;*DDK*************************************************************************/
;
; COPYRIGHT    Copyright (C) 1995 IBM Corporation
;
;    The following IBM OS/2 WARP source code is provided to you solely for
;    the purpose of assisting you in your development of OS/2 WARP device
;    drivers. You may use this code in accordance with the IBM License
;    Agreement provided in the IBM Device Driver Source Kit for OS/2. This
;    Copyright statement may not be removed.;
;*****************************************************************************/
;***********************************************************************
;
;   Module          = EDDHLINE
;
;   Description     = Private DD code
;
;   Function        = Does line drawing work.
;
;acw This will go into dl2, look at horz/vert optimization
;***********************************************************************

INCL_GPIBITMAPS EQU     1
include os2.inc
include eddinclt.inc

include eddhcone.inc
include eddhmacr.inc
include eddhtype.inc

include eddfcone.inc
include eddfmacr.inc
include eddftype.inc

IFDEF _8514
?DF     equ     1       ; we do not want _TEXT segment defined by cmacros.
include cmacros.inc
include 8514.inc
ENDIF

;----------------------------------------------------------------------

; Constants

LINETYPE_SOLID equ 07
GUARD_X_MIN     equ -2048
GUARD_X_MAX     equ 6143
GUARD_Y_MIN     equ -2048
GUARD_Y_MAX     equ 6143
MIN_COORD       equ -8192      ;minimum coordinate value clipping can handle
MAX_COORD       equ 8191       ;maximum coordinate value clipping can handle
POLYLINE_CALL   equ 1
DISJOINT_CALL   equ 2

;----------------------------------------------------------------------


; The parameter block
; Note. This must also serve as a parameter block for the plotstep
; routine.

POLYLINEPB     struc
ulPixelOp       dd      ?       ; pixel op values to use
cClipRects      dw      ?       ; number of clip rectangles
pClipRects      dd      ?       ; pointer to the clip rectangles        ;**
ptsFirstPos     dd      ?       ; first position (device coords)
ptsDevOffsets   dd      ?       ; device offsets to be applied to points
cPoints         dd      ?       ; number of points
pPoints         dd      ?       ; pointer to the points
usPatPos        dw      ?       ; pattern position
usPatType       dw      ?       ; line type
pPlotStepCodes  dd      ?       ; pointer to plot step codes            ;**
usPatEnd        dw      ?       ; pattern position after operation
usCallType      dw      ?       ; identifies polyline and disjointline
pbmhDest        dd      ?       ; pointer to the destination bitmap (PMLINES);**
POLYLINEPB      ends

;----------------------------------------------------------------------

; The scratch pad block

PolyLineSP      struc
ptsStartPoint   dd      ?       ; start point of line (before clipping)
ptsEndPoint     dd      ?       ; end point of line (before clipping)
usPatPosTmp     dw      ?       ; store for pattern position
sETstart        dw      ?       ; preclipped bresenham error term
ptsNewStart     dd      ?       ; clipped start point
ptsNewEnd       dd      ?       ; clipped end point
sDeltaX         dw      ?       ; end point X - start point X
sDeltaY         dw      ?       ; end point Y - start point Y
fsClipStatus    dw      ?       ; position of points relative to clip rectangle
sK1             dw      ?       ; temp copy of bresenham constant K1
sK2             dw      ?       ; temp copy of bresenham constant K2
sK1_K2          dw      ?       ; temp copy of K1-K2
ulPointCount    dd      ?       ; count of lines left to draw
usClipCount     dw      ?       ; count of clip rectangles left to draw into
pOldClipRects   dd      ?       ; pointer to original clip rectangles in PMHline
                                                                        ;**
cOldClips       dw      ?       ; original number of clip rectangles in PMHline
ulPixOpWork     dd      ?       ; working value of pixel op
IFDEF _8514
usMajorAxis     dw      ?
usErrorTerm     dw      ?
usCmdFlags      dw      ?
ENDIF

; ********* If you add anything to this you MUST change the definition in
; eddhtype.h which matches this *********

PolyLineSP     ends

;***********************************************************************
; handle the soft draw/hard draw differences
;***********************************************************************
IFDEF HARD_DRAW
public  _eddh_PMLINES
public  _eddh_PMPLOTSTEP

PMLines         equ     <_eddh_PMLINES>
PMPlotStep      equ     <_eddh_PMPLOTSTEP>
bitmap_address  equ     phys_addr
ENDIF ; HARD_DRAW

IFDEF SOFT_DRAW
public  _eddf_PMLINES
public  _eddf_PMPLOTSTEP

PMLines         equ     <_eddf_PMLINES>
PMPlotStep      equ     <_eddf_PMPLOTSTEP>
bitmap_address  equ     virt_addr

extrn _eddf_MESS        :proc
ENDIF ; SOFT_DRAW

ifndef   _8514
pXGARegs        equ     <_pXGARegs>
else
p8514Regs       equ     <_p8514Regs>
endif

;----------------------------------------------------------------------
;----------------------------------------------------------------------

_DATA           segment dword public use32 'DATA'                       ;**

; Define external data for this module

ifndef   _8514
extrn pXGARegs          :dword
else
extrn p8514Regs         :dword
extrn _LinePatternsBitmap :dword
externDP Conv8514toXGA
extrn _DDT              :byte
extrn   _Shadow8514Regs :dword
endif
extrn _AIxfer           :dword
extrn _SPad             :dword

extrn _LinePatternCur   :dword
extrn _pbHWPollRegister :dword

_DATA           ends

;----------------------------------------------------------------------
;----------------------------------------------------------------------

_TEXT           segment dword use32 public 'CODE'                       ;**
                assume  cs:FLAT, ds:FLAT, es:FLAT

; Define external functions

extrn _eddl_BresenhamOne :proc                                           ;**

;----------------------------------------------------------------------



;***********************************************************************
;
; Function: PMlines
;
; Called by: eddl_PolyLineWork in eddlpoly.c
;
; Draws a set of lines between the coordinates stored at .pPoints
; Draws either disjoint lines or continuous lines depending on value
; in .usCallType
;
; .ulPixOpValue holds the pixel operation for the first vector
;
; .cClipRects and .pClipRects allow this function to draw each line
; into all the clip rectangles
;
; .ptsFirstPos is set up before this function is called to the start
; point
;
; .ptsDevOffsets.x and .ptsDevOffsets.y are the offsets which must be applied
; to convert device coordinates to hardware coordinates.
;
; This function calls our bresenham software simulation if the line is
; too long (>4096 pels) or its coordinate values are too large to be
; handled by the clipping algorithm. The simulation _must_ be transparent
; to this function so any important values must be saved and restored by it.
;
;
; 386 only code
;***********************************************************************

        align   4
PMLines         proc

        push    edi
        push    esi
        push    ebx
        pushxga

; Get access to the hardware registers
        ifndef  _8514
        movxga  pXGARegs
        else
        movxga  p8514Regs
        endif

; set up the destination bitmap (as pixel map A)
        mov     ebx, _AIxfer.pbmhDest                                    ;**
        memregwrite     pi_map_index_A, SEL_PIX_MAP_A

; bitmap address
        mov     eax, [ebx].bitmap_address
        memregwrite     pi_map_base_ptr_A, eax

; width and height
        mov     ax, [ebx].hw_height                                     ;**
        swap    eax
        mov     ax, [ebx].hw_width                                      ;**
        memregwrite     pi_map_size_A, eax

; format
        mov     ax, [ebx].hw_format                                     ;**
        memregwrite     pi_map_format_A, al


; We need a temporary copy of this to save us stomping on the one set
; up by polylinework because we may be called multiple times (ie by
; walkthebitmap).
IFDEF _8514
        mov     _SPad.usCmdFlags,0
ENDIF
        mov     eax, _AIxfer.ulPixelOp
        mov     _SPad.ulPixOpWork, eax

; Set up loop and get access to the coordinate data
        mov     ecx, _AIxfer.cPoints
        mov     _SPad.ulPointCount, ecx
        mov     edi, _AIxfer.pPoints                                     ;**

; Is there a pattern which we need to set up
        cmp     _AIxfer.usPatType, LINETYPE_SOLID
        jz      short nopattern

; Set up the pattern pixel map as map C

; No need to wait as calling routine has already done it

; Get access to pixel map C and set up its parameters
        memregwrite     pi_map_index_C, SEL_PIX_MAP_C
        memregwrite     pi_map_width_C, LINE_PATTERN_LENGTH-1
        memregwrite     pi_map_height_C, 0
        memregwrite     pi_map_format_C, ONE_BPP+MOTOROLA

IFDEF IBMJ                                                  
; Get physical address of pattern (hardcoded for 32 bit patterns)
ELSE 
; Get physical address of pattern (hardcoded for 16 bit patterns)
ENDIF 

        mov     eax, _LinePatternCur
        movzx   edx, _AIxfer.usPatType                                   ;**
IFDEF IBMJ                                                  
 IFDEF XGA                                                  
        lea     eax, [eax+edx*4]
 ELSE 
        add     eax, edx
        add     eax, edx
 ENDIF 
ELSE 
        add     eax, edx
        add     eax, edx
ENDIF 
        memregwrite     pi_map_base_ptr_C, eax
;
; @RCW - We need to adjust pattern biasing for the 8514/A.
;

; Set up the start position in the pattern and also initialise our
; running total of where the pattern position is (.usPatEnd)
        memregwrite     patt_map_y, 0
        mov     ax, _AIxfer.usPatPos
        mov     _AIxfer.usPatEnd, ax
        mov     _SPad.usPatPosTmp, ax
        memregwrite     patt_map_x, ax

; Alter the pixel op because we are drawing with a pattern
        mov     eax, _SPad.ulPixOpWork
        and     eax, 0FFFF0FFFh
        or      eax, 000003000h
        mov     _SPad.ulPixOpWork, eax

nopattern:
; Get the start point for the lines
        mov     eax, _AIxfer.ptsFirstPos
        mov     _SPad.ptsStartPoint, eax

; Don't check disjoint lines for closed
;       test    _AIxfer.usCallType, DISJOINT_CALL
;       jnz     openpline

; Only check PolyLines for closed
        cmp     _AIxfer.usCallType, POLYLINE_CALL
        jne     short openpline

; Work out if the polyline is closed

; Get the x coordinate of the last point (.cPoints will be > 0)
        mov     ebx, _AIxfer.cPoints                                    ;**
        dec     ebx                                                     ;**

; if this decrement took us to zero we have a single point
; and a single point will always have start and end points equal!
        jz      short openpline

        shl     ebx, 3                                                  ;**
        mov     dx, [edi+ebx]                                           ;**

; Apply the appropriate offset
        add     dx, _AIxfer.ptsDevOffsets.x

; Is the x coordinate the same? Quit if not
        cmp     dx, ax
        jne     short openpline

; Get the y coordinate of the last point and apply the offset
        mov     dx, _AIxfer.ptsDevOffsets.y
        sub     dx, [edi+ebx+4]                                         ;**

; And the y coordinate of the first point
        swap    eax

; Are they the same? Quit if not
        cmp     dx, ax
        jne     short openpline

; The polyline is a closed one so adjust the pixel op accordingly
        mov     eax, _SPad.ulPixOpWork
        or      al, 00010000b            ;set omit first pel mode
        mov     _SPad.ulPixOpWork, eax
IFDEF _8514
        or      _SPad.usCmdFlags, CMD_LP_NULL  ;set omit last pel mode
ENDIF

openpline:
; The polyline is an open one and this is the default set up for
; .ulPixOpValue

;this turns off scissoring and clear octant field
        mov     eax, _SPad.ulPixOpWork
        and     al, 00111000b
        mov     _SPad.ulPixOpWork, eax

; Now go into the per-vector loop..
        mov     eax, _SPad.ptsStartPoint
vloop:

; Get the next ptsEndPoint. (eax already contains start point).
; Get the next y coordinate and apply the device to hardware offsets
        mov     dx, _AIxfer.ptsDevOffsets.y
        sub     dx, [edi+4]                                             ;**

; Save the coord for later use
        swap    edx

; Now do the same for the x coordinate
        mov     dx, _AIxfer.ptsDevOffsets.x
        add     dx, [edi]                                               ;**

; Save the ptsEndPoint for future use
        mov     _SPad.ptsEndPoint, edx

; Adjust edi ready for next point
        add     edi, 8                                                  ;**


; Test the ptsStartPoint to see if clipping algorithm can handle it
; ptsStartPoint - eax
; ptsEndPoint - edx
; Note that .ptsStartPoint and .ptsEndPoint are set up ready for a 'softdraw'

; Start with x coordinate
        cmp     ax, MIN_COORD
        jle     softdraw
        cmp     ax, MAX_COORD
        jge     softdraw

; Now get y coordinate
        swap    eax
        cmp     ax, MIN_COORD
        jle     softdraw
        cmp     ax, MAX_COORD
        jge     softdraw

; We are going to want the start point for working out bresenham
; registers
        xchg    edx, eax

; Test the ptsEndPoint to see if clipping algorithm can handle it
; First the x coordinate
        cmp     ax, MIN_COORD
        jle     softdraw
        cmp     ax, MAX_COORD
        jge     softdraw

; Then the y coordinate
        swap    eax
        cmp     ax, MIN_COORD
        jle     softdraw
        cmp     ax, MAX_COORD
        jge     softdraw

ifdef HARD_DRAW
ifdef _8514
; Is there a pattern which we need to set up?
; If so, the 8514/A can't handle it.  We must us the Bresenham line
; drawing simulation code.
        cmp     _AIxfer.usPatType, LINETYPE_SOLID
        jnz     softdraw

ifdef   BPP24
        test    [_DDT],USE_24BPP
        jnz     softdraw
endif
endif
endif

; Now..
; eax contains the end point with y in low word and x in high word
; edx contains the start point with y in low word and x in high word

; Now calculate the bresenham values.  This is taken from AI macro
; and altered to avoid memory access.  BX is the octant code.

; Clear octant mask first
        sub     bx, bx

; Calculate delta-y
        sub     ax, dx                  ;y2 - y1

; If delta-y < 0 then
        jns     short ypos

;   Set dy flag for negative
        or      bx, OCT_DY_BIT
        neg     ax

ypos:
; Hide the delta-y (ax) and get back the xstart and xend
        swap    eax
        swap    edx

; Calculate delta-x
        sub     ax, dx                  ;x2 - x1

; If delta-x < 0 then
        jns     short xpos

;   Set dx flag for negative
        or      bx, OCT_DX_BIT
        neg     ax

xpos:
; Put the delta-x into dx and get back delta-y
        mov     dx, ax
        swap    eax

; If Y is major then
        cmp     dx, ax
        ja      short ymin

;   Set dz flag and make dx the major dimension
        or      bx, OCT_DZ_BIT
        xchg    ax, dx

ymin:

IFDEF _8514
        and     _SPad.usCmdFlags,NOT CMD_DYZX

        test    bx,OCT_DY_BIT
        jnz     short @F
        or      _SPad.usCmdFlags,CMD_DY
@@:
        test    bx,OCT_DX_BIT
        jnz     short @F
        or      _SPad.usCmdFlags,CMD_DX
@@:
        test    bx,OCT_DZ_BIT
        jz      short @F
        or      _SPad.usCmdFlags,CMD_DZ
@@:
ENDIF

; dx is now the major dimension. Can now increase the pattern count
; (we don't trust the hardware with this) and check that the line can
; be drawn using bresenham line drawing and software clipping.

        add     _AIxfer.usPatEnd, dx
        and     _AIxfer.usPatEnd, 00001111b ;pattern width is always 16 bits

ifndef  _8514
        cmp     dx, MAX_LINE_LENGTH      ;line is too long so is drawn using
        jae     softdraw                 ;software simulation
endif

; Wait for the hardware before writing to the registers
;acw  NO NEED TO DO THE REGISTER SAVES AT THIS POINT.
;     COMMENT OUT THE WAITSHORT AND MOVE THE MEMREGWRITES DOWN TO THE END
;        waitshort

; Calculate K1, K2, DimX and ET
        sal     ax,1                    ; AX=2*b
        mov     _SPad.sK1, ax            ; keep copy of sK1
;acw commented out
;        memregwrite  bres_k1, ax        ; Set sK1
        sub     ax,dx                   ; AX=2*b-a
        mov     _SPad.sETstart, ax       ; ET=2*b-a
        sub     ax,dx                   ; AX=2*b-2*a
        mov     _SPad.sK2, ax            ; keep copy of k2
;acw commented out
;        memregwrite  bres_k2, ax        ; Set K2
        sub     ax, _SPad.sK1            ; ax=k2-K1
        neg     ax
        jne     short @F        ;avoid overflow later when dividing by sK1_K2
        mov     ax, 0ffh
@@:     mov     _SPad.sK1_K2, ax         ; keep copy of K1-k2

        ;Adjust ET so that lines are retraceable when points are swapped.
        ;(ie. line x1,y1 -> x2,y2 is the same as line x2,y2 -> x1,y1)
        cmp     bx, 3                   ;bx still contains the direction octant
        jl      short @F
        cmp     bx, 5
        je      short @F
        dec     _SPad.sETstart
@@:

; Set up the correct pixel operation by or'ing in the octant code
        mov     eax, _SPad.ulPixOpWork
        or      ax, bx                  ;bx still contains the direction octant
        mov     _SPad.ulPixOpWork, eax

; Set up the per clip rectangle loop
        mov     ax, _AIxfer.cClipRects
        mov     _SPad.usClipCount, ax
        mov     ebx, _AIxfer.pClipRects                                  ;**

IFDEF HARD_DRAW
; this is just here for debugging !
public clip_loop
ENDIF

clip_loop:

;***********************************************************************
;
; General clipping algorithm.
;
;
; The lines are drawn using Bresenhams well known algorithm. The
; start and end points and the error term must be adjusted so that
; the pels drawn are completely reproducable.
;
; Through out the code comments (sx,sy) and (ex,ey) refer to the original
; unclipped start and end points respectively, and (nsx,nsy) and (nex,ney)
; refer to the new clipped start and end points respectively.
;
;
; Consider an X major line in the first octant.
;
; The initial error term and the constants are given by:
;
;       ETstart = 2dY - dX , K1 = 2dY , K2 = 2dY - 2dX
;
; where dX = ex - sx and dY = ey - sy
;
;
; To clip against the vertical clip boundary x0 (ie. in the major direction):
;
;       nsx = x0 , nsy = round1( sy + (x0 - sx)*dY/dX )
;
; To clip it against the horizontal clip boundary y0 (in the minor direction):
;
;       nsy = y0 , nsx = round2( sx + (y0 - sy - 0.5)*dX/dY )
;
; The -0.5 in the above equation effectively lowers the clip boundary
; by a half so that all of the pels that lie on the real clip boundary
; are certain to be drawn.
;
; In both cases the adjusted error term ET is given by:
;
; ET = K2 + (ETstart - K2 + K1*(x0 - sx))mod(K1 - K2)
;
; The rounding functions used to convert the floating point result to an
; actual integer coordinate are crucial to ensuring we follow our line drawing
; metric correctly. In the above cases round1 rounds any fractions to the
; nearest integer rounding halves away from zero, and round2 rounds any
; fraction up to the nearest integer.
;
; The equations above refer to only one octant. The equivilant equations
; for the other 7 octants are identical in form. Care should be taken
; in choosing the rounding function correctly:
;
; ͻ
; clipping   effect on major  rounding function             
; dimension  coordinate                                     
; ͹
; major      increase         round to nearest integer,     
;                             rounding halves away from zero
; Ķ
; major      decrease         round to nearest integer,     
;                             rounding halves towards zero  
; Ķ
; minor      increase         round up to next integer      
; Ķ
; minor      decrease         round up to next integer      
;                             and subtract one              
; ͼ
;
;
;***********************************************************************

; First set up "clip status" in dx which gives the position of the start
; and end points relative to the clip rectangle boundaries.
; NB. coordinate values have a range of 14 bits to avoid overflow in the
; clipping algrorithm.

        mov     eax, _SPad.ptsStartPoint         ;eax = sy:sx
        mov     _SPad.ptsNewStart, eax           ;nsy:nsx = eax
        add     ax, 2000h                       ;bias sx by 8k so it will always
                                                ;be +ve (can use carry of cmp)

        add     [ebx].clip_x0, 2000h    ;bias clip bounds by 8k so they match ;**
        add     [ebx].clip_x1, 2000h    ;the start and end point biasing for ;**
        add     [ebx].clip_y0, 2000h    ;following comparisons          ;**
        add     [ebx].clip_y1, 2000h                                    ;**

        xor     dx, dx                  ;clear dx

        cmp     ax, [ebx].clip_x0       ;compare sx with clip rectangle bounds ;**
        rcl     dl, 1                   ;set sx < x0 bit
        cmp     [ebx].clip_x1, ax                                       ;**
        rcl     dl, 1                   ;set sx > x1 bit
        swap    eax
        add     ax, 2000h               ;bias sy by 8k so it will always be +ve
        cmp     ax, [ebx].clip_y0       ;compare sy with clip rectangle bounds ;** ;**
        rcl     dl, 1                   ;set sy < y0 bit
        cmp     [ebx].clip_y1, ax                                       ;**
        rcl     dl, 1                   ;set sy > y1 bit

        mov     eax, _SPad.ptsEndPoint      ;eax = ey:ex
        mov     _SPad.ptsNewEnd, eax        ;ney:nex = eax
        add     ax, 2000h               ;bias ex by 2k so is always +ve

        cmp     ax, [ebx].clip_x0       ;compare ex with clip rectangle bounds ;**
        rcl     dl, 1                   ;set ex < x0 bit
        cmp     [ebx].clip_x1, ax                                       ;**
        rcl     dl, 1                   ;set ex > x1 bit
        swap    eax
        add     ax, 2000h               ;bias ey by 2k so is always +ve
        cmp     ax, [ebx].clip_y0       ;compare ey with clip rectangle bounds ;**
        rcl     dl, 1                   ;set ey < y0 bit
        cmp     [ebx].clip_y1, ax                                       ;**
        rcl     dl, 1                   ;set ey > y1 bit

        sub     [ebx].clip_x0, 2000h    ;unbias the clip bounds         ;**
        sub     [ebx].clip_x1, 2000h                                    ;**
        sub     [ebx].clip_y0, 2000h                                    ;**
        sub     [ebx].clip_y1, 2000h                                    ;**


; dl now contains bits set to indicate clipping status in the following format:
; (bit 7) sx<x0 | sx>x1 | sy<y0 | sy>y1 | ex<x0 | ex>x1 | ey<y0 | ey>y1 (bit 0)

        cmp     dl, 0                   ;fast path for lines which lie entirely
        je      no_clipping_needed      ;within the clip rectangle

        shl     dx, 4           ;seperate the start point and end point bits
        shr     dl, 4           ;dl = end point bits, dh = start point bits
                                ;(in bottom 4 bits of each)

        test    dl, dh                  ;fast path for lines which lie entirely
        jne     completely_clipped      ;outside the clip rectangle

        sub     ax, 2000h                       ;unbias y end point
        sub     ax, _SPad.ptsStartPoint.y        ;calculate sDeltaY
        swap    eax                             ;get biased x end point
        sub     ax, 2000h                       ;unbias x end point
        sub     ax, _SPad.ptsStartPoint.x        ;calculate sDeltaX
        mov     dword ptr _SPad.sDeltaX, eax     ;store sDeltaX:sDeltaY in _SPad

        mov     _SPad.fsClipStatus, dx   ;save clip status bits

; Now calculate the start point switch index so that we can jump to case
; specific code to deal with each type of clipping correctly

; NB we change si here, so must reload esi before we use it to access
; the XGA hardware registers
        mov     si, dx
        shr     si, 5                   ;get start point status into bits
        and     esi, 01111000b          ;3 to 6 of low byte of si       ;**
        test    byte ptr _SPad.ulPixOpWork, OCT_DZ_BIT  ;test if X or Y major
        je      short @F
        or      si, 00000100b  ;line is Y major so adjust case index accordingly
@@:

        jmp     dword ptr cs:clip_start_switch[esi]
clip_start_switch:
                dd      offset FLAT:start_clipped_ok                                ;**
                dd      offset FLAT:start_clipped_ok                                ;**
                dd      offset FLAT:xm_sy_gt_y1                                     ;**
                dd      offset FLAT:ym_sy_gt_y1                                     ;**
                dd      offset FLAT:xm_sy_lt_y0                                     ;**
                dd      offset FLAT:ym_sy_lt_y0                                     ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:xm_sx_gt_x1                                     ;**
                dd      offset FLAT:ym_sx_gt_x1                                     ;**
                dd      offset FLAT:xm_sx_gt_x1_sy_gt_y1                            ;**
                dd      offset FLAT:ym_sx_gt_x1_sy_gt_y1                            ;**
                dd      offset FLAT:xm_sx_gt_x1_sy_lt_y0                            ;**
                dd      offset FLAT:ym_sx_gt_x1_sy_lt_y0                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:xm_sx_lt_x0                                     ;**
                dd      offset FLAT:ym_sx_lt_x0                                     ;**
                dd      offset FLAT:xm_sx_lt_x0_sy_gt_y1                            ;**
                dd      offset FLAT:ym_sx_lt_x0_sy_gt_y1                            ;**
                dd      offset FLAT:xm_sx_lt_x0_sy_lt_y0                            ;**
                dd      offset FLAT:ym_sx_lt_x0_sy_lt_y0                            ;**

xm_sy_gt_y1:

; The line is X major and the start point lies directly below the clip rectangle

        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     ax, 0                           ;now convert back to integer:
        jge     short @F                        ;if nsx-sx is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nsx-sx is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx
        cmp     ax, [ebx].clip_x0               ;check nsx is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

ym_sy_gt_y1:

; The line is Y major and the start point lies directly below the clip rectangle

        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     dx, 0                           ;now convert back to integer
        jne     short abit1                     ;by rounding to the nearest
        cmp     ax, 0                           ;integer, and rounding halves
        jge     short @F                        ;towards zero.
        test    ax, 01b
        je      short @F
        add     ax, 10b
        jmp     short @F
abit1:
        cmp     ax, 0
        jl      short @F
        inc     ax
@@:     sar     ax, 1
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx
        cmp     ax, [ebx].clip_x0               ;check nsx is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

xm_sy_lt_y0:

; The line is X major and the start point lies directly above the clip rectangle

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     ax, 0                           ;now convert back to integer:
        jge     short @F                        ;if nsx-sx is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nsx-sx is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx
        cmp     ax, [ebx].clip_x0               ;check nsx is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

ym_sy_lt_y0:

; The line is Y major and the start point lies directly above the clip rectangle

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     ax, 0                           ;now convert back to integer
        jl      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, rounding halves away
@@:     sar     ax, 1                           ;from zero.
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx
        cmp     ax, [ebx].clip_x0               ;check nsx is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

xm_sx_gt_x1:

; The line is X major and the start point lies directly to the right of the
; clip rectangle

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        cmp     dx, 0                           ;now convert back to integer
        jne     short abit2                     ;by rounding to the nearest
        cmp     ax, 0                           ;integer, and rounding halves
        jge     short @F                        ;towards zero.
        test    ax, 01b
        je      short @F
        add     ax, 10b
        jmp     short @F
abit2:
        cmp     ax, 0
        jl      short @F
        inc     ax
@@:     sar     ax, 1
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy
        cmp     ax, [ebx].clip_y0               ;check nsy is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

ym_sx_gt_x1:

; The line is Y major and the start point lies directly to the right of the
; clip rectangle

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaX                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     ax, 0                           ;now convert back to integer
        jge     short @F                        ;if nsy-sy is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nsy-sy is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, word ptr _SPad.ptsStartPoint[2] ;ax = sy + (nsx-sx)*dY/dX = nsy
        cmp     ax, [ebx].clip_y0               ;check nsy is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

xm_sx_gt_x1_sy_gt_y1:

; The line is X major and the start point lies below and to the right of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        cmp     dx, 0                           ;now convert back to integer
        jne     short @F                        ;by rounding to the nearest
        test    ax, 01b                         ;integer, and rounding halves
        je      short @F                        ;towards zero.
        add     ax, 10b                         ;(nsy-sy is always -ve)
@@:
        sar     ax, 1
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x1 resulted in nsy > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nsx-sx is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsStartPoint.x        ;ax=sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x0       ;check nsx is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

ym_sx_gt_x1_sy_gt_y1:

; The line is Y major and the start point lies below and to the right of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nsy-sy is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x1 resulted in nsy > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     dx, 0                           ;now convert back to integer
        jne     short @F                        ;by rounding to the nearest
        test    ax, 01b                         ;integer, and rounding halves
        je      short @F                        ;towards zero.
        add     ax, 10b                         ;(nsy-sy is always -ve)
@@:
        sar     ax, 1
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x0       ;check nsx is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

xm_sx_gt_x1_sy_lt_y0:

; The line is X major and the start point lies above and to the right of the
; clip rectangle

; First try clipping against x1

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        cmp     dx, 0                           ;now convert back to integer
        je      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, and rounding halves
@@:                                             ;towards zero.
        sar     ax, 1                           ;(nsy-sy is always +ve)
        add     ax, _SPad.ptsStartPoint.y        ;ax = sx + (nsx-sx)*dY/dX = nsy

                                        ;now check nsy is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x1 resulted in nsy < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nsx-sx is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x0       ;now check nsx is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

ym_sx_gt_x1_sy_lt_y0:

; The line is Y major and the start point lies above and to the right of the
; clip rectangle

; First try clipping against x1

        mov     ax, [ebx].clip_x1               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nsy-sy is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x1 resulted in nsy < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1
        imul    _SPad.sDeltaX                    ;ax = (nsy-sy)* dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        sar     ax, 1                           ;now convert back to integer
                                                ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nsx-sx is always -ve)
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x0       ;check nsx is within clip bounds
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

xm_sx_lt_x0:

; The line is X major and the start point lies directly to the left of the
; clip rectangle

        mov     ax, [ebx].clip_x0               ;nsx = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        cmp     ax, 0                           ;now convert back to integer
        jl      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, rounding halves away
@@:     sar     ax, 1                           ;from zero.
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy
        cmp     ax, [ebx].clip_y0               ;check nsy is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

ym_sx_lt_x0:

; The line is Y major and the start point lies directly to the left of the
; clip rectangle

        mov     ax, [ebx].clip_x0               ;nsx = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaX                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     ax, 0                           ;now convert back to integer
        jge     short @F                        ;if nsy-sy is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nsy-sy is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy
        cmp     ax, [ebx].clip_y0               ;check nsy is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax           ;store nsy:nsx in _SPad
        jmp     start_clipped_ok                ;now clip end point

xm_sx_lt_x0_sy_gt_y1:

; The line is X major and the start point lies below and to the left of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x0               ;nsx = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsx               ;**
        sub     ax, word ptr _SPad.ptsStartPoint    ;ax = nsx - sx
        shl     ax ,1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        sar     ax, 1                           ;now convert back to integer
                                                ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nsy-sy is always -ve)
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
                                        ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x0 resulted in nsy > y1 so now try clipping to y1
        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nsx-sx is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsStartPoint.x        ;ax=sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x1       ;check nsx is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

ym_sx_lt_x0_sy_gt_y1:

; The line is Y major and the start point lies below and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nsx = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax ,1                           ;shift binary point
        dec     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nsy-sy is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x0 resulted in nsy > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;nsy = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        cmp     dx, 0                           ;now convert back to integer
        je      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, and rounding halves
@@:                                             ;towards zero.
        sar     ax, 1                           ;(nsy-sy is always +ve)
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx

        cmp     ax, [ebx].clip_x1       ;check nsx is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

xm_sx_lt_x0_sy_lt_y0:

; The line is X major and the start point lies above and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nsx = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        inc     ax                              ;now convert back to integer
        sar     ax, 1                           ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nsy-sy is always +ve)
        add     ax, _SPad.ptsStartPoint.y        ;ax = sx + (nsx-sx)*dY/dX = nsy

                                        ;now check nsy is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point
@@:

; Clipping to x0 resulted in nsy < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;ax = (nsy-sy) * dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nsx-sx is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx ;**

        cmp     ax, [ebx].clip_x1       ;now check nsx is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     start_clipped_ok        ;now clip end point

ym_sx_lt_x0_sy_lt_y0:

; The line is Y major and the start point lies above and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nsx = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nsx               ;**
        sub     ax, _SPad.ptsStartPoint.x        ;ax = nsx - sx
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nsx-sx) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nsx-sx)*dY / dX = nsy-sy
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nsy-sy is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsStartPoint.y        ;ax = sy + (nsx-sx)*dY/dX = nsy

                                        ;check nsy is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
        jmp     short start_clipped_ok  ;now clip end point
@@:

; Clipping to x0 resulted in nsy < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;nsy = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = nsy               ;**
        sub     ax, _SPad.ptsStartPoint.y        ;ax = nsy - sy          ;**
        shl     ax, 1
        imul    _SPad.sDeltaX                    ;ax = (nsy-sy)* dX
        idiv    _SPad.sDeltaY                    ;ax = (nsy-sy)*dX / dY = nsx-sx
        inc     ax                              ;now convert back to integer
        sar     ax, 1                           ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nsx-sx is always +ve)
        add     ax, _SPad.ptsStartPoint.x        ;ax = sx + (nsy-sy)*dX/dY = nsx


        cmp     ax, [ebx].clip_x1       ;check nsx is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewStart, eax   ;store nsy:nsx in _SPad
                                        ;now clip end point


IFDEF HARD_DRAW
; this is just here for debugging !
public start_clipped_ok
ENDIF

start_clipped_ok:
        mov     dl, byte ptr _SPad.fsClipStatus
        mov     si, dx
        shl     si, 3          ;shift end point status into bits 3 to 6 of
        and     esi, 01111000b ;low byte of si                          ;**
        test    byte ptr _SPad.ulPixOpWork, OCT_DZ_BIT  ;test if X or Y major
        je      short @F
        or      si, 00000100b  ;line is Y major so adjust case index accordingly
@@:
        jmp     dword ptr cs:clip_end_switch[esi]                       ;**

clip_end_switch:
                dd      offset FLAT:clipped_ok                                      ;**
                dd      offset FLAT:clipped_ok                                      ;**
                dd      offset FLAT:xm_ey_gt_y1                                     ;**
                dd      offset FLAT:ym_ey_gt_y1                                     ;**
                dd      offset FLAT:xm_ey_lt_y0                                     ;**
                dd      offset FLAT:ym_ey_lt_y0                                     ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:xm_ex_gt_x1                                     ;**
                dd      offset FLAT:ym_ex_gt_x1                                     ;**
                dd      offset FLAT:xm_ex_gt_x1_ey_gt_y1                            ;**
                dd      offset FLAT:ym_ex_gt_x1_ey_gt_y1                            ;**
                dd      offset FLAT:xm_ex_gt_x1_ey_lt_y0                            ;**
                dd      offset FLAT:ym_ex_gt_x1_ey_lt_y0                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:clip_rectangle_error                            ;**
                dd      offset FLAT:xm_ex_lt_x0                                     ;**
                dd      offset FLAT:ym_ex_lt_x0                                     ;**
                dd      offset FLAT:xm_ex_lt_x0_ey_gt_y1                            ;**
                dd      offset FLAT:ym_ex_lt_x0_ey_gt_y1                            ;**
                dd      offset FLAT:xm_ex_lt_x0_ey_lt_y0                            ;**
                dd      offset FLAT:ym_ex_lt_x0_ey_lt_y0                            ;**

xm_ey_gt_y1:

; The line is X major and the end point lies directly below the clip rectangle

        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     ax, 0                           ;now convert back to integer:
        jge     short @F                        ;if nex-ex is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nex-ex is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex
        cmp     ax, [ebx].clip_x0               ;check nex is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

ym_ey_gt_y1:

; The line is Y major and the end point lies directly below the clip rectangle

        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     dx, 0                           ;now convert back to integer
        jne     short abit3                     ;by rounding to the nearest
        cmp     ax, 0                           ;integer, and rounding halves
        jge     short @F                        ;towards zero.
        test    ax, 01b
        je      short @F
        add     ax, 10b
        jmp     short @F
abit3:
        cmp     ax, 0
        jl      short @F
        inc     ax
@@:     sar     ax, 1
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex
        cmp     ax, [ebx].clip_x0               ;check nex is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

xm_ey_lt_y0:

; The line is X major and the end point lies directly above the clip rectangle

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     ax, 0                           ;now convert back to integer:
        jge     short @F                        ;if nex-ex is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if nex-ex is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex
        cmp     ax, [ebx].clip_x0               ;check nex is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

ym_ey_lt_y0:

; The line is Y major and the end point lies directly above the clip rectangle

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     ax, 0                           ;now convert back to integer
        jl      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, rounding halves away
@@:     sar     ax, 1                           ;from zero.
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex
        cmp     ax, [ebx].clip_x0               ;check nex is within clip bounds ;** ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_x1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

xm_ex_gt_x1:

; The line is X major and the end point lies directly to the right of the
; clip rectangle

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        cmp     dx, 0                           ;now convert back to integer
        jne     short abit4                     ;by rounding to the nearest
        cmp     ax, 0                           ;integer, and rounding halves
        jge     short @F                        ;towards zero.
        test    ax, 01b
        je      short @F
        add     ax, 10b
        jmp     short @F
abit4:
        cmp     ax, 0
        jl      short @F
        inc     ax
@@:     sar     ax, 1
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney
        cmp     ax, [ebx].clip_y0               ;check ney is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

ym_ex_gt_x1:

; The line is Y major and the end point lies directly to the right of the
; clip rectangle

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaX                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     ax, 0                           ;now convert back to integer
        jge     short @F                        ;if ney-ey is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if ney-ey is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney
        cmp     ax, [ebx].clip_y0               ;check ney is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

xm_ex_gt_x1_ey_gt_y1:

; The line is X major and the end point lies below and to the right of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        cmp     dx, 0                           ;now convert back to integer
        jne     short @F                        ;by rounding to the nearest
        test    ax, 01b                         ;integer, and rounding halves
        je      short @F                        ;towards zero.
        add     ax, 10b                         ;(nsy-sy is always -ve)
@@:
        sar     ax, 1
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x1 resulted in ney > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nex-ex is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsEndPoint.x          ;ax=ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x0       ;check nex is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

ym_ex_gt_x1_ey_gt_y1:

; The line is Y major and the end point lies below and to the right of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;ney-ey is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x1 resulted in ney > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     dx, 0                           ;now convert back to integer
        jne     short @F                        ;by rounding to the nearest
        test    ax, 01b                         ;integer, and rounding halves
        je      short @F                        ;towards zero.
        add     ax, 10b                         ;(nsy-sy is always -ve)
@@:
        sar     ax, 1
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x0       ;check nex is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

xm_ex_gt_x1_ey_lt_y0:

; The line is X major and the end point lies above and to the right of the
; clip rectangle

; First try clipping against x1

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        cmp     dx, 0                           ;now convert back to integer
        je      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, and rounding halves
@@:                                             ;towards zero.
        sar     ax, 1                           ;(nsy-sy is always +ve)
        add     ax, _SPad.ptsEndPoint.y          ;ax = ex + (nex-ex)*dY/dX = ney

                                        ;now check ney is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x1 resulted in ney < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;nex-ex is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x0       ;now check nex is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

ym_ex_gt_x1_ey_lt_y0:

; The line is Y major and the end point lies above and to the right of the
; clip rectangle

; First try clipping against x1

        mov     ax, [ebx].clip_x1               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x1               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;ney-ey is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x1 resulted in ney < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1
        imul    _SPad.sDeltaX                    ;ax = (ney-ey)* dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        sar     ax, 1                           ;now convert back to integer
                                                ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nex-ex is always -ve)
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x0       ;check nex is within clip bounds ;**
        jl      completely_clipped      ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

xm_ex_lt_x0:

; The line is X major and the end point lies directly to the left of the
; clip rectangle

        mov     ax, [ebx].clip_x0               ;nex = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        cmp     ax, 0                           ;now convert back to integer
        jl      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, rounding halves away
@@:     sar     ax, 1                           ;from zero.
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney
        cmp     ax, [ebx].clip_y0               ;check ney is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

ym_ex_lt_x0:

; The line is Y major and the end point lies directly to the left of the
; clip rectangle

        mov     ax, [ebx].clip_x0               ;nex = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaY                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaX                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     ax, 0                           ;now convert back to integer
        jge     short @F                        ;if ney-ey is -ve then
        dec     ax                              ; round up and subtract one
@@:     sar     ax, 1                           ;if ney-ey is +ve then
        js      short @F                        ; just round up
        rcl     dx, 1
        cmp     dx, 0
        je      short @F
        inc     ax
@@:     add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney
        cmp     ax, [ebx].clip_y0               ;check ney is within clip bounds ;**
        jl      completely_clipped              ;if not then line lies
        cmp     ax, [ebx].clip_y1               ;completely outside clip ;**
        jg      completely_clipped              ;rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax             ;store ney:nex in _SPad
        jmp     clipped_ok                      ;finished clipping

xm_ex_lt_x0_ey_gt_y1:

; The line is X major and the end point lies below and to the left of the
; clip rectangle

; First try clipping to x1

        mov     ax, [ebx].clip_x0               ;nex = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax ,1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        sar     ax, 1                           ;now convert back to integer
                                                ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(ney-ey is always -ve)
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
                                        ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x0 resulted in ney > y1 so now try clipping to y1
        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        inc     ax                              ;add a half
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nex-ex is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsEndPoint.x          ;ax=ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x1       ;check nex is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

ym_ex_lt_x0_ey_gt_y1:

; The line is Y major and the end point lies below and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nex = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax ,1                           ;shift binary point
        dec     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        dec     ax                              ;now convert back to integer
        sar     ax, 1                           ;ney-ey is always -ve so round
                                                ;up and subtract one
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y1       ;if has not reach nearest edge of clip ;**
        jg      short @F                ;rectangle (y1) then try clipping to y1
        cmp     ax, [ebx].clip_y0       ;if has passed opposite edge of clip ;**
        jl      completely_clipped      ;rectangle (y0) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x0 resulted in ney > y1 so now try clipping to y1

        mov     ax, [ebx].clip_y1               ;ney = y1               ;**
        swap    eax
        mov     ax, [ebx].clip_y1               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaX                    ;dx:ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        cmp     dx, 0                           ;now convert back to integer
        je      short @F                        ;by rounding to the nearest
        inc     ax                              ;integer, and rounding halves
@@:                                             ;towards zero.
        sar     ax, 1                           ;(nsy-sy is always +ve)
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x1       ;check nex is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

xm_ex_lt_x0_ey_lt_y0:

; The line is X major and the end point lies above and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nex = x0               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        inc     ax                              ;now convert back to integer
        sar     ax, 1                           ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(ney-ey is always +ve)
        add     ax, _SPad.ptsEndPoint.y          ;ax = ex + (nex-ex)*dY/dX = ney

                                        ;now check ney is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      completely_clipped      ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping
@@:

; Clipping to x0 resulted in ney < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;subtract a half
        imul    _SPad.sDeltaX                    ;ax = (ney-ey) * dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;nex-ex is always -ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex

        cmp     ax, [ebx].clip_x1       ;now check nex is within clip bounds ;**
        jg      completely_clipped      ;no need to check against x0 since we
                                        ;have already tried clipping against x0

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     clipped_ok              ;finished clipping

ym_ex_lt_x0_ey_lt_y0:

; The line is Y major and the end point lies above and to the left of the
; clip rectangle

; First try clipping against x0

        mov     ax, [ebx].clip_x0               ;nex = x1               ;**
        swap    eax
        mov     ax, [ebx].clip_x0               ;ax = nex               ;**
        sub     ax, _SPad.ptsEndPoint.x          ;ax = nex - ex
        shl     ax, 1                           ;shift binary point
        dec     ax                              ;add a half
        imul    _SPad.sDeltaY                    ;dx:ax = (nex-ex) * dY
        idiv    _SPad.sDeltaX                    ;ax = (nex-ex)*dY / dX = ney-ey
        sar     ax, 1                           ;now convert back to integer
        rcl     dx, 1                           ;ney-ey is always +ve so just
        cmp     dx, 0                           ;round up
        je      short @F
        inc     ax
@@:
        add     ax, _SPad.ptsEndPoint.y          ;ax = ey + (nex-ex)*dY/dX = ney

                                        ;check ney is within clip bounds
        cmp     ax, [ebx].clip_y0       ;if has not reach nearest edge of clip ;**
        jl      short @F                ;rectangle (y0) then try clipping to y0
        cmp     ax, [ebx].clip_y1       ;if has passed opposite edge of clip ;**
        jg      short completely_clipped  ;rectangle (y1) then line lies
                                        ;completely outside clip rectangle
        swap    eax
        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     short clipped_ok        ;finished clipping
@@:

; Clipping to x0 resulted in ney < y0 so now try clipping to y0

        mov     ax, [ebx].clip_y0               ;ney = y0               ;**
        swap    eax
        mov     ax, [ebx].clip_y0               ;ax = ney               ;**
        sub     ax, _SPad.ptsEndPoint.y          ;ax = ney - ey
        shl     ax, 1
        imul    _SPad.sDeltaX                    ;ax = (ney-ey)* dX
        idiv    _SPad.sDeltaY                    ;ax = (ney-ey)*dX / dY = nex-ex
        inc     ax                              ;now convert back to integer
        sar     ax, 1                           ;by rounding to the nearest
                                                ;integer, rounding halves away
                                                ;from zero.
                                                ;(nex-ex is always +ve)
        add     ax, _SPad.ptsEndPoint.x          ;ax = ex + (ney-ey)*dX/dY = nex


        cmp     ax, [ebx].clip_x1       ;check nex is within clip bounds ;**
        jg      short completely_clipped  ;no need to check against x1 since we
                                        ;have already tried clipping against x1

        mov     _SPad.ptsNewEnd, eax     ;store ney:nex in _SPad
        jmp     short clipped_ok        ;finished clipping

clip_rectangle_error:
        int     3           ; firewall

completely_clipped:
        ; we have been using esi as a working register, so we must reload
        ; it before we have access to the XGA memory mapped registers
        ifndef  _8514
        movxga  pXGARegs
        else
        movxga  p8514Regs
        endif
        jmp     line_completely_clipped

IFDEF HARD_DRAW
; this is just here for debugging !
public clipped_ok
ENDIF

no_clipping_needed:
        mov     _SPad.fsClipStatus, dx     ;set fsClipStatus to zero (used later to
                                        ;help decide whether to draw first pel)
clipped_ok:
        ; we have been using esi as a working register, so we must reload
        ; it before we have access to the XGA memory mapped registers
        ifndef  _8514
        movxga  pXGARegs
        else
        movxga  p8514Regs
        endif


; Now we have clipped the start and end points we need to find the new
; clipped values for the Bresenham error term, the line length,  and
; the pattern position

; Find deltaMajor and set up dim1 (line length)
        test    byte ptr _SPad.ulPixOpWork, OCT_DZ_BIT
        jne     short @F                        ; see if line is X or Y major
        mov     dx, _SPad.ptsNewStart.x         ; line is X major
        mov     cx, dx                          ; cx = dx = nsx
        sub     cx, _SPad.ptsStartPoint.x       ; cx = nsx-sx = +/- deltaMajor
        sub     dx, _SPad.ptsNewEnd.x           ; dx = nsx-nex = +/- dim1
        jmp     short deltaMajor_set
@@:
        mov     dx, _SPad.ptsNewStart.y         ; line is Y major
        mov     cx, dx                          ; cx = dx = nsy
        sub     cx, _SPad.ptsStartPoint.y       ; cx = nsy-sy = +/- deltaMajor
        sub     dx, _SPad.ptsNewEnd.y           ; dx = nsy-ney = +/- dim1

deltaMajor_set:
        jge     short @F                ; make sure that dim1 is positive
        neg     dx
@@:
        cmp     cx, 0                   ; make sure that deltaMajor is positive
        jge     short @F
        neg     cx
@@:
; Set up new pattern position
        mov     ax, _SPad.usPatPosTmp      ;ax = unclipped pattern position
        add     ax, cx          ;ax = unclipped pattern position + deltaMajor
        and     ax, 00001111b   ;pattern size is always 16 bits

        waitshort                       ; wait for hardware
        memregwrite  patt_map_x, ax     ; set new pattern position
        memregwrite  dim1, dx           ; set line length
IFDEF _8514
        mov     _SPad.usMajorAxis,dx
ENDIF

; Set up new error term (biased to compensate for clipping)
;acw mov added and memregwrite moved from top of loop to here
        mov     ax, _SPad.sK2            ;ax = sK2
        memregwrite  bres_k2, ax        ; Set sK2

        mov     ax, _SPad.sK1            ;ax = sK1
;acw memregwrite moved from top of loop to here
        memregwrite  bres_k1, ax        ; Set sK1

;acw enhancement #2, if sK1 is ZERO, line is horz or vert! -> ET=ETstart
        cmp     ax, 0                   ; test for rectilinear
        jne     short @F
        mov     dx, _SPad.sETstart               ; dx = ETstart
        jmp     short ET_write

@@:
        imul    cx                      ;dx:ax = sK1*deltaMajor
        movsx   ecx, _SPad.sK2           ;ecx = sK2;
        sub     ax, cx                  ;dx:ax = -sK2 + sK1*deltaMajor
        sbb     dx, 0                   ;(sK2 is -ve or zero, so we need to
        swap    ecx                     ; adjust dx accordingly not forgetting
        sub     dx, cx                  ; that swap macro corrupts carry flag)
        movsx   ecx, _SPad.sETstart      ;dx:ax = sETstart - sK2+sK1*deltaMajor
        add     ax, cx                  ;(sETstart can be -ve or +ve so
        adc     dx, 0                   ; must adjust dx accordingly, not
        swap    ecx                     ; forgetting that swap macro corrupts
        add     dx, cx                  ; the carry flag)
        idiv    _SPad.sK1_K2     ;dx = (sETstart-sK2+sK1*deltaMajor) mod (sK1-sK2)
        add     dx, _SPad.sK2    ;dx = sK2 + (sETstart-sK2+sK1*deltaMajor)mod(sK1-sK2)
ET_write:
        memregwrite  bres_error, dx     ; Set ET
IFDEF _8514
        mov     _SPad.usErrorTerm,dx
ENDIF

; Set up the clipped start position
        mov eax, _SPad.ptsNewStart
        memregwrite dest_map, eax

; Draw the line
        mov     eax, _SPad.ulPixOpWork
        cmp     byte ptr _SPad.fsClipStatus[1], 0  ;see if start point clipped
        je      short @F
        and     al, 11101111b    ;start point clipped so set draw first pel mode
@@:     memregwrite     pixel_op, eax

IFDEF HARD_DRAW
IFDEF _8514

        push    ebx
        push    ecx
        push    edi
        push    esi

        WaitQIdle
        WaitQ   8

        mov     eax,_SPad.ptsNewStart
        outwQ   X0,ax
        swap    eax
        outwQ   Y0,ax

        outwQ   LX,_SPad.usMajorAxis
        outwQ   K1,_SPad.sK1
        outwQ   K2,_SPad.sK2
        outwQ   ERR_TERM,_SPad.usErrorTerm

        ; edb fix for defect 76602.  8514 and S3 need the command register
        ; last pixel bit set for polylines in FM_INVERT (!DST) mode
        outwQ   MODE,(MODE_2DECODE+MD_UP_FALSE)
        mov     eax,(CMD_C_VECTOR+CMD_MA_ACCESS+CMD_CD_ANGLED+CMD_LP_NULL+CMD_RW_W)
        or      ax,_SPad.usCmdFlags

        ; edb fix for defect 071632
        ; draw entire line (i.e., last pel) if end point is clipped
        cmp     byte ptr _SPad.fsClipStatus[0], 0
        je      short @F
        and     al, NOT CMD_LP_NULL
@@:
        outwQ   CMD_FLAGS,ax

        pop     esi
        pop     edi
        pop     ecx
        pop     ebx


ENDIF ;_8514

ELSE ;SOFT_DRAW

IFDEF _8514

        mov             eax,_SPad.ptsNewStart
        memregwrite     dest_map,eax

        mov             ax,_SPad.sK1
        memregwrite     sr_K1,ax

        mov             ax,_SPad.sK2
        memregwrite     sr_K2,ax

        mov             ax,_SPad.usErrorTerm
        memregwrite     Err_Term,ax

        mov             ax,_SPad.usMajorAxis
        memregwrite     dim1,ax

ENDIF ;_8514

        saveregs
        call    _eddf_MESS
        restoreregs
ENDIF

line_completely_clipped:

; More than one clip rectangle ?

        add     ebx, len_clip_rect                                      ;**
        dec     _SPad.usClipCount
        jne     clip_loop      ;jump to beginning of the per clip rectangle loop

;acw  tried inserting a delay here if looping back for additional
;     segments since initial 0-wait poll might be interfering with
;     the engine.  It degraded the min time by the amount of the poll loop,
;     and failed to improve longer line times AT ALL.  No change.
;     FYI, on a model 90 486-33 an ECX=100 loopz gave 210ns delay...
;
;     Hmmm, should really retry this with the delay loop down below, this is
;     only for multiple clip rects...

vloopend:
; By now the first vector will have been drawn into all the clip rectangles
; If this is a polyline then we want future vectors to be drawn with
; first pel null.  If it is a disjoint line then we want future vectors
; to be drawn with first pel on.

; Get the .ulPixOpWork into eax.
        mov     eax, _SPad.ulPixOpWork

; First clear draw mode and direction octant fields
        and     al, 11001000b

; Check to see if there are more lines to be drawn
        dec     _SPad.ulPointCount
        jz      short exit

; Is this a disjoint line
        mov     bx, _AIxfer.usCallType
        cmp     bx, POLYLINE_CALL
        jne     short disjoint_line

; This is a polyline so set draw mode to omit first pel
        or      al, 00010000b
        mov     _SPad.ulPixOpWork, eax

; Copy usPatEnd to _SPad.usPatPosTmp (initial pattern position for unclipped line)
        mov     ax, _AIxfer.usPatEnd
        mov     _SPad.usPatPosTmp, ax

; Copy old ptsEndPoint to ptsStartPoint
        mov     eax, _SPad.ptsEndPoint
        mov     _SPad.ptsStartPoint, eax

; Loop back to draw next line
        jmp     vloop

disjoint_line:
; This is a disjoint line so draw all pels
        mov     _SPad.ulPixOpWork, eax

; Set _SPad.usPatPosTmp to zero (initial pattern position for unclipped line)
; and reset usPatEnd to zero (disjoint lines always reset pattern position)
        mov     ax, 0
        mov     _SPad.usPatPosTmp, ax
        mov     _AIxfer.usPatEnd, ax

; Set the ptsStartPoint to be the next point in the list and advance
; the pointer to the next coordinates

        mov     ax, _AIxfer.ptsDevOffsets.y
        sub     ax, [edi+4]                                             ;**

        swap    eax

        mov     ax, _AIxfer.ptsDevOffsets.x
        add     ax, [edi]                                               ;**

        mov     _SPad.ptsStartPoint, eax
        add     edi, 8                                                  ;**

; Loop back to draw next line
        jmp     vloop


softdraw:
; Right, this spells T.r.o.u.b.l.e.
; We are now going to call a 'C' function in the driver which emulates
; the hardware line draw.  To make things simple we will save some
; important registers here.
; These registers should be saved: xga, edi, esi, ecx, ebx
; For safety we save all the necessary registers
        pushxga
        push    edi
        push    esi
        push    ebx
        push    ecx

        call    _eddl_BresenhamOne

        pop     ecx
        pop     ebx
        pop     esi
        pop     edi
        popxga
        jmp     vloopend

exit:
        mov     ax, _AIxfer.usPatEnd
        inc     ax
        and     ax, 0Fh
        mov     _AIxfer.usPatEnd, ax
        popxga
        pop     ebx
        pop     esi
        pop     edi
        retn

PMLines endp

;***********************************************************************
;
; Function: PMPlotStep
;
; Called by: eddl_BresenhamOne in eddlpoly.c
;
; This outputs all the plot step orders which have been accumulated
; by the software line drawing.  The plot step orders are output into all
; current clip rectangles.
; Because this function is called from the software emulation of the line
; drawing it must be careful what it alters and the set up it leaves as
; the software emulation is supposed to be transparent to the calling
; function.
;
; 386 only code
;***********************************************************************

        align   4
PMPlotStep      proc

; For safety (eddl_BresenhamOne uses these)
        push    edi
        push    esi
        push    ebx
        pushxga

; Get the base address of the hardware registers
        ifndef  _8514
        movxga  pXGARegs
        else
        movxga  p8514Regs
        endif

; Set up the correct pixel operation
        mov     eax, _SPad.ulPixOpWork

; Change pixel op to omit last pel and draw and step write mode
; Clear current step function and drawing mode
        and     eax, 11110000111111111111111111001111b

; Set draw and step write and omit last pel modes
        or      eax, 00000100000000000000000000100000b

; Get the offset to the first plot step order
        mov     edi, _AIxfer.pPlotStepCodes                              ;**

; Wait for the hardware before using it
        waitshort

; Set the pixel op register.  This _does not_ initiate a draw and step
; operation.  It only needs to be done once.
        memregwrite  pixel_op, eax

; Reset the start point
        mov     eax, _SPad.ptsStartPoint
        memregwrite     dest_map, eax

; Do the drawing for all plot step orders.  Alter this if plot/step
; cache is enlarged
        irp     offset,<0, 4, 8, 12>
        mov     eax, [edi+offset]                                       ;**
        memregwrite     dir_steps, eax

IFDEF  ;SOFT_DRAW
        saveregs
        call    _eddf_MESS
        restoreregs
ENDIF

        waitshort                              ;maybe a waitmedium?
        endm

IFDEF HARD_DRAW
IFDEF _8514

        push    edx
        push    ecx
        push    eax

        WaitQIdle

        ifndef  S3
        setclip 0,0,SCR_8514_WIDTH,SCR_8514_HEIGHT
        else
        setclip 0,0,SCR_S3_WIDTH,SCR_S3_HEIGHT
        endif
        WaitQIdle
        WaitQ   1
        outwQ   MODE,MODE_2DECODE

        mov     cx,16
ShortLineLoop1:
        WaitQ   2
        mov     bl, BYTE PTR [edi]
        inc     edi
        movzx   ax,bl
        and     ax,0fh
        outwQ   LX,ax
        movzx   ax,bl
        and     ax,0f0h
        or      ax,(CMD_C_VECTOR+CMD_WORD+CMD_FV_FIX+CMD_CD_45+CMD_LP_NULL+CMD_RW_W)
        outwQ   CMD_FLAGS,ax
        loop    ShortLineLoop1

        pop     eax
        pop     ecx
        pop     edx

ENDIF   ;_8514
ENDIF   ;HARD_DRAW

; Update the start position storage for next time
        memregread      eax, dest_map
        mov     _SPad.ptsStartPoint, eax

IFDEF HARD_DRAW
IFDEF _8514
        ; When in hard draw mode for the 8514, we must
        ; read the current position from X0,Y0 and
        ; overwrite the data just loaded above into
        ; _SPad.ptsStartPoint.
        push    edx

        ; Wait for card to become idle.
        WaitQIdle

        ; Read in the x,y coordinates.
        mov     dx,Y0
        in      ax,dx
        swap    eax
        mov     dx,X0
        in      ax,dx
        mov     _SPad.ptsStartPoint, eax

        pop     edx

        ; Restore the pointer to our plot step codes.
        mov     edi, _AIxfer.pPlotStepCodes                              ;**

ENDIF
ENDIF

; Clean up the plot step storage
        sub     eax, eax
        irp     offset,<0, 4, 8, 12>
        mov     [edi+offset], eax                                       ;**
        endm

@@:
        popxga
        pop     ebx
        pop     esi
        pop     edi
        retn

PMPlotStep      endp

IFDEF HARD_DRAW
IFDEF _8514

cProc DrawPel_8514,     <PUBLIC>, <edi,ebx,ecx>
        parmD   xPos
        parmD   yPos
cBegin

ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      @f
        mov     eax,xPos
        mov     dx,3
        mul     dx
        mov     xPos,ax
        mov     ebx,_Shadow8514Regs.Color_1
        swap    ebx
        mov     ecx,_Shadow8514Regs.Color_0
        swap    ecx
        mov     edi,3
next_pel_byte:
        WaitQ   2
        movzx   ax,bl
        outwQ   COLOR_1,ax
        movzx   ax,cl
        outwQ   COLOR_0,ax
@@:
endif
        WaitQIdle
        WaitQ   6

        mov     eax,xPos
        outwQ   X0,ax

        mov     eax,yPos
        outwQ   Y0,ax

        outwQ   LX,0
        outwQ   LY,0

        outwQ   MODE,MODE_2DECODE
        outwQ   CMD_FLAGS,(CMD_C_HRECT+CMD_MA_ACCESS+CMD_RW_W)

ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      @f
        rol     ebx,8
        rol     ecx,8
        inc     xPos
        dec     di
        jnz     next_pel_byte
@@:
endif

cEnd

ENDIF ;_8514
ENDIF ;HARD_DRAW

_TEXT            ends

END
