;*DDK*************************************************************************/
;
; COPYRIGHT    Copyright (C) 1995 IBM Corporation
;
;    The following IBM OS/2 WARP source code is provided to you solely for
;    the purpose of assisting you in your development of OS/2 WARP device
;    drivers. You may use this code in accordance with the IBM License
;    Agreement provided in the IBM Device Driver Source Kit for OS/2. This
;    Copyright statement may not be removed.;
;*****************************************************************************/
;***********************************************************************
;
;   Module          = EDDHGCHS
;
;   Description     = Assembler text drawing code
;
;   Function        = Text interface to Expressway hardware
;
;
; CHANGE ACTIVITY =
;   DATE      FLAG        APAR   CHANGE DESCRIPTION
;   --------  ----------  -----  ------------------------------------
;   mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
;   04/10/93              65290  Use unclipped glyph height when converting
;                                from font file to bitmap format.
;***********************************************************************
;***********************************************************************
;  CHANGE ACTIVITY:
;
;  FLAG RLSE  DATE  ORIGIN              COMMENTS
;  ---- ---- ------ ------  --------------------------------------------
;  @001 0205 930309 Change  Modified DrawText so it would not attempt to
;                   Team    to draw the extra background (required when
;                           drawing bold text) if it is not located on the
;                           screen. (Defect 63056 - BZ)
;
;  @002 0205 930309 Change  Modified DrawText so it would draw the extra
;                   Team    background required if the actual character
;                           width was less than the user supplied increment
;                           vector char width. (Defect 63056 - BZ)
;
;  @003 0205 930312 Change  Fixed a     in DrawText so the extra background
;                   Team    would get drawn in the case of the extra
;                           background falling on the border of the
;                           right clip edge. (Defect 63056 - BZ)
;
;***********************************************************************

.386

INCL_GPIBITMAPS EQU     1
INCL_FONTFILEFORMAT     equ     1       ; to include pmfont.inc
include os2.inc
include eddinclt.inc

include eddhmacr.inc
include eddhcone.inc
include eddhtype.inc

include eddfcone.inc
include eddfmacr.inc
include eddftype.inc

?DF     equ     1       ; we do not want _TEXT segment defined by cmacros.
include cmacros.inc

;***********************************************************************
; Cope with Microsoft compiler conventions
;***********************************************************************
ifndef   _8514
pXGARegs         equ <_pXGARegs>
ShadowXGARegs    equ <_ShadowXGARegs>
else
include 8514.inc
include cacheman.inc
p8514Regs        equ <_p8514Regs>
Shadow8514Regs   equ <_Shadow8514Regs>
p8514FontCache   equ <_p8514FontCache>
pCacheMap        equ <_pCacheMap>
pPhunkVirt       equ <_pPhunkVirt>
endif
AIxfer           equ <_AIxfer>
pbHWPollRegister equ <_pbHWPollRegister>
eddf_Mess        equ <_eddf_Mess>

;***********************************************************************
; handle the soft draw/hard draw differences
;***********************************************************************
IFDEF HARD_DRAW
bitmap_address  equ     phys_addr
extrn _pCurCacheBasePhy :dword
pCacheStart     equ     <_pCurCacheBasePhy>
ENDIF ; HARD_DRAW

IFDEF SOFT_DRAW
bitmap_address  equ     virt_addr
extrn _pSysCacheStart   :dword
pCacheStart     equ     <_pSysCacheStart>
extrn _eddf_MESS        :proc
ENDIF ; SOFT_DRAW

extrn AIxfer            :dword
ifndef   _8514
extrn pXGARegs          :dword
extrn ShadowXGARegs     :dword
else
extrn p8514Regs         :dword
extrn Shadow8514Regs    :dword
extrn p8514FontCache    :dword
extrn pPhunkVirt        :dword
extrn pCacheMap         :dword
extrn _Screen8514VisHeight :dword
ifdef BPP24
extrn _Copy24MonoToVRAM    :NEAR
endif ;BPP24
extrn _DDT                 :byte
endif
extrn pbHWPollRegister  :dword


;***********************************************************************
; Define the parameter block (AIxfer) for this file.
;***********************************************************************
TEXTPB                  struc
pbmhDest                dd      ?
cDestClipRects          dw      ?
pNextDestClipRect       dd      ?
ptsStartCoord           dd      ?
cChars                  dw      ?
pCodePoints             dd      ?
pFocaFnt                dd      ?
ifndef  BPP24
usBColor                dw      ?
usFColor                dw      ?
else
usBColor                dd      ?
usFColor                dd      ?
endif
usTheCharSpacing        dw      ?
pFontCacheInfo          dd      ?
pCharCache              dd      ?
bCharWidth              db      ?
bCharHeight             db      ?
pCharMask               dd      ?
IFDEF _8514                             ; For the 8514, we add this
usCacheInUse            dw      ?       ; value to indicate which cache
ENDIF                                   ; (0-7) to use.
bfgmix                  db      ?       ; these fields are not (yet)
bbgmix                  db      ?       ;  in the 'C' version

IFDEF DBCS                                                  ;YOJN
psCacheIndex            dd      ?      ; should be set before calling
ENDIF                                                       ;YOJN
TEXTPB                  ends

;***********************************************************************
; Set up structure for FontCacheInfo (text caching)
;***********************************************************************
MAX_NUM_CODEPOINTS      equ   256

IFDEF DBCS                                                  ;YOJN
MAX_NUM_DBCODEPOINTS    equ 256        ; limit of DBCS char in cache
MAX_TOTAL_CODEPOINTS    equ MAX_NUM_CODEPOINTS + MAX_NUM_DBCODEPOINTS

FcCharDef               struc          ; 
fccd_aspace             DW      ?      ; A space for ABC font
fccd_bspace             DW      ?      ; B
fccd_cspace             DW      ?      ; C
FcCharDef               ends           ; 

fccd_width              equ     fccd_aspace
                                       ; width field for non-ABC font
ENDIF                                                       ;YOJN

FontCacheInfo           struc
fci_fmFontMetrics         DB      SIZE FOCAMETRICS DUP (?)
fci_usHashMetrics         DW      (?)
fci_usUsageCount          DW      (?)
fci_usFontId              DW      (?)
fci_usCodePage            DW      (?)

IFNDEF DBCS                                                 ;YOJN
fci_aulCachedCharOffset   DD      MAX_NUM_CODEPOINTS DUP (?)
ifdef FULL_ADDRESS
fci_apCharDef             DD      MAX_NUM_CODEPOINTS DUP (?)
else ; ndef FULL_ADDRESS
fci_apCharDef             DW      MAX_NUM_CODEPOINTS DUP (?)
endif ; ndef FULL_ADDRESS
ifdef  _8514                                                
fci_ausFontPlaneOffset    DW      MAX_NUM_CODEPOINTS DUP (?)
endif 
ELSE                                                        ;YOJN
fci_aulCachedCharOffset   DD      MAX_TOTAL_CODEPOINTS DUP (?)

ifdef BPP24                                                 
ifdef FULL_ADDRESS
fci_apCharDef             DD      MAX_TOTAL_CODEPOINTS DUP (?)
else ; ndef FULL_ADDRESS
fci_apCharDef             DW      MAX_TOTAL_CODEPOINTS DUP (?)
endif ; ndef FULL_ADDRESS
endif 

fci_aCharDef    DB       (type FcCharDef)*MAX_TOTAL_CODEPOINTS DUP (?)
fci_aCharOffset           DD      MAX_TOTAL_CODEPOINTS DUP (?)  
ifdef  _8514                                                
fci_ausFontPlaneOffset    DW      MAX_TOTAL_CODEPOINTS DUP (?)  
endif 
fci_usDummy               DW      ?    ; fields for DBCS cache mgt
fci_usDBGITableNext       DW      ?    ; 
fci_ausDBGIndexTable      DW      MAX_NUM_DBCODEPOINTS+1 DUP (?)
fci_usFirstGlyph          DW      ?
fci_usLastGlyphOffset     DW      ?
fci_usDefGlyphOffset      DW      ?
ENDIF                                                       ;YOJN
FontCacheInfo           ends

non_ABC_CharDef         struc
non_abc_image_offset    dd      ?
char_width              dw      ?
non_ABC_CharDef         ends

ABC_CharDef             struc
abc_image_offset        dd      ?
a_space                 dw      ?
b_space                 dw      ?
c_space                 dw      ?
ABC_CharDef             ends

clip_xs                 struc
right                   dw      ?
left                    dw      ?
clip_xs                 ends

_DATA           segment use32 dword public 'DATA'
_DATA           ends

_TEXT           segment use32 dword public 'CODE'
                assume  cs:FLAT, ds:FLAT, es:FLAT

;***********************************************************************
; Function: DrawText
;
; Called by: eddt_charstringpos in eddngchs.c
;
; Purpose: Draw characters at given position
;
;***********************************************************************
IFDEF HARD_DRAW
cProc eddh_DrawText,<PUBLIC>, <edi, esi, ebx>
ENDIf ; HARD_DRAW

IFDEF SOFT_DRAW
cProc eddf_DrawText,<PUBLIC>, <edi, esi, ebx>
ENDIf ; SOFT_DRAW

        localV  ptsCharCoord, %(size POINTS)
        localV  ptsCharHWsize, %(size POINTS)
        localW  usClipsToDo
        localW  usPelsToClip
        localD  pNextClipRect
        localV  xsClips, %(size clip_xs)
        localD  pGlyphImage
IFNDEF DBCS                                                             ;YOJN
        localD  pCodePoint
ELSE ;DBCS                                                              ;YOJN
        localD  pCacheIndex                                             ;YOJN
ENDIF ;DBCS                                                             ;YOJN
        localD  flastchar
        localW  usExtraSpacing                               ; @002

IFDEF _8514
        ;************************************************************
        ; 8514 NOTES
        ; 
        ; For the 8514, we allow the XGA code to set the Shadow regs
        ; as usual, even though the 8514 doesn't really have most of
        ; those registers.  This allows the MESS code to access the
        ; original data structures when drawing text to a memory
        ; bitmap instead of the screen.
        ; 
        ; However, while we set the shadow regs, we also save important
        ; values in these locals from time to time so that when we
        ; set the 8514 regs, we don't have to recalculate their values.
        ;************************************************************
        localW  bltheight
        localW  bltwidth
        localD  dest_xy
        localW  usClipFactor
        localW  usHorizOffset
        localW  usTotalGlyphHeight              ;          
ENDIF

cBegin
        pushxga

IFDEF _8514
        mov     usHorizOffset,0
ENDIF

;********************************************************************
; We now need to change the mixes within this routine, so save the
; ones the calling code thinks it has set so we can restore them later
;********************************************************************
ifndef  _8514
        mov     al, ShadowXGARegs.fg_mix
        mov     AIxfer.bfgmix, al
        mov     al, ShadowXGARegs.bg_mix
        mov     AIxfer.bbgmix, al
else
        mov     ax, Shadow8514Regs.Function_1
        mov     AIxfer.bfgmix, al
        mov     ax, Shadow8514Regs.Function_0
        mov     AIxfer.bbgmix, al
endif

;********************************************************************
; Set up our local copy of the CodePoints pointer.
;********************************************************************
IFNDEF DBCS                                                             ;YOJN
        mov     eax, AIxfer.pCodePoints
        mov     pCodePoint, eax
ELSE ;DBCS                                                              ;YOJN
; Now we use cache index array instead of string                        ;YOJN
        mov     eax, AIxfer.psCacheIndex                                ;YOJN
        mov     pCacheIndex, eax                                        ;YOJN
ENDIF ;DBCS                                                             ;YOJN

;********************************************************************
; Get the base address of the hardware registers
;********************************************************************
ifndef  _8514
        movxga  pXGARegs
else
        movxga  p8514Regs
endif

;********************************************************************
; Use ebx as a pointer to the destination bitmap header.
;********************************************************************
        mov     ebx, AIxfer.pbmhDest

;********************************************************************
; Set up our local copy of the start coordinates.
;********************************************************************
        mov     eax, AIxfer.ptsStartCoord
        mov     ptsCharCoord, eax

;********************************************************************
; Set up our count of the number of clip regions still to go.
;********************************************************************
        mov     ax, AIxfer.cDestClipRects
        mov     usClipsToDo, ax

;********************************************************************
; Set up our pointer to the current clip region and keep a copy in edi.
;********************************************************************
        mov     edi, AIxfer.pNextDestClipRect
        mov     pNextClipRect, edi

;********************************************************************
; Set up our copy of the left and right clip edges.
;********************************************************************
        mov     ax, [edi].clip_x0
        shl     eax, 16
        mov     ax, [edi].clip_x1
        mov     xsClips, eax

;********************************************************************
; We have got to the point where we need to set up the hardware
;********************************************************************
        waitshort

;********************************************************************
; Set up the destination bitmap in the hardware.
; First, select mapA.
;********************************************************************
        memregwrite     pi_map_index_A, SEL_PIX_MAP_A

;********************************************************************
; Write destination bitmap address to hardware mapA base.
;********************************************************************
        mov     eax, [ebx].bitmap_address
        memregwrite     pi_map_base_ptr_A, eax

;********************************************************************
; Write mapA width and height from bitmap header.
;********************************************************************
        mov     eax, dword ptr [ebx].hw_width
        memregwrite     pi_map_size_A, eax

;********************************************************************
; Write mapA format from bitmap header.
;********************************************************************
        mov     al, byte ptr [ebx].hw_format
        memregwrite     pi_map_format_A, al

;********************************************************************
; Set up the source bitmap in the hardware: this is the glyph images
; first, select mapC.
;********************************************************************
        memregwrite     pi_map_index_C, SEL_PIX_MAP_C

;********************************************************************
; The source format is always one bit per pel.
;********************************************************************
        memregwrite     pi_map_format_C, MOTOROLA+ONE_BPP

;********************************************************************
; The bitmap width and height are the character width and height less 1.
; At this point we can only set the height because the width can vary for
; each character.
;********************************************************************
        mov     ebx, AIxfer.pFocaFnt
        mov     ax, [ebx].ff_fdDefinitions.fdh_yCellHeight
        dec     ax
        memregwrite     pi_map_height_C, ax

;********************************************************************
; Keep a copy of the bitmap size since we will need it later for
; calculating the operating dimension.
;********************************************************************
        mov     ptsCharHWsize.y, ax
IFDEF _8514
        mov     usTotalGlyphHeight,ax                   ;          
ENDIF


output_to_cliprect:


;********************************************************************
; Since string are always left to right horizontal, clipping of the y
; component can be done outside the per character loop as this will
; affect the blt y-dimension by the same amount for each character.
; The blt direction is bottom (of bitmap) to top. Check if the
; bottom of the character is below the clip region first.
;
;   bitmapĿ
;    bitmap origin                                                    
;                                                                     
;          clip regionĿ clip y0     
;                                                                   
;                                                                   
;                                                                   
;                                                                   
;                                                                   
;                                                                   
;                                                                   
;                                                                   
;                   .........                                       
;                   : ***** :                                       
;                   :   *   :                                       
;          :*: clip y1     
;                    : ***** :                                        
;                    :.......:                                        
;                                                                    
;                    character start position                         
;                                                                     
;                                                                     
;   
;
;********************************************************************


;********************************************************************
; flag that we are not currently drawing the last character in the string
;********************************************************************
        mov     flastchar, FALSE

;********************************************************************
; Register usage:
;       edi holds pointer to the current clip rectangle
;********************************************************************

;********************************************************************
; Use cx as a counter for the number of characters to output.
;********************************************************************
        mov     cx, AIxfer.cChars

;********************************************************************
; First clip to the bottom of the bitmap.
;********************************************************************
        mov     ax, ptsCharHWsize.y
        mov     dx, ptsCharCoord.y
        sub     dx, [edi].clip_y1
        jle     short inside_y1_clip

;********************************************************************
; The bottom of the character is below the bottom of the clip so reduce
; the y dimension: if it goes negative there is no intersection.
;********************************************************************
        sub     ax, dx
        jc      try_next_clip

;********************************************************************
; Adjust the character hardware height from cell height less one to cell
; height less one less the number of pels below the clip region.
; This is needed since the blt direction is bottom to top of bitmap, ie.
; top to bottom of character (and screen). This value is used later to
; calculate the start position in the source - by knocking off the top
; of the character the start point is nearer the start of the character
; definition
;********************************************************************
        sub     ptsCharHWsize.y, dx

;********************************************************************
; The destination y coordinate should also be adjusted by the same amount.
;********************************************************************
        sub     ptsCharCoord.y, dx

inside_y1_clip:
;********************************************************************
; Now repeat for the clip y0 coordinate: if the top of the character is
; above this then the operating dimension must be adjusted accordingly.
;********************************************************************
        mov     dx, [edi].clip_y0
        sub     dx, ptsCharCoord.y
        add     dx, ptsCharHWsize.y
        js      short inside_y0_clip

;********************************************************************
; The top of the character is above the top of the clip region
; subtract the overshoot from the dimension: if this is negative there is
; no intersection with the clip region.
;********************************************************************
        sub     ax, dx
        jc      try_next_clip

    IFDEF _8514
        mov     usClipFactor,dx
        jmp     short write_height
    ENDIF

inside_y0_clip:

    IFDEF _8514
        mov     usClipFactor,0
    ENDIF

write_height:
;********************************************************************
; Write the height of the blt to be done to the y dimension register.
;********************************************************************
        waitshort
        memregwrite     dim2, ax
    IFDEF _8514
        mov     bltheight,ax    ; Saved the clipped height
    ENDIF


next_char:
IFNDEF DBCS                                                             ;YOJN
;********************************************************************
; Start of per character output loop.
; Get the next codepoint.
;********************************************************************
        mov     ebx, pCodePoint
        movzx   ebx, byte ptr [ebx]

;********************************************************************
; Increment code_poinys_address to point to the next codepoint.
;********************************************************************
        inc     pCodePoint
ELSE ; DBCS                                                             ;YOJN
;********************************************************************
; Now we already have array of cache index.  Using this, we need not    ;YOJN
; care about SBCS/MBCS/DBCS difference.                                 ;YOJN
;********************************************************************
        mov     ebx, pCacheIndex                                        ;YOJN
        movzx   ebx, word ptr [ebx]                                     ;YOJN
                                                                        ;YOJN
        add     pCacheIndex, 2         ; advance to next cache index    ;YOJN
ENDIF ; DBCS                                                            ;YOJN

;********************************************************************
; cx contains the number of characters to be output: shift it left 16
; to give us a spare register (value 0) to work with for now.
;********************************************************************
        shl     ecx, 16

;********************************************************************
; Get the character's cached address and CharDef address from the
; FontCacheInfo table.
;********************************************************************
        mov     edi, AIxfer.pFontCacheInfo
IFDEF _8514
ifndef DBCS                                                 
IFDEF HARD_DRAW
        ;************************************************************
        ; 8514 NOTES
        ; 
        ; When drawing a character to the screen, we simply need to
        ; get the 8514 VRAM pointer where this particular character
        ; has been cached.  We convert this value into an X,Y pair
        ; later and use it as the source origin for a 1-plane to
        ; 8-plane blt.
        ;************************************************************
        mov     eax, [edi].fci_aulCachedCharOffset[ebx*4]

ELSE ;SOFT_DRAW for the _8514
        ;************************************************************
        ; 8514 NOTES
        ; 
        ; However, when drawing a character to a software bitmap, it
        ; easier (and hopefully faster) to grab the original bitmap
        ; of the character, convert it to a format we can use, and
        ; pass it to the MESS code.  For now, just load the address
        ; off the character bitmap (in font file format) and store it
        ; in pGlyphImage.  See the Font File Definition for more info.
        ;************************************************************
        movzx   eax, [edi].fci_apCharDef[ebx*2]
        add     eax, AIxfer.pFocaFnt
        mov     eax, [eax]
        add     eax, AIxfer.pFocaFnt

ENDIF ; _8514
else 
IFDEF HARD_DRAW
        mov     eax, [edi].fci_aulCachedCharOffset[ebx*4]
;;      add     eax, pCacheStart        ; Not necessary for 8514
ELSE ;SOFT_DRAW for the _8514
        mov     eax, [edi].fci_aCharOffset[ebx*4]
ENDIF ; _8514
endif 

ELSE ; XGA
        mov     eax, [edi].fci_aulCachedCharOffset[ebx*4]
        add     eax, pCacheStart        ; Not necessary for 8514
ENDIF
;********************************************************************
; Set our copy of the character image address.
;********************************************************************
        mov     pGlyphImage, eax

;********************************************************************
; Get the address of the CharDef of the current character.
;********************************************************************
IFNDEF DBCS                                                             ;YOJN
ifdef FULL_ADDRESS
        mov     ebx, [edi].fci_apCharDef[ebx*4]
        mov     edi, AIxfer.pFocaFnt
else ; ndef FULL_ADDRESS
        movzx   ebx, [edi].fci_apCharDef[ebx*2]
        mov     edi, AIxfer.pFocaFnt
        add     ebx, edi
endif ; ndef FULL_ADDRESS
ELSE                                                                    ;YOJN
        lea     ebx, [ebx + ebx*2]     ; EBX <- EBX * 3                 ;YOJN
        lea     ebx, [edi].fci_aCharDef[ebx*2]                          ;YOJN
                                       ;so we assume 1 entry is 6 bytes ;YOJN
        .errnz  type FcCharDef - 6                                      ;YOJN
        ;-------------------------------------------------------------- ;YOJN
        ; Note that EBX points slightly different struct from CharDef   ;YOJN
        ; included in actual font data.  That is because we now keep    ;YOJN
        ; necessary chardef info in FONTCACHEINFO struct, instead of    ;YOJN
        ; keeping only pointer.                                         ;YOJN
        ;-------------------------------------------------------------- ;YOJN
        mov     edi, AIxfer.pFocaFnt                                    ;YOJN
ENDIF                                                                   ;YOJN

;********************************************************************
; See if the font is ABC spaced.
;********************************************************************
        cmp     [edi].ff_fdDefinitions.fdh_fsChardef, FONTDEFCHAR3
        je      short abc_spaced_font

        ; This font is not ABC spaced
IFNDEF DBCS                                                             ;YOJN
        mov     dx, [ebx].char_width
ELSE                                                                    ;YOJN
        mov     dx, [ebx].fccd_width                                    ;YOJN
ENDIF                                                                   ;YOJN
        jmp     short store_char_width

abc_spaced_font:
IFNDEF DBCS                                                             ;YOJN
        mov     dx, [ebx].a_space
        add     ptsCharCoord.x, dx
        mov     dx, [ebx].b_space
ELSE                                                                    ;YOJN
        mov     dx, [ebx].fccd_aspace                                   ;YOJN
        add     ptsCharCoord.x, dx                                      ;YOJN
        mov     dx, [ebx].fccd_bspace                                   ;YOJN
ENDIF                                                                   ;YOJN

store_char_width:
; @002 - Begin
; Store the extra width required to draw the additional background
; if the actual character width is less than the increment vector
; character width.  Add to that any extra spacing required by any
; special character attributes (i.e. bolding).
        movzx   ax, AIxfer.bCharWidth
        sub     ax, dx
        jle     short char_spacing_only
        add     ax, AIxfer.usTheCharSpacing
        jmp     short save_extra_spacing
char_spacing_only:
        mov     ax, AIxfer.usTheCharSpacing
save_extra_spacing:
        mov     usExtraSpacing, ax
; @002 - End

;********************************************************************
; The character cell width is now in dx.
; Decrement it to get the operation dimension we need to write to the hardware.
; For the hardware bitmap width we need the width padded to byte boundaries.
; To do the clipping/operation dimension stuff we want the true width.
;********************************************************************
        mov     ax, dx
        add     dx, 7
        and     dx, 0fff8h
        dec     dx
        swap    edx
        mov     dx, ax
        dec     dx
        mov     ptsCharHWsize.x, dx


;********************************************************************
; Now we do the clipping to the right and left edge of the current clip region.
; Clip to the right edge of the bitmap.
;********************************************************************

IFDEF _8514
        mov     usHorizOffset,0         ; Initialize for no clipping.
ENDIF
        mov     eax, xsClips
        sub     ax, dx
        sub     ax, ptsCharCoord.x
        jns     short inside_x1_clip

;********************************************************************
; Reduce the x dimension: if this goes negative there is no intersection
; and, since the text direction is always left to right, no further
; characters need be output.
;********************************************************************
        add     dx, ax
        jnc     try_next_clip

inside_x1_clip:
        shr     eax, 10h        ; clip left edge
        sub     ax, ptsCharCoord.x
        jg      short clip_left_edge

; Write the destination coordinates.
        mov     eax, ptsCharCoord
        jmp     short write_dst_coords


clip_left_edge:
;********************************************************************
; Left edge is outside the clip region so reduce the x dimension: if this
; goes negative then there is no intersection.
;********************************************************************
IFDEF _8514
        mov     usHorizOffset,ax
ENDIF
        sub     dx, ax
        jc      per_char_endloop

;********************************************************************
; Adjust the source and destination x start by the same amount.
; The source x start is the clip adjustment - save it in for now in cx.
;********************************************************************
        mov     cx, ax

;********************************************************************
; The destination x start is the original x-coord + clip adjustment.
;********************************************************************
        add     ax, ptsCharCoord.x
        shl     eax, 16
        mov     ax, ptsCharCoord.y
        ror     eax, 16

write_dst_coords:
;********************************************************************
; We can't wait any longer before updating the hardware so check it's
; free and write the destination coordinates.
;********************************************************************
        waitshort
        memregwrite     dest_map, eax

;********************************************************************
; Changes to the processing of bold simulation mean that we must write
; the foreground mix and color each time
;********************************************************************
ifndef  _8514
        mov             al, AIxfer.bfgmix
        memregwrite     fg_mix, al
        mov             ax, AIxfer.usFColor
        memregwrite     fg_colour, ax
endif

;********************************************************************
; Source is x = left clip adjustment (currently in cx),
;           y = cell height - 1
;********************************************************************
        mov     ax, ptsCharHWsize.y
        shl     eax, 16
        mov     ax, cx
        memregwrite     patt_map, eax

;********************************************************************
; Write the x dimension to the hardware.
;********************************************************************
        memregwrite     dim1, dx
IFDEF _8514
IFDEF HARD_DRAW
        ;************************************************************
        ; 8514 NOTES
        ; 
        ; When drawing text to the screen, we want only the specific
        ; width of the character with no padding on the right.
        ;************************************************************
        mov     bltwidth,dx
ENDIF
ENDIF

;********************************************************************
; Write the character width to the hardware.
;********************************************************************
        swap    edx
        memregwrite     pi_map_width_C,dx
IFDEF _8514
IFDEF SOFT_DRAW
        ;************************************************************
        ; 8514 NOTES
        ; 
        ; When drawing text to a bitmap, we want the "padded" width,
        ; rounded up to the nearest byte, so that we can properly
        ; convert the glyph image data from font file format to bitmap
        ; format.
        ;************************************************************
        mov     bltwidth,dx
ENDIF
ENDIF

;********************************************************************
; Set up the source address.
;********************************************************************
        mov     eax, pGlyphImage
        memregwrite     pi_map_base_ptr_C, eax

;********************************************************************
; Set the pixelop and kick off the blt
; -  background source: background colour
; -  foreground source: foreground colour
; -  step: PxBlt
; -  source pixel map: don't care
; -  destination pixel map: Map A
; -  pattern pixel map: Map C
; -  mask pixel map: boundary disabled
; -  drawing mode: don't care
; -  direction octant: left to right, bottom to top
;********************************************************************
        memregwrite     pixel_op, 08013002h

IFDEF HARD_DRAW
IFDEF _8514

        ;************************************************************
        ; 8514 NOTES - Drawing Text to Screen
        ; 
        ; The basic idea here is that we have 8 different font caches,
        ; one on each bit plane.  We use the usCacheInUse value in
        ; AIxfer to determine which bit plane we want to use as the
        ; source and expand this monochrome data to a location on the
        ; visible part of the screen.  During expansion, all 1's in
        ; the monochrome bitmap, use the foreground mix and color.
        ; Similarly, all 0's use the background mix and color.
        ; 
        ; Before setting 8514 regs, we save the current values is
        ; the registers that we use while accessing the 8514.
        ;************************************************************

        push    ebx
        push    ecx
        push    edx
        push    edi

        WaitQIdle
        WaitQ   8

        mov     eax,1
        mov     cx,AIxfer.usCacheInUse
        shl     eax,cl
        ifndef  S3
        rol     al,1                            ; Read only one cache
        endif
        outwQ   READ_ENABLE,ax                  ;  located on a bit plane.
        outwQ   WRITE_ENABLE,WRITE_ALL_PLANES   ; Expand to all planes.

        mov     ax,ptsCharCoord.x
        add     ax,usHorizOffset
ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      @f
        mov     dx,3
        mul     dx
@@:
endif
        outwQ   X1,ax
        mov     ax,ptsCharCoord.y
        sub     ax,bltheight                    ; Flip the Y-coord.
        outwQ   Y1,ax

        movzx   ax, AIxfer.bfgmix
        outwQ   FUNCTION_1, ax
        movzx   ax, AIxfer.bbgmix
        outwQ   FUNCTION_0, ax


ifdef   BPP24
        mov     ax,bltwidth
        test    [_DDT],USE_24BPP
        jz      @f
        inc     ax
        mov     dx,3
        mul     dx
        dec     ax
@@:
        outwQ   LX,ax
else
        outwQ   LX,bltwidth
endif
        outwQ   LY,bltheight

        WaitQ   6
        outwQ   COLOR_1, AIxfer.usFColor
        outwQ   COLOR_0, AIxfer.usBColor

        calc8514xy pGlyphImage                  ; Convert the VRAM ptr
        mov     ebx,edx                         ;  into an X,Y pair.
        add     ax,usClipFactor
        outwQ   Y0,ax
        add     bx,usHorizOffset

ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      @f
        mov     ax,bx
        mov     dx,3
        mul     dx
        mov     bx,ax
@@:
endif
        outwQ   X0,bx

        outwQ   MODE,(MODE_2DECODE+MD_PS_COPY+MD_UP_FALSE)
        outwQ   CMD_FLAGS,CMD_DRAWTEXT

        pop     edi
        pop     edx
        pop     ecx
        pop     ebx

ENDIF ;_8514
ELSE ; SOFT_DRAW
        saveregs

IFDEF _8514
ifndef DBCS                                                 
        ;************************************************************
        ; 8514 NOTES - Drawing Text to a Memory Bitmap
        ; 
        ; Instead of pulling down the character bitmap from the
        ; off-screen VRAM cache, we will use the pointer to the font
        ; file format of the glyph to create a bitmap format that
        ; can be used by the MESS code to draw text to a memory
        ; bitmap.  When converting from font file to bitmap format,
        ; we copy the first 8-bits of each row into the target bitmap,
        ; followed by the next 8-bit if present, followed by the next
        ; 8-bits,.... until the target bitmap is full.  We use the
        ; PHUNK as the area for our target bitmap.
        ; 
        ; Before setting 8514 regs, we save the current values is
        ; the registers that we use while accessing the 8514.
        ;************************************************************
        push    ebx
        push    ecx
        push    edx
        push    edi

        mov     eax,pGlyphImage         ; Get font file format - Src.
        mov     edi,pPhunkVirt          ; Get PHUNK bitmap addr. - Targ.
        movzx   ecx,bltwidth            ; Get padded width (in bits)
        inc     ecx                     ; Make one-based
        shr     ecx,3                   ; Calc. width in bytes
        mov     bltwidth,cx
outer_font_hack:

        push    ecx                     ; Save outer loop count
        mov     edx,edi                 ; Load column addr.
        inc     edi                     ; Bump to next column

        movzx   ecx,usTotalGlyphHeight  ; Fetch glyph height -           
        inc     ecx                     ; Make height one-based

inner_font_hack:                        ; Copy from src to target.
        mov     bl, byte ptr[eax]
        mov     byte ptr [edx], bl
        add     dx, bltwidth            ; Move dest. down to next row.
        inc     eax                     ; Incr. src.

        dec     ecx
        jnz     inner_font_hack         ; Loop for each row.

        pop     ecx                     ; Restore outer loop count.
        dec     ecx
        jnz     outer_font_hack         ; Loop for each column.

;********************************************************************
; Set up the source address.
;********************************************************************
        mov     eax, pPhunkVirt
        memregwrite     pi_map_base_ptr_C, eax

        pop     edi
        pop     edx
        pop     ecx
        pop     ebx
ENDIF ; _8514
endif 

        call    _eddf_MESS
        restoreregs
ENDIF ; SOFT_DRAW

;********************************************************************
; Break out if no more characters to do.
;********************************************************************
per_char_endloop:
        shr     ecx, 10h
        loop    next_char_setup

;********************************************************************
; We have drawn all the characters, but if we are doing a bold simulation
; then we must draw an extra bit of background for the last character just
; drawn
;********************************************************************
        cmp     usExtraSpacing, 0                                      ; @002
        jne     short last_char_spacing

;********************************************************************
; Output all the characters intersecting the clip region so move to the
; next clip and repeat.
;********************************************************************

try_next_clip:
;********************************************************************
; Decrement the clip counter and break out if zero.
;********************************************************************
        dec     usClipsToDo
        jnz     still_more_clips

;********************************************************************
; and exit...
;********************************************************************
        ifdef   _8514
        ifdef   HARD_DRAW
        WaitQ   1
        outwQ   READ_ENABLE,READ_ALL_PLANES
        endif
        endif

        popxga
cEnd



last_char_spacing:
        mov     flastchar, TRUE

next_char_setup:
;********************************************************************
; Adjust the x coordinate stored on the stack by the cell width.
; If the font is ABC spaced then also add on the C space.
;********************************************************************
        mov     ax, ptsCharCoord.x              ; the last char position
        add     ax, ptsCharHWsize.x             ; add the h/w width
        inc     ax                              ; make that the real width
        cmp     [edi].ff_fdDefinitions.fdh_fsChardef,FONTDEFCHAR3
        jne     short no_c_space
IFNDEF DBCS                                                             ;YOJN
        add     ax, [ebx].c_space               ; add in the c_space
ELSE                                                                    ;YOJN
        add     ax, [ebx].fccd_cspace                                   ;YOJN
ENDIF                                                                   ;YOJN
no_c_space:
        mov     ptsCharCoord.x, ax              ; save the new value

;********************************************************************
; We now have the position of the next character, assuming no bold simulations
; etc are in use.
; Now we check if there is any extra spacing required by simulations.
;********************************************************************
        cmp     usExtraSpacing, 0                                      ; @002
        je      next_char

;********************************************************************
; TheCharSpacing is non zero.
; Its a simple add if the background mix is leave-alone
;********************************************************************
        mov     bx, ax                          ; x coord
        add     bx, usExtraSpacing                                     ; @002
        mov     ptsCharCoord.x, bx              ; save the new value
        jle     check_if_last                   ; not on screen dont bother @001

IFNDEF _8514
        cmp     AIxfer.bbgmix, HWMIX_DEST
ELSE
        push    ebx
        movzx   bx,AIxfer.bbgmix
        and     bx,FUNC_ALU
        cmp     bx,FUNC_D
        pop     ebx
ENDIF
        je      check_if_last

;********************************************************************
; We need to draw a solid box the height of the current character, and
; the width of the extra space, using the backgound color and mix
; The destination map is already set up.
; We need to temporarily place the bg mix and color in the fg regs, since
; the XGA wont draw a solid block of background!
; ax holds the right of this extra box to be drawn
;********************************************************************
        dec     bx                              ; @003
        sub     bx, xsClips.right               ; do we need to clip the box
        jbe     short not_clipped

        neg     bx                              ; negate the amount
        add     bx, usExtraSpacing              ; add the width of spacing @002
        jle     check_if_last                   ; width <= is a NOP!
        jmp     short clip_at_left

not_clipped:
        mov     bx, usExtraSpacing      ; get the width of the spacing   @002

clip_at_left:
        shl     ebx, 16                 ; save width
        mov     bx, ax                  ; x start coord
        sub     bx, xsClips.left        ; negative of # of pels to clip out
        jge     short got_box_width

;********************************************************************
; need to clip at the left edge
;********************************************************************
        mov     usPelsToClip, bx
        shr     ebx, 16                 ; width of spacing
        add     bx, usPelsToClip
        jle     check_if_last           ; NOP if new width <= 0
        shl     ebx, 16
        sub     ax, usPelsToClip        ; adjust start x


got_box_width:
        shr     ebx, 16                 ; change to hw dimension
        waitshort                       ; wait for the character to finish
        memregwrite     dim1, bx        ; write the width
        shl     eax, 16                 ; save x coord
        mov     ax, ptsCharCoord.y      ; put y coord in place
        swap    eax                     ; swap x and y coords
        memregwrite     dest_map, eax   ; write the start coord
IFDEF _8514
        mov     dest_xy,eax
        mov     bltwidth,bx
ENDIF

;********************************************************************
; we now change the foreground mix and color
; these will be set back to their true values within the per char loop
;********************************************************************
IFNDEF _8514
        mov             al, AIxfer.bbgmix
        memregwrite     fg_mix, al
        mov             ax, AIxfer.usBColor
        memregwrite     fg_colour, ax
ELSE ;_8514

IFDEF HARD_DRAW

        WaitQIdle
        WaitQ   8

        mov     eax,dest_xy
ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      @f
        mov     dx,3
        mul     dx
@@:
endif
        outwQ   X1,ax
        outwQ   X0,ax
        swap    eax
        outwQ   Y1,ax
        outwQ   Y0,ax

ifdef   BPP24
        mov     ax,bltwidth
        test    [_DDT],USE_24BPP
        jz      @f
        inc     ax
        mov     dx,3
        mul     dx
        dec     ax
@@:
        outwQ   LX,ax
else
        outwQ   LX,bltwidth
endif

        outwQ   LY,bltheight            ;Defect 73600 - there is no guarantee
                                        ; (for 8514) that LY was set above @EKF
        movzx   ax, AIxfer.bbgmix
        outwQ   FUNCTION_1, ax
        outwQ   COLOR_1,AIxfer.usBColor

        WaitQ   2
        outwQ   MODE,(MODE_2DECODE+MD_PS_ONES+MD_UP_FALSE)

        ;outwQ   CMD_FLAGS,(CMD_C_HRECT+CMD_DY+CMD_DX+CMD_MA_ACCESS+CMD_RW_W)

        ; MRC @@61946 - 03/25/93
        ; Took out the Y_INC - it caused the opaque rect to be drawn in the
        ;       direction

        outwQ   CMD_FLAGS,(CMD_C_HRECT+CMD_DX+CMD_MA_ACCESS+CMD_RW_W)

ENDIF ; HARD_DRAW
ENDIF ; _8514

;********************************************************************
; Set the pixelop and kick off the blt
;********************************************************************
        mov             eax,  BACK_SRC_BACK_COL    or \
                              FORE_SRC_FORE_COL    or \
                              STEP_PXBLT           or \
                              SRC_PIX_MAP_DONTCARE or \
                              DST_PIX_MAP_A        or \
                              PAT_PIX_MAP_FORE     or \
                              MASK_PIX_MAP_OFF     or \
                              DRAW_MODE_DONTCARE   or \
                              DIR_OCTANT_LRBT
        memregwrite     pixel_op, eax

;********************************************************************
; If we have just done the spacing after the last character then we move
; on to the next clip, otherwise we loop back for the next character
;********************************************************************
check_if_last:
        cmp     flastchar, TRUE
        jne     next_char
        jmp     try_next_clip


still_more_clips:
;********************************************************************
; Restore the original character heights and destination start y on the stack.
;********************************************************************
        mov     edi, AIxfer.pFocaFnt
        mov     ax, [edi].ff_fdDefinitions.fdh_yCellHeight
        dec     ax
        mov     ptsCharHWsize.y, ax
        mov     edx, AIxfer.ptsStartCoord
        mov     ptsCharCoord, edx

IFNDEF DBCS                                                             ;YOJN
;********************************************************************
; Set pCodePoint back to the first codepoint.
;********************************************************************
        mov     eax, AIxfer.pCodePoints
        mov     pCodePoint, eax
ELSE ;DBCS                                                              ;YOJN
; Set ptr to Cache Index array to the first one.                        ;YOJN
        mov     eax, AIxfer.psCacheIndex                                ;YOJN
        mov     pCacheIndex, eax                                        ;YOJN
ENDIF ;DBCS                                                             ;YOJN

;********************************************************************
; Get a pointer to the next clip region.
;********************************************************************
        add     pNextClipRect, len_clip_rect
        mov     edi, pNextClipRect

;********************************************************************
; Update the left and right clip edges on the stack.
;********************************************************************
        mov     ax, [edi+clip_x0]
        shl     eax, 16
        mov     ax, [edi+clip_x1]
        mov     xsClips, eax

        jmp     output_to_cliprect





IFNDEF DBCS     ; be replaced by eddj_PutStringInCache                  ;YOJN
;***********************************************************************
; Function: eddh_update_cache
;
; Called by: eddt_charstringpos in eddngchs.c
;
; Purpose: To ensure that all of the codepoints in the current string have
; been cached in VRAM. When this function returns all of the codepoints will
; have valid cache entries.
;
; Errors: If the function is successful it returns 1
;         If the function encounters an error then it returns 0
;
;         The only error that can occur is if the cache becomes full
;         (and therefore is trashed) when we attempt to cache a character.
;         If this happens, then all previous characters in the string
;         will now no longer be cached. Therefore the calling routine
;         must call LocateFont to get a new CacheInfo entry and
;         then call this routine again to cache the whole string.
;
;***********************************************************************

IFDEF SOFT_DRAW
; Only assemble this routine once - we chose the SOFT_DRAW pass because
; the MATROX driver only has SOFT_DRAW

extrn   _eddt_CacheCharacter     : proc
extrn   _eddt_TidyUpAfterCaching : proc

cProc   eddh_update_cache,      <PUBLIC>, <edi, esi, ebx>
        parmD   pCodePoint
        parmD   ulCharNum
        parmD   pFntCacheInfo
        parmD   pFontDetails
        parmD   pHorzExtent

cBegin

cache_trashed:

        ; Load character count into ecx.
        mov     ecx, ulCharNum
        jecxz   short no_chars

; CHANGED

        ; Get a copy of the pFocaFont field of FontDetails into eax.
        mov     edi, pFontDetails
        mov     eax, [edi].pFocaFont

        ; Clear dx, which will store the total width of the string.
        xor     dx, dx

        ; Make edi point to the codepoint.
        mov     edi, pCodePoint

        ; Make esi point to the FontCacheInfo entry.
        mov     esi, pFntCacheInfo

check_next_char:
        ; Read the codepoint into ebx and move pointer to next codepoint.
        movzx   ebx, byte ptr [edi]
        inc     edi

        ; Now read the CachedCharOffset.
        cmp     dword ptr [esi].fci_aulCachedCharOffset.[ebx*4], 0
        je      short char_not_cached

char_cached:
        ; eax still contains the pFocaFont field.
ifdef FULL_ADDRESS
        mov     ebx, [esi].fci_apCharDef[ebx*4]
else ; ndef FULL_ADDRESS
        movzx   ebx, [esi].fci_apCharDef[ebx*2]
        add     ebx, eax
endif ; ndef FULL_ADDRESS
        cmp     [eax].ff_fdDefinitions.fdh_fsChardef, FONTDEFCHAR3
        je      short abc_font
        add     dx, [ebx].char_width
endloop:
        loop    check_next_char

no_chars:
        ; Store the width of the string
        mov     ebx, pHorzExtent
        mov     [ebx], dx

        ; Allow the caching code to tidy up
        call    _eddt_TidyUpAfterCaching

        ; Indicate success
        mov     eax, 1
        jmp     short exit_ok

        ; Calculate the width of the abc character
        ; - just sum the a, b and c spaces.
abc_font:
        add     dx, [ebx].a_space
        add     dx, [ebx].b_space
        add     dx, [ebx].c_space
        jmp     short endloop


char_not_cached:
        ; Save the registers in use
        push    eax
        push    esi
        push    edi
        push    ecx
        push    edx
        push    ebx

        ; push the FONTDETAILS
        mov     eax, pFontDetails
        push    eax

        ; Push the codepoint we want to be cached
        push    ebx

        call    _eddt_CacheCharacter
        add     esp, 8

        pop     ebx
        pop     edx
        pop     ecx
        pop     edi
        pop     esi

        ; Check whether the character was successfully cached
        or      ax, ax
        pop     eax
        jnz     char_cached

        ; Allow the caching code to tidy up
        call    _eddt_TidyUpAfterCaching

        ; Indicate failure
        xor     eax, eax

exit_ok:

cEnd

ENDIF ; SOFT_DRAW
ENDIF ; DBCS                                                            ;YOJN


;***********************************************************************
; Function: cache8514char
;
; Called by: eddt_charstringpos in eddngchs.c
;
; Purpose: To copy a monochrome image of a character onto one of
;          the eight bit planes.  Each bit plane contains a separate
;          font cache, each stored on top of one another.
;
; Errors: If the function is successful it returns a VRAM pointer
;         where the character was stored.
;
;         @RCW 11/25/92
;         Currently, there is no error handling.  We must thoroughly
;         define off-screen VRAM before we know what our font cache
;         limits are.
;***********************************************************************

IFDEF HARD_DRAW
IFDEF _8514
extrn _DVRAM        :proc

cProc   cache8514char,      <PUBLIC>, <edi, esi, ebx>
        parmD   ulBitPlane
        parmD   ulWidth     ;One-based width
        parmD   ulHeight
        parmD   pData
        parmD   pPlaneOffset
        localD  dest_x
        localD  dest_y
        localD  byteWidth
        localD  PlaneOffset

cBegin

        WaitQIdle
        WaitQ   8

        ;;***********************************************************
        ;; Ensure that we're not getting clipped out by previous calls
        ;; to the cursor code.  Adjust the bottom of the clip rectangle
        ;; to include off-screen VRAM.
        ;;***********************************************************
        ifndef S3
        outwQ   YMAX,(YMAX_2DECODE+SCR_8514_HEIGHT)
        else
        outwQ   YMAX,(YMAX_2DECODE+SCR_S3_HEIGHT)
        endif

        mov     ebx,pPlaneOffset        ; 75206
        mov     ebx,[ebx]               ; 75206
        mov     PlaneOffset,ebx         ; 75206
        cmp     ebx,1                   ; 75206
        jz      @f                      ; 75206
        xor     ebx,ebx                 ; 75206
@@:                                     ; 75206
        shl     ebx,3                   ; 75206
        ;;***********************************************************
        ;; Load the BitPlane number and use it to enable one of
        ;; the 8 bit-planes(font cache areas) for writing.
        ;;***********************************************************
        mov     eax,1
        mov     ecx,ulBitPlane
        add     ecx,ebx                 ; 75206
        shl     eax,cl
        outwQ   WRITE_ENABLE,ax
        sub     ecx,ebx                 ; 75206

        ;;***********************************************************
        ;; Fetch the VRAM pointer of the location where this character
        ;; will be cached.  Push it onto the stack so that we can
        ;; return this location to the calling function.
        ;;***********************************************************
        lea     edi,p8514FontCache+(ecx*4)
        mov     eax,[edi]
        push    eax                     ; Save current pos

        ;;***********************************************************
        ;; Convert the VRAM pointer into an X,Y pair and store in
        ;; our local vars.
        ;;***********************************************************
        calc8514xy  eax
        mov     dest_x,edx
        mov     dest_y,eax

        ;;***********************************************************
        ;; If the character cache fits on the current row, we can
        ;; simply update the pointer for the next character and store
        ;; the current char.  However, if the character is too wide
        ;; to fit on the current row, we must move to the next row of
        ;; character caches starting in column zero.  The beginning
        ;; row for this character is the current row plus the one-based
        ;; height of the current font.
        ;;***********************************************************
        add     edx,ulWidth             ; Add the char width.
        mov     esi,[pCacheMap]
        cmp     edx,[esi].cm_font_cache_right  ; If it fits, we're OK
        jl      cache_fits

        ;; starting ne row
        mov     edx,[esi].cm_font_cache_left    ; Otherwise start new row
        add     eax,ulHeight                    ;  and Y = current row + Height
        inc     eax                             ; Make height one-based
        ; start 75206
        mov     ecx,eax
        add     ecx,ulHeight                    ; set ecx to last row we need
        cmp     ecx,[esi].cm_font_cache_bottom  ; did it fit?
        jle     UpdatedRow                      ; yes
        cmp     PlaneOffset,0                   ; are we in the first plane
        jnz     UpdatedRow                      ; no, we can't wrap to plane+8
        inc     PlaneOffset
        push    edx
        WaitQ   1
        mov     eax,pPlaneOffset                ; update passed plane offset
        mov     dword ptr [eax],1
        mov     eax,1
        mov     ecx,ulBitPlane
        add     ecx,8
        shl     eax,cl
        outwQ   WRITE_ENABLE,ax
        pop     edx
        mov     eax,[esi].cm_font_cache_top     ; back to top of cache
UpdatedRow:
        ; end 75206
        mov     dest_x,edx
        mov     dest_y,eax

        ;; When we switch scans we need to increment the pvram pointer
        ;; after we change the dest_x and dest_y vars.  The assumption
        ;; below is that we will not have a nugget alignment problem on
        ;; the first char of a scan line.  MRC - 4/5/93

        shl     edx,16
        or      eax,edx
        or      eax,0f0000000h
        add     esp,4                   ; need to reset current position
        push    eax                     ; Save current pos
        mov     eax,dest_y              ; reset eax
        shr     edx,16                  ; put edx back
        add     edx,ulWidth             ; Update next char pos for pvram

cache_fits:

        ;;***********************************************************
        ;; Convert new X,Y into a VRAM pointer and store for use by
        ;; the next char to be cached.
        ;;
        ;; NOTE: During development, we found a quirk in our 8514
        ;;       card that requires the x coordinate to be a mulitple
        ;;       of 4.  Otherwise the monochrome data will be garbage.
        ;;       Therefore, when saving the location for the next
        ;;       character, we will round up.
        ;;
        ;; NOTE: We also need to check if the full width of the next
        ;;       charater will fit on the current scan line, if not
        ;;       we will need to adjust the next char pos.  MRC
        ;;
        ;;***********************************************************
        ifndef  S3
        add     edx,3
        and     edx,NOT 3       ; round up to dword alignment
        else
        add     edx,7
        and     edx,NOT 7       ; round up to dword alignment
        endif

        ;; MRC - NOTE : we can speed up this process by only using the
        ;; x position and letting it wrap around.  Y would be a constant
        ;; equalling the start of the offscreen memory cache

        cmp     edx,[esi].cm_font_cache_right  ; If it fits, we're OK
        jl      short store_pvram

        mov     edx,[esi].cm_font_cache_left ; Otherwise set X=0
        add     eax,ulHeight              ;  and Y = current row + Height
        inc     eax                       ; Make height one-based

store_pvram:
        ror     edx,16
        or      eax,edx
        or      eax,0f0000000h
        mov     [edi],eax

        ;;***********************************************************
        ;; Finally, set the destination of the Blt.
        ;;***********************************************************
ifdef   BPP24
        test    [_DDT],USE_24BPP
        jz      cc_not_24

        ;ifdef   PSUEDO_CACHE
        ;cache the font offscreen
        mov     eax,1
        mov     ecx,ulBitPlane
        shl     eax,cl

        mov     ecx,eax
        rol     eax,8
        mov     al,cl
        rol     eax,8
        mov     al,cl
        mov     Shadow8514Regs.Color_1,eax
        mov     Shadow8514Regs.Color_0,0
        ;endif

        xor     ax,ax
        push    ax                      ; start bits
        push    ONE_BPP
        mov     edx,ulHeight
        push    edx             ; format
        mov     eax,ulWidth
        add     eax,7           ; Round-up to nearest byte
        and     eax,NOT 7
        dec     eax             ; Make zero-based
        push    eax             ; bmap width
        push    eax             ; blt width
        mov     ax,dest_x
        ror     eax,16
        mov     ax,dest_y
        push    eax             ; vram dest
        push    pData           ; memory source
        ;mov     ecx,ulBitPlane
        ;call    _Copy24FontToVRAM
        call    _Copy24MonoToVRAM
        add     esp,26

        ;call    _DVRAM
        jmp     done_char_cache
cc_not_24:
endif

        WaitQ   6
        mov     eax,dest_x
        outwQ   X0,ax
        mov     eax,dest_y
        outwQ   Y0,ax

        ;;***********************************************************
        ;; The Width and Height are provided as parameters and are
        ;; zero-based.  However, the source bitmap data contains extra
        ;; padding on the right to the next byte boundary.  We will
        ;; blt this entire character bitmap to make our lives easier.
        ;; Later, when the next cache request comes in, the padding
        ;; will be overwritten by the next character's bitmap.
        ;;***********************************************************
        mov     eax,ulWidth
        add     eax,7           ; Round-up to nearest byte
        and     eax,NOT 7
        mov     byteWidth,eax
        dec     eax             ; Make zero-based
        outwQ   LX,ax
        mov     eax,ulHeight
        outwQ   LY,ax

        ;;***********************************************************
        ;; Set up the drawing controls to
        ;;***********************************************************
        outwQ   FUNCTION_0,(FUNC_2OPSRC+FUNC_ZEROS)
        outwQ   FUNCTION_1,(FUNC_2OPSRC+FUNC_ONES)

        WaitQIdle
        WaitQ   2
        outwQ   MODE,(MODE_2DECODE+MD_PS_VAR+MD_UP_FALSE)
        outwQ   CMD_FLAGS,CMD_CACHEF

        ;;***********************************************************
        ;; Start feeding mono font data to the board.
        ;;***********************************************************
        mov     eax,ulHeight        ; for # of scanlines in a pointer.
        inc     eax                 ; eax = ulHeight+1 ( make one-based ).
        mov     ebx,byteWidth       ; ebx = width rounded up to byte.

ifndef  S3
        ifdef   WORKS
        shr     ebx,3               ; 8 bits at a time.
        mul     ebx                 ; calc. total # of bytes
        ;; Performance      @RCW
        inc     eax
        shr     eax,1
        else
        ; @DMS this is minutely better ; put in to check for a board hang !!!
        mul     ebx                 ; calc. total # of bytes
        shr     eax,4
        inc     eax
        endif

        mov     ecx,eax             ;  and use as loop counter.
        ;mov     dx,COLOR_1          ; variable data port.
        mov     dx,COLOR_0_WAIT     ; variable data port.
        mov     esi,pData           ; pointer to character data.
        cld                         ; work up in address.

load_char_cache_loop:
                                    ; Load up a byte of monochrome data
        twistbits16nomap            ; Massage into planar format and
                                    ; send it to the board.
        loop load_char_cache_loop

else    ; S3
        mov     ecx,eax
        mov     dx,COLOR_0_WAIT     ; variable data port.
        mov     esi,pData           ; pointer to character data.
        cld                         ; work up in address.
load_char_cache_loop:

        push    ecx                     ; Save the outer loop counter.
        mov     ecx,ebx
        inc     ecx                     ; make one based
        mov     eax,ecx
        shr     ecx,4                   ; round off and get word count
        jcxz    @f
        rep     outsw                   ; to the board.
@@:     shr     eax,4                   ; get the odd word
        jnc     @f
        lodsb
        out     dx,ax
@@:     pop     ecx                     ; Restore outer loop counter

        loop load_char_cache_loop
endif   ;S3

ifdef   BPP24
done_char_cache:
endif


        ;;***********************************************************
        ;; Reset the Write Enable to All planes.
        ;;***********************************************************
        WaitQ   2
        outwQ   WRITE_ENABLE,WRITE_ALL_PLANES
        mov     ax,WORD PTR _Screen8514VisHeight
        or      ax,YMAX_2DECODE
        outwQ   YMAX,ax

        ;;***********************************************************
        ;; Return the VRAM pointer where the character was cached.
        ;;***********************************************************
        pop     eax
cEnd

ENDIF ;; _8514
ENDIF ;; HARD_DRAW

_TEXT          ends

END
