/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   File            = EDDESAVE                                       */
/*                                                                    */
/*   Description     = Enable : subfunction Save DC state             */
/*                                                                    */
/*   Function        = SaveDC makes a copy of the current DC state    */
/*                     and saves it.                                  */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design                   */
/*                                                                    */
/*                                                                    */
/**********************************************************************/

#define INCL_GRE_PALETTE
#define INCL_DDIMISC
#include <eddinclt.h>

#include <eddetypt.h>
#include <eddeextf.h>
#include <eddbextf.h>
#include <eddgextf.h>

#include <eddbcone.h>
#include <eddecone.h>

#include <memman.h>

extern BitmapHeader        NewPatternTable[];

/**********************************************************************/
/* SaveDC saves the current DC state and sets up a new DC with        */
/* identical attributes.                                              */
/**********************************************************************/

USHORT SaveDC (PDC   pdcArg)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    PDC         pdcSave;               /* pointer to saved DC data    */


    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Allocate memory for the new DC instance data.                  */
    /******************************************************************/
    pdcSave = AllocateMemory( sizeof(DC),
                              MT_DC,
                              MO_PRIVATE );
    if (pdcSave == NULL)
    {
        /**************************************************************/
        /* Allocation has failed.                                     */
        /**************************************************************/
        goto SAVEDC_ERROR_EXIT;
    }

    /******************************************************************/
    /* Copy the current DC instance data into the new one             */
    /******************************************************************/
    *pdcSave = *pdc;

    /******************************************************************/
    /* Now allocate memory for the fields accessed via pointers       */
    /******************************************************************/

    if (pdc->DCICorrSize)
    {
        /**************************************************************/
        /* Set up the corr rectangles                                 */
        /**************************************************************/
        pdc->DCICorrRects = AllocateMemory( pdc->DCICorrSize * sizeof(DevRect),
                                            MT_CORRRECTS,
                                            MO_SHARED );
        if (pdc->DCICorrRects == NULL)
        {
            /**********************************************************/
            /* The memory allocation has failed.                      */
            /**********************************************************/
            goto SAVEDC_CORR_EXIT;
        }

        /**************************************************************/
        /* Initialise the correlation rectangles                      */
        /**************************************************************/
        memcpy (pdcSave->DCICorrRects,
                pdc->DCICorrRects,
                pdc->DCICorrSize * sizeof(DevRect));
    }

    /******************************************************************/
    /* If the color mode is not RGB then a logical color table will   */
    /* be present and hence must be copied.                           */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_INDRGB)
    {
        /**************************************************************/
        /* Allocate memory for logical color table                    */
        /**************************************************************/
        pdcSave->DCIColorTable =
            AllocateMemory( pdc->DCIColTabSize * sizeof(COLORTABLETYPE),
                            MT_COLORTABLE,
                            MO_PRIVATE );

        if (pdcSave->DCIColorTable == NULL)
        {
            /**********************************************************/
            /* The allocation has failed.                             */
            /**********************************************************/
            goto SAVEDC_COLOR_EXIT;
        }

        /**************************************************************/
        /* Initialise the color table                                 */
        /**************************************************************/
        memcpy (pdcSave->DCIColorTable,
                pdc->DCIColorTable,
                pdc->DCIColTabSize * sizeof(COLORTABLETYPE));
    }

    /******************************************************************/
    /* Check for font table presence                                  */
    /******************************************************************/
    if (pdc->DCIFontTabNum != 0)
    {
        /**************************************************************/
        /* Allocate memory for font table copy.                       */
        /**************************************************************/
        pdcSave->DCIRealizedFonts =
            AllocateMemory( pdc->DCIFontTabNum * sizeof(RealizedFontType),
                            MT_FONTTABLE,
                            MO_PRIVATE );

        if (pdcSave->DCIRealizedFonts == NULL)
        {
            /**********************************************************/
            /* The allocation has failed.                             */
            /**********************************************************/
            goto SAVEDC_FONT_EXIT;
        }

        /**************************************************************/
        /* Initialise the font table                                  */
        /**************************************************************/
        memcpy ((PVOID)pdcSave->DCIRealizedFonts,
                (PVOID)pdc->DCIRealizedFonts,
                sizeof(RealizedFontType) * pdc->DCIFontTabNum);
    }

#ifdef PALETTE_MGR
    if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        /**************************************************************/
        /* There is a palette selected in this DC so we must          */
        /* increase its usage count since it will be used by another  */
        /* DC (the saved one)                                         */
        /**************************************************************/
        pdc->Palette->cUsage++;
    }
#endif /* PALETTE_MGR */

    /*******************************************************************/
    /* no need to save the pattern definition - just mark it as        */
    /* changed and it will be set up when next used                    */
    /*******************************************************************/
    /* no need to save the marker definition - just mark it as         */
    /* changed in the saved DC and it will be set up when next used    */
    /*******************************************************************/
    pdc->DCIChanged |= NEW_PATTERN_SYMBOL | NEW_MARKER_SYMBOL;
    pdc->DCIChanged &= ~(PATTERN_BITMAP_CREATED |
                            PATTERN_HEADER_CREATED |
                            MARKER_BITMAP_CREATED);
    pdc->TempPattern = 0;
    pdc->DCIMarker = 0;

    /******************************************************************/
    /* Chain the old instance data                                    */
    /******************************************************************/
    pdc->DCINextEntry = pdcSave;

    /******************************************************************/
    /* Increment number of DC's saved                                 */
    /******************************************************************/
    pdc->DCISaveCount++;

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);


SAVEDC_FONT_EXIT:
    /*****************************************************************/
    /* Release colour table memory if allocated                      */
    /*****************************************************************/
    if (pdc->DCIColFormat == LCOLF_INDRGB)
    {
        FreeMemory( pdcSave->DCIColorTable );
    }


SAVEDC_COLOR_EXIT:
    /*******************************************************************/
    /* Free the pick rectangle memory that was just acquired           */
    /*******************************************************************/
    FreeMemory( pdcSave->DCICorrRects );

SAVEDC_CORR_EXIT:
    /******************************************************************/
    /* Free the DC memory that was acquired.                          */
    /******************************************************************/
    FreeMemory( pdcSave );

SAVEDC_ERROR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(ERROR_ZERO);

}
