/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   File            = EDDERSTR                                       */
/*                                                                    */
/*   Description     = Enable : subfunction Restore DC state          */
/*                                                                    */
/*   Function        = RestoreDC sets the DC instance data to a       */
/*                     previously saved state.                        */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design                   */
/*                                                                    */
/*                                                                    */
/**********************************************************************/


#define INCL_GRE_PALETTE
#include <eddinclt.h>

#include <eddecone.h>
#include <eddetypt.h>

#include <eddaextf.h>
#include <eddeextf.h>
#include <eddgextf.h>

#include <memman.h>

#ifdef PALETTE_MGR
#include <twozero.h>
#endif /* PALETTE_MGR */

#ifdef VRAMPTR
#include <eddncach.h>
#endif /* VRAMPTR */


/**********************************************************************/
/* private functions                                                  */
/**********************************************************************/
VOID PASCAL NEAR free_all_dc_memory(VOID);

/**********************************************************************/
/* RestoreDC restores the DC Instance data to a previously saved      */
/* state. (Saved via SaveDC)                                          */
/**********************************************************************/

USHORT RestoreDC (PDC    pdcArg,
                  LONG   Count)

{
#define TFUNC "edde_RestoreDC"
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    PDC            pdcNext;            /* next DC instance in list    */
    PDC            pdcOld;             /* pointer to current DC data  */
    LONG           Pops;               /* no. pops required           */
    LONG           i;                  /* loop counter                */
    DevRect        user_bounds;
    BOOL           user_bounds_are_def;
    POINTL         origin_adjust;


    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);

    PROFILE_ENTER(71);

    /******************************************************************/
    /* SaveDC stores the DC's in a stack. The bottom being the first  */
    /* saved. It works on a first in last out principle where the     */
    /* number of pops is the number that be taken from the stack to   */
    /* reach the desired DC data.                                     */
    /* Negative Count => number of pops is the absolute value of Count*/
    /* Positive Count => pop until the specified DC (eg first saved)  */
    /*                   is reached.                                  */
    /******************************************************************/
    Pops = 0;
    if (Count <= 0)
    {
        Pops = -Count;
    }
    else
    {
        Pops = pdc->DCISaveCount + 1 - Count;
    }

    /******************************************************************/
    /* see if this is a valid number                                  */
    /******************************************************************/
    if ((Pops <= 0) || (Pops > pdc->DCISaveCount))
    {
        LOGERR(TFUNC, "Invalid Pop", &Pops, 1,
                                              PMERR_INVALID_POP_NUMBER);
        goto RESTOREDC_ERR_EXIT;
    }

    /******************************************************************/
    /* initialise pdcNext: this is the pointer to the next DC         */
    /* instance to be deallocated, so start the DC pointed to by the  */
    /* current DC                                                     */
    /******************************************************************/
    pdcNext = pdc->DCINextEntry;

    /******************************************************************/
    /* now free the memory pointed to by entries in the first DC      */
    /* instance (but keep the DC memory for the new DC). This frees   */
    /* clip rectangles and correlation rectangles as well.            */
    /******************************************************************/
    free_all_dc_memory();

#ifdef PALETTE_MGR
#ifdef FIREWALLS
    /**************************************************************/
    /* The engine should have cleaned up the DC before trying to  */
    /* restore it.                                                */
    /* This means it should have deleted any palette which may    */
    /* have been selected into it - however just to               */
    /* make sure we will check for the existence of a palette     */
    /* ourselves, and delete it if necessary                      */
    /**************************************************************/
    if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        DebugOutput("Restore DC - palette still selected\n\r");

        InnerDeletePalette(pdc->Palette);
    }
#endif /* FIREWALLS */
#endif /* PALETTE_MGR */

    /******************************************************************/
    /* "Pop" a DC off the list for the required number of pops,       */
    /* returning the freed memory for each pop.                       */
    /* NB. this loop is executed for one less than the number of pops */
    /* required as the last pop gives the desired DC.                 */
    /******************************************************************/
    for (i = 1; i < Pops; i++)
    {
        /**************************************************************/
        /* Maintain pointers to next DC Instance Data                 */
        /**************************************************************/
        pdcOld = pdcNext;
        pdcNext = pdcOld->DCINextEntry;

        /**************************************************************/
        /* Call routine to free memory items pointed to from within   */
        /* the DC structure.                                          */
        /**************************************************************/
        edde_FreeDCItems(pdcOld);

        /**************************************************************/
        /* Free the dc instance data memory.                          */
        /**************************************************************/
        FreeMemory( pdcOld );

#ifdef PALETTE_MGR
#ifdef FIREWALLS
        /**************************************************************/
        /* The engine should have cleaned up the DC before trying to  */
        /* restore it.                                                */
        /* This means it should have deleted any palette which may    */
        /* have been selected into it - however just to               */
        /* make sure we will check for the existence of a palette     */
        /* ourselves, and delete it if necessary                      */
        /*                                                            */
        /* NB. This has never been spec'ed - this comment is based on */
        /* one in the 8514 source: however I dont see how the engine  */
        /* can clean up intermediate pops on the stack, so this may   */
        /* be a necessary feature - not just a firewall.              */
        /**************************************************************/
        if (pdcOld->DCIColFormat == LCOLF_PALETTE)
        {
            DebugOutput("Restore DC - popped palette still selected\n\r");

            InnerDeletePalette(pdcOld->Palette);
        }
#endif /* FIREWALLS */
#endif /* PALETTE_MGR */
    }

    /******************************************************************/
    /* don't restore user bounds (vga ptr sm83744) ...                */
    /******************************************************************/
    user_bounds[0] = pdc->DCIUserBounds[0];
    user_bounds[1] = pdc->DCIUserBounds[1];
    user_bounds_are_def = pdc->DCIDefUserBounds;

    /******************************************************************/
    /* ...or the dc origin                                            */
    /******************************************************************/
    origin_adjust.x = pdc->DCIOrigin.X;
    origin_adjust.y = pdc->DCIOrigin.Y;

    /******************************************************************/
    /* copy the new DC instance to the memory previously              */
    /* occupied by the current DC prior to this Restore call.         */
    /******************************************************************/
    *pdc = *pdcNext;

    /******************************************************************/
    /* Throw away the memory the new DC was occupying                 */
    /******************************************************************/
    FreeMemory( pdcNext );

    /******************************************************************/
    /* change the dc origin of the newly restored dc back to that of  */
    /* the dc we've just discarded                                    */
    /******************************************************************/
    make_dc_origin_adjustments(pdc, &origin_adjust);

    /******************************************************************/
    /* keep the user bounds we got out the discarded dc               */
    /******************************************************************/
    pdc->DCIUserBounds[0] = user_bounds[0];
    pdc->DCIUserBounds[1] = user_bounds[1];
    pdc->DCIDefUserBounds = user_bounds_are_def;

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);
    PROFILE_EXIT(71);
    return(OK);

RESTOREDC_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FUNN_UNKNOWN, EDF_STANDARD | EDF_DONT_CLEAN);
    PROFILE_EXIT(71);

    return(ERROR_ZERO);
}
#undef TFUNC











VOID PASCAL NEAR free_all_dc_memory(VOID)

 /*********************************************************************/
 /* Frees all the memory allocated to a DC including that given to it */
 /* when first created. Used by restoreDC to destroy stacked DCs.     */
 /*********************************************************************/

{
    /******************************************************************/
    /* first free the memory allocated at creation                    */
    /******************************************************************/
    /* Free memory used for coor rectangles.                          */
    /******************************************************************/
    if ( pdc->DCICorrSize )
    {
        FreeMemory( pdc->DCICorrRects );
    }


    /******************************************************************/
    /* now free the memory allocated since creation                   */
    /******************************************************************/
    /* if there is a logical color table present then free the        */
    /* memory allocated for it                                        */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_INDRGB )
    {
        FreeMemory( pdc->DCIColorTable );
    }


    if ( pdc->DCIFontTabNum )
    {
        /**************************************************************/
        /* Free the memory used for the realized font table.          */
        /**************************************************************/
        FreeMemory( pdc->DCIRealizedFonts );
    }

    /**************************************************************/
    /* free the character pattern bitmap if one was created       */
    /**************************************************************/
    if ( pdc->DCIChanged & PATTERN_BITMAP_CREATED )
    {
        /**********************************************************/
        /* Get rid of the bitmap memory                           */
        /**********************************************************/
        destroy_character_bitmap(pdc->TempPattern);
    }

    if (pdc->DCIChanged & PATTERN_HEADER_CREATED)
    {
        /**********************************************************/
        /* DEFECT - 55770  (Joe Celi 10/15/92)                    */
        /*                                                        */
        /* Get rid of the bitmap memory                           */
        /**********************************************************/
       #ifdef VRAMPTR
        if ( BITMAP_IS_CACHED(pdc->TempPattern) )
           evict_cached_bitmap(pdc->TempPattern->bm_cache_slot);
       #endif /* VRAMPTR */

        /**********************************************************/
        /* Get rid of the bitmap header                           */
        /**********************************************************/
        FreeMemory(pdc->TempPattern);
    }

    /**************************************************************/
    /* free the character marker bitmap if one was created        */
    /**************************************************************/
    if ( pdc->DCIChanged & MARKER_BITMAP_CREATED )
    {
        /**********************************************************/
        /* Get rid of the bitmap memory                           */
        /**********************************************************/
        destroy_character_bitmap(pdc->DCIMarker);

        /**********************************************************/
        /* Get rid of the bitmap header                           */
        /**********************************************************/
        FreeMemory(pdc->DCIMarker);
    }

} /* free_all_dc_memory */
