/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   Module          = EDDCQERY                                       */
/*                                                                    */
/*   Description     = Display Device Driver minor function handler   */
/*                     QueryRealColors, QueryNearestColor,            */
/*                     QueryColorIndex, QueryRGBColor.                */
/*                                                                    */
/*   Function        = Provides information about current color       */
/*                     status                                         */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_WINSYS
#define INCL_DDIMISC
#define INCL_GRE_PALETTE
#include <eddinclt.h>

#include <edddtypt.h>
#include <eddcextf.h>
#include <eddgextf.h>
#include <eddccone.h>

#include <eddhcone.h>       /* required by eddhtype.h */
#include <eddhtype.h>       /* required by convfuns.h */
#include <convfuns.h>

#include <twozero.h>

extern RGB2             HWPalette[HW_PAL_SIZE];
extern RGB2             RealizedLCTPalette[HW_PAL_SIZE];
extern COLORTABLETYPE   DirectSpecialColorTable[];
extern PRGB2            MemoryDeviceDefaultPalette;

extern USHORT           MaxLogColorIndex;
extern ULONG            cPhysicalColors;
extern USHORT           SizeOfHWPalette;

extern DDTType          DDT;
extern BOOL             fRealizeSupported;
extern HDC              ColorTableRealized;


/******************************************************************/
/* QueryRealColors returns RGB values available on the current    */
/* device.                                                        */
/*                                                                */
/* ArgOptions can specify:                                        */
/* LCOLOPT_REALIZED - the RGB values required are those available */
/*                    after the LCT has been realized.            */
/* LCOLOPT_INDEX    - the routine must return results in the form */
/*                    (index, RGB value). RGB values that are     */
/*                    present on the device but have no logical   */
/*                    index are given the special index           */
/*                    CLR_NOINDEX.                                */
/* ArgStart is the ordinal number of the first RGB value to be    */
/*     returned. Note that this is NOT the logical index, and the */
/*     order in which the colors are returned is undefined. This  */
/*     is therefore used as an index into the physical color      */
/*     table.                                                     */
/* ArgCount is the number of elements available in the results    */
/*     array.                                                     */
/* ArgArray is where the values are written to.                   */
/******************************************************************/
DDIENTRY eddc_QueryRealColors (HDC           hdc,
                               ULONG         ArgOptions,
                               ULONG         ArgStart,
                               ULONG         ArgCount,
                               PULONG        ArgArray,
                               PDC           pdcArg,
                               ULONG         FunN)
{
    /******************************************************************/
    /* Local variables.                                               */
    /******************************************************************/
    ULONG      ulLoop;                 /* loop variable               */
    ULONG      ulIndex;                /* Index into return array     */
    ULONG      ulRGB;                  /* value in RGB mode           */

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Check for an invalid start index.                              */
    /******************************************************************/
    if (ArgStart >= cPhysicalColors)
    {
        LogError(PMERR_INV_COLOR_START_INDEX);
        goto QRYREALCOLORS_ERR_EXIT;
    }

    /******************************************************************/
    /* We don't specifically validate the count parameter.            */
    /* In theory at 16bpp we have 65536 valid items to return, and    */
    /* we basically return as much as we can.                         */
    /******************************************************************/

    /******************************************************************/
    /* Check for illegal options. The only valid options are          */
    /* LCOLOPT_REALIZED and LCOLOPT_INDEX.                            */
    /******************************************************************/
    if ( ArgOptions & ~(ULONG)(LCOLOPT_REALIZED | LCOLOPT_INDEX) )
    {
        /**************************************************************/
        /* At least one invalid option has been specified. Log an     */
        /* error and exit.                                            */
        /**************************************************************/
        LogError(PMERR_INV_COLOR_OPTIONS);
        goto QRYREALCOLORS_ERR_EXIT;
    }

    /******************************************************************/
    /* If we dont support realization, or the color table is not      */
    /* realizable then force the LCOLOPT_REALIZED option to be off.   */
    /******************************************************************/
    if ( !fRealizeSupported ||
         !(pdc->DCIColStatus & LCOL_REALIZABLE) )
    {
        ArgOptions &= ~LCOLOPT_REALIZED;
    }

    /******************************************************************/
    /* Calculate Maximum count.                                       */
    /******************************************************************/
    if (ArgOptions & LCOLOPT_INDEX)
    {
        /**************************************************************/
        /* ArgCount must be even, as pairs of values are being        */
        /* returned.                                                  */
        /**************************************************************/
        ArgCount &= 0xFFFFFFFE;
    }

    /******************************************************************/
    /* Now enter a loop to pass back each item of data.               */
    /******************************************************************/
    for (ulLoop = ArgStart, ulIndex = 0;
         (ulLoop < cPhysicalColors) && (ulIndex < ArgCount);
         ulLoop++)
    {
        /**********************************************************/
        /* We need the RGB value.                                 */
        /**********************************************************/
        #ifdef   BPP24
        if (DDT.BitCount == 24)
        {
            /**************************************************/
            /* 24bpp means we form the RGB value from the     */
            /* loop count.                                    */
            /**************************************************/
            ulRGB = ulLoop;
        }
        else if (DDT.BitCount == 16)
        #else
        if (DDT.BitCount == 16)
        #endif
        {
            /**************************************************/
            /* 16bpp means we form the RGB value from the     */
            /* loop count.                                    */
            /**************************************************/
            ulRGB = ULONGFromRGB16(ulLoop);
        }
        else
        {
            /**********************************************************/
            /* If a color table is realized pick up the value from    */
            /* there, otherwise pick it up from the HWPalette table   */
            /* which all other situations use.                        */
            /*                                                        */
            /* This information may be out of date by the time an     */
            /* application gets to use it! (because palette manager   */
            /* and realizable color tables may get in and change the  */
            /* h/w palette).                                          */
            /**********************************************************/
            if (ColorTableRealized)
            {
                ulRGB = URGB(RealizedLCTPalette[ulLoop]);
            }
            else
            {
                ulRGB = URGB(HWPalette[ulLoop]);
            }
            ulRGB &= RGB_MASK;
        }
        /**************************************************************/
        /* Check whether we need to return the index.                 */
        /**************************************************************/
        if (ArgOptions & LCOLOPT_INDEX)
        {
            if (pdc->DCIColFormat == LCOLF_RGB)
            {
                /******************************************************/
                /* Mode is RGB so there is no index -  write RGB      */
                /* value.                                             */
                /******************************************************/
                ArgArray[ulIndex++] = ulRGB;
            }
            else
            {
                /******************************************************/
                /* Mode is not RGB, so write logical index which      */
                /* gives the current physical index.                  */
                /******************************************************/
                ArgArray[ulIndex++] = PhyToLogIndex(ulLoop);
            }
        }
        /**************************************************************/
        /* Now we need to write the color value.                      */
        /**************************************************************/
        if ( (ArgOptions & LCOLOPT_REALIZED) &&
             (ulLoop <= pdc->DCIHighIndex) &&
             (pdc->DCIColorTable[ulLoop].PhyIndex != CLR_NOPHYINDEX))
        {
            /**********************************************************/
            /* Return the RGB value that will be used when the LCT is */
            /* realized.                                              */
            /**********************************************************/
            ArgArray[ulIndex++] =
                    URGB(pdc->DCIColorTable[ulLoop].LogRGB) & RGB_MASK;
        }
        else
        {
            /**********************************************************/
            /* Return the RGB value used when the LCT has NOT been    */
            /* realized, or when the LCT has been realized and there  */
            /* is no valid entry. The value is returned from the      */
            /* default physical palette.                              */
            /**********************************************************/
            ArgArray[ulIndex++] = ulRGB;
        }
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return the number of values written to the array               */
    /******************************************************************/
    return(ulIndex);


QRYREALCOLORS_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return GPI_ALTERROR to indicate an error.                      */
    /******************************************************************/
    return(GPI_ALTERROR);
} /* eddc_QueryRealColors */


/******************************************************************/
/* QueryNearestColor returns the nearest RGB color available on   */
/* the device to the specified RGB value. Note that the nearest   */
/* RGB color is returned even if it is not available using the    */
/* current LCT.                                                   */
/*                                                                */
/* ArgOptions can specify:                                        */
/* LCOLOPT_REALIZED - the RGB value required is that available    */
/*                    after the LCT has been realized.            */
/******************************************************************/
DDIENTRY eddc_QueryNearestColor (HDC          hdc,
                                 ULONG        ArgOptions,
                                 ULONG        ArgRGBColorIn,
                                 PDC          pdcArg,
                                 ULONG        FunN)
{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG      i;                      /* loop variable               */
    RGB2       RGBVal;                 /* Stores current color        */
    RGB2       PassedRGB;
    RGB2       NearestRGB;             /* Stores nearest RGB          */
    ULONG      Diff;                   /* Stores color difference     */
    ULONG      MinDiff;                /* Stores minimum difference   */

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Check for illegal options. The only valid option is            */
    /* LCOLOPT_REALIZED.                                              */
    /******************************************************************/
    if ( ArgOptions & ~(LCOLOPT_REALIZED) )
    {
        /**************************************************************/
        /* At least one invalid option has been specified. Log an     */
        /* error and exit.                                            */
        /**************************************************************/
        LogError(PMERR_INV_COLOR_OPTIONS);
        goto QRYNEARESTCOLOR_ERR_EXIT;
    }

    /******************************************************************/
    /* Check that the supplied RGB value is valid.                    */
    /******************************************************************/
    if (ArgRGBColorIn & 0xFF000000)
    {
        /**************************************************************/
        /* The RGB value is invalid. Log an error and exit.           */
        /**************************************************************/
        LogError(PMERR_INV_RGBCOLOR);
        goto QRYNEARESTCOLOR_ERR_EXIT;
    }

    /**************************************************************/
    /* Get the RGB we were given into an RGB2 structure.          */
    /**************************************************************/
    PassedRGB = *((PRGB2)&ArgRGBColorIn);

    /******************************************************************/
    /* Only process the LCOLOPT_REALIZED option if the color table    */
    /* is realizable, and the configuration supports realization.     */
    /******************************************************************/
    if ( fRealizeSupported                     &&
         (ArgOptions & LCOLOPT_REALIZED)       &&
         (pdc->DCIColStatus & LCOL_REALIZABLE) )
    {
        /**************************************************************/
        /* Return nearest color after LCT has been realized. Have to  */
        /* step through all the LCT entries comparing their RGB       */
        /* values with the given RGB value. The default color palette */
        /* supplies the RGB values for any invalid LCT entries.       */
        /**************************************************************/

        /**************************************************************/
        /* Initialise MinDiff to be a jolly big number, bigger than   */
        /* any other MinDiff in the whole wide world.                 */
        /**************************************************************/
        MinDiff = 0xFFFFFFFFL;

        /**************************************************************/
        /* Search through all entries in the color palette            */
        /**************************************************************/
        for (i = 0; i < SizeOfHWPalette; i++)
        {
            if ( (i <= pdc->DCIHighIndex) &&
                 (pdc->DCIColorTable[i].PhyIndex != CLR_NOPHYINDEX))
            {
                /******************************************************/
                /* Valid entry - use RGB entry in LCT                 */
                /******************************************************/
                RGBVal = pdc->DCIColorTable[i].LogRGB;
            }
            else
            {
                /******************************************************/
                /* There is a gap in the color table, so pick up the  */
                /* value in the default 256 color palette.            */
                /* (this palette is always used in memory DC's)       */
                /******************************************************/
                RGBVal = MemoryDeviceDefaultPalette[i];
            }

            /**********************************************************/
            /* Calculate the difference between the two RGB values    */
            /**********************************************************/
            Diff = rgb2_diff(PassedRGB,RGBVal);

            if (Diff == 0)
            {
                /******************************************************/
                /* Exact match has been found - return RGB value      */
                /******************************************************/
                NearestRGB = RGBVal;
                break;
            }

            if (Diff < MinDiff)
            {
                /******************************************************/
                /* This RGB value is the nearest to the desired RGB   */
                /* value so far. Store the difference and RGB value.  */
                /******************************************************/
                MinDiff = Diff;
                NearestRGB = RGBVal;
            }
        }
    }
    else if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        /**************************************************************/
        /* Use RGB values in current selected palette.                */
        /**************************************************************/
        NearestRGB = pdc->Palette->entries[NearestPaletteIndex(PassedRGB)].rgb;
        NearestRGB.fcOptions = 0;
    }
    else
    {
        /**************************************************************/
        /* Use RGB values in the default color palette.               */
        /**************************************************************/
        NearestRGB = NearestDefaultPhysicalColor(PassedRGB);
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(URGB(NearestRGB));


QRYNEARESTCOLOR_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return GPI_ALTERROR to indicate an error.                      */
    /******************************************************************/
    return(GPI_ALTERROR);

} /* eddc_QueryNearestColor */


/******************************************************************/
/* QueryColorIndex returns the index into the logical color table */
/* which will generate the nearest color to ArgRGBColor on the    */
/* current device. If the mode is RGB then there is no LCT, so    */
/* the RGB value is just returned.                                */
/*                                                                */
/* ArgOptions can specify:                                        */
/* LCOLOPT_REALIZED - the RGB values to be used are those         */
/*                    available after the LCT has been realized.  */
/******************************************************************/
DDIENTRY eddc_QueryColorIndex (HDC          hdc,
                               ULONG        ArgOptions,
                               ULONG        ArgRGBColor,
                               PDC          pdcArg,
                               ULONG        FunN)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG               ulLogIndex;    /* Logical index return value  */
    RGB2                RGBValue;

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Check that the supplied RGB value is valid.                    */
    /******************************************************************/
    if (ArgRGBColor & 0xFF000000)
    {
        /**************************************************************/
        /* The RGB value is invalid. Log an error and exit.           */
        /**************************************************************/
        LogError(PMERR_INV_RGBCOLOR);
        goto QRYCOLORINDEX_ERR_EXIT;
    }

    /******************************************************************/
    /* Check for invalid options. The only valid option is            */
    /* LCOLOPT_REALIZED.                                              */
    /******************************************************************/
    if ( ArgOptions & ~(ULONG)(LCOLOPT_REALIZED) )
    {
        /**************************************************************/
        /* At least one invalid option has been specified. Log an     */
        /* error and exit.                                            */
        /**************************************************************/
        LogError(PMERR_INV_COLOR_OPTIONS);
        goto QRYCOLORINDEX_ERR_EXIT;
    }

    if (pdc->DCIColFormat == LCOLF_RGB)
    {
        /**************************************************************/
        /* Mode is RGB, so the return value is the RGB that was       */
        /* passed in.                                                 */
        /**************************************************************/
        ulLogIndex = ArgRGBColor;
    }
    else if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        /**************************************************************/
        /* Mode is palette, so return the index of the nearest color  */
        /* in the palette.                                            */
        /**************************************************************/
        RGBValue   = *((PRGB2)&ArgRGBColor);
        ulLogIndex = NearestPaletteIndex(RGBValue);

    }
    else
    {
        /**************************************************************/
        /* Mode is index. Search the LCT to find the nearest logical  */
        /* color.                                                     */
        /**************************************************************/
        RGBValue   = *((PRGB2)&ArgRGBColor);
        ulLogIndex = NearestLogicalColor(ArgOptions,RGBValue);
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(ulLogIndex);


QRYCOLORINDEX_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return GPI_ALTERROR to indicate an error.                      */
    /******************************************************************/
    return(GPI_ALTERROR);
} /* eddc_QueryColorIndex */


/******************************************************************/
/* QueryRGBColor returns the actual RGB color that results from   */
/* using the specified color index on the display.                */
/* ArgColor may be an index into a logical color table or an RGB  */
/* value, according to the current index mode.  If the index mode */
/* is RGB the nearest RGB color in the Physical Color Table is    */
/* returned, otherwise the RGB color (in the Physical Color       */
/* Table) which results from using the specified logical index is */
/* returned.                                                      */
/*                                                                */
/* ArgOptions can specify:                                        */
/* LCOLOPT_REALIZED - the RGB value required is that available    */
/*                    after the LCT has been realized.            */
/* LCOLOPT_INDEX    - the routine must return the RGB color       */
/*                    originally specified for this index,        */
/*                    rather than the RGB value that actually     */
/*                    results from drawing on the device.         */
/******************************************************************/
DDIENTRY eddc_QueryRGBColor (HDC           hdc,
                             ULONG         ArgOptions,
                             LONG          lLogIndex,
                             PDC           pdcArg,
                             ULONG         FunN)

{
    /******************************************************************/
    /* The action of this routine is very similar to the action of    */
    /* LogToPhyIndex, on which it is based.                           */
    /******************************************************************/

    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG     ulIndex;
    RGB2      ReqRGB;

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Check for illegal options. The only valid options are          */
    /* LCOLOPT_REALIZED and LCOLOPT_INDEX.                            */
    /******************************************************************/
    if ( ArgOptions & ~(ULONG)(LCOLOPT_REALIZED | LCOLOPT_INDEX) )
    {
        /**************************************************************/
        /* At least one invalid option has been specified. Log an     */
        /* error and exit.                                            */
        /**************************************************************/
        LogError(PMERR_INV_COLOR_OPTIONS);
        goto QRYRGBCOLOR_ERR_EXIT;
    }

    /******************************************************************/
    /* If the color table is not realizable then force the            */
    /* LCOLOPT_REALIZED option to be off.                             */
    /******************************************************************/
    if ( !(pdc->DCIColStatus & LCOL_REALIZABLE) )
    {
        ArgOptions &= ~LCOLOPT_REALIZED;
    }

    /**************************************************************/
    /* Check for a system (negative) index                        */
    /**************************************************************/
    if ( lLogIndex < 0 )
    {
        /**********************************************************/
        /* Index is negative - either a special colour or a       */
        /* system colour                                          */
        /**********************************************************/
        /**********************************************************/
        /* Negate LogIndex to get a positive index we can look    */
        /* it up in the table                                     */
        /**********************************************************/
        ulIndex = -lLogIndex;
        if ( ulIndex < SPECIAL_COL_TAB_SIZE &&
             DirectSpecialColorTable[ulIndex].PhyIndex != CLR_NOPHYINDEX)
        {
            ReqRGB   = DirectSpecialColorTable[ulIndex].LogRGB;
        }
        else
        {
            /**********************************************************/
            /* Invalid negative index                                 */
            /**********************************************************/
            LogError(PMERR_INV_COLOR_INDEX);
            goto QRYRGBCOLOR_ERR_EXIT;
        }
    }
    else
    {
        /**************************************************************/
        /* Value was positive - a 'normal' logical colour.            */
        /**************************************************************/
        /**************************************************************/
        /* Find out what the current mode is.                         */
        /**************************************************************/
        if (pdc->DCIColFormat == LCOLF_PALETTE)
        {
            /**********************************************************/
            /* we have a palette                                      */
            /**********************************************************/
            if (lLogIndex < (LONG)(pdc->Palette->usCountStored))
            {
                /******************************************************/
                /* This entry is valid so pick up the RGB from the    */
                /* palette.                                           */
                /******************************************************/
                ReqRGB = pdc->Palette->entries[lLogIndex].rgb;
                ReqRGB.fcOptions = 0;
            }
            else
            {
                /******************************************************/
                /* The required index is outside the palette range    */
                /******************************************************/
                LogError(PMERR_INV_COLOR_INDEX);
                goto QRYRGBCOLOR_ERR_EXIT;
            }
        }
        else if (pdc->DCIColFormat != LCOLF_RGB)
        {
            /**********************************************************/
            /* Mode is index.                                         */
            /* Get the physical index directly from the LCT.          */
            /**********************************************************/

            /**********************************************************/
            /* Index is positive                                      */
            /**********************************************************/
            if (lLogIndex <= (LONG)(pdc->DCIHighIndex))
            {
                ReqRGB   = pdc->DCIColorTable[lLogIndex].LogRGB;

                /******************************************************/
                /* The color table we just accessed may have gaps in  */
                /* it - in which case we will have just picked up an  */
                /* 'RGB' of CLR_NOINDEX.                              */
                /* If so this is an error.                            */
                /******************************************************/
                if (URGB(ReqRGB) == CLR_NOINDEX)
                {
                    LogError(PMERR_INV_COLOR_INDEX);
                    goto QRYRGBCOLOR_ERR_EXIT;
                }
            }
            else
            {
                /******************************************************/
                /* Logical index is invalid.                          */
                /******************************************************/
                LogError(PMERR_INV_COLOR_INDEX);
                goto QRYRGBCOLOR_ERR_EXIT;
            }
        }
        else
        {
            /**********************************************************/
            /* Mode is RGB                                            */
            /**********************************************************/
            ReqRGB = *((PRGB2)&lLogIndex);
        }
    }

    /******************************************************************/
    /* now consider the options flags                                 */
    /******************************************************************/
    if (ArgOptions & LCOLOPT_INDEX)
    {
        /**************************************************************/
        /* We already have the RGB color originally specified         */
        /**************************************************************/
    }
    else
    {
        /**************************************************************/
        /* Find the nearest RGB actually supported                    */
        /**************************************************************/
        if (fRealizeSupported &&
            ArgOptions & LCOLOPT_REALIZED)
        {
            /**********************************************************/
            /* We need the value when the color table is realized.    */
            /**********************************************************/
            ulIndex = NearestRealizableIndex(ReqRGB);
            ReqRGB = pdc->DCIColorTable[ulIndex].LogRGB;
        }
        else
        {
            /**********************************************************/
            /* Use the nearest RGB in the default HW palette          */
            /**********************************************************/
            ReqRGB = NearestDefaultPhysicalColor(ReqRGB);
        }
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(URGB(ReqRGB));


QRYRGBCOLOR_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return GPI_ALTERROR to indicate an error.                      */
    /******************************************************************/
    return(GPI_ALTERROR);
} /* eddc_QueryRGBColor */
