/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDCCTAB                                       */
/*                                                                    */
/*   Description     = Display Device Driver minor function handler   */
/*                     CreateLogColorTable, RealizeColorTable,        */
/*                     UnRealizeColorTable.                           */
/*                                                                    */
/*   Function        = Create or alter a Logical Color Table.         */
/*                     Load current logical colors into physical      */
/*                     palette.                                       */
/*                     Load default colors into physical palette.     */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_WINSYS
#define INCL_DDIMISC
#define INCL_DDICOMFLAGS
#define INCL_DOSSEMAPHORES
#include <eddinclt.h>

#include <eddccone.h>

#include <edddtypt.h>

#include <eddbextf.h>
#include <eddcextf.h>
#include <eddgextf.h>

#include <memman.h>
#include <semaphor.h>

extern RGB2                 HWPalette[HW_PAL_SIZE];
extern PRGB2                MemoryDeviceDefaultPalette;
extern PRGB2                DirectDeviceDefaultPalette;
extern ULONG                ulDirectDeviceDefaultPaletteSize;
extern PCOLORTABLE          DefaultDirectLogicalColorTable;
extern PCOLORTABLE          DefaultMemoryLogicalColorTable;
extern COLORTABLETYPE       DirectSpecialColorTable[];
extern COLORTABLETYPE       MemorySpecialColorTable[];
extern COLORTABLETYPE       DefaultEightBppTable256[];
extern COLORTABLETYPE       DefaultEightBppTable128[];
extern COLORTABLETYPE       DefaultEightBppTable64[];
extern COLORTABLETYPE       DefaultEightBppTable32[];
extern COLORTABLETYPE       DefaultEightBppTable16[];

extern USHORT               SizeOfHWPalette;
extern USHORT               MaxLogColorIndex;

extern USHORT               CurrentSysColState;

extern DDTType              DDT;

extern HDC                  ColorTableRealized;
extern BOOL                 fRealizeSupported;
extern ULONG                RealizedWhiteIndex;
extern ULONG                RealizedBlackIndex;
extern RGB2                 RGBBlack;
extern RGB2                 RGBWhite;
extern RGB2                 RealizedLCTPalette[HW_PAL_SIZE];


/******************************************************************/
/*                                                                */
/* UpdateAttributes updates the physical indices stored in the    */
/* attribute bundles to match the values in the LCT and the SCT.  */
/*                                                                */
/******************************************************************/
VOID DRIVERCALL eddc_UpdateAttributes(VOID)
{
    /******************************************************************/
    /* Local variable.                                                */
    /******************************************************************/
    ULONG    PhyIndex;       /* Physical index                        */

    /******************************************************************/
    /* We can only update the physical indices if there is actually   */
    /* a bitmap selected into the dc.                                 */
    /* (Direct DCs always have a bitmap selected).                    */
    /******************************************************************/
    if (pdc->DCISelListEntry != NULL)
    {
        /******************************************************************/
        /* Character (text) attributes.                                   */
        /******************************************************************/
        PhyIndex = LogToPhyIndex(pdc->DCICurTxtAts.cbnd.lColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCICharColatts.ForeColor = PhyIndex;
        }

        PhyIndex = LogToPhyIndex(pdc->DCICurTxtAts.cbnd.lBackColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCICharColatts.BackColor = PhyIndex;
        }

        /******************************************************************/
        /* Line attributes.                                               */
        /******************************************************************/
        PhyIndex = LogToPhyIndex(pdc->DCICurLinAts.lbnd.lColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCILineColatts.ForeColor = PhyIndex;
        }

#ifdef LINESBG
        PhyIndex = LogToPhyIndex(pdc->DCICurLinAts.lbnd.lBackColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCILineColatts.BackColor = PhyIndex;
        }
#endif /* LINESBG */

        /******************************************************************/
        /* Marker attributes.                                             */
        /******************************************************************/
        PhyIndex = LogToPhyIndex(pdc->DCICurMrkAts.mbnd.lColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIMarkColatts.ForeColor = PhyIndex;
        }

        PhyIndex = LogToPhyIndex(pdc->DCICurMrkAts.mbnd.lBackColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIMarkColatts.BackColor = PhyIndex;
        }

        /******************************************************************/
        /* Pattern attributes.                                            */
        /******************************************************************/
        PhyIndex = LogToPhyIndex(pdc->DCICurPtnAts.abnd.lColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIPattColatts.ForeColor = PhyIndex;
        }

        PhyIndex = LogToPhyIndex(pdc->DCICurPtnAts.abnd.lBackColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIPattColatts.BackColor = PhyIndex;
        }

        /******************************************************************/
        /* Image attributes.                                              */
        /******************************************************************/
        PhyIndex = LogToPhyIndex(pdc->DCICurImgAts.ibnd.lColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIImagColatts.ForeColor = PhyIndex;
        }

        PhyIndex = LogToPhyIndex(pdc->DCICurImgAts.ibnd.lBackColor);
        if (PhyIndex != CLR_NOPHYINDEX)
        {
            pdc->DCIImagColatts.BackColor = PhyIndex;
        }
    }
} /* eddc_UpdateAttributes */


/**********************************************************************/
/* PropagateSysColorChanges is called by a DC when it discovers that  */
/* the system colors have changed.  The routine updates the physical  */
/* indices that are stored in the LCT and all of the attribute        */
/* bundles, because they may have been invalidated by the system      */
/* color change.                                                      */
/**********************************************************************/
VOID DRIVERCALL eddc_PropagateSysColorChanges(VOID)
{
    /******************************************************************/
    /* Check whether the DC has a color table that needs updating.    */
    /*                                                                */
    /* Defect 53146 - If no color table then don't update.            */
    /******************************************************************/
    if ( pdc->DCIColorTable &&
         pdc->DCIColFormat == LCOLF_INDRGB ||
         pdc->DCIColFormat == LCOLF_DEFAULT )
    {
        /**************************************************************/
        /* There is a color table.                                    */
        /* Check whether background and foreground have been defined. */
        /* If they have not, then copy the values from SYS_CLRWINDOW  */
        /* and SYS_CLRWINDOWTEXT respectively.                        */
        /**************************************************************/
        if (!pdc->DCIBackgndDefined)
        {
            if (pdc->DCIDCType == OD_DIRECT)
            {
                pdc->DCIColorTable[CLR_BACKGROUND] =
                              DirectSpecialColorTable[-(SYSCLR_WINDOW)];
            }
            else
            {
                pdc->DCIColorTable[CLR_BACKGROUND] =
                              MemorySpecialColorTable[-(SYSCLR_WINDOW)];
            }
        }

        if (!pdc->DCINeutralDefined)
        {
            if (pdc->DCIDCType == OD_DIRECT)
            {
                pdc->DCIColorTable[CLR_NEUTRAL] =
                          DirectSpecialColorTable[-(SYSCLR_WINDOWTEXT)];
            }
            else
            {
                pdc->DCIColorTable[CLR_NEUTRAL] =
                          MemorySpecialColorTable[-(SYSCLR_WINDOWTEXT)];
            }
        }
    }

    /******************************************************************/
    /* Update all of the attribute bundles.                           */
    /******************************************************************/
    eddc_UpdateAttributes();

    /******************************************************************/
    /* Update the count in pdc to indicate that the system colors  */
    /* have been propagated.                                          */
    /******************************************************************/
    pdc->DCISysColState = CurrentSysColState;
} /* eddc_PropagateSysColorChanges */


/**********************************************************************/
/* AlterSysColorTable defines the entries in the System Color Table   */
/* (SCT).  The system color table is stored by the driver in (part    */
/* of) the SpecialColorTables.  There are actually two versions of    */
/* this table corresponding to memory and direct DCs because the      */
/* device default palette used by direct DCs is not always the same   */
/* as the device default palette used by memory DCs (because HW slots */
/* are allocated to palette manager palettes).                        */
/*                                                                    */
/* This function is called by CreateLogColorTable when the            */
/* LCOL_SYSCOLORS bit in ulOptions is set.                            */
/*                                                                    */
/* ulFormat can specify:                                              */
/* LCOLF_INDRGB     - The supplied data is an array of pairs,         */
/*                    (logical index, RGB value)                      */
/* LCOLF_CONSECRGB  - The supplied data is an array of RGB values     */
/*                    which are the colors for the logical            */
/*                    indices from lStart upwards.                    */
/*                                                                    */
/* pData is where the input values are stored.                        */
/*                                                                    */
/* ulCount is the number of elements supplied in pData. If the        */
/* mode is LCOLF_INDRGB then this must be even.                       */
/*                                                                    */
/**********************************************************************/

ULONG DRIVERCALL eddc_AlterSysColorTable (ULONG         ulFormat,
                                          LONG          lStart,
                                          ULONG         ulCount,
                                          PULONG        pData)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    LONG           LogIndex;           /* Stores logical index        */
    ULONG          i;                  /* Loop counter                */

    if (ulFormat == LCOLF_CONSECRGB)
    {
        /**************************************************************/
        /* Format is LCOLF_CONSECRGB.                                 */
        /**************************************************************/

        /**************************************************************/
        /* Check lStart is a valid value.                             */
        /* SYSCLR_HELPHILITE is the 'first' system color (negative    */
        /* number nearest to zero).                                   */
        /**************************************************************/
        if ((lStart > SYSCLR_HELPHILITE) ||
            (lStart <= (SYSCLR_HELPHILITE - SYSCLR_CSYSCOLORS) ) )
        {
            /**********************************************************/
            /* Start index is invalid.                                */
            /**********************************************************/
            LOGERR(NULL, "Invalid start index", &lStart, 1,
                                         PMERR_INV_COLOR_START_INDEX);
            return(ERROR_ZERO);
        }

        if ( (lStart + ulCount - 1) > SYSCLR_HELPHILITE)
        {
            /**********************************************************/
            /* Count is invalid.                                      */
            /**********************************************************/
            LOGERR(NULL, "Invalid count", &ulCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
            return(ERROR_ZERO);
        }

        /**************************************************************/
        /* Read values from pData                                     */
        /* The loop counts LogIndex down because what we were actually*/
        /* passed was a negative LogIndex to be counted up!           */
        /**************************************************************/
        for (i = 0, LogIndex = -lStart;
             i < ulCount;
             i++, LogIndex--)
        {
            /**********************************************************/
            /* Read logical RGB value.                                */
            /**********************************************************/
            DirectSpecialColorTable[LogIndex].LogRGB = ((PRGB2)pData)[i];
            MemorySpecialColorTable[LogIndex].LogRGB = ((PRGB2)pData)[i];

            /**********************************************************/
            /* Calculate nearest physical indexes using the default   */
            /* physical palettes for OD_MEMORY and OD_DIRECT DCs.     */
            /**********************************************************/
            DirectSpecialColorTable[LogIndex].PhyIndex =
                        NearestDirectDefaultPhysicalIndex(
                              DirectSpecialColorTable[LogIndex].LogRGB);

            MemorySpecialColorTable[LogIndex].PhyIndex =
                        NearestMemoryDefaultPhysicalIndex(
                              MemorySpecialColorTable[LogIndex].LogRGB);
        }
    }
    else
    {
        /**************************************************************/
        /* Format is LCOLF_INDRGB.                                    */
        /**************************************************************/
        /**************************************************************/
        /* Check that ArgCount is even.                               */
        /**************************************************************/
        if (ulCount & 0x00000001)
        {
            /**********************************************************/
            /* ArgCount is odd when it should be even.                */
            /**********************************************************/
            LOGERR(NULL, "Invalid ArgCount", &ulCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
            return(ERROR_ZERO);
        }

        /**************************************************************/
        /* Divide ArgCount by two so that it contains the number      */
        /* of new RGB values being passed.                            */
        /**************************************************************/
        ulCount /= 2;

        for (i = 0; i < ulCount; i++)
        {
            /**********************************************************/
            /* Read logical index.                                    */
            /**********************************************************/
            LogIndex = pData[i*2];
            if ((LogIndex > SYSCLR_HELPHILITE) ||
                (LogIndex <= (SYSCLR_HELPHILITE - SYSCLR_CSYSCOLORS)))
            {
                /******************************************************/
                /* System color index is invalid.                     */
                /******************************************************/
                LOGERR(NULL, "Invalid index", &LogIndex, 1,
                                           PMERR_INV_COLOR_START_INDEX);
                return(ERROR_ZERO);
            }

            /**********************************************************/
            /* Negate the Logical index to get the SpecialColorTable  */
            /* index.                                                 */
            /**********************************************************/
            LogIndex = -LogIndex;

            /**********************************************************/
            /* Read logical RGB value.                                */
            /**********************************************************/
            DirectSpecialColorTable[LogIndex].LogRGB = ((PRGB2)pData)[i*2+1];
            MemorySpecialColorTable[LogIndex].LogRGB = ((PRGB2)pData)[i*2+1];

            /**********************************************************/
            /* Calculate nearest physical indexes using the default   */
            /* physical palettes for OD_MEMORY and OD_DIRECT DCs.     */
            /**********************************************************/
            DirectSpecialColorTable[LogIndex].PhyIndex =
                        NearestDirectDefaultPhysicalIndex(
                              DirectSpecialColorTable[LogIndex].LogRGB);
            MemorySpecialColorTable[LogIndex].PhyIndex =
                        NearestMemoryDefaultPhysicalIndex(
                              MemorySpecialColorTable[LogIndex].LogRGB);
        }
    }

    /******************************************************************/
    /* Update the Default LCTs for OD_MEMORY and OD_DIRECT DCs so     */
    /* that entry 0 (CLR_BACKGROUND) and entry 7 (CLR_NEUTRAL) have   */
    /* the values of SYSCLR_WINDOW and SYSCLR_WINDOWTEXT              */
    /* respectively.  (Note: if we are at 8 bits per pel then palette */
    /* manager will be running so we must update all of the default   */
    /* logical color tables (corresponding to the five possible size  */
    /* of device default palette).  If we are not at 8 bits per pel   */
    /* then the default OD_DIRECT logical color table will be the     */
    /* same as the one for OD_MEMORY DCs.)                            */
    /******************************************************************/
    if (DDT.BitCount == 8)
    {
        DefaultEightBppTable256[CLR_BACKGROUND] =
        DefaultEightBppTable128[CLR_BACKGROUND] =
        DefaultEightBppTable64[CLR_BACKGROUND] =
        DefaultEightBppTable32[CLR_BACKGROUND] =
        DefaultEightBppTable16[CLR_BACKGROUND] =
                              DirectSpecialColorTable[-(SYSCLR_WINDOW)];
        DefaultEightBppTable256[CLR_NEUTRAL] =
        DefaultEightBppTable128[CLR_NEUTRAL] =
        DefaultEightBppTable64[CLR_NEUTRAL] =
        DefaultEightBppTable32[CLR_NEUTRAL] =
        DefaultEightBppTable16[CLR_NEUTRAL] =
                          DirectSpecialColorTable[-(SYSCLR_WINDOWTEXT)];
    }
    else
    {
        DefaultDirectLogicalColorTable[CLR_BACKGROUND] =
                              DirectSpecialColorTable[-(SYSCLR_WINDOW)];
        DefaultDirectLogicalColorTable[CLR_NEUTRAL] =
                          DirectSpecialColorTable[-(SYSCLR_WINDOWTEXT)];
    }

    /******************************************************************/
    /* Update system color state.                                     */
    /* Each DC stores its own system color state. Whenever the driver */
    /* is entered, the DC's system color state is compared with the   */
    /* current global system color state. If they are different then  */
    /* the system colors have been altered and the DC has to go       */
    /* through all of its attribute bundles and update the physical   */
    /* indices that are stored there (because the system color change */
    /* may have altered them).                                        */
    /******************************************************************/
    CurrentSysColState++;

    /******************************************************************/
    /* Propagate the system color changes.                            */
    /******************************************************************/
    eddc_PropagateSysColorChanges();

    return(OK);
} /* eddc_AlterSysColorTable */


/******************************************************************/
/* ResetLogColorTable resets the entries in the Logical Color     */
/* Table (LCT) to their default state.                            */
/******************************************************************/
VOID DRIVERCALL eddc_ResetLogColorTable(VOID)
{
    if (pdc->DCIColFormat == LCOLF_INDRGB)
    {
        /**************************************************************/
        /* A color table currently exists so free its memory          */
        /**************************************************************/
        FreeMemory( pdc->DCIColorTable );
    }

    /******************************************************************/
    /* Set the values in DCIData to their default values              */
    /******************************************************************/
    if (pdc->DCIDCType == OD_DIRECT)
    {
        pdc->DCIColorTable = DefaultDirectLogicalColorTable;
        pdc->DCIDeviceDefaultPalette = DirectDeviceDefaultPalette;
    }
    else
    {
        pdc->DCIColorTable = DefaultMemoryLogicalColorTable;
        pdc->DCIDeviceDefaultPalette = MemoryDeviceDefaultPalette;
    }

    pdc->DCIColFormat      = LCOLF_DEFAULT;
    pdc->DCIColStatus      = 0;
    pdc->DCIColTabSize     = DFLT_LOG_COL_TABLE_SIZE;
    pdc->DCILowIndex       = 0;
    pdc->DCIHighIndex      = DFLT_LOG_COL_TABLE_SIZE - 1;
    pdc->DCIBackgndDefined = FALSE;
    pdc->DCINeutralDefined = FALSE;

} /* eddc_ResetLogColorTable */


/**********************************************************************/
/* CreateLogColorTable defines the entries in the Logical Color       */
/* Table (LCT).                                                       */
/*                                                                    */
/* ArgOptions can specify:                                            */
/* LCOL_RESET       - the color table is reset to its default         */
/*                    before processing the supplied values.          */
/*                    If old mode was RGB then this option is         */
/*                    forced.                                         */
/* LCOL_REALIZABLE  - indicates that the color table can be           */
/*                    realized using RealizeColorTable. This          */
/*                    affects the way in which the routine maps       */
/*                    the logical color table onto the physical       */
/*                    color table.                                    */
/* LCOL_PURECOLOR   - If set then only pure colors can be used        */
/*                    i.e. dithering is not allowed.                  */
/*                                                                    */
/* LCOL_SYSCOLORS   - The colors supplied are system colors           */
/*                    i.e. the logical indices are negative and       */
/*                    the colors must be updated in the global        */
/*                    array SpecialColorTable.                        */
/*                                                                    */
/* ArgFormat can specify:                                             */
/* LCOLF_INDRGB     - The supplied data is an array of pairs,         */
/*                    (logical index, RGB value)                      */
/* LCOLF_CONSECRGB  - The supplied data is an array of RGB values     */
/*                    which are the colors for the logical            */
/*                    indices from ArgStart upwards.                  */
/* LCOLF_RGB        - Index mode is RGB, which does not use a         */
/*                    color table                                     */
/*                                                                    */
/* ArgData is where the input values are stored.                      */
/*                                                                    */
/* ArgCount is the number of elements supplied in ArgData. If the     */
/* mode is LCOLF_INDRGB then this must be even.                       */
/**********************************************************************/
DDIENTRY eddc_CreateLogColorTable (HDC           hdc,
                                   ULONG         ArgOptions,
                                   ULONG         ArgFormat,
                                   ULONG         ArgStart,
                                   ULONG         ArgCount,
                                   PULONG        ArgData,
                                   PDC           pdcArg,
                                   ULONG         FunN)
{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG          Size;               /* Color Table Size            */
    ULONG          LogIndex;           /* Stores logical index        */
    ULONG          i;                  /* Loop counter                */
    PCOLORTABLE    NewTable;           /* Pointer to new color table  */

    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

#ifdef PALETTE_MGR
    /******************************************************************/
    /* this call cannot be used if there is a palette selected into   */
    /* this DC                                                        */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_PALETTE)
    {
        LOGERR(NULL, "palette error", 0L, 0 ,PMERR_PALETTE_SELECTED);
        goto CREATECOLTABLE_ERR_EXIT;
    }
#endif /* PALETTE_MGR */

    /******************************************************************/
    /* If the function is being called within an area or path         */
    /* definition then we raise an error and leave immediately.       */
    /******************************************************************/
    if (FunNTest(COM_AREA))
    {
        /**************************************************************/
        /* We have been called within an area definition              */
        /**************************************************************/
        LOGERR(NULL, "Invalid within area", FNULL, 0,
                                                     PMERR_INV_IN_AREA);
        goto CREATECOLTABLE_ERR_EXIT;
    }

    if (FunNTest(COM_PATH))
    {
        /**************************************************************/
        /* We have been called within a path definition               */
        /**************************************************************/
        LOGERR(NULL, "Invalid within path", FNULL, 0,
                                                     PMERR_INV_IN_PATH);
        goto CREATECOLTABLE_ERR_EXIT;
    }

    if (ArgFormat == LCOLF_RGB)
    {
        /**************************************************************/
        /* New mode is RGB                                            */
        /**************************************************************/
        /**************************************************************/
        /* Check options                                              */
        /**************************************************************/
        if (ArgOptions & LCOL_REALIZABLE)
        {
            /**********************************************************/
            /* Log an error and exit                                  */
            /**********************************************************/
            LOGERR(NULL, "Invalid option", &ArgOptions, 1,
                                          PMERR_INCOMPAT_COLOR_OPTIONS);
            goto CREATECOLTABLE_ERR_EXIT;
        }

        if (pdc->DCIColFormat == LCOLF_INDRGB)
        {
            /**********************************************************/
            /* A color table currently exists so free its memory      */
            /**********************************************************/
            FreeMemory( pdc->DCIColorTable );
        }
        pdc->DCIColFormat = (USHORT)LCOLF_RGB;
        pdc->DCIColStatus = (USHORT)ArgOptions;
        pdc->DCIColTabSize = 0;
        pdc->DCIHighIndex  = 0;

        goto CREATECOLTABLE_OK_EXIT;
    }

    /******************************************************************/
    /* Check for an invalid ArgFormat                                 */
    /******************************************************************/
    if ( (ArgFormat != LCOLF_INDRGB) &&
         (ArgFormat != LCOLF_CONSECRGB) )
    {
        /**************************************************************/
        /* Log an error and exit                                      */
        /**************************************************************/
        LOGERR(NULL, "Invalid color format", &ArgFormat, 1,
                                                PMERR_INV_COLOR_FORMAT);
        goto CREATECOLTABLE_ERR_EXIT;
    }

    if (ArgOptions & LCOL_SYSCOLORS)
    {
        /**************************************************************/
        /* Colors are system colors                                   */
        /**************************************************************/
        if (eddc_AlterSysColorTable(ArgFormat,
                                    ArgStart,
                                    ArgCount,
                                    ArgData))
        {
            /**********************************************************/
            /* The system colors were successfully updated so exit.   */
            /**********************************************************/
            goto CREATECOLTABLE_SYS_OK_EXIT;
        }
        else
        {
            /**********************************************************/
            /* An error occured during the system color update. An    */
            /* error will already have been logged so just exit.      */
            /**********************************************************/
            goto CREATECOLTABLE_ERR_EXIT;
        }
    }

    /******************************************************************/
    /* If current mode is LCOLF_RGB then force LCOL_RESET option.     */
    /******************************************************************/
    if (pdc->DCIColFormat == LCOLF_RGB)
    {
        ArgOptions |= LCOL_RESET;
    }

    /******************************************************************/
    /* Check whether the LCT is being reset to the default.           */
    /******************************************************************/
    if ( (ArgOptions & LCOL_RESET) &&
         !(ArgOptions & LCOL_REALIZABLE) &&
         (ArgCount == 0) )
    {
        /**************************************************************/
        /* Table will be the default.                                 */
        /**************************************************************/
        eddc_ResetLogColorTable();

        goto CREATECOLTABLE_OK_EXIT;
    }

    /******************************************************************/
    /* Initialise Size to the best value we know of so far...         */
    /******************************************************************/
    if (ArgOptions & LCOL_RESET)
    {
        /**************************************************************/
        /* Table is being reset so set size to that of default        */
        /* logical table.                                             */
        /**************************************************************/
        Size = DFLT_LOG_COL_TABLE_SIZE;
    }
    else
    {
        /**************************************************************/
        /* Set size to the same as the current logical table          */
        /**************************************************************/
        Size = pdc->DCIColTabSize;
    }

    /******************************************************************/
    /* Calculate size of new color table.                             */
    /******************************************************************/
    if (ArgFormat == LCOLF_CONSECRGB)
    {
        /**************************************************************/
        /* Format is LCOLF_CONSECRGB.                                 */
        /* Calculate size of color table given that the values passed */
        /* are all consecutive.                                       */
        /**************************************************************/

        /**************************************************************/
        /* Check that start index is valid.                           */
        /**************************************************************/
        if (ArgStart > MaxLogColorIndex)
        {
            /**********************************************************/
            /* Log an error and exit                                  */
            /**********************************************************/
            LOGERR(NULL, "Invalid start index", &ArgStart, 1,
                                           PMERR_INV_COLOR_START_INDEX);
            goto CREATECOLTABLE_ERR_EXIT;
        }
        /**************************************************************/
        /* Check that count is valid i.e. that it does not take us    */
        /* past the end of the color table.                           */
        /**************************************************************/
        if (ArgStart + ArgCount > (ULONG)(MaxLogColorIndex + 1))
        {
            /**********************************************************/
            /* Log an error and exit                                  */
            /**********************************************************/
            LOGERR(NULL, "Invalid count", &ArgCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
            goto CREATECOLTABLE_ERR_EXIT;
        }
        /**************************************************************/
        /* Calculate the size                                         */
        /**************************************************************/
        Size = max( Size, (ArgStart + ArgCount) );
    }
    else
    {
        /**************************************************************/
        /* Format is LCOLF_INDRGB.                                    */
        /* Calculate size of color table given that the indices of    */
        /* the RGB values are all individually specified.             */
        /**************************************************************/

        /**************************************************************/
        /* Check ArgCount is even.                                    */
        /**************************************************************/
        if (ArgCount & 0x00000001)
        {
            /**********************************************************/
            /* ArgCount is odd when it should be even.                */
            /**********************************************************/
            LOGERR(NULL, "Invalid ArgCount", &ArgCount, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
            goto CREATECOLTABLE_ERR_EXIT;
        }

        /**************************************************************/
        /* Divide ArgCount by two so that it contains the number of   */
        /* new RGB values being passed.                               */
        /**************************************************************/
        ArgCount /= 2;

        /**************************************************************/
        /* Enter a loop which looks for the highest index being       */
        /* passed.                                                    */
        /**************************************************************/
        for (i = 0; i < ArgCount; i++)
        {
            LogIndex = ArgData[i*2];
            if (LogIndex >= Size)
            {
                /******************************************************/
                /* Update Size. Size is set to (Index + 1) because    */
                /* the indices begin at zero                          */
                /******************************************************/
                Size = LogIndex + 1;
            }
        }
        /**************************************************************/
        /* Check that the maximum index is valid                      */
        /**************************************************************/
        if ( Size > (MaxLogColorIndex + 1) )
        {
            Size--;

            LOGERR(NULL, "Invalid max index", &Size, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
            goto CREATECOLTABLE_ERR_EXIT;
        }
    }

    /******************************************************************/
    /* Set NewTable to point to current table                         */
    /******************************************************************/
    NewTable = pdc->DCIColorTable;

    /******************************************************************/
    /* Check whether we need to acquire new storage.                  */
    /* New storage is fetched if the new table size is different to   */
    /* the current table size or if the current table is default (if  */
    /* we get this far the new table is not going to be the default). */
    /******************************************************************/
    if ( (Size != pdc->DCIColTabSize) ||
         (pdc->DCIColFormat == LCOLF_DEFAULT) )
    {
        NewTable = AllocateMemory( Size * sizeof(COLORTABLETYPE),
                                   MT_COLORTABLE,
                                   MO_SHARED );
        if (NewTable == NULL)
        {
            LOGERR(NULL, "No memory for colour table", NULL, 0,
                                             PMERR_INSUFFICIENT_MEMORY);
            goto CREATECOLTABLE_ERR_EXIT;
        }

        /**************************************************************/
        /* Set logical RGB values to the special value CLR_NOINDEX.   */
        /* (This is the value we must return to QueryLogColorTable).  */
        /*                                                            */
        /* For internal checks within the driver we check the         */
        /* PhyIndex field, so mark all these with CLR_NOPHYINDEX.     */
        /*                                                            */
        /* We only have to fill from index 16 upwards as indices 0-15 */
        /* always have defined values (due to the default color       */
        /* table).                                                    */
        /**************************************************************/
        for (i = DFLT_LOG_COL_TABLE_SIZE; i < Size; i++)
        {
            *(PULONG)(&NewTable[i].LogRGB) = CLR_NOINDEX;
            NewTable[i].PhyIndex           = CLR_NOPHYINDEX;
        }

        /**************************************************************/
        /* Copy old logical color table to new if required (if table  */
        /* is not being reset).                                       */
        /*                                                            */
        /* We only copy the RGB value because we always recalculate   */
        /* physical indices (because the realizable state may have    */
        /* changed).                                                  */
        /**************************************************************/
        if ( !(ArgOptions & LCOL_RESET) )
        {
            for (i = 0;
                 (i < pdc->DCIColTabSize) && (i < Size);
                 i++)

            {
                NewTable[i].LogRGB = pdc->DCIColorTable[i].LogRGB;
                /******************************************************/
                /* Make sure that this value is not thought to be     */
                /* invalid by making the PhyIndex 0 (any value which  */
                /* is not CLR_NOPHYINDEX would do!).  The true        */
                /* PhyIndex will be calculated later.                 */
                /******************************************************/
                NewTable[i].PhyIndex = 0;
            }
        }

        /**************************************************************/
        /* If a table had previously been established then free       */
        /* its memory.                                                */
        /**************************************************************/
        if (pdc->DCIColFormat == LCOLF_INDRGB)
        {
            FreeMemory( pdc->DCIColorTable );
        }
    }

    /******************************************************************/
    /* If table is being reset                                        */
    /******************************************************************/
    if (ArgOptions & LCOL_RESET)
    {
        /**************************************************************/
        /* Copy default color table across                            */
        /**************************************************************/
        for (i = 0; i < DFLT_LOG_COL_TABLE_SIZE; i++)
        {
            NewTable[i].LogRGB = DefaultDirectLogicalColorTable[i].LogRGB;
            /******************************************************/
            /* Make sure that this value is not thought to be     */
            /* invalid by making the PhyIndex 0 (any value which  */
            /* is not CLR_NOPHYINDEX would do!).  The true        */
            /* PhyIndex will be calculated later.                 */
            /******************************************************/
            NewTable[i].PhyIndex = 0;
        }
        /**************************************************************/
        /* Reset flags to say that application has not defined        */
        /* CLR_BACKGROUND and CLR_NEUTRAL.                            */
        /**************************************************************/
        pdc->DCIBackgndDefined = FALSE;
        pdc->DCINeutralDefined = FALSE;
    }

    /******************************************************************/
    /* Update format.                                                 */
    /* Format will always be LCOLF_INDRGB by this stage.              */
    /******************************************************************/
    pdc->DCIColFormat = LCOLF_INDRGB;

    /******************************************************************/
    /* Update HighIndex to new value.                                 */
    /******************************************************************/
    pdc->DCIHighIndex = (USHORT)(Size - 1);

    /******************************************************************/
    /* Write new values into table.                                   */
    /******************************************************************/
    for (i = 0; i < ArgCount; i++)
    {
        if (ArgFormat == LCOLF_INDRGB)
        {
            /**********************************************************/
            /* Read index from ArgData                                */
            /**********************************************************/
            LogIndex = ArgData[i*2];
            NewTable[LogIndex].LogRGB = ((PRGB2)ArgData)[i*2+1];
            /******************************************************/
            /* Make sure that this value is not thought to be     */
            /* invalid by making the PhyIndex 0 (any value which  */
            /* is not CLR_NOPHYINDEX would do!).  The true        */
            /* PhyIndex will be calculated later.                 */
            /******************************************************/
            NewTable[LogIndex].PhyIndex = 0;

            if (LogIndex == CLR_BACKGROUND)
            {
                pdc->DCIBackgndDefined = TRUE;
            }
            else if (LogIndex == CLR_NEUTRAL)
            {
                pdc->DCINeutralDefined = TRUE;
            }
        }
        else
        {
            /**********************************************************/
            /* Calculate index directly from loop count               */
            /**********************************************************/
            NewTable[ArgStart+i].LogRGB = ((PRGB2)ArgData)[i];
            /******************************************************/
            /* Make sure that this value is not thought to be     */
            /* invalid by making the PhyIndex 0 (any value which  */
            /* is not CLR_NOPHYINDEX would do!).  The true        */
            /* PhyIndex will be calculated later.                 */
            /******************************************************/
            NewTable[ArgStart+i].PhyIndex = 0;

            if (ArgStart+i == CLR_BACKGROUND)
            {
                pdc->DCIBackgndDefined = TRUE;
            }
            else if (ArgStart+i == CLR_NEUTRAL)
            {
                pdc->DCINeutralDefined = TRUE;
            }
        }
    }

    /******************************************************************/
    /* Include new table in DC instance data                          */
    /******************************************************************/
    pdc->DCIColorTable = NewTable;
    pdc->DCIColTabSize = (USHORT)Size;
    pdc->DCIColStatus  = (USHORT)ArgOptions;

    /******************************************************************/
    /* Calculate physical indices                                     */
    /******************************************************************/
    CalculateColorTablePhysicalIndices();

CREATECOLTABLE_OK_EXIT:
    /******************************************************************/
    /* Update physical indices in attribute bundles.                  */
    /******************************************************************/
    eddc_UpdateAttributes();

CREATECOLTABLE_SYS_OK_EXIT:
    /******************************************************************/
    /* Setting SYSCLRs will already have called UpdateAttributes      */
    /* (via PropagateSysColors).                                      */
    /******************************************************************/

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(OK);

CREATECOLTABLE_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(ERROR_ZERO);
} /* eddc_CreateLogColorTable */


VOID DRIVERCALL CalculateColorTablePhysicalIndices(VOID)
{
    ULONG ulLogIndex;

#ifdef FIREWALLS
    /******************************************************************/
    /* This function should only be called if the color table is in   */
    /* rgb-index pair format.                                         */
    /******************************************************************/
    if (pdc->DCIColFormat != LCOLF_INDRGB)
    {
        haltproc();
    }
#endif /* FIREWALLS */

    /******************************************************************/
    /* We are about to change the physical indexes used by the color  */
    /* table. We must call ChangedColorTable so that the current      */
    /* "palette to palette" mapping will be invalidated if necessary. */
    /******************************************************************/
    ChangedColorTable (pdc->DCIColorTable);

    /******************************************************************/
    /* Update the DCs device default palette so that we will          */
    /* calculate the correct physical indices when we call off to     */
    /* NearestDefaultPhysicalIndex.                                   */
    /******************************************************************/
    if (pdc->DCIDCType == OD_DIRECT)
    {
        pdc->DCIDeviceDefaultPalette = DirectDeviceDefaultPalette;
    }
    else
    {
        pdc->DCIDeviceDefaultPalette = MemoryDeviceDefaultPalette;
    }

    /******************************************************************/
    /* Calculate physical indices using the current device default    */
    /* palette.                                                       */
    /******************************************************************/
    for (ulLogIndex = pdc->DCILowIndex;
         ulLogIndex <= pdc->DCIHighIndex;
         ulLogIndex++)
    {
        if (pdc->DCIColorTable[ulLogIndex].PhyIndex != CLR_NOPHYINDEX)
        {
            if (fRealizeSupported && (pdc->DCIColStatus & LCOL_REALIZABLE))
            {
                /******************************************************/
                /* Realizable was specified and we actually support it*/
                /* Thus physical index is the same as logical index.  */
                /******************************************************/
                pdc->DCIColorTable[ulLogIndex].PhyIndex = ulLogIndex;
            }
            else
            {
                /******************************************************/
                /* Find physical index which will give nearest RGB    */
                /* value to specified RGB using the current device    */
                /* default palette.                                   */
                /******************************************************/
                pdc->DCIColorTable[ulLogIndex].PhyIndex =
                              NearestDefaultPhysicalIndex(
                                        pdc->DCIColorTable[ulLogIndex].LogRGB);
            }
        }
    }
}

/******************************************************************/
/* RealizeColorTable ensures that the physical color table is set */
/* to the closest possible match to the logical color table.      */
/******************************************************************/
DDIENTRY eddc_RealizeColorTable (HDC          hdc,
                                 PDC          pdcArg,
                                 ULONG        FunN)
{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG          i;              /* Loop counter                    */

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* If the function is being called within an area or path         */
    /* definition then we raise an error and leave immediately.       */
    /******************************************************************/
    if (FunNTest(COM_AREA))
    {
        /**************************************************************/
        /* We have been called within an area definition              */
        /**************************************************************/
        LOGERR(NULL, "Invalid within area", FNULL, 0,
                                                     PMERR_INV_IN_AREA);
        goto REALIZECOLTABLE_ERR_EXIT;
    }

    if (FunNTest(COM_PATH))
    {
        /**************************************************************/
        /* We have been called within a path definition               */
        /**************************************************************/
        LOGERR(NULL, "Invalid within path", FNULL, 0,
                                                     PMERR_INV_IN_PATH);
        goto REALIZECOLTABLE_ERR_EXIT;
    }

    /******************************************************************/
    /* Check whether color table can be realized.                     */
    /******************************************************************/
    if ( !(pdc->DCIColStatus & LCOL_REALIZABLE) )
    {
        /**************************************************************/
        /* If it cannot, log an error and return                      */
        /**************************************************************/
        LOGERR(NULL, "Unrealizable", FNULL, 0,
                                        PMERR_COL_TABLE_NOT_REALIZABLE);
        goto REALIZECOLTABLE_ERR_EXIT;
    }

    if (!fRealizeSupported)
    {
        /**************************************************************/
        /* We dont support realizable color tables at all bits per    */
        /* pel.  If this is one where we dont support them, we log a  */
        /* warning but return success.                                */
        /**************************************************************/
        LOGWARNING(NULL, "Not supported", FNULL, 0,
                                           PMERR_REALIZE_NOT_SUPPORTED);
        goto REALIZECOLTABLE_OK_EXIT;
    }

#ifdef FIREWALLS
#pragma message( __FILE__"(1052) : Warning: Can a LCT in a non direct DC be realized ?")
    if (pdc->DCIDCType != OD_DIRECT)
    {
        DebugOutput("RealizeColorTable: DC is not direct\n\r");
    }
#endif /* FIREWALLS */

    /******************************************************************/
    /* Check whether the currently loaded palette is not the default  */
    /******************************************************************/
    if (ColorTableRealized != 0)
    {
        /**************************************************************/
        /* Load the default palette                                   */
        /* We pick this up from the memory version, because this      */
        /* will always be the full size that the device supports.     */
        /**************************************************************/
        for (i = 0; i < SizeOfHWPalette; i++)
        {
            RealizedLCTPalette[i] = MemoryDeviceDefaultPalette[i];
        }
    }

    /******************************************************************/
    /* Copy RGB values from logical color table to physical color     */
    /* table                                                          */
    /******************************************************************/
    for (i = pdc->DCILowIndex;
         i <= min(pdc->DCIHighIndex, (USHORT)(SizeOfHWPalette - 1));
         i++)
    {
        if (pdc->DCIColorTable[i].PhyIndex != CLR_NOPHYINDEX )
        {
            /**********************************************************/
            /* Copy over the RGB value                                */
            /**********************************************************/
            RealizedLCTPalette[i] = pdc->DCIColorTable[i].LogRGB;
        }
    }

    /******************************************************************/
    /* Realize it !                                                   */
    /******************************************************************/
    ColorTableRealized = hdc;
    #ifndef   _8514
    LoadPaletteEntries(0,
    #else
    Load8514PaletteEntries(0,
    #endif
                           SizeOfHWPalette,
                           RealizedLCTPalette);

    /******************************************************************/
    /* Set up the realized indices for black and white now, because   */
    /* from experience we know these are common colors !              */
    /******************************************************************/
    RealizedWhiteIndex = InnerSearchRealizedLCTPalette(RGBWhite);
    RealizedBlackIndex = InnerSearchRealizedLCTPalette(RGBBlack);

REALIZECOLTABLE_OK_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);

REALIZECOLTABLE_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(ERROR_ZERO);

} /* eddc_RealizeColorTable */


/******************************************************************/
/* UnrealizeColorTable reverses RealizeColorTable by causing the  */
/* default color table to be reinstated.                          */
/******************************************************************/
DDIENTRY eddc_UnrealizeColorTable (HDC          hdc,
                                   PDC          pdcArg,
                                   ULONG        FunN)

{
    /******************************************************************/
    /* Prevent compiler warnings.                                     */
    /******************************************************************/
    IgnoreParam(hdc);

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* If the function is being called within an area or path         */
    /* definition then we raise an error and leave immediately.       */
    /******************************************************************/
    if (FunNTest(COM_AREA))
    {
        /**************************************************************/
        /* We have been called within an area definition              */
        /**************************************************************/
        LOGERR(NULL, "Invalid within area", FNULL, 0,
                                                     PMERR_INV_IN_AREA);
        goto UNREALIZECOLTABLE_ERR_EXIT;
    }

    if (FunNTest(COM_PATH))
    {
        /**************************************************************/
        /* We have been called within a path definition               */
        /**************************************************************/
        LOGERR(NULL, "Invalid within path", FNULL, 0,
                                                     PMERR_INV_IN_PATH);
        goto UNREALIZECOLTABLE_ERR_EXIT;
    }

    /******************************************************************/
    /* Only execute the code if there is a realized color table       */
    /******************************************************************/
    if (ColorTableRealized)
    {
        /**************************************************************/
        /* Since we are unrealizing a logical color table, we are     */
        /* basically stepping back into the 2.0 world of Palette      */
        /* Manager palettes, and (unrealizable) color tables.  Thus   */
        /* we simply restore the HWPalette contents in to the         */
        /* hardware.                                                  */
        /**************************************************************/

        /**************************************************************/
        /* Unrealize it !                                             */
        /**************************************************************/
        ColorTableRealized = NULL;
        #ifndef   _8514
        LoadPaletteEntries(0,
        #else
        Load8514PaletteEntries(0,
        #endif
                           SizeOfHWPalette,
                           HWPalette);
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);

UNREALIZECOLTABLE_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(ERROR_ZERO);

} /* eddc_UnrealizeColorTable */
