/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = EDDAQERY                                       */
/*                                                                    */
/*   Description     = Display Device Driver text query functions     */
/*                     DeviceQueryFontAttributes,                     */
/*                     DeviceQueryFonts                               */
/*                                                                    */
/*   Function        =                                                */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design Specification     */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_DDICOMFLAGS                                    
#include <eddinclt.h>

#include <eddacone.h>
#include <eddtcone.h>
#include <edddtypt.h>
#include <eddttypt.h>

#include <eddaextf.h>
#include <eddgextf.h>

#ifdef DBCS                                                 
#include <memman.h>
#include <eddjfm.h>
#include <eddjdef.h>
#include <eddjfont.h>

extern PFOCAFONT pResourceFont;
#endif 

extern DDTType   DDT;
extern USHORT    ResidentFonts;

#ifndef DBCS                                                
/**********************************************************************/
/* DeviceQueryFontAttributes obtains the metrics of the currently     */
/* selected font                                                      */
/**********************************************************************/
DDIENTRY edda_DeviceQueryFontAttributes (HDC           hdc,
                                         ULONG         ArgSize,
                                         PFONTMETRICS  ArgMetrics,
                                         PDC           pdcArg,
                                         ULONG         FunN)

{
    /******************************************************************/
    /* With the engine managing our font this should never be called. */
    /* Just return TRUE in case it is.                                */
    /******************************************************************/
    IgnoreParam(hdc);
    IgnoreParam(ArgSize);
    IgnoreParam(ArgMetrics);
    IgnoreParam(pdcArg);
    IgnoreParam(FunN);

    return(TRUE);
}

#else 
/**********************************************************************/
/* CopyMetrics is a subroutine which copies a set of font metrics from*/
/* a source to a target data structure. The source metrics are in     */
/* Winthorn format and the target in GPI font format.                 */
/**********************************************************************/

#define CPMET_XFORM     COM_TRANSFORM  /* means must xformed to WORLD */
#define CPMET_BOLDSIM   0x00000001     /* inc width for bold font     */
                                       /*                             */
#if CPMET_XFORM == CPMET_BOLDSIM       /* must be different!          */
  #error "Constant definition conflicts!\n"
#endif                                 /*                             */

VOID DRIVERCALL CopyMetrics (PFOCAMETRICS    SrcMetrics,
                             PFONTMETRICS    TrgMetrics,
                             ULONG           ArgSize,
                             ULONG           flOptions,
                             HDC             hdc)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG           i;                 /* Loop variable               */
    PSHORT          SrcAdd;            /* Pointer to field in source  */
                                       /* metrics structure           */
    PLONG           TrgAdd;            /* Pointer to field in target  */
                                       /* metrics structure           */
    FONTMETRICS     TargetBuffer;      /* temp buffer for target      */

    /******************************************************************/
    /* zero the entire buffer                                         */
    /******************************************************************/
    memset((PVOID)&TargetBuffer, 0, sizeof(FONTMETRICS));

    /******************************************************************/
    /* A direct copy is possible from Familyname to Codepage,         */
    /* which is 68 bytes                                              */
    /******************************************************************/
    memcpy((PVOID)TargetBuffer.szFamilyname,
           (PVOID)SrcMetrics->szFamilyname,
           ENOUGHSIZE(FONTMETRICS, usCodePage) );

    /******************************************************************/
    /* The parameters from EmHeight to MaxBaselineExt must be changed */
    /* from short to long. The number of fields is 12.  The post      */
    /* increment on TrgAdd and SrcAdd works correctly because of      */
    /* their types.                                                   */
    /******************************************************************/
    TrgAdd = &TargetBuffer.lEmHeight;
    SrcAdd = &SrcMetrics->yEmHeight;
    for (i = 0; i < 12; i++)
    {
        *TrgAdd++ = (LONG)*SrcAdd++;
    }

    /******************************************************************/
    /* The 18 values from CharSlope to Capabilities are identical     */
    /* and are all 2 byte values.                                     */
    /******************************************************************/
    memcpy((PVOID)&TargetBuffer.sCharSlope,
           (PVOID)&SrcMetrics->sCharSlope, 36);

    /******************************************************************/
    /* The 12 fields from SubscriptXSize to StrikeoutPosition must be */
    /* changed from short to long.                                    */
    /******************************************************************/
    TrgAdd = (PLONG)&TargetBuffer.lSubscriptXSize;
    SrcAdd = (PSHORT)&SrcMetrics->ySubscriptXSize;
    for (i = 0; i < 12; i++)
    {
        *TrgAdd++ = (LONG)*SrcAdd++;
    }

    /******************************************************************/
    /* The remaining field can be copied directly                     */
    /******************************************************************/
    TargetBuffer.sKerningPairs = SrcMetrics->usKerningPairs;
    TargetBuffer.sFamilyClass = SrcMetrics->sFamilyClass;

    /******************************************************************/
    /*  Set the Match field to zero                                   */
    /******************************************************************/
    TargetBuffer.lMatch = (LONG)0;

    /*================================================================*/
    /* Fix some resolution-dependent parameters                       */
    /*================================================================*/
    eddj_FixFontMetrics( &TargetBuffer );

    /*================================================================*/
    /* If BOLD simulation is required for the current font, Driver    */
    /* should increment width values in metrics by 1 dot.  I.e.,      */
    /* average char width, max char inc, and em inc.                  */
    /* (DCR077 on JP12)                                               */
    /*================================================================*/
    if (flOptions & CPMET_BOLDSIM)
      {
        if (pdc->DCICurTxtAts.cdef.fFlags & CDEF_BOLD)
          {
            TargetBuffer.lAveCharWidth ++;
            TargetBuffer.lMaxCharInc ++;
            TargetBuffer.lEmInc ++;
          }                            /* end of if:                  */
      }                                /* end of if:                  */

    /******************************************************************/
    /* All fields which are sizes must be converted from device to    */
    /* world coordinates.                                             */
    /******************************************************************/
    /* X Coordinates: lAveCharWidth, lMaxCharInc, lEmInc              */
    /******************************************************************/
    /******************************************************************/
    /* Y Coordinates: lEmHeight to lExternalLeading (8 values)        */
    /*                lMaxBaselineExt                                 */
    /******************************************************************/
    /*                lSubscriptSize to lStrikeoutPosition (12values) */
    /******************************************************************/
    if ((flOptions & CPMET_XFORM) &&
        !pdc->DCIXFrmSimple)
    {
        xform_metrics(hdc, &TargetBuffer);
    }

    /******************************************************************/
    /* Copy the transformed buffer to the user buffer but no more     */
    /* information than can legally be held in a FONTMETRICS          */
    /* structure                                                      */
    /******************************************************************/
    memcpy((PVOID)TrgMetrics, (PVOID)&TargetBuffer,
           min( ArgSize, sizeof(FONTMETRICS) ) );

    return;
}

/**********************************************************************/
/* DeviceQueryFontAttributes obtains the metrics of the currently     */
/* selected font                                                      */
/**********************************************************************/
DDIENTRY edda_DeviceQueryFontAttributes (HDC           hdc,
                                         ULONG         ArgSize,
                                         PFONTMETRICS  ArgMetrics,
                                         PDC           pdcArg,
                                         ULONG         FunN)

{
    /*================================================================*/
    /* Since we manage DBCS fonts by ourselves, it is necessary to    */
    /* handle this query also.                                        */
    /*================================================================*/
    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Presentation Driver Interface says that ArgSize is a LONG so   */
    /* check for a signed value                                       */
    /******************************************************************/
    if (ArgSize & 0x80000000)
    {
        LOGERR(TFUNC, "-ve ArgSize invalid", &ArgSize, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
        goto QUERYFONTATTRS_ERR_EXIT;
    }

    /******************************************************************/
    /* Apply the DC command mask to the command bits                  */
    /******************************************************************/
    COMMANDBITS(FunN) &= pdc->DCICommandMask;

    /******************************************************************/
    /* Copy selected font metrics into structure ArgMetrics indicates */
    /******************************************************************/
    CopyMetrics( &(pdc->CurrentFont.pFocaFont->fmMetrics),
                 ArgMetrics,
                 ArgSize,
                 (FunN & CPMET_XFORM) | CPMET_BOLDSIM,
                 hdc);

    /******************************************************************/
    /* See if we can fill in the codepage                             */
    /******************************************************************/
    if (ArgSize >= 68)
    {
        /**************************************************************/
        /* For a FocaFont pick up the code page from the metrics      */
        /**************************************************************/
        ArgMetrics->usCodePage = pdc->CurrentFont.usECodePage;
                                       /* get effective codepage      */
                                       /* since in DBCS env, fonts can*/
                                       /* be realized for any CP reg- */
                                       /* ardless its capability.     */
                                       /* (DCR for JP13)              */
    }

    /******************************************************************/
    /* See if we want to fill in the match number                     */
    /******************************************************************/
    if (ArgSize >= ENOUGHSIZE(FONTMETRICS, lMatch))
    {
      ArgMetrics->lMatch = pdc->CurrentFont.lMatch;
                                       /* now use saved match value   */
                                       /* Note: for Engine font, 0 is */
                                       /*   saved here, but must not  */
                                       /*   be problem anyway.        */
    }

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(OK);

QUERYFONTATTRS_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return(ERROR_ZERO);
}
#endif

#ifndef DBCS                                                
/**********************************************************************/
/* DeviceQueryFonts enumerates the fonts on a device. For each font on*/
/* the device the function returns the metrics of the font.           */
/* A filter can be supplied in which case font metrics are only       */
/* returned for fonts whose facename matches the filter.              */
/**********************************************************************/
DDIENTRY edda_DeviceQueryFonts (HDC           hdc,
                                ULONG         ArgOptions,
                                PCHAR         ArgFilter,
                                PFONTMETRICS  ArgMetrics,
                                ULONG         ArgSize,
                                PULONG        ArgCount,
                                PDC           pdcArg,
                                ULONG         FunN)

{
    /******************************************************************/
    /* With the engine managing our font we have nothing to do here.  */
    /* We have to set ArgCount to be the number of metrics returned   */
    /* (always 0), and we return the number of fonts left to fill in  */
    /* (also 0 as we don't have any)                                  */
    /******************************************************************/
    IgnoreParam(hdc);
    IgnoreParam(ArgOptions);
    IgnoreParam(ArgFilter);
    IgnoreParam(ArgMetrics);
    IgnoreParam(ArgSize);
    IgnoreParam(pdcArg);
    IgnoreParam(FunN);

    *ArgCount = 0;
    return(0);
}

#else 
/**********************************************************************/
/* DeviceQueryFonts enumerates the fonts on a device. For each font on*/
/* the device the function returns the metrics of the font.           */
/* A filter can be supplied in which case font metrics are only       */
/* returned for fonts whose facename matches the filter.              */
/**********************************************************************/
DDIENTRY edda_DeviceQueryFonts (HDC           hdc,
                                ULONG         ArgOptions,
                                PCHAR         ArgFilter,
                                PFONTMETRICS  ArgMetrics,
                                ULONG         ArgSize,
                                PULONG        ArgCount,
                                PDC           pdcArg,
                                ULONG         FunN)

{
    /******************************************************************/
    /* Local variables                                                */
    /******************************************************************/
    ULONG             cMetricsReturned;
    ULONG             cMetricsNotReturned;
    ULONG             i;
    PFOCAMETRICS      pFontMetrics;

    USHORT            usNumDBCSFont;   /* number of DBCS Font Mgr Font*/
    PFOCAFONT         pFont;           /* ptr to font segment         */
    PFMFONTINFO       pFMFontInfo;     /* ptr to buffer rtn'd from FM */
    ULONG             ulOptions;

    /******************************************************************/
    /* Get driver semaphore and perform entry checks                  */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Presentation Driver Interface says that ArgSize is a LONG so   */
    /* check for a signed value                                       */
    /******************************************************************/
    if (ArgSize & 0x80000000)
    {
        LOGERR(TFUNC, "-ve ArgSize invalid", &ArgSize, 1,
                                             PMERR_INV_LENGTH_OR_COUNT);
        goto QUERYFONTS_ERR_EXIT;
    }

    /******************************************************************/
    /* Apply the DC command mask to the command bits                  */
    /******************************************************************/
    COMMANDBITS(FunN) &= pdc->DCICommandMask;

    /******************************************************************/
    /* Initialize count of metrics returned and metrics not returned  */
    /* to be zero.                                                    */
    /* These value will be returned if private fonts are being        */
    /* queried - we never have any of these!                          */
    /******************************************************************/
    cMetricsReturned = 0;
    cMetricsNotReturned = 0;

    /*================================================================*/
    /* Check SBCS Resource font: note it is public...                 */
    /*================================================================*/
    if (ArgOptions & QF_PUBLIC)        /*                             */
      {                                /*                             */
            /**********************************************************/
            /* If there is no filter, then the font metrics should    */
            /* be returned.                                           */
            /* If there is a filter, then it must match the font      */
            /* facename if we are to return it's metrics.             */
            /* Note that strcmp returns zero to indicate a match.     */
            /**********************************************************/
            if ( (ArgFilter == FNULL) ||
                 (strcmp( ArgFilter,
                        pResourceFont->fmMetrics.szFacename ) == 0) )
            {
                /******************************************************/
                /* Only return the metrics if there is room in the    */
                /* external buffer.                                   */
                /******************************************************/
                if (cMetricsReturned < *ArgCount)
                {
                    /**************************************************/
                    /* There is room. Increment the count, then copy  */
                    /* the metrics across.                            */
                    /**************************************************/
                    cMetricsReturned++;

                    CopyMetrics (&pResourceFont->fmMetrics,
                                 ArgMetrics,
                                 ArgSize,
                                 FunN & CPMET_XFORM,
                                 hdc);

                    /**************************************************/
                    /* Set Match number to negative of font number    */
                    /* (Match number is last field in font metrics).  */
                    /**************************************************/
                    if (ArgSize >= ENOUGHSIZE(FONTMETRICS, lMatch))
                    {
                        ArgMetrics->lMatch = LMATCH_SYS_SBCS;
                    }

                    /*================================================*/
                    /* force codepage=0 when codepage=UGL CP          */
                    /*    (DCR on JP13)                               */
                    /*================================================*/
                    if ((ArgSize >= ENOUGHSIZE(FONTMETRICS, usCodePage))
                     && (ArgMetrics->usCodePage == CP_UGL_FONT))
                      {                /* if codepage is required     */
                                       /*   and 850,                  */
                        ArgMetrics->usCodePage = 0;
                                       /* change it to 0!             */
                      }                /* end of if:                  */

                    /**************************************************/
                    /* Move the destination pointer onto next metrics */
                    /* structure.                                     */
                    /**************************************************/
                    (PBYTE)ArgMetrics += ArgSize;
                }
                else
                {
                    /**************************************************/
                    /* There is no room in the external buffer -      */
                    /* increment the count of the Metrics that we     */
                    /* can't copy.                                    */
                    /**************************************************/
                    cMetricsNotReturned++;
                }
            }

      }                                /* end of if:                  */

    /*================================================================*/
    /* DBCS Font Mgr fonts may be private; so let FM decide it.       */
    /*================================================================*/
    if (CHECKENV(ENV_DBCS_CAPABLE))    /* if have DBCS font also...   */
      {
        /**************************************************************/
        /* now we will check DBCS font manager's font                 */
        /**************************************************************/
        ulOptions = (ArgOptions & QF_PUBLIC ? FM_PUBLIC : 0) |
                    (ArgOptions & QF_PRIVATE ? FM_PRIVATE : 0);

        usNumDBCSFont = eddj_GetFontList( &pFMFontInfo, ulOptions );
        for (i = 0; i < usNumDBCSFont; i++)
        {
            if ( pFMFontInfo[i].lMatch == LMATCH_SYS_SBCS )
              {                        /* if it is SBCS Resource font */
                continue;              /* it is already counted       */
              }

            /**********************************************************/
            /* Get access to the font metrics                         */
            /**********************************************************/
//          pFont = SegToFlat( pFMFontInfo[i].pFont );
//                                     /* need to convert to flat ptr */
            pFont = pFMFontInfo[i].pFont;
                                       /* not need to convert ptr     */
            pFontMetrics = &(pFont->fmMetrics);

            /**********************************************************/
            /* If there is no filter, then the font metrics should    */
            /* be returned.                                           */
            /* If there is a filter, then it must match the font      */
            /* facename if we are to return it's metrics.             */
            /* Note that strcmp returns zero to indicate a match.     */
            /**********************************************************/
            if ( (ArgFilter == FNULL) ||
                 (strcmp( (PVOID)ArgFilter,
                          (PVOID)pFontMetrics->szFacename ) == 0) )
            {
                /******************************************************/
                /* Only return the metrics if there is room in the    */
                /* external buffer.                                   */
                /******************************************************/
                if (cMetricsReturned < *ArgCount)
                {
                    /**************************************************/
                    /* There is room. Increment the count, then copy  */
                    /* the metrics across.                            */
                    /**************************************************/
                    cMetricsReturned++;

                    CopyMetrics (pFontMetrics,
                                 ArgMetrics,
                                 ArgSize,
                                 FunN & CPMET_XFORM,
                                 hdc);

                    /**************************************************/
                    /* Copy match number from returned buffer         */
                    /**************************************************/
                    if (ArgSize >= ENOUGHSIZE(FONTMETRICS, lMatch))
                    {
                        ArgMetrics->lMatch = pFMFontInfo[i].lMatch;
                    }

                    /*================================================*/
                    /* force codepage=0 when codepage=UGL CP          */
                    /*    (DCR on JP13)                               */
                    /*================================================*/
                    if ((ArgSize >= ENOUGHSIZE(FONTMETRICS, usCodePage))
                     && (ArgMetrics->usCodePage == CP_UGL_FONT))
                      {                /* if codepage is required     */
                                       /*   and 850,                  */
                        ArgMetrics->usCodePage = 0;
                                       /* change it to 0!             */
                      }                /* end of if:                  */

                    /**************************************************/
                    /* Move the destination pointer onto next metrics */
                    /* structure.                                     */
                    /**************************************************/
                    (PBYTE)ArgMetrics += ArgSize;
                }
                else
                {
                    /**************************************************/
                    /* There is no room in the external buffer -      */
                    /* increment the count of the Metrics that we     */
                    /* can't copy.                                    */
                    /**************************************************/
                    cMetricsNotReturned++;
                }
            }
        }

//      eddj_Free16Segment( pFMFontInfo );
        if (usNumDBCSFont)             /* only if font mgr returns    */
          {                            /*   some font...              */
            FreeMemory( pFMFontInfo );
          }                            /* end of if:                  */
                                       /* discard returned buffer     */
      }                                /* end of if:                  */

    /******************************************************************/
    /* Set ArgCount to number of metrics returned.                    */
    /******************************************************************/
    *ArgCount = cMetricsReturned;

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* The return value is the number of metrics that should have    */
    /* been returned, but couldn't be because the output buffer was   */
    /* too small.                                                     */
    /******************************************************************/
    return(cMetricsNotReturned);

QUERYFONTS_ERR_EXIT:
    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    return GPI_ALTERROR;
}
#endif 
