/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/

/**********************************************************************/
/*                                                                    */
/*   File            = EDDADC                                         */
/*                                                                    */
/*   Description     = Display device driver minor functions          */
/*                     DeviceSetDCOrigin                              */
/*                     GetDCOrigin                                    */
/*                                                                    */
/*   Function        = DeviceSetDCOrigin sets a new DC origin.        */
/*                     GetDCOrigin returns the current DC origin      */
/*                                                                    */
/*   Reference       = Winthorn Functional Specification              */
/*                     Device Driver Interface Specification          */
/*                     Display Device Driver Design                   */
/*                                                                    */
/*                                                                    */
/**********************************************************************/


#define INCL_DOSMEMMGR
#define INCL_DDICOMFLAGS
#define INCL_DOSSEMAPHORES
#include <eddinclt.h>
#include <eddaextf.h>
#include <eddgextf.h>

#include <semaphor.h>

/**********************************************************************/
/* DeviceSetDCOrigin sets the new DC origin.                          */
/* It also updates those DC data items that are in AI co-ordinates,   */
/* which depend on the current DC origin.                             */
/**********************************************************************/

DDIENTRY edda_DeviceSetDCOrigin (HDC           hdc,
                                 PPOINTL       ArgOrigin,
                                 PDC           pdcArg,
                                 ULONG         FunN)

{
#define TFUNC "edda_DeviceSetDCOrigin"

    /******************************************************************/
    /* Get driver semaphore and perform entry checks.                 */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    PROFILE_ENTER(48);

    /******************************************************************/
    /* update all dc fields affected by a dc origin change            */
    /******************************************************************/
    make_dc_origin_adjustments(pdc, ArgOrigin);

    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);
    PROFILE_EXIT(48);
    return(OK);

} /* DeviceSetDCOrigin */
#undef TFUNC





VOID PASCAL NEAR make_dc_origin_adjustments(PDC     pdcAdjust,
                                            PPOINTL new_origin)

{

    POINTS offset;

    /******************************************************************/
    /* Calculate the difference between the old and the new origins   */
    /******************************************************************/
    offset.x = pdcAdjust->DCIOrigin.X - (SHORT)new_origin->x;
    offset.y = pdcAdjust->DCIOrigin.Y - (SHORT)new_origin->y;

    if (offset.x == 0 && offset.y == 0)
    {
        /**************************************************************/
        /* Both offsets are zero so there is no work to do.           */
        /**************************************************************/
        return;
    }

    /******************************************************************/
    /* Adjust the current position (in AI co-ords) to take account of */
    /* the change in DC Origin.                                       */
    /* Note that the current world coordinate position does not       */
    /* change.                                                        */
    /******************************************************************/
    pdcAdjust->DCICurrPosAI.Y += offset.y;
    pdcAdjust->DCICurrPosAI.X -= offset.x;

    /******************************************************************/
    /* Adjust the user bounds (which are in device coords) to         */
    /* ensure that they are correct when returned in screen coords.   */
    /******************************************************************/
    pdcAdjust->DCIUserBounds[0].X += offset.x;
    pdcAdjust->DCIUserBounds[1].X += offset.x;
    pdcAdjust->DCIUserBounds[0].Y += offset.y;
    pdcAdjust->DCIUserBounds[1].Y += offset.y;

    /******************************************************************/
    /* Adjust the pattern origin (in AI co-ords) to take account of   */
    /* the change in the DC Origin.                                   */
    /******************************************************************/
    pdcAdjust->DCIPatternOrigin.X -= offset.x;
    pdcAdjust->DCIPatternOrigin.Y += offset.y;

    /******************************************************************/
    /* Adjust the AI-Dev conversion factor                            */
    /******************************************************************/
    pdcAdjust->DCIConvFactor += offset.y;

    /******************************************************************/
    /* Mark correlation rectangles as invalid                         */
    /******************************************************************/
    pdcAdjust->DCICorrInvalid = TRUE;

    /******************************************************************/
    /* Set the new DC origin                                          */
    /******************************************************************/
    pdcAdjust->DCIOrigin.X -= offset.x;
    pdcAdjust->DCIOrigin.Y -= offset.y;

} /* make_dc_origin_adjustments */





/**********************************************************************/
/* GetDCOrigin returns the current DC origin.                         */
/**********************************************************************/

DDIENTRY edda_GetDCOrigin (HDC           hdc,
                           pWcsPoint     ArgOrigin,
                           PDC           pdcArg,
                           ULONG         FunN)

{
#define TFUNC "edda_GetDCOrigin"
    /******************************************************************/
    /* Get driver semaphore and perform entry checks.                 */
    /******************************************************************/
    EnterDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    /******************************************************************/
    /* Return the current DC origin from DCIData                      */
    /******************************************************************/
    ArgOrigin->X = (LONG)pdc->DCIOrigin.X;
    ArgOrigin->Y = (LONG)pdc->DCIOrigin.Y;

    /******************************************************************/
    /* Release driver semaphore                                       */
    /******************************************************************/
    ExitDriver(pdcArg, FunN, EDF_STANDARD | EDF_DONT_CLEAN);

    return(OK);
}
#undef TFUNC
