/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1995 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 WARP source code is provided to you solely for  */
/*    the purpose of assisting you in your development of OS/2 WARP device   */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Device Driver Source Kit for OS/2. This  */
/*    Copyright statement may not be removed.                                */
/*                                                                           */
/*****************************************************************************/
/**********************************************************************/
/*                                                                    */
/*   Module          = DCAFCNVT.C                                     */
/*                                                                    */
/*   Description     = Internal functions to set up DCAF conversion   */
/*                     tables.                                        */
/*                                                                    */
/*                                                                    */
/**********************************************************************/
#define INCL_DDIMISC
#include <eddinclt.h>
#include <memman.h>

#include <edddtypt.h>   /* required by eddhtype.h */
#include <eddhcone.h>   /* required by eddhtype.h */
#include <eddhtype.h>   /* required by convfuns.h */

#include <convfuns.h>

extern RGB2     FullSizeDeviceDefaultPalette[];
extern RGB2     Reduced16DeviceDefaultPalette[];
extern RGB2     HWPalette[];

ULONG DRIVERCALL NearestRestrictedColourIndex(PBYTE  pbColourTable,
                                              ULONG  ulColTabIncrement,
                                              RGB2   rgb,
                                              ULONG  ulMaxIndex);

/**********************************************************************/
/* Define the colors in the Default VGA palette.                      */
/**********************************************************************/
ULONG StandardVGADefaultPalette[16] =
{
    0x00000000,
    0x00000080,
    0x00008000,
    0x00008080,
    0x00800000,
    0x00800080,
    0x00808000,
    0x00808080,
    0x00cccccc,
    0x000000ff,
    0x0000ff00,
    0x0000ffff,
    0x00ff0000,
    0x00ff00ff,
    0x00ffff00,
    0x00ffffff
};

/**********************************************************************/
/* Define a macro to swap a pair of bytes within a word.              */
/* This is used to convert 16bpp values from Intel format to          */
/* Motorola format.                                                   */
/**********************************************************************/
#define SWAP_BYTES(x) x = (((x) & 0x00FF)<<8) | (((x) & 0xFF00)>>8)


/**********************************************************************/
/* FreeConvertTable frees up the memory of the supplied convert       */
/* table.                                                             */
/**********************************************************************/
VOID    FreeConvertTable(PBYTE  *ppConvertTable)
{
    FreeMemory(*ppConvertTable);
    *ppConvertTable = NULL;
}

/**********************************************************************/
/* CreateConvertTable_8int_4ext creates a 256-entry table of bytes to */
/* convert 8bpp to 4bpp.                                              */
/**********************************************************************/
PBYTE CreateConvertTable_8int_4ext()
{
    int     i;
    PBYTE   pConvertTable;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(BYTE),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 256 ;i++ )
        {
            pConvertTable[i] = NearestRestrictedColourIndex(
                                        (PBYTE)StandardVGADefaultPalette,
                                        sizeof(RGB2),
                                        HWPalette[i],
                                        16L );
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}


/**********************************************************************/
/* CreateConvertTable_4ext_8int creates a 16-entry table of bytes to  */
/* convert 4bpp to 8bpp.                                              */
/**********************************************************************/
PBYTE CreateConvertTable_4ext_8int()
{
    int     i;
    PBYTE   pConvertTable;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 16*sizeof(BYTE),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 16 ;i++ )
        {
            pConvertTable[i] = NearestRestrictedColourIndex(
                                  (PBYTE)FullSizeDeviceDefaultPalette,
                                  sizeof(RGB2),
                                  ((PRGB2)StandardVGADefaultPalette)[i],
                                  256L );
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}

/**********************************************************************/
/* CreateConvertTable_4ext_16int creates a 16-entry table of words to */
/* convert 4bpp to 16bpp.                                             */
/**********************************************************************/
PBYTE CreateConvertTable_4ext_16int()
{
    int     i;
    PUSHORT pConvertTable;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 16*sizeof(USHORT),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 16 ;i++ )
        {
            pConvertTable[i] =
                   RGB16FromPRGB2((PRGB2)&StandardVGADefaultPalette[i]);

            /**********************************************************/
            /* The 16bpp value is currently in Intel format. i.e.     */
            /*   Bit7..Bit 0   Bit15..Bit8                            */
            /*     BYTE 0        BYTE 1                               */
            /*                                                        */
            /* Swap the bytes so that the RGB16 is in Motorola format */
            /* i.e. :                                                 */
            /*   Bit15...Bit8   Bit7..Bit0                            */
            /*      BYTE 0        BYTE 1                              */
            /*                                                        */
            /**********************************************************/
            SWAP_BYTES(pConvertTable[i]);
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}


/**********************************************************************/
/* CreateConvertTable_8ext_16int creates a 256-entry table of words   */
/* to convert 8bpp external to 16bpp internal.                        */
/**********************************************************************/
PBYTE CreateConvertTable_8ext_16int()
{
    int     i;
    PUSHORT pConvertTable;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(USHORT),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 256 ;i++ )
        {
            pConvertTable[i] =
                RGB16FromPRGB2((PRGB2)&FullSizeDeviceDefaultPalette[i]);

            /**********************************************************/
            /* The 16bpp value is currently in Intel format. i.e.     */
            /*   Bit7..Bit 0   Bit15..Bit8                            */
            /*     BYTE 0        BYTE 1                               */
            /*                                                        */
            /* Swap the bytes so that the RGB16 is in Motorola format */
            /* i.e. :                                                 */
            /*   Bit15...Bit8   Bit7..Bit0                            */
            /*      BYTE 0        BYTE 1                              */
            /*                                                        */
            /**********************************************************/
            SWAP_BYTES(pConvertTable[i]);
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return((PBYTE)pConvertTable);
}


#ifdef S3
/**********************************************************************/
/* CreateConvertTable_4ext_24int creates a 16-entry table of words to */
/* convert 4bpp to 24bpp.                                             */
/**********************************************************************/
PBYTE CreateConvertTable_4ext_24int()
{
    int     i;
    PULONG  pConvertTable;
    BYTE    temp;
    RGB2    Color24;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 16*sizeof(ULONG),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 16 ;i++ )
        {
            Color24 = *((PRGB2)&StandardVGADefaultPalette[i]);
            pConvertTable[i] = *((PULONG)&Color24);
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}


/**********************************************************************/
/* CreateConvertTable_8ext_24int creates a 256-entry table of words   */
/* to convert 8bpp external to 24bpp internal.                        */
/**********************************************************************/
PBYTE CreateConvertTable_8ext_24int()
{
    int     i;
    PULONG  pConvertTable;
    RGB2    Color24;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(ULONG),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 256 ;i++ )
        {
            Color24 = *((PRGB2)&FullSizeDeviceDefaultPalette[i]);
            pConvertTable[i] = *((PULONG)&Color24);
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return((PBYTE)pConvertTable);
}

#endif


/**********************************************************************/
/* CreateConvertTable_4int_4ext creates a 256-entry table of bytes to */
/* convert 4bpp internal (fudged) format to 4bpp external (VGA) format*/
/* We convert pairs of pels at a time, which is why the conversion    */
/* table is 256 entries, rather than the expected 16.                 */
/**********************************************************************/
PBYTE CreateConvertTable_4int_4ext()
{
    int     i;
    PBYTE   pConvertTable;
    BYTE    abLocalConvertTable[16];

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(BYTE),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        /**************************************************************/
        /* First build up a 16-entry table which contains our basic   */
        /* 4bpp conversion values.                                    */
        /**************************************************************/
        for (i = 0; i < 16 ;i++ )
        {
            abLocalConvertTable[i] =
                 NearestRestrictedColourIndex(
                                     (PBYTE)StandardVGADefaultPalette,
                                     sizeof(RGB2),
                                     Reduced16DeviceDefaultPalette[i],
                                     16L );
        }
        /**************************************************************/
        /* Now combine pairs of conversion values into the main 256   */
        /* entry conversion table.                                    */
        /**************************************************************/
        for (i = 0; i < 256; i++)
        {
            pConvertTable[i] = (abLocalConvertTable[i >> 4] << 4) |
                               (abLocalConvertTable[i & 0x0F]) ;
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}

/**********************************************************************/
/* CreateConvertTable_4ext_4int creates a 256-entry table of bytes to */
/* convert 4bpp external (VGA) format to 4bpp internal (fudged).      */
/* We convert pairs of pels at a time, which is why the conversion    */
/* table is 256 entries, rather than the expected 16.                 */
/**********************************************************************/
PBYTE CreateConvertTable_4ext_4int()
{
    int     i;
    PBYTE   pConvertTable;
    BYTE    abLocalConvertTable[16];

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(BYTE),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        /**************************************************************/
        /* First build up a 16-entry table which contains our basic   */
        /* 4bpp conversion values.                                    */
        /**************************************************************/
        for (i = 0; i < 16 ;i++ )
        {
            abLocalConvertTable[i] =
               NearestRestrictedColourIndex(
                                  (PBYTE)Reduced16DeviceDefaultPalette,
                                  sizeof(RGB2),
                                  ((PRGB2)StandardVGADefaultPalette)[i],
                                  16L );
        }

        /**************************************************************/
        /* Now combine pairs of conversion values into the main 256   */
        /* entry conversion table.                                    */
        /**************************************************************/
        for (i = 0; i < 256; i++)
        {
            pConvertTable[i] = (abLocalConvertTable[i >> 4] << 4) |
                               (abLocalConvertTable[i & 0x0F]) ;
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}

/**********************************************************************/
/* CreateConvertTable_8int_8ext creates a 256-entry table of bytes to */
/* convert 8bpp internal indices to 8bpp external indices.            */
/*                                                                    */
/* If the XGA hardware palette contains the default value then this   */
/* table will not be required (and will not be used - the values      */
/* will be directly compressed).                                      */
/**********************************************************************/
PBYTE CreateConvertTable_8int_8ext()
{
    int     i;
    PBYTE   pConvertTable;

    /******************************************************************/
    /* Allocate memory for the conversion table                       */
    /******************************************************************/
    pConvertTable = AllocateMemory( 256*sizeof(BYTE),
                                    MT_CONVERT_TABLE,
                                    MO_SHARED );

    /******************************************************************/
    /* If the memory was successfully allocated then go ahead and     */
    /* initialise the table.                                          */
    /******************************************************************/
    if (pConvertTable != NULL)
    {
        for (i = 0; i < 256 ;i++ )
        {
            pConvertTable[i] =
               NearestRestrictedColourIndex(
                                  (PBYTE)FullSizeDeviceDefaultPalette,
                                  sizeof(RGB2),
                                  HWPalette[i],
                                  256L );
        }
    }

    /******************************************************************/
    /* Return the conversion table pointer.                           */
    /******************************************************************/
    return(pConvertTable);
}
