;/*****************************************************************************
;*
;* SOURCE FILE NAME =   THUNKMAC.INC
;*
;* DESCRIPTIVE NAME =   16 to 32 bit thunking support
;*
;*
;* COPYRIGHT    COPYRIGHT IBM CORPORATION, 1998
;*              LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
;*              REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
;*              RESTRICTED MATERIALS OF IBM
;*              IBM CONFIDENTIAL
;*
;* VERSION      V1.0
;*
;* DATE
;*
;* DESCRIPTION :
;*
;*
;* CHANGE ACTIVITY =
;*   DATE      FLAG        APAR   CHANGE DESCRIPTION
;*   --------  ----------  -----  --------------------------------------
;*   mm/dd/yy  @Vnnnnn     xxxxx  xxxxxxx
;*****************************************************************************/

CODE32          segment dword use32 public 'CODE'
        extrn   DOSFLATDS:word           
        extrn   _DiskDDHeader:byte
CODE32          ends

;----------------------------------------------------------------------------
; 16 to 32 bit thunk support macros
;
; A 16 to 32 bit thunk routine can be constructed using the following macros:
;
;    ThunkPrologue  - Prologue code for the thunk routine
;    ThunkEpilogue  - Epilogue code for the thunk routine
;    PushParm  i    - Utility macro to push ith parameter on stack
;
; The structure of a thunk routine is:
;
;    ThunkPrologue  _myFunc
;
;       ; Perform any parameter thunking and push parms on stack
;       ;
;       PushParm 1
;       PushParm 2
;       ...
;
;       ; Call the 32 bit routine
;       ;
;       call  myFunc32
;
;    ThunkEpilogue  _myFunc
;
; The calling convention for the 32 bit routines is Pascal, so parameters should 
; be pushed on stack in order from first to last.  This convention can be changed
; on a routine to routine basis by changing the parameter handling logic.
; The PushParm macro is just a convenience for routines whose parameters are all
; 32 bit DWORD in size.
;
; The resulting thunk routine is a near 16-bit function with C calling conventions,
; callable from Microsoft C 6 (register saving conventions tailored to MSC, but
; could be altered to support other compilers).
;
;----------------------------------------------------------------------------

;----------------------------------------------------------------------------
; ThunkPrologue  fname
;
; Macro for thunk routine prologue.
;
; Parameters:
;    fname  -  Name of the routine
;
; Implementation:
;    - Set up stack frame, ESP, EBP
;    - Align stack on DWORD boundary
;    - Save caller's registers according to calling environment's conventions
;      (currently Microsoft C 6)
;    - Switch to 32 bit code segment
;    - Move 32 bit flat DS segment into DS, ES
;    - Thunk stack to 32 bit. NOTE: This is currently NOT done. See Issues.
;
; Issues:
;    On fast path IO (no LVM filtering such as Bad Block Relocation needed),
;    not sure it is necessary or desirable to thunk the stack to 32 bit.
;    If it is possible to write this path without requiring a 32 bit stack
;    (i.e. don't pass address of local var as parameter), then we shouldn't
;    incur the overhead.  Then, logic paths that desire a 32 bit stack can
;    do their own switching.
;
;----------------------------------------------------------------------------

ThunkPrologue macro fname,ptype
        local   lbl32

        public  fname
        align 4
fname           proc   ptype

        ; Set up stack frame.
        ; Make sure high order word of esp, ebp is 0 for 32 bit code.
        ;
        push    bp
        and     esp, 0ffffh
        mov     ebp, esp

        ; Align stack on DWORD boundary
        ;
        and     sp, not 03h

        ; Save caller's registers for the environment from which we're called.
        ; Called from MSC code: Save SI,DI,DS (and SS,BP already preserved)
        ; 
        push    si
        push    di
        push    ds
        push    es                      ; just to keep stack dword aligned

        ; Switch to 32 bit code segment
        ;
        jmp     far ptr FLAT:lbl32

        ; May need to add 1 to 3 bytes here to align the continuation
        ; of this segment (at label lbl16) on a DWORD boundary.
        ; You will get an assembler phase error at lbl16 if not aligned.
        ; I don't know why the align 4 doesn't do this.
        ;
        db      2 dup (0) 

CODE32  SEGMENT DWORD PUBLIC USE32 'CODE'
        ASSUME  CS:FLAT
        PUBLIC  lbl32

lbl32   label   near

        ; Set up flat ES, DS
        ;
        mov     eax, offset DOSFLATDS
        mov     ds, ax
        mov     es, ax
        ASSUME  DS:FLAT
        ASSUME  ES:FLAT

        ; Thunk to 32 bit stack
        ;
;;;        call   KernThunkStackTo32
endm

;----------------------------------------------------------------------------
; ThunkEpilogue  fname
;
; Macro for thunk routine epilogue.
;
; Parameters:
;    fname  -  Name of the routine
;
; Implementation:
;    - Thunk stack back to 16 bit. NOTE: This is currently NOT done.
;      See Issues in comments for ThunkPrologue macro.
;    - Switch to 16 bit code segment
;    - Restore caller's registers, restore stack, and return
;
;----------------------------------------------------------------------------

ThunkEpilogue macro fname
        local   lbl16

        ; Thunk to 16 bit stack
        ;
;;;        call   KernThunkStackTo16

        ; Switch back to 16 bit code segment
        ;
        jmp     far ptr _TEXT:lbl16
CODE32  ends


        ; This align is needed to avoid assembler phase error.
        ; See also, ThunkPrologue for phase error avoidance code.
        ;
        align   4       

        ASSUME  CS:_TEXT
        ASSUME  DS:_DATA
        ASSUME  ES:_DATA
lbl16   label   far

        ; Restore caller's registers, restore stack, and return
        ;
        pop     es
        pop     ds
        pop     di
        pop     si
        leave
        ret
fname           endp
endm

;----------------------------------------------------------------------------
; PushParm  i
;
; Macro for accessing a DWORD parameter and re-pushing it on the stack
; in preparation for calling the 32-bit routine.
;
; Parameters:
;    i  -  Index of parameter in the parameter list.
;
; This macro is only intended for use by routines whose parameters are
; ALL 32 bit DWORD in size.  It accesses the 'i'th parameter passed to
; this routine, and re-pushes it on the stack.
;
;----------------------------------------------------------------------------

PushParm    macro  i
        push    [ebp + (i * 4)]
endm

;----------------------------------------------------------------------------
; Linear off
; 
; Macro to update a DWORD parameter (off), which contains the near address of
; an item in the default data segment, to a linear address.
;
; Accomplished by adding the start of the default data segment (_DiskDDHeader).
;
;----------------------------------------------------------------------------

Linear      macro  off 
        add     off, offset FLAT:_DiskDDHeader
endm

;----------------------------------------------------------------------------
; LinToOff linear
; 
; Macro to update a DWORD parameter (linear), which contains the linear address of
; an item in the default data segment, to the offset (near address) of the item.
;
; Accomplished by subtracting the start of the default data segment (_DiskDDHeader).
;
;----------------------------------------------------------------------------

LinToOff    macro  lin 
        sub     lin, offset FLAT:_DiskDDHeader
endm

