/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 *
 *   OCO Source Materials
 *
 *   Program number (when available)
 *
 *
 *   The source code for this program is not published or otherwise divested of its
 *   tradesecrets, irrespective of what has been deposited with the U.S. Copyright Office.
 *
 ****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = DMTRAC32.C
 *
 * DESCRIPTIVE NAME = OS2DASD.DMD - OS/2 DASD Device Manager
 *
 * DESCRIPTION  32 bit trace support
 *
 *
*/
#include "dmh32.h"
#include "rqd.h"
#include "perfhook.h"
#include "dmtrac32.h"

/*------------------------------------------------------------------------
;** TraceStrat3Pre - Trace Strategy-3 Pre-Invocation
;
;   ENTRY:    pRLCB            - pointer to RLCB (Req List Chain Block)
;             pVolCB           - pointer to VolCB
;             TraceEnabled     - Which type of tracing is enabled
;
;   RETURN:   VOID
;
;   FUNCTION: Traces RLH and RLE(s) on entry to Strat3
;
;   NOTES:    1) Only called if tracing is enabled
;             2) Must be called on a FLAT stack.
------------------------------------------------------------------------*/
VOID TraceStrat3Pre (PRLCB pRLCB, PVOLCB pVolCB, USHORT TraceEnabled)
{
   UCHAR  TraceBuffer[TCBNewSize];
   PTCB   pTraceBuffer = (PTCB)TraceBuffer;
   UCHAR  UnitChar;

   PReq_List_Header pRLH;
   PPB_Read_Write  pRLE;
   PRENODE pRENode;
   PUNITCB pUnitCB = linear(pVolCB->pUnitCB);

   /* Table mapping RLE Command to a Letter for tracing */
   static char RLECmdToLetter[] = {
      'X',                   // PB_READV_X      0x1D
      'R',                   // PB_READ_X       0x1E
      'W',                   // PB_WRITE_X      0x1F
      'V',                   // PB_WRITEV_X     0x20
      'P'                    // PB_PREFETCH_X   0x21
   };
#define MapRLECmdToLetter(cmd)  RLECmdToLetter[(cmd) - PB_READV_X]

   /*
    * Trace Request List Header (RLH)
    */
   pRLH = pRLCB->pRLHeader;
   ((PTRLHS)pTraceBuffer)->pRLH = (PBYTE)pRLH;
   ((PTRLHS)pTraceBuffer)->Count = pRLH->Count;
   ((PTRLHS)pTraceBuffer)->Unit = pUnitCB->PhysDriveNum;
   ((PTRLHS)pTraceBuffer)->pRQD = pRLCB->pRQD;

   if (pRLH->Block_Dev_Unit <= MAX_LOGICAL_UNITID  &&  pVolCB->DriveLetter != 0)
      UnitChar = pVolCB->DriveLetter;
   else
      UnitChar = '?';                  /* I/O to a physical disk */

   ((PTRLHS)pTraceBuffer)->Drive[0] = UnitChar;
   ((PTRLHS)pTraceBuffer)->Drive[1] = '\0';
   ((PTRLHS)pTraceBuffer)->Request_Control = pRLH->Request_Control;

   DoTrace32(RAS_MINOR_STRAT3_RLH, sizeof(TRLHS), (PBYTE)pTraceBuffer, TraceEnabled);

   /*
    * Trace each Request List Entry (RLE)
    */
   pTraceBuffer->Drive[0] = UnitChar;
   pTraceBuffer->Drive[1] = '\0';

   for (pRENode = pRLCB->pReqElement; pRENode != NULL; pRENode = pRENode->pNextElement) {
      pRLE = pRENode->pRE;
      pTraceBuffer->Unit = pUnitCB->PhysDriveNum;
      pTraceBuffer->CommandCode = pRLE->RqHdr.Command_Code;

      pTraceBuffer->CmdString[0] = MapRLECmdToLetter(pTraceBuffer->CommandCode);
      pTraceBuffer->CmdString[1] = pTraceBuffer->CmdString[2] = '\0';

      pTraceBuffer->RequestControl = pRLE->RqHdr.Req_Control;
      pTraceBuffer->Priority = pRLE->RqHdr.Priority;
      pTraceBuffer->pRequest = (PBYTE)pRLE;
      pTraceBuffer->pRLH32 = (PBYTE)pRLH;
      pTraceBuffer->cSGList = pRLE->SG_Desc_Count;
      pTraceBuffer->RBA = pRLE->Start_Block + pVolCB->PartitionOffset +
                                              pVolCB->MediaBPB.HiddenSectors;
      pTraceBuffer->BlockCount = pRLE->Block_Count;
      pTraceBuffer->LSN = pRLE->Start_Block;

      DoTrace32(RAS_MINOR_STRAT3_RLE, TCBNewSize, (PBYTE)pTraceBuffer, TraceEnabled);
   }
}

/*------------------------------------------------------------------------
;** TraceStrat3Post - Trace Strategy-3 Post-Invocation
;
;   ENTRY:    pRLH             - pointer to RLH
;             TraceEnabled     - Which type of tracing is enabled
;
;   RETURN:   VOID
;
;   FUNCTION: Traces RLH notification back to OS2LVM
;
;   NOTES:    1) Only called if tracing is enabled
;             2) Must be called on a FLAT stack.
;             3) Assumes all RLEs have completed.
;             4) Since OS2DASD doesn't do RLE notifications anymore I
;                don't support tracing that.  If it's required then it
;                must be done in OS2LVM.
------------------------------------------------------------------------*/
VOID TraceStrat3Post (PReq_List_Header pRLH, USHORT TraceEnabled)
{
   UCHAR  TraceBuffer[sizeof(TRLHD)];
   PTRLHD pTraceBuffer = (PTRLHD)TraceBuffer;

   /*
    * RLH Notification
    */
   pTraceBuffer->pRLH = (PBYTE)pRLH;
   pTraceBuffer->DoneCount = pRLH->y_Done_Count;
   pTraceBuffer->Status = (USHORT)pRLH->Lst_Status;

   DoTrace32(RAS_MINOR_STRAT3_RLH|0x80, sizeof(TRLHD), (PBYTE)pTraceBuffer, TraceEnabled);
}

/*------------------------------------------------------------------------
;** TraceIORB32Pre  - Trace IORB Pre_Invocation Request
;
;   ENTRY:    pIORB            - ptr to first IORB in chain
;             pUnitCB          - ptr to Unit CB
;             TraceEnabled     - Which type of tracing is enabled
;
;   RETURN:   VOID
;
;   FUNCTION: Traces IORBs right before they're sent to the ADD
;
;   NOTES:    1) Only called if tracing is enabled
;             2) Must be called on a FLAT stack.
------------------------------------------------------------------------*/
VOID TraceIORB32Pre (PIORB pIORB, PUNITCB pUnitCB, USHORT TraceEnabled)
{
   UCHAR  TraceBuffer[TCBOrgSize];
   PTCB   pTraceBuffer = (PTCB) TraceBuffer;
   PIORB_DMWORK pDMWork;
   PRENODE pRENode;
   PFORMAT_CMD_TRACK pFmtCmd;

   /*
    * Map an EXECUTE_IO Command Modifier to a letter.
    */
   static char CmdModifierToLetter[] = {
      '?',
      'R',                   // IOCM_READ           0x0001
      'X',                   // IOCM_READ_VERIFY    0x0002
      'P',                   // IOCM_READ_PREFETCH  0x0003
      'W',                   // IOCM_WRITE          0x0004
      'V'                    // IOCM_WRITE_VERIFY   0x0005
   };

   /*
    * Map an IORB command, other than EXECUTE_IO, to a letter.
    */
   static char IORBCmdToLetter[] = {
      '?',
      'C',                   // IOCC_CONFIGURATION     0x0001
      'U',                   // IOCC_UNIT_CONTROL      0x0002
      'G',                   // IOCC_GEOMETRY          0x0003
      'I',                   // IOCC_EXECUTE_IO        0x0004
      'F',                   // IOCC_FORMAT            0x0005
      'S',                   // IOCC_UNIT_STATUS       0x0006
      'D',                   // IOCC_DEVICE_CONTROL    0x0007
      'A'                    // IOCC_ADAPTER_PASSTHRU  0x0008
   };

TraceNextIORB:

   pDMWork = (PIORB_DMWORK) &(pIORB->DMWorkSpace[0]);
   pRENode = (PRENODE)(pDMWork->pRequest);
   pTraceBuffer->pRequest = (PBYTE)(pRENode->pRE);
   pTraceBuffer->Unit = pUnitCB->PhysDriveNum;
   pTraceBuffer->RequestControl = pIORB->RequestControl;
   pTraceBuffer->CommandCode = (UCHAR) pIORB->CommandCode;
   pTraceBuffer->CommandModifier= (UCHAR) pIORB->CommandModifier;
   pTraceBuffer->CmdString[1] = pTraceBuffer->CmdString[2] = '\0';

   if (pTraceBuffer->CommandCode == IOCC_EXECUTE_IO) {
      pTraceBuffer->CmdString[0] = CmdModifierToLetter[pTraceBuffer->CommandModifier];
      pTraceBuffer->cSGList = ((PIORB_EXECUTEIO_X)pIORB)->cSGList;
      pTraceBuffer->RBA = ((PIORB_EXECUTEIO_X)pIORB)->RBA;
      pTraceBuffer->BlockCount=((PIORB_EXECUTEIO_X)pIORB)->BlockCount;
   }
   else {
      pTraceBuffer->CmdString[0] = IORBCmdToLetter[pTraceBuffer->CommandCode];
      if (pTraceBuffer->CommandCode == IOCC_FORMAT) {
         pTraceBuffer->cSGList = ((PIORB_FORMAT)pIORB)->cSGList;
         pFmtCmd = (PFORMAT_CMD_TRACK)((PIORB_FORMAT)pIORB)->pFormatCmd;
         pTraceBuffer->RBA = pFmtCmd->RBA;
         pTraceBuffer->BlockCount = (ULONG) pFmtCmd->cTrackEntries;
         pTraceBuffer->Flags = (UCHAR) pFmtCmd->Flags;
      }
   }

   DoTrace32(RAS_MINOR_IORB32, TCBOrgSize, (PBYTE)pTraceBuffer, TraceEnabled);

   /*
    * Follow IORB chain if necessary ...
    * Don't use pNxtIORB because that's a 16:16 virtual *NOT* in
    * our data segment and can't be (easily) converted to linear.
    * Instead, use DMWork field "Reserved_2" which contains the
    * linear address of the next IORB in the chain.
    */
   if (pIORB->RequestControl & IORB_CHAIN) {
      pIORB = (PIORB) (pDMWork->Reserved_2);
      goto TraceNextIORB;
   }
}

/*------------------------------------------------------------------------
;** TraceIORB32Post - Trace IORB Post_Invocation Request
;
;   ENTRY:    pRENode          - pointer to RE node
;             TraceEnabled     - Which type of tracing is enabled
;
;   RETURN:   VOID
;
;   FUNCTION: Traces IORB completion
;
;   NOTES:    1) Only called if tracing is enabled
;             2) Must be called on a FLAT stack.
------------------------------------------------------------------------*/
VOID TraceIORB32Post (PRENODE pRENode, USHORT TraceEnabled)
{
   UCHAR  TraceBuffer[sizeof(TCBD)];
   PTCBD  pTraceBuffer = (PTCBD)TraceBuffer;
   PIORB pIORB = (PIORB) &(pRENode->IORB);

   pTraceBuffer->pRequest = (PBYTE) pRENode->pRE;
   pTraceBuffer->Status = pIORB->Status;
   pTraceBuffer->ErrorCode = pIORB->ErrorCode;
   pTraceBuffer->BlocksXferred = ((PIORB_EXECUTEIO_X)pIORB)->BlocksXferred;

   DoTrace32(RAS_MINOR_IORB32|0x80, sizeof(TCBD), (PBYTE)pTraceBuffer, TraceEnabled);
}

/*------------------------------------------------------------------------
;** DoTrace32 - Perform Dekko/STrace/RAS tracing
;
;   VOID DoTrace32 (ULONG MinorCode, ULONG TraceSize, PBYTE pTraceBuffer, USHORT TraceEnabled)
;
;   ENTRY:    MinorCode        - Minor Code
;             TraceSize        - Size of data to trace
;             pTraceBuffer     - Pointer to trace buffer
;             TraceEnabled     - Which type of tracing is enabled
;
;   RETURN:   VOID
;
;   FUNCTION: Sends trace data (setup by callers) to either
;             - Dekko (Performance/hardware)
;             - STrace (Performance/software)
;             - RAS (RAS/software)
;
;   Notes: 1) The stack *MUST* be FLAT before calling this function.
              If it isn't KernPerfSysTrace and/or KernRASSysTrace will trap.
;          2) Caller *MUST* check if tracing is enabled
------------------------------------------------------------------------*/
VOID DoTrace32 (ULONG MinorCode, ULONG TraceSize, PBYTE pTraceBuffer, USHORT TraceEnabled)
{
   volatile struct Dekko_Addr *pDekkoBuf;
   int i;

   /*
    * Caller has (or should have) checked that either major 0x68 or 0x07 is
    * enabled ...
    * If it was 0x68 (OS2DASD Performance Major Code) then trace to either
    * Dekko (hardware) or STrace (software).
    * If it was 0x07 (OS2DASD RAS Major Code) then trace to RAS.
    */
   if (TraceEnabled & TF_DEKKO) {                               /* Performance */
      if (TEST_DEKKO((unsigned char *)_plSIS_perf_mec_table)) { /* Dekko? */
         pDekkoBuf = (struct Dekko_Addr *)(((struct InfoSegGDT *)_plSysInfoSeg)->SIS_MMIOAddr);
         pDekkoBuf->majmin_code = (USHORT)(DEKKO_MAJOR_DISK * 256 + MinorCode);
         for (i = 0; i < TraceSize/2; i++) {
            pDekkoBuf->perf_data = (USHORT) *((PUSHORT)pTraceBuffer + i);
         }
      }
      else                                                      /* No Dekko - STrace */
         KernPerfSysTrace(DEKKO_MAJOR_DISK, MinorCode, (PVOID)pTraceBuffer, TraceSize);
   } else {                                                     /* RAS */
      KernRASSysTrace(RAS_MAJOR_DISK, MinorCode, (PVOID)pTraceBuffer, TraceSize);
   }
}
