/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 *
 *   OCO Source Materials
 *
 *   Program number (when available)
 *
 *
 *   The source code for this program is not published or otherwise divested of its
 *   tradesecrets, irrespective of what has been deposited with the U.S. Copyright Office.
 *
 ****************************************************************************/
/**************************************************************************
 *
 * SCCSID: src/basedd/dasd/os2dasd/dmstrat3.c, dsdm, w45.fs32, 20000823.1 99/03/12
 *
 * SOURCE FILE NAME = DMSTRAT3.C
 *
 * DESCRIPTIVE NAME = OS2DASD.DMD - OS/2 DASD Device Manager
 *                    Strategy 3 interface for OS/2 DASD Manager
 *
 * DESCRIPTION  Processes Strategy 3 requests
 *
 *
*/
#include "dmh32.h"
#include "rqd.h"
#include "dmtrac32.h"
#include "dmtrc.h"

/* Forward declarations
 */
extern PVOLCB UnitToVolCB(USHORT unit);
extern VOID NotifyRequestError(PRLCB pRLCB, USHORT error);
extern VOID ExecReqList (PRLCB pRLCB, PVOLCB pVolCB, PUNITCB pUnitCB);
extern VOID QueueRLH(PRLCB pRLCB, PUNITCB pUnitCB);
extern VOID SubmitRLERequests(PUNITCB pUnitCB);
extern PIORB AllocIORB(VOID);
extern PIORB AllocIORBList(ULONG sendCount);
extern void FreeIORB(PIORB pIORB);
extern VOID SetupIORB(PUNITCB pUnitCB, PVOLCB pVolCB, PRENODE pRENode);
extern VOID CallADD(PUNITCB pUnitCB, PIORB pIORB);
extern BOOL NotifyRLE( PRENODE pRENode, PUNITCB pUnitCB, USHORT TraceEnabled);
extern VOID NotifyIFS32( PBYTE pReq, PVOID callback);

/* Cover defines for indirect Far16 calls
 */
#define SubmitRPRequests(pUnitCB) \
      (* pfSubmitRPs)(pUnitCB)

#define ThunkToAdd16( ADDEntry, pIORB16 ) \
      (* pfThunkToAdd16)(ADDEntry, pIORB16)

/* Return the virtual address of the IORB associated with a given RENode
 */
#define IORB_VirtualAddr(pRENode) \
   pointer_add((pRENode)->VirtualAddr, (PBYTE) &(pRENode)->IORB - (PBYTE) (pRENode))

/* Type of callback function to OS2LVM
 */
typedef VOID (* _System PCALLBACK32)(PVOID);

#ifdef DM_STRICT

#define ValidateRLE(pRLE)  Assert(pRLE->RqHdr.Old_Command == 0x1c)

/* Support to keep track of the RLCBs currently being processed.
 * Used to check for lost or duplicate RLCBs.
 * A set of the current RLCBs which have been received but not yet
 * returned is maintained.
 */
#define RLCBSET_SIZE   4

PRLCB RLCB_Set[RLCBSET_SIZE] = {0};

BOOL RLCB_Set_NoOverflow = TRUE;

SpinLock_t  RLCB_Set_Lock = {0};

VOID InsertReceivedRLCB(PRLCB pRLCB)
{
   int i;
   int free = RLCBSET_SIZE;

   Lock(&RLCB_Set_Lock);
   for (i = 0; i < RLCBSET_SIZE; i++) {
      if (RLCB_Set[i] == pRLCB) {
         /* Received same RLCB before returning it! */
         Unlock(&RLCB_Set_Lock);
         Assert(FALSE);
      }
      if (RLCB_Set[i] == NULL)
         free = i;
   }
   if (free < RLCBSET_SIZE)
      RLCB_Set[free] = pRLCB;
   else
      RLCB_Set_NoOverflow = FALSE;
   Unlock(&RLCB_Set_Lock);
}

VOID RemoveReturnedRLCB(PRLCB pRLCB)
{
   int i;
   BOOL found = FALSE;

   Lock(&RLCB_Set_Lock);
   for (i = 0; i < RLCBSET_SIZE && !found; i++) {
      if (RLCB_Set[i] == pRLCB) {
         found = TRUE;
         RLCB_Set[i] = NULL;
      }
   }
   if (!found && RLCB_Set_NoOverflow) {
      Unlock(&RLCB_Set_Lock);
      Assert(FALSE);
   }
   Unlock(&RLCB_Set_Lock);
}

/*------------------------------------------------------------------------
   ValidateRLCB

   Routine to perform consistency checks on an RLCB, which has just been
   passed to us (some checks may fail if this is called after the RLCB has
   undergone any processing).

   Uses Assert() to perform the checks.

   Parameters:
      pRLCB         - pointer to an RLCB structure  

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/
VOID ValidateRLCB(PRLCB pRLCB)
{
   PReq_List_Header pRLH = pRLCB->pRLHeader;
   PRENODE pRENode;
   ULONG nodeCount;

   /* Until     Block, the following 2 assertions should hold
    */
   Assert(pRLH->Lst_Status == 0);
   Assert(pRLH->y_Done_Count == 0);

   /* We don't handle sequential requests yet */
   Assert(!(pRLH->Request_Control & RLH_Exe_Req_Seq));

   /* Make sure the number of RENODEs matches the RLH count.
    * Also verify each RLE.
    */
   for (pRENode = pRLCB->pReqElement, nodeCount = 0;
        pRENode != NULL;
        pRENode = pRENode->pNextElement, ++nodeCount
       ) {

      PPB_Read_Write pRLE = pRENode->pRE;
      ValidateRLE(pRLE);

      /* Until     Block, the following should be true */
      Assert(pRLE->RqHdr.Status == RH_NOT_QUEUED);
   }
   Assert(nodeCount == pRLH->Count);
}

VOID Strat3Init()
{
   KernAllocSpinLock(&RLCB_Set_Lock);
}

#else
#define ValidateRLE(pRLE)
#define ValidateRLCB(pRLCB)
#define InsertReceivedRLCB(pRLCB)
#define RemoveReturnedRLCB(pRLCB)

VOID Strat3Init() {}
#endif

#ifdef DM_DEBUG
/* Note: the following variables are initialized to 1, rather than 0 to avoid
 * being placed in BSS data segment.
 */
ULONG Strat3Count = 1L;    // Keep track of number of times DMStrat3 has been called.
                           // Useful for recreateable bugs which occur after some # invocations
ULONG NumRLEsReceived = 1L;
ULONG NumRLEsReturned = 1L;
#endif

/* The following is a workaround to a compiler optimization which
 * messes up calls to KernThunkStackxx().  Sometimes the compiler optimizes
 * a call to a routine into a jump to the routine (when the call is 
 * immediately followed by a return, so folds two returns into a jump return).
 * For the stack thunking routines, this is a problem since they NEED to
 * do a return to properly finish adjusting the stack.
 * Workaround is to use 'CompilerThunkWorkaround()' immediately after the
 * call to the thunk routine, which will eliminate the compiler's optimization.
 * (note this optimization is only a problem for the thunk routines).
 */
BOOL globalWorkaround = TRUE;
#define   CompilerThunkWorkaround()  (globalWorkaround = TRUE)

/*------------------------------------------------------------------------
   DMStrat3

   Entry point for augmented strat3 requests from os2lvm.
   See header file RQD.H for a description of these requests.

   Parameters:
      pRLCB         - pointer to an RLCB structure  

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

VOID _Pascal DMStrat3(PRLCB pRLCB)
{
   PVOLCB pVolCB;
   PReq_List_Header pRLH = pRLCB->pRLHeader;
   PUNITCB pUnitCB;
   USHORT validityError;
   USHORT TraceEnabled;                 /* Whether tracing is on or off */

// _interrupt(3);

   /*
    * Performance/RAS Tracing
    */
   SetTraceState(TraceEnabled);
   if (TraceEnabled)
      DoTrace32(RAS_MINOR_OS2DASD, (ULONG)NULL, (PBYTE)NULL, TraceEnabled);

   trace(TR_CALL_DMSTRAT3, pRLCB, pRLH->Block_Dev_Unit, pRLH->Count);

#ifdef DM_DEBUG
   ++Strat3Count;
   NumRLEsReceived += pRLH->Count;
#endif

   pVolCB = UnitToVolCB(pRLH->Block_Dev_Unit);
   Assert(pVolCB);
   pUnitCB = linear(pVolCB->pUnitCB);

   /*
    * Performance/RAS Tracing
    */
   if (TraceEnabled)
      TraceStrat3Pre(pRLCB, pVolCB, TraceEnabled);

   /* Perform validity checks
    *
    * Since our clients (FS / IFSM) are trusted code, certain checks
    * are only done under the strict version.
    */
   validityError = 0;

   /* Note: when a validity error is detected, do loop is exited with a break.
    */
   do {
      ValidateRLCB(pRLCB);
      InsertReceivedRLCB(pRLCB);

      /* If this is an unassigned PRM volume, return NOT_READY
       */
      if (pVolCB->Flags & (vf_NoAssignedPartition | vf_noDisk))
         { validityError = ERROR_I24_NOT_READY;  break; }

   } while (FALSE);

   /* If request list not aborted from validate, then execute the list 
    * else notify client of error.
    */
   if ( !validityError )
      ExecReqList(pRLCB, pVolCB, pUnitCB);
   else
      NotifyRequestError(pRLCB, validityError);

   /*
    * Performance/RAS Tracing
    */
   if (TraceEnabled)
      DoTrace32(RAS_MINOR_OS2DASD|0x80, (ULONG)NULL, (PBYTE)NULL, TraceEnabled);
}

/*------------------------------------------------------------------------
   NotifyRequestError

   Notify (through callback) that the given request list is in error.
   Called when an error is detected for which the request list as a whole 
   cannot be executed.

   Parameters:
      PRLCB   pRLCB           - pointer to request list chain block
      USHORT  error           - error code to return

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

VOID NotifyRequestError (PRLCB pRLCB, USHORT error) 
{
   PRENODE pRENode;
   Req_List_Header *pRLH = pRLCB->pRLHeader;
   USHORT TraceEnabled;

   SetTraceState(TraceEnabled);

   /* Mark all RLEs as completed with error.
    */
   for (pRENode = pRLCB->pReqElement;
        pRENode != NULL;
        pRENode = pRENode->pNextElement
       ) {
      PPB_Read_Write pRLE = pRENode->pRE;
      pRLE->Blocks_Xferred = 0;
      pRLE->RqHdr.Status = RH_UNREC_ERROR;
      pRLE->RqHdr.Error_Code = (UCHAR) (STDON | STERR | error);
   }

   /* Mark the request list as done with error.
    */
   pRLH->Lst_Status &= ~(RLH_Req_Not_Queued | RLH_All_Req_Queued);
   pRLH->Lst_Status |= (RLH_All_Req_Done | RLH_Unrec_Error);


   /* Callback the client  (perform RAS before callback)
    */
   if (TraceEnabled)
      TraceStrat3Post(pRLH, TraceEnabled);

   trace(TR_CALL_CALLBACK, 0, pRENode->pRLChain, 0);

#ifdef DM_DEBUG
   NumRLEsReturned += pRLH->Count;
#endif
   RemoveReturnedRLCB(pRLCB);

   (* (PCALLBACK32)pRLH->Notify_Address)(pRLCB);
}


/*------------------------------------------------------------------------
   ExecReqList

   Process a request list.

   Parameters:
      PRLCB   pRLCB           - pointer to request list chain block
      PVOLCB  pVolCB          - pointer to associated VolCB
      PUNITCB pUnitCB         - pointer to associated UnitCB

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

VOID ExecReqList (PRLCB pRLCB, PVOLCB pVolCB, PUNITCB pUnitCB)
{
   PRENODE pRENode;
   PPB_Read_Write pRLE;
   Req_List_Header *pRLH;

   trace(TR_CALL_EXECREQLIST, pRLCB, pVolCB, 0);

   pRLH = pRLCB->pRLHeader;

   Lock(&pUnitCB->QueueLock);

   /* Fastpath logic
    */
   if (pRLH->Count == 1               &&
       pUnitCB->NumRLEWaiting == 0    &&
       pUnitCB->NumReqsInProgress < pUnitCB->QueueDepth) {

      pRLH->Lst_Status   &= ~RLH_Req_Not_Queued;
      pRLH->Lst_Status   |= RLH_All_Req_Queued;

      pRENode = pRLCB->pReqElement;
      pRENode->pRE->RqHdr.Status  = RH_QUEUED;
      pRENode->pRE->RqHdr.Waiting = 0;

      pUnitCB->NumReqsInProgress++;
      _NumReqsInProgress++;

      Unlock(&pUnitCB->QueueLock);

      SetupIORB(pUnitCB, pVolCB, pRENode);
      CallADD(pUnitCB, (PIORB) &pRENode->IORB);
      return;
   }

   /* Non-fastpath logic
    */
   QueueRLH(pRLCB, pUnitCB);
   if (pUnitCB->NumRLEWaiting)
      SubmitRLERequests(pUnitCB);    // Will Unlock UnitCB's QueueLock
   else
      Unlock(&pUnitCB->QueueLock);
}

/*------------------------------------------------------------------------
   QueueRLH

   Queue the RLEs from a given RLCB on a given UnitCB 32 bit RLE queue

   Parameters:
      PRLCB   pRLCB           - pointer to request list chain block
      PUNITCB pUnitCB         - pointer to UnitCB to queue against

   Returns:
      none

   Assumptions:
      The UnitCB's QueueLock has been locked by caller.

   Implementation:

------------------------------------------------------------------------*/

VOID QueueRLH(PRLCB pRLCB, PUNITCB pUnitCB)
{
   PRTYQ *pQueue;          
   Req_List_Header *pRLH = pRLCB->pRLHeader;
   PRENODE pRENode;

   trace(TR_CALL_QUEUERLH, pRLCB, pUnitCB, 0);

   pRLH->Lst_Status   &= ~RLH_Req_Not_Queued;
   pRLH->Lst_Status   |= RLH_All_Req_Queued;

   pQueue = &pUnitCB->FifoQRLE;

   for (pRENode = pRLCB->pReqElement;
        pRENode != NULL;
        pRENode = pRENode->pNextElement
       ) {
      PB_Read_Write *pRLE = pRENode->pRE;
      if (pRLE->RqHdr.Status == RH_NOT_QUEUED) {
         /* Queue the RLE
          */
         ValidateRLE(pRLE);
         pRLE->RqHdr.Status = RH_QUEUED;
         pRLE->RqHdr.Waiting = 0;
         if (pQueue->Tail) {
            ((PRENODE)(pQueue->Tail))->pQueued = pRENode;
            pQueue->Tail = (PBYTE) pRENode;
         }
         else {
            pQueue->Head = (PBYTE) pRENode;
            pQueue->Tail = (PBYTE) pRENode;
         }
         pUnitCB->NumRLEWaiting++;
         _NumRLEReqsWaiting++;                                                                                                                         /*@RAWIO*/

         /* If RLEs are to be executed sequentially, can only queue
          * one at a time to be sent to ADD (since no guarantee ADD will
          * execute requests it receives in order received).
          */
         if (pRLH->Request_Control & RLH_Exe_Req_Seq)
            break;
      }
   }
   /* NULL terminate the queue */
   if (pQueue->Tail)
      ((PRENODE)(pQueue->Tail))->pQueued = NULL; 
}

/*------------------------------------------------------------------------
   SubmitRLERequests

   Submit queued RLE requests for a given UnitCB to the ADD.

   Parameters:
      PUNITCB pUnitCB         - pointer to UnitCB

   Returns:
      none

   Assumptions:
      The UnitCB's QueueLock has been locked by caller.
      SubmitRLERequests() will unlock the QueueLock.

   Implementation:

------------------------------------------------------------------------*/

VOID SubmitRLERequests(PUNITCB pUnitCB)
{
   int sendCount, i;
   PIORB iorbList, pIORB;
   PRENODE nodeList, pRENode;
   PRTYQ *pQueue = &pUnitCB->FifoQRLE;
   PVOLCB pVolCB;
   PIORB_DMWORK pDMWork;

   trace(TR_CALL_SUBMITRLE, pUnitCB, pUnitCB->NumReqsInProgress, pUnitCB->NumRLEWaiting);

   /* Calculate number of requests which can be sent
    */
   sendCount = pUnitCB->QueueDepth - pUnitCB->NumReqsInProgress;
   if (pUnitCB->NumRLEWaiting < sendCount)
      sendCount = pUnitCB->NumRLEWaiting;

   /* If nothing to send, cleanup and return. */
   if (sendCount == 0) {
      Unlock(&pUnitCB->QueueLock);
      return;
   }

   /* Remove RLEs to be processed from the queue
    */
   nodeList = (PRENODE) pQueue->Head;
   for (i = 0, pRENode = nodeList;
        i < sendCount;
        i++, pRENode = pRENode->pQueued
       )
      ;
   /* pRENode points to RENODE just past last one to process. */
   pQueue->Head = (PBYTE) pRENode;
   if (pRENode == NULL)
      pQueue->Tail = NULL;

   /* Update counts
    */
   pUnitCB->NumRLEWaiting -= sendCount;
   _NumRLEReqsWaiting -= sendCount;
   pUnitCB->NumReqsInProgress += sendCount;
   _NumReqsInProgress += sendCount;

   /* Unlock UnitCB's QueueLock
    */
   Unlock(&pUnitCB->QueueLock);

   /* Setup the IORBs from the RLEs
    */
   pRENode = nodeList;
   iorbList = pIORB = (PIORB) &pRENode->IORB;

   for (i = 0; i < sendCount; i++) {
      pVolCB = UnitToVolCB(pRENode->pRLChain->pRLHeader->Block_Dev_Unit);
      Assert(pVolCB);
      SetupIORB(pUnitCB, pVolCB, pRENode);
      if (i+1 < sendCount) {
         /* IORB chaining */
         pIORB->RequestControl |= IORB_CHAIN;
         pRENode = pRENode->pQueued;             
         pIORB->pNxtIORB = IORB_VirtualAddr(pRENode);
         /*
          * Save linear address of chained (next) IORB in the
          * work area.  It is used by the Performance/RAS trace
          * code (TraceIORB32Pre) when walking the IORB chain.
          * Can't use pNxtIORB because that's a virtual *NOT* in
          * our data segment, and there's no (easy) way to
          * convert it to linear.
          */
         pDMWork = (PIORB_DMWORK) &(pIORB->DMWorkSpace);
         pDMWork->Reserved_2 = (ULONG) &(pRENode->IORB);

         pIORB = (PIORB) &pRENode->IORB;
      }
      else
         pIORB->pNxtIORB = NULL;
   }
   CallADD(pUnitCB, iorbList);
}

/*------------------------------------------------------------------------
   SetupIORB  

   Setup an IORB from a given RENODE.

   Parameters:
      PUNITCB     pUnitCB     - pointer to associated UnitCB
      PVOLCB      pVolCB      - pointer to associated VolCB
      PRENODE     pRENode     - pointer to RENODE

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

/* Table mapping RLE extended commands to IORB command modifier values.
 */
static USHORT RLEtoIORBCmdMod[] = {
   IOCM_READ_VERIFY,          // PB_READV_X      0x1D
   IOCM_READ,                 // PB_READ_X       0x1E
   IOCM_WRITE,                // PB_WRITE_X      0x1F
   IOCM_WRITE_VERIFY,         // PB_WRITEV_X     0x20
   IOCM_READ_PREFETCH         // PB_PREFETCH_X   0x21
};

/* Return TRUE if given RLE extended command is valid. FALSE otherwise.
 */
#define isValidRLECommand(rleCmd) \
   ((rleCmd) >= PB_READV_X && (rleCmd) <= PB_PREFETCH_X)

/* Map a valid RLE extended command to an IORB command modifier.
 */
#define MapCommandRLEtoIORB(rleCmd)  RLEtoIORBCmdMod[ (rleCmd) - PB_READV_X ]
   

VOID SetupIORB(PUNITCB pUnitCB, PVOLCB pVolCB, PRENODE pRENode)
{
   PIORB             pIORB = (PIORB) &pRENode->IORB;
   PIORB_EXECUTEIO_X pXIO  = (PIORB_EXECUTEIO_X) pIORB;
   PPB_Read_Write    pRLE  = pRENode->pRE;
   PIORB_DMWORK      pDMWork;
   PSCATGATENTRY     pRLE_SGList;
   int i;

   trace(TR_CALL_SETUPIORB, pUnitCB, pVolCB, pRENode);

   ValidateRLE(pRLE);

   /* Set IORB command code and command modifier
    */
   pIORB->CommandCode = IOCC_EXECUTE_IO;
   pIORB->Length = sizeof(IORB_EXECUTEIO);

#ifdef VACPP_BUG
   /* Compiler     v3.6 VAC++. Used PB_xxx values as index into table
    * containing IOCM_xxx values.  However, table contained all zeros !!
    */
   switch (pRLE->RqHdr.Command_Code) {
      case PB_READ_X:  
         pIORB->CommandModifier = IOCM_READ;           break;
      case PB_WRITE_X: 
         pIORB->CommandModifier = IOCM_WRITE;          break;
      case PB_WRITEV_X:
         pIORB->CommandModifier = IOCM_WRITE_VERIFY;   break;
      case PB_PREFETCH_X:
         pIORB->CommandModifier = IOCM_READ_PREFETCH;  break;
      default:
         _interrupt(3);
   }
#else
   if (!isValidRLECommand(pRLE->RqHdr.Command_Code))
      { Assert(FALSE); }
   else {
      pIORB->CommandModifier = MapCommandRLEtoIORB(pRLE->RqHdr.Command_Code);
   }
#endif

   /* Setup remaining IORB header fields
    */
   pIORB->UnitHandle = pUnitCB->UnitInfo.UnitHandle;
   pIORB->RequestControl = IORB_ASYNC_POST;
   pIORB->Status = 0;
   pIORB->ErrorCode = 0;
   pIORB->Timeout = 0;
   pIORB->StatusBlockLen = 0;
   pIORB->pStatusBlock = 0;
   pIORB->NotifyAddress = pvThunkDoneIORB;
   pIORB->pNxtIORB = NULL;

   /* Setup DM work area
    */
   pDMWork = (PIORB_DMWORK) &(pIORB->DMWorkSpace);
   pDMWork->pUnitCB = offset(pUnitCB);
   pDMWork->pRequest = (PBYTE) pRENode;
   pDMWork->Flags = 0;

   /* Setup Execute IO IORB 
    */
   pXIO->cSGList = pRLE->SG_Desc_Count;

   pRLE_SGList = (PSCATGATENTRY) pointer_add(pRLE, sizeof(PB_Read_Write));

   if (pXIO->cSGList <= INLINE_SGLIST_SZ) {
      /* Scatter Gather list fits in our local copy area.
       * Copy the Scatter Gather list to the IORB.
       */
      ULONG  sgOffset;

      for (i = 0; i < pXIO->cSGList; ++i)
         pXIO->SGList[i] = pRLE_SGList[i];

      /* Set up pointers to SGList based on addresses provided in the RENODE */
      sgOffset = (PBYTE)&pXIO->SGList - (PBYTE)pRENode;
      pXIO->pSGList  = (PSCATGATENTRY) pointer_add(pRENode->VirtualAddr, sgOffset);
      pXIO->ppSGList = pRENode->PhysAddr + sgOffset;
   }
   else {
      /* Scatter Gather list DOES NOT fit in our local copy area.
       * Do things the hard way.  Call KEE services to calculate
       * physical address.  Obtain Virtual address from saved 16:16 pointer.
       * NOTE: 16:16 pointer is only saved for 16 bit IFSs. This WONT WORK for JFS.
       */
      PReq_List_Header pRLH;     /* Linear address of RLH */
      PVOID pvRLE;               /* Virtual (16:16) address of RLE */
      KernPageList_t pages[2];   /* 2 to allow straddling a page boundary */
      ULONG  pageCount = 2;
      APIRET rc;

      /* Calculate virtual 16:16 address of SG List
       */
      pRLH = pRENode->pRLChain->pRLHeader;

      Assert(pRLH->y_PhysAddr);   // Must have 16:16 address of RLH.  

      pvRLE = pointer_add(pRLH->y_PhysAddr, pRLE->RqHdr.Head_Offset);
      pXIO->pSGList  = (PSCATGATENTRY) pointer_add(pvRLE, sizeof(PB_Read_Write));

      /* Calculate physical address of SG List
       */
      rc = KernLinToPageList(pRLE_SGList, 
                             pXIO->cSGList * sizeof(SCATGATENTRY),
                             pages,
                             &pageCount );

      if (rc || pageCount > 2)  // Failed Kern call
         Assert(FALSE);

      pXIO->ppSGList = pages[0].Addr;
   }

   /* Set up remaining IO info */
   pXIO->RBA = pRLE->Start_Block +
               pVolCB->PartitionOffset + pVolCB->MediaBPB.HiddenSectors;
   pXIO->BlockCount = pRLE->Block_Count;
   pXIO->BlockSize = 512;
   pXIO->BlocksXferred = 0;
   pXIO->Flags = 0;

   /* Disable caching if requested */
   if (  ( !(pRLE->RW_Flags & RW_Cache_Req) ) ||      
         (pUnitCB->Flags & UCF_REMOVABLE_NON_FLOPPY) ) {
      pXIO->Flags |=  (XIO_DISABLE_HW_WRITE_CACHE |
                       XIO_DISABLE_HW_READ_CACHE);
   }
}

/*------------------------------------------------------------------------
   CallADD

   Pass an IORB request to the appropriate ADD.

   Parameters:
      PUNITCB pUnitCB         - pointer to UnitCB whose ADD is to be called
      PIORB   pIORB           - pointer to IORB to pass to ADD

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

VOID CallADD(PUNITCB pUnitCB, PIORB pIORB)
{
   PIORB_DMWORK  pDMWork;
   PRENODE pRENode;
   PVOID pIORB16;
   USHORT TraceEnabled;

   /*
    * Performance/RAS Tracing
    */
   SetTraceState(TraceEnabled);
   if (TraceEnabled)
      TraceIORB32Pre(pIORB, pUnitCB, TraceEnabled);

   trace(TR_CALL_CALLADD, pUnitCB, pIORB, 0);

   pDMWork = (PIORB_DMWORK) &(pIORB->DMWorkSpace);
   pRENode = (PRENODE) pDMWork->pRequest;
   pIORB16 = IORB_VirtualAddr(pRENode);

   KernSerialize16BitDD();   // OK to call even if already serialized
   KernThunkStackTo16();

   ThunkToAdd16(pUnitCB->AdapterDriverEP, pIORB16);

   KernThunkStackTo32();
   KernUnserialize16BitDD();
}

/*------------------------------------------------------------------------
   MapIORBError

   Map an IORB error code to a Request Packet (ERROR_I24) error code.

   Parameters:
      USHORT  IORB_ErrorCode  - the IORB error code to be mapped

   Returns:
      UCHAR    - The corresponding request packet error code

   Implementation:
   Just a 32 bit version of the 16 bit _MapIORBError() function.

------------------------------------------------------------------------*/

UCHAR MapIORBError(USHORT IORB_ErrorCode)
{
   int i;

   /* Map the IORB error to the corresponding ERROR_I24 error code */

   for (i = 0; _ErrorTable[i].IORB_ErrorCode != (USHORT) -1; i++)
     if (_ErrorTable[i].IORB_ErrorCode == IORB_ErrorCode)
        return(_ErrorTable[i].I24_ErrorCode);

   return(ERROR_I24_GEN_FAILURE);
}

/*------------------------------------------------------------------------
   NotifyDoneIORB32

   32 bit portion of callback logic called by ADD when an IORB completes.
   This routine is setup as the handler for flat strat2 / strat3 requests.

   Parameters:
      PRENODE  pRENode          - pointer to request's RENODE

   Returns:
      none

   Implementation:

------------------------------------------------------------------------*/

VOID _Pascal NotifyDoneIORB32(PRENODE pRENode)
{
   PIORB pIORB;
   PIORB_DMWORK pDMWork;
   PPB_Read_Write  pRLE; 
   PUNITCB pUnitCB;
   BOOL allDone;
   USHORT TraceEnabled;

   KernThunkStackTo32();

   /*
    * Performance/RAS Tracing
    */
   SetTraceState(TraceEnabled);
   if (TraceEnabled)
      TraceIORB32Post(pRENode, TraceEnabled);

   trace(TR_CALL_NOTIFYDONE, pRENode, pRENode->pRE->Start_Block, pRENode->pRE->Block_Count);

   pIORB = (PIORB) &pRENode->IORB;
   pDMWork = (PIORB_DMWORK) &(pIORB->DMWorkSpace);
   pRLE = pRENode->pRE;
   ValidateRLE(pRLE);

   /* Lock the UnitCB to protect the RLE, RLH, and UnitCB
    */
   pUnitCB = (PUNITCB) linear(pDMWork->pUnitCB);
   Lock(&pUnitCB->QueueLock);

   /* Set the RLE status from the IORB
    */
   if (  pIORB->Status & IORB_ERROR        &&
       !(pIORB->Status & IORB_RECOV_ERROR)
      ) {
      /* Unrecoverable error
       */
      pRLE->RqHdr.Status = RH_UNREC_ERROR;
      pRLE->RqHdr.Error_Code = MapIORBError(pIORB->ErrorCode);
      pRLE->Blocks_Xferred = 0;
   }
   else {
      /* Recoverable error or No error
       */
      pRLE->RqHdr.Status = RH_NO_ERROR;
      pRLE->RqHdr.Error_Code = 0;
      pRLE->Blocks_Xferred = pRLE->Block_Count;
   }

   Assert(pUnitCB->NumReqsInProgress > 0);
   pUnitCB->NumReqsInProgress --;
   _NumReqsInProgress--;

   /* NotifyRLE performs IFS notifications and Unlocks the UnitCB
    */
   allDone = NotifyRLE(pRENode, pUnitCB, TraceEnabled);

   if (!allDone) {
      /* If Uncertain Media error or Abort List on Error specified, */
      /* then abort the entire request list.                        */
      /* TBD */
   }

   /* Redrive IO
    */
   if (pUnitCB->NumReqsWaiting) {
      KernThunkStackTo16();
      SubmitRPRequests(pUnitCB);
      KernThunkStackTo32();
   }
   if (pUnitCB->NumRLEWaiting) {
      Lock(&pUnitCB->QueueLock);
      SubmitRLERequests(pUnitCB);   // Will Unlock UnitCB's QueueLock
   }

   KernThunkStackTo16();
}


/*------------------------------------------------------------------------
   NotifyRLE

   Update status of given RLE which has completed and associated RLH,
   and notify the appropriate IFS of completion if necessary.

   Parameters:
      PRENODE         pRENode - pointer to RENODE
      PUNITCB         pUnitCB - pointer to associate UnitCB

   Returns:
      BOOL      TRUE if all RLEs in associated RLH have completed
                FALSE otherwise

   Implementation:

------------------------------------------------------------------------*/

BOOL NotifyRLE( PRENODE pRENode, PUNITCB pUnitCB, USHORT TraceEnabled)
{
   PPB_Read_Write pRLE;
   Req_List_Header *pRLH;
   BOOL allDone, error;

   trace(TR_CALL_NOTIFYRLE, pRENode, pUnitCB, 0);

   pRLE = pRENode->pRE;
   ValidateRLE(pRLE);

   /* Make sure this request is not already done.
    * (Is this needed?     return FALSE?)
    */
   if (pRLE->RqHdr.Status & (RH_DONE | RH_ABORTED) ) {
      Unlock(&pUnitCB->QueueLock);
      return(FALSE);
   }

   /* Set status to done */
   pRLE->RqHdr.Status |= RH_DONE;

   pRLH = pRENode->pRLChain->pRLHeader;

   /* Keep track of whether all requests are done.
    */
   pRLH->y_Done_Count++;
   allDone = pRLH->y_Done_Count == pRLH->Count;
   if (allDone) {
      pRLH->Lst_Status &= ~(RLH_Req_Not_Queued | RLH_All_Req_Queued);
      pRLH->Lst_Status |= RLH_All_Req_Done;                          
   }

   /* Update RLH for recoverable error
    */
   if ( (pRLE->RqHdr.Status & RH_RECOV_ERROR) &&
       !(pRLH->Lst_Status & RLH_Unrec_Error) ) 
      pRLH->Lst_Status |= RLH_Rec_Error;

   /* Check for unrecoverable error */
   error = pRLE->RqHdr.Status & RH_UNREC_ERROR;
   if (error) {
      pRLH->Lst_Status |= RLH_Unrec_Error;
   }

   /* Unlock to end critical region accessing RLH / RLE
    */
   Unlock(&pUnitCB->QueueLock);

   /* Perform notification to OS2LVM if all requests are done.
    * Only doing RLH notification to LVM.  LVM fakes RLE notifications
    * to IFS if they requested it.
    */
   if (allDone) {
      /*
       * Performance/RAS Tracing
       */
      if (TraceEnabled)
         TraceStrat3Post(pRLH, TraceEnabled);

      trace(TR_CALL_CALLBACK, pRENode, pRENode->pRLChain, 0);
#ifdef DM_DEBUG
      NumRLEsReturned += pRLH->Count;
#endif
      RemoveReturnedRLCB(pRENode->pRLChain);
      (* (PCALLBACK32)pRLH->Notify_Address)(pRENode->pRLChain);
   }
   return allDone;
}

/*------------------------------------------------------------------------
   UnitToVolCB

   Given a unit number, return the associated VolCB.

   Parameters:
      USHORT      unit        - Unit number 

   Returns:
      PVOLCB                  - pointer to VolCB associated with given unit

   Implementation:

------------------------------------------------------------------------*/

PVOLCB UnitToVolCB(USHORT unit)
{
   NPVOLCB npVolCB;
   PVOLCB  pVolCB;

   /* Try fast lookup
    */
   npVolCB = _DriveToVolCB[unit];

   if (npVolCB)
      return linear(npVolCB);

   /* Fast lookup failed.  Perform linear search.
    */
   Lock(&_VolCB_Lock);
   npVolCB = _VolCB_Head;

   while (npVolCB) {
      pVolCB = linear(npVolCB);
      if (pVolCB->LogDriveNum == unit) {
         Unlock(&_VolCB_Lock);
         return(pVolCB);
      }
      npVolCB = pVolCB->pNextVolCB;
   }
   Unlock(&_VolCB_Lock);

   return NULL;  /* Failed to find */
}

