/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 *
 *   OCO Source Materials
 *
 *   Program number (when available)
 *
 *
 *   The source code for this program is not published or otherwise divested of its
 *   tradesecrets, irrespective of what has been deposited with the U.S. Copyright Office.
 *
 ****************************************************************************/
/**************************************************************************
 *
 * SCCSID: src/basedd/dasd/os2dasd/dmidc32.c, dsdm, w45.fs32, 20000823.1 99/03/16
 *
 * SOURCE FILE NAME = DMIDC32.C
 *
 * DESCRIPTIVE NAME = OS2DASD.DMD - OS/2 DASD Device Manager
 *
 * DESCRIPTION  32 bit IDC interfaces
 *
 *
*/
#include "dmh32.h"
#include "dmidc.h"


/*------------------------------------------------------------------------
   NextPhysDiskInfo

   Return information on the next physical disk, as specified by a given handle.

   Parameters:
      PHYSINFO_HANDLE *pHandle - pointer to an iteration handle
      PHYSDISK_INFO   *pInfo   - pointer to structure to return physical disk info

   Returns:
      BOOL             TRUE  if returned information is valid.
                       FALSE if there are no more physical disks, and no
                             information has been returned through pInfo.
   Usage:      

   NextPhysDiskInfo is exported through OS2DASD's IDC interface.
   See header dmidc.h for the IDC interface and physinfo.h for the data types
   used by the physical disk info interface.

   Suggested sample code for using this interface is given below:

   PHYSINFO_HANDLE handle = PHYSINFO_HANDLE_FIRST;   // Initialize for first element.
   PHYSDISK_INFO   info;                             // Disk information structure.

   while (NextPhysDiskInfo(&handle, &info)) {
      // Process the disk info returned in 'info
      ...
   }

   Implementation:

   Walk the UnitCB list (headed by UnitCB_Head), returning info on each disk.
   Handle maintains the iteration state by being a linear pointer to the UnitCB
   whose information was previously returned to the user. The iteration is started
   by PHYSINFO_HANDLE_FIRST, which is a unique pointer value (NULL).

------------------------------------------------------------------------*/

BOOL NextPhysDiskInfo(PHYSINFO_HANDLE *pHandle, PHYSDISK_INFO *pInfo)
{
   NPUNITCB npUnitCB;
   PUNITCB  pUnitCB;

   /* Obtain pointer to next UnitCB from HANDLE
    */
   if (*pHandle == PHYSINFO_HANDLE_FIRST) {
      if (_pPseudoFloppyUnitCB)                                      /*@221316*/
         // If have pseudo A: B:, report their psuedo floppy         /*@221316*/
         // UnitCB so os2lvm sees them.                              /*@221316*/
         npUnitCB = _pPseudoFloppyUnitCB;                            /*@221316*/
      else                                                           /*@221316*/
         npUnitCB = _UnitCB_Head;
   }
   else
      npUnitCB = ((PUNITCB)(*pHandle))->pNextUnitCB; 

   /* If there are no more UnitCBs, return FALSE
    */
   if (!npUnitCB)
      return FALSE;

   pUnitCB = linear(npUnitCB);

   /* Return information on this physical disk.
    */
   pInfo->Flags = 0;
   if (pUnitCB->UnitInfo.UnitFlags & UF_REMOVABLE)
      pInfo->Flags |= PDiskRemovable;
   if (pUnitCB->Flags & UCF_REMOVABLE_AS_FIXED)
      pInfo->Flags |= PDiskPRM;
// if (???)
//    pInfo->Flags |= PDiskBootDevice;
   if (pUnitCB->UnitInfo.UnitFlags & UF_PREFETCH)
      pInfo->Flags |= PDiskPrefetch;
   if (pUnitCB->Flags & UCF_HW_SCATGAT)
      pInfo->Flags |= PDiskHWSGList;
   if (pUnitCB->UnitInfo.UnitType == UIB_TYPE_CDROM)
      pInfo->Flags |= PDiskReadOnly; 
   if (pUnitCB->Flags & UCF_16M)
      pInfo->Flags |= PDiskAbove16M;

   pInfo->DiskBPB = ((PVOLCB)linear(pUnitCB->pCurrentVolCB))->MediaBPB;
   pInfo->UnitID = pUnitCB->PhysDriveNum;
   pInfo->MediaSerialNum = pUnitCB->MediaSerialNum;
   pInfo->MaxHWSGList = pUnitCB->MaxHWSGList;
   pInfo->DiskName[0] = 0;

   /* Update handle and return TRUE for success
    */
   *pHandle = pUnitCB;
   return TRUE;
}

/*------------------------------------------------------------------------
   NextPhysPartInfo

   Return information on the next physical partition of a particular disk,
   as specified by a given handle.

   Note: OS2DASD refers to these as logical volumes, but our client views 
   them as physical partitions.

   Parameters:
      PHYSINFO_HANDLE *pHandle - pointer to an iteration handle
      ULONG           driveNum - Physical disk number of disk whose partitions are 
                                 to be returned.  This will be the number returned 
                                 in PHYSDISK_INFO field 'UnitID' by NextPhysDiskInfo().
                                 (For fixed disks, this is BIOS convention, origin 0x80)
      PHYSPART_INFO   *pInfo   - pointer to structure to return physical partition info

   Returns:
      BOOL             TRUE  if returned information is valid.
                       FALSE if there are no more partitions, and no
                             information has been returned through pInfo.
   Usage:      

   NextPhysPartInfo is exported through OS2DASD's IDC interface.
   See header dmidc.h for the IDC interface and physinfo.h for the data types
   used by the physical partition info interface.

   Suggested sample code for using this interface is given below:

   PHYSINFO_HANDLE handle = PHYSINFO_HANDLE_FIRST;   // Initialize for first element.
   PHYSPART_INFO   info;                             // Partition information structure.
   ULONG           driveNum;   // Set to drive number as returned in PHYSDISK_INFO 'UnitID' field                

   while (NextPhysPartInfo(&handle, driveNum, &info)) {
      // Process the partition info returned in 'info'
      ...
   }

   Implementation:

   Walk the VolCB list (headed by VolCB_Head), and return information on each
   logical volume associated with the given disk.  Note: once the first VolCB
   for a given disk is found in the list, the remaining VolCBs for that disk
   are consecutive in the list.  
   Fault Tolerance partitions are NOT reported to caller.

   Handle maintains the iteration state by being a linear pointer to the VolCB
   whose information was previously returned to the user. The iteration is started
   by PHYSINFO_HANDLE_FIRST, which is a unique pointer value (NULL).

------------------------------------------------------------------------*/

BOOL NextPhysPartInfo(PHYSINFO_HANDLE *pHandle, ULONG driveNum, PHYSPART_INFO *pInfo)
{
   NPVOLCB npVolCB;
   PVOLCB  pVolCB;

   /* Obtain pointer to next VolCB from HANDLE
    */
   if (*pHandle == PHYSINFO_HANDLE_FIRST) {
      /* Find the first logical volcb for this drive */
      npVolCB = _VolCB_Head;
      while (npVolCB) {
         pVolCB = linear(npVolCB);
         if ((UCHAR)pVolCB->PhysDriveNum == driveNum)
            break;               /*@221316  Need to cast (UCHAR) for PhysDriveNum value of 0xffff */
                                 /* since 'driveNum' is really UCHAR (PHYSPART_INFO 'UnitId' is UCHAR) */
         npVolCB = pVolCB->pNextVolCB;
      }
   }
   else
      npVolCB = ((PVOLCB)(*pHandle))->pNextVolCB; 

   pVolCB = linear(npVolCB);

   /* Skip over (don't report) any Fault Tolerance partitions
    */
   while (npVolCB && pVolCB->Flags & vf_FTPartition) {
      npVolCB = pVolCB->pNextVolCB;
      pVolCB = linear(npVolCB);
   }

   /* If there are no more logical VolCBs for this disk, return FALSE
    */
   if (!npVolCB  || isPhysicalVolCB(pVolCB)  ||  (UCHAR)pVolCB->PhysDriveNum != driveNum)
      return FALSE;

   /* Return information on this physical partition.
    */
   pInfo->Flags = 0;
   if (pVolCB->PartitionType == PARTITION_LVM)
      pInfo->Flags |= PPartType;
   pInfo->PartBPB = pVolCB->MediaBPB;

   if (pVolCB->pUnitCB == _pPseudoFloppyUnitCB)                      //@221316
      /* This is a pseudo A: or B:.  It's VolCB doesn't have a valid //@221316
       * unit ID (LogDriveNum), so calculate from drive letter.      //@221316
       */                                                            //@221316
      pInfo->UnitID = pVolCB->DriveLetter == 'A' ? 0 : 1;            //@221316
   else                                                              //@221316
      pInfo->UnitID = pVolCB->LogDriveNum;

   pInfo->DriveLetter = pVolCB->DriveLetter;
   pInfo->PartitionStart = pVolCB->PartitionOffset + pVolCB->MediaBPB.HiddenSectors;
   if (pVolCB->MediaBPB.TotalSectors != 0)
      pInfo->PartitionSize = pVolCB->MediaBPB.TotalSectors;
   else
      pInfo->PartitionSize = pVolCB->MediaBPB.BigTotalSectors;
   pInfo->PartitionCylinders = pVolCB->NumLogCylinders;
   pInfo->PartitionName[0] = 0;
   pInfo->VolumeName[0] = 0;
   pInfo->MediaSerialNum = ((PUNITCB)linear(pVolCB->pUnitCB))->MediaSerialNum;
   pInfo->PartitionSerialNum = pVolCB->PartitionSerialNum;

   /* Update handle and return TRUE for success
    */
   *pHandle = pVolCB;
   return TRUE;
}
