{******************************************************************************}
{**}
{**  Project Name:	DropShell}
{**     File Name:	DSAppleEvents.p}
{**}
{**   Description:	Generic AppleEvent handling routines}
{**					}
{**					This is the set of routines for handling the required Apple events.}
{**					You should NEVER have to modify this file!!!}
{**					Simply add code in DSUserProcs to the routines called by these.}
{**}
{*******************************************************************************}
{**                       A U T H O R   I D E N T I T Y}
{*******************************************************************************}
{**}
{**	Initials	Name}
{**	--------	-----------------------------------------------}
{**	LDR			Leonard Rosenthol}
{**}
{*******************************************************************************}
{**                      R E V I S I O N   H I S T O R Y}
{*******************************************************************************}
{**}
{**	  Date		Time	Author	Description}
{**	--------	-----	------	---------------------------------------------}
{**	11/24/91			LDR		Added a handler for 'pdoc' as per DTS recommendation}
{**								This caused some reorg & userProc routine changes}
{**								I also created a new common AEVT doc extractor}
{**								Pass the new userDataHandle to odoc/pdoc routines}
{**								FailErr now uses the ErrorAlert routine to report probs}
{**	10/30/91			LDR		Modified USES clause for new include & ifc'ed ThP}
{**	10/28/91			LDR		Officially renamed DropShell (from QuickShell)}
{**								Added a bunch of comments for clarification}
{**	04/09/91	00:04	LDR		Original Version}
{**}
{******************************************************************************}

unit DSAppleEvents;
interface

	uses
		AppleTalk, Processes, PPCToolbox, EPPC, Notification, AppleEvents, DSGlobals, DSUtils, DSUserProcs;	{just the DropShell files}

{---------------------}
{Interface Definitions}
{---------------------}

	procedure InitAEVTStuff;
	procedure DoHighLevelEvent (event: EventRecord);

	function HandleOAPP (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
	function HandleODOC (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
	function HandlePDOC (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
	function HandleQuit (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;

implementation

	{$S Initialize}
	{ }
{		This routine does all initialization for AEM, including the}
{		creation and then population of the dispatch table.}
{	}
	procedure InitAEVTStuff;
		var
			aevtErr: OSErr;

	begin
		aevtErr := noErr;

		if (aevtErr = noErr) then
			aevtErr := AEInstallEventHandler(kCoreEventClass, kAEOpenApplication, @HandleOAPP, 0, false);
		if (aevtErr = noErr) then
			aevtErr := AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, @HandleODOC, 0, false);
		if (aevtErr = noErr) then
			aevtErr := AEInstallEventHandler(kCoreEventClass, kAEPrintDocuments, @HandlePDOC, 0, false);
		if (aevtErr = noErr) then
			aevtErr := AEInstallEventHandler(kCoreEventClass, kAEQuitApplication, @HandleQUIT, 0, false);

		if (aevtErr = noErr) then
			InstallOtherEvents;

		if (aevtErr <> noErr) then
			; {Report an error to the user, if you are so inclinded!}
	end;


	{$S Main}
	{ }
{		This routine is a utility routine for checking that all required }
{		parameters in the Apple event have been used.}
{	}

	function GotRequiredParams (theAppleEvent: AppleEvent): OSErr;
		var
			typeCode: DescType;
			actualSize: Size;
			err: OSErr;

	begin
		err := AEGetAttributePtr(theAppleEvent, keyMissedKeywordAttr, typeWildCard, typeCode, nil, 0, actualSize);
		if err = errAEDescNotFound then { we got all the required params: all is ok }
			GotRequiredParams := noErr
		else if err = noErr then
			GotRequiredParams := errAEEventNotHandled
		else
			GotRequiredParams := err;
	end; { GotRequiredParams }

	{}
{		This is another routine useful for showing debugging info.}
{		}
{		It calls the ErrorAlert routine from DSUtils to put up the }
{		error message.}
{	}
	procedure FailErr (err: OSErr);
	begin
		if err <> noErr then
			ErrorAlert(kErrStringID, kAEVTErr, err);
	end;



	{}
{		This routine is the handler for the oapp (Open Application) event.}
{		}
{		It first checks the number of parameters to make sure we got them all }
{		(even though we don't want any) and then calls the OpenApp userProc in QSUserProcs.}
{		Finally it checks to see if the caller wanted a reply & sends one, setting any error.}
{	}

	function HandleOAPP (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
		var
			errStr: Str255;

	begin
		FailErr(GotRequiredParams(theAppleEvent));

		{ let's show the user the splash screen }
		ShowWindow(gSplashScreen);

		OpenApp; 	{pass it on to the app specific routine}

		if reply.dataHandle <> nil then { a reply is sought }
			begin
				errStr := 'Opening';
				HandleOAPP := AEPutParamPtr(reply, 'errs', 'TEXT', Ptr(@errStr[1]), length(errStr));
			end
		else
			HandleOAPP := noErr;
	end;

	{}
{		This routine is the handler for the quit (Quit Application) event.}
{		}
{		It first checks the number of parameters to make sure we got them all }
{		(even though we don't want any) and then calls the QuitApp userProc in QSUserProcs.}
{		Finally it checks to see if the caller wanted a reply & sends one, setting any error.}
{	}

	function HandleQUIT (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
		var
			errStr: Str255;

	begin
		FailErr(GotRequiredParams(theAppleEvent));

		QuitApp;	{pass it on to the app specific routine}

		if reply.dataHandle <> nil then { a reply is sought }
			begin
				errStr := 'Quiting';
				HandleQUIT := AEPutParamPtr(reply, 'errs', 'TEXT', Ptr(@errStr[1]), length(errStr));
			end
		else
			HandleQUIT := noErr;
	end;

	{}
{		This routine is the low level processing routine for both the }
{		odoc (Open Document) and pdoc (Print Document) events.}
{		}
{		This routine is the key one, since this is how we get the list of}
{		files/folders/disks to process.  The first thing to do is the get the}
{		list of files, and then make sure that's all the parameters (should be!).}
{		We then send call the PreflightDocs routine (from DSUserProcs), process}
{		each file in the list by calling OpenDoc (again in DSUserProcs), and finally}
{		call PostflightDocs (you know where) and setting a return value.}
{	}

	function _HandleDocs (theAppleEvent, reply: AppleEvent; opening: Boolean): OSErr;
		var
			myFSS: FSSpec;
			docList: AEDescList;
			index, itemsInList: LONGINT;
			actualSize: Size;
			keywd: AEKeyword;
			typeCode: descType;
			userDataHandle: Handle;

	begin
		FailErr(AEGetParamDesc(theAppleEvent, keyDirectObject, typeAEList, docList));
		FailErr(GotRequiredParams(theAppleEvent));

		if PreFlightDocs(opening, userDataHandle) then {let the app do any preflighting it might need}
			begin
				FailErr(AECountItems(docList, itemsInList));	{count the number of items}
				for index := 1 to itemsInList do
					begin
						FailErr(AEGetNthPtr(docList, index, typeFSS, keywd, typeCode, @myFSS, sizeof(myFSS), actualSize));
						OpenDoc(@myFSS, opening, userDataHandle);	{call the open routine}
					end;

				PostFlightDocs(opening, userDataHandle); {and any cleanup}

				_HandleDocs := noErr;
			end
		else
			_HandleDocs := errAEEventNotHandled;	{this tells AEM we didn't handle it}

		FailErr(AEDisposeDesc(docList));
	end; { _HandleDocs }


	{}
{		This routine is the handler for the odoc (Open Document) event.}
{		}
{		The odoc event simply calls the common _HandleDocs routines, which will}
{		do the dirty work of parsing the AEVT & calling the userProcs.}
{	}

	function HandleODOC (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
	begin
		HandleODOC := _HandleDocs(theAppleEvent, reply, TRUE);	{call the low level routine}
	end; { HandleODOC }

	{}
{		This routine is the handler for the pdoc (Print Document) event.}
{		}
{		The pdoc event like the odoc simply calls the common _HandleDocs routines}
{	}

	function HandlePDOC (theAppleEvent, reply: AppleEvent; handlerRefcon: LONGINT): OSErr;
	begin
		HandlePDOC := _HandleDocs(theAppleEvent, reply, FALSE);	{call the low level routine}
	end; { HandlePDOC }

	{}
{		This is the routine called by the main event loop, when a high level}
{		event is found.  Since we only deal with Apple events, and not other}
{		high level events, we just pass everything onto the AEM via AEProcessAppleEvent}
{	}
	procedure DoHighLevelEvent (event: EventRecord);
	begin
		FailErr(AEProcessAppleEvent(event));
	end;

end.