/*********************************************************/
/*                                                       */
/*  Name:         vcfgcid.cmd                            */
/*                                                       */
/*  Description:  REXX command file for selecting        */
/*                resolution (<Horiz>x<Vert>x<colors>),  */
/*                refresh rate and display type during   */
/*                CID install of supported adapters.     */
/*                                                       */
/*  Requirements: rxvidcfg.dll needs to be in libpath    */
/*                                                       */
/*  Valid Parameters:                                    */
/*               /L1:<Error Log>                         */
/*               /L2:<History Log>                       */
/*               /R:<Response file> {not yet implemented}*/
/*               /RES:<Horizontal>x<Vertical>x<Colors>   */
/*               /RR:<Refresh Rate>                      */
/*               /MON:<Display type index>               */
/*               /PMI            - If not specified then */
/*                                 assume no             */
/*********************************************************/

/*
** MRI Strings which must be translated
*/
linefeed = x2c('0D0A')

MRI_CANNOT_OPEN_LOG = 'Kunne ikke pne loggfilen: '
MRI_CANNOT_OPEN_HISTORY = 'Kunne ikke pne historikkfilen: '
MRI_CANNOT_OPEN_RESPONSE = 'Kunne ikke pne responsfilen: '
MRI_COMPLETE_ERRORS = 'Fullfrt med feil'
MRI_COMPLETE_SUCCESS = 'Fullfrt'
MRI_ERROR = 'Feil: '
MRI_INVALID_MONITOR = 'Ugyldig skjerm-ID er oppgitt: '
MRI_INVALID_REFRESH = 'Ugyldig fornyingsfrekvens er oppgitt: '
MRI_INVALID_RESOLUTION = 'Ugyldig opplsning er oppgitt: '
MRI_MONITOR_INDEX_SET = 'Skjermen er satt til skjermindeks '
MRI_NO_ARGUMENTS = 'Ingen argumenter er oppgitt!'
MRI_RESOLUTION_SET = 'Opplsningen er satt til '
MRI_REFRESH_SET = 'Fornyingsfrekvens er satt til '
MRI_RXVIDCFG_NOT_IN_LIBPATH = 'RXVIDCFG.DLL ER IKKE I LIBPATH'
MRI_SPECIFY_L1 = 'Filnavn m oppgis sammen med kommandolinjeparameteren /L1:'
MRI_SPECIFY_L2 = 'Filnavn m oppgis sammen med kommandolinjeparameteren /L2:'
MRI_SPECIFY_R = 'M oppgi en eller flere av /RES: /MON: /RR:'
MRI_SYNTAX_ERROR = 'Syntaksfeil p linje '
MRI_UNKNOWN_COMMANDS = 'Ukjent(e) kommando(er): '
MRI_USAGE = linefeed'Syntaks:  vcfgcid /L1:<Feillogg> /L2:<Historikklogg>'linefeed,
            '       /RES:<HORISONTALxVERTIKALxFARGER> /MON:<Skjermindeks>'linefeed,
            '       /PMI /RR:<Fornyingsfrekvens>'

/*
** Global Constants
*/
PARAMS = '/L1: /L2: /R: /RES: /MON: /PMI /RR:'     /* Valid command line argument switches) */
KEYS   = 'logFile histLog respFile resolution monID genPMI refresh' /* Command line mnemonics */
/*  SDM return codes */
InvocationError   = x2d('1600',4)
SPTCallback       = x2d('FF01',4)
SPTReboot         = x2d('FE00',4)
SPTRebootErr      = x2d('FE08',4)
SPTWithErr        = x2d('0008',4)

signal on syntax name syntaxError

/*
** Initialized Global variables
*/
TRUE         = 1
FALSE        = 0
ENV          = 'OS2ENVIRONMENT'

doCallback   = FALSE
exitRC       = ''

isLog        = FALSE
isHist       = FALSE
isResp       = FALSE
isResolution = FALSE
isMonID      = FALSE
doSVGA       = FALSE
isRefresh    = FALSE

parse source . . me          /* Get fully qualified path name of our invocation */
exePath   = substr(me, 1, lastpos('\', me) - 1)


GLOBALS = 'logFile histLog input inputArgs. PARAMS KEYS TRUE FALSE HorizRes VertRes NumColors'


/*
** Start of main processing
*/

/* trace '?I' */
parse arg input

if ProcessArgs() \= 0               /* See what arguments were passed to us and validate them */
   then cleanup(InvocationError)

if isResp && ProcessResp() \= 0
   then cleanup(InvocationError)


'@set ENDLIBPATH='exePath
call RxFuncAdd 'SysVcfgSetCurrentCfg', 'RXVIDCFG', 'SysVcfgSetCurrentCfg'
call RxFuncAdd 'SysSearchPath', 'REXXUTIL', 'SysSearchPath'


state = value('REMOTE_INSTALL_STATE',, ENV)
if state = '' then cidInstall = FALSE
              else cidInstall = TRUE

if isMonID then config.MonitorID = inputArgs.monID
if cidInstall & ( state \= 0 ) then drop config.MonitorID
if doSVGA
then do
   config.RunSVGA = 1
   PMIstate = getPMI()    /* Save date and size of current PMI file to see latter if it changed */
end
if isMonID & cidInstall & ( state = 0 ) & doSVGA
then do
   signal on syntax name NoDllError
   result = SysVcfgSetCurrentCfg('config')
   signal on syntax name syntaxError
   parse var result rc errMessage
   if rc \= 0
   then do
      call doLog error, MRI_ERROR || rc errMessage
      exitRC = SPTWithErr
   end
   else do
      call doLog history, MRI_MONITOR_INDEX_SET || config.MonitorID
      exitRC = SPTReboot
      if PMIstate \= getPMI() & (isResolution | isRefresh)
      then do
         doCallback = TRUE
         exitRC = SPTCallback
      end
      drop config.MonitorID
   end
end
if \( doCallback | (exitRC = SPTWithErr))
then do
   if isResolution
   then do
      config.HorizResolution = HorizRes
      config.VertResolution  = VertRes
      config.Colors          = NumColors
   end
   if isRefresh then config.VertRefresh = inputArgs.refresh
   if isResolution | isRefresh | isDefined('config.MonitorID')
   then do
      signal on syntax name NoDllError
      result = SysVcfgSetCurrentCfg('config')
      signal on syntax name syntaxError
      parse var result rc errMessage
      if rc \= 0
      then do
         call doLog error, MRI_ERROR || rc errMessage
         if exitRC \= '' then exitRC = SPTRebootErr
                         else exitRC = SPTWithErr
      end
      else do
         logMessage = ''
         if isMonID & \cidInstall then call doLog history, MRI_MONITOR_INDEX_SET || config.MonitorID
         if isResolution then logMessage = MRI_RESOLUTION_SET || inputArgs.resolution' '
         if isRefresh then logMessage = logMessage || MRI_REFRESH_SET || config.VertRefresh
         call doLog history, logMessage
         exitRC = SPTReboot
      end
   end
end

call cleanup exitRC


/*********************************************************/
/*                                                       */
/* Function ProcessArgs:                                 */
/*                                                       */
/*********************************************************/

ProcessArgs:

parse upper var input commandLine

returnState = 0

if commandLine \= ''
then do
   do i = 1 to words(PARAMS)
      if pos(word(PARAMS, i), commandLine) \= 0
      then do
         parseCmd = "parse var commandLine begin '"word(PARAMS,i)"'value end"
         interpret parseCmd
         commandLine = begin || end
         if pos(':', value) \= 0
         then
            if wordpos(substr(value, 1, pos(':', value)), PARAMS) \= 0
            then do
               commandLine = commandLine || value
               value = ''
            end
         call value 'inputArgs.'word(KEYS,i) , value
         drop value
      end
   end

   /* First get log file names so that any subsequent messages get logged if possible*/

   if isDefined('inputArgs.logFile')
   then
      if inputArgs.logFile \= ''
      then do
         if stream(inputArgs.logFile, 'C', 'query exists') \= '' then '@del ' inputArgs.logFile
         if stream(inputArgs.logFile, 'C', 'open') \= 'READY:'
            then call  doLog error, MRI_CANNOT_OPEN_LOG || inputArgs.logFile
            else isLog = TRUE
      end
      else do
         call doLog error, MRI_SPECIFY_L1
         returnState = InvocationError
      end

   if isDefined('inputArgs.histLog')
   then
      if inputArgs.histLog \= ''
      then do
         if stream(inputArgs.histLog, 'C', 'open write') \= 'READY:'
         then call doLog error, MRI_CANNOT_OPEN_HISTORY || inputArgs.histLog
         else do
            isHist = TRUE
            parse source . . me
            call doLog history, me input
         end
       end
       else do
          call doLog error, MRI_SPECIFY_L2
          returnState = InvocationError
       end

   if commandLine \= ''
   then do
      call doLog error, MRI_UNKNOWN_COMMANDS || commandLine
      returnState = InvocationError
   end


   if isDefined('inputArgs.respFile')
   then
      if inputArgs.respFile == ''
      then do
         call doLog error, MRI_CANNOT_OPEN_RESPONSE || inputArgs.respFile
         returnState = InvocationError
      end
      else if (stream(inputArgs.respFile,'C','query exists') = '')
      then do
      call doLog error, MRI_CANNOT_OPEN_RESPONSE || inputArgs.respFile
      returnState = InvocationError
      end
      else isResp = TRUE

   if isDefined('inputArgs.resolution')
   then do
      parse var inputArgs.resolution HorizRes 'X' VertRes 'X' NumColors
      if (datatype(HorizRes, 'W')) & (datatype(VertRes, 'W')) & (datatype(NumColors, 'W'))
      then isResolution = TRUE
      else do
         call doLog error, MRI_INVALID_RESOLUTION || inputArgs.resolution
         returnState = InvocationError
      end
   end

   if isDefined('inputArgs.monID')
   then
      if datatype(inputArgs.monID, 'W')
      then isMonID = TRUE
      else do
         call doLog error, MRI_INVALID_MONITOR || inputArgs.monID
         returnState = InvocationError
      end

   if isDefined('inputArgs.genPMI')
   then  doSVGA = TRUE

   if isDefined('inputArgs.refresh')
   then
      if datatype(inputArgs.refresh, 'W')
         then isRefresh = TRUE
         else do
            call doLog error, MRI_INVALID_REFRESH || inputArgs.refresh
            returnState = InvocationError
         end

   /* Must have /R: specified and/or one or more of /RES: /MON: /RR: in order for a valid invocation */
   if \( isResp | isResolution | isMonID | isRefresh )
   then do
      call doLog error, MRI_SPECIFY_R
      returnState = InvocationError
   end
end

else do
   call doLog error, MRI_NO_ARGUMENTS
   returnState = InvocationError
end

if returnState = InvocationError then call usage

return  returnState


/*********************************************************/
/*                                                       */
/* Function ProcessResp:                                 */
/*                                                       */
/*********************************************************/
ProcessResp:

/* TBD */

return 0
/*********************************************************/
/*                                                       */
/* Function: isDefined                                   */
/*                                                       */
/* Purpose: See if variable has been defined(set)        */
/*********************************************************/
isDefined:

parse arg varName

return symbol(varName) = 'VAR'


/*********************************************************/
/*                                                       */
/* Function: doLog                                       */
/*                                                       */
/* Purpose: output messages to stdout and log file if    */
/* available                                             */
/*********************************************************/

doLog:

type    = arg(1)
message = arg(2)

timeStamp = '[' || date('L') time() || ']'

if type = 'ERROR'
   then do
      say message
      if isLog then call lineout inputArgs.logFile, timeStamp message
   end
   else
      if isHist then call lineout inputArgs.histLog, timeStamp message

return



/*********************************************************/
/*                                                       */
/* Function: cleanup                                     */
/*                                                       */
/* Purpose: check exit code passed, log appropriate      */
/* messages, close open files exit with code passed      */
/*********************************************************/

cleanup:

exitCode = arg(1)

if (exitCode = InvocationError) | (exitCode = SPTRebootErr) | (exitCode = SPTWithErr)
then call doLog history, MRI_COMPLETE_ERRORS
else call doLog history, MRI_COMPLETE_SUCCESS

if isLog then call stream inputArgs.logFile, 'c', 'close'
if isHist then call stream inputArgs.histLog, 'c', 'close'

exit exitCode

/*********************************************************/
/*                                                       */
/* Function: getPMI                                      */
/*                                                       */
/*********************************************************/
getPMI: Procedure

PMIFile = SysSearchPath('PATH', 'SVGADATA.PMI')
data = PMIFile
if PMIFile \= ''
then do
  data = data || stream(PMIFile, 'c', 'query size')
  data = data || stream(PMIFile, 'c', 'query datetime')
end

return data

/*********************************************************/
/*                                                       */
/* Function: usage                                       */
/*                                                       */
/*********************************************************/

usage:

call doLog error, MRI_USAGE

return

/*********************************************************/
/*                                                       */
/* Function: syntaxError                                 */
/*                                                       */
/*********************************************************/

syntaxError:

call doLog error, MRI_SYNTAX_ERROR || SIGL
call usage
call cleanup SPTWithErr

/*********************************************************/
/*                                                       */
/* Function: NoDllError                                  */
/*                                                       */
/*********************************************************/

NoDllError:

call doLog error, MRI_RXVIDCFG_NOT_IN_LIBPATH
call cleanup SPTWithErr
