/****************************************************************************
 * COMPRESS.H
 *
 *      This header file contains the function and constant declarations for
 *      the data compression functions in the COMPRESS.DLL dynamic link
 *      library.
 *
 *      Modifications : 05-Mar-94 : Initial version.
 *                      20-Mar-94 : Added programming notes comments.
 *                                  Changed PBYTE to const BYTE * in
 *                                  CompressObject function declaration.
 *                                  Changed PCOMPRESSED_DATA to
 *                                  const COMPRESSED_DATA * in ExpandObject
 *                                  function declaration.
 *                                  Eliminated dependencies on OS2 toolkit
 *                                  headers.
 *                      10-Apr-94 : Fixed problem with CompressObject
 *                                  and zero length input data blocks.
 *                      14-May-94 : Added ReadAndExpand and CompressAndWrite
 *                                  APIs.
 *                      31-May-94 : Fixed OS2 common definitions section of
 *                                  header file so that it is compatible
 *                                  with the GNU C++ compiler.
 *
 *      (c)Copyright 1994 Rick Yoder
 ****************************************************************************/

/*****************************************************************************
 * Programming notes
 *
 * 1. All entry points into this DLL are implemented using the same calling
 *    convention as the OS2 system APIs so it should be compiler independent.
 *
 * 2. The compression functions are fully reentrant.
 *
 * 3. The 32-bit CRCs used for data integrity checking are built using the same
 *    32-bit generator polynomial used by the ZMODEM protocol: 0xEDB88320. The
 *    CRC is initialized to 0xFFFFFFFF and the final calculated value is inverted.
 *
 * 3. Currently, the only compression method available is METHOD_DEFLATE, which
 *    compresses your data using the pkzip 2.04 deflate method.
 *
 * 4. You should never receive a protection violation exception generated from
 *    within the library functions. The compression functions are designed to
 *    (hopefully) fail gracefully and return the appropriate error code if
 *    they are given bad data or some sort of internal error occurs. If one of
 *    the library functions returns COMPRESS_INTERNAL_ERROR or
 *    EXPAND_INTERNAL_ERROR please report this to the author as this means
 *    there is probably a bug in the library.
 *****************************************************************************/

#if !defined(_COMPRESS_INCLUDED)
#define _COMPRESS_INCLUDED

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************
 * If the OS2 common definitions header from the OS2 toolkit (OS2DEF.H) has
 * not been included, define the types from it that we need.
 *****************************************************************************/
#if !defined(OS2DEF_INCLUDED) && !defined(_OS2EMX_H)
    #if defined(__EMX__)
        #define EXPENTRY
    #else
        #define EXPENTRY _System
    #endif
    typedef unsigned long APIRET;

    typedef unsigned short USHORT;
    typedef unsigned long  ULONG;
    typedef char BYTE;
    typedef BYTE *PBYTE;
#endif

#if !(defined(INCL_DOSINCLUDED) || defined(_OS2EMX_H)) || !(defined (INCL_DOSFILEMGR) || !defined (INCL_NOCOMMON))
    typedef unsigned long  HFILE;
#endif
/*****************************************************************************/

/*****************************************************************************/
/* Definition of compression constants                                       */
/*****************************************************************************/
#define METHOD_STORE    0           /* store object with no compression */
#define METHOD_DEFLATE  1           /* use ZIP style deflation */

#define METHOD_BEST     255         /* use best available compression method */
                                    /*   (currently METHOD_DEFLATE)          */
/*****************************************************************************/

/*****************************************************************************/
/* Error constants                                                           */
/*****************************************************************************/
#define COMPRESS_INVALID_METHOD     ((APIRET)0xF0000000)
#define COMPRESS_INVALID_LEVEL      ((APIRET)0xF0000001)
#define COMPRESS_INVALID_PARAMETER  ((APIRET)0xF0000002)
#define COMPRESS_INTERNAL_ERROR     ((APIRET)0xF0000003)

#define EXPAND_INVALID_PARAMETER    ((APIRET)0xFF000000)
#define EXPAND_INVALID_METHOD       ((APIRET)0xFF000001)
#define EXPAND_INTERNAL_ERROR       ((APIRET)0xFF000002)
#define EXPAND_CRC                  ((APIRET)0xFF000003)

#ifndef ERROR_NOT_ENOUGH_MEMORY
    #define ERROR_NOT_ENOUGH_MEMORY 8
#endif
#ifndef ERROR_INVALID_PARAMETER
    #define ERROR_INVALID_PARAMETER 87
#endif
#ifndef ERROR_INTERRUPT
    #define ERROR_INTERRUPT 95
#endif
#ifndef ERROR_INVALID_ADDRESS
    #define ERROR_INVALID_ADDRESS 487
#endif
#ifndef ERROR_ACCESS_DENIED
    #define ERROR_ACCESS_DENIED 5
#endif
#ifndef ERROR_LOCKED
    #define ERROR_LOCKED 212
#endif
#ifndef ERROR_CROSSES_OBJECT_BOUNDARY
    #define ERROR_CROSSES_OBJECT_BOUNDARY 32798
#endif
#ifndef ERROR_INVALID_HANDLE
    #define ERROR_INVALID_HANDLE 6
#endif
#ifndef ERROR_NOT_DOS_DISK
    #define ERROR_NOT_DOS_DISK 26
#endif
#ifndef ERROR_LOCK_VIOLATION
    #define ERROR_LOCK_VIOLATION 33
#endif
#ifndef ERROR_WRITE_PROTECT
    #define ERROR_WRITE_PROTECT 19
#endif
#ifndef ERROR_WRITE_FAULT
    #define ERROR_WRITE_FAULT 29
#endif
#ifndef ERROR_BROKEN_PIPE
    #define ERROR_BROKEN_PIPE 109
#endif
#ifndef ERROR_MORE_DATA
    #define ERROR_MORE_DATA 234
#endif

/*****************************************************************************/

/*****************************************************************************/
/* Definition of compressed data object data structure                       */
/*****************************************************************************/
#pragma pack(1)
typedef struct _COMPRESSED_DATA {
    ULONG  crc;                 /* 32 bit CRC of rest of structure + compressed data */
    ULONG  ulCompressedSize;    /* compressed size of data */
    ULONG  crcOriginal;         /* 32 bit CRC of the original uncompressed data */
    ULONG  ulOriginalSize;      /* original uncompressed size of data */
    USHORT version;             /* version of COMPRESS.DLL used to compress data */
    BYTE   method;              /* compression method */
    BYTE   reserved;            /* currently unused, set to 0 */
    BYTE   data[1];             /* compressed data starts here */
    } COMPRESSED_DATA;

typedef COMPRESSED_DATA * PCOMPRESSED_DATA;
#pragma pack()

#define COMPRESSED_HEADER_SIZE  ( sizeof(COMPRESSED_DATA) - 1 )     /* 20 bytes */
/*****************************************************************************/

/*****************************************************************************
 * Function declarations
 *****************************************************************************/
extern APIRET EXPENTRY CompressObject( const BYTE * pData,
                                       ULONG dataSize,
                                       PCOMPRESSED_DATA * ppCompressed,
                                       BYTE method,
                                       BYTE level );
/* Description          This function compresses the designated memory object.
 *
 * Parameters           pData points to the data object that is to be compressed.
 *                      This parameter may be set to NULL if dataSize is zero.
 *
 *                      dataSize contains the size in bytes of the input data object.
 *                      If dataSize is zero, the METHOD_STORE compression
 *                      method will always be used.
 *
 *                      ppCompressed points to the location where the pointer to
 *                      the compressed data object created by CompressObject is to
 *                      be stored. The memory used to store the compressed data
 *                      object is sparse allocated by the DosAllocMem function.
 *
 *                      method contains the compression method to be used to
 *                      compress the data object. This parameter can be set to
 *                      one of the following values:
 *
 *                          METHOD_STORE    ..... store with no compression.
 *                          METHOD_DEFLATE  ..... use ZIP style deflation.
 *                          METHOD_BEST     ..... use best available method
 *                                                  (currently METHOD_DEFLATE).
 *
 *                      level contains a value from 0 - 9 indicating the compression
 *                      level to be used. A value of 0 forces the data to be STORED.
 *                      A value of 1 gives fast but minimal compression. A value of
 *                      9 gives slow but maximal compression. Usually a compression
 *                      level of 4 or 5 provides a decent speed/compression tradeoff.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      CompressObject returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_INVALID_ADDRESS
 *                      ERROR_ACCESS_DENIED
 *                      ERROR_LOCKED
 *                      ERROR_CROSSES_OBJECT_BOUNDARY
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      COMPRESS_INVALID_METHOD     ..... The specified compression method
 *                                                        is not supported by this version of
 *                                                        the library.
 *
 *                      COMPRESS_INVALID_LEVEL      ..... The compression level parameter
 *                                                        contained an invalid value.
 *
 *                      COMPRESS_INVALID_PARAMETER  ..... The pData, dataSize, or ppCompressed
 *                                                        parameter contained an invalid
 *                                                        value.
 *
 *                      COMPRESS_INTERNAL_ERROR     ..... An internal error occured in the
 *                                                        compression function.
 */

extern APIRET EXPENTRY ExpandObject( const COMPRESSED_DATA * pData,
                                     PBYTE * ppExpanded );
/* Purpose              This function decompresses the designated compressed data
 *                      object.
 *
 * Parameters           pData points to the compressed data object to be
 *                      expanded.
 *
 *                      ppExpanded points to the location where the pointer to the
 *                      expanded data object created by ExpandObject is to be stored.
 *                      The memory used to store the expanded data object is allocated
 *                      by the DosAllocMem function.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      ExpandObject returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_ACCESS_DENIED
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      EXPAND_INVALID_PARAMETER    ..... The pData or ppExpanded parameters
 *                                                        contain a pointer to an invalid
 *                                                        address.
 *
 *                      EXPAND_INVALID_METHOD       ..... The compressed data object was
 *                                                        compressed using a method not
 *                                                        supported by this version of the
 *                                                        library.
 *
 *                      EXPAND_CRC                  ..... The contents of the input compressed
 *                                                        data object appear to have been
 *                                                        corrupted.
 *
 *                      EXPAND_INTERNAL_ERROR       ..... An internal error occurred in the
 *                                                        data expansion function.
 */

extern USHORT EXPENTRY CompressVersion( void );
/* Purpose              This function retrieves the currently installed version
 *                      of the data compression library.
 *
 * Return Value         The major version number is stored in the high order
 *                      byte. The minor version number is stored in the low
 *                      order byte.
 */

extern APIRET EXPENTRY ReadAndExpand( HFILE hfile,PBYTE * ppExpanded );
/* Purpose              This function reads a compressed data object
 *                      from a disk file, expands the compressed
 *                      data, and returns the expanded data to
 *                      the caller.
 *
 * Parameters           hfile contains the handle to the file
 *                      being read. ReadAndExpand assumes that
 *                      the file pointer has already been set
 *                      to the start of the compressed data
 *                      object.
 *
 *                      ppExpanded points to the location where the pointer to the
 *                      expanded data object created by ExpandObject is to be stored.
 *                      The memory used to store the expanded data object is allocated
 *                      by the DosAllocMem function.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      ReadAndExpand returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_ACCESS_DENIED
 *                      ERROR_INVALID_HANDLE
 *                      ERROR_NOT_DOS_DISK
 *                      ERROR_LOCK_VIOLATION
 *                      ERROR_BROKEN_PIPE
 *                      ERROR_MORE_DATA
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      EXPAND_INVALID_PARAMETER    ..... The pData or ppExpanded parameters
 *                                                        contain a pointer to an invalid
 *                                                        address.
 *
 *                      EXPAND_INVALID_METHOD       ..... The compressed data object was
 *                                                        compressed using a method not
 *                                                        supported by this version of the
 *                                                        library.
 *
 *                      EXPAND_CRC                  ..... The contents of the input compressed
 *                                                        data object appear to have been
 *                                                        corrupted.
 *
 *                      EXPAND_INTERNAL_ERROR       ..... An internal error occurred in the
 *                                                        data expansion function.
 */

extern APIRET EXPENTRY CompressAndWrite( HFILE hfile,
                                         const BYTE * pData,
                                         ULONG dataSize,
                                         BYTE method,
                                         BYTE level );
/* Description          This function compresses the designated memory object
 *                      and then writes the compressed data to the specified
 *                      disk file.
 *
 * Parameters           hfile contains the handle to the file being
 *                      written. CompressAndWrite assumes that the
 *                      file pointer has already been set to the
 *                      location where the compressed object is to
 *                      be stored.
 *
 *                      pData points to the data object that is to be compressed.
 *                      This parameter may be set to NULL if dataSize is zero.
 *
 *                      dataSize contains the size in bytes of the input data object.
 *                      If dataSize is zero, the METHOD_STORE compression
 *                      method will always be used.
 *
 *                      method contains the compression method to be used to
 *                      compress the data object. This parameter can be set to
 *                      one of the following values:
 *
 *                          METHOD_STORE    ..... store with no compression.
 *                          METHOD_DEFLATE  ..... use ZIP style deflation.
 *                          METHOD_BEST     ..... use best available method
 *                                                  (currently METHOD_DEFLATE).
 *
 *                      level contains a value from 0 - 9 indicating the compression
 *                      level to be used. A value of 0 forces the data to be STORED.
 *                      A value of 1 gives fast but minimal compression. A value of
 *                      9 gives slow but maximal compression. Usually a compression
 *                      level of 4 or 5 provides a decent speed/compression tradeoff.
 *
 * Return Value         A return value of 0 indicates that no error occured, otherwise
 *                      CompressAndWrite returns one of the following error codes:
 *
 *                      Possible OS/2 errors
 *                      --------------------
 *                      ERROR_NOT_ENOUGH_MEMORY
 *                      ERROR_INVALID_PARAMETER
 *                      ERROR_INTERRUPT
 *                      ERROR_INVALID_ADDRESS
 *                      ERROR_ACCESS_DENIED
 *                      ERROR_LOCKED
 *                      ERROR_CROSSES_OBJECT_BOUNDARY
 *                      ERROR_INVALID_HANDLE
 *                      ERROR_WRITE_PROTECT
 *                      ERROR_NOT_DOS_DISK
 *                      ERROR_WRITE_FAULT
 *                      ERROR_LOCK_VIOLATION
 *                      ERROR_BROKEN_PIPE
 *
 *                      Compression library specific errors
 *                      -----------------------------------
 *                      COMPRESS_INVALID_METHOD     ..... The specified compression method
 *                                                        is not supported by this version of
 *                                                        the library.
 *
 *                      COMPRESS_INVALID_LEVEL      ..... The compression level parameter
 *                                                        contained an invalid value.
 *
 *                      COMPRESS_INVALID_PARAMETER  ..... The pData, dataSize, or ppCompressed
 *                                                        parameter contained an invalid
 *                                                        value.
 *
 *                      COMPRESS_INTERNAL_ERROR     ..... An internal error occured in the
 *                                                        compression function.
 */
/*****************************************************************************/

#ifdef __cplusplus
}
#endif

#endif /* _COMPRESS_INCLUDED */
