/*****************************************************************************
 * UnSmash.c -- A sample program to demonstrate the usage of the
 *              ExpandObject API.
 *
 *              Usage: UNSMASH sourcefile destfile
 *
 *              Reads the contents of sourcefile into memory, decompresses it,
 *              and writes the expanded data to destfile.
 *****************************************************************************/

#define INCL_DOSFILEMGR
#define INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>
#include <Compress.h>

int main( int argc,char *argv[] )
{
    APIRET              rc          = 0;
    HFILE               hsource     = NULLHANDLE,
                        hdest       = NULLHANDLE;
    ULONG               filesize    = 0;
    PBYTE               pData       = NULL;
    PCOMPRESSED_DATA    pCompressed = NULL;

    if ( argc != 3 )
        {
        printf( "Usage: UNSMASH sourcefile destfile\n" );
        return 1;
        }

    /* Open the source file */
    {
        ULONG ulAction = 0;

        rc = DosOpen( argv[1],
                      &hsource,
                      &ulAction,
                      0,
                      FILE_NORMAL,
                      OPEN_ACTION_FAIL_IF_NEW|OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_FLAGS_SEQUENTIAL|OPEN_SHARE_DENYWRITE|OPEN_ACCESS_READONLY,
                      0 );
        if ( rc )
            {
            printf( "Error %u opening file %s.\n",rc,argv[1] );
            return 1;
            }
    }

    /* Determine size of source file */
    {
        FILESTATUS3 fs;

        rc = DosQueryFileInfo( hsource,FIL_STANDARD,&fs,sizeof(fs) );
        if ( rc )
            {
            printf( "Error %u returned by DosQueryFileInfo.\n",rc );
            return 1;
            }
        filesize = fs.cbFile;
    }

    /* Allocate storage for contents of source file. */
    rc = DosAllocMem( (PPVOID)&pCompressed,filesize,PAG_READ|PAG_WRITE|PAG_COMMIT );
    if ( rc )
        {
        printf( "Error %u allocating memory for source file contents.\n",rc );
        return 1;
        }

    /* Read the source file into memory */
    {
        ULONG ulBytesRead = 0;

        rc = DosRead( hsource,pCompressed,filesize,&ulBytesRead );
        if ( rc )
            {
            printf( "Error %u reading source file into memory.\n",rc );
            return 1;
            }
        if ( ulBytesRead != filesize )
            {
            printf( "Unable to read entire source file into memory.\n" );
            return 1;
            }
    }

    /* Now expand the source file */
    rc = ExpandObject( pCompressed,&pData );
    switch ( rc ) {
        case 0:
            break;

        case EXPAND_INVALID_PARAMETER:
            printf( "An invalid parameter was passed to ExpandObject.\n" );
            return 1;

        case EXPAND_INVALID_METHOD:
            printf( "The compressed data object was compressed using\n" );
            printf( "a method not supported by this version of the library.\n" );
            return 1;

        case EXPAND_CRC:
            printf( "The contents of the compressed data object appear to have been corrupted.\n" );
            return 1;

        case EXPAND_INTERNAL_ERROR:
            printf( "An internal error occurred in ExpandObject.\n" );
            return 1;

        default:
            printf( "Error %u occurred in ExpandObject.\n",rc );
            return 1;
        }

    /* Create the destination file */
    {
        ULONG ulAction = 0;

        rc = DosOpen( argv[2],
                      &hdest,
                      &ulAction,
                      0,
                      FILE_NORMAL,
                      OPEN_ACTION_CREATE_IF_NEW|OPEN_ACTION_REPLACE_IF_EXISTS,
                      OPEN_FLAGS_SEQUENTIAL|OPEN_SHARE_DENYREADWRITE|OPEN_ACCESS_READWRITE,
                      0 );
        if ( rc )
            {
            printf( "Error %u opening file %s.\n",rc,argv[1] );
            return 1;
            }
    }

    /* Write the expanded data to the destination file */
    {
        ULONG ulBytesWritten = 0;

        rc = DosWrite( hdest,pData,pCompressed->ulOriginalSize,&ulBytesWritten );
        if ( rc )
            {
            printf( "Error %u writing to destination file.\n",rc );
            return 1;
            }
        if ( ulBytesWritten != pCompressed->ulOriginalSize )
            {
            printf( "Unable to write entire expanded data block to destination file.\n" );
            return 1;
            }
    }

    /* Clean up used resources and exit */
    DosClose( hsource );
    DosClose( hdest );
    DosFreeMem( pData );
    DosFreeMem( pCompressed );

    return 0;
}

