/*****************************************************************************
 * Smash.c -- A sample program to demonstrate the usage of the
 *            CompressObject API.
 *
 *            Usage: SMASH sourcefile destfile
 *
 *            Reads the contents of sourcefile into memory, compresses it,
 *            and writes the compressed data to destfile.
 *****************************************************************************/

#define INCL_DOSFILEMGR
#define INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>
#include <Compress.h>

int main( int argc,char *argv[] )
{
    APIRET              rc          = 0;
    HFILE               hsource     = NULLHANDLE,
                        hdest       = NULLHANDLE;
    ULONG               filesize    = 0;
    PBYTE               pData       = NULL;
    PCOMPRESSED_DATA    pCompressed = NULL;

    if ( argc != 3 )
        {
        printf( "Usage: SMASH sourcefile destfile\n" );
        return 1;
        }

    /* Open the source file */
    {
        ULONG ulAction = 0;

        rc = DosOpen( argv[1],
                      &hsource,
                      &ulAction,
                      0,
                      FILE_NORMAL,
                      OPEN_ACTION_FAIL_IF_NEW|OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_FLAGS_SEQUENTIAL|OPEN_SHARE_DENYWRITE|OPEN_ACCESS_READONLY,
                      0 );
        if ( rc )
            {
            printf( "Error %u opening file %s.\n",rc,argv[1] );
            return 1;
            }
    }

    /* Determine size of source file */
    {
        FILESTATUS3 fs;

        rc = DosQueryFileInfo( hsource,FIL_STANDARD,&fs,sizeof(fs) );
        if ( rc )
            {
            printf( "Error %u returned by DosQueryFileInfo.\n",rc );
            return 1;
            }
        filesize = fs.cbFile;
    }

    /* Allocate storage for contents of source file. */
    rc = DosAllocMem( (PPVOID)&pData,filesize,PAG_READ|PAG_WRITE|PAG_COMMIT );
    if ( rc )
        {
        printf( "Error %u allocating memory for source file contents.\n",rc );
        return 1;
        }

    /* Read the source file into memory */
    {
        ULONG ulBytesRead = 0;

        rc = DosRead( hsource,pData,filesize,&ulBytesRead );
        if ( rc )
            {
            printf( "Error %u reading source file into memory.\n",rc );
            return 1;
            }
        if ( ulBytesRead != filesize )
            {
            printf( "Unable to read entire source file into memory.\n" );
            return 1;
            }
    }

    /* Now compress the source file */
    rc = CompressObject( pData,filesize,&pCompressed,METHOD_DEFLATE,5 );
    switch ( rc ) {
        case 0:
            /* Print compression statistics */
            printf( "Compress.DLL version    = %u.%.2u\n",
                    (ULONG)HIBYTE(pCompressed->version),
                    (ULONG)LOBYTE(pCompressed->version) );
            printf( "Compression method      = " );
            switch ( pCompressed->method ) {
                case METHOD_STORE:      printf( "STORED" ); break;
                case METHOD_DEFLATE:    printf( "DEFLATED" ); break;
                default:                printf( "unknown" ); break;
                }
            printf( "\n" );
            printf( "Size of input data      = %u bytes\n",pCompressed->ulOriginalSize );
            printf( "Size of compressed data = %u bytes\n",pCompressed->ulCompressedSize );
            if ( filesize == 0 ) filesize = 1; /* prevent divide by zero error */
            printf( "Compression ratio       = %.2f%%\n",
                    100.0 - (100.0*(double)pCompressed->ulCompressedSize/(double)filesize) );
            break;

        case COMPRESS_INVALID_METHOD:
            printf( "Invalid compression method specified.\n" );
            return 1;

        case COMPRESS_INVALID_LEVEL:
            printf( "Invalid compression level specified.\n" );
            return 1;

        case COMPRESS_INVALID_PARAMETER:
            printf( "An invalid parameter was passed to CompressObject.\n" );
            return 1;

        case COMPRESS_INTERNAL_ERROR:
            printf( "An internal error occurred in CompressObject.\n" );
            return 1;

        default:
            printf( "Error %u occurred in CompressObject.\n",rc );
            return 1;
        }

    /* Create the destination file */
    {
        ULONG ulAction = 0;

        rc = DosOpen( argv[2],
                      &hdest,
                      &ulAction,
                      0,
                      FILE_NORMAL,
                      OPEN_ACTION_CREATE_IF_NEW|OPEN_ACTION_REPLACE_IF_EXISTS,
                      OPEN_FLAGS_SEQUENTIAL|OPEN_SHARE_DENYREADWRITE|OPEN_ACCESS_READWRITE,
                      0 );
        if ( rc )
            {
            printf( "Error %u opening file %s.\n",rc,argv[1] );
            return 1;
            }
    }

    /* Write the compressed data to the destination file */
    {
        ULONG ulBytesWritten = 0;

        rc = DosWrite( hdest,
                       pCompressed,
                       COMPRESSED_HEADER_SIZE + pCompressed->ulCompressedSize,
                       &ulBytesWritten );
        if ( rc )
            {
            printf( "Error %u writing to destination file.\n",rc );
            return 1;
            }
        if ( ulBytesWritten != COMPRESSED_HEADER_SIZE+pCompressed->ulCompressedSize )
            {
            printf( "Unable to write entire compressed data block to destination file.\n" );
            return 1;
            }
    }

    /* Clean up used resources and exit */
    DosClose( hsource );
    DosClose( hdest );
    DosFreeMem( pData );
    DosFreeMem( pCompressed );

    return 0;
}

