// Revision: 42 1.41.3.1 source/ui/extapp/itbar.cpp, toolbar, ioc.v400 
/*******************************************************************************
* FILE NAME: itbar.cpp                                                         *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in itbar.hpp.                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C" {
  #define INCL_WINFRAMEMGR
  #define INCL_WININPUT
  #define INCL_WINTRACKRECT
  #define INCL_WINPOINTERS
  #define INCL_WINWINDOWMGR
  #define INCL_WINSYS
  #include <iwindefs.h>
}

#include <itbar.hpp>

#include <icmdhdr.hpp>
#include <icmnfun.hpp>
#include <icolor.hpp>
#include <icvhdr.hpp>
#include <idmhndlr.hpp>
#include <iexcept.hpp>
#include <igrport.hpp>
#include <ibundles.hpp>
#include <igbase2d.hpp>
#include <igline2d.hpp>
#include <inotifev.hpp>
#include <iplatfrm.hpp>
#include <ipoint.hpp>
#include <ireslib.hpp>
#include <istring.hpp>
#include <isysmenu.hpp>
#include <ititle.hpp>
#include <itbarbut.hpp>
#include <itbarcnr.hpp>
#include <itrace.hpp>
#include <itbarfrm.hpp>
#include <itimer.hpp>
#include <itbpriv.hpp>
#include <imcevt.hpp>
#include <icoordsy.hpp>
#include <iobjwin.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ITBAR_CPP_
  #include <ipagetun.h>
#endif

#include <iseq2.h>               // for ISequence

/*------------------------------------------------------------------------------
| Public styles.                                                               |
------------------------------------------------------------------------------*/
const IToolBar::Style
  IToolBar::filterMisfits              ( 0, ICNV_HIDEMISFITS ),
  IToolBar::buttonBitmapVisible        ( 0, ICNV_BTNBMPVISIBLE ),
  IToolBar::buttonTextVisible          ( 0, ICNV_BTNTEXTVISIBLE ),
  IToolBar::buttonBitmapAndTextVisible ( 0, ICNV_BTNBMPANDTEXTVISIBLE ),
  IToolBar::dragDrop                   ( 0, ICNV_DRAGDROP ),
#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  IToolBar::noDragDrop                 ( 0 ),
#endif // IC_OBSOLETE
  IToolBar::classDefaultStyle          ( WS_VISIBLE, (ICNV_HIDEMISFITS |
                                                      ICNV_BTNBMPVISIBLE) );

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
IToolBar::Style
  IToolBar::fgCurrentDefaultStyle ( WS_VISIBLE, (ICNV_HIDEMISFITS |
                                                 ICNV_BTNBMPVISIBLE) );

/*------------------------------------------------------------------------------
| Class:   IToolBarPaintHandler                                                |
|                                                                              |
| Purpose: Draws the border of the tool bar.                                   |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarPaintHandler : public IHandler {
typedef IHandler
  Inherited;
public:
  IToolBarPaintHandler ();
  ~IToolBarPaintHandler ();
protected:
virtual bool
  dispatchHandlerEvent ( IEvent& event );
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarPaintHandler::IToolBarPaintHandler ()
{
}

IToolBarPaintHandler::~IToolBarPaintHandler ()
{
}

/*------------------------------------------------------------------------------
| Class:   IToolBarObject                                                      |
|                                                                              |
| Purpose: Points to a window object that has been added to the toolbar.       |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarObject  {
public:
  IToolBarObject ( IWindow *childWindow );

void
  setVisible        ( bool visibility );

bool
  isVisible         ( ) const { return fVisible; }

IWindow
 *window            ( ) const { return pwinCl; }

private:
IWindow
 *pwinCl;
bool
  fVisible;
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarObject::IToolBarObject ( IWindow* childWindow )
              : pwinCl(childWindow),
                fVisible(true)
{
}

void IToolBarObject::setVisible ( bool visibility )
{
  fVisible = visibility;
}

/*------------------------------------------------------------------------------
| IToolBarObjectList                                                           |
|                                                                              |
| Implementation class.                                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarObjectList : public ISequence < IToolBarObject* >
{
public:
  IToolBarObjectList ( );
 ~IToolBarObjectList ( );
};

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IToolBarObjectList::IToolBarObjectList                                       |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IToolBarObjectList :: IToolBarObjectList ( )
  : ISequence < IToolBarObject* > ( )
{ }

/*------------------------------------------------------------------------------
| IToolBarObjectList::~IToolBarObjectList                                      |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IToolBarObjectList :: ~IToolBarObjectList ( )
{ }

/*------------------------------------------------------------------------------
| IToolBarList class                                                           |
|                                                                              |
| Implementation class.                                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarList : public IVPtrSequence<IToolBar*>
{
public:
  IToolBarList ();
 ~IToolBarList ();
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarList::IToolBarList ()
{
}

IToolBarList::~IToolBarList ()
{
}

/*------------------------------------------------------------------------------
| Class:   IToolBarFloatingHandler                                             |
|                                                                              |
| Purpose: Hides the floating frame window when the user requests that it be   |
|          closed.                                                             |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarFloatingHandler : public ICommandHandler {
typedef ICommandHandler
  Inherited;
public:
  IToolBarFloatingHandler ();
  ~IToolBarFloatingHandler ();
protected:
virtual bool
  systemCommand ( ICommandEvent& event );
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarFloatingHandler::IToolBarFloatingHandler ()
{
}

IToolBarFloatingHandler::~IToolBarFloatingHandler ()
{
}

/*------------------------------------------------------------------------------
| IToolBarFloatingHandler::systemCommand                                       |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBarFloatingHandler::systemCommand ( ICommandEvent& event )
{
#ifdef IC_WIN
  switch  ( event.commandId() & 0xFFFFFFF0 )
#endif
#ifdef IC_MOTIFPM
  switch  ( event.commandId() )
#endif
  {
     case SC_CLOSE:
     {
       /********************************************************************/
       /* Clicking on the close button on the toolbar floating frame will  */
       /* not destruct the frame and the toolbar in it.  Instead, the      */
       /* toolbar/frame will be hidden.  This behavior is documented and   */
       /* is consistent with past toolbar behavior, in which the toolbar's */
       /* was set to hidden.                                               */
       /********************************************************************/
       ((IFrameWindow*)event.window())->client()->hide();
       return true;
     }
  } // switch
  return false;
}

/*------------------------------------------------------------------------------
| Class:   IToolBarTrackHandler                                                |
|                                                                              |
| Purpose: This handler is attached to the floating frame of a toolbar to      |
|          handle move requests and allow the user to "drag" the toolbar back  |
|          to the owning frame as a frame extension.                           |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarTrackHandler : public IHandler {
typedef IHandler
  Inherited;
public:
  IToolBarTrackHandler ( IToolBar* tbar, IToolBarData* tbardata );
  ~IToolBarTrackHandler ( );
void
  checkCursor ();

#ifdef IC_WIN
  void DrawTrackRect(IGrafPort& gc, LPRECT lprc);
  bool WinTrackRect (IWindowHandle hwnd, void* , PTRACKINFO lpTrackInfo);
#endif

protected:
virtual bool
  dispatchHandlerEvent ( IEvent& event );

private:
IToolBar*
  toolBar;
IToolBarData*
  fToolBarData;
IPointerHandle
  pointer;
IPointerHandle
  tbarLeft;
IPointerHandle
  tbarRight;
IPointerHandle
  tbarBottom;
IPointerHandle
  tbarTop;
IPointerHandle
  tbarLeft2;
IPointerHandle
  tbarRight2;
IPointerHandle
  tbarBottom2;
IPointerHandle
  tbarTop2;
IRectangle
  frameRect;
IToolBar::Location
  newLocation;
IToolBar*
  targetTbar;
IToolBarContainer*
  targetCnr;
IWindowHandle
  lastHandle;
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarTrackHandler::IToolBarTrackHandler ( IToolBar* tbar, IToolBarData* tbardata )
                     : toolBar(tbar), fToolBarData(tbardata), pointer(0),
                       tbarLeft(0), tbarRight(0), tbarBottom(0), tbarTop(0),
                       tbarLeft2(0), tbarRight2(0), tbarBottom2(0), tbarTop2(0),
                       targetTbar(0), targetCnr(0)
{
}

IToolBarTrackHandler::~IToolBarTrackHandler ()
{
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IToolBarHandler::DrawTrackRect                                               |
------------------------------------------------------------------------------*/
void IToolBarTrackHandler::DrawTrackRect(IGrafPort& gc, LPRECT lprc)
{
  gc.draw(IGRect2D(lprc->left, lprc->top, lprc->right, lprc->bottom));
}

/*------------------------------------------------------------------------------
| IToolBarHandler::WinTrackRect                                                |
| This function lifted from cclnt\cnr\cnrs3p1.c.  It is an incomplete          |
| implementation of the PM function of the same name.  This version is         |
| designed specifically for use in the toolbar track handler.                  |
------------------------------------------------------------------------------*/
bool IToolBarTrackHandler::WinTrackRect (IWindowHandle hwnd,
                                            void* ,
                                            PTRACKINFO lpTrackInfo)
{
  IMODTRACE_DEVELOP("IToolBarTrackHandler::WinTrackRect");

  bool       fTrack = TRUE;
  IWindowHandle hwndFocus = GetFocus();
  RECT          rcTrack, rcBoundary;
  POINT         ptCursor;
  int           xLast, yLast;
  MSG           msg;
  WORD          wQuitMessage;

  // Create a window to capture the mouse. The hwnd that is passed in may
  // be the desktop which can not be used to capture the mouse so we need
  // a new window for the capture.
  IFrameWindow objWindow;

  // Getting the Desktop's device context.  Be sure to release it later.
  IPresSpaceHandle presSpace(IGETPS(HWND_DESKTOP), HWND_DESKTOP);
  IRootGrafPort rootPort(presSpace);

  IColor white(IColor::white);
  IGrafBundle whiteFrame(IBasicColor(white), IAttributeState::kFrame);
  whiteFrame.setFrameTransferMode(IColorTransferMode::kXOR);
  whiteFrame.setFramePen(IPen(IPen::kSolid,
                              (GCoordinate)lpTrackInfo->cxBorder));
  ILinkedGrafPort gc(&rootPort, &whiteFrame);

  /*-----------------------------------------------------------*/
  /* figure out which button will signal the end of the drag;  */
  /* if more than one button is pressed, the left takes        */
  /* precedence, followed by the right, followed by the middle */
  /*-----------------------------------------------------------*/
  wQuitMessage = (GetKeyState (VK_LBUTTON) < 0) ? WM_LBUTTONUP :
                 (GetKeyState (VK_RBUTTON) < 0) ? WM_RBUTTONUP :
                                                  WM_MBUTTONUP;

  /*------------------------------------------------------*/
  /* get the current mouse position in window coordinates */
  /* (we're overloading rcTrack here, but it saves stack) */
  /*------------------------------------------------------*/
//  GetCursorPos (&ptCursor);
  DWORD messagePos = GetMessagePos();
  ptCursor.x = LOWORD(messagePos);
  ptCursor.y = HIWORD(messagePos);

  GetWindowRect (hwnd, &rcTrack);
  xLast = (int)(ptCursor.x - rcTrack.left);
  yLast = (int)(ptCursor.y - rcTrack.top);

  ITRACE_DEVELOP("Cursor pos is " + IPoint(ptCursor).asString() +
                 "Window rect is " + IRectangle(rcTrack).asString() );

  /*------------------------------------------------*/
  /* make a working copy of the tracking rectangles */
  /*------------------------------------------------*/
  rcTrack    = lpTrackInfo->rclTrack;

  /*-----------------------------------------------------------------*/
  /* make sure the mouse doesn't move outside the boundary rectangle */
  /*-----------------------------------------------------------------*/
  ITRACE_DEVELOP(IString("lpTrackInfo->rclBoundary=(") +
             IString( lpTrackInfo->rclBoundary.left) + IString(",") +
             IString( lpTrackInfo->rclBoundary.top) + IString(",") +
             IString( lpTrackInfo->rclBoundary.right) + IString(",") +
             IString( lpTrackInfo->rclBoundary.bottom) + IString(")") );
  // We add 1 to the right/bottom to prevent oscillation of the tracking
  // rectangle by 1 pixel.  Seems that if limits of the boundary are ==
  // the mouse position oscillates around the point.
  rcBoundary.left   = ptCursor.x -
                      (rcTrack.left - lpTrackInfo->rclBoundary.left );
  rcBoundary.right  = ptCursor.x +
                      (lpTrackInfo->rclBoundary.right - rcTrack.right ) + 1 ;
  rcBoundary.top    = ptCursor.y -
                      (rcTrack.top  - lpTrackInfo->rclBoundary.top );
  rcBoundary.bottom = ptCursor.y +
                      (lpTrackInfo->rclBoundary.bottom - rcTrack.bottom ) + 1;
  ITRACE_DEVELOP(IString("rcBoundary=(") +
             IString( rcBoundary.left) + IString(",") +
             IString( rcBoundary.top) + IString(",") +
             IString( rcBoundary.right) + IString(",") +
             IString( rcBoundary.bottom) + IString(")") );

  SetCapture ( objWindow.handle() );
  ITRACE_DEVELOP(IString("Capturing pointer, hwnd=") +
                 IString(objWindow.handle().asUnsigned()).d2x());
//  ClipCursor (&rcBoundary);

  DrawTrackRect (gc, &rcTrack);

  while ( fTrack &&
          GetMessage (&msg, 0, 0, 0))
  {
    if (msg.message == wQuitMessage)
      fTrack = FALSE;

    else
    {
      switch (msg.message)
      {
        /*-----------------------*/
        /* eat keyboard messages */
        /*-----------------------*/
        case WM_KEYUP:
        case WM_SYSKEYUP:
          break;
        case WM_CHAR:
        case WM_KEYDOWN:
        case WM_SYSCHAR:
        case WM_SYSKEYDOWN:
        {
          switch (msg.wParam )
          {
            case VK_ESCAPE:
            {
              fTrack = FALSE;
              break;
            }
          }
        }
        break;

        case WM_MOUSEMOVE:
        {
          POINT pt;
          GetCursorPos( &pt );

          int  xDelta = (int)(pt.x - xLast);
          int  yDelta = (int)(pt.y - yLast);

          /*-------------------*/
          /* erase, move, draw */
          /*-------------------*/
          this->DrawTrackRect (gc, &rcTrack);
          OffsetRect (&rcTrack, xDelta, yDelta);
          this->DrawTrackRect (gc, &rcTrack);
          xLast += xDelta;
          yLast += yDelta;
        }
        break;

        /*---------------------------------*/
        /* eat other mouse button messages */
        /*---------------------------------*/
        case WM_LBUTTONUP:
        case WM_MBUTTONUP:
        case WM_RBUTTONUP:
        case WM_LBUTTONDOWN:
        case WM_MBUTTONDOWN:
        case WM_RBUTTONDOWN:
        case WM_LBUTTONDBLCLK:
        case WM_MBUTTONDBLCLK:
        case WM_RBUTTONDBLCLK:
          break;

        default:
          DispatchMessage (&msg);
          break;
      }
    }
  }

  /*----------------------------------------------------------------------*/
  /* if we fell out of the loop because GetMessage returned FALSE         */
  /* (i.e. PostQuitMessage was called), put the WM_QUIT back on the queue */
  /*----------------------------------------------------------------------*/
  if (fTrack && (GetFocus() == hwndFocus) && (GetCapture() == hwnd))
  {
    PostQuitMessage (msg.wParam);
  }

  this->DrawTrackRect (gc, &rcTrack);
  ClipCursor (NULL);
  ReleaseCapture ();
  ITRACE_DEVELOP(IString("Releasing capture of pointer, hwnd=") +
                 IString(objWindow.handle().asUnsigned()).d2x());

  /*---------------------------------------*/
  /* if the user pressed ESC, return FALSE */
  /*---------------------------------------*/
  if ((msg.message == WM_KEYDOWN) && (msg.wParam == VK_ESCAPE))
  {
    IRELEASEPS( HWND_DESKTOP, presSpace );
    return (FALSE);
  }
  /*---------------------------------------*/
  /* otherwise return the drop coordinates */
  /*---------------------------------------*/
  else
  {
    lpTrackInfo->rclTrack = rcTrack;
    IRELEASEPS( HWND_DESKTOP, presSpace );
    return (TRUE);
  }
}
#endif

/*------------------------------------------------------------------------------
| Class:   IToolBarStatics                                                     |
|                                                                              |
| Purpose: Static data used for all instances of IToolBars                     |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarStatics {
public:
  IToolBarStatics ( ) { };
  ~IToolBarStatics ( );
static IColor
  *colorButtonDark,
  *colorButtonLight;
};
#pragma pack(pop)
#pragma enum(pop)

static IToolBarStatics toolBarStatics;
IColor*  IToolBarStatics::colorButtonDark  = 0;
IColor*  IToolBarStatics::colorButtonLight = 0;

/*------------------------------------------------------------------------------
| IToolBarStatics::~IToolBarStatics                                            |
------------------------------------------------------------------------------*/
IToolBarStatics::~IToolBarStatics( )
{
    IMODTRACE_DEVELOP("IToolBarStatics::~IToolBarStatics");
#if IC_STATIC_PRIORITY_SUPPORTED
    // Only do deletes if static object initialization ordering is supported.
    if (colorButtonDark)
    {
      delete colorButtonDark;
      colorButtonDark = 0;
    }
    if (colorButtonLight)
    {
      delete colorButtonLight;
      colorButtonLight = 0;
    }
#endif
}

/*------------------------------------------------------------------------------
| Class:   IToolBarData                                                        |
|                                                                              |
| Purpose: Store private data used by the class.                               |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarData
{
public:
  IToolBarData ( IFrameWindow* frame = 0 );

  ~IToolBarData();

IFrameWindow*
  owningFrame;
IToolBarContainer*
  toolBarContainer;
IString
  toolBarName;
IToolBarPaintHandler
  paintHandler;
bool
  skipLayout;
static long
  currentDefaultPad;
static unsigned long
  currentDefaultGroupPad,
  currentDefaultMisfitWidth;
static ISize
  currentDefaultMargin;
IToolBarObjectList*
  objectList;
IToolBar*
  precedingToolBar;
IToolBarFrameWindow*
  floatingFrame;
IToolBarFloatingHandler*
  floatingHandler;
IToolBarTrackHandler*
  trackHandler;
IPoint
  floatingPosition;
bool
  isInitialized,
  isDragDropInitialized;
IToolBar::Location
  currentLocation;
ISize
  margin;
unsigned long
  misfitWidth;
bool
  isExpanded,
  isShown,
  skipShowProcessing;
unsigned long
  windowId;
#ifdef IC_WIN
bool
  bCheckingDragStart;
IPoint
  pointLButtonDown;
IEventData
  parameter1,
  parameter2;
#endif
};

#pragma pack(pop)
#pragma enum(pop)


/*------------------------------------------------------------------------------
| IToolBarData::IToolBarData                                                   |
------------------------------------------------------------------------------*/
IToolBarData::IToolBarData ( IFrameWindow* frame )
             : owningFrame(frame)
             , toolBarContainer(0)
             , toolBarName("")
             , paintHandler()
             , skipLayout(false)
             , objectList(new IToolBarObjectList)
             , precedingToolBar(0)
             , floatingFrame(0)
             , floatingHandler(0)
             , trackHandler(0)
             , floatingPosition(IPoint(0,0))
             , isInitialized(false)
             , isDragDropInitialized(false)
             , currentLocation(IToolBar::hidden)
             , isExpanded(true)
             , isShown(true)
             , skipShowProcessing(false)
             , windowId(0)
#ifdef IC_WIN
             , bCheckingDragStart( false)
#endif
{ }

/*------------------------------------------------------------------------------
| IToolBarData::~IToolBarData                                                  |
------------------------------------------------------------------------------*/
IToolBarData::~IToolBarData ()
{
}

long          IToolBarData::currentDefaultPad = -1;
unsigned long IToolBarData::currentDefaultGroupPad = 8;
unsigned long IToolBarData::currentDefaultMisfitWidth = 65;
ISize         IToolBarData::currentDefaultMargin = ISize(7,4);

/*------------------------------------------------------------------------------
| IToolBarTrackHandler::checkCursor                                            |
|                                                                              |
------------------------------------------------------------------------------*/
void IToolBarTrackHandler::checkCursor ()
{
  IMODTRACE_DEVELOP("IToolBarTrackHandler::checkCursor");

//IC_NOTYET Motif (IToolBarTrackHandler::checkCursor not implemented yet)
#ifdef IC_PMWIN
  POINTL pnt;
  IQUERYPOINTERPOSITION(pnt.x,pnt.y);

  IWindowHandle testHandle = IWINDOWFROMPOINT
                             ( IWindow::desktopWindow()->handle(), &pnt, true );

  // Be sure we have all the toolbar pointers loaded
  if ( !tbarLeft )
  {
    try
    {
      tbarLeft = IApplication::current().resourceLibrary().loadPointer
                 (IC_TOOLBAR_POINTER_LEFT);
      tbarRight = IApplication::current().resourceLibrary().loadPointer
                  (IC_TOOLBAR_POINTER_RIGHT);
      tbarTop = IApplication::current().resourceLibrary().loadPointer
                (IC_TOOLBAR_POINTER_TOP);
      tbarBottom = IApplication::current().resourceLibrary().loadPointer
                   (IC_TOOLBAR_POINTER_BOTTOM);
      tbarLeft2 = IApplication::current().resourceLibrary().loadPointer
                  (IC_TOOLBAR_POINTER_LEFT_2);
      tbarRight2 = IApplication::current().resourceLibrary().loadPointer
                   (IC_TOOLBAR_POINTER_RIGHT_2);
      tbarTop2 = IApplication::current().resourceLibrary().loadPointer
                 (IC_TOOLBAR_POINTER_TOP_2);
      tbarBottom2 = IApplication::current().resourceLibrary().loadPointer
                    (IC_TOOLBAR_POINTER_BOTTOM_2);
    }
    catch ( IException& )
    {
      // Could not load pointers from our DLL so use system app pointer instead
      tbarLeft = ISystemPointerHandle ( ISystemPointerHandle::standardApplication );
      tbarRight = ISystemPointerHandle ( ISystemPointerHandle::standardApplication );
      tbarTop = ISystemPointerHandle ( ISystemPointerHandle::standardApplication );
      tbarBottom = ISystemPointerHandle ( ISystemPointerHandle::standardApplication );
      tbarLeft2 = tbarLeft;
      tbarRight2 = tbarRight;
      tbarTop2 = tbarTop;
      tbarBottom2 = tbarBottom;
    }
  }

  if ( testHandle == fToolBarData->owningFrame->handle() )
  {
    if ( !pointer )
    {
      // Save old pointer (so we can restore it later)
      pointer = IQUERYPOINTER;
    }

    targetTbar = 0;
    targetCnr = 0;

    IPoint pt( pnt );

    IRectangle newRect = frameRect;

    if ( ICoordinateSystem::nativeOrientation() !=
             ICoordinateSystem::kOriginLowerLeft )
      // The logic to determine which border we are over is written based on
      // the lower left.  The point is native, so if the native orientation
      // is not lower left the point needs to be converted.
      pt.setY( IWindow::desktopWindow()->size().height() - pt.y() );

    if ( ICoordinateSystem::applicationOrientation() !=
             ICoordinateSystem::kOriginLowerLeft )
      // Additionally, if the framerect is not lower left, it too needs to be
      // converted.
      newRect.moveTo( IPoint( newRect.minX(),
              IWindow::desktopWindow()->size().height() - newRect.maxY()));

    long cxLeft   = pt.x() - newRect.left();
    long cxRight  = newRect.right() - pt.x();
    long cyTop    = newRect.top() - pt.y();
    long cyBottom = pt.y() - newRect.bottom();
    if (( cxRight < cxLeft ) && ( cxRight < cyTop ) && ( cxRight < cyBottom ))
    {
      ISETPOINTER(tbarRight2);
      newLocation = IToolBar::rightOfClient;
    }
    else
    {
      if (( cxLeft < cyTop ) && ( cxLeft < cyBottom ))
      {
        ISETPOINTER(tbarLeft2);
        newLocation = IToolBar::leftOfClient;
      }
      else
      {
        if ( cyBottom < cyTop )
        {
          ISETPOINTER(tbarBottom2);
          newLocation = IToolBar::belowClient;
        }
        else
        {
          ISETPOINTER(tbarTop2);
          newLocation = IToolBar::aboveClient;
        }
      }
    }
  }
  else
  {
    // If we are over the source tool bar, then we cannot drop and it
    // must be floating (so just reset pointer if needed)
    if ( toolBar->handle() == testHandle )
    {
      if ( pointer )
        ISETPOINTER ( pointer );
      pointer = 0;
      return;
    }

    // Assume that the tool bar will be floating
    newLocation = IToolBar::floating;

    // Check if cursor is over another toolbar or toolbar container
    targetTbar = 0;
    targetCnr = 0;
    IToolBar::FrameCursor cursor(fToolBarData->owningFrame);
    for ( cursor.setToFirst();
        ( cursor.isValid() && ( newLocation == IToolBar::floating ));
          cursor.setToNext() )
    {
      IToolBar*          tbarTest = IToolBar::toolBarAt(cursor);
      IToolBarContainer* cnr = 0;

      if ( tbarTest )
        cnr = tbarTest->toolBarContainer();

      if ( tbarTest && cnr && ( tbarTest->handle() == testHandle ))
      {
        targetTbar = tbarTest;
        newLocation = targetTbar->location();
      }
      else
      {
        if ( cnr && ( cnr->handle() == testHandle ))
        {
          targetCnr = cnr;
          newLocation = tbarTest->location();
        }
      }
    }

    if ( newLocation == IToolBar::floating )
    {
       // Reset the pointer if needed
      if ( pointer )
        ISETPOINTER ( pointer );
      pointer = 0;
    }
    else
    {
      // We are not floating, so save the system pointer handle
      if ( pointer == 0 )
        pointer = IQUERYPOINTER;

      // Set the new pointer based on the new location
      switch ( newLocation )
      {
         case IToolBar::aboveClient:
            if ( targetTbar )
              ISETPOINTER(tbarTop);
            else
              ISETPOINTER(tbarTop2);
            break;
         case IToolBar::belowClient:
            if ( targetTbar )
              ISETPOINTER(tbarBottom);
            else
              ISETPOINTER(tbarBottom2);
            break;
         case IToolBar::leftOfClient:
            if ( targetTbar )
              ISETPOINTER(tbarLeft);
            else
              ISETPOINTER(tbarLeft2);
            break;
         case IToolBar::rightOfClient:
            if ( targetTbar )
              ISETPOINTER(tbarRight);
            else
              ISETPOINTER(tbarRight2);
            break;
      }
    }
  }
#endif //IC_PMWIN
}

/*------------------------------------------------------------------------------
| IToolBarTrackHandler::dispatchHandlerEvent                                   |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBarTrackHandler::dispatchHandlerEvent ( IEvent& event )
{
//IC_NOTYET Motif (IToolBarTrackHandler::dispatchHandlerEvent not implemented yet)
#ifdef IC_PMWIN
  // Check for move events
  if (( event.eventId() == WM_TRACKFRAME ) &&
      ( event.parameter1().asUnsignedLong() & TF_MOVE ) &&
      ( toolBar->allowsDragDrop() ))
  {
    bool   trackOk;
    IPoint    startPos;
    TRACKINFO trackInformation;

#ifdef IC_PM
    trackOk = event.window()->sendEvent ( WM_QUERYTRACKINFO,
                                          event.parameter1(),
                                          &trackInformation ).asUnsignedLong();
#endif
#ifdef IC_WIN
    {
       trackInformation.cxBorder   = 2;
       trackInformation.cyBorder   = 2;
       trackInformation.cxGrid     = 1;
       trackInformation.cyGrid     = 1;
       trackInformation.cxKeyboard = 8;
       trackInformation.cyKeyboard = 8;

       RECT rect;
       IQUERYWINDOWRECT( event.handle(), &rect );

       trackInformation.rclTrack = rect;

       trackInformation.rclBoundary.top    = 0;
       trackInformation.rclBoundary.left   = 0;
       trackInformation.rclBoundary.bottom = IWindow::desktopWindow()->size().height();
       trackInformation.rclBoundary.right  = IWindow::desktopWindow()->size().width();
       // We are not changing size
       trackInformation.ptlMinTrackSize.x =
          trackInformation.rclTrack.right - trackInformation.rclTrack.left;
       trackInformation.ptlMinTrackSize.y =
          trackInformation.rclTrack.bottom - trackInformation.rclTrack.top;
       trackInformation.ptlMaxTrackSize = trackInformation.ptlMinTrackSize;

       trackInformation.fs = TF_MOVE;
       trackOk = true;
    }
#endif

    if ( trackOk )
    {
      // Set up information to use during tracking
      // to determines which edge we are over.
      frameRect = fToolBarData->owningFrame->rect();

      if ( ICoordinateSystem::isConversionNeeded() )
      {
         startPos = IRectangle(trackInformation.rclTrack).topLeft();
         startPos = ICoordinateSystem::convertToApplication (startPos,
                                      IWindow::desktopWindow()->size() );
      }
      else
         startPos = IRectangle(trackInformation.rclTrack).bottomLeft();

      newLocation = IToolBar::floating;

      lastHandle = 0;
      ITimer timer(new ITimerMemberFn0<IToolBarTrackHandler>(
                            *this,
                            &IToolBarTrackHandler::checkCursor),
                   100);
      trackOk = WinTrackRect ( IWindow::desktopWindow()->handle(), 0,
                               &trackInformation );
      timer.stop();
    }

    if ( pointer )
    {
      ISETPOINTER( pointer );
      pointer = 0;
    }

    if ( trackOk )
    {
      // The trackInformation rectangle is native, convert if needed.
      IPoint natPos = IRectangle(trackInformation.rclTrack).bottomLeft();
      IPoint newPos = natPos;
      if ( ICoordinateSystem::isConversionNeeded() )
      {
         newPos = IRectangle(trackInformation.rclTrack).topLeft();
         newPos = ICoordinateSystem::convertToApplication (newPos,
                                      IWindow::desktopWindow()->size() );
      }

      event.setResult(true);

      // Check if the toolbar should be a frame extension
      if ( newLocation != IToolBar::floating )
      {
        toolBar->postEvent(IC_UM_TOOLBAR, IC_UM_TBAR_MOVE, newLocation);
        if ( targetTbar )
        {
          IPoint pt = IWindow::mapPoint(IWindow::pointerPosition(),
                                        IWindow::desktopWindow()->handle(),
                                        targetTbar->handle());

          if ( pt.x() < (targetTbar->size().width()/2) )
          {
            toolBar->postEvent(IC_UM_TOOLBAR, IC_UM_TBAR_BEFORE,
                               (unsigned long) (void*) targetTbar);
          }
          else
          {
            toolBar->postEvent(IC_UM_TOOLBAR, IC_UM_TBAR_AFTER,
                               (unsigned long) (void*) targetTbar);
          }
        }
        else
          toolBar->postEvent(IC_UM_TOOLBAR, IC_UM_TBAR_AFTER, 0);
        return true;
      }

      // Check for a frame extension being dragged off the frame
      if ( fToolBarData->currentLocation != IToolBar::floating )
      {
        // setFloatingPosition uses native origin
        if ( ICoordinateSystem::nativeOrientation() ==
             ICoordinateSystem::kOriginLowerLeft )
          {
            // If on OS/2, do not use converted position.  Always use native
            // position, which is relative to the lower left.
            toolBar->setFloatingPosition(IWindow::mapPoint(natPos,
                                         IWindow::desktopWindow()->handle(),
                                         fToolBarData->owningFrame->handle()));
          }
         else
          {
            // If on Windows, use to new position.
            toolBar->setFloatingPosition(IWindow::mapPoint(newPos,
                                        IWindow::desktopWindow()->handle(),
                                        fToolBarData->owningFrame->handle()));
          }
         toolBar->setLocation(IToolBar::floating);
         return true;
      }

      // Normal processing (move of an existing floating toolbar)
      if ( newPos != startPos )
      {
        event.window()->moveTo(newPos);
      }
      event.window()->setFocus();
    }

    targetTbar = 0;
    targetCnr = 0;
    return true;
  }

#ifdef IC_WIN
  // These two messages show up before we actually move the tool bar and they
  // cause us to wipe out the targetTbar value. Eat them!
  if ( event.eventId() == WM_WINDOWPOSCHANGING ||
       event.eventId() == WM_WINDOWPOSCHANGED )
    return false;
#endif

  targetTbar = 0;
  targetCnr = 0;

#endif //IC_PMWIN
  // This is not a move event, so do default processing
  return false;
}

/*------------------------------------------------------------------------------
| IToolBarPaintHandler::dispatchHandlerEvent                                   |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBarPaintHandler::dispatchHandlerEvent ( IEvent& event )
{
  IMODTRACE_DEVELOP("TBarPaintHdr::dispatchHandlerEvent");

#ifdef IC_PMWIN
  // To solve a BIDI problem, we are going to also attach this handler to
  // the owning frame.  This means that we will look for the BIDI changed
  // message and then return false for all other frame events.
  if ( event.window()->isFrameWindow() )
  {
     if (( event.eventId() == IC_UM_TOOLBAR ) &&
         ( event.parameter1().asUnsignedLong() == IC_UM_TBAR_BIDI_CHANGE ))
     {
        // The frame BIDI attributes changed, so we need to update all of
        // the floating (or hidden) toolbars
        IBidiSettings bidi(*(event.window()));
        IToolBar::FrameCursor cursor( (IFrameWindow*) event.window() );
        for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
        {
          IToolBar* toolBar = IToolBar::toolBarAt(cursor);
          if ( toolBar )
          {
             if ( toolBar->location() == IToolBar::floating )
             {
               bidi.apply(*toolBar);
               bidi.apply(*(toolBar->floatingFrame()), false );
             }
             if ( toolBar->location() == IToolBar::hidden )
             {
               bidi.apply(*toolBar);
             }
          }
        }
        return true;
     }
     return false;
  }
#endif

  ITRACE_DEVELOP(event.window()->rect().asString());

  // Check for events
  switch (event.eventId())
  {
//IC_NOTYET Motif (drag/drop not implemented yet)
#ifdef IC_PMWIN
    case WM_BEGINDRAG:
    {
      IToolBar* toolBar = (IToolBar*)(event.window());
      if ( toolBar->fToolBarData->floatingFrame )
      {
        // The toolbar is already floating, so just track the frame window
        toolBar->fToolBarData->floatingFrame->sendEvent(WM_TRACKFRAME,TF_MOVE);
      }
      else
      {
        // The toolbar is in a frame extension, so create a dummy frame that
        // can be used for tracking.  Note that if the toolbar does not allow
        // drag/drop then nothing should happen
        if ( toolBar->allowsDragDrop() )
        {
          IFrameWindow trackFrame ( IC_DEFAULT_FRAME_ID,
                                    IWindow::desktopWindow(),
                                    toolBar,
                                    IRectangle(IWindow::mapPoint(IPoint(0,0),
                                               toolBar->handle(),
                                               IWindow::desktopWindow()->handle()),
                                               toolBar->size()),
                                    IFrameWindow::border );
          toolBar->fToolBarData->trackHandler->handleEventsFor(&trackFrame);
          trackFrame.sendEvent(WM_TRACKFRAME,TF_MOVE);
          toolBar->fToolBarData->trackHandler->stopHandlingEventsFor(&trackFrame);
        }
      }
    }
    return true;
#endif //IC_PMWIN

#ifdef IC_MOTIF
    case xEvent(Expose):
#endif
#ifdef IC_PMWIN
    case WM_PAINT:
#endif
    {
      // Call default painting, now in custom canvas window procedure
      event.dispatchingWindow()->dispatchRemainingHandlers(event);

      bool drawBorder = true;

#ifdef IC_MOTIF
      XExposeEvent* Xevent = (XExposeEvent*)(char*)event.parameter2();
      if ( Xevent->count != 0 )
         drawBorder = false;
#endif
      if(drawBorder)
      {
        IToolBar* toolBar = (IToolBar*)(event.window());
        ISize     size    = toolBar->size();

        IManagedPresSpaceHandle presSpace(toolBar);
        IRootGrafPort gc(presSpace);

        // Draw a border around the tool bar.  The default is a 3d border.
        // We make it a simple box, however, if one of the lines matches
        // the toolbar background color.  This is the case with the NT 3.51
        // default color scheme.  Otherwise, it looks partially painted.
        IGrafBundle bundle( IBasicColor(toolBar->backgroundColor() ==
                                        *toolBarStatics.colorButtonLight ?
                                        *toolBarStatics.colorButtonDark :
                                        *toolBarStatics.colorButtonLight));
        bundle.setDrawingOperation( IAttributeState::kFrame );

        //Set pen to hairline, the default is solid.  We must use a hairline
        //pen to assure pixel perfect drawing accross platforms.
        bundle.setFramePen(IPen(IPen::kHairline,(GCoordinate)1));

        IGPolyline2D line(3);
            GCoordinate width = (GCoordinate)size.width();
            GCoordinate height = (GCoordinate)size.height();

        line.setPoint(0, IGPoint2D(1., height-1.));
        line.setPoint(1, IGPoint2D(1., 1.));
        line.setPoint(2, IGPoint2D(width-1., 1.));
        gc.draw(line, bundle);

        bundle.setFrameColor( IBasicColor(toolBar->backgroundColor() ==
                                          *toolBarStatics.colorButtonDark ?
                                          *toolBarStatics.colorButtonLight :
                                          *toolBarStatics.colorButtonDark));

        line.setPoint(1, IGPoint2D(width-1., height-1.));
        //Reset point 2's Y-coord to one beyond what we really want, because
        //the definition of hairline pens is to not draw the end point.
        line.setPoint(2, IGPoint2D(width-1., 0));

        gc.draw(line, bundle);
      }
    }
    return true;

    case IC_UM_TOOLBAR:
    {
      IToolBar* toolbar = (IToolBar*) event.window();
      switch ( event.parameter1().asUnsignedLong() )
      {
        case IC_UM_TBAR_SET_LOCATION:
          toolbar->fToolBarData->currentLocation =
                  (IToolBar::Location) event.parameter2().asUnsignedLong();
          break;
        case IC_UM_TBAR_MOVE:
          toolbar->setLocation(
                  (IToolBar::Location) event.parameter2().asUnsignedLong());
          break;
        case IC_UM_TBAR_BEFORE:
          {
            IToolBar* targetTBar = (IToolBar*) event.parameter2().asUnsignedLong();
            toolbar->positionOnSiblings();
            toolbar->positionBehindSibling(targetTBar->handle());
            targetTBar->positionBehindSibling(toolbar->handle());
            if (targetTBar->isGroup())
            {
              toolbar->enableGroup();
              targetTBar->disableGroup();
            }
            else
            {
              toolbar->disableGroup();
            }
            toolbar->toolBarContainer()->setLayoutDistorted
                    (IWindow::layoutChanged | IWindow::immediateUpdate,0);
          }
          break;
        case IC_UM_TBAR_AFTER:
          if ( event.parameter2().asUnsignedLong() == 0 )
          {
            toolbar->enableGroup();
            toolbar->positionBehindSiblings();
            toolbar->toolBarContainer()->setLayoutDistorted
                    (IWindow::layoutChanged | IWindow::immediateUpdate,0);
          }
          else
          {
            IToolBar* tBar = (IToolBar*) event.parameter2().asUnsignedLong();
            toolbar->disableGroup();
            toolbar->positionOnSiblings();
            toolbar->positionBehindSibling(tBar->handle());
            toolbar->toolBarContainer()->setLayoutDistorted
                    (IWindow::layoutChanged | IWindow::immediateUpdate,0);
          }
          break;
        case IC_UM_TBAR_TARGET_ENTER:
          event.setResult( true );
          return true;
      }
    }
    break;

#ifdef IC_WIN
    case WM_LBUTTONDOWN:
    {
      IToolBar* toolbar = (IToolBar*) event.window();
      IMouseClickEvent mevt( event );

      // We need to use ChildWindowFromPoint instead of WindowFromPoint
      // which IMouseEvent::windowUnderPointer uses to find disabled window
      POINT pt = { (short)mevt.parameter2().number1(),
                   (short)mevt.parameter2().number2() };
      IWindowHandle childHwnd (ChildWindowFromPoint( mevt.handle(), pt));

      if ( ( mevt.mouseButton() == IMouseClickEvent::button1 ) &&
           (childHwnd) && !(ISTYLEOF( childHwnd ) & WS_DISABLED) )
      {
        // May need checking to see if this is a button or the
        // toolbar.
        // In Windows 95, the system apparently captures the pointer itself.
        // during the down event and releases it in the up event.  If we
        // mess with this incorrectly it causes no WM_COMMANDs for the
        // button clicks to be sent.
        IWindowHandle captureHandle = GetCapture( );
        if (( captureHandle != NULL ) && (captureHandle != event.handle()) )
        {
          // This means that someone else has set a capture.
          // We don't want to override it.
          // SetCapture( captureHandle );
          ITRACE_DEVELOP(IString("Pointer already captured hwnd=") +
                         IString(captureHandle.asUnsigned()).d2x() +
                         IString(" event.handle()=")+
                         IString(event.handle().asUnsigned()).d2x());
        }
        else
        {
          if (captureHandle == NULL)
             {
             ITRACE_DEVELOP(IString("capturing pointer hwnd=") +
                            IString(event.handle().asUnsigned()).d2x());
             SetCapture( event.handle() );
             }
          toolbar->fToolBarData->bCheckingDragStart = True;
          toolbar->fToolBarData->pointLButtonDown =
              mevt.mousePosition();
          toolbar->fToolBarData->parameter1 =
              event.parameter1().asUnsignedLong();
          toolbar->fToolBarData->parameter2 =
              event.parameter2().asUnsignedLong();
        }
      }
    }
    break;

    case WM_LBUTTONUP:
    {
      IToolBar* toolbar = (IToolBar*) event.window();
      ITRACE_DEVELOP("Got a button up event");
      IMouseClickEvent mevt( event );
      if ( mevt.mouseButton() == IMouseClickEvent::button1 )
      {
        if ( toolbar->fToolBarData->bCheckingDragStart )
        {
	  if (!( IPlatform::isWin9x() || IPlatform::isNTNewShell() ))
          {
            // Win95 appears to release the capture when processing buttonup
            // If we release it we get no WM_COMMAND for button clicks.
            ITRACE_DEVELOP(IString("Releasing capture GetCapture=") +
                           IString((unsigned long)GetCapture()).d2x()  );
            SetCapture( NULL );
          }
          else
          {
            // On 95, if captured by us, and not already released by WM_MOUSEMOVE
            // processing, (for example user clicks and releases mouse, without moving
            // mouse) release it here.
            IWindowHandle captureHandle = GetCapture( );
            if (( captureHandle != NULL ) && (captureHandle == event.handle()) )
            {
              ITRACE_DEVELOP(IString("Releasing capture GetCapture=") +
                             IString((unsigned long)GetCapture()).d2x()  );
              SetCapture( NULL );
            }
          }
          toolbar->fToolBarData->bCheckingDragStart = False;
        }

        toolbar->fToolBarData->pointLButtonDown = IPoint(0,0);
      }
      break;
    }
    case WM_MOUSEMOVE:
    {
      IToolBar* toolbar = (IToolBar*) event.window();
      if ( toolbar->fToolBarData->bCheckingDragStart)
      {
        IMouseClickEvent mevt( event );
        IPoint mPoint( toolbar->fToolBarData->pointLButtonDown );
        if ( abs(mevt.mousePosition().x() - mPoint.x()) >
                 2 /*DD_DEFDRAGMINDIST*/ ||
             abs(mevt.mousePosition().y() - mPoint.y()) >
                 2 /*DD_DEFDRAGMINDIST*/ )
        {
          IWindowHandle captureHandle = GetCapture( );
          ITRACE_DEVELOP(IString("mouse move event.handle()=") +
                         IString(event.handle().asUnsigned()).d2x());
          if (( captureHandle != NULL ) && (captureHandle == event.handle()) )
            {
             ITRACE_DEVELOP(IString("Releasing capture (mouse move), hwnd=") +
                            IString((unsigned long)GetCapture()).d2x() );
             SetCapture( NULL );
            }
          toolbar->fToolBarData->bCheckingDragStart = FALSE;
          // Use the original point that started the drag, not the point
          // associated with the move.
          toolbar->sendEvent( WM_BEGINDRAG,
                        IEventParameter1(mPoint.x(),
                                         mPoint.y()),
                        IEventParameter2( 0 ));
        }
      }
    }
    break;
#ifdef IC_TRACE_DEVELOP
    // Windows 95 only
    case WM_CAPTURECHANGED:
    {
       ITRACE_DEVELOP(IString("WM_CAPTURECHANGED new capture=") +
                      IString(event.parameter2().asUnsignedLong()).d2x() +
                      IString(" event.handle()=")+
                      IString(event.handle().asUnsigned()).d2x());
    }
    break;
#endif
#endif

  }

  return false;
}

/*------------------------------------------------------------------------------
| IToolBar::setLocation                                                        |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::setLocation( IToolBar::Location location )
{
  IMODTRACE_DEVELOP("IToolBar::setLocation");

  if (( location != fToolBarData->currentLocation ) ||
      ( fToolBarData->isInitialized == false ))
  {
    //If move is not valid, throw exception.  Movement restrictions only
    //apply to Motif.  isMoveValid() returns true for OS/2 and Windows
    if (!isMoveValid(location))
    {
      ITHROWLIBRARYERROR(IC_TOOLBAR_SETLOCATION_INVALID,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
    }

#ifdef IC_PMWIN
    // In Motif, this code is skipped, since Motif does not allows a
    // setLocation() that would change the parent.  SetParent() is not
    // supported on Motif.
    if ( fToolBarData->isInitialized )
    {
       // Tool bar window already exists.
       // Before we change the toolbar parent and owner, set the
       // focus to the owning frame.  We need to do this so that we do
       // not cause focus loop problems later on.
       if ( IISCHILD ( IQUERYFOCUS ( IWindow::desktopWindow()->handle() ),
                       handle() ))
       {
         fToolBarData->owningFrame->setFocus();
       }

       // Remove the toolbar from the toolbar container, (toolbar
       // container will only exist in non-floating case).
       if (fToolBarData->toolBarContainer)
          {
            //This special flag is so that when ISetCanvas::remove() calls
            //hide(), the IToolBar::show(false) function does not cause
            //another call to remove(), nor does it call Inherited::hide()
            //to change the visible/hidden state of the toolbar.
            fToolBarData->skipShowProcessing = true;
            fToolBarData->toolBarContainer->remove(this);
            fToolBarData->skipShowProcessing = false;
          }

       // Change the parent and owner to frame.  This is needed for
       // toolbar container parent/child test below.
       setParent(objectWindow());

       // In Windows setOwner does nothing for top level windows.  That is
       // not a problem here since the toolbar is a child window.
       setOwner(fToolBarData->owningFrame);
       setId(fToolBarData->windowId);
    }
#endif //IC_PMWIN

#ifdef IC_PMWIN
    /***************************************************************/
    /* In Motif, it is not valid to change a floating toolbar to   */
    /* a non-floating toolbar.  Therefore deleting the floating    */
    /* frame only applies to OS/2 and Windows                      */
    /***************************************************************/
    // If there is a floating frame, delete it
    if ( fToolBarData->floatingFrame )
    {
 #ifdef IC_WIN
      fToolBarData->owningFrame->setFocus();
 #endif
      // Save the old floating position first
      fToolBarData->floatingPosition = floatingPosition();
      delete fToolBarData->floatingHandler;
      delete fToolBarData->floatingFrame;
      fToolBarData->floatingFrame = 0;
      fToolBarData->floatingHandler = 0;
    }
#endif //IC_PMWIN

#ifdef IC_PMWIN
    /***************************************************************/
    /* In Motif, a change of toolbar location will not delete and  */
    /* re-create the toolbar container, it will move the existing  */
    /* container, and all toolbars in it to the new location.      */
    /* Therefore deleting the toolbar container only applies to    */
    /* OS/2 and Windows.                                           */
    /***************************************************************/
    // If there is a container, check if we need to delete it
    if ( fToolBarData->toolBarContainer )
    {
      // Delete the container if no child windows.  It is possible
      // since we have an explicitAddsNeeded style container the container's
      // children may be removed (hidden toolbar), but we do not want
      // the container to be destroyed.  So, check for children
      // using the parent/child relationship
      bool child = false;
      {
        IWindow::ChildCursor cursor(*(fToolBarData->toolBarContainer));
        cursor.setToFirst();
        if ( cursor.isValid() )
          child = true;
      }
      if (child == false)
      {
 #ifdef IC_PM
        // As a visual optimization, before removing the extension,
        // see if we will create another one and if so, disable
        // update to prevent flashing.  The flashing would occur because
        // of a frame update here to remove the extension, and a frame
        // updated later to add the extension.
        bool tbarCnr = true;   // Set true for floating/hidden cases
        IFrameWindow::Location frameLocation;
        switch ( location )
          {
            case aboveClient:
              frameLocation = IFrameWindow::aboveClient;
              tbarCnr = (bool)IToolBarContainer::frameToolBarContainer(
                          fToolBarData->owningFrame, frameLocation);
              break;
            case belowClient:
              frameLocation = IFrameWindow::belowClient;
              tbarCnr = (bool)IToolBarContainer::frameToolBarContainer(
                          fToolBarData->owningFrame, frameLocation);
              break;
            case leftOfClient:
              frameLocation = IFrameWindow::leftOfClient;
              tbarCnr = (bool)IToolBarContainer::frameToolBarContainer(
                          fToolBarData->owningFrame, frameLocation);
              break;
            case rightOfClient:
              frameLocation = IFrameWindow::rightOfClient;
              tbarCnr = (bool)IToolBarContainer::frameToolBarContainer(
                          fToolBarData->owningFrame, frameLocation);
              break;
            }
        if (!tbarCnr)
        {
          //Disable update if toolbar is being moved to a frame extension
          //location that does not already contain a toolbar because a
          //frame update will occur when the extension is added.
          fToolBarData->owningFrame->disableUpdate();
          fToolBarData->owningFrame->removeExtension
                       (fToolBarData->toolBarContainer);
          fToolBarData->owningFrame->enableUpdate();
        }
        else
 #endif //IC_PM
          fToolBarData->owningFrame->removeExtension
                       (fToolBarData->toolBarContainer);

        delete fToolBarData->toolBarContainer;
      }
      else
      {
        // Call layout on the container (since we did not delete it)
        fToolBarData->toolBarContainer->setLayoutDistorted
                                       (IWindow::layoutChanged, 0);
      }
      fToolBarData->toolBarContainer = 0;
    }
#endif //IC_PMWIN

    /*******************************************************************/
    /* Completed:  Removing toolbar from previous location             */
    /* Now begin:  Adding toolbar to new location                      */
    /*******************************************************************/

    // Show toolbar and reset layout for new location
    fToolBarData->currentLocation = location;

#ifdef IC_PMWIN
    setLayoutDistorted(IWindow::layoutChanged, 0);
#endif

    // Be sure that the toolbar is not collapsed
    if ( !fToolBarData->isExpanded )
      fToolBarData->isExpanded = true;

    // Handle floating toolbar
    if ( location == floating )
    {
      // Get a new floating frame
      fToolBarData->floatingFrame = createFloatingFrame
                                   ( fToolBarData->owningFrame );

      // Set the state of the floating toolbar
      // expand/collapse button.
      fToolBarData->floatingFrame->sendEvent( IC_UM_TOOLBAR,
                                             IC_UM_TBAR_SETEXPAND_BTN,
                                             isExpanded() );

      // Add handlers to frame
      fToolBarData->floatingHandler = new IToolBarFloatingHandler;
      fToolBarData->floatingHandler->handleEventsFor(fToolBarData->floatingFrame);
      fToolBarData->trackHandler->handleEventsFor(fToolBarData->floatingFrame);

      if ( ! fToolBarData->isInitialized )
      {
         // Still need to create the tool bar window.
         ISetCanvas::Style initialStyle( ISetCanvas::classDefaultStyle |
                                         ISetCanvas::explicitAddsNeeded );
         this->Inherited::initialize( fToolBarData->windowId,
                                      fToolBarData->floatingFrame,
                                      fToolBarData->floatingFrame,
                                      IRectangle(),
                                      initialStyle.asUnsignedLong(),
                                      initialStyle.asExtendedUnsignedLong() );
      }
#ifdef IC_PMWIN
      else
      {
         // This case will not be met in Motif.  The operation of changing
         // a non-floating toolbar to floating is not supported.  An earlier
         // check disallowed this.

         // Make the toolbar the client area of the frame
         setParent(fToolBarData->floatingFrame);

         // In Windows setOwner does nothing for top level windows.  That is
         // not a problem here since the toolbar is a child window.
         setOwner(fToolBarData->floatingFrame);
      }
#endif //IC_PMWIN

      fToolBarData->skipShowProcessing = true;
      fToolBarData->floatingFrame->setClient(this);
      fToolBarData->skipShowProcessing = false;

      // Set the frame title text
      fToolBarData->floatingFrame->setTitleText( fToolBarData->toolBarName );

      // Set the frame position and size
      IPoint newPos = mapPoint(fToolBarData->floatingPosition,
                               fToolBarData->owningFrame->handle(),
                               desktopWindow()->handle());
      newPos.setX(newPos.x()+1);
      newPos.setY(newPos.y()+1);

      // Check if collapsed
      ISize minSize = calcMinimumSize();
      if (!isExpanded())
      {
        minSize = Inherited::calcMinimumSize();
        minSize.setHeight(0);
      }

      //IC_NOTYET, should probably remove this call.
      fToolBarData->floatingFrame->moveSizeToClient(IRectangle(newPos,minSize));

      //Only show the frame if the toolbar is in visible state.  If the
      //toolbar is in hidden state, and its location is set to floating, do
      //not show the frame.  The frame will be shown if IToolBar::show()
      //is called.
      if ( fToolBarData->isShown == true )
      {
        // Show the new frame
        fToolBarData->floatingFrame->show();
#ifdef IC_WIN
        if (isExpanded())
        {
          // We need to show maximized
          fToolBarData->floatingFrame->maximize();
        }
#endif

        // Put the floating frame on top of sibling windows if the owning frame
        // has the focus (otherwise let the z-order be the default)
        if ( fToolBarData->owningFrame->hasFocus() )
        {
          fToolBarData->floatingFrame->positionOnSiblings();
        }
      } /* end isShown */
    }

    // Handle frame extension toolbar
    if (( location != floating ) && ( location != hidden ))
    {
      bool bPositionBehindSiblings = true;
      IFrameWindow::Location frameLocation;
      switch ( location )
      {
        case aboveClient:
          frameLocation = IFrameWindow::aboveClient;

          //setMargin calls setLayoutDistorted
          //Reverse the width and height when changing to horizontal
          setMargin(ISize(fToolBarData->margin.height(),
                          fToolBarData->margin.width()));
          //Setting of deck orientation must occur after setting margin
          //due to toolbar's setLayoutDistorted processing
          setDeckOrientation(ISetCanvas::horizontal);
          break;

        case belowClient:
          frameLocation = IFrameWindow::belowClient;

          //setMargin calls setLayoutDistorted
          //Reverse the width and height when changing to horizontal
          setMargin(ISize(fToolBarData->margin.height(),
                          fToolBarData->margin.width()));
          //Setting of deck orientation must occur after setting margin
          //due to toolbar's setLayoutDistorted processing
          setDeckOrientation(ISetCanvas::horizontal);
          break;

        case leftOfClient:
          frameLocation = IFrameWindow::leftOfClient;

          //setMargin calls setLayoutDistorted
          //Honor the margin width and height when changing to vertical
          setMargin(fToolBarData->margin);

          //Setting of deck orientation must occur after setting margin
          //due to toolbar's setLayoutDistorted processing
          setDeckOrientation(ISetCanvas::vertical);
          break;

        case rightOfClient:
          frameLocation = IFrameWindow::rightOfClient;

          //setMargin calls setLayoutDistorted
          //Honor the margin width and height when changing to vertical
          setMargin(fToolBarData->margin);

          //Setting of deck orientation must occur after setting margin
          //due to toolbar's setLayoutDistorted processing
          setDeckOrientation(ISetCanvas::vertical);
          break;
      }

#ifdef IC_PMWIN
      /*********************************************************************/
      /* In OS/2 and Windows, a new toolbar container used for the toolbar */
      /* Check to see if one already exists at new location, otherwise,    */
      /* create a new one.                                                 */
      /*********************************************************************/
      fToolBarData->toolBarContainer = IToolBarContainer::frameToolBarContainer
                                      (fToolBarData->owningFrame, frameLocation);

      if (!fToolBarData->toolBarContainer)
      {
        fToolBarData->toolBarContainer = createAndVerifyToolBarContainer
                                         (fToolBarData->owningFrame);
      }
#endif
#ifdef IC_MOTIF
      if ( !fToolBarData->isInitialized )
      {
        /*********************************************************************/
        /* In Motif, a new toolbar container is created only if the toolbar  */
        /* has not been initialized.  Otherwise, the same container re-used. */
        /*********************************************************************/
        fToolBarData->toolBarContainer = IToolBarContainer::frameToolBarContainer
                                    (fToolBarData->owningFrame, frameLocation);

        if (!fToolBarData->toolBarContainer)
        {
          fToolBarData->toolBarContainer = createAndVerifyToolBarContainer
                                           (fToolBarData->owningFrame);
        }
      }
#endif //IC_MOTIF

      if ( !fToolBarData->isInitialized )
      {
         // Still need to create the tool bar window.
         ISetCanvas::Style initialStyle( ISetCanvas::classDefaultStyle |
                                         ISetCanvas::explicitAddsNeeded );
         this->Inherited::initialize( fToolBarData->windowId,
                                      fToolBarData->toolBarContainer,
                                      fToolBarData->toolBarContainer,
                                      IRectangle(),
                                      initialStyle.asUnsignedLong(),
                                      initialStyle.asExtendedUnsignedLong() );

         // Add the toolbar to the toolbar container
         // This special flag is so that when ISetCanvas::add() calls
         // show(), the IToolBar::show() function does not cause another
         // call to add().  Call Inherited::show() now to set the state
         // of the toolbar to visible.
         fToolBarData->skipShowProcessing = true;
         fToolBarData->toolBarContainer->add(this);
         fToolBarData->skipShowProcessing = false;
         Inherited::show();
      }
      else
      {
#ifdef IC_PMWIN
         /*****************************************************************/
         /* Add the toolbar to the container.  In this case, you already  */
         /* have an existing toolbar which you are adding to a new        */
         /* toolbar container.                                            */
         /*****************************************************************/
         setParent(fToolBarData->toolBarContainer);
#endif //IC_PMWIN

         // If the state of the toolbar is hidden, we do not want to add it
         // to the container.  The toolbar will be added to the container
         // when it is shown by a call to IToolBar::show().
         if ( fToolBarData->isShown == true )
         {
            // Add the toolbar to the toolbar container
            // This special flag is so that when ISetCanvas::add() calls
            // show(), the IToolBar::show() function does not cause
            // another call to add(), nor does it call Inherited::show() to
            // change the visible/hidden state of the toolbar.
            fToolBarData->skipShowProcessing = true;
            fToolBarData->toolBarContainer->add(this);
            fToolBarData->skipShowProcessing = false;
         }

#ifdef IC_MOTIF
         /*****************************************************************/
         /* SetParent not supported on Motif.                             */
         /* In motif, the same toolbar container is re-used.  It's        */
         /* location as a frame extension is changed.                     */
         /*****************************************************************/
         fToolBarData->owningFrame->disableUpdate();
         fToolBarData->owningFrame->removeExtension
                                    (fToolBarData->toolBarContainer);
         fToolBarData->owningFrame->enableUpdate();

         /*****************************************************************/
         /* Set alignment on container before adding it as an extension   */
         /*****************************************************************/
         if ((location == leftOfClient) || (location == rightOfClient))
         {
           fToolBarData->toolBarContainer->setDeckOrientation(
                                                   ISetCanvas::vertical);
           fToolBarData->toolBarContainer->setAlignment(
                                                   ISetCanvas::topCenter);
         }
         else
         {
           fToolBarData->toolBarContainer->setDeckOrientation(
                                                   ISetCanvas::horizontal);
           fToolBarData->toolBarContainer->setAlignment(
                                                   ISetCanvas::centerLeft);
         }

         /*****************************************************************/
         /* Set alignment of all toolbars in the container.  Even though  */
         /* the toolbar that setLocation() was called on, already has the */
         /* proper alignment set above, there is no harm in re-setting it */
         /*****************************************************************/
         IWindow::ChildCursor cursor(*(fToolBarData->toolBarContainer));
         for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
         {
           IToolBar* toolbar = (IToolBar*)IWindow::windowWithHandle(
                                                   childAt(cursor));
           if ( toolbar )
           {
             //Set the location of each toolbar, so any subsequent
             //setLocation() calls by the user have no effect, since
             //the location is being changed here.
             toolbar->fToolBarData->currentLocation = location;

             if ((location == leftOfClient) || (location == rightOfClient))
               {
                 toolbar->setMargin(toolbar->fToolBarData->margin);
                 //Setting of deck orientation must occur after setting margin
                 //due to toolbar's setLayoutDistorted processing
                 toolbar->setDeckOrientation(ISetCanvas::vertical);
               }
             else
               {
                 toolbar->setMargin(ISize(
                                    toolbar->fToolBarData->margin.height(),
                                    toolbar->fToolBarData->margin.width()));
                 //Setting of deck orientation must occur after setting margin
                 //due to toolbar's setLayoutDistorted processing
                 toolbar->setDeckOrientation(ISetCanvas::horizontal);
               }

           }
         }
         // Because we've moved all toolbars in the container to the 
         // new extension area, we do not want to reposition this toolbar.
         bPositionBehindSiblings = false;

         fToolBarData->owningFrame->addExtension
                                    (fToolBarData->toolBarContainer,
                                     frameLocation,
                                     IFrameWindow::none);
#endif //IC_MOTIF
      }
      fToolBarData->toolBarContainer->setLayoutDistorted
                   (IWindow::layoutChanged | IWindow::immediateUpdate, 0);
    if(bPositionBehindSiblings)
      positionBehindSiblings();
    }

    /*********************************************************************/
    /* In Motif, movement from non-hidden to hidden toolbar is not       */
    /* allowed.  This movement was prohibited by a check above.          */
    /* Therefore, if Motif, at this point a toolbar is being initialized */
    /* for the first time.                                               */
    /* In OS/2 and Windows, movement from any state to any other state   */
    /* is allowed.                                                       */
    /*********************************************************************/
    /* Note that a location of 'hidden' is marked as obsolete.  It is    */
    /* still supported in this release for compatibility with existing   */
    /* applications.  'Hidden' is being replaced by IToolBar             */
    /* hide() and show().                                                */
    /*********************************************************************/
    if ( location == hidden )
    {
      //Need to create toolbar if not already created.
      if ( ! fToolBarData->isInitialized )
      {
         // Still need to create the tool bar window.
         // A hidden toolbar's parent and owner is the owning frame specified
         // as a parameter to IToolBar::initialize().
         ISetCanvas::Style initialStyle( ISetCanvas::classDefaultStyle |
                                         ISetCanvas::explicitAddsNeeded );
         this->Inherited::initialize( fToolBarData->windowId,
                                      fToolBarData->owningFrame,
                                      fToolBarData->owningFrame,
                                      IRectangle(),
                                      initialStyle.asUnsignedLong(),
                                      initialStyle.asExtendedUnsignedLong() );
      }
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::location                                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::Location IToolBar::location ( ) const
{
  return fToolBarData->currentLocation;
}

/*------------------------------------------------------------------------------
| IToolBar::isMoveValid                                                        |
|                                                                              |
| Used to determine if the movement of a toolbar (through setLocation) is      |
| valid.  If you specifiy a location that is the same as the existing          |
| location of the toolbar, this function returns true.  Calling setLocation()  |
| in this case does no harm - the toolbar does not move.                       |
------------------------------------------------------------------------------*/
bool IToolBar::isMoveValid (Location newLocation) const
{
#ifdef IC_MOTIF
  if (fToolBarData->isInitialized)
    {
      /********************************************************************/
      /* return true if new location is the same as the existing toolbar  */
      /* location.                                                        */
      /********************************************************************/
      if ( newLocation == fToolBarData->currentLocation )
         return true;

      /********************************************************************/
      /* If toolbar is currently floating, it must always remain floating */
      /* If toolbar is currently hidden, it must always remain hidden     */
      /********************************************************************/
      if ( (fToolBarData->currentLocation == floating) ||
           (fToolBarData->currentLocation == hidden) )
        {
          return false;
        }
      else
        {
          /****************************************************************/
          /* toolbar is a frame extension.  It can only be moved to a     */
          /* position is which no other toolbar already exists.  Test on  */
          /* the existence of a toolbar container in the new location.    */
          /****************************************************************/
          IFrameWindow::Location frameLocation;
          switch ( newLocation )
            {
              case aboveClient:
                frameLocation = IFrameWindow::aboveClient;
                break;
              case belowClient:
                frameLocation = IFrameWindow::belowClient;
                break;
              case leftOfClient:
                frameLocation = IFrameWindow::leftOfClient;
                break;
              case rightOfClient:
                frameLocation = IFrameWindow::rightOfClient;
                break;
              case floating:
                return false;  //Cannot move from docked to any other location
                break;
              case hidden:
                return false;  //Cannot move from docked to any other location
                break;
            }
          if (IToolBarContainer::frameToolBarContainer
                                 (fToolBarData->owningFrame, frameLocation))
            {
              return false;   //A toolbar container already exists.
            }
          else
            {
              return true;
            }
        }
    }
  else
    {
      /********************************************************************/
      /* If toolbar has not been initialized, return true to allow        */
      /* toolbar to be created in any state                               */
      /********************************************************************/
      return true;
    }
#endif //IC_MOTIF

#ifdef IC_PMWIN
  return true;
#endif
}

/*------------------------------------------------------------------------------
| IToolBar::IToolBar                                                           |
|                                                                              |
| Constructor to create a tool bar using a tool bar area.  The actual window   |
| is created in setLocation(), by a call to the ISetCanvas's initialize()      |
| function.  It is done at that point so the toolbar container can be created  |
| and the container specified as the parent.                                   |
|                                                                              |
| If the tool bar is a floating tool bar, its parent is the frame window       |
| containing it.  If the tool bar is a frame extension, its parent is the      |
| tool bar container in the frame extension area.                              |
------------------------------------------------------------------------------*/
IToolBar::IToolBar ( unsigned long id,
                     IFrameWindow* owner,
                     Location      frameLocation,
                     bool          groupWithPreceding,
                     const Style&  style)
         :ISetCanvas(),
         fToolBarData(new IToolBarData(owner))
{
  IMODTRACE_DEVELOP("TBar::TBar(location)");

  this->initialize( id,
                    owner,
                    frameLocation,
                    groupWithPreceding,
                    style.asUnsignedLong(),
                    style.asExtendedUnsignedLong() );

  if (toolBarStatics.colorButtonDark == 0)
     toolBarStatics.colorButtonDark = new IColor(IColor::buttonDark);
  if (toolBarStatics.colorButtonLight == 0)
     toolBarStatics.colorButtonLight = new IColor(IColor::buttonLight);
}

/*------------------------------------------------------------------------------
| IToolBar::IToolBar                                                           |
|                                                                              |
| Constructor to create a tool bar using a tool bar area.  The actual window   |
| is created in setLocation(), by a call to the ISetCanvas's initialize()      |
| function.  It is done at that point so the toolbar container can be created  |
| and the container specified as the parent.                                   |
|                                                                              |
| If the tool bar is a floating tool bar, its parent is the frame window       |
| containing it.  If the tool bar is a frame extension, its parent is the      |
| tool bar container in the frame extension area.                              |
------------------------------------------------------------------------------*/
IToolBar::IToolBar ( unsigned long id,
                     IToolBar*     precedingToolBar,
                     bool          groupWithPreceding,
                     const Style&  style)
         :ISetCanvas(),
         fToolBarData(new IToolBarData
                     (precedingToolBar->fToolBarData->owningFrame))
{
  IMODTRACE_DEVELOP("TBar::TBar(Tbar)");

  // Save default values
  fToolBarData->precedingToolBar = precedingToolBar;

  this->initialize( id,
                    precedingToolBar->fToolBarData->owningFrame,
                    precedingToolBar->fToolBarData->currentLocation,
                    groupWithPreceding,
                    style.asUnsignedLong(),
                    style.asExtendedUnsignedLong() );

  if (toolBarStatics.colorButtonDark == 0)
     toolBarStatics.colorButtonDark = new IColor(IColor::buttonDark);
  if (toolBarStatics.colorButtonLight == 0)
     toolBarStatics.colorButtonLight = new IColor(IColor::buttonLight);
}

/*------------------------------------------------------------------------------
| IToolBar::IToolBar                                                           |
|                                                                              |
| Protected constructor for derived classes.                                   |
------------------------------------------------------------------------------*/
IToolBar::IToolBar ( )
  : ISetCanvas(),
    fToolBarData(new IToolBarData)
{
  if (toolBarStatics.colorButtonDark == 0)
     toolBarStatics.colorButtonDark = new IColor(IColor::buttonDark);
  if (toolBarStatics.colorButtonLight == 0)
     toolBarStatics.colorButtonLight = new IColor(IColor::buttonLight);
}

/*------------------------------------------------------------------------------
| IToolBar::~IToolBar                                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::~IToolBar ( )
{
  IMODTRACE_DEVELOP("TBar::~TBar");

  // Remove ourselves from the frames list of tool bars and delete
  // the list if we are the last one in it.
  bool sharedLocation = false;
  bool removed = false;

  IToolBarList* tbarList = fToolBarData->owningFrame->toolBarList();
  if (tbarList != 0)
  {
    IToolBarList::Cursor cursor(*tbarList);
    cursor.setToFirst();
    while (cursor.isValid() && !removed)
    {
      IToolBar* currentToolBar = tbarList->elementAt(cursor);
      if (this == currentToolBar)
      {
         tbarList->removeAt(cursor);
         removed = true;
      }
      else
        cursor.setToNext();
    }

    // Delete the list if it is now empty
    if(tbarList->numberOfElements() == 0)
    {
      delete tbarList;
      fToolBarData->owningFrame->setToolBarList(0);
      tbarList = 0;
    }
  }

  // Check if any other toolbars have the same location (if so then
  // they must also be sharing the same toolbar container)
  if (tbarList != 0)
  {
    IToolBarList::Cursor cursor(*tbarList);
    for(cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
    {
      IToolBar* currentToolBar = tbarList->elementAt(cursor);
      if ( currentToolBar->location() == location() )
        sharedLocation = true;
    }
  }

  // Delete tracking handler (if needed)
  if ( fToolBarData->trackHandler )
  {
    delete fToolBarData->trackHandler;
  }

  // Delete the floating frame (if needed)
  if ( fToolBarData->floatingFrame )
  {
    delete fToolBarData->floatingHandler;
    delete fToolBarData->floatingFrame;
  }

  // Delete the toolbar container (if needed)
  if ( fToolBarData->toolBarContainer )
  {
    // Delete the container if no other toolbars share the same location
    // Note: The toolbar container frame extension should only be removed
    //       if the frame is a valid PM window (this is needed to prevent
    //       the removeExtension from throwing an exception)
    if ( sharedLocation == false )
    {
      if ( fToolBarData->owningFrame->isValid() )
        fToolBarData->owningFrame->removeExtension(fToolBarData->toolBarContainer);
      delete fToolBarData->toolBarContainer;
    }
    else
    {
      // Call layout on the container (since we did not delete it)
      if ( fToolBarData->toolBarContainer->isValid() )
        fToolBarData->toolBarContainer->setLayoutDistorted
                                       (IWindow::layoutChanged, 0);
    }
  }

  // Delete any IToolBarObjects still in the list
  {
    IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));
    for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
    {
      IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
      delete winObj;
    }
    delete fToolBarData->objectList;
  }

  // Delete our local data
  delete fToolBarData;
}

/*------------------------------------------------------------------------------
| IToolBar::initialize                                                         |
|                                                                              |
| Setup the initial state of the tool bar.  This includes:                     |
|                                                                              |
| 1) Save styles and verify style information                                  |
| 2) Call setlocation, which calls ISetCanvas::initialize() to create the      |
|    actual toolbar                                                            |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::initialize ( unsigned long     windowIdentifier,
                                 IFrameWindow*     ownerFrame,
                                 Location          frameLocation,
                                 bool              groupWithPreceding,
                                 unsigned long     style,
                                 unsigned long     extendedStyle )
{
  IMODTRACE_DEVELOP("TBar::initialize");

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( this->extendedStyle() | extendedStyle );

  // Save the style
  unsigned long ulStyle = this->extendedStyle() & ICNV_EXTTBAR ;
  unsigned long ulOldStyle = ulStyle;

  // Convert redundant styles
  if (((ulStyle & buttonTextVisible.asExtendedUnsignedLong()) &&
      (ulStyle & buttonBitmapVisible.asExtendedUnsignedLong())) ||
      (ulStyle & buttonBitmapAndTextVisible.asExtendedUnsignedLong()))
  {
    ulStyle &= ~(buttonTextVisible.asExtendedUnsignedLong() |
                 buttonBitmapVisible.asExtendedUnsignedLong());
    ulStyle |= buttonBitmapAndTextVisible.asExtendedUnsignedLong();
  }

  // Reject mutually exclusive styles
  if (((ulStyle & buttonTextVisible.asExtendedUnsignedLong()) &&
       (ulStyle & buttonBitmapAndTextVisible.asExtendedUnsignedLong())) ||
      ((ulStyle & buttonBitmapVisible.asExtendedUnsignedLong())  &&
       (ulStyle & buttonBitmapAndTextVisible.asExtendedUnsignedLong())))
  {
     ITHROWLIBRARYERROR(IC_INVALIDSTYLE,
                        IBaseErrorInfo::invalidParameter,
                        IException::recoverable);
  }

  // Update if redundant style was eliminated
  if (ulStyle != ulOldStyle)
    setExtendedStyle( ulStyle );

  // No need to do multiple layout requests.  (setLayoutDistorted is called
  // by setPad() and setGroupPad() and setMargin() ).  Later, initiate
  // layout once, after all settings are complete.
  fToolBarData->skipLayout = true;

  // Create a track handler for this toolbar
  fToolBarData->trackHandler = new IToolBarTrackHandler ( this, fToolBarData );

  // Save default values
  setPad(ISize(defaultPad(), defaultPad()));
  setGroupPad(defaultGroupPad());

  //A toolbar's margin variable is set to defaultMargin here in initialize.
  //SetLocation() queries the margin variable and reverses margin width and
  //height as appropriate.  When a user calls setMargin(), we update the
  //toolbar's margin variable in setLayoutDistored(), so that it can be used
  //in setLocation().  This means that a user's margin setting will be honored,
  //after the first call to setLocation() which occurs in initialization.
  fToolBarData->margin = defaultMargin();

  fToolBarData->misfitWidth = defaultMisfitWidth();
  fToolBarData->windowId = windowIdentifier;
  fToolBarData->currentLocation = frameLocation;
  //Need to set owningFrame, because it's not set if protected ctor is called.
  fToolBarData->owningFrame = ownerFrame;

  // Set the location of the tool bar.  This causes initialization of the
  // ISetCanvas, which creates the actual system window.  This must be done
  // in setLocation() because that is where the toolbar container is created.
  // The container must be created before the initialization of the canvas.
  setLocation(frameLocation);

#ifdef IC_PMWIN
  // Copy correct bidi settings from the owning frame window to the tool bar.
  if (IBidiSettings::isBidiSupported() )
  {
    IBidiSettings ownerSettings(*ownerFrame);
    ownerSettings.apply(*this);
  }
#endif

  if(!groupWithPreceding)
    enableGroup();
  else
    enableGroup(false);

  // set the default color
  // do not cache this value because this is the only place its used.
  // On Motif, do not set the color, the default color is what we want.
#ifdef IC_PMWIN
  setBackgroundColor(IColor(IColor::menuBgnd));
#endif

  // Setup Orientation based on the parent container
  setAlignment(ISetCanvas::centerCenter);

  // Add our paint handler to paint a different border
  fToolBarData->paintHandler.handleEventsFor(this);

#ifdef IC_PMWIN
  // Add the paint handler to the frame as well so that we can pick up
  // any BIDI changes
  if ( IBidiSettings::isBidiSupported() )
       fToolBarData->paintHandler.handleEventsFor(
                                  (ICanvas*)fToolBarData->owningFrame);
#endif

  // Add our drag drop handler if style set and not already added
  if (allowsDragDrop() && (!fToolBarData->isDragDropInitialized))
  {
    IDMHandler::enableDropOn(this);
    fToolBarData->isDragDropInitialized = true;
  }
  // Store the tool bar in the frames list
  IToolBarList* tbarList = fToolBarData->owningFrame->toolBarList();

  // Allocate the tool bar list if necessary and store it back
  // into the frame
  if (tbarList == 0)
  {
    tbarList = new IToolBarList;
    fToolBarData->owningFrame->setToolBarList(tbarList);
  }

  // Add ourselves to the list
  tbarList->addAsLast(this);

  fToolBarData->isInitialized = true;

  //No need to do multiple layout requests.  Now, initiate layout
  //once.  Note that colorChange not needed in this call to setLayoutDistorted.
  fToolBarData->skipLayout = false;
  setLayoutDistorted(IWindow::layoutChanged | IWindow::immediateUpdate, 0);

  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::toolBarContainer                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBarContainer* IToolBar::toolBarContainer( ) const
{
  IMODTRACE_DEVELOP("TBar::toolBarContainer");
  return fToolBarData->toolBarContainer;
}

/*------------------------------------------------------------------------------
| IToolBar::floatingFrame                                                      |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBarFrameWindow* IToolBar::floatingFrame( )
{
  IMODTRACE_DEVELOP("TBar::floatingFrame");
  return fToolBarData->floatingFrame;
}

/*------------------------------------------------------------------------------
| IToolBar::createAndVerifyToolBarContainer                                    |
|                                                                              |
| Create a container by calling createToolBarContainer.  Verify that the       |
| container was created witht the explicitAddsNeeded style.  Throw an          |
| exception if this condition is not met.                                      |
| This verification is needed for the case where the user overrides            |
| creatToolBarContainer() and creates the container oneself.                   |
------------------------------------------------------------------------------*/
IToolBarContainer* IToolBar::createAndVerifyToolBarContainer
                             (IFrameWindow* frame)
{
  IToolBarContainer * cnr = createToolBarContainer(frame);
  if (!cnr->areExplicitAddsNeeded() )
    {
      ITHROWLIBRARYERROR(IC_TOOLBARCNR_EXPLICITADDS,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
    }
  return cnr;
}

/*------------------------------------------------------------------------------
| IToolBar::createToolBarContainer                                             |
|                                                                              |
| Create a container to be used to house tool bars in one of the frame         |
| extensions around the client.  The current location of the toolbar is used   |
| when creating the container.                                                 |
------------------------------------------------------------------------------*/
IToolBarContainer* IToolBar::createToolBarContainer ( IFrameWindow* frame )
{
  IMODTRACE_DEVELOP("TBar::createToolBarContainer");

  IToolBarContainer* cnr = 0;

  switch ( fToolBarData->currentLocation )
  {
    case aboveClient:
      cnr = new IToolBarContainer((unsigned long) fToolBarData->currentLocation,
                                  frame,
                                  IFrameWindow::aboveClient);
      break;
    case belowClient:
      cnr = new IToolBarContainer((unsigned long) fToolBarData->currentLocation,
                                  frame,
                                  IFrameWindow::belowClient);
      break;
    case leftOfClient:
      cnr = new IToolBarContainer((unsigned long) fToolBarData->currentLocation,
                                  frame,
                                  IFrameWindow::leftOfClient);
      break;
    case rightOfClient:
      cnr = new IToolBarContainer((unsigned long) fToolBarData->currentLocation,
                                  frame,
                                  IFrameWindow::rightOfClient);
      break;
  }

#ifdef IC_PMWIN
  if ( cnr && IBidiSettings::isBidiSupported() )
  {
     IBidiSettings bidi(*(fToolBarData->owningFrame));
     bidi.apply(*cnr);
  }
#endif

  return cnr;
}

/*------------------------------------------------------------------------------
| IToolBar::createFloatingFrame                                                |
|                                                                              |
| Create the frame window for a floating tool bar.  This includes establishing |
| its initial style and removing any unnecessary items from the system menu.   |
------------------------------------------------------------------------------*/
IToolBarFrameWindow* IToolBar::createFloatingFrame ( IFrameWindow* owner )
{
  IMODTRACE_DEVELOP("TBar::createFloatingFrame");

  // Floating tool bars need a frame window.
  IToolBarFrameWindow* floater = new IToolBarFrameWindow( IC_DEFAULT_FRAME_ID,
                                                          owner );
#ifdef IC_PMWIN
  if ( floater && IBidiSettings::isBidiSupported() )
  {
     IBidiSettings bidi(*(fToolBarData->owningFrame));
     bidi.apply(*floater);
  }
#endif

  return floater;
}

/*------------------------------------------------------------------------------
| IToolBar::addControl                                                         |
|                                                                              |
| Private function to handle common code when adding a new control to the      |
| tool bar.                                                                    |
------------------------------------------------------------------------------*/
void IToolBar::addControl ( IWindow* control )
{
  /****************************************************************/
  /* Use RTTI to determine the type. If IToolbarButton,           */
  /* ensure the correct button state.  The toolbar's view has     */
  /* precedent over the button's view.                            */
  /* Also, setup drag/drop for the button                         */
  /****************************************************************/
  IToolBarButton *tBarButton = dynamic_cast<IToolBarButton*>(control);
  if (tBarButton)
    {
      unsigned long ulExtStyle = extendedStyle();

      if (ulExtStyle & buttonTextVisible.asExtendedUnsignedLong())
         tBarButton->setView(IToolBarButton::textView );
      else if (ulExtStyle & buttonBitmapAndTextVisible.asExtendedUnsignedLong())
         tBarButton->setView(IToolBarButton::bitmapAndTextView );
      else
         tBarButton->setView(IToolBarButton::bitmapView );

      //If allowsDragDrop is true, in this case drag/drop will
      //always have been initialized for the toolbar.
      if (allowsDragDrop())
        {
          if (!tBarButton->itemProvider())
            IDMHandler::enableDragDropFor(tBarButton);
        }
    }

  // Add the control to the setcanvas
  this->add( control );

  // Force layout
  setLayoutDistorted(IWindow::layoutChanged | IWindow::minimumSizeChanged |
                     IWindow::immediateUpdate, 0);
}

/*------------------------------------------------------------------------------
| IToolBar::addAsFirst                                                         |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::addAsFirst( IWindow* window,
                                bool  startNewGroup )
{
  IMODTRACE_DEVELOP("TBar::addAsFirst");
  IASSERTPARM( window != 0 );
	return addAsPrevious( window, 0, startNewGroup);
}

/*------------------------------------------------------------------------------
| IToolBar::addAsNext                                                          |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::addAsNext( IWindow* window,
                               IWindow* referenceWindow,
                               bool  startNewGroup )
{
  IMODTRACE_DEVELOP("TBar::addAsNext");
  ITRACE_DEVELOP(IString("startNewGroup =")+IString(startNewGroup));
  IASSERTPARM( window != 0 );

#ifdef IC_MOTIF
  // SetParent not supported on Motif.
  // It is a documented Motif restriction that the window must already have
  // the toolbar as its parent.  Throw an exception if violated.
  if (window->parent() != this)
     ITHROWLIBRARYERROR(IC_TOOLBAR_ADD_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
#endif
#ifdef IC_PMWIN
  window->setParent(this);
#endif

  // Create the object
  IToolBarObject* newObject = new IToolBarObject(window);
  IToolBarObject* lastObject = 0;
  remove(window);

  // Find the last visible window (before the add location) and add object
  IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));
  bool done = false;

  //Search for the reference window in toolbar list
  for (cursor.setToFirst();(cursor.isValid() && !done); cursor.setToNext())
  {
    lastObject = fToolBarData->objectList->elementAt(cursor);
    if ( lastObject->window() == referenceWindow )
    {
      fToolBarData->objectList->addAsNext(newObject,cursor);
      done = true;

		  // set Z order of the windows.
  		window->positionBehindSibling( referenceWindow->handle() );
    }
  }

  if (!done)	//Reference window was not found
  {
  	if( lastObject ) //If controls exist, append to the last one
  	{
    	fToolBarData->objectList->addAsLast(newObject);
		  // set Z order of the window.
	  	window->positionBehindSibling( lastObject->window()->handle() );
    }
    else
    {
    	fToolBarData->objectList->addAsFirst(newObject);
		  // set Z order of the window.
  		window->positionOnSiblings ();
    }
  }

  // If this is the start of a new group, enable group style
  if ( startNewGroup )
    window->enableGroup();
	else
    window->disableGroup();		
	
  // Finish adding the control
  this->addControl( window );
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::addAsPrevious                                                      |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::addAsPrevious( IWindow* window,
                                   IWindow* referenceWindow,
                                   bool  startNewGroup )
{
  IMODTRACE_DEVELOP("TBar::addAsPrevious");
  IASSERTPARM( window != 0 );

#ifdef IC_MOTIF
  // SetParent not supported on Motif.
  // It is a documented Motif restriction that the window must already have
  // the toolbar as its parent.  Throw an exception if violated.
  if (window->parent() != this)
     ITHROWLIBRARYERROR(IC_TOOLBAR_ADD_INVALID,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
#endif
#ifdef IC_PMWIN
  window->setParent(this);
#endif

  // Plan to add the object at the beginning (if we can't find a previous)
  IToolBarObject* newObject = new IToolBarObject(window);
  IToolBarObject* prevObject = 0;
  IWindow* nextControl = 0;

  //Make sure the button is not on the toolbar
  remove(window);

  // Find the last visible window (before the add location)
  //Search for the reference window in toolbar list
  bool done = false;
  IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));

  //Set a default reference control
  cursor.setToFirst();
  if( cursor.isValid() )
  {
    IToolBarObject* nextObj = fToolBarData->objectList->elementAt(cursor);
    nextControl = nextObj->window();    
  }
  
  //Search for the reference control and determine if the new control is to 
  //be the first control.  
  //NOTE: the new window can be the first control if the reference window 
  //was the first control OR the reference window does not exist 
  //(ie. AddAsFirst calls addAsPrevious with a 0 pointer for the ref. window)
  for (cursor.setToFirst();(cursor.isValid() && !done); cursor.setToNext())
  {
    IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
    if ( winObj->window() == referenceWindow )
    {
      done = true;
      nextControl = referenceWindow;
    }
    else
      prevObject = winObj;
  }

  // Add the object (after prevObject or at beginning)
  // We need the previous object so we can set the Z order
  if ( done && prevObject )  //Add the object after prevObject
  {
    cursor.setToFirst();
    while (prevObject != fToolBarData->objectList->elementAt(cursor))
      cursor.setToNext();

    fToolBarData->objectList->addAsNext(newObject,cursor);
	  // set Z order of the windows
 		window->positionBehindSibling ( prevObject->window()->handle() ); 		

    if( done && nextControl)
    {
      if ( nextControl->isGroup() )
      {
        window->enableGroup();
        if( !startNewGroup )
          nextControl->disableGroup();
      }      
      else
      {	
        if( startNewGroup )
          window->enableGroup();
      }
    }  
  }
  else  //add as the first control
  {
    fToolBarData->objectList->addAsFirst(newObject);
  	// set Z order of the windows
		window->positionOnSiblings();
    window->disableGroup();
    if( nextControl )
    {
      if( startNewGroup )
        nextControl->enableGroup();
      else
        nextControl->disableGroup();          
    }
  }

  // Finish adding the control
  this->addControl( window );
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::addAsLast                                                          |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::addAsLast( IWindow* window,
                               bool  startNewGroup )
{
  IMODTRACE_DEVELOP("TBar::addAsLast");
  IASSERTPARM( window != 0 );

	return addAsNext( window, 0, startNewGroup );
}

/*------------------------------------------------------------------------------
| IToolBar::moveToFirst                                                        |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::moveToFirst ( IWindow* window,
                                  bool  startNewGroup )
{
  IASSERTPARM( window != 0 );
  remove(window);
  addAsFirst(window,startNewGroup);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::moveAfter                                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::moveAfter ( IWindow* window,
                                IWindow* referenceWindow,
                                bool  startNewGroup )
{
  IASSERTPARM( window != 0 );
  IASSERTPARM( window != referenceWindow );
  remove(window);
  addAsNext(window,referenceWindow,startNewGroup);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::moveBefore                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::moveBefore ( IWindow* window,
                                 IWindow* referenceWindow,
                                 bool  startNewGroup )
{
  IASSERTPARM( window != 0 );
  IASSERTPARM( window != referenceWindow );
  remove(window);
  addAsPrevious(window,referenceWindow,startNewGroup);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::moveToLast                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::moveToLast ( IWindow* window,
                                 bool  startNewGroup )
{
  IASSERTPARM( window != 0 );
  remove(window);
  addAsLast(window,startNewGroup);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::remove                                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::remove ( IWindow* window )
{
  bool removed = false;

  IASSERTPARM( window != 0 );
  this->Inherited::remove(window);

  IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));

  // Check if we need to move the group style to the next control (if it exists)
  if ( window->isGroup() )
  {
    bool found = false;
    for (cursor.setToFirst();(cursor.isValid() && !found); cursor.setToNext())
    {
      IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
      if ( winObj->window() == window )
         found = true;
    }
    if (cursor.isValid())
    {
      IToolBarObject* nextObj = fToolBarData->objectList->elementAt(cursor);
      nextObj->window()->enableGroup();
    }
  }

  // Remove the object
  cursor.setToFirst();
  while (cursor.isValid() && !removed)
  {
    IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
    if ( winObj->window() == window )
    {
      fToolBarData->objectList->removeAt(cursor);
      removed = true;

      // Check if we need to refresh display
      if ( winObj->isVisible() )
      {
        setLayoutDistorted (IWindow::childWindowDestroyed |
                            IWindow::immediateUpdate, 0);
      }
      delete winObj;
    }
    else
      cursor.setToNext();
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::setButtonView                                                      |
|                                                                              |
| Tells all the tool bar buttons to show the view.                             |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::setButtonView ( IToolBarButton::View buttonView )
{
  unsigned long ulExtStyle = extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;
  unsigned long ulMask = ~(buttonTextVisible.asExtendedUnsignedLong()   |
                           buttonBitmapVisible.asExtendedUnsignedLong() |
                           buttonBitmapAndTextVisible.asExtendedUnsignedLong());

  if (buttonView == IToolBarButton::textView)
  {
     ulExtStyle = (ulExtStyle & ulMask) |
                  buttonTextVisible.asExtendedUnsignedLong();
  }
  else if (buttonView == IToolBarButton::bitmapAndTextView)
  {
     ulExtStyle = (ulExtStyle & ulMask) |
                  buttonBitmapAndTextVisible.asExtendedUnsignedLong();
  }
  else if (buttonView == IToolBarButton::bitmapView)
  {
     ulExtStyle = (ulExtStyle & ulMask) |
                  buttonBitmapVisible.asExtendedUnsignedLong();
  }

  // Update if style changed
  if (ulExtStyle != ulOldExtStyle)
  {
    setExtendedStyle( ulExtStyle );

    //Do not allow the layout on each button to cause the toolbar to refresh
    //before all butttons have been set to new view.
    fToolBarData->skipLayout = true;

    IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));
    for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
    {
      IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
      IToolBarButton *tBarButton =
                  dynamic_cast<IToolBarButton*>(winObj->window());
      //If windows type is toolbar button, set the view.
      if (tBarButton)
        {
          if (buttonView == IToolBarButton::textView)
            tBarButton->setView(IToolBarButton::textView );
          else if (buttonView == IToolBarButton::bitmapAndTextView)
            tBarButton->setView(IToolBarButton::bitmapAndTextView );
          else if (buttonView == IToolBarButton::bitmapView)
            tBarButton->setView(IToolBarButton::bitmapView );
        }
    }

    fToolBarData->skipLayout = false;
    setLayoutDistorted(IWindow::minimumSizeChanged |
                       IWindow::childMinimumSizeChanged |
                       IWindow::immediateUpdate, 0);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::buttonView                                                         |
|                                                                              |
| Tells all the tool bar buttons to show their bitmaps.                        |
------------------------------------------------------------------------------*/
IToolBarButton::View IToolBar::buttonView ( ) const
{
  unsigned long ulExtStyle = extendedStyle();

  if (ulExtStyle & buttonBitmapVisible.asExtendedUnsignedLong())
     return IToolBarButton::bitmapView;
  else if (ulExtStyle & buttonTextVisible.asExtendedUnsignedLong())
     return IToolBarButton::textView;
  return IToolBarButton::bitmapAndTextView;
}

/*------------------------------------------------------------------------------
| IToolBar::enableMisfitFiltering                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::enableMisfitFiltering ( bool enable )
{
  if ( isMisfitFilteringEnabled() != enable )
  {
    // Change the extended style
    unsigned long ulExtStyle = extendedStyle();
    if (enable)
      ulExtStyle |= filterMisfits.asExtendedUnsignedLong();
    else
      ulExtStyle &= ~filterMisfits.asExtendedUnsignedLong();

    setExtendedStyle(ulExtStyle);

    // Check for misfit filtering (in or out) if vertical
    if ( deckOrientation() == vertical )
    {
      filter();
      setLayoutDistorted(IWindow::childMinimumSizeChanged |
                         IWindow::immediateUpdate, 0);
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::disableMisfitFiltering                                             |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::disableMisfitFiltering ( )
{
  enableMisfitFiltering(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::isMisfitFilteringEnabled                                           |
------------------------------------------------------------------------------*/
bool IToolBar::isMisfitFilteringEnabled ( ) const
{
  return( (extendedStyle() & filterMisfits.asExtendedUnsignedLong()) ? true
                                                                     : false );
}

/*------------------------------------------------------------------------------
| IToolBar::defaultStyle                                                       |
|                                                                              |
| Returns the current default style.                                           |
| Notes: Can not inline this since it is in the constructor declaration.       |
------------------------------------------------------------------------------*/
IToolBar::Style  IToolBar::defaultStyle ( )
{
  return fgCurrentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IToolBar::setDefaultStyle                                                    |
|                                                                              |
| Sets the current default style.                                              |
------------------------------------------------------------------------------*/
void IToolBar::setDefaultStyle ( const Style& style )
{
  fgCurrentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IToolBar::convertToGUIStyle                                                  |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IToolBar::convertToGUIStyle(const IBitFlag& guiStyle,
                                          bool bExtOnly) const
{
  // Obtain the style from the class (ISetCanvas) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // All the styles for the toolbar are extended styles, so we do not mask.
    ulStyle |= guiStyle.asUnsignedLong();
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IToolBar::setFloatingTitle                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::setFloatingTitle ( const char* pszText )
{
  fToolBarData->toolBarName = pszText;

  if ( fToolBarData->floatingFrame )
  {
    fToolBarData->floatingFrame->setTitleText(fToolBarData->toolBarName);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::setFloatingTitle                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar &IToolBar::setFloatingTitle ( const IResourceId& residText)
{
  IString aString = residText.resourceLibrary().loadString(residText);
  return setFloatingTitle((char*)aString);
}

/*------------------------------------------------------------------------------
| IToolBar::floatingTitle                                                      |
|                                                                              |
------------------------------------------------------------------------------*/
IString  IToolBar::floatingTitle() const
{
  return fToolBarData->toolBarName;
}

/*------------------------------------------------------------------------------
| IToolBar::setFloatingPosition                                                |
|                                                                              |
| Set the Frame relative tool bar postition.                                   |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::setFloatingPosition ( const IPoint& frameRelativePosition)
{
  fToolBarData->floatingPosition = frameRelativePosition;
  if ( fToolBarData->floatingFrame )
  {
    fToolBarData->floatingFrame->moveTo(mapPoint(frameRelativePosition,
                                       fToolBarData->owningFrame->handle(),
                                       desktopWindow()->handle()));
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::floatingPosition                                                   |
|                                                                              |
| Returns the frame relative tool bar position.                                |
------------------------------------------------------------------------------*/
IPoint IToolBar::floatingPosition ( ) const
{
  // If we are floating get the current floating position
  // Otherwise return our stored value
  if ( fToolBarData->floatingFrame )
  {
    fToolBarData->floatingPosition = mapPoint
                                  ( fToolBarData->floatingFrame->position(),
                                    desktopWindow()->handle(),
                                    fToolBarData->owningFrame->handle());
  }
  return fToolBarData->floatingPosition;
}

/*------------------------------------------------------------------------------
| IToolBar::setDefaultPad                                                      |
|                                                                              |
| Sets the current default pad.                                                |
------------------------------------------------------------------------------*/
void IToolBar::setDefaultPad ( long pixelsOfPad)
{
  IToolBarData::currentDefaultPad = pixelsOfPad;
}

/*------------------------------------------------------------------------------
| IToolBar::setDefaultGroupPad                                                 |
|                                                                              |
| Sets the current default group pad.                                          |
------------------------------------------------------------------------------*/
void IToolBar::setDefaultGroupPad ( unsigned long pixelsOfGroupPad)
{
  IToolBarData::currentDefaultGroupPad = pixelsOfGroupPad;
}

/*------------------------------------------------------------------------------
| IToolBar::setDefaultMisfitWidth                                              |
|                                                                              |
| Sets the default misfit width.                                               |
------------------------------------------------------------------------------*/
void IToolBar::setDefaultMisfitWidth ( unsigned long  maximumWidth )
{
  IToolBarData::currentDefaultMisfitWidth = maximumWidth;
}

/*------------------------------------------------------------------------------
| IToolBar::setDefaultMargin                                                   |
|                                                                              |
| Sets the current default margin.                                             |
------------------------------------------------------------------------------*/
void IToolBar::setDefaultMargin ( const ISize& sizeOfMargin)
{
  IToolBarData::currentDefaultMargin = sizeOfMargin;
}

/*------------------------------------------------------------------------------
| IToolBar::defaultPad                                                         |
|                                                                              |
| Returns the current pad.                                                     |
------------------------------------------------------------------------------*/
long IToolBar::defaultPad ( )
{
  return IToolBarData::currentDefaultPad;
}

/*------------------------------------------------------------------------------
| IToolBar::defaultGroupPad                                                    |
|                                                                              |
| Returns the current group pad.                                               |
------------------------------------------------------------------------------*/
unsigned long IToolBar::defaultGroupPad ( )
{
  return IToolBarData::currentDefaultGroupPad;
}

/*------------------------------------------------------------------------------
| IToolBar::defaultMisfitWidth                                                 |
|                                                                              |
| Returns the current misfit width.                                            |
------------------------------------------------------------------------------*/
unsigned long IToolBar::defaultMisfitWidth ( )
{
  return IToolBarData::currentDefaultMisfitWidth;
}

/*------------------------------------------------------------------------------
| IToolBar::defaultMargin                                                      |
|                                                                              |
| Returns the current default margin.                                          |
------------------------------------------------------------------------------*/
ISize IToolBar::defaultMargin ( )
{
  return IToolBarData::currentDefaultMargin;
}

/*------------------------------------------------------------------------------
| IToolBar::layout                                                             |
|                                                                              |
| Filters the toolbar if necessary.                                            |
------------------------------------------------------------------------------*/
IToolBar&  IToolBar::layout ( )
{
  if ( !fToolBarData->skipLayout )
    {
      if (isLayoutDistorted(IWindow::layoutChanged))
        {
          filter();
          Inherited::layout();
        }
    }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::setLayoutDistorted                                                 |
|                                                                              |
| Provide the response specific for a tool bar to various distortion           |
| notifications.  Some will cause the canvas to need to update how it has      |
| layed out its child windows.                                                 |
------------------------------------------------------------------------------*/
IToolBar&  IToolBar::setLayoutDistorted ( unsigned long layoutAttributeOn,
                                          unsigned long layoutAttributeOff )
{
  IMODTRACE_DEVELOP("ITBar::setLayoutDistorted");
  ITRACE_DEVELOP(IString("Layout Attribute is: ")+
                 IString(layoutAttributeOn).d2x());

  // Ensure that changes to the children of the tool bar cause a change
  // to the tool bar container so that it can resize its frame when floating
  if (( fToolBarData->currentLocation == floating) &&
      ( layoutAttributeOn & childMinimumSizeChanged ))
  {
    layoutAttributeOn |= IWindow::layoutChanged;
  }

  if ( !fToolBarData->skipLayout )
  {
    if (layoutAttributeOn & IWindow::layoutChanged)
    {
      //If vertical, then store margin, it is used as expected.
      if (deckOrientation() == ISetCanvas::vertical)
         fToolBarData->margin = margin();
      else //If horizontal, then store inverse of margin.
         fToolBarData->margin = ISize(margin().height(),
                                      margin().width() );
    }

    // If color changed, re-query system colors so next WM_PAINT will use
    // new colors
    if (layoutAttributeOn & IWindow::colorChanged )
    {
      // Rather than create an IColor (which queries the system for the
      // actual color) each time a color is needed, cache the value
      // and update the cache here.  When the system color changes, all
      // windows will receive a setLayoutDistorted with colorChanged.
      if (toolBarStatics.colorButtonDark)
      {
        delete toolBarStatics.colorButtonDark;
      }
      toolBarStatics.colorButtonDark = new IColor(IColor::buttonDark);
      if (toolBarStatics.colorButtonLight)
      {
        delete toolBarStatics.colorButtonLight;
      }
      toolBarStatics.colorButtonLight = new IColor(IColor::buttonLight);
    }

    ITRACE_DEVELOP("Called Inherited::setLayoutDistorted");
    Inherited::setLayoutDistorted(layoutAttributeOn, layoutAttributeOff);

    // Need to adjust the size of a floating tool bar
    if (( fToolBarData->floatingFrame ) &&
        ( layoutAttributeOn & minimumSizeChanged ))
    {
      IPoint newPos(fToolBarData->floatingFrame->position());
      ISize  minimum(minimumSize());

      // Check if the toolbar is empty or has been collapsed.
      // If so make sure the floating toolbar frame retains
      // its width.
      if (minimum == ISize(0,0))
      {
        minimum.setWidth(size().width());
        minimum.setHeight(1);                   // 1 pel height
      }

      // Position the frame window such that the title bar remains
      // in the same location regardless of changes to the toolbar.

      IRectangle frameRect(fToolBarData->floatingFrame->frameRectFor(
                           IRectangle( newPos, minimum )));

      if (ICoordinateSystem::applicationOrientation() ==
          ICoordinateSystem::kOriginLowerLeft)
        {
          // need to move position to adjust for changes in the size of
          // the client area.
          IPoint topLeft(fToolBarData->floatingFrame->rect().topLeft());

          frameRect.moveTo( IPoint(
                            topLeft.x(),
                            topLeft.y() - frameRect.size().height() ));
        }
      else
        {
          frameRect.moveTo( newPos );
        }

      fToolBarData->floatingFrame->moveSizeTo(frameRect);
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::expand                                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::expand( bool expand )
{
  if ( fToolBarData->isExpanded != expand )
  {
    fToolBarData->isExpanded = expand;
    setLayoutDistorted(IWindow::layoutChanged | IWindow::minimumSizeChanged |
                       IWindow::immediateUpdate, 0);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::collapse                                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::collapse ( )
{
  expand ( false );
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::isExpanded                                                         |
------------------------------------------------------------------------------*/
bool IToolBar::isExpanded( ) const
{
  return fToolBarData->isExpanded;
}

/*------------------------------------------------------------------------------
| IToolBar::enableDragDrop                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar :: enableDragDrop ( bool bTurnOn )
{
  //This function enables drag/drog regardless if the dragDrop style was
  //specified on toolbar creation.

  unsigned long ulExtStyle = extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  if (bTurnOn)
  {
    ulExtStyle |= dragDrop.asExtendedUnsignedLong();

    //Initialize drag/drop if initialization has not already taken place
    if (!fToolBarData->isDragDropInitialized)
    {
      IDMHandler::enableDropOn(this);

      IToolBar *nonConstThis = (IToolBar*)this;
      IWindow::ChildCursor cursor(*nonConstThis);
      for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
      {
        IWindow* childWindow = IWindow::windowWithHandle(childAt(cursor));
        //If the child is a toolbar button, setup drag/drop for the button.
        IToolBarButton *tBarButton = dynamic_cast<IToolBarButton*>(childWindow);
        if (tBarButton)
          {
            if (!tBarButton->itemProvider())
              IDMHandler::enableDragDropFor(tBarButton);
          }
      }
    }
  }
  else
  {
    ulExtStyle &= ~dragDrop.asExtendedUnsignedLong();
  }

  if (ulExtStyle != ulOldExtStyle)
  {
    setExtendedStyle( ulExtStyle );
  }

  return( *this );
}

/*------------------------------------------------------------------------------
| IToolBar::disableDragDrop                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar :: disableDragDrop ( )
{
  enableDragDrop( false );
  return( *this );
}


/*------------------------------------------------------------------------------
| IToolBar::allowsDragDrop                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar :: allowsDragDrop ( ) const
{
  return( (extendedStyle() & dragDrop.asExtendedUnsignedLong()) ? true
                                                                  : false );
}

/*------------------------------------------------------------------------------
| IToolBar::filter                                                             |
|                                                                              |
| The IToolBar::filter() function iterates through the toolbar's children      |
| and filters each one.  A list is maintained so filter() can add windows      |
| back in their original position.                                             |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::filter ( )
{
  IToolBarObjectList::Cursor cursor(*(fToolBarData->objectList));

  IToolBarObject* lastObj = 0;
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
  {
    IToolBarObject* winObj = fToolBarData->objectList->elementAt(cursor);
    if (( deckOrientation() == vertical ) &&
        ( extendedStyle() & filterMisfits.asExtendedUnsignedLong() ))
    {
      // Check if any windows need to be hidden
      if ( winObj->isVisible() &&
         ( winObj->window()->minimumSize().width() >
           fToolBarData->misfitWidth ))
      {
        this->Inherited::remove( winObj->window() );
        winObj->setVisible(false);
      }

      // Check if any hidden windows need to be reshown
      if ( ( winObj->isVisible() == false ) &&
           ( winObj->window()->minimumSize().width() <
             fToolBarData->misfitWidth ))
      {
        this->add( winObj->window() );
        winObj->setVisible(true);
        winObj->window()->positionOnSiblings();
        if ( lastObj )
          winObj->window()->positionBehindSibling(lastObj->window()->handle());
      }

    }
    else
    {
      // Check if any hidden windows need to be reshown
      if ( winObj->isVisible() == false )
      {
        this->add( winObj->window() );
        winObj->setVisible(true);
        winObj->window()->positionOnSiblings();
        if ( lastObj )
          winObj->window()->positionBehindSibling(lastObj->window()->handle());
      }
    }

    if ( winObj->isVisible() )
      lastObj = winObj;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::calcMinimumSize                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
ISize IToolBar::calcMinimumSize( ) const
{
  if (fToolBarData->isExpanded)
  {
    ISize minimum;

    if (hasChildrenToLayout())
    {
      minimum = Inherited::calcMinimumSize();
    }
    else
    {
#ifdef IC_PMWIN
      // Use standard button size for default minimum size of an empty toolbar
      unsigned long cxBorder = IQUERYSYSVALUE(SV_CXBORDER);
      unsigned long cyBorder = IQUERYSYSVALUE(SV_CYBORDER);
#endif
#ifdef IC_MOTIF
      unsigned long cxBorder = 1;
      unsigned long cyBorder = 1;
#endif
      minimum.setWidth  ( cxBorder * 5 + 10 +
                          IToolBarButton::standardBitmapSize().width());
      minimum.setHeight ( cyBorder * 5 + 16 +
                          IToolBarButton::standardBitmapSize().height());
    }
    ITRACE_ALL(IString("IToolBar::calcMinimumSize =") + minimum.asString() );
    return minimum;
  }
  else
  {
    ITRACE_ALL(IString("IToolBar::calcMinimumSize =") + ISize(0,0).asString() );
    return ISize(0,0);
  }
}

/*------------------------------------------------------------------------------
| Class:   IToolBarFrameCursorData                                             |
|                                                                              |
| Purpose: Data for an IToolBar::FrameCursor.                                  |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarFrameCursorData
{
public:
IFrameWindow
  *frame;
IToolBarList
  *list;
IToolBarList::Cursor
  *cursor;
};

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::FrameCursor                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::FrameCursor::FrameCursor ( const IFrameWindow* frame )
{
   fToolBarFrameCursorData = new IToolBarFrameCursorData;
   fToolBarFrameCursorData->frame = (IFrameWindow*) frame;
   fToolBarFrameCursorData->list = 0;
   fToolBarFrameCursorData->cursor = 0;
}

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::~FrameCursor                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::FrameCursor::~FrameCursor ()
{
  delete fToolBarFrameCursorData->cursor;
  delete fToolBarFrameCursorData;
}

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::setToFirst                                            |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::FrameCursor::setToFirst ()
{
  if ( fToolBarFrameCursorData->list !=
       fToolBarFrameCursorData->frame->toolBarList() )
  {
    delete fToolBarFrameCursorData->cursor;
    fToolBarFrameCursorData->cursor = 0;
    fToolBarFrameCursorData->list = fToolBarFrameCursorData->frame->toolBarList();
  }

  if ( fToolBarFrameCursorData->cursor )
  {
    return fToolBarFrameCursorData->cursor->setToFirst();
  }
  else
  {
    if ( fToolBarFrameCursorData->list )
    {
      fToolBarFrameCursorData->cursor = new IToolBarList::Cursor
                                        (*(fToolBarFrameCursorData->list));
      return fToolBarFrameCursorData->cursor->setToFirst();
    }
  }
  return false;
}

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::setToNext                                             |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::FrameCursor::setToNext ()
{
  if ( fToolBarFrameCursorData->list !=
       fToolBarFrameCursorData->frame->toolBarList() )
  {
    delete fToolBarFrameCursorData->cursor;
    fToolBarFrameCursorData->cursor = 0;
    fToolBarFrameCursorData->list = fToolBarFrameCursorData->frame->toolBarList();
  }

  if ( fToolBarFrameCursorData->cursor )
  {
    return fToolBarFrameCursorData->cursor->setToNext();
  }
  else
  {
    if ( fToolBarFrameCursorData->list )
    {
      fToolBarFrameCursorData->cursor = new IToolBarList::Cursor
                                        (*(fToolBarFrameCursorData->list));
      return fToolBarFrameCursorData->cursor->setToNext();
    }
  }
  return false;
}

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::isValid                                               |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::FrameCursor::isValid () const
{
  if ( fToolBarFrameCursorData->list !=
       fToolBarFrameCursorData->frame->toolBarList() )
  {
    delete fToolBarFrameCursorData->cursor;
    fToolBarFrameCursorData->cursor = 0;
    fToolBarFrameCursorData->list = fToolBarFrameCursorData->frame->toolBarList();
  }

  if ( fToolBarFrameCursorData->cursor )
  {
    return fToolBarFrameCursorData->cursor->isValid();
  }
  return false;
}

/*------------------------------------------------------------------------------
| IToolBar::FrameCursor::invalidate                                            |
|                                                                              |
------------------------------------------------------------------------------*/
void IToolBar::FrameCursor::invalidate ()
{
  if ( fToolBarFrameCursorData->cursor )
  {
    delete fToolBarFrameCursorData->cursor;
    fToolBarFrameCursorData->cursor = 0;
  }
}

/*------------------------------------------------------------------------------
| IToolBar::toolBarAt                                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar* IToolBar::toolBarAt ( const FrameCursor& cursor )
{
  if ( cursor.fToolBarFrameCursorData->cursor )
  {
    return cursor.fToolBarFrameCursorData->list->elementAt
                  (*(cursor.fToolBarFrameCursorData->cursor));
  }
  // The cursor is invalid so throw exception
  ITHROWLIBRARYERROR(IC_UNEXPECTED_ERROR,
                     IBaseErrorInfo::invalidParameter,
                     IException::recoverable);
  return (0);
}

/*------------------------------------------------------------------------------
| Class:   IToolBarWindowCursorData                                            |
|                                                                              |
| Purpose: Data for an IToolBar::WindowCursor.                                 |
|                                                                              |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IToolBarWindowCursorData
{
public:
  IToolBarWindowCursorData (IToolBarObjectList* objectList);
  ~IToolBarWindowCursorData ( );
IToolBarObjectList::Cursor
  cursor;
};

#pragma pack(pop)
#pragma enum(pop)

IToolBarWindowCursorData::IToolBarWindowCursorData (IToolBarObjectList* objectList)
                         :cursor(*objectList)
{
}

IToolBarWindowCursorData::~IToolBarWindowCursorData ()
{
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::WindowCursor                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::WindowCursor::WindowCursor ( const IToolBar* toolbar )
{
  fToolBarWindowCursorData = new IToolBarWindowCursorData
                             (toolbar->fToolBarData->objectList);
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::~WindowCursor                                        |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar::WindowCursor::~WindowCursor ()
{
  delete fToolBarWindowCursorData;
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::setToFirst                                           |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::WindowCursor::setToFirst ()
{
  return fToolBarWindowCursorData->cursor.setToFirst();
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::setToNext                                            |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::WindowCursor::setToNext ()
{
  return fToolBarWindowCursorData->cursor.setToNext();
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::isValid                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IToolBar::WindowCursor::isValid () const
{
  return fToolBarWindowCursorData->cursor.isValid();
}

/*------------------------------------------------------------------------------
| IToolBar::WindowCursor::invalidate                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
void IToolBar::WindowCursor::invalidate ()
{
  fToolBarWindowCursorData->cursor.invalidate();
}

/*------------------------------------------------------------------------------
| IToolBar::windowAt                                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow* IToolBar::windowAt ( const WindowCursor& cursor ) const
{
  if ( cursor.isValid() )
  {
    IToolBarObject* winObj = fToolBarData->objectList->elementAt
                             (cursor.fToolBarWindowCursorData->cursor);
    return winObj->window();
  }
  // The cursor is invalid so throw exception
  ITHROWLIBRARYERROR(IC_UNEXPECTED_ERROR,
                     IBaseErrorInfo::invalidParameter,
                     IException::recoverable);
  return (0);
}

/*------------------------------------------------------------------------------
| IToolBar::show                                                               |
|                                                                              |
| - Variable checks:                                                           |
|    1) skipShowProcessing:  When this flag is set, show does nothing but      |
|       return. This flag is used during toolbar setLocation(), so that        |
|       changing the location (which causes hide()/show() calls by ISetCanvas) |
|       will not affect the hidden/visible state of the toolbar.               |
|                                                                              |
|       This allows for the following scenario - User hides toolbar, then      |
|       issues setLocation();  The toolbar should remain hidden.               |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::show( bool showWindow )
{
  if (fToolBarData->skipShowProcessing)
    return *this;

  // ------- Hide case ----------
  // If toolbar is to be hidden, and it is in shown state then proceed,
  // otherwise just call inherited::show().
  if ( (!showWindow) &&
       (fToolBarData->isShown == true) )
  {
    fToolBarData->isShown = false;
    if (location() == IToolBar::floating)
    {
       floatingFrame()->hide();
    }
    else  /* what about 'hidden' location */
    {
       toolBarContainer()->remove(this);
    }
  }

  // ------- Show case ----------
  // If toolbar is to be shown, and it is in hidden state then proceed,
  // otherwise just call inherited::show().
  if ( (showWindow) &&
       (fToolBarData->isShown == false) )
  {
    fToolBarData->isShown = true;
    if (location() == IToolBar::floating)
    {
       floatingFrame()->show();
       #ifdef IC_WIN
       if (isExpanded())
       {
         // We need to show maximized
         fToolBarData->floatingFrame->maximize();
       }
       #endif
       // Put the floating frame on top of sibling windows if the owning frame
       // has the focus (otherwise let the z-order be the default)
       if ( fToolBarData->owningFrame->hasFocus() )
       {
         fToolBarData->floatingFrame->positionOnSiblings();
       }
    }
    else  /* what about 'hidden' location */
    {
       toolBarContainer()->add(this);

       //There are certain cases where after adding the toolbar to the
       //container, the toolbar does not show up.  The following code is
       //to force the relayout/repaint so that the toolbar does show up.
       Inherited::show(showWindow);
       toolBarContainer()->setLayoutDistorted(IWindow::layoutChanged |
                                              IWindow::minimumSizeChanged |
                                              IWindow::childMinimumSizeChanged |
                                              IWindow::immediateUpdate, 0);
       toolBarContainer()->refresh();
       // No need to drop out and issue second call to Inherited::show()
       return *this;
    }
  }
  Inherited::show(showWindow);
  return *this;
}

/*------------------------------------------------------------------------------
| IToolBar::hide                                                               |
|                                                                              |
------------------------------------------------------------------------------*/
IToolBar& IToolBar::hide()
{
  return IToolBar::show(false);
}
