/* Revision: 66 1.4.1.6 source/ui/cclnt/spinbtn/spinbtn.c, cclspin, ioc.v400  */
/****************************************************************************/
/*                                                                          */
/*                         IBM CONFIDENTIAL                                 */
/*                                                                          */
/*        (IBM Confidential-Restricted when combined with the               */
/*         aggregated OCO source modules for this program)                  */
/*                                                                          */
/*                       OCO SOURCE MATERIALS                               */
/*                                                                          */
/*         20G0400 (C) Copyright IBM Corp. 1992, 1997 (Unpublished)         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                                                                          */
/* MODULE = SPINBTN.C                                                       */
/* COMPONENT = SPIN BUTTON                                                  */
/* USAGE = SPIN BUTTON SOURCE                                               */
/*                                                                          */
/*                                                                          */
/* NOTES =                                                                  */
/*     NONE.                                                                */
/*                                                                          */
/*   RESTRICTIONS =                                                         */
/*                                                                          */
/*     NONE.                                                                */
/*                                                                          */
/*                                                                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/*                           Windows Modifications                          */
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* @W01 1           06/10/91  mab   1   Initial pop!                        */
/* @XXX 1           07/03/91  mab   1   Problem areas - no PM/Windows sync  */
/* @OS2 1           07/03/91  mab   1   Port fixes to OS/2                  */
/* @L01 1  8500198  11/27/91  mab   1   Fix bug caused by CCL name changes  */
/* @L02 1  8500198  01/25/92  mab   1   Spin Button disable support         */
/* @L03 1  8500425  02/13/92  mab   4   Selection/Editing bug               */
/* WBB1             02/21/92  WBB       Error handling                      */
/* @L04 1  8500501  02/27/92  mab   4   WM_CTLCOLOR not passed to parent    */
/* @L05 1  8500502  02/27/92  mab   9   Add WS_TABSTOP support for dialogs  */
/* @B01 1  8510002  06/02/92  mab   1   Cannot scroll to right on large font*/
/* @B02 1  8510043  06/02/92  mab   1   Part of servant background clr wrong*/
/* @J01 1  8500     07/31/92  jeh       Buttons not correct on VGA          */
/* @B03 1  8510061  08/12/92  mab   1   WM_CHAR instead of WM_KEYDOWN       */
/* @B04 1  8510062  08/12/92  mab   1   Fix problems with fix for 8510002   */
/* @B05 1  8510064  08/13/92  mab   1   Full deletion on SPBQ_ALWAYSUPDATE  */
/* @B06 1  8510065  08/16/92  mab   1   String comparison failing           */
/* @B07 1  8510066  08/26/92  mab   1   Bug in numeric select/replace logic */
/* @B08 1  8510069  10/02/92  mab   1   Escape destroys spinbtn in 3.1 dlg  */
/* @B09 1  8510070  10/02/92  mab   1   Spin notifications going to master  */
/* DAB1 1  85xxxxx  06/02/94  dab  11   Map message APIs for NT to OS/2 mapg*/
/* EVB1 1  85xxxxx  10/31/95  evb       Add color support for spinbtn       */
/* DAB2 1  85xxxxx  01/22/96  dab   7   Filter delete key if readonly spinbt*/
/* RKK1 1  85xxxxx  04/30/96  rkk       Change "SPINBTNP.H" to "spinbtnp.h" */
/* RKK2 1  85xxxxx  04/30/96  rkk       Use _strrev() for Wind/U instead.   */
/****************************************************************************/
/****************************************************************************/
/* Includes                                                                 */
/****************************************************************************/
#include <windows.h>
#include <iclccl.h>     /* CCL common include file */
#include "iclspbw.h"

#include "spinbtnp.h"                                           /*RKK1*/

extern HANDLE hSpinInstance;
WNDPROC pfnOldWindowProc = NULL;

/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinButtonWndProc                              */
/*                                                                    */
/*  Descriptive Name:  Spin Button Window Procedure                   */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to process messages for the spin button      */
/*  control.                                                          */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          msg  - Message to process                                 */
/*          wParam  - Message Parameter 1                             */
/*          lParam  - Message Parameter 2                             */
/*                                                                    */
/*  Output:  See returns from individual messages                     */
/*                                                                    */
/*  Exit-Normal: See returns from individual messages                 */
/*                                                                    */
/*  Exit-Error:  See returns from individual messages                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/**********************************************************************/
LRESULT CALLBACK SpinButtonWndProc (HWND hwnd, UINT msg,
                                    WPARAM wParam, LPARAM lParam)
{
 /*-------------------------------------------------------------------*
  *  Get pointer to internally stored data from window words.
  *-------------------------------------------------------------------*/
  PSPINDATA pSpinData = (PSPINDATA)GetWindowLong (hwnd, GWL_SPIN);

  switch(msg)
  {
    case WM_CREATE:
      return (SpinCreate (hwnd, (LPCREATESTRUCT)lParam));
      break;

    /*----------------------------------------------------------------*
     *  Send entryfield focus: component has no business with focus.
     *     You can't change focus on a WM_SETFOCUS.  Therefore I post
     *     a message to myself so that when the message I set gets
     *     through the QUEUE, all of the focus processing will have
     *     finished and it will be safe to change the focus.
     *----------------------------------------------------------------*/
    case WM_SETFOCUS:
      SENDMESSAGE (hwnd, WM_SETFOCUS_PRIVATE, 0, 0);
      break;

    /*----------------------------------------------------------------*
     *  Send entryfield focus: component has no business with focus.
     *----------------------------------------------------------------*/
    case WM_SETFOCUS_PRIVATE:
      SetFocus (pSpinData->hwndSpinField);
      break;

    case WM_DESTROY:
      SpinDestroy (hwnd, pSpinData);
      break;

    case WM_PAINT:
      SpinPaint (hwnd, pSpinData);
      break;

    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
      /*------------------------------------------------------*/
      /* Return a value that will indicate that we're not     */
      /* subclassed by the Dialog Editor (see the button down */
      /* cases of ArrowWndProc and SpinSubclassEntryfield).   */
      /*------------------------------------------------------*/
      return (SB_GOTBUTTONDOWN);
      break;

    case WM_SIZE:
     /*---------------------------------------------------------------*
      *  Reflow the spinbutton control entryfield and scrollbar.
      *---------------------------------------------------------------*/
      SpinAdjustWindowPos (pSpinData, hwnd, wParam,
                           LOWORD(lParam), HIWORD(lParam));
      return ((long)TRUE);
      break;


    case WM_ENABLE:
     /*---------------------------------------------------------------*
      *  Enable child windows of spinbutton
      *---------------------------------------------------------------*/
      EnableWindow (pSpinData->hwndSpinField, (BOOL)wParam);
      if (pSpinData->hwndArrows)
         EnableWindow (pSpinData->hwndArrows, (BOOL)wParam);
      break;


    case WM_COMMAND:
      SpinControl (hwnd, pSpinData, LOWORD(wParam),
                   HIWORD(wParam), (HWND)(UINT)lParam);
      break;


    case WM_CTLCOLOREDIT:
     /*---------------------------------------------------------------*
      *  Reflow the spinbutton control entryfield and scrollbar.
      *---------------------------------------------------------------*/
// JEH if (HIWORD(lParam) == CTLCOLOR_EDIT)
        {
// EVB1  HBRUSH hBrush;                                                 //->@L04

// EVB1  if (!(hBrush = (HBRUSH)SENDMESSAGE (PARENTOF(hwnd), msg,
// EVB1                                      wParam, lParam)))
// EVB1  {
           return (LRESULT)(SpinSetEditCtrlBrush ((HDC)wParam, pSpinData));  //@EVB1
// EVB1  }
// EVB1  else
// EVB1  {
// EVB1    return (LRESULT)(hBrush);
// EVB1  }                                                              //@L04<-
      }
      break;


    /******************************************************************/
    /* If a WM_SETFONT message comes in, set the hSpbFont field of    */
    /* the spin button's internal data.                               */
    /*   wParam: font handle - handle of the new font, if NULL then   */
    /*                      system font will be used                  */
    /*   lParam: if TRUE, redraw spin btn immediately, FALSE, don't   */
    /******************************************************************/
    case WM_SETFONT:
    {
      RECT rect;
      BYTE  Temp[CCHMAXLONG * 2];                                       //@B04

      pSpinData->hSpbFont = (HFONT)wParam;
      GetClientRect(hwnd, &rect);
      if (SpinAdjustWindowPos (pSpinData, hwnd, SIZENORMAL,
                           rect.right, rect.bottom))
      {
        SENDMESSAGE((HWND)GetDlgItem(hwnd, ID_EDIT), WM_SETFONT,
                    wParam, FALSE);
      }
      else
        return (FALSE);

#ifndef WIN31
      if (pSpinData->fl & SBF_ARRAYDATA)                                //->@B04
      {
        SetWindowText (pSpinData->hwndSpinField,                        //@B01
                       pSpinData->anpszText[pSpinData->lCurrentValue]);
      }
      else
      {
        ltoa (hwnd, pSpinData, (LPSTR)Temp);
        SetWindowText (pSpinData->hwndSpinField, (LPSTR)Temp);
      }                                                                 //@B04<-
#endif

      InvalidateRect (hwnd, NULL, TRUE);
    }
      break;

    /******************************************************************/
    /* If a WM_GETFONT message comes in, return the hSpbFont field    */
    /* of the spin button's internal data.                            */
    /******************************************************************/
    case WM_GETFONT:

      return ((long)pSpinData->hSpbFont);
      break;


    case SPBM_QUERYLIMITS:
     /*---------------------------------------------------------------*
      *  Be sure that the two message parameters actually contain
      *  addresses and that the spinbutton contains data (ie not in
      *  default state) and that the data is not in an array.
      *---------------------------------------------------------------*/
      if (wParam && lParam && (pSpinData->fl & SBF_DATAEXIST)  &&
          !(pSpinData->fl & SBF_ARRAYDATA))
      {
         *(PLONG)wParam = pSpinData->lUpperLimit;                       /*DAB1*/
         *(PLONG)lParam = pSpinData->lLowerLimit;
         return ((long)TRUE);
      } /* end if */
      break;


    case SPBM_SETMASTER:
      return (SpinSetMaster (pSpinData, (HWND)wParam));
      break;


    case SPBM_SETTEXTLIMIT:
      return (SpinSetTextLimit (hwnd, pSpinData, LOWORD(wParam)));
      break;


    case SPBM_SETCURRENTVALUE:
      return (SpinSetCurrentValue (hwnd, pSpinData, (LONG)wParam));     /*DAB1*/
      break;


    case SPBM_OVERRIDESETLIMITS:
    case SPBM_SETLIMITS:
      return (SpinSetLimits (hwnd, pSpinData, (LONG)wParam,             /*DAB1*/
                        (LONG)lParam, (msg == SPBM_OVERRIDESETLIMITS)));
      break;


    case SPBM_SETARRAY:
     /*---------------------------------------------------------------*
      *  If we were given an array with size > 0, copy in the new data
      *---------------------------------------------------------------*/
      if (wParam && lParam)
         return (SpinSetArray (hwnd, pSpinData, (LPPSTR)wParam,         /*DAB1*/
                               (WORD)lParam));                          /*DAB1*/
      break;


    case SPBM_SPINUP:
    case SPBM_SPINDOWN:
      return (SpinSpinValue (hwnd, pSpinData, wParam,                   /*DAB1*/
                            (msg == SPBM_SPINUP)));
      break;


    case SPBM_QUERYVALUE:
      return (SpinQueryValue (hwnd, pSpinData, (LPVOID)wParam, lParam));/*DAB1*/
      break;

    case SPBM_QUERYSPINCOLORS:                                             //@EVB
      return ((long)SpinQueryPresParam(pSpinData, LOWORD(wParam)));        //@EVB
      break;                                                               //@EVB

    case SPBM_SETSPINCOLORS:                                               //@EVB
      if (SpinSetPresParam(pSpinData, LOWORD(wParam), (COLORREF)lParam))   //@EVB
      {                                                                    //@EVB
         InvalidateRect(hwnd, NULL, FALSE);                                //@EVB
         return((long)TRUE);                                               //@EVB
      }                                                                    //@EVB
      else                                                                 //@EVB
         return((long)FALSE);                                              //@EVB
      break;                                                               //@EVB

    case SPBM_RESETSPINCOLORS:                                             //@EVB
      if (SpinResetPresParam(pSpinData, LOWORD(wParam)))                   //@EVB
      {                                                                    //@EVB
         InvalidateRect(hwnd, NULL, FALSE);                                //@EVB
         return((long)TRUE);                                               //@EVB
      }                                                                    //@EVB
      else                                                                 //@EVB
         return((long)FALSE);                                              //@EVB
      break;                                                               //@EVB

    default:
      return (DefWindowProc (hwnd, msg, wParam, lParam));
      break;
  } /* end switch */

  return (DefWindowProc (hwnd, msg, wParam, lParam));
} /* end SpinButtonWndProc */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinCreate                                     */
/*                                                                    */
/*  Descriptive Name:  Spin button Creation procedure                 */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is create a spin button and handle all               */
/*  initializations associated with the creation process.             */
/*                                                                    */
/*                                                                    */
/*  Notes:  Must use CS_GLOBALCLASS style when registering class      */
/*          in a DLL ...                                              */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pCreate - Pointer to create structure                     */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  FALSE - successful                                  */
/*                                                                    */
/*  Exit-Error:  TRUE - unsuccessful                                  */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinCreate (HWND hwnd, LPCREATESTRUCT pCreate)
{
  ENTRYFDATA  ctlEntry;            /* Creation data for entryfield */
  PSPINDATA   pSpinData;
//WNDCLASS    wndclass;

 /*-------------------------------------------------------------------*
  *  Allocate the instance data for this spinbutton from the heap
  *-------------------------------------------------------------------*/
  if (!(pSpinData = (PSPINDATA)AllocHeapMem (sizeof(SPINDATA))))
     goto Err;

//#ifdef DEBUG
  memset (pSpinData, 0, sizeof(SPINDATA));                              /*@L01*/
//#endif

  SetWindowLong (hwnd, GWL_SPIN, (LONG)pSpinData);

 /*-------------------------------------------------------------------*
  *  Get window instance
  *-------------------------------------------------------------------*/
  pSpinData->hInstance = pCreate->hInstance;

 /*-------------------------------------------------------------------*
  *  Load bitmaps, register arrow window class, and create arrow
  *  window iff the WM_CREATE we're processing is for a master ...
  *-------------------------------------------------------------------*/
  if (STYLEOF(hwnd) & SPBS_MASTER)
  {
    /*----------------------------------------------------------------*
     *  Try to load spin button bitmaps
     *----------------------------------------------------------------*/
     if (!(pSpinData->hbmUp = LoadBitmap (hSpinInstance, (LPSTR)"Up")))
        goto ErrBitmapUp;

     if (!(pSpinData->hbmDown = LoadBitmap (hSpinInstance, (LPSTR)"Down")))
        goto ErrBitmapDn;
                                                                        //->$L02
     if (!(pSpinData->hbmUpDisabled = LoadBitmap (hSpinInstance,
                                                  (LPSTR)"UpDisabled")))
        goto ErrBitmapUpDis;

     if (!(pSpinData->hbmDownDisabled = LoadBitmap (hSpinInstance,
                                                    (LPSTR)"DownDisabled")))
        goto ErrBitmapDnDis;                                            //$L02<-

    /*----------------------------------------------------------------*
     *  Create the up/down arrow window
     *----------------------------------------------------------------*/
     if (!(pSpinData->hwndArrows = CreateWindow (WC_ARROWS, NULL,
                                                 WS_VISIBLE | WS_CHILD,
                                                 0, 0, 0, 0, hwnd, 0,
                                                 pSpinData->hInstance,
                                                 NULL)))
     {
        goto ErrArrows;
     } /* end if */
  } /* end if - MASTER */

 /*-------------------------------------------------------------------*
  *  Init internal spin button values
  *-------------------------------------------------------------------*/
//pSpinData->hInstance           = pCreate->hInstance;   /* Apps inst */
  pSpinData->usNumericFormatSize = CCHMAXLONG;
  pSpinData->ulStyleChange       = GetSpinButtonStyle (hwnd);
  pSpinData->ulWarpFactor        = 1;
  pSpinData->usTextLimit         = CCHMAXEDIT;
  pSpinData->ulColorVal[CLR_SPBBACKGROUND] = GetSysColor(COLOR_WINDOW);     //@EVB
  pSpinData->ulColorVal[CLR_SPBFOREGROUND] = GetSysColor(COLOR_WINDOWTEXT); //@EVB
  pSpinData->hBrush                        = NULL;                          //@EVB

 /*-------------------------------------------------------------------*
  *  Tell this entryfield that its default font is the system font
  *-------------------------------------------------------------------*/
  pSpinData->hSpbFont = NULL;

 /*-------------------------------------------------------------------*
  *  Tell this entryfield that its master is its own spinarrows
  *-------------------------------------------------------------------*/
  pSpinData->hwndMaster = hwnd;

 /*-------------------------------------------------------------------*
  *  Initialize control data for edit control
  *-------------------------------------------------------------------*/
  ctlEntry.cb           = sizeof(ENTRYFDATA);
  ctlEntry.cchEditLimit = CCHMAXEDIT;
  ctlEntry.ichMaxSel    = 0;
  ctlEntry.ichMinSel    = 0;

 /*-------------------------------------------------------------------*
  *  Create the entryfield invisible and arbitrary size
  *-------------------------------------------------------------------*/
  if (!(pSpinData->hwndSpinField = CreateWindow ((LPSTR)"Edit",
                                                  (LPSTR)NULL,
                                                  pSpinData->ulStyleChange,
                                                  0, 0, 0, 0,
                                                  hwnd,
                                                  (HMENU)ID_EDIT,
                                                  pSpinData->hInstance,
                                                  (LPSTR)&ctlEntry)))
  {
     goto ErrEdit;
  } /* end if */

 /*-------------------------------------------------------------------*
  *  Subclass the entryfield for character filter
  *-------------------------------------------------------------------*/
  pfnOldWindowProc = (WNDPROC)GetWindowLong (pSpinData->hwndSpinField, GWL_WNDPROC);
  SetWindowLong (pSpinData->hwndSpinField,
                 GWL_WNDPROC,
                 (LONG)(FARPROC)SpinSubclassEntryfield);

  SENDMESSAGE(pSpinData->hwndSpinField, WM_SETFONT, 0, TRUE);
  return ((long)FALSE);

ErrEdit:
  if (pSpinData->hwndArrows)
     DestroyWindow (pSpinData->hwndArrows);

ErrArrows:                                                              //->$L02
  if (pSpinData->hwndArrows)
     DeleteObject (pSpinData->hbmDownDisabled);

ErrBitmapDnDis:
  if (pSpinData->hwndArrows)
     DeleteObject (pSpinData->hbmUpDisabled);

ErrBitmapUpDis:                                                         //$L02<-
  if (pSpinData->hwndArrows)
     DeleteObject (pSpinData->hbmDown);

ErrBitmapDn:
  if (pSpinData->hwndArrows)
     DeleteObject (pSpinData->hbmUp);

ErrBitmapUp:
  FreeHeapMem ((PBYTE)pSpinData);

Err:
  return ((long)SPBERR_CREATE_FAILED);
} /* end SpinCreate */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinPaint                                      */
/*                                                                    */
/*  Descriptive Name:  Spin button Paint procedure                    */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine handles processing of the WM_PAINT message           */
/*  for the spin button component (i.e. handles repainting chores).   */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
VOID SpinPaint (HWND hwnd, PSPINDATA pSpinData)
{
//  RECT    wrc;
  DWORD   ulSpinfieldStyle;
  HDC     hdc;
  PAINTSTRUCT  paint;
  HBRUSH  hBrushOld;
  HPEN    hPenOld;
//#if 0      /*-------------------------< MAB >--------------------------------*/
  HBRUSH  hBrushNew;
  HPEN    hPenNew;
  int     oldBkMode;
//#endif     /*-------------------------< MAB >--------------------------------*/
  HBRUSH  hNullBrush = NULL;                                            /*WBB1*/
  HPEN    hBlackPen = NULL;                                             /*WBB1*/

  ulSpinfieldStyle = GetSpinButtonStyle (hwnd);

 /*-------------------------------------------------------------------*
  *  If the styles have changed, then reflow the windows
  *-------------------------------------------------------------------*/
  if (((LONG)ulSpinfieldStyle != STYLEOF(pSpinData->hwndSpinField))  ||
      (pSpinData->ulStyleChange != ulSpinfieldStyle))
  {
     pSpinData->ulStyleChange = ulSpinfieldStyle;

    /*----------------------------------------------------------------*
     *  Change the entryfield styles to match that specified in
     *  the spinbutton styles.
     *----------------------------------------------------------------*/
     SetWindowLong (pSpinData->hwndSpinField, GWL_STYLE, ulSpinfieldStyle);
     InvalidateRect (pSpinData->hwndSpinField, NULL, TRUE);             //@B02
  } /* end if */

 /*-------------------------------------------------------------------*
  *  Get edit field client coordinates
  *-------------------------------------------------------------------*/
//  GetClientRect (pSpinData->hwndSpinField, (LPRECT)&wrc);

 /*-------------------------------------------------------------------*
  *  The window for the up arrow button is one greater than the
  *  bottom (in the case of an odd height)
  *-------------------------------------------------------------------*/
  if (pSpinData->hwndArrows)
  {
    MoveWindow (pSpinData->hwndArrows,
                pSpinData->rctSpinField.right+2,
                pSpinData->rctSpinField.top-1,
                (WORD)GetSystemMetrics (SM_CXVSCROLL),
                pSpinData->rctSpinField.bottom+2,
                TRUE);

    /*------------------------------------------------------------------*/
    /* This extra invalidation is here so the arrows will be drawn      */
    /* correctly when the spin button is selected in the Dialog Editor. */
    /*------------------------------------------------------------------*/
    InvalidateRect (pSpinData->hwndArrows, NULL, FALSE);
  }

  hdc = BeginPaint (hwnd, (LPPAINTSTRUCT)&paint);

  if (hNullBrush = GetStockObject (NULL_BRUSH))
  {
     hBrushOld  = SelectObject (hdc, hNullBrush);
     hPenNew = CreatePen (PS_SOLID, 0, GetSysColor (COLOR_WINDOW));
     hPenOld = SelectObject (hdc, hPenNew);
     Rectangle(hdc,
               pSpinData->rctSpinField.left-1,
               pSpinData->rctSpinField.top-1,
               pSpinData->rctSpinField.right+2,
               pSpinData->rctSpinField.bottom+2);
     SelectObject (hdc, hBrushOld);
     SelectObject (hdc, hPenOld);
  }
  EndPaint (hwnd, (LPPAINTSTRUCT)&paint);

#if 0
  hdc = BeginPaint (hwnd, (LPPAINTSTRUCT)&paint);

  if ((GetWindowLong( pSpinData->hwndSpinField, GWL_STYLE )
        & WS_BORDER))
  {
    if ((hNullBrush = GetStockObject (NULL_BRUSH)) &&
        (hBlackPen = GetStockObject (BLACK_PEN)))                       /*WBB1*/
    {                                                                   /*WBB1*/
      hBrushOld  = SelectObject (hdc, hNullBrush);                      /*WBB1*/
      hPenOld    = SelectObject (hdc, hBlackPen);                       /*WBB1*/
      Rectangle(hdc,
                pSpinData->rctSpinField.left-1,
                pSpinData->rctSpinField.top-1,
                pSpinData->rctSpinField.right+2,
                pSpinData->rctSpinField.bottom+2);
      SelectObject (hdc, hBrushOld);
      SelectObject (hdc, hPenOld);
    }                                                                   /*WBB1*/
  }
  EndPaint (hwnd, (LPPAINTSTRUCT)&paint);
#endif

#if 0      /*-------------------------< MAB >--------------------------------*/
 /*-------------------------------------------------------------------*
  *  Clear the component window
  *-------------------------------------------------------------------*/

 /*-------------------------------------------------------------------*
  *  Set Background Mode to Transparent
  *-------------------------------------------------------------------*/
  oldBkMode = SetBkMode (hdc, TRANSPARENT);

// Get the current window background color ...
  hBrushNew = CreateSolidBrush (GetSysColor (COLOR_WINDOW));
  hBrushOld = SelectObject (hdc, hBrushNew);
  hPenNew = CreatePen (PS_SOLID, NULL, GetSysColor (COLOR_WINDOW));
  hPenOld = SelectObject (hdc, hPenNew);
  FillRect (hdc, (LPRECT)&paint.rcPaint, hBrushNew);
  DeleteObject (SelectObject (hdc, hPenOld));
  DeleteObject (SelectObject (hdc, hBrushOld));

 /*-------------------------------------------------------------------*
  *  Reset Background Mode to default
  *-------------------------------------------------------------------*/
  SetBkMode (hdc, oldBkMode);



  /*-------------------------------------------------------*/
  /* Make the arrows and the entryfied repaint themselves. */
  /*-------------------------------------------------------*/
  //Should only repaint affected areas ...
  InvalidateRect (pSpinData->hwndSpinField, NULL, FALSE);
  if (pSpinData->hwndArrows)
     InvalidateRect (pSpinData->hwndArrows, NULL, FALSE);
#endif     /*-------------------------< MAB >--------------------------------*/

} /* end SpinPaint */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinAdjustWindowPos                            */
/*                                                                    */
/*  Descriptive Name:  Spin Button Adjust Window Position             */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function calculates the sizes of the entryfield and the      */
/*  scrollbar.  It is called when the component is created, when      */
/*  certain styles change (such as margin for the entryfield),        */
/*  and if the component is resized.                                  */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*          hwnd - Window handle of the spin button                   */
/*          fResize - Resizing flag                                   */
/*          sWidth - New component width                              */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
BOOL SpinAdjustWindowPos (PSPINDATA pSpinData, HWND hwnd, UINT fResize,
                          int sWidth, int sHeight)
{
  int           SBWidth;           /* Temp spinbutton width           */
  int           ComponentWidth;    /* Width of the component window   */
  int           fieldWidth;        /* Width of the entryfield         */
//int           fieldHeight;       /* height of the entryfield        */
  DWORD           ulStyle;         /* Store window style temporarily  */
//TEXTMETRIC      tm;              /* Place to query text metrics     */
//HDC             hdc;             /* Temp hdc for text metrics query */
//HFONT           hOldFont;

 /*-------------------------------------------------------------------*
  *  If the resizing type is not normal or fullscreen, then don't
  *  process anything else ...
  *-------------------------------------------------------------------*/
  if ((fResize == SIZEICONIC) ||
      (fResize == SIZEZOOMHIDE) ||
      (fResize == SIZEZOOMSHOW))
  {
     return(TRUE);
  } /* end if */

  ulStyle = STYLEOF(hwnd);

 /*-------------------------------------------------------------------*
  * This is the width of an arrow
  *-------------------------------------------------------------------*/
  SBWidth = (WORD)GetSystemMetrics (SM_CXVSCROLL);

 /*-------------------------------------------------------------------*
  *  This is the width that the user specified in WCW
  *-------------------------------------------------------------------*/
  ComponentWidth = sWidth;

 /*-------------------------------------------------------------------*
  *  Ensure that the spinbutton is at least a certain width
  *-------------------------------------------------------------------*/
  if (ComponentWidth < SBWidth)         //@XXX
     ComponentWidth = SBWidth;

  fieldWidth = ComponentWidth;

 /*-------------------------------------------------------------------*
  *  Query the current font metrics
  *-------------------------------------------------------------------*/
#if 0
  hdc = GetDC (pSpinData->hwndSpinField);
  hOldFont = NULL;

  if(pSpinData->hSpbFont)
    hOldFont=SelectObject(hdc, pSpinData->hSpbFont);

  GetTextMetrics (hdc, &tm);

  if(hOldFont)
    SelectObject(hdc, hOldFont);

  ReleaseDC (pSpinData->hwndSpinField, hdc);
#endif
 /*-------------------------------------------------------------------*
  *  If we have a master, set up spinarrows & size data.
  *-------------------------------------------------------------------*/
  if (ulStyle & SPBS_MASTER)
     fieldWidth = ComponentWidth - SBWidth;

 /*-------------------------------------------------------------------*
  *  Resize & show the entryfield.
  *-------------------------------------------------------------------*/
#if 0
  if (pSpinData->ulStyleChange & WS_BORDER)
  {
     fieldHeight =((tm.tmHeight + tm.tmExternalLeading) +
                     (tm.tmHeight + tm.tmExternalLeading) / 2);
//                 ( ( (tm.tmHeight + tm.tmExternalLeading) * 3)/4));
  }
  else
  {
     fieldHeight = (tm.tmHeight + tm.tmExternalLeading);
  }

  if (fieldHeight > sHeight )
    return(FALSE);                                   //return error here
#endif

  pSpinData->rctSpinField.left  = 1;
  pSpinData->rctSpinField.top   = 1;
  pSpinData->rctSpinField.right = fieldWidth-2;
  pSpinData->rctSpinField.bottom = sHeight-2;
  MoveWindow (pSpinData->hwndSpinField,
              pSpinData->rctSpinField.left,
              pSpinData->rctSpinField.top,
              pSpinData->rctSpinField.right,
              pSpinData->rctSpinField.bottom,
              FALSE);

  return(TRUE);
} /* end SpinAdjustWindowPos */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetEditCtrlBrush                           */
/*                                                                    */
/*  Descriptive Name:  Spin button set edit control brush procedure   */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine handles processing of the WM_CTLCOLOR message        */
/*  for the edit control.  It returns a brush to the edit control     */
/*  to allow it to properly paint its background.                     */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hdcEditCtrl - Edit Control device context                 */
/*          pSpinData   - Pointer to internally stored data           */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
HBRUSH SpinSetEditCtrlBrush (HDC hdcEditCtrl, PSPINDATA pSpinData)          //@EVB
{
  HBRUSH  hBrush;

 /*-------------------------------------------------------------------*
  *  Use current background color as the brush color
  *  Return NULL if brush can't be created
  *-------------------------------------------------------------------*/
//if (!(hBrush = CreateSolidBrush (GetSysColor (COLOR_WINDOW))))        /*WBB1*/
  if (pSpinData->hBrush)                                                    //@EVB
     DeleteObject(pSpinData->hBrush);                                       //@EVB
  if (!(pSpinData->hBrush =                                                 //@EVB
         CreateSolidBrush (pSpinData->ulColorVal[CLR_SPBBACKGROUND])))      //@EVB
    return (NULL);                                                      /*WBB1*/

 /*-------------------------------------------------------------------*
  *  Set background and foreground color of spinbtn data
  *-------------------------------------------------------------------*/
  SetBkColor(hdcEditCtrl, pSpinData->ulColorVal[CLR_SPBBACKGROUND]);        //@EVB
  SetTextColor(hdcEditCtrl, pSpinData->ulColorVal[CLR_SPBFOREGROUND]);      //@EVB

 /*-------------------------------------------------------------------*
  *  Align origin of the brush with the edit ctrl window coordinates
  *-------------------------------------------------------------------*/
  UnrealizeObject (pSpinData->hBrush);                                      //@EVB

 /*-------------------------------------------------------------------*
  *  Set brush origin to the upper-left corner of the edit ctrl
  *-------------------------------------------------------------------*/
  SetBrushOrgEx (hdcEditCtrl, 0, 0, NULL);

  return(pSpinData->hBrush);                                                //@EVB
} /* end SpinSetEditCtrlBrush */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSearch                                     */
/*                                                                    */
/*  Descriptive Name:  Spin button binary Search                      */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function performs a binary case insensitive character array  */
/*  search.  If a match is found, the index in the arry where it was  */
/*  found will be returned.                                           */
/*                                                                    */
/*                                                                    */
/*  Notes:  0 is a valid index                                        */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*          lpszSearchText - Near pointer to search text              */
/*          fPartial - Partial match flag                             */
/*                     {TRUE = partial match   FALSE = full match}    */
/*                                                                    */
/*  Output:  Array index                                              */
/*                                                                    */
/*  Exit-Normal:  Index where match was found                         */
/*                                                                    */
/*  Exit-Error:  -1 to indicate no match                              */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
int SpinSearch (PSPINDATA pSpinData, LPSTR lpszSearchText, BOOL fPartial)
{
  BYTE npszTemp[CCHMAXEDIT + 1];
  WORD cchCompare,
       i,
       delta;

 /*-------------------------------------------------------------------*
  *  Uppercase the copy of the entryfield string and find it's
  *  string length.
  *-------------------------------------------------------------------*/
  if (i = lstrlen (lpszSearchText))                                      //@B06
    AnsiUpperBuff (lpszSearchText, i);                                  //@B06

  if (fPartial)
     cchCompare = lstrlen (lpszSearchText);

 /*-------------------------------------------------------------------*
  *  Loop through each of the array strings looking for a match
  *  We need to loop through the array starting from the current
  *  position to properly handle duplicates within the array.
  *-------------------------------------------------------------------*/
  for (delta=0; delta < pSpinData->cArrayItems; delta++)
  {
     i = (pSpinData->lCurrentValue + delta) % pSpinData->cArrayItems;
     lstrcpy (npszTemp, pSpinData->anpszText[i]);

     AnsiUpperBuff (npszTemp, sizeof(npszTemp));

    /*----------------------------------------------------------------*
     *  If we're only looking for a partial match, terminate the
     *  temp string at the maximum length we want to compare
     *----------------------------------------------------------------*/
     if (fPartial)
        npszTemp[cchCompare] = '\0';

    /*----------------------------------------------------------------*
     *  if we found a match, return the index of the matching
     *  string; otherwise, loop around for the next one
     *----------------------------------------------------------------*/
     if (!(lstrcmp (lpszSearchText, npszTemp)))
        return (i);
  } /* end for */

  return (-1);
} /* end SpinSearch */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     UpdateField                                    */
/*                                                                    */
/*  Descriptive Name:  Update spin button entryField                  */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function will update the the entryfield with the currently   */
/*  selected value.  It will check what type of data is in the        */
/*  component (e.g. Character/numeric)                                */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  {TRUE = Edit field updated    FALSE = No updates}   */
/*                                                                    */
/*  Exit-Error:  None                                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*              SpinSetArray                                          */
/*                                                                    */
/**********************************************************************/
BOOL UpdateField (HWND hwnd, PSPINDATA pSpinData)
{
  BYTE  Temp[CCHMAXLONG * 2];
  BYTE  npszEditText[CCHMAXEDIT + 1];

  GetWindowText (pSpinData->hwndSpinField, npszEditText, sizeof(npszEditText));

 /*-------------------------------------------------------------------*
  *  Check the type of data in the spinbutton.
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_ARRAYDATA)
  {
    /*----------------------------------------------------------------*
     *  Don't update if the update won't change the text
     *----------------------------------------------------------------*/
     if  (lstrcmp ((LPSTR)pSpinData->anpszText[pSpinData->lCurrentValue],
                 (LPSTR)npszEditText))
     {
        SetWindowText (pSpinData->hwndSpinField,
                       pSpinData->anpszText[pSpinData->lCurrentValue]);
        UpdateWindow(pSpinData->hwndSpinField);

        return (TRUE);
     } /* end if */
  }
  else
  {
    /*----------------------------------------------------------------*
     *  If the style is PAD_WITH_ZEROS, then format the output of the
     *  value with zeros padded on front.
     *----------------------------------------------------------------*/
     ltoa (hwnd, pSpinData, (LPSTR)Temp);

    /*----------------------------------------------------------------*
     *  Don't update if the update won't change the text
     *----------------------------------------------------------------*/
     if (lstrcmp ((LPSTR)Temp, (LPSTR)npszEditText))
     {
        SetWindowText (pSpinData->hwndSpinField, (LPSTR)Temp);
        UpdateWindow(pSpinData->hwndSpinField);

        return (TRUE);
     } /* end if */
  } /* end if */

  return (FALSE);

} /* end UpdateField */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetArray                                   */
/*                                                                    */
/*  Descriptive Name:  Set (Resize) character Array                   */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function is called each time a new array of data is          */
/*  requested.  It will resize the current segment so that the data   */
/*  can be tacked on to the end of the internal data.                 */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          alpszNew - Pointer to array of new values                 */
/*          cItems - Count of items in the array                      */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE                                                */
/*                                                                    */
/*  Exit-Error:  FALSE - Out of Memory                                */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSetArray (HWND hwnd, PSPINDATA pSpinData, LPPSTR alpszNew,
                   WORD cItems)
{
  int  i;
  WORD cch;
  NPPSTR  anpszText;

 /*-------------------------------------------------------------------*
  *  Allocate the array of string pointers
  *-------------------------------------------------------------------*/
  if (!(anpszText = (NPPSTR)AllocHeapMem (sizeof(NPSTR) * cItems)))
     goto Err;

 /*-------------------------------------------------------------------*
  *  Search through the array and add up how much space is needed to
  *  store it.
  *-------------------------------------------------------------------*/
  for (i=0; i < (int)cItems; i++)
  {
     cch = (lstrlen ((LPSTR)alpszNew[i]) + 1);   /* find the length of the string */

     if (cch > pSpinData->cchMax)            /* is this longer than the prev- */
        pSpinData->cchMax = cch;             /* yes, remember the new length  */

     if (!(anpszText[i] = AllocHeapMem (cch)))  /* alloc a buffer to copy to */
        goto ErrArray;

     lstrcpy ((LPSTR)anpszText[i], alpszNew[i]);    /* alloc succeeded, copy string  */
  } /* end for */

 /*-------------------------------------------------------------------*
  *  If we get here, everything was allocated alright, so we can
  *  free the old buffers
  *-------------------------------------------------------------------*/
  for (i=0; i < (int)pSpinData->cArrayItems; i++)
     FreeHeapMem ((PBYTE)pSpinData->anpszText[i]);

  if (pSpinData->anpszText)
     FreeHeapMem ((PBYTE)pSpinData->anpszText);

  pSpinData->anpszText = anpszText;

 /*-------------------------------------------------------------------*
  *  Make the upper limit equal to the number of items - 1 since 0
  *  is the first
  *-------------------------------------------------------------------*/
  pSpinData->lLowerLimit = 0;
  pSpinData->lUpperLimit = cItems - 1;
  pSpinData->cArrayItems = cItems;

 /*-------------------------------------------------------------------*
  *  If the current value is out of range, make it in range
  *-------------------------------------------------------------------*/
  AlignRange (pSpinData);

  pSpinData->fl |= SBF_DATAEXIST;
  pSpinData->fl |= SBF_ARRAYDATA;

 /*-------------------------------------------------------------------*
  *  HACK - Send message to owner that the user has
  *  changed the edit field.  (Not receiving EN_CHANGE
  *  from Windows to trigger)
  *-------------------------------------------------------------------*/
  if (UpdateField (hwnd, pSpinData))
     SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);   //@XXX

  return ((long)TRUE);

ErrArray:
  do
  {
     FreeHeapMem ((PBYTE)anpszText[i]);
  } while (--i >= 0);

  FreeHeapMem ((PBYTE)anpszText);

Err:
  return ((long)FALSE);
} /* end SpinSetArray */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinDestroy                                    */
/*                                                                    */
/*  Descriptive Name:  Destroy the Spin button window                 */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function processes the WM_DESTROY message.  It stops         */
/*  the timer (if in use), destroys the entryfield and the            */
/*  arrows (bitmap) window, and frees the heap.                       */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
VOID SpinDestroy (HWND hwnd, PSPINDATA pSpinData)
{

 /*-------------------------------------------------------------------*
  *  If timer is in use, stop it.
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_TIMER)
     KillTimer (hwnd, pSpinData->idTimer);

 /*-------------------------------------------------------------------*
  *  Delete bitmap objects
  *-------------------------------------------------------------------*/
  if (pSpinData->hwndArrows)
  {
     DeleteObject (pSpinData->hbmUp);
     DeleteObject (pSpinData->hbmDown);
     DeleteObject (pSpinData->hbmUpDisabled);                           //$L02
     DeleteObject (pSpinData->hbmDownDisabled);                         //$L02
  } /* endif */

 /*-------------------------------------------------------------------*
  *  Delete background brush
  *-------------------------------------------------------------------*/
  if (pSpinData->hBrush != 0)                                               //@EVB
     DeleteObject (pSpinData->hBrush);                                      //@EVB

 /*-------------------------------------------------------------------*
  *  Undo subclassing
  *-------------------------------------------------------------------*/
  SetWindowLong (pSpinData->hwndSpinField,
                 GWL_WNDPROC,
                 (LONG)(FARPROC)pfnOldWindowProc);

 /*-------------------------------------------------------------------*
  *  Destroy Entryfield - important because of subclassing!
  *    The entryfield uses the static data.  It must be destroyed
  *    before freeing the segment.
  *-------------------------------------------------------------------*/
  DestroyWindow (pSpinData->hwndSpinField);
  if (pSpinData->hwndArrows)
     DestroyWindow (pSpinData->hwndArrows);

  FreeHeapMem ((PBYTE)pSpinData);
} /* end SpinDestroy */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinChar                                       */
/*                                                                    */
/*  Descriptive Name:  Spin button Character processing               */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function processes the WM_CHAR message.  It handles          */
/*  processing for the directional keys (e.g. up, down, etc ...).     */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          msg  - Message (WM_KEYDOWN/WM_KEYUP)                      */
/*          wParam - Message parameter 1 (virtual key)                */
/*          lParam - Message parameter 2 (other info)                 */
/*          pSpinData - Pointer to internally stored data             */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE - Message Processed                            */
/*                FALSE - Not processed                               */
/*                                                                    */
/*  Exit-Error:  None                                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinSubclassEntryfield                                */
/*//Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinChar (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam,
               PSPINDATA pSpinData)
{

  switch (wParam)
  {
    /*----------------------------------------------------------------*
     *  Check to see if up arrow key or page up key has been
     *  pressed.  If so, spin up.
     *----------------------------------------------------------------*/
     case VK_UP:
     case VK_PRIOR:
        SendSpinKeybrdMsgs (pSpinData,
                            hwnd,
                            SPBM_SPINUP,
                            SPBN_UPARROW,
                            (msg == WM_KEYUP));   /* Key pressed/released */

        return ((long)TRUE);
        break;

    /*----------------------------------------------------------------*
     *  Check to see if down arrow key or page down key has been
     *  pressed.  If so, spin down.
     *----------------------------------------------------------------*/
     case VK_DOWN:
     case VK_NEXT:
        SendSpinKeybrdMsgs (pSpinData,
                            hwnd,
                            SPBM_SPINDOWN,
                            SPBN_DOWNARROW,
                            (msg == WM_KEYUP));   /* Key pressed/released */

        return ((long)TRUE);
        break;

     default:
        break;
  } /* end switch */

  return ((long)FALSE);
} /* end SpinChar */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinControl                                    */
/*                                                                    */
/*  Descriptive Name:  Spin button Control (entryfield) processing    */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function processes the WM_COMMAND message.  It handles       */
/*  processing for the spin button entryfield.                        */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          idSender - Window ID of the control sender                */
/*          usNotifyCode - Entryfield processing notification code    */
/*          hwndSelected - Entryfield currently selected              */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
VOID SpinControl (HWND hwnd, PSPINDATA pSpinData, WORD idSender,
                  WORD usNotifyCode, HWND hwndSelected)
{
  UINT idSelf = IDOF(hwnd);

 /*-------------------------------------------------------------------*
  *  Check to see if the message came from the entryfield.
  *-------------------------------------------------------------------*/
  if (idSender == ID_EDIT)                                              //@L03
  {
    /*----------------------------------------------------------------*
     * If the entryfield just got focus, perform focus processing
     *----------------------------------------------------------------*/
     if (usNotifyCode == EN_SETFOCUS)
     {
       /*-------------------------------------------------------------*
        * Send a message to the owner that this spinbutton was selected
        *-------------------------------------------------------------*/
        SENDCONTROL (PARENTOF(hwnd), idSelf, SPBN_SETFOCUS, hwnd);

       /*-------------------------------------------------------------*
        *  Send a message to the master component that this
        *  component was selected.
        *-------------------------------------------------------------*/
        SENDCONTROL (pSpinData->hwndMaster, idSelf, SPBN_SETFOCUS, hwnd);
     } /* end if */

    /*----------------------------------------------------------------*
     *  If the entryfield has changed, then send a change message
     *  to the owner.
     *----------------------------------------------------------------*/
     if (usNotifyCode == EN_CHANGE)
        SENDCONTROL (PARENTOF(hwnd), idSelf, SPBN_CHANGE, hwnd);

    /*----------------------------------------------------------------*
     *  Do focus loss processing
     *----------------------------------------------------------------*/
     if (usNotifyCode == EN_KILLFOCUS)
     {
       /*-------------------------------------------------------------*
        *  Send a message to the owner that says this component
        *  lost focus
        *-------------------------------------------------------------*/
        SENDCONTROL (PARENTOF(hwnd), idSelf, SPBN_KILLFOCUS, hwnd);

       /*-------------------------------------------------------------*
        *  Send a message to the master that says this component
        *  lost focus
        *-------------------------------------------------------------*/
        SENDCONTROL (pSpinData->hwndMaster, idSelf, SPBN_KILLFOCUS, hwnd);

       /*-------------------------------------------------------------*
        *  If a timer is in use, turn off fast spinning.
        *  (i.e. If focus changes during a fast spin, turn it
        *  off so that when focus returns, it won't still be
        *  spinning.)
        *-------------------------------------------------------------*/
        if (pSpinData->fl & SBF_TIMER)
           FastSpinOff (hwnd, pSpinData, TID_SECONDARY);
     } /* end if */
  } /* end if */

 /*-------------------------------------------------------------------*
  *  If the spinfield didn't send the message and the
  *  spinarrow didn't send it, then another component must
  *  have sent it.
  *-------------------------------------------------------------------*/
  if (usNotifyCode == SPBN_SETFOCUS)
  {
     pSpinData->hwndCurrentlySelected = hwndSelected;
     FastSpinOff (hwnd, pSpinData, TID_SECONDARY);
  } /* end if */

  if (usNotifyCode == SPBN_KILLFOCUS)
  {
     pSpinData->hwndCurrentlySelected = NULL;
     FastSpinOff (hwnd, pSpinData, TID_SECONDARY);
  } /* end if */
} /* end SpinControl */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetMaster                                  */
/*                                                                    */
/*  Descriptive Name:  Spin button Set Master component               */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function determines who will control the spinning: the       */
/*  master component or the app.  The owner control bit is set        */
/*  in the status flags to indicate who has control.                  */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*          hwndMaster - Window handle of the master component        */
/*                                                                    */
/*  Output:  Always TRUE                                              */
/*                                                                    */
/*  Exit-Normal:  TRUE                                                */
/*                                                                    */
/*  Exit-Error:  None                                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSetMaster (PSPINDATA pSpinData, HWND hwndMaster)
{

  if (hwndMaster)
  {
    /*----------------------------------------------------------------*
     *  The spinbutton will control the spinning - set link to
     *  the master
     *----------------------------------------------------------------*/
     pSpinData->hwndMaster = hwndMaster;

    /*----------------------------------------------------------------*
     *  NULL the owner control bit
     *----------------------------------------------------------------*/
     pSpinData->fl &= (~SBF_OWNERCONTROL);
  }
  else
  {
    /*----------------------------------------------------------------*
     *  The spinbutton will not control the spinning - the app will
     *  set the owner control bit
     *----------------------------------------------------------------*/
     pSpinData->fl |= SBF_OWNERCONTROL;
  } /* end if */

  return ((long)TRUE);
} /* end SpinSetMaster */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetTextLimit                               */
/*                                                                    */
/*  Descriptive Name:  Spin button Set Text Limit                     */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function sets the text limit for the entryfield, and         */
/*  redisplays the text array to its new limit.                       */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          usTextLimit - New text limit for the entryfield           */
/*                                                                    */
/*  Output:  rc from EM_LIMITTEXT/FALSE                               */
/*                                                                    */
/*  Exit-Normal:  rc from EM_LIMITTEXT                                */
/*                                                                    */
/*  Exit-Error:  FALSE                                                */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSetTextLimit (HWND hwnd, PSPINDATA pSpinData, WORD usTextLimit)
{
  long mr;
  int   i;
  BYTE    npszEditText[CCHMAXEDIT + 1];

 /*-------------------------------------------------------------------*
  *  Don't allow the text limit to be set greater than the default
  *-------------------------------------------------------------------*/
  if (usTextLimit > CCHMAXEDIT)
     return ((long)FALSE);

 /*-------------------------------------------------------------------*
  *  Ensure that the Numeric format size is 11 characters or less.
  *-------------------------------------------------------------------*/
  pSpinData->usNumericFormatSize = (usTextLimit > CCHMAXLONG) ? CCHMAXLONG :
                                                                usTextLimit;

  mr = SENDMESSAGE (pSpinData->hwndSpinField,
                    EM_LIMITTEXT,
                    (WORD)usTextLimit,
                    0L);

 /*-------------------------------------------------------------------*
  *  For dynamic updating
  *-------------------------------------------------------------------*/
  if ((pSpinData->fl & SBF_ARRAYDATA) &&
      (usTextLimit > pSpinData->usTextLimit))
  {
     GetWindowText (pSpinData->hwndSpinField,
                    npszEditText,
                    sizeof(npszEditText));

    /*----------------------------------------------------------------*
     *  If no match is found then the return value will be -1.
     *  If a match is found then update the spinfield.
     *----------------------------------------------------------------*/
     if ((i = SpinSearch (pSpinData, (LPSTR)npszEditText, TRUE)) != -1)
     {
        pSpinData->lCurrentValue = i;

       /*-------------------------------------------------------------*
        *  HACK - Send message to owner that the user has
        *  changed the edit field.  (Not receiving EN_CHANGE
        *  from Windows to trigger)
        *-------------------------------------------------------------*/
        if (UpdateField (hwnd, pSpinData))
           SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);  //@XXX
     } /* end if */
  } /* end if */

  pSpinData->usTextLimit = usTextLimit;
  return (mr);
} /* end SpinSetTextLimit */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetCurrentValue                            */
/*                                                                    */
/*  Descriptive Name:  Spin button Set Current Value                  */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function sets a new value (or text) in the spin button       */
/*  entryfield as the current (visible) value.                        */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          lNewValue - New value (numeric) or index (array)          */
/*                      for the spin button entryfield.               */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE                                                */
/*                                                                    */
/*  Exit-Error:  FALSE                                                */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSetCurrentValue (HWND hwnd, PSPINDATA pSpinData, LONG lNewValue)
{

 /*-------------------------------------------------------------------*
  *  Ensure that the spinbutton has been initialized and isn't in
  *  a default state.
  *-------------------------------------------------------------------*/
  if (!(pSpinData->fl & SBF_DATAEXIST))
     return (FALSE);

 /*-------------------------------------------------------------------*
  *  Check to see what kind of data is in the spinbutton.
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_ARRAYDATA)
  {
    /*----------------------------------------------------------------*
     *  Current value must be between 0 and Upper limit inclusive
     *----------------------------------------------------------------*/
     if ((lNewValue < 0) || (lNewValue > pSpinData->lUpperLimit))
        return (FALSE);
  }
  else
  {
    /*----------------------------------------------------------------*
     *  Current value must be between lower limit an upper
     *  limit inclusive.
     *----------------------------------------------------------------*/
     if ((lNewValue < pSpinData->lLowerLimit) ||
         (lNewValue > pSpinData->lUpperLimit))
     {
        return (FALSE);
     } /* end if */
  } /* end if */

  pSpinData->lCurrentValue = lNewValue;

 /*-------------------------------------------------------------------*
  *  Now that the current value is reset, update the entryfield
  *  to display the proper entry.
  *-------------------------------------------------------------------*/

 /*-------------------------------------------------------------------*
  *  HACK - Send message to owner that the user has
  *  changed the edit field.  (Not receiving EN_CHANGE
  *  from Windows to trigger)
  *-------------------------------------------------------------------*/
  if (UpdateField (hwnd, pSpinData))
     SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);   //@XXX

  return ((long)TRUE);
} /* end SpinSetCurrentValue */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSetLimits                                  */
/*                                                                    */
/*  Descriptive Name:  Spin button Set Limits                         */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function sets the upper and lower limits for the spin        */
/*  buttons.  It will override the current limits, if the user        */
/*  so desired.                                                       */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          lUpperLimit - New upper limit                             */
/*          lLowerLimit - New lower limit                             */
/*          fOverride   - Override the current limits                 */
/*                        {0 = SBM_OVERRIDESETLIMITS                  */
/*                         Nonzero = SBM_SETLIMITS}                   */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE                                                */
/*                                                                    */
/*  Exit-Error:  FALSE                                                */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSetLimits (HWND hwnd, PSPINDATA pSpinData, LONG lUpperLimit,
                    LONG lLowerLimit, BOOL fOverride)
{
  LONG  lQueryData;
  LPSTR lpszEndStr;          /* Ptr to char. Used in Str2Long# */
  BYTE  npszEditText[CCHMAXEDIT + 1];

 /*-------------------------------------------------------------------*
  *  Ensure the upper limit is greater than the lower limit
  *-------------------------------------------------------------------*/
  if (lUpperLimit >= lLowerLimit)
  {
     pSpinData->lLowerLimit = lLowerLimit;
     pSpinData->lUpperLimit = lUpperLimit;
     pSpinData->fl |= SBF_DATAEXIST;
     pSpinData->fl &= (~SBF_ARRAYDATA);

     if (!fOverride)
     {
       /*-------------------------------------------------------------*
        *  Get the content in the spinfield
        *-------------------------------------------------------------*/
        GetWindowText (pSpinData->hwndSpinField,
                       npszEditText,
                       sizeof(npszEditText));

       /*-------------------------------------------------------------*
        *  If we get to this section of code, the spinbutton is
        *  spinning numeric values between an upper and lower
        *  limit.
        *-------------------------------------------------------------*/
        lQueryData = strtol (npszEditText, &lpszEndStr);

       /*-------------------------------------------------------------*
        *  Check that the conversion from a string to an integer
        *  stopped because of a NULL character and that the
        *  length of the string is not zero.
        *  If it's a numeric number then assign it to the
        *  current value.
        *-------------------------------------------------------------*/
        if (!*lpszEndStr && lstrlen (npszEditText) > 0)
           pSpinData->lCurrentValue = lQueryData;

       /*-------------------------------------------------------------*
        *  If out of range, make it in range
        *-------------------------------------------------------------*/
        AlignRange (pSpinData);

       /*-------------------------------------------------------------*
        *  HACK - Send message to owner that the user has
        *  changed the edit field.  (Not receiving EN_CHANGE
        *  from Windows to trigger)
        *-------------------------------------------------------------*/
        if (UpdateField (hwnd, pSpinData))
           SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);   //@XXX
     } /* end if */

     return ((long)TRUE);
  } /* end if */

  return ((long)FALSE);
} /* end SpinSetLimits */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSpinValue                                  */
/*                                                                    */
/*  Descriptive Name:  Spin button Spin to Value                      */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function spins the value in the entryfield up or down        */
/*  n numbers of items (if possible). {n is the increment parameter}  */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          ulIncrement - Increment for spinning distance             */
/*          fUp         - Indicates whether to spin up or down        */
/*                        {0 = SPBM_SPINUP                            */
/*                         Nonzero = SPBM_SPINDOWN}                   */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE                                                */
/*                                                                    */
/*  Exit-Error:  FALSE                                                */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinSpinValue (HWND hwnd, PSPINDATA pSpinData, DWORD ulIncrement,
                    BOOL fUp)
{
  DWORD   lNumToTop;
  BYTE    npszEditText[CCHMAXEDIT + 1];
  LONG    lQueryData;
  LPSTR   lpszEndStr;

 /*-------------------------------------------------------------------*
  *  Ensure that the spinbutton has been initialized.  Can't spin
  *  up if there is nothing to spin
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_DATAEXIST)
  {
    /*----------------------------------------------------------------*
     *  If current value is out of range, make it in range
     *----------------------------------------------------------------*/
     AlignRange (pSpinData);

     if (!(pSpinData->fl & SBF_ARRAYDATA))
     {
        GetWindowText (pSpinData->hwndSpinField,
                       npszEditText,
                       sizeof(npszEditText));

        lQueryData = strtol (npszEditText, &lpszEndStr);
        if ((lpszEndStr[0] == '\0') && (lstrlen (npszEditText) > 0))   //@OS2
        {
           if ((lQueryData > pSpinData->lUpperLimit)  ||
               (lQueryData < pSpinData->lLowerLimit))
           {
              pSpinData->lCurrentValue = (lQueryData > pSpinData->lUpperLimit) ?
                  pSpinData->lUpperLimit : pSpinData->lLowerLimit;

             /*-------------------------------------------------------*
              *  HACK - Send message to owner that the user has
              *  changed the edit field.  (Not receiving EN_CHANGE
              *  from Windows to trigger)
              *-------------------------------------------------------*/ //@XXX
              if (UpdateField (hwnd, pSpinData))
                 SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);

              return ((long)TRUE);
           } /* end if */
        } /* end if */
     } /* end if */

    /*----------------------------------------------------------------*
     *  Disable default processing if master handle is NULL
     *----------------------------------------------------------------*/
     if (pSpinData->hwndMaster != NULL)
     {
        SENDMESSAGE( hwnd, SPBM_QUERYVALUE, (WPARAM)0L,                 /*DAB1*/
                     (LPARAM)MAKELONG( 0, SPBQ_ALWAYSUPDATE ) );        /*DAB1*/
     } /* end if */

     lNumToTop = pSpinData->lUpperLimit - pSpinData->lLowerLimit + 1;

    /*----------------------------------------------------------------*
     *  This is a little check to be sure that you don't spin up
     *  15 times if there are only 10 #'s in the list.  You might
     *  as well spin up 5
     *----------------------------------------------------------------*/
     ulIncrement -= lNumToTop * (ulIncrement / lNumToTop);

     if (fUp)
     {
        if ((pSpinData->lUpperLimit - pSpinData->lCurrentValue)
            < (LONG)ulIncrement)
        {
           ulIncrement -= pSpinData->lUpperLimit - pSpinData->lCurrentValue + 1;
           pSpinData->lCurrentValue = pSpinData->lLowerLimit;
        } /* end if */
        pSpinData->lCurrentValue += ulIncrement;
     }
     else
     {
        if ((pSpinData->lCurrentValue - pSpinData->lLowerLimit)
            < (LONG)ulIncrement)
        {
           ulIncrement -= pSpinData->lCurrentValue - pSpinData->lLowerLimit + 1;
           pSpinData->lCurrentValue = pSpinData->lUpperLimit;
        } /* end if */
        pSpinData->lCurrentValue -= ulIncrement;
     } /* end if */

    /*----------------------------------------------------------------*
     *  HACK - Send message to owner that the user has
     *  changed the edit field.  (Not receiving EN_CHANGE
     *  from Windows to trigger)
     *----------------------------------------------------------------*/
     if (UpdateField (hwnd, pSpinData))
        SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);   //@XXX

     return ((long)TRUE);
  } /* end if */

  return ((long)FALSE);
} /* end SpinSpinValue */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinQueryValue                                 */
/*                                                                    */
/*  Descriptive Name:  Spin button Query Value                        */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function searches for the value entered in the entryfield    */
/*  in either the numerical range or the text array.  If found        */
/*  it will update the field and current value settings if the        */
/*  update the field and current value settings if the update flag    */
/*  is set.                                                           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          pBuf - Pointer to return value buffer                     */
/*          lOther - Other query info:                                */
/*                   {lo = buffer size   hi = update flag}            */
/*                                                                    */
/*  Output:  Pointer to queried value                                 */
/*                                                                    */
/*  Exit-Normal:  TRUE - successful                                   */
/*                                                                    */
/*  Exit-Error:  FALSE - unsuccessful                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
long SpinQueryValue (HWND hwnd, PSPINDATA pSpinData, LPVOID pBuf, DWORD lOther)
{
  LONG  lQueryData;             /* Query variable                  */
  LPSTR lpszEndStr;             /* Ptr to char. Used in Str2Long#  */
  BYTE  npszEditText[CCHMAXEDIT + 1];
  BYTE  Temp[CCHMAXLONG * 2];                                           //@B07

 /*-------------------------------------------------------------------*
  *  Check and make sure that there is data in the spinbutton.
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_DATAEXIST)
  {
     GetWindowText (pSpinData->hwndSpinField,
                    npszEditText,
                    sizeof(npszEditText));

    /*----------------------------------------------------------------*
     *  Check if array or numeric limits
     *----------------------------------------------------------------*/
     if (pSpinData->fl & SBF_ARRAYDATA)
     {
        lQueryData = SpinSearch (pSpinData, (LPSTR)npszEditText, FALSE);

       /*-------------------------------------------------------------*
        *  If a return of -1, the entryfield contained invalid data
        *-------------------------------------------------------------*/
        if (lQueryData == -1)  /* if it isn't valid */
        {

          /*----------------------------------------------------------*
           *  Update the field if SPBQ_ALWAYSUPDATE style
           *----------------------------------------------------------*/
           if (HIWORD(lOther) == SPBQ_ALWAYSUPDATE)
           {
             /*-------------------------------------------------------*
              *  HACK - Send message to owner that the user has
              *  changed the edit field.  (Not receiving EN_CHANGE
              *  from Windows to trigger)
              *-------------------------------------------------------*/  //@XXX
              if (UpdateField (hwnd, pSpinData))
                 SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);
           } /* end if */

          /*----------------------------------------------------------*
           *  Ensure that pBuf is not NULL
           *----------------------------------------------------------*/
           PBUFNOTNULL(pSpinData->lCurrentValue);
           return ((long)FALSE);
        }
        else
        {
          /*----------------------------------------------------------*
           *  If we get to this part of the code, the spinbutton
           *  contains a string that is not in the internal list.
           *----------------------------------------------------------*/
           pSpinData->lCurrentValue = lQueryData;

           if ((HIWORD(lOther) == SPBQ_UPDATEIFVALID)  ||
               (HIWORD(lOther) == SPBQ_ALWAYSUPDATE))
           {
             /*-------------------------------------------------------*
              *  HACK - Send message to owner that the user has
              *  changed the edit field.  (Not receiving EN_CHANGE
              *  from Windows to trigger)
              *-------------------------------------------------------*/  //@XXX
              if (UpdateField (hwnd, pSpinData))
                 SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);
           } /* end if */

          /*----------------------------------------------------------*
           *  Check that pBuf is not NULL
           *----------------------------------------------------------*/
           PBUFNOTNULL(pSpinData->lCurrentValue);
           return ((long)TRUE);
        } /* end if */
     }
     else
     {
       /*-------------------------------------------------------------*
        *  If we get to this section of code, the spinbutton is
        *  spinning numeric values between an upper and lower
        *  limit.
        *-------------------------------------------------------------*/
        lQueryData = strtol (npszEditText, &lpszEndStr);

       /*-------------------------------------------------------------*
        *  Check that the conversion from a string to an integer
        *  stopped because of a NULL character and that the
        *  length of the string is not zero.
        *-------------------------------------------------------------*/
        if ((lpszEndStr[0] == '\0')  &&
            (lstrlen (npszEditText) > 0))
        {

          /*----------------------------------------------------------*
           *  We have a valid number if here
           *----------------------------------------------------------*/
           if ((lQueryData >= pSpinData->lLowerLimit) &&
               (lQueryData <= pSpinData->lUpperLimit))
           {

             /*-------------------------------------------------------* //->@L03
              *  We have a situation where the current spin field
              *  contains 00, 000, 0000, 00000, etc ... This situation
              *  occurs (for example) in the following scenario:
              *      - Spin Field contains 100
              *      - Select the 1 with the mouse
              *      - Type in a 2
              *-------------------------------------------------------*/
//            if ((lQueryData == 0) && (lstrlen (npszEditText) > 1))     //->@B07
//            {
//              return ((long)TRUE);
//            }                                                         //@L03<-

             /*-------------------------------------------------------*
              *  We have a situation where the current spin field
              *  contains 0X, 0XX, 0XXX, 0XXXX, etc ... This situation
              *  occurs (for example) in the following scenario:
              *      - Spin Field contains 102
              *      - Select the 1 with the mouse
              *      - Type in a 2
              *  To prevent the 0 from being lost perform check:
              *-------------------------------------------------------*/
              if ((HIWORD(lOther) == SPBQ_UPDATEIFVALID) ||
                  (HIWORD(lOther) == SPBQ_ALWAYSUPDATE))
              {
                pSpinData->lPreviousValue = pSpinData->lCurrentValue;
                pSpinData->lCurrentValue = lQueryData;
                ltoa (hwnd, pSpinData, (LPSTR)Temp);
                if ((lstrcmp ((LPSTR)Temp, (LPSTR)npszEditText)) &&
                    (pSpinData->fl &  SBF_DELETEORREPLACE)      &&
                    (!(STYLEOF(hwnd) & SPBS_PADWITHZEROS)))
                {
                  pSpinData->lCurrentValue = pSpinData->lPreviousValue;
                  pSpinData->fl &= ~SBF_DELETEORREPLACE;
                  return ((long)TRUE);
                }

               /*-----------------------------------------------------*
                *  Update previous spin value if not select/replace
                *  or delete operation
                *-----------------------------------------------------*/
                if (!(pSpinData->fl &  SBF_DELETEORREPLACE))
                  pSpinData->lPreviousValue = pSpinData->lCurrentValue;

               /*-----------------------------------------------------*
                *  If current value == spin window value and delete
                *  or select/replace flag set, reset flag to allow
                *  future numeric updates
                *-----------------------------------------------------*/
                if (!(lstrcmp ((LPSTR)Temp, (LPSTR)npszEditText)) &&
                    (pSpinData->fl &  SBF_DELETEORREPLACE))
                  pSpinData->fl &= ~SBF_DELETEORREPLACE;

               /*-----------------------------------------------------*
                *  We have a valid number that is in range if here
                *-----------------------------------------------------*/

                /*----------------------------------------------------*
                 *  HACK - Send message to owner that the user has
                 *  changed the edit field.  (Not receiving EN_CHANGE
                 *  from Windows to trigger)
                 *----------------------------------------------------*/
                 if (UpdateField (hwnd, pSpinData))                      //@XXX
                    SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);
              }
              else
              {
                pSpinData->lCurrentValue = lQueryData;
              } /* end if */                                            //@B07<-

             /*-------------------------------------------------------*
              *  Ensure that the address to store the data is not NULL.
              *  If the buffer length is not null, the address in pBuf
              *  is assumed to be the address of string variable of
              *  size buffer length.
              *
              *  If pBuf is the address of a LONG variable, the current
              *  value is returned in it.  If it is the address of a
              *  string variable, the current value is returned in
              *  string format.
              *-------------------------------------------------------*/
              PBUFNOTNULL(lQueryData);
              return ((long)TRUE);
           }
           else
           {
             /*-------------------------------------------------------*
              *  We have a valid number that is out of range
              *
              *  Update the entryfield if the spinbutton is of
              *  ALWAYSUPDATE
              *-------------------------------------------------------*/
              if (HIWORD(lOther) == SPBQ_ALWAYSUPDATE)
              {
                /*----------------------------------------------------*
                 *  Reset current spin value to previous value to
                 *  preserve integrity of spin window value
                 *----------------------------------------------------*/
                 pSpinData->lCurrentValue = pSpinData->lPreviousValue;  //@B07

                /*----------------------------------------------------*
                 *  HACK - Send message to owner that the user has
                 *  changed the edit field.  (Not receiving EN_CHANGE
                 *  from Windows to trigger)
                 *----------------------------------------------------*/
                 if (UpdateField (hwnd, pSpinData))                     //@XXX
                    SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);

                 lQueryData = pSpinData->lCurrentValue;
              } /* end if */

              PBUFNOTNULL(lQueryData);
              return ((long)FALSE);
           } /* end if */
        }
        else
        {
           if ((HIWORD(lOther) == SPBQ_ALWAYSUPDATE) &&                 //@B05
               (lstrlen (npszEditText) > 0))
           {
             /*-------------------------------------------------------*
              *  HACK - Send message to owner that the user has
              *  changed the edit field.  (Not receiving EN_CHANGE
              *  from Windows to trigger)
              *-------------------------------------------------------*/
              if (UpdateField (hwnd, pSpinData))                      //@XXX
                 SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_CHANGE, hwnd);
           } /* end if */

          /*----------------------------------------------------------*
           *  If the buffer length is NULL, the address in pBuf
           *  is assumed to be the address of a LONG variable.
           *  If the buffer length is not null, the address
           *  in pBuf is assumed to be the address of a string
           *  variable of size buffer lenght.
           *
           *  If pBuf is the address of a LONG variable, the
           *  current value is returned in it.  If it is
           *  the address of a string variable, the contents
           *  of the entryfield is returned.
           *----------------------------------------------------------*/
           PBUFNOTNULL(pSpinData->lCurrentValue);
           return ((long)FALSE);
        } /* end if */
     } /* end if */
  } /* end if */
  return 0;
} /* end SpinQueryValue */

/**********************************************************************/
/* Procedure: SpinSetPresParam - This routine will update the         */
/*            presentation parameter which has been modified for the  */
/*            spinbtn                                                 */
/*                                                                    */
/* Inputs:    pSpinData   - Pointer to the spinbtn control data block */
/*            ulPp        - presentation parameter attribute identity */
/*            cr          - rgb value to set for identified attribute */
/**********************************************************************/
BOOL SpinSetPresParam (PSPINDATA pSpinData, WORD wColorToChange,            //@EVB
                       COLORREF crColor )                                   //@EVB
{

   /*******************************************************************/
   /* Check if the foreground color has been changed                  */
   /*******************************************************************/
   if ( wColorToChange == SPBA_FOREGROUNDCOLOR )
   {
     pSpinData->ulColorVal[CLR_SPBFOREGROUND] = crColor;
     return (TRUE);
   }
   /*******************************************************************/
   /* Check if the background color has been changed                  */
   /*******************************************************************/
   else if ( wColorToChange == SPBA_BACKGROUNDCOLOR )
   {
     pSpinData->ulColorVal[CLR_SPBBACKGROUND] = crColor;
     return (TRUE);
   }

   return (FALSE);
}

/**********************************************************************/
/* Procedure: SpinResetPresParam - This routine will update the       */
/*            presentation parameter which has been modified for the  */
/*            spinbtn                                                 */
/*                                                                    */
/* Inputs:    pSpinData   - Pointer to the spinbtn control data block */
/*            ulPp        - presentation parameter attribute identity */
/**********************************************************************/
BOOL SpinResetPresParam (PSPINDATA pSpinData, WORD wColorToChange )         //@EVB
{

   /*******************************************************************/
   /* Check if the foreground color has been changed                  */
   /*******************************************************************/
   if ( wColorToChange == SPBA_FOREGROUNDCOLOR )
   {
     pSpinData->ulColorVal[CLR_SPBFOREGROUND] =
           GetSysColor(COLOR_WINDOWTEXT);

     return (TRUE);
   }
   /*******************************************************************/
   /* Check if the background color has been changed                  */
   /*******************************************************************/
   else if ( wColorToChange == SPBA_BACKGROUNDCOLOR )
   {
     pSpinData->ulColorVal[CLR_SPBBACKGROUND] =
           GetSysColor(COLOR_WINDOW);

     return (TRUE);
   }

   return (FALSE);
}

/**********************************************************************/
/* Procedure: SpinQueryPresParam - This routine will return the       */
/*            presentation parameter which currently exists for the   */
/*            spinbtn                                                 */
/*                                                                    */
/* Inputs:    pSpinData   - Pointer to the spinbtn control data block */
/*            ulPp        - presentation parameter attribute identity */
/**********************************************************************/
LONG SpinQueryPresParam (PSPINDATA pSpinData, USHORT wColorToReturn )       //@EVB
{
   /*******************************************************************/
   /* Return the current foreground color                             */
   /*******************************************************************/
   if ( wColorToReturn == SPBA_FOREGROUNDCOLOR )
   {
     return (long)(pSpinData->ulColorVal[CLR_SPBFOREGROUND]);
   }
   /*******************************************************************/
   /* Return the current background color                             */
   /*******************************************************************/
   else if ( wColorToReturn == SPBA_BACKGROUNDCOLOR )
   {
     return (long)(pSpinData->ulColorVal[CLR_SPBBACKGROUND]);
   }
   else
     return (0);
}


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SendSpinKeybrdMsgs                             */
/*                                                                    */
/*  Descriptive Name:  Send Spin Button Keyboard Messages             */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to send messages which were caused by usage  */
/*  of the keyboard to the spin button control.                       */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*          hwnd - Window handle of the spin button                   */
/*          SpinMsg - Spin button msg to either spin up or down       */
/*          SpinNotify - Spin notification that the up arrow key was  */
/*                       pressed.  Also used in the same manner for   */
/*                       down arrow.                                  */
/*          bReleased - Indicates whether key was pressed or released */
/*                      {TRUE = Released   FALSE = Pressed}           */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
VOID SendSpinKeybrdMsgs (PSPINDATA pSpinData, HWND hwnd, WORD SpinMsg,
                         WORD SpinNotify, BOOL bReleased)
{
//register int loop;

  if (bReleased)
  {
    /*----------------------------------------------------------------*
     *  Send Message to owner that the spin has ended since a
     *  key was released.
     *----------------------------------------------------------------*/
     SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SPBN_ENDSPIN, hwnd);

    /*----------------------------------------------------------------*
     *  Attempt to turn off fast spinning
     *----------------------------------------------------------------*/
     FastSpinOff (hwnd, pSpinData, TID_SECONDARY);
  }
  else
  {
    /*----------------------------------------------------------------*
     *  Check to see that a master handle is present.  If no
     *  master handle, then do not spin up by yourself.
     *----------------------------------------------------------------*/
     if (!(pSpinData->fl & SBF_OWNERCONTROL))
     {
        SENDMESSAGE( hwnd, SpinMsg,
                     (LPARAM)pSpinData->ulWarpFactor, 0 );              /*DAB1*/
     } /* end if */

    /*----------------------------------------------------------------*
     *  Send message to owner that the user has requested a spin up
     *  or a spin down.
     *----------------------------------------------------------------*/
     SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SpinNotify, hwnd);

    /*----------------------------------------------------------------*
     *  Attempt to turn on fast spinning
     *----------------------------------------------------------------*/
     FastSpinOn (hwnd, pSpinData, hwnd, TID_SECONDARY);
  } /* end if */

 /*-------------------------------------------------------------------*
  * Slow down the spinbutton
  *-------------------------------------------------------------------*/
//for (loop = 0; loop < DELAY; loop++);                                 //@B09

} /* end SendSpinKeybrdMsgs */


/**********************************************************************///OS2
/*                                                                    */
/*  Function Name:     GetSpinButtonStyle                             */
/*                                                                    */
/*  Descriptive Name:  Get Style of the Spin Button control           */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to query for the style of a spin button.     */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*                                                                    */
/*  Output:  Spin button style                                        */
/*                                                                    */
/*  Exit-Normal: Spin button style                                    */
/*                                                                    */
/*  Exit-Error:  NULL - Invalid hwndComponent                         */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  Underlying operating system                           */
/*                                                                    */
/**********************************************************************/
DWORD GetSpinButtonStyle (HWND hwnd)
{
  ULONG ulStyle;
  ULONG ulSpinFieldStyle = 0;

  ulStyle = STYLEOF(hwnd);

 /*-------------------------------------------------------------------*
  *  Set up entryfield justification...
  *-------------------------------------------------------------------*/
  if ((ulStyle & SPBS_JUSTCENTER) == SPBS_JUSTCENTER)
     ulSpinFieldStyle = ES_CENTER;
  else
  {
     if (ulStyle & SPBS_JUSTRIGHT)
        ulSpinFieldStyle = ES_RIGHT;
     else
        ulSpinFieldStyle = ES_LEFT;
  } /* end if */

 /*-------------------------------------------------------------------*
  *  Entry field is read only.  User will not see any visual
  *  diferences ... (Windows 3.1 only)
  *-------------------------------------------------------------------*/
#ifdef WIN31
  if (ulStyle & SPBS_READONLY)
     ulSpinFieldStyle |= ES_READONLY;
#endif

  if (!(ulStyle & SPBS_NOBORDER))
     ulSpinFieldStyle |= WS_BORDER;

  if (ulStyle & SPBS_FASTSPIN)
     ulSpinFieldStyle |= SPBS_FASTSPIN;

  if (ulStyle & WS_TABSTOP)
     ulSpinFieldStyle |= WS_TABSTOP;

  if (ulStyle & WS_DISABLED)
     ulSpinFieldStyle |= WS_DISABLED;

 /*-------------------------------------------------------------------*
  *  Windows required style settings (except ES_AUTOHSCROLL)
  *-------------------------------------------------------------------*/
  ulSpinFieldStyle |= WS_CHILD | WS_VISIBLE | ES_AUTOHSCROLL | ES_MULTILINE;

  return (ulSpinFieldStyle);

} /* end GetSpinButtonStyle */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SpinSubclassEntryfield                         */
/*                                                                    */
/*  Descriptive Name:  Spin button Subclass Entryfield                */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This is the subclass entryfield procedure.  It acts as the        */
/*  character input filter for the entryfield.  It also fixes         */
/*  a bug in the PM 1.1 version if selected text is deleted via       */
/*  the DELETE key.                                                   */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Handle to the Entryfield window                    */
/*          msg - Message to process                                  */
/*          wParam - Message parameter 1                              */
/*          lParam - Message parameter 2                              */
/*                                                                    */
/*  Output:  See returns from individual messages                     */
/*                                                                    */
/*  Exit-Normal: See returns from individual messages                 */
/*                                                                    */
/*  Exit-Error:  See returns from individual messages                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
LRESULT CALLBACK SpinSubclassEntryfield (HWND hwnd, UINT msg,
                                         WPARAM wParam, LPARAM lParam)
{
  HWND          hwndSpin;
  PSPINDATA     pSpinData = NULL;
  POINT         dwMousePos;

  hwndSpin = PARENTOF(hwnd);
  pSpinData = (PSPINDATA)GetWindowLong (hwndSpin, GWL_SPIN);

  switch (msg)
  {
    case WM_KEYDOWN:

     /*---------------------------------------------------------------*
      *  This is the logic to process virtual key ...
      *  Return any virtual key except the spacebar, up/down,
      *  and pageup/pagedown to the entryfield.
      *---------------------------------------------------------------*/
      switch (wParam)
      {
        case VK_UP:
        case VK_DOWN:
        case VK_PRIOR:
        case VK_NEXT:
         /*-----------------------------------------------------------*
          *  Must process in subclass window since Windows
          *  is processing these keys in the Edit Control.
          *-----------------------------------------------------------*/
          return (SpinChar (hwndSpin, msg, wParam, lParam, pSpinData));
          break;

       /*-------------------------------------------------------------*
        *  Return following virtual keys to the app.
        *-------------------------------------------------------------*/
        case VK_RETURN:
          SENDMESSAGE (PARENTOF(hwndSpin), WM_KEYDOWN, wParam, lParam);
          return ((long)TRUE);
          break;

        case VK_DELETE:                                                 /*DAB2*/
          if (STYLEOF(hwndSpin) & SPBS_READONLY)                        /*DAB2*/
          {                                                             /*DAB2*/
            MessageBeep (0);                                            /*DAB2*/
            return ((long)TRUE);                                        /*DAB2*/
          }                                                             /*DAB2*/
          break;                                                        /*DAB2*/

       /*-------------------------------------------------------------*
        *  Process the tab key for the dialog control ...
        *-------------------------------------------------------------*/
        case VK_TAB:                                                    //->@L05
         {
          char  szClass[8];

          if (GetClassName (PARENTOF(hwndSpin), (LPSTR)szClass, 8))
          {
            if (!(lstrcmp (szClass, "#32770")))
            {
              SetFocus (GetNextDlgTabItem (PARENTOF(hwndSpin),
                                           hwndSpin,
                                           ((GetKeyState (VK_SHIFT) < 0) ?
                                             TRUE   :
                                             FALSE)));

              return ((long)TRUE);
            }
          }

          SENDMESSAGE (PARENTOF(hwndSpin), WM_KEYDOWN, wParam, lParam);
          return ((long)TRUE);
         }
         break;                                                         //@L05<-

        case VK_SPACE:
         /*-----------------------------------------------------------*
          *  Check to see if we should trap out non-numerics
          *  Beep if the spacebar was pressed
          *-----------------------------------------------------------*/
          if (STYLEOF(hwndSpin) & SPBS_NUMERICONLY)
          {
             MessageBeep (0);
             return ((long)TRUE);
          } /* end if */

        default:
          return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                  msg, wParam, lParam));
          break;
      } /* end switch */
      break;


    case WM_KEYUP:

     /*---------------------------------------------------------------*
      *  Reset delete or select/replace flag
      *---------------------------------------------------------------*/
      pSpinData->fl &= ~SBF_DELETEORREPLACE;                            //@B07

     /*---------------------------------------------------------------*
      *  This is the logic to process virtual key ...
      *  Return any virtual key except the spacebar, up/down,
      *  and pageup/pagedown to the entryfield.
      *---------------------------------------------------------------*/
      switch (wParam)
      {
        case VK_UP:
        case VK_DOWN:
        case VK_PRIOR:
        case VK_NEXT:
         /*--------------------------------------------------------*
          *  Must process in subclass window since Windows
          *  is processing these keys in the Edit Control.
          *--------------------------------------------------------*/
          return (SpinChar (hwndSpin, msg, wParam, lParam, pSpinData));
          break;

        case VK_SPACE:
         /*--------------------------------------------------------*
          *  Check to see if we should trap out non-numerics
          *  Beep if the spacebar was pressed
          *--------------------------------------------------------*/
          if (STYLEOF(hwndSpin) & SPBS_NUMERICONLY)
             return ((long)TRUE);

        default:
          return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                  msg, wParam, lParam));
          break;
      } /* end switch */
      break;

    case WM_CHAR:
     /*-------------------------------------------------------------*
      *  Process the tab key for the dialog control ...
      *-------------------------------------------------------------*/
      if (wParam == VK_TAB)                                             //->@L05
      {
       char  szClass[8];

       if (GetClassName (PARENTOF(hwndSpin), (LPSTR)szClass, 8))
       {
         if (!(lstrcmp (szClass, "#32770")))
         {
           return ((long)TRUE);
         }
       }

       SENDMESSAGE (PARENTOF(hwndSpin), WM_CHAR, wParam, lParam);       //@B03
       return ((long)TRUE);
      }                                                                 //@L05<-

     /*---------------------------------------------------------------*
      *  If it is a read only spinbutton then no processing is
      *  done here.  The spinbutton will not beep or respond in any
      *  way, as entry field will not respond to read only control.
      *---------------------------------------------------------------*/
      if (STYLEOF(hwndSpin) & SPBS_READONLY)
      {
        /*------------------------------------------------------------*
         *  Ignore following virtual keys
         *------------------------------------------------------------*/
         if (wParam == VK_RETURN)
         {
             return ((long)TRUE);
         }
         else
         {
            if (wParam == VK_EXECUTE)
            {
               return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                       msg, wParam, lParam));
            } /* end if */
         } /* end if */

         MessageBeep (0);
         return ((long)TRUE);
      }
      else
      {
        /*------------------------------------------------------------*
         *  If it is not a read only spinbutton
         *  Check to see if we should trap out non-numerics
         *------------------------------------------------------------*/
         if (STYLEOF(hwndSpin) & SPBS_NUMERICONLY)
         {
            long Result;

           /*---------------------------------------------------------*
            *  Ignore following virtual keys.
            *---------------------------------------------------------*/
            if (wParam == VK_RETURN)
            {
               return ((long)TRUE);
            }
            else
            {
               if ((wParam == VK_BACK) || (wParam == VK_EXECUTE))
               {
                  return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                          msg, wParam, lParam));
               } /* end if */

              /*------------------------------------------------------*
               *  Check for deletion and set delete or select/replace
               *  flag if necessary
               *------------------------------------------------------*/
               if (wParam == VK_DELETE)                                 //->@B07
               {
                  pSpinData->fl |= SBF_DELETEORREPLACE;
                  return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                          msg, wParam, lParam));
               } /* end if */                                           //@B07<-

            } /* end if */

           /*---------------------------------------------------------*
            *  In WM_CHAR handling, KC_CHAR should always be
            *  processed before KC_VIRTUALKEY flag.  This is done
            *  because the number pad generates both virtual codes
            *  and character codes if Num Lock is on.
            *---------------------------------------------------------*/

            Result = SENDMESSAGE (pSpinData->hwndSpinField,
                                  EM_GETSEL,
                                  0,
                                  0);

           /*---------------------------------------------------------*
            *  Code fix:  allow check if a selection exists
            *---------------------------------------------------------*/
            if (LOWORD(Result) == 0)
            {
               BYTE szQuery[2];

               GetWindowText (pSpinData->hwndSpinField, szQuery, 2);

              /*------------------------------------------------------*
               *  Only check for an existing - if the string size
               *  is greater than zero.
               *------------------------------------------------------*/
               if (lstrlen ((LPSTR)szQuery) > 0)
               {
                 /*---------------------------------------------------*
                  *  Check for selection and set delete or
                  *  select/replace flag if necessary
                  *---------------------------------------------------*/
                  if (HIWORD(Result) != 0)                           //@B07
                    pSpinData->fl |= SBF_DELETEORREPLACE;               //@B07

                 /*---------------------------------------------------*
                  *  If a - already exists, don't allow the user
                  *  to type any more.  unless it is selected
                  *---------------------------------------------------*/
                  if ((szQuery[0] == '-') &&
                      (HIWORD(Result) == 0))
                  {
                    /*------------------------------------------------*
                     *  Beep if this is a key down transition
                     *------------------------------------------------*/
                     if ((GetKeyState (wParam)) < 0)
                     {
                        MessageBeep (0);
                        return ((long)TRUE);
                     } /* end if */
                  } /* end if */
               } /* end if */

              /*------------------------------------------------------*
               *  If there isn't currently a - in the entryfield,
               *  then allow them to type it, if array data is
               *  being spun ...
               *------------------------------------------------------*/
               if (wParam == (WORD)'-')
               {
                  if (pSpinData->fl & SBF_ARRAYDATA)
                  {
                     return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                             msg, wParam, lParam));
                  }
                  else
                  {
                    /*------------------------------------------------*
                     *  If there isn't currently a - in the
                     *  entryfield, then allow the user to type
                     *  one unless the lower limit is greater
                     *  than or equal to zero.
                     *------------------------------------------------*/
                     if (pSpinData->lLowerLimit < 0)
                     {
                        return (CallWindowProc ((FARPROC)pfnOldWindowProc,
                                                 hwnd, msg, wParam, lParam));
                     } /* end if */
                  } /* end if */
               } /* end if */
            } /* end if */

           /*---------------------------------------------------------*
            *  If the key pressed is less than or equal to '9' and
            *  greater than or equal to '0' then allow the user to
            *  type it.
            *---------------------------------------------------------*/
            if ((wParam <= (WORD)'9') && (wParam >= (WORD)'0'))
            {
               return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd,
                                       msg, wParam, lParam));
            } /* end if */

           /*------------------------------------------------------*
            *  Beep if the key is pressed
            *------------------------------------------------------*/
            if ((GetKeyState (wParam)) < 0)
               MessageBeep (0);

            return ((long)TRUE);
         } /* end if */
      } /* endif */
      break;

    case WM_GETDLGCODE:                                                 //->@B08
     /*-------------------------------------------------------------*
      * If a WM_GETDLGCODE message comes in, the spin button is in
      * in a dialog.  Tell the dialog which keys we want to process.
      *-------------------------------------------------------------*/
      return ((long)DLGC_WANTARROWS | DLGC_WANTCHARS |
                    DLGC_WANTTAB | DLGC_HASSETSEL);
      break;                                                            //@B08<-

    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
      /*------------------------------------------------------------*/
      /* Forward this button down to the spin button window (which  */
      /* won't do anything with it) so the Dialog Editor's subclass */
      /* function can select the window or open the style dialog.   */
      /*                                                            */
      /* If the spin button gets this message, it will return SB_-  */
      /* GOTBUTTONDOWN, in which case we know this is a "normal"    */
      /* suituation.  Otherwise, the Dialog Editor has subclassed   */
      /* the spin button and is handling the button down for us.    */
      /* In that case, we just want to eat the message.             */
      /*------------------------------------------------------------*/
      dwMousePos.x = LOWORD(lParam);            
      dwMousePos.y = HIWORD(lParam);            
      ClientToScreen (hwnd, (LPPOINT) &dwMousePos);
      ScreenToClient (hwndSpin, (LPPOINT) &dwMousePos);

      if ( SENDMESSAGE (  hwndSpin
                          , msg
                          , wParam
                          , (LPARAM)MAKELONG( dwMousePos.x, dwMousePos.y ))  
           != SB_GOTBUTTONDOWN)
      {
        return (TRUE);
      }
      break;

  } /* end switch */

  return (CallWindowProc ((FARPROC)pfnOldWindowProc, hwnd, msg, wParam, lParam));

} /* end SpinSubclassEntryfield */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     FastSpinOn                                     */
/*                                                                    */
/*  Descriptive Name:  Fast Spin On                                   */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function turns on the fast spin for a particular spinbutton  */
/*  component.                                                        */
/*                                                                    */
/*                                                                    */
/*  Notes: Windows allows only 16 timers .... Therefore, SetTimer     */
/*         may fail!                                                  */
/*                                                                    */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          hwndCurrent - Handle to the current window                */
/*          idTimer - Timer ID                                        */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SendSpinKeybrdMsgs                                    */
/*                                                                    */
/**********************************************************************/
VOID FastSpinOn (HWND hwnd, PSPINDATA pSpinData, HWND hwndCurrent,
                 int idTimer)
{

 /*-------------------------------------------------------------------*
  *  Check to see if fast spin style is set
  *-------------------------------------------------------------------*/
  if ((STYLEOF(hwndCurrent) & SPBS_FASTSPIN) && !(pSpinData->fl & SBF_TIMER))
  {
     while (!(pSpinData->idTimer = SetTimer (NULL, 0, TIME_DURATION, NULL)))
     {
        if (IDCANCEL == MessageBox (hwnd,
                                    "Too Many Timers!",
                                    "Spin Button",
                                    MB_ICONEXCLAMATION | MB_RETRYCANCEL))
        {
           return;
        } /* end if */
     } /* end while */

     KillTimer  (NULL, pSpinData->idTimer);
     while (!(SetTimer (hwnd, idTimer, TIME_DURATION, NULL)))   /* 2 seconds */
     {
        if (IDCANCEL == MessageBox (hwnd,
                                    "Too Many Timers!",
                                    "Spin Button",
                                    MB_ICONEXCLAMATION | MB_RETRYCANCEL))
        {
           return;
        } /* end if */
     } /* end while */

     pSpinData->fl |= SBF_TIMER;
  } /* end if */
} /* end FastSpinOn */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     FastSpinOff                                    */
/*                                                                    */
/*  Descriptive Name:  Fast Spin Off                                  */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function turns off fast spinning.  The window handle of the  */
/*  button to stop spinning is not necessary since the only thing     */
/*  needed to stop fast spinning is to stop the WM_TIMER messages     */
/*  from coming from the timer.                                       */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          idTimer - Timer ID                                        */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*              SendSpinKeybrdMsgs                                    */
/*                                                                    */
/**********************************************************************/
VOID FastSpinOff (HWND hwnd, PSPINDATA pSpinData, int idTimer)
{

 /*-------------------------------------------------------------------*
  *  If the timer is currently turned on, stop the timer messages
  *-------------------------------------------------------------------*/
  if (pSpinData->fl & SBF_TIMER)
  {
     KillTimer (hwnd, idTimer);
     while (!(pSpinData->idTimer = SetTimer (NULL, 0, TIME_DURATION, NULL)))
     {
        if (IDCANCEL == MessageBox (hwnd,
                                    "Too Many Timers - Buy OS/2!",
                                    "Spin Button",
                                    MB_ICONEXCLAMATION | MB_RETRYCANCEL))
        {
           return;
        } /* end if */
     } /* end while */

     KillTimer  (NULL, pSpinData->idTimer);
     pSpinData->fl &= (~SBF_TIMER);
     pSpinData->ulWarpFactor = 1;
  } /* end if */
} /* end FastSpinOff */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     ArrowWndProc                                   */
/*                                                                    */
/*  Descriptive Name:  Arrow Window Procedure                         */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to process messages for the spin arrows      */
/*  window.                                                           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin arrows (bitmaps)         */
/*          msg - Message to process                                  */
/*          wParam - Message Parameter 1                              */
/*          lParam - Message Parameter 2                              */
/*                                                                    */
/*  Output:  See returns from individual messages                     */
/*                                                                    */
/*  Exit-Normal: See returns from individual messages                 */
/*                                                                    */
/*  Exit-Error:  See returns from individual messages                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
LRESULT CALLBACK ArrowWndProc (HWND hwnd, UINT msg, WPARAM wParam,
                               LPARAM lParam)
{
  PSPINDATA    pSpinData;
  HWND         hwndSpin;
//HWND         hwndFocus;                   //jdr                       //@B09
//char         szClassName[32];             //jdr                       //@B09
  HWND         hwndFocusParent;             //jdr
  HDC          hdc;
  RECT         rc;
  RECT         rcWnd;
  MSG          qmsg;
  DWORD        dwStyle;
  PAINTSTRUCT  paint;
  HPEN         hpenNew;
  HPEN         hpenOld;
  HBRUSH       hbrushNew;
  HBRUSH       hbrushOld;
  POINT        dwMousePos;
  static   WORD count;
//register int  loop;
  register WORD SpinDirection;
  register WORD SpinNotify;

 /*-------------------------------------------------------------------*
  *  Get pointer to internally stored data from window words.
  *  hwndSpin is the hwnd to the spin button control.
  *-------------------------------------------------------------------*/
  hwndSpin = PARENTOF(hwnd);
  pSpinData = (PSPINDATA)GetWindowLong (hwndSpin, GWL_SPIN);

  switch (msg)
  {
    case WM_PAINT:
     /*---------------------------------------------------------------*
      *  Set current spinbtn style flags
      *---------------------------------------------------------------*/
      dwStyle = STYLEOF(hwndSpin);

     /*---------------------------------------------------------------*
      *  rcPaint is the invalid region
      *---------------------------------------------------------------*/
      hdc = BeginPaint (hwnd, (LPPAINTSTRUCT)&paint);

     /*---------------------------------------------------------------*
      *  rcWnd is the region of the spin arrows window
      *---------------------------------------------------------------*/
      GetClientRect (hwnd, &rcWnd);

     /*---------------------------------------------------------------*
      *  Define appropriate pen
      *---------------------------------------------------------------*/
      hpenNew = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_WINDOWFRAME));
      hpenOld = (hpenNew) ? SelectObject (hdc, hpenNew) : NULL;

     /*---------------------------------------------------------------*
      *  Define appropriate brush
      *---------------------------------------------------------------*/
      hbrushNew = CreateSolidBrush (GetSysColor (COLOR_WINDOW));
//    hbrushNew = (HBRUSH)SENDMESSAGE (PARENTOF(hwnd),
//                                     WM_CTLCOLOR,
//                                     hdc,
//                                     MAKELONG(hwnd, CTLCOLOR_BTN));
      hbrushOld = (hbrushNew) ? SelectObject (hdc, hbrushNew) : NULL;

      if ((rcWnd.top == paint.rcPaint.top) ||                       //@OS2
          (paint.rcPaint.top < ((rcWnd.bottom - rcWnd.top)/2)))
      {
        /*------------------------------------------------------------*
         *  The up arrow region is invalid,  so draws the up arrow
         *  portion ...
         *------------------------------------------------------------*/
         rcWnd.bottom -= ((rcWnd.bottom - rcWnd.top)/2);
//       DrawArrow (hdc, &rcWnd, pSpinData->hbmUp,                      //->$L02
//                  (BOOL)pSpinData->fl & SBF_UPARROWPRESSED);
         DrawArrow (hdc, &rcWnd,
                    (dwStyle & WS_DISABLED) ? pSpinData->hbmUpDisabled :
                                              pSpinData->hbmUp,
                    (BOOL)pSpinData->fl & SBF_UPARROWPRESSED);          //$L02<-
      } /* end if */

      GetClientRect (hwnd, &rcWnd);
      if ((rcWnd.bottom == paint.rcPaint.bottom)  ||                //@OS2
          (paint.rcPaint.bottom > ((rcWnd.bottom - rcWnd.top)/2)))
      {
        /*------------------------------------------------------------*
         *  The down arrow region is invalid,  so draws the down arrow
         *  portion ...
         *------------------------------------------------------------*/
         rcWnd.top += ((rcWnd.bottom - rcWnd.top)/2);
//       DrawArrow (hdc, &rcWnd, pSpinData->hbmDown,                    //->$L02
//                  (BOOL)pSpinData->fl & SBF_DOWNARROWPRESSED);
         DrawArrow (hdc, &rcWnd,
                    (dwStyle & WS_DISABLED) ? pSpinData->hbmDownDisabled :
                                              pSpinData->hbmDown,
                    (BOOL)pSpinData->fl & SBF_DOWNARROWPRESSED);        //$L02<-
      } /* end if */

     /*---------------------------------------------------------------*
      *  Restore original brush
      *---------------------------------------------------------------*/
      if (hbrushNew)
         DeleteObject (SelectObject (hdc, hbrushOld));

     /*---------------------------------------------------------------*
      *  Restore original pen
      *---------------------------------------------------------------*/
      if (hpenNew)
         DeleteObject (SelectObject (hdc, hpenOld));

      EndPaint (hwnd, (LPPAINTSTRUCT)&paint);
      break;

    case WM_LBUTTONUP:
      GetClientRect (hwnd, &rc);

     /*---------------------------------------------------------------*
      *  Determines the rect to be invalidated
      *---------------------------------------------------------------*/
      if (pSpinData->fl & SBF_UPARROWPRESSED)
      {
         rc.bottom -= ((rc.bottom - rc.top)/2);
         pSpinData->fl &= ~SBF_UPARROWPRESSED;
         pSpinData->fl &= ~SBF_INITIALUPARROWPRESS;
      }
      else
      {
         if (pSpinData->fl & SBF_DOWNARROWPRESSED)
         {
            rc.top += ((rc.bottom - rc.top)/2);
            pSpinData->fl &= ~SBF_DOWNARROWPRESSED;
            pSpinData->fl &= ~SBF_INITIALDOWNARROWPRESS;
         } /* end if */
      } /* end if */

     /*---------------------------------------------------------------*
      *  Stop the timer, release the mouse capture, let the owner
      *  know we stopped spinning, and reset warp factor ...
      *---------------------------------------------------------------*/
      KillTimer (hwnd, TID_PRIMARY);
      ReleaseCapture ();

      if (IsWindow (pSpinData->hwndCurrentlySelected))                  //->@B09
        hwndFocusParent = pSpinData->hwndCurrentlySelected;
      else
        hwndFocusParent = hwndSpin;

      SENDCONTROL (PARENTOF(hwndSpin), IDOF(hwndSpin), SPBN_ENDSPIN,
                   hwndFocusParent);

//    SENDCONTROL (PARENTOF(hwndSpin), IDOF(hwndSpin), SPBN_ENDSPIN,
//                 hwndSpin);                                           //@B09<-
      pSpinData->ulWarpFactor = 1;

     /*---------------------------------------------------------------*
      *  Invalidate on the button that was let up
      *---------------------------------------------------------------*/
      InvalidateRect (hwnd, &rc, TRUE);

      dwMousePos.x = LOWORD(lParam);             
      dwMousePos.y = HIWORD(lParam);             
      ClientToScreen (hwnd, (LPPOINT) &dwMousePos);
      ScreenToClient (hwndSpin, (LPPOINT) &dwMousePos);

      SENDMESSAGE (   hwndSpin
                    , msg
                    , wParam
                    , (LPARAM)MAKELONG( dwMousePos.x, dwMousePos.y ));

      return (TRUE);
      break;

    case WM_LBUTTONDOWN:
    case WM_LBUTTONDBLCLK:
      /*------------------------------------------------------------*/
      /* Forward this button down to the spin button window (which  */
      /* won't do anything with it) so the Dialog Editor's subclass */
      /* function can select the window or open the style dialog.   */
      /*                                                            */
      /* If the spin button gets this message, it will return SB_-  */
      /* GOTBUTTONDOWN, in which case we know this is a "normal"    */
      /* situation.  Otherwise, the Dialog Editor has subclassed    */
      /* the spin button and is handling the button down for us.    */
      /* In that case, we just want to eat the message.             */
      /*------------------------------------------------------------*/
      dwMousePos.x = LOWORD(lParam);            
      dwMousePos.y = HIWORD(lParam);            
      ClientToScreen (hwnd, (LPPOINT) &dwMousePos);
      ScreenToClient (hwndSpin, (LPPOINT) &dwMousePos);
      
      if (SENDMESSAGE (   hwndSpin
                        , msg
                        , wParam
                        , (LPARAM)MAKELONG( dwMousePos.x, dwMousePos.y ))  
          != SB_GOTBUTTONDOWN)
      {
        return (TRUE);
      }

     /*---------------------------------------------------------------*
      *  Coordinates for the invalid region are calculated by whether
      *  or not the user clicked on the up arrow or the down arrow.
      *---------------------------------------------------------------*/
      GetClientRect (hwnd, &rc);

      if ((int)HIWORD(lParam) < (rc.top + (rc.bottom - rc.top)/2))
      {
        /*------------------------------------------------------------*
         *  Only have to lower the .bottom variable
         *------------------------------------------------------------*/
         rc.bottom -= ((rc.bottom - rc.top)/2);
         pSpinData->fl |= SBF_UPARROWPRESSED;
         SpinDirection = SPBM_SPINUP;
         SpinNotify = SPBN_UPARROW;
      }
      else
      {
        /*------------------------------------------------------------*
         *  Only have to raise the .top variable
         *------------------------------------------------------------*/
         rc.top += ((rc.bottom - rc.top)/2);
         pSpinData->fl |= SBF_DOWNARROWPRESSED;
         SpinDirection = SPBM_SPINDOWN;
         SpinNotify = SPBN_DOWNARROW;
      } /* end if */

      pSpinData->ulWarpFactor = 1;

     /*---------------------------------------------------------------*
      *  If the user has clicked down on the arrow and then moves
      *  the mouse off the arrow, the arrow window will still
      *  receive the button 1 up message
      *---------------------------------------------------------------*/
      SetCapture (hwnd);

      SENDMESSAGE (pSpinData->hwndCurrentlySelected ?
                   pSpinData->hwndCurrentlySelected : PARENTOF(hwnd),
                   WM_SETFOCUS_PRIVATE, 0, 0L);


      //jdr begin =============================================================

      /*-----------------------------------------------------------*/
      /* If a spin button has the "focus", the focus will really   */
      /* be on its edit field.  If the parent of the focus window  */
      /* is a spin button, we want to use hwndFocusParent instead  */
      /* of hwndSpin since hwndSpin will always be the master spin */
      /* button, not the servant spin button if there is one.      */
      /*-----------------------------------------------------------*/
//    hwndFocus = GetFocus ();                                          //->@B09
//
//    if (IsWindow (hwndFocus))
//    {
//      hwndFocusParent = PARENTOF(hwndFocus);
//
//      if (IsWindow (hwndFocusParent))
//      {
//        GetClassName (hwndFocusParent, szClassName,
//                      sizeof(szClassName));
//        if (lstrcmp (szClassName, ICL_SPINBTN))
//          hwndFocusParent = hwndSpin;
//      }
//    }
//    else
//      hwndFocusParent = hwndSpin;

      if (IsWindow (pSpinData->hwndCurrentlySelected))
        hwndFocusParent = pSpinData->hwndCurrentlySelected;
      else
        hwndFocusParent = hwndSpin;                                     //@B09<-

      SendSpinMouseMsgs (pSpinData, hwndFocusParent, SpinDirection,
                         SpinNotify);

//jdr SendSpinMouseMsgs (pSpinData, hwndSpin, SpinDirection, SpinNotify);

      //jdr end ===============================================================


     /*---------------------------------------------------------------*
      *  Count used for the spinning delay in the spinfield if
      *  the arrow button is held down.  This delay is not used
      *  when the spinfield increments too fast on the initial
      *  WM_LBUTTONDOWN message.
      *---------------------------------------------------------------*/
      count = 0;

     /*---------------------------------------------------------------*
      *  Only invalidate the region that contains the button that
      *  was pressed ...
      *
      *  Track initial press to prevent bitblt's of the button when
      *  user holds the mouse button down ...
      *---------------------------------------------------------------*/
      if (pSpinData->fl & SBF_UPARROWPRESSED)                       //->@OS2
      {
         if (!(pSpinData->fl & SBF_INITIALUPARROWPRESS))
         {
            InvalidateRect (hwnd, &rc, TRUE);
            pSpinData->fl |= SBF_INITIALUPARROWPRESS;
         } /* end if */
      }
      else
      {
         if (!(pSpinData->fl & SBF_INITIALDOWNARROWPRESS))
         {
            InvalidateRect (hwnd, &rc, TRUE);
            pSpinData->fl |= SBF_INITIALDOWNARROWPRESS;
         } /* end if */
      } /* end if */                                                //@OS2<-

#if 0      /*-------------------------< JDR >--------------------------------*/
      KillTimer  (hwnd, TID_PRIMARY);                               //@OS2
#endif     /*-------------------------< JDR >--------------------------------*/

      while (!(SetTimer (hwnd, TID_PRIMARY, INITIAL_TIMEDELAY, NULL)))
      {
         if (IDCANCEL == MessageBox (hwnd,
                                     "Too Many Timers!",
                                     "Spin Button",
                                     MB_ICONEXCLAMATION | MB_RETRYCANCEL))
         {
            return (FALSE);
         } /* end if */
      } /* end while */
      return (TRUE);
      break;

    case WM_TIMER:
      if (pSpinData->fl & SBF_UPARROWPRESSED)
      {
         SpinDirection  = SPBM_SPINUP;
         SpinNotify = SPBN_UPARROW;
      }
      else
      {
         SpinDirection  = SPBM_SPINDOWN;
         SpinNotify = SPBN_DOWNARROW;
      } /* end if */

      /*-----------------------------------------------------------*/   //->@B09
      /* If a spin button has the "focus", the focus will really   */
      /* be on its edit field.  If the parent of the focus window  */
      /* is a spin button, we want to use hwndFocusParent instead  */
      /* of hwndSpin since hwndSpin will always be the master spin */
      /* button, not the servant spin button if there is one.      */
      /*-----------------------------------------------------------*/
//    hwndFocusParent = PARENTOF(GetFocus ());
//    if (!(IsWindow (hwndFocusParent)))
//       hwndFocusParent = hwndSpin;

      if (IsWindow (pSpinData->hwndCurrentlySelected))
        hwndFocusParent = pSpinData->hwndCurrentlySelected;
      else
        hwndFocusParent = hwndSpin;

      dwStyle = STYLEOF(hwndFocusParent);                               //@B09<-
      KillTimer (hwnd, TID_PRIMARY);

      while (!PeekMessage (&qmsg, NULL,
                           WM_LBUTTONUP, WM_LBUTTONUP, PM_NOREMOVE))
      {
         if ((dwStyle & SPBS_FASTSPIN) && ((count++ % 200) == 0))
         {
            if (pSpinData->ulWarpFactor < 0x10000000)
               pSpinData->ulWarpFactor *= 2;
         } /* end if */

         SendSpinMouseMsgs (pSpinData, hwndFocusParent, SpinDirection,
                            SpinNotify);                                //@B09

//       SendSpinMouseMsgs (pSpinData, hwndSpin, SpinDirection,
//                          SpinNotify);                                //@B09

        /*------------------------------------------------------------*
         * Slow down the spinbutton so it will be as slow as the
         * arrow keys
         *------------------------------------------------------------*/
//       for (loop = 0; loop < DELAY; loop++);                      //@OS2
      } /* end while */
      return (TRUE);
      break;

    default:
      break;
  }
  return DefWindowProc (hwnd, msg, wParam, lParam);

} /* end ArrowWndProc */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     DrawArrow                                      */
/*                                                                    */
/*  Descriptive Name:  Draw Arrow (Bitmap)                            */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to draw the up/down arrow bitmaps.           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hdc - Presentation space handle                           */
/*          pRect - Pointer to rectangle for drawing bounds           */
/*          hbmArrow - Arrow bitmap to draw                           */
/*          fButtonDown - Indicates position of the button            */
/*                        {TRUE = INWARD  FALSE = OUTWARD}            */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  ArrowWndProc                                          */
/*                                                                    */
/**********************************************************************/
VOID DrawArrow (HDC hdc, LPRECT pRect, HBITMAP hbmArrow, BOOL fButtonDown)
{
  POINT     ptl;
  BITMAP    BitmapHdr;
  HBITMAP   hbmOld;             /* Old bitmap handle                  */
  HDC       hdcMemory;          /* handle to memory device context    */
  int       ibmHeight;          /* height to blit bitmap              *///@J01
  int       isrcHeight;         /* height of source bitmap            *///@J01
  int       ichopBM;            /* drop bits from down arrow stem     *///@J01

 /*-------------------------------------------------------------------*
  *  Compute the point where we need to draw the bitmap so it will
  *  appear to be centered
  *-------------------------------------------------------------------*/
  GetObject (hbmArrow, sizeof(BITMAP), (LPSTR)&BitmapHdr);

  ptl.x = pRect->left  + ((pRect->right - pRect->left
                       - (BitmapHdr.bmWidth - 1)) / 2);
  ibmHeight = (pRect->bottom-2) - (pRect->top+2);                       //@J01

  /* hack bitmap height to limit ugliness caused by compaction on VGA *///@J01
  ichopBM = 0;                                                          //@J01
  if(ibmHeight< BitmapHdr.bmHeight)                                     //@J01
  {                                                                     //@J01
    if (pRect->top)                                                     //@J01
      ichopBM = 3;                                                      //@J01
    isrcHeight = BitmapHdr.bmHeight-3;                                  //@J01
    ibmHeight = isrcHeight;                                             //@J01
  }                                                                     //@J01
  else                                                                  //@J01
  {                                                                     //@J01
    isrcHeight = BitmapHdr.bmHeight;                                    //@J01
  }                                                                     //@J01

  ptl.y = pRect->top + ((pRect->bottom - pRect->top - ibmHeight)/2);    //@J01

  DrawButtonArea (hdc, pRect, fButtonDown);

  if (fButtonDown)
  {
    /*----------------------------------------------------------------*
     * Moves the arrow down and to the right to make it appear as
     * though the button is being pushed in ...
     *----------------------------------------------------------------*/
     ptl.x++;
     ptl.y++;
  } /* end if */

  if (hdcMemory = CreateCompatibleDC (hdc))                             /*WBB1*/
  {                                                                     /*WBB1*/
    if (hbmOld = SelectObject (hdcMemory, hbmArrow))                    /*WBB1*/
    {                                                                   /*WBB1*/
      StretchBlt ( hdc, ptl.x, ptl.y,                                   //@J01
                  (BitmapHdr.bmWidth - 1), ibmHeight,                   //@J01
                  hdcMemory, 0, ichopBM, BitmapHdr.bmWidth,isrcHeight,  //@J01
                  SRCCOPY);                                             //@J01

//    BitBlt (hdc, ptl.x, ptl.y, (BitmapHdr.bmWidth - 1), BitmapHdr.bmHeight,
//            hdcMemory, 0, 0, SRCCOPY);

      SelectObject (hdcMemory, hbmOld);
    }                                                                   /*WBB1*/
    DeleteDC (hdcMemory);                                               /*WBB1*/
  }                                                                     /*WBB1*/

} /* end DrawArrow */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     DrawButtonArea                                 */
/*                                                                    */
/*  Descriptive Name:  Draw Button Area                               */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to fill the rectangle around the up/down     */
/*  arrow buttons to simulate inward/outward appearance of the        */
/*  bitmaps.                                                          */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hdc - Presentation space handle                           */
/*          pRect - Pointer to rectangle for drawing bounds           */
/*          fButtonDown - Indicates position of the button            */
/*                        {TRUE = INWARD  FALSE = OUTWARD}            */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  DrawArrow                                             */
/*                                                                    */
/**********************************************************************/
VOID DrawButtonArea (HDC hdc, LPRECT pRect, BOOL fButtonDown)
{
  POINT  ptl[5];
  HPEN   hpenOld;
//HPEN   hpenCurrent;
  HBRUSH hbrushOld;
  HBRUSH hFaceBrush = NULL;                                             /*WBB1*/
  HPEN   hShadowPen = NULL;                                             /*WBB1*/
  HPEN   hWhitePen = NULL;                                              /*WBB1*/

  Rectangle (hdc, pRect->left, pRect->top, pRect->right, pRect->bottom);

 /*-------------------------------------------------------------------*
  *  Define button polygon
  *-------------------------------------------------------------------*/
  ptl[0].x = pRect->left;        /* ptl[0] - upper left   */
  ptl[0].y = pRect->top;
  ptl[1].x = pRect->right - 1;   /* ptl[1] - upper right  */
  ptl[1].y = pRect->top;
  ptl[2].x = pRect->right - 1;   /* ptl[2] - lower right  */
  ptl[2].y = pRect->bottom - 1;
  ptl[3].x = pRect->left;        /* ptl[3] - lower left   */
  ptl[3].y = pRect->bottom - 1;
  ptl[4].x = pRect->left;        /* ptl[4] - upper left - */
  ptl[4].y = pRect->top;         /* needed to finish drawing */
 /*-------------------------------------------------------------------*
  *  Draw bitmap outline
  *-------------------------------------------------------------------*/
//hpenCurrent = CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNFACE));
  if (hFaceBrush = CreateSolidBrush (GetSysColor (COLOR_BTNFACE)))      /*WBB1*/
  {                                                                     /*WBB1*/
    hbrushOld = SelectObject (hdc, hFaceBrush);                         /*WBB1*/
  //hpenOld = SelectObject (hdc, hpenCurrent);
    Polygon (hdc, ptl, 4);
  //DeleteObject (SelectObject (hdc, hpenOld));
    DeleteObject (SelectObject (hdc, hbrushOld));
  }                                                                     /*WBB1*/

  if (fButtonDown)
  {
    /*----------------------------------------------------------------*
     *  Draw shadow border around left and top
     *----------------------------------------------------------------*/
     if (hShadowPen =
         CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNSHADOW)))        /*WBB1*/
     {                                                                  /*WBB1*/
       hpenOld = SelectObject (hdc, hShadowPen);                        /*WBB1*/
       MoveToEx (hdc, (ptl[3].x + 1), (ptl[3].y - 1), NULL);
       LineTo (hdc, (ptl[3].x + 1), (ptl[4].y + 1));
       LineTo (hdc, ptl[1].x, (ptl[1].y + 1));
       DeleteObject (SelectObject (hdc, hpenOld));
     }                                                                  /*WBB1*/
  }
  else
  {
    /*----------------------------------------------------------------*
     *  Draw white border around left and top
     *----------------------------------------------------------------*/
     if (hWhitePen = GetStockObject (WHITE_PEN))                        /*WBB1*/
     {                                                                  /*WBB1*/
       hpenOld = SelectObject (hdc, hWhitePen);                         /*WBB1*/
  //                           CreatePen (PS_SOLID, 1,
  //                                      GetSysColor (COLOR_WINDOW)));
       MoveToEx (hdc, (ptl[3].x + 1), (ptl[3].y - 1), NULL);
       LineTo (hdc, (ptl[3].x + 1), (ptl[4].y + 1));
       LineTo (hdc, ptl[1].x, (ptl[1].y + 1));
       SelectObject (hdc, hpenOld);                                     /*WBB1*/
     }                                                                  /*WBB1*/

    /*----------------------------------------------------------------*
     *  Draw shadow border around right and bottom
     *----------------------------------------------------------------*/
     if (hShadowPen =
         CreatePen (PS_SOLID, 1, GetSysColor (COLOR_BTNSHADOW)))        /*WBB1*/
     {                                                                  /*WBB1*/
       hpenOld = SelectObject (hdc, hShadowPen);                        /*WBB1*/
       MoveToEx (hdc, (ptl[3].x + 1), (ptl[3].y - 1), NULL);
       LineTo (hdc, (ptl[2].x - 1), (ptl[2].y - 1));
       LineTo (hdc, (ptl[1].x - 1), ptl[1].y);
  //   MoveTo (hdc, (ptl[3].x + 2), (ptl[3].y - 2));
  //   LineTo (hdc, (ptl[2].x - 2), (ptl[2].y - 2));
  //   LineTo (hdc, (ptl[1].x - 2), ptl[1].y);
       DeleteObject (SelectObject (hdc, hpenOld));
     }                                                                  /*WBB1*/
  } /* end if */

} /* end DrawButtonArea */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     SendSpinMouseMsgs                              */
/*                                                                    */
/*  Descriptive Name:  Send Spin button Mouse Messages                */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine is used to send messages to control the spining of   */
/*  the spinbutton. If the OWNERCONTROL flag is set, the spinbutton   */
/*  control sends a message to the owner requesting the application   */
/*  update the spinfield.  If this flag is not set, messages are sent */
/*  internally to the spinbutton code to spin the value.  Afterwards, */
/*  a notification message is sent to the owner.                      */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*          hwnd - Window handle of the spin button                   */
/*          SpinMsg - Spin button msg to either spin up or down       */
/*          SpinNotify - Spin notification that the up arrow was      */
/*                       clicked on.  Also used in the same manner    */
/*                       manner for down arrow.                       */
/*                                                                    */
/*  Output:  rc from WinSendMsg                                       */
/*                                                                    */
/*  Exit-Normal:  rc from WinSendMsg                                  */
/*                                                                    */
/*  Exit-Error:  rc from WinSendMsg                                   */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  ArrowWndProc                                          */
/*                                                                    */
/**********************************************************************/
long SendSpinMouseMsgs (PSPINDATA pSpinData, HWND hwnd,
                        WORD SpinMsg, WORD SpinNotify)
{
  long result;
  PSPINDATA pSelectedSpinData;

  pSelectedSpinData =
          (PSPINDATA)GetWindowLong (pSpinData->hwndCurrentlySelected, GWL_SPIN);

  if( pSelectedSpinData )
  {
    if (!(pSelectedSpinData->fl & SBF_OWNERCONTROL))
    {
       SENDMESSAGE (pSpinData->hwndCurrentlySelected, SpinMsg,
                    (DWORD)pSpinData->ulWarpFactor, 0 );                  /*DAB1*/
    } /* end if */
  }
  
  result = SENDCONTROL (PARENTOF(hwnd), IDOF(hwnd), SpinNotify, hwnd);

  return (result);
} /* end SendSpinMouseMsgs */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     AlignRange                                     */
/*                                                                    */
/*  Descriptive Name:  Align Range                                    */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function aligns the current value within the set limits.     */
/*  If the current value is larger than the upper limit, the current  */
/*  value is changed to the upper limit.  If the current value is     */
/*  less than the lower limit, the current value is reset to the      */
/*  lower limit.                                                      */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  pSpinData - Pointer to internally stored data             */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
VOID AlignRange (PSPINDATA pSpinData)
{
  if (pSpinData->lCurrentValue > pSpinData->lUpperLimit)
     pSpinData->lCurrentValue = pSpinData->lUpperLimit;
  else
  {
     if (pSpinData->lCurrentValue < pSpinData->lLowerLimit)
        pSpinData->lCurrentValue = pSpinData->lLowerLimit;
  } /* end if */
} /* end AlignRange */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     AllocHeapMem                                   */
/*                                                                    */
/*  Descriptive Name:  Allocate Heap Memory                           */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function allocates a block of memory on the heap.            */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  cbSize - Size of block to allocate                        */
/*                                                                    */
/*  Output:  Near pointer                                             */
/*                                                                    */
/*  Exit-Normal:  Near pointer to memory block                        */
/*                                                                    */
/*  Exit-Error:  NULL - Out of Memory                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
NPVOID AllocHeapMem (UINT cbSize)
{
  PBYTE nptr;

 /*-------------------------------------------------------------------*
  *  Allocate a block from the heap
  *-------------------------------------------------------------------*/
  if (!(nptr = (PBYTE)LocalAlloc (LMEM_FIXED | LMEM_ZEROINIT, cbSize)))
     return (NULL);

#ifdef DEBUG
  memset (nptr, 0xCC, cbSize);             /* initialize with known value   */
  *(nptr + cbSize - 1) = 0x7c;             /* put an eyecatcher at the end  */
#endif

  return ((NPVOID)nptr);
} /* end AllocHeapMem */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     FreeHeapMem                                    */
/*                                                                    */
/*  Descriptive Name:  Free Heap Memory                               */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This function frees a block of memory on the heap.                */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  nptr - pointer to block to free                           */
/*                                                                    */
/*  Output:  TRUE/FALSE                                               */
/*                                                                    */
/*  Exit-Normal:  TRUE - Success                                      */
/*                                                                    */
/*  Exit-Error:  FALSE - Failure                                      */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
BOOL FreeHeapMem (PBYTE nptr)
{
  return (!((BOOL)LocalFree ((HANDLE)nptr)));
} /* end FreeHeapMem */

#if 0 //JEH
/**********************************************************************/
/*                                                                    */
/*  Function Name:     memset                                         */
/*                                                                    */
/*  Descriptive Name:  Set Memory                                     */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine Sets first cb bytes of p to character c.             */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  p  - Pointer to destination                               */
/*          c  - Character to set                                     */
/*          cb - Number of characters                                 */
/*                                                                    */
/*  Output:  Pointer                                                  */
/*                                                                    */
/*  Exit-Normal:  Pointer to destination                              */
/*                                                                    */
/*  Exit-Error:  None                                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  UpdateField                                           */
/*                                                                    */
/**********************************************************************/
NPVOID memset (NPVOID p, BYTE c, WORD cb)
{
  PBYTE pb;

  for (pb=p; cb--; *pb++ = c);
  return (p);
} /* end memset */
#endif

/**********************************************************************/
/*                                                                    */
/*  Function Name:     strtol                                         */
/*                                                                    */
/*  Descriptive Name:  String To Long                                 */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine converts an alphanumeric (asciiz) string into a      */
/*  signed long integer.                                              */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  lpszValue - Alphanumeric string containing value          */
/*          lppEndChar - Character where conversion stopped           */
/*                                                                    */
/*  Output:  Long signed integer                                      */
/*                                                                    */
/*  Exit-Normal:  Converted long signed integer                       */
/*                                                                    */
/*  Exit-Error:  ???                                                  */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  SpinButtonWndProc                                     */
/*                                                                    */
/**********************************************************************/
LONG strtol (LPSTR lpszValue, LPPSTR lppEndChar)
{
  LONG   result = 0L;
  BOOL   bInvertRes = FALSE;
  int  i;

  i = CCHMAXLONG - 1;

  if (*lpszValue == '-')
  {
     lpszValue++;
     bInvertRes = TRUE;
  } /* end if */

  while (*lpszValue == '0')
     lpszValue++;

  while ((*lpszValue) && (i > 0))
  {
     if (*lpszValue >= '0' && *lpszValue <= '9')
     {
        result = result * 10L;
        result += (DWORD)(*lpszValue - '0');
     }
     else
     {
         break;
     }
     lpszValue++;
     i--;
  } /* end while */

  *lppEndChar = lpszValue;

  if ((DWORD)result > MAXLONG)
    result = MAXLONG;

  if (bInvertRes)
     result = ~result + 1;

  return (result);

} /* end strtol */

#if 0  //JEH
/**********************************************************************/
/*                                                                    */
/*  Function Name:     strrev                                         */
/*                                                                    */
/*  Descriptive Name:  Reverse string                                 */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine reverses (inverts) a string.                         */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  lpString - Pointer to string to reverse                   */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  ltoa                                                  */
/*                                                                    */
/**********************************************************************/
VOID strrev (LPSTR lpString)
{
  int   i, j;
  BYTE    c;

  for (i = 0, j = lstrlen (lpString) - 1; i<j; i++, j-- ) {
      c = lpString[i];
      lpString[i] = lpString[j];
      lpString[j] = c;
  } /* end for */
} /* end strrev */
#endif

/**********************************************************************/
/*                                                                    */
/*  Function Name:     ltoa                                           */
/*                                                                    */
/*  Descriptive Name:  Long to ASCII                                  */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine converts a signed long integer into an asciiz        */
/*  string.                                                           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  hwnd - Window handle of the spin button                   */
/*          pSpinData - Pointer to internally stored data             */
/*          lpString - Pointer to string to store converted integer   */
/*                                                                    */
/*  Output:  None                                                     */
/*                                                                    */
/*  Exit-Normal:                                                      */
/*                                                                    */
/*  Exit-Error:                                                       */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  UpdateField                                           */
/*                                                                    */
/**********************************************************************/
VOID ltoa (HWND hwnd, PSPINDATA pSpinData, LPSTR lpString)
{
  LONG  n;
  int i;
  BOOL  bInvertRes = FALSE;

  n = pSpinData->lCurrentValue;

  i = 0;

 /*-------------------------------------------------------------------*
  *  If the number is negative, set negative flag and invert
  *-------------------------------------------------------------------*/
  if (n < 0)
  {
   n = ~n + 1;
   bInvertRes = TRUE;
  } /* end if */

  do {
     lpString[i++] = (BYTE)((n % 10) + '0');
  } while ((n /= 10) > 0);

  if (STYLEOF(hwnd) & SPBS_PADWITHZEROS)
  {
     while (i < (int)pSpinData->usNumericFormatSize)
        lpString[i++] = '0';

     if (bInvertRes)
        lpString[i-1] = '-';

  }
  else
  {
     if (bInvertRes)
        lpString[i++] = '-';
  } /* end if */

  lpString[i] = '\0';
#ifdef IC_WU
  _strrev (lpString);                                           /*RKK2*/
#else
  strrev (lpString);
#endif

} /* end ltoa */

#if 0
/**********************************************************************/
/*                                                                    */
/*  Function Name:     strlen                                         */
/*                                                                    */
/*  Descriptive Name:  String Length                                  */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine calculates the length of an asciiz string.           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  lpString - NULL terminated string                         */
/*                                                                    */
/*  Output:  Length                                                   */
/*                                                                    */
/*  Exit-Normal:  Length of asciiz string                             */
/*                                                                    */
/*  Exit-Error:  0                                                    */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  UpdateField                                           */
/*                                                                    */
/**********************************************************************/
WORD strlen (LPSTR lpString)
{
  register WORD length;

  for (length = 0; *lpString++; length++);
  return (length);
} /* end strlen */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     strcpy                                         */
/*                                                                    */
/*  Descriptive Name:  String Copy                                    */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine copies an asciiz string.                             */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  lpszDest - Destination buffer                             */
/*          lpszSrc  - Source string                                  */
/*                                                                    */
/*  Output:  Pointer                                                  */
/*                                                                    */
/*  Exit-Normal:  Pointer to destination buffer                       */
/*                                                                    */
/*  Exit-Error:  0                                                    */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  UpdateField                                           */
/*                                                                    */
/**********************************************************************/
LPSTR strcpy (LPSTR lpszDest, LPSTR lpszSrc)
{
  LPSTR lpszSave;

  lpszSave = lpszDest;
  while (*lpszDest++ = *lpszSrc++);
  return (lpszSave);
} /* end strcpy */


/**********************************************************************/
/*                                                                    */
/*  Function Name:     strcmp                                         */
/*                                                                    */
/*  Descriptive Name:  String Compare                                 */
/*                                                                    */
/*  Functional Description:                                           */
/*                                                                    */
/*  This routine compares 2 asciiz strings.                           */
/*                                                                    */
/*                                                                    */
/*  Notes:                                                            */
/*                                                                    */
/*  Input:  lpString1 - String 1                                      */
/*          lpString2 - String 2                                      */
/*                                                                    */
/*  Output:  Comparison Indicator                                     */
/*                                                                    */
/*  Exit-Normal: less than 0  - lpString1 is less than lpString2      */
/*               0            - lpString1 and lpString2 are identical */
/*               greater than 0 - lpString1 is greater than lpString2 */
/*                                                                    */
/*  Exit-Error:  None                                                 */
/*                                                                    */
/*  Side Effects:  None                                               */
/*                                                                    */
/*  Called By:  UpdateField                                           */
/*                                                                    */
/**********************************************************************/
int strcmp (LPSTR lpString1, LPSTR lpString2)
{
  while (*lpString1 && (*lpString1 == *lpString2))
  {
     lpString1++;  lpString2++;
  } /* end while */

  return (*lpString1 - *lpString2);
} /* end strcmp */
#endif
