#ifndef _ILISTBX4_
#define _ILISTBX4_

// Revision: 98 1.4.2.3 source/ui/basectl/ilistbx4.hpp, listctls, ioc.v400  
/*NOSHIP*/
/*******************************************************************************
* FILE NAME: ilistbx4.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in both ilistbox.hpp and icombobx.hpp.                                     *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#ifdef IC_MOTIF
  #include <Xm/List.h>
  #include <Xm/ScrolledW.h>
  #include <stdio.h>

  #include <ilistbox.hpp>
  #include <istring.hpp>
  #include <irect.hpp>
  #include <ireslib.hpp>
  #include <iapp.hpp>
  #include <ihandle.hpp>
  #include <iexcept.hpp>
  #include <itrace.hpp>
#endif //IC_MOTIF

#ifdef IC_USE_CB
  #define IC_CLASSNAME IComboBox
  #define IC_CURSOR_CLASSNAME IBaseComboBox
#else
  #define IC_CLASSNAME IListBox
  #define IC_CURSOR_CLASSNAME IBaseListBox
#endif

#ifdef IC_MOTIF
// Carry over from PM implementation.
  #define LIT_ERROR           (-3)
  #define LIT_END             (-1)

#ifdef IC_USE_CB
  #define IC_BASECLASSPRIV IBaseComboBoxData
  #define PRIVATEDATA      fBaseComboBoxData
#else
  #define IC_BASECLASSPRIV IBaseListBoxData
  #define PRIVATEDATA      fBaseListBoxData
#endif
#endif //IC_MOTIF


/*------------------------------------------------------------------------------
   IItemHandlesListDeleter - Static function used as input to allElementsDo(),
   resulting in all members of the collection being deleted.
------------------------------------------------------------------------------*/
#ifdef IC_MOTIF
bool IItemHandlesListDeleter ( IItemHandleElement* const& handleElement, void* )
{
  delete (IItemHandleElement*)handleElement;
  return true;
}
#endif

/*------------------------------------------------------------------------------
| IC_CLASSNAME::add                                                            |
|                                                                              |
| Add an item to the list using a string resource id.                          |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::add ( unsigned long lIndex,
                                  const IResourceId& resid )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return add(lIndex,newText);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::add                                                            |
|                                                                              |
| Add an item to the list using a const char *.  This method is also used by   |
| the other add item methods to actually insert the item.                      |
------------------------------------------------------------------------------*/
unsigned long   IC_CLASSNAME::add(unsigned long lIndex, 
                                  const char* pszString)
{
#ifdef IC_PMWIN
//   needValidation++;   // add may result in an invalid cursor.
   incrementChangeCount( );   // add may result in an invalid cursor.
   unsigned long anIndex = count();
   if ((long)anIndex > (long)lIndex)
      anIndex = lIndex;
   IEventResult evt = handle().sendEvent(LM_INSERTITEM,
                               IEventParameter1(anIndex),
                               IEventParameter2((unsigned long)pszString));
   switch (evt.asLong())
   {
      case LIT_MEMERROR:
      case LIT_ERROR:
         ITHROWGUIERROR("LM_INSERTITEM");
         break;
      default:
         break;
   } /* endswitch */

#ifndef IC_USE_CB
#ifdef IC_WIN
   // If horizontal scroll bar style specified and listbox not owner draw
   if ( fListBoxData->horizontalScrollStyleSet && !isDrawItem() )
   {
      unsigned long newSize = fListBoxData->stringSize(this, pszString);
      if (newSize > fListBoxData->largestString)
      {
         handle().sendEvent( LB_SETHORIZONTALEXTENT,
                             newSize + 4,  //add fudge of 4 pixels
                             0 );
         fListBoxData->largestString = newSize;
      }
   }
#endif //IC_WIN
#endif

#ifdef IC_USE_CB
#ifdef IC_WIN
   // If horizontal scroll bar style specified and listbox not owner draw
  if( isHorizontalScroll() )
  {
    IFont tempFont = font();
    unsigned long newSize =  tempFont.textWidth(pszString);    
    if (newSize > fComboBoxData->fLargestStringSize)
    {
       handle().sendEvent( CB_SETHORIZONTALEXTENT,
                           newSize + 4,  //add fudge of 4 pixels
                         0 );
       fComboBoxData->fLargestStringSize = newSize;
    }
  }
#endif //IC_WIN
#endif

   notifyObservers(INotificationEvent(IC_CLASSNAME::addId,
                                      *this, true, (void *)1));
   return evt.asUnsignedLong();  // Return index of inserted item.
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Widget        listBox;
  char          *addItem;
  XmString      string;
  int           position, itemCount;
  IItemHandlesList::Cursor handleCursor (*(PRIVATEDATA->itemHandlesList));
  IItemHandleElement *handleElement;
  unsigned long keyAtCursor;

  // needValidation++;
  incrementChangeCount( );   // add may result in an invalid cursor.

  // Insert the item.  The character string will be converted to a
  // simple compound string prior to insertion.

  listBox = IC_BASECLASSPRIV::listBoxHandle (*this);
  position = lIndex;
  position++;

  // Create a compound string.
  addItem = (char*) XtMalloc (strlen(pszString) + 1);
  sprintf (addItem, "%s", pszString);
  string = XmStringCreateLocalized (addItem);

  XmListAddItemUnselected(listBox,
                          (XmString) string,
                          position);

  XmStringFree (string);
  XtFree (addItem);

  // Get index of item added.
  itemCount = count();
  if (position == 0)
  {
     // Item added to end of list.
     position = itemCount - 1;
  }
  else
  {
    // If the insertion index was beyond the current end of list, return the
    // index of the last item in the list.
    //
    // Note that in PM if the insertion point is beyond the end of the list,
    // nothing will be added to the list, but here the items are inserted.

    if (--position > itemCount - 1)
       position = itemCount - 1;
  }

  // Adjust the itemHandlesList since keys for items following the new item
  // added need to be incremented by 1.
  for (handleCursor.setToLast();
       handleCursor.isValid();)
  {
     keyAtCursor = key (PRIVATEDATA->itemHandlesList->elementAt(handleCursor));
     if (keyAtCursor < position)
        handleCursor.invalidate();
     else
     {
        handleElement = new IItemHandleElement (keyAtCursor + 1,
                                                itemHandle (keyAtCursor));
        IItemHandleElement*  handleElementOld = PRIVATEDATA->itemHandlesList->elementAt(handleCursor);                                      
        PRIVATEDATA->itemHandlesList->removeAt (handleCursor);
        delete handleElementOld;
        PRIVATEDATA->itemHandlesList->add (handleElement, handleCursor);
        handleCursor.setToPrevious();
     }
  }

  return (unsigned long) position;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::add                                                            |
|                                                                              |
| Add a list of items to the listbox.  An array of character strings are       |
| passed in.  Return index of first inserted item.                             |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::add ( unsigned long lIndex,
                                  const char* const * itemList,
                                  unsigned long addCount )
{
#ifdef IC_PMWIN
  int  i;
  unsigned long rc, firstIndex;

  // assertions on input parms
  IASSERTPARM(addCount != 0 && itemList != 0);
  for (i=0; i < addCount; i++)
  {
    rc = add( lIndex+i, (char*)itemList[i] );
    if (!i)
      firstIndex = rc;
  }
  return firstIndex;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Widget        listBox;
  char          *strings, *addItem;
  XmStringTable stringTable;
  int           i, position, itemCount;
  IItemHandlesList::Cursor handleCursor (*(PRIVATEDATA->itemHandlesList));
  IItemHandleElement *handleElement;
  unsigned long keyAtCursor;

  // needValidation++;
  incrementChangeCount( );   // add may result in an invalid cursor.

  /***************************************************************************/
  /* Insert the item(s).  The character strings will be converted            */
  /* to simple compound strings prior to insertion.                          */
  /***************************************************************************/

  listBox = IC_BASECLASSPRIV::listBoxHandle (*this);
  position = lIndex;
  position++;

  XtVaGetValues (listBox,
                 XmNitemCount, &itemCount,
                 NULL);

  /***************************************************************************/
  /* Create an array of compound strings.                                    */
  /***************************************************************************/
  strings = XtMalloc (addCount * sizeof (XmString *));
  stringTable = (XmStringTable) strings;
  for (i=0; i < addCount; i++)
  {
     addItem = (char*) itemList[i];
     stringTable[i] = XmStringCreateLocalized (addItem);
  }

  /***************************************************************************/
  /* Add items to the listbox.                                               */
  /***************************************************************************/
  XmListAddItems(listBox,
                 stringTable,
                 addCount,
                 position);

  /***************************************************************************/
  /* Free strings.                                                           */
  /***************************************************************************/
  for (i=0; i < addCount; i++)
     XmStringFree (stringTable[i]);
  XtFree (strings);

  /***************************************************************************/
  /* Calculate the actual index of the first item added.                     */
  /***************************************************************************/
  if (lIndex == first)
     position = 0;
  else if (lIndex > itemCount)
     position = itemCount;
  else
     position--;

  // Adjust the itemHandlesList since keys for items following the new item
  // added need to be incremented by addCount.
  for (handleCursor.setToLast();
       handleCursor.isValid();)
  {
     keyAtCursor = key (PRIVATEDATA->itemHandlesList->elementAt(handleCursor));
     if (keyAtCursor < position)
        handleCursor.invalidate();
     else
     {
        handleElement = new IItemHandleElement (keyAtCursor + addCount,
                                                itemHandle (keyAtCursor));
        IItemHandleElement*  handleElementOld = PRIVATEDATA->itemHandlesList->elementAt(handleCursor);                                     
        PRIVATEDATA->itemHandlesList->removeAt (handleCursor);
        delete handleElementOld;
        PRIVATEDATA->itemHandlesList->add (handleElement, handleCursor);
        handleCursor.setToPrevious();
     }
  }

  return position;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsFirst                                                     |
|                                                                              |
| Add an item at the beginning of the list using a string resource id.         |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAsFirst ( const IResourceId& resid )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAsFirst(newText);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsFirst                                                     |
|                                                                              |
| Add an item at the beginning of the list using a const char *.               |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAsFirst ( const char* pszString )
{
   return add(0, pszString);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsLast                                                      |
|                                                                              |
| Add an item at the end of the list using a string resource id.               |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAsLast ( const IResourceId& resid )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAsLast(newText);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsLast                                                      |
|                                                                              |
| Add an item at the end of the list using a const char *.                     |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAsLast ( const char* pszString )
{
   return add(LIT_END, pszString);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAscending                                                   |
|                                                                              |
| Add an item to the list in ascending order using a string resource id.       |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAscending ( const IResourceId& resid )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAscending(newText);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAscending                                                   |
|                                                                              |
| Add an item to the list in ascending order using a const char *.             |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addAscending ( const char* item )
{
#ifdef IC_WIN           //mkb
   // Use binary search to find position at which item is to be inserted.
   long upperBound = count() - 1;
   long lowerBound = 0;
   while ( upperBound >= lowerBound )
     {
     long i;
     i = lowerBound + ((upperBound - lowerBound) / 2);

     if ( lstrcmp( itemText(i), item ) > 0)
       upperBound = i - 1;
     else
       lowerBound = i + 1;
     }
   ITRACE_ALL( IString("addAscending index=") + IString(lowerBound));
   return add( lowerBound, item );
#endif //IC_WIN

#ifdef IC_PM
   return add(LIT_SORTASCENDING, item);
#endif //IC_PM

#ifdef IC_MOTIF
   Widget        listBox;
   char          *addItem, *currentItem;
   int           i, upperBound, lowerBound;
   XmString      itemString, *itemList;
   IItemHandlesList::Cursor handleCursor (*(PRIVATEDATA->itemHandlesList));
   IItemHandleElement *handleElement;
   unsigned long keyAtCursor;

   // needValidation++;
   incrementChangeCount( );   // add may result in an invalid cursor.

   // Get the current list box entries.
   listBox = IC_BASECLASSPRIV::listBoxHandle (*this);
   XtVaGetValues (listBox,
                  XmNitemCount, &upperBound,
                  XmNitems, &itemList,
                  NULL);

   // Use binary search to find position at which item is to be inserted.
   upperBound--;
   lowerBound = 0;
   while (upperBound >= lowerBound)
   {
      i = lowerBound + ((upperBound - lowerBound) / 2);
      if (!XmStringGetLtoR (itemList[i], XmFONTLIST_DEFAULT_TAG, &currentItem))
         break;
      if (strcmp (currentItem, item) > 0)
         upperBound = i - 1;
      else
         lowerBound = i + 1;
      XtFree (currentItem);
   }
   addItem = (char*) item;
   itemString = XmStringCreateLocalized (addItem);
   XmListAddItemUnselected (listBox,
                            itemString,
                            lowerBound+1);
   XmStringFree (itemString);

  // Adjust the itemHandlesList since keys for items following the new item
  // added need to be incremented by 1.
  for (handleCursor.setToLast();
       handleCursor.isValid();)
  {
     keyAtCursor = key (PRIVATEDATA->itemHandlesList->elementAt (handleCursor));
     if (keyAtCursor < lowerBound)
        handleCursor.invalidate();
     else
     {
        handleElement = new IItemHandleElement (keyAtCursor + 1,
                                                itemHandle (keyAtCursor));
        IItemHandleElement*  handleElementOld = PRIVATEDATA->itemHandlesList->elementAt(handleCursor);                                      
        PRIVATEDATA->itemHandlesList->removeAt (handleCursor);
        delete handleElementOld;
        PRIVATEDATA->itemHandlesList->add (handleElement, handleCursor);
        handleCursor.setToPrevious();
     }
  }

  return lowerBound;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addDescending                                                  |
|                                                                              |
| Add an item to the list in descending order using a string resource id.      |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addDescending ( const IResourceId& resid )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addDescending(newText);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addDescending                                                  |
|                                                                              |
| Add an item to the list in descending order using a const char *.            |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::addDescending ( const char* item )
{
#ifdef IC_WIN   //mkb
   // Use binary search to find position at which item is to be inserted.
   long upperBound = count() - 1;
   long lowerBound = 0;
   while ( upperBound >= lowerBound )
     {
     long i;
     i = lowerBound + ((upperBound - lowerBound) / 2);
     if ( lstrcmp( item, itemText(i)) > 0 )
       upperBound = i - 1;
     else
       lowerBound = i + 1;
     }
   return add( lowerBound, item );
#endif //IC_WIN

#ifdef IC_PM 
   return add(LIT_SORTDESCENDING, item);
#endif //IC_PM

#ifdef IC_MOTIF
   Widget        listBox;
   char          *addItem, *currentItem;
   int           i, upperBound, lowerBound;
   XmString      itemString, *itemList;
   IItemHandlesList::Cursor handleCursor (*(PRIVATEDATA->itemHandlesList));
   IItemHandleElement *handleElement;
   unsigned long keyAtCursor;

   // needValidation++;
   incrementChangeCount( );   // add may result in an invalid cursor.

   // Get the current list box entries.
   listBox = IC_BASECLASSPRIV::listBoxHandle (*this);
   XtVaGetValues (listBox,
                  XmNitemCount, &upperBound,
                  XmNitems, &itemList,
                  NULL);

   // Use binary search to find position at which item is to be inserted.
   upperBound--;
   lowerBound = 0;
   while (upperBound >= lowerBound)
   {
      i = lowerBound + ((upperBound - lowerBound) / 2);
      if (!XmStringGetLtoR (itemList[i], XmFONTLIST_DEFAULT_TAG, &currentItem))
         break;
      if (strcmp (item, currentItem) > 0)
         upperBound = i - 1;
      else
         lowerBound = i + 1;
      XtFree (currentItem);
   }
   addItem = (char*) item;
   itemString = XmStringCreateLocalized (addItem);
   XmListAddItemUnselected (listBox,
                            itemString,
                            lowerBound+1);
   XmStringFree (itemString);

  // Adjust the itemHandlesList since keys for items following the item
  // added need to be incremented by 1.
  for (handleCursor.setToLast();
       handleCursor.isValid();)
  {
     keyAtCursor = key (PRIVATEDATA->itemHandlesList->elementAt (handleCursor));
     if (keyAtCursor < lowerBound)

        handleCursor.invalidate();
     else
     {
        handleElement = new IItemHandleElement (keyAtCursor + 1,
                                                itemHandle (keyAtCursor));
        IItemHandleElement*  handleElementOld = PRIVATEDATA->itemHandlesList->elementAt(handleCursor);                                      
        PRIVATEDATA->itemHandlesList->removeAt (handleCursor);
        delete handleElementOld;
        
        PRIVATEDATA->itemHandlesList->add (handleElement, handleCursor);
        handleCursor.setToPrevious();
     }
  }

  return lowerBound;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::removeAll                                                      |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::removeAll ( )
{
#ifdef IC_PMWIN
   unsigned long selectCount = numberOfSelections();
   IEventResult evt = handle().sendEvent(LM_DELETEALL,
                                         IEventParameter1(0),
                                         IEventParameter2(0));
#ifndef IC_USE_CB
#ifdef IC_WIN
    //reset extents to 0  
    fListBoxData->largestString = 0;
    handle().sendEvent( LB_SETHORIZONTALEXTENT,
                        0,  //add fudge of 4 pixels
                        0 );
#endif //IC_WIN
#endif //IC_USE_CB

#ifdef IC_USE_CB
#ifdef IC_WIN
        // reset extents to 0  
        fComboBoxData->fLargestStringSize = 0;
        handle().sendEvent( CB_SETHORIZONTALEXTENT,
                          0,  //add fudge of 4 pixels
                          0 );
#endif //IC_WIN
#endif //IC_USE_CB

#if  !(defined(IC_WIN) && !(defined IC_USE_CB))      //mkb
   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("LM_DELETEALL");
#endif
// needValidation++;  // removeAll may result in an invalid cursor.
   incrementChangeCount( );   // add may result in an invalid cursor.
   if (selectCount)
   {
     INotificationEvent anEvent( IC_CLASSNAME::selectId,
                                 *this );
     notifyObservers( anEvent );
   }
   notifyObservers(INotificationEvent(IC_CLASSNAME::removeId,
                                      *this, true, (void *)0));
#endif //IC_PMWIN

#ifdef IC_MOTIF
   incrementChangeCount( ); 

   XmListDeleteAllItems(IC_BASECLASSPRIV::listBoxHandle(*this));
   PRIVATEDATA->itemHandlesList->removeAll(&IItemHandlesListDeleter);

#ifdef IC_USE_CB
   // remove entryfield text
   XmTextSetString(IC_BASECLASSPRIV::textHandle(*this),
                   "");
#endif //IC_USE_CB
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::remove                                                         |
|                                                                              |
| Remove one item from the list and return the number of remaining items.      |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::remove ( unsigned long lIndex )
{
#ifdef IC_WIN
        
  IString itemRemoved;
  if( lIndex < count() )
  {
    itemRemoved = itemText( lIndex );
  }
#endif //IC_WIN

#ifdef IC_PMWIN
// needValidation++;  // remove may result in an invalid cursor.
   incrementChangeCount( );   // remove may result in an invalid cursor.
   bool selected = isSelected( lIndex );
   IEventResult evt = handle().sendEvent(LM_DELETEITEM,
                                         IEventParameter1(lIndex),
                                         IEventParameter2(0));


#ifndef IC_USE_CB
#ifdef IC_WIN
   // If horizontal scroll bar style specified and listbox not owner draw
   if ( fListBoxData->horizontalScrollStyleSet && !isDrawItem() )
   {
      IFont tempFont = font();
      unsigned long removedSize = tempFont.textWidth(itemRemoved);

      if( removedSize == fListBoxData->largestString ) 
      {
        fListBoxData->largestString = fListBoxData->largestStringSize(this);
        handle().sendEvent( LB_SETHORIZONTALEXTENT,
                            fListBoxData->largestString + 4,  //add fudge of 4 pixels
                            0 );
      }
   }
#endif //IC_WIN
#endif

#ifdef IC_USE_CB
#ifdef IC_WIN
   // If horizontal scroll bar style specified and listbox not owner draw
   if ( isHorizontalScroll() )
   {
      IFont tempFont = font();
      unsigned long removedSize = tempFont.textWidth(itemRemoved);

      if( removedSize == fComboBoxData->fLargestStringSize ) 
      {
        fComboBoxData->fLargestStringSize = fComboBoxData->largestStringSize(this);
        handle().sendEvent( CB_SETHORIZONTALEXTENT,
                          fComboBoxData->fLargestStringSize + 4,  //add fudge of 4 pixels
                          0 );
      }
   }
#endif //IC_WIN
#endif

   if ( selected )
   {
     INotificationEvent  anEvent( IC_CLASSNAME::selectId,
                                 *this );
     notifyObservers( anEvent );
   }
   notifyObservers(INotificationEvent(IC_CLASSNAME::removeId,
                                      *this, true, (void *)1));
   return evt.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int                position;
  IItemHandlesList::Cursor handleCursor (*(PRIVATEDATA->itemHandlesList));
  IItemHandleElement *handleElement;
  unsigned long      keyAtCursor;

  // needValidation++;
  incrementChangeCount( );

  position = lIndex;
  position++;

  XmListDeleteItemsPos(IC_BASECLASSPRIV::listBoxHandle(*this), 1, position);
  PRIVATEDATA->itemHandlesList->removeElementWithKey (lIndex);

  // Adjust the itemHandlesList since keys for items following the new item
  // added need to be incremented by 1.
  for (handleCursor.setToFirst();
       handleCursor.isValid();
       handleCursor.setToNext())
  {
     keyAtCursor = key (PRIVATEDATA->itemHandlesList->elementAt (handleCursor));
     if (keyAtCursor > lIndex)
     {
        handleElement = new IItemHandleElement (keyAtCursor - 1,
                                                itemHandle (keyAtCursor));
        IItemHandleElement*  handleElementOld = PRIVATEDATA->itemHandlesList->elementAt(handleCursor);                                      
        PRIVATEDATA->itemHandlesList->removeAt (handleCursor);
        delete handleElementOld;
        PRIVATEDATA->itemHandlesList->add (handleElement, handleCursor);
     }
  }

  return count();
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IC_CLASSNAME::removeAt                                                       |
|                                                                              |
| Remove the item at the cursor's position.                                    |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::removeAt ( IC_CURSOR_CLASSNAME::Cursor& cursor )
{
  IASSERTSTATE(cursor.isValid());
  unsigned long i = cursor.asIndex();
  cursor.setToNext();
  if (cursor.isValid())
  {
     remove(i);
#ifdef IC_MOTIF
     PRIVATEDATA->itemHandlesList->removeElementWithKey (i);
#endif //IC_MOTIF
#ifdef IC_MOTIFPM
     cursor.setToIndex(cursor.asIndex() - 1);
#endif  //IC_MOTIFPM
#ifdef IC_WIN
     // Store away current pos prior to cursor reset
     unsigned long curPos = cursor.asIndex();
     // Reset cursor data after removing listbox item
     cursor.setToFirst();
     // Set cursor position to "next" item after remove
     cursor.setToIndex(curPos-1);
#endif //IC_WIN
  }
  else
  {
     remove(i);
#ifdef IC_MOTIF
     PRIVATEDATA->itemHandlesList->removeElementWithKey (i);
#endif //IC_MOTIF
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::replaceAt                                                      |
|                                                                              |
| Replace the item at the cursor's position using a string resource id.        |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::replaceAt ( const IResourceId& resid,
                                        const IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return replaceAt(newText, cursor);
}

IC_CLASSNAME& IC_CLASSNAME::replaceAt ( const char* strNew,
                                        const IC_CURSOR_CLASSNAME::Cursor& cursor )
{
  IASSERTSTATE(cursor.isValid());
  #ifdef IC_WIN
     IString strRemoved = itemText( cursor.asIndex() );
  #endif //IC_WIN

  setItemText(cursor.asIndex(), strNew);

#ifndef IC_USE_CB
#ifdef IC_WIN
  // If horizontal scroll bar style specified and listbox not owner draw
  if ( fListBoxData->horizontalScrollStyleSet && !isDrawItem() )
  {
    unsigned long newSize = fListBoxData->stringSize(this, strNew);
    unsigned long removedSize = fListBoxData->stringSize(this, strRemoved);
          
    if (newSize > fListBoxData->largestString ) 
    {
      fListBoxData->largestString = newSize;
      handle().sendEvent( LB_SETHORIZONTALEXTENT,
                          fListBoxData->largestString + 4,  //add fudge of 4 pixels
                          0 );
    }         
    else if (( newSize < fListBoxData->largestString )
          && (removedSize == fListBoxData->largestString))      
    {
        fListBoxData->largestString = fListBoxData->largestStringSize(this);
        handle().sendEvent( LB_SETHORIZONTALEXTENT,
                            fListBoxData->largestString + 4,  //add fudge of 4 pixels
                            0 );
    }
  }
#endif //IC_WIN
#endif

#ifdef IC_USE_CB
#ifdef IC_WIN
   // If horizontal scroll bar style specified and listbox not owner draw
   if ( isHorizontalScroll() )
   {
      IFont tempFont = font();
      unsigned long newSize =  tempFont.textWidth(strNew);
      unsigned long removedSize = tempFont.textWidth( strRemoved);

      if (newSize > fComboBoxData->fLargestStringSize ) 
      {
        fComboBoxData->fLargestStringSize = newSize;                          
        handle().sendEvent( CB_SETHORIZONTALEXTENT,
                          fComboBoxData->fLargestStringSize + 4,  //add fudge of 4 pixels
                          0 );
      }
      else if (( newSize < fComboBoxData->fLargestStringSize )
              && (removedSize == fComboBoxData->fLargestStringSize))      
      {    
        fComboBoxData->fLargestStringSize = fComboBoxData->largestStringSize(this);
        handle().sendEvent( CB_SETHORIZONTALEXTENT,
                          fComboBoxData->fLargestStringSize + 4,  //add fudge of 4 pixels
                          0 );
      }
   } //if isHorizontalScroll
#endif //IC_WIN
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::add                                                            |
|                                                                              |
| Add an item at the cursor's position using a string resource id.  The cursor |
| is set to the new item.                                                      |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::add ( const IResourceId& resid,
                                  IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return add(newText, cursor);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::add                                                            |
|                                                                              |
| Add an item at the cursor's position using a const char *.  The cursor is    |
| set to the new item.                                                         |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::add ( const char* strItem,
                                  IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IASSERTSTATE(cursor.isValid());
   unsigned long index = add(cursor.asIndex(), strItem);
   cursor.setToIndex(index);
   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsFirst                                                     |
|                                                                              |
| Add an item at the beginning of the list using a string resource id.  The    |
| cursor is set to the new item.                                               |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsFirst ( const IResourceId& resid,
                                         IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAsFirst(newText, cursor);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsFirst                                                     |
|                                                                              |
| Add an item at the beginning of the list using a const char *.  The cursor   |
| is set to the new item.                                                      |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsFirst ( const char* strItem,
                                         IC_CURSOR_CLASSNAME::Cursor& cursor )
{
  unsigned long index = addAsFirst(strItem);
  cursor.setToIndex(index);
  return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsLast                                                      |
|                                                                              |
| Add an item at the end of the list using a string resource id.  The cursor   |
| is set to the new item.                                                      |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsLast ( const IResourceId& resid,
                                        IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAsLast(newText, cursor);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsLast                                                      |
|                                                                              |
| Add an item at the end of the list using a const char *.  The cursor is set  |
| to the new item.                                                             |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsLast ( const char* strItem,
                                        IC_CURSOR_CLASSNAME::Cursor& cursor )
{
  unsigned long index = addAsLast(strItem);
  cursor.setToIndex(index);
  return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsNext                                                      |
|                                                                              |
| Add an item after the item at the current cursor position using a string     |
| resource id.  The cursor is set to the new item.                             |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsNext ( const IResourceId& resid,
                                        IC_CURSOR_CLASSNAME::Cursor& cursor )
{
   IString newText = resid.resourceLibrary().loadString(resid);
   return addAsNext(newText, cursor);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::addAsNext                                                      |
|                                                                              |
| Add an item after the item at the current cursor position using a const      |
| char *.  The cursor is set to the new item.                                  |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::addAsNext ( const char* strItem,
                                        IC_CURSOR_CLASSNAME::Cursor& cursor )
{
  IASSERTSTATE(cursor.isValid());
  unsigned long index = cursor.asIndex();
  long lElems = count();

  if (index >= 0  &&  index < lElems)
  {
    if (index == lElems - 1)
       index = LIT_END;
    else
       index++;
    index = add(index, strItem);
    cursor.setToIndex(index);
  }
  return *this;
}

#endif // ILISTBX4 header guard
