// Revision: 64 1.39.1.3 source/ui/basectl/icvhdr.cpp, canvas, ioc.v400 
/*******************************************************************************
* FILE NAME: icvhdr.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icvhdr.hpp.                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C" {
  #define INCL_GPILOGCOLORTABLE  // GpiCreateLogColorTable, etc.
  #define INCL_WINBUTTONS        // BS_xxx, BM_xxx
  #define INCL_WINDIALOGS        // WinEnumDlgItem, etc.
  #define INCL_WINFRAMEMGR       // WM_QUERYFOCUSCHAIN, SC_CLOSE
  #define INCL_WININPUT          // WM_CHAR, WinGetKeyState, WinQueryFocus
  #define INCL_WINMESSAGEMGR     // WM_COMMAND, etc., CMDSRC_OTHER
  #define INCL_WINSTATICS        // SS_BKGNDRECT, SS_FGNDRECT
  #define INCL_WINWINDOWMGR      // WinDefWindowProc, WinQueryWindowULong, etc.

  #include <iwindefs.h>
#ifdef IC_WIN
  #include <commctrl.h>          // WC_TABCONTROL
#endif
}

#ifdef IC_MOTIF
extern "C" {
  #include <Xm/Xm.h>
  #include <Xm/PushB.h>
  #include <Xm/CascadeB.h>
  #include <Xm/ToggleB.h>
  #include <Xm/BulletinB.h>
  }
#endif

#include <icvhdr.hpp>
#include <icanvas.hpp>
#include <icconst.h>
#include <icmnfun.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <iradiobt.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>
#ifdef IC_WIN
#include <ientryfd.hpp>
#endif
// #include <iplatfrm.hpp>

// Suppress const conversion messages.
#pragma info(nocnv)

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ICVHDR_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_PMWIN
bool ICanvasStatics :: fHasBeenRegistered = false;
bool ICanvasStatics :: fAllPaintsHasBeenRegistered = false;
#endif

/*------------------------------------------------------------------------------
| ICanvasStatics::~ICanvasStatics                                              |
|                                                                              |
| Destructor to clean up.                                                      |
------------------------------------------------------------------------------*/
ICanvasStatics :: ~ICanvasStatics ( )
{
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICanvasStatics::setWindowClassRegistered                                     |
|                                                                              |
| Indicates that the window class has been registered.                         |
------------------------------------------------------------------------------*/
void ICanvasStatics :: setWindowClassRegistered ( EWindowClassStyle cs )
{
  if ( cs == kClassStyleMaxPaints )
    fAllPaintsHasBeenRegistered = true;
  else if ( cs == kClassStyleMinPaints )
    fHasBeenRegistered = true;
}

/*------------------------------------------------------------------------------
| ICanvasStatics::isWindowClassRegistered                                      |
|                                                                              |
| Returns window class registration status.                                    |
------------------------------------------------------------------------------*/
bool ICanvasStatics :: isWindowClassRegistered ( EWindowClassStyle cs )
{
  if ( cs == kClassStyleMaxPaints )
    return( fAllPaintsHasBeenRegistered );
  else if ( cs == kClassStyleMinPaints )
    return( fHasBeenRegistered );
  else
    return false;
}
#endif // IC_PMWIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvasHandler::ICanvasHandler                                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ICanvasHandler :: ICanvasHandler ( )
  : ICanvasHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| ICanvasHandler::~ICanvasHandler                                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ICanvasHandler :: ~ICanvasHandler ( )
{ }

/*------------------------------------------------------------------------------
| ICanvasHandler::dispatchHandlerEvent                                         |
|                                                                              |
| Router and default event processing for events.                              |
| Notes: Event processing includes:                                            |
|    --  Expose                                                                |
|  (with IC_UM_CANVAS_PAINT, calls the layout routine to provide               |
|  specialized sizing and positioning of child controls)                       |
|    --  IC_UM_CANVAS_PAINT                                                    |
|  (with Expose, calls the layout routine to provide specialized sizing        |
|  and positioning of child controls)                                          |
|    --  VisibilityNotify                                                      |
|  (with IC_UM_CANVAS_PAINT, calls the layout routine to provide               |
|  specialized sizing and positioning of child controls)                       |
|    --  MapNotify                                                             |
|  (Removes any bit gravity so the entire canvas repaints when its size        |
|  changes to avoid leaving border and grid line artifacts)                    |
------------------------------------------------------------------------------*/
bool ICanvasHandler::dispatchHandlerEvent ( IEvent &evt )
{
  bool bProcessed = false;

  switch (evt.eventId())
  {
    case IC_UM_CANVAS_PAINT:
      {
        ICanvas* pcvCanvas = (ICanvas*)evt.dispatchingWindow();
        pcvCanvas->setLayoutDistorted(0, IWindow::fontPropogated);
        bProcessed = layoutCanvas(evt);
        break;
      } /* endcase */

    case xEvent(Expose):
      {
        XEvent
         *xEvent = (XEvent*)(char*) evt.parameter2();
        XExposeEvent
         *exposeEvent = (XExposeEvent*)xEvent;
        unsigned long
          exposeCount = exposeEvent->count;
        if ( exposeCount != 0 )
        {  // More expose messages still to come.
           break;
        }
        // Else fall into the IC_UM_PAINT case.
      }
    case IC_UM_PAINT:
      {
        IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(Expose)");
        ICanvas* pcvCanvas = (ICanvas*)evt.dispatchingWindow();

        if ( pcvCanvas->isLayoutDistorted(IWindow::layoutChanged)  &&
             pcvCanvas->isVisible())
        {    // Use layout to show child changes on screen
          ITRACE_DEVELOP("Send IC_UM_CANVAS_PAINT message");
          pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
        }       // Pass processing on to operating system.

        // Now paint the canvas.
        bProcessed = pcvCanvas->paintWindow( evt );
        break;
      } /* endcase */

    case xEvent(VisibilityNotify):
      {
        ICanvas* pcvCanvas = (ICanvas*)evt.dispatchingWindow();
        if ( pcvCanvas->isLayoutDistorted(IWindow::layoutChanged)  )
        {        // Window being shown and need new layout
          IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(VisibilityNotify)");
          pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
        }       // Pass processing on to operating system.
        break;
      } /* endcase */

    case xEvent(MapNotify):
      {
        IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(MapNotify)");
        if ( evt.parameter1().asUnsignedLong() == true )
        {    // First time the window is being shown.
          ICanvas* pcvCanvas = (ICanvas*)evt.dispatchingWindow();
          IWindowHandle hwnd( evt.handle() );

          XSetWindowAttributes attributes;
          if ( pcvCanvas->isClassStyleAllPaints()  ||
               pcvCanvas->hasBorder() )
          {
             // Don't use bit gravity, so the canvas paints its entire
             // window whenever it is sized, including resized smaller.
             // This maintains backwards compatibility with ICanvas
             // behavior in past releases, and keeps the borders from
             // a previous paint from being left on the screen after a
             // partial paint.
             attributes.bit_gravity = ForgetGravity;
          }
          else
          {
             // Add bit gravity, so the canvas paints only invalidated
             // areas when it is resized. This is a performance
             // feature to reduce the number of paints that occur
             // occur during resizing.
             attributes.bit_gravity = StaticGravity;
          }
          unsigned long valuemask = CWBitGravity;
          XChangeWindowAttributes( XtDisplay( (Widget) hwnd ),
                                   XtWindow( (Widget) hwnd ),
                                   valuemask,
                                   &attributes );

          // Note that the bit gravity is also changed in
          // ICanvas::addBorder and removeBorder.
        }
        break;
      } /* endcase */

    default:
      break;
  }
  return bProcessed;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::layoutCanvas                                                 |
|                                                                              |
| Process a paint event by calling the canvas's layout() routine.              |
------------------------------------------------------------------------------*/
bool ICanvasHandler :: layoutCanvas ( IEvent& event )
{
   IMODTRACE_DEVELOP("ICanvasHandler::layoutCanvas");
   ICanvas* pcvCanvas = (ICanvas*)event.window();
   if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged))
   {
      pcvCanvas->layout();
   }
   return true;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::handleEventsFor                                              |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ICanvasHandler& ICanvasHandler :: handleEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::stopHandlingEventsFor                                        |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ICanvasHandler& ICanvasHandler :: stopHandlingEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::handleEventsFor                                              |
------------------------------------------------------------------------------*/
IHandler& ICanvasHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}

/*------------------------------------------------------------------------------
| ICanvasHandler::stopHandlingEventsFor                                        |
------------------------------------------------------------------------------*/
IHandler& ICanvasHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}
#endif // IC_MOTIF


/*------------------------------------------------------------------------------
| ICVKeyboardHandler::ICVKeyboardHandler                                       |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ICVKeyboardHandler :: ICVKeyboardHandler ( )
  : fCVKeyboardHandlerData( 0 ),
    ICVKeyboardHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::~ICVKeyboardHandler                                      |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ICVKeyboardHandler :: ~ICVKeyboardHandler ( )
{ }

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::dispatchHandlerEvent                                     |
|                                                                              |
| Router and default event processing for events.                              |
| Notes: Event processing includes:                                            |
|    -- WM_CHAR / WM_KEYDOWN / WM_SYSCHAR                                      |
|       (processes tab and cursor movement keys in a similar manner as is done |
|       for frame windows by PM)                                               |
|    -- WM_SETFOCUS                                                            |
|       (with IC_UM_CANVAS_SETFOCUS, sets input focus to a tab-able child      |
|       control)                                                               |
|    -- IC_UM_CANVAS_SETFOCUS                                                  |
|       (with WM_SETFOCUS, sets input focus to a tab-able child control)       |
------------------------------------------------------------------------------*/
bool ICVKeyboardHandler :: dispatchHandlerEvent ( IEvent & event )
{
   IMODTRACE_ALL("ICVKeyboardHandler::dispatchHandlerEvent");
   bool bProcessed = false;

   switch (event.eventId())
   {
#ifdef IC_MOTIF
     case xEvent(KeyPress):
#endif
#ifdef IC_PMWIN
      case WM_CHAR:
#ifdef IC_WIN
      case WM_KEYDOWN:
      case WM_SYSCHAR:
      // defect: 29278
      // Removed WM_SYSKEYDOWN from the above case list.  We do not want
      // this msg to be passed on since it will cause duplicate command
      // events to fire on mnemonic pushbuttons in the key() method.
#endif // IC_WIN
#endif // IC_PMWIN
      {
         IKeyboardEvent keyevt(event);
         bProcessed = key(keyevt);
         if (bProcessed)
         {                             // Key processed by canvas.
            event.setResult(keyevt.result());
         }
         break;
      } /* endcase */

#ifdef IC_PMWIN
      case WM_SETFOCUS:
      {                   // Assign to first/last tab-able control.
         ITRACE_ALL( IString( "WM_SETFOCUS hwnd=" ) +
                     IString( event.handle().asUnsigned() ).d2x() +
                     IString( " wParam=" ) +
                     IString( event.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( event.parameter2().asUnsignedLong() ).d2x() );
#ifdef IC_PM     // In Windows loss of focus is separate message
         if (event.parameter2().number1())
#endif // IC_PM
         {                             // Canvas window getting focus.
            /*******************************************************/
            /* Post the IC_UM_CANVAS_SETFOCUS message to avoid     */
            /* changing the window with the input focus during     */
            /* processing of a WM_SETFOCUS message.                */
            /*******************************************************/
            event.handle().postEvent(IC_UM_CANVAS_SETFOCUS, 0, 0);
            bProcessed = true;
         }
         break;
      } /* endcase */

      case IC_UM_CANVAS_SETFOCUS:
      {                                // From WM_SETFOCUS processing.
         ITRACE_ALL( IString( "IC_UM_CANVAS_SETFOCUS hwnd=" ) +
                     IString( event.handle().asUnsigned() ).d2x() +
                     IString( " wParam=" ) +
                     IString( event.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( event.parameter2().asUnsignedLong() ).d2x() );
         bProcessed = true;
         IWindow* pwndParent;
         IWindowHandle hwndFocus =
                   IQUERYFOCUS(IWindow::desktopWindow()->handle());
         bool bClientFocus = false;
         if (!(event.window()->isFrameWindow()))
            {
            bClientFocus =
              (event.window()->id() == FID_CLIENT  &&
               (pwndParent = event.window()->parent())  &&
               hwndFocus == pwndParent->handle()  &&
               pwndParent->isFrameWindow() );
            }
         if (hwndFocus == event.window()->handle()  ||
             bClientFocus )
         {                             // Canvas has input focus.
            /*******************************************************/
            /* Canvas still has the focus from when WM_SETFOCUS    */
            /* was detected.  We should ignore this                */
            /* IC_UM_CANVAS_SETFOCUS message if PM has already     */
            /* changed the window with the input focus since we    */
            /* posted this message--otherwise we will be changing  */
            /* the input focus incorrectly back to ourselves.      */
            /*                                                     */
            /* The one exception to the above is if the frame      */
            /* gained the input focus, but is letting the client   */
            /* process the notification (the frame is still the    */
            /* focus window in this case, but the canvas is sent a */
            /* WM_SETFOCUS notification).  Presumably this means   */
            /* the frame is not providing any special processing,  */
            /* and does not care if we take the focus away (so     */
            /* we will go ahead and take the focus away).          */
            /*******************************************************/
            bProcessed = gotFocus(event);
         }
         break;
      } /* endcase */
#endif // IC_PMWIN

      default:
         break;
   }  /* end switch */

   return bProcessed;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::gotFocus                                                 |
|                                                                              |
| Gives input focus to the appropriate child window.                           |
------------------------------------------------------------------------------*/
bool ICVKeyboardHandler :: gotFocus ( IEvent& event )
{
#ifdef IC_PMWIN
#ifdef IC_PM
   unsigned long ulKeyState =                // Check for a back-tab key.
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_BACKTAB);
#endif // IC_PM
#ifdef IC_WIN
   unsigned long ulKeyState =                // Check for a back-tab key.
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_TAB) &
      IGETKEYSTATE(IWindow::desktopWindow()->handle(), VK_SHIFT);
#endif //IC_WIN
   unsigned long ulTabCode = (ulKeyState & 0x8000) ?      // Key pressed?
                                EDI_LASTTABITEM : EDI_FIRSTTABITEM;
   /****************************************************************/
   /* No check is made for cursor movement keys (e.g. VK_UP),      */
   /* since the canvas is not assumed to belong to a group and no  */
   /* children in the canvas are assumed to belong to a group      */
   /* outside the canvas.                                          */
   /*                                                              */
   /* The use of WinEnumDlgItem here assumes that either all child */
   /* windows have been created with HWND_BOTTOM positioning, or   */
   /* the z-order of the child windows has been reversed by the    */
   /* canvas, so that the order that child windows are returned by */
   /* WinEnumDlgItem matches the expected order that the windows   */
   /* are layed out and tabbed through.                            */
   /****************************************************************/
   IWindow* pcvCanvas = event.window();
   IWindowHandle hwndNextFocus =
                          IENUMDLGITEM(event.handle(), 0, ulTabCode);
   if (hwndNextFocus == 0)        // No window to tab to.
   {                              // Let next sibling try.
#ifdef IC_WIN
      if ( pcvCanvas->isFrameWindow() )
      {
         // Windows only case.  There are no tabbable children of the
         // frame.  If there is a client, set focus to it.  If there
         // is no client, set focus to the first child. (as Windows dialogs
         // do)  Otherwise, punt.
         hwndNextFocus = IWINDOWFROMID( pcvCanvas->handle(), FID_CLIENT );
         if (hwndNextFocus == 0)
         {           // no client
            hwndNextFocus = GetWindow( pcvCanvas->handle(), GW_CHILD);
         }
      }
      else
#endif //IC_WIN
      {
         /*************************************************************/
         /* We will probably fall into here only for the case of a    */
         /* canvas with no child windows.  Try to find a sibling      */
         /* window that can take the input focus.                     */
         /*************************************************************/
         IWindow* pwndParent = pcvCanvas->parent();
         if (pwndParent)
         {                                // Has parent window.
            hwndNextFocus =
               IENUMDLGITEM(pwndParent->handle(),
                            event.handle(),
                            (ulTabCode == EDI_LASTTABITEM) ?
                                EDI_PREVTABITEM : EDI_NEXTTABITEM);
         }
      }
   }                              // End no window to tab to.

   if (hwndNextFocus)
   {                              // Have window to give focus to.
      ICVKeyboardHandler::buttonSetup(pcvCanvas, hwndNextFocus,
                                      ICVKeyboardHandler::enteringGroup);
#ifdef IC_WIN
      if (IEntryField::isTextSelectionEnabled()) {
         IWindow* efWindow = IWindow::windowWithHandle( hwndNextFocus, true );
         IEntryField* eField = dynamic_cast<IEntryField*>(efWindow);
         if ( eField )
         {
		   			eField->selectRange();
         }
	   	}
#endif
      if (hwndNextFocus != event.handle())
      {                           // Giving the focus to someone else.
         ISETFOCUS(IWindow::desktopWindow()->handle(),
                   hwndNextFocus);
#ifdef IC_PM
         // Note: On OS/2 there is a bug that entry fields do not highlight
         //       selected text when they get focus from a window outside
         //       of the canvas.  To fix this, we need to check for entry
         //       fields and force them to repaint
         IWindowClassName className( hwndNextFocus );
         if ( className == WC_ENTRYFIELD )
         {
            IINVALIDATERECT( hwndNextFocus, 0, true );
         }
#endif // IC_PM
      }
   }
#endif // IC_PMWIN
   return true;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::key                                                      |
|                                                                              |
| Process tabs and cursor movement keys.                                       |
| Notes: Process tab and cursor movement since PM's frame window procedure     |
|          will only set focus to a direct child window (e.g. a canvas window  |
|          instead of a control on the canvas, which is what is needed).       |
------------------------------------------------------------------------------*/
bool ICVKeyboardHandler :: key ( IKeyboardEvent& keyevt )
{
   bool bProcessed = false;

   if (!(keyevt.isUpTransition()))
   {                                   // Key press.
#ifdef IC_PMWIN
#ifdef IC_PM
      if ( keyevt.isCharacter() ||     // Character key (mnemonic?)
           (( keyevt.isAltDown()  ||  keyevt.isCtrlDown() )  &&
            keyevt.parameter2().number1() ))
#endif //IC_PM
#ifdef IC_WIN
      if ( ( keyevt.isCharacter()  ||  // Character key (mnemonic?)
             ( keyevt.isAltDown()  &&
               MapVirtualKey( keyevt.parameter1(), 2 ) ) )  &&
           ! keyevt.isCtrlDown() )
#endif // IC_WIN
      {                                // Data key or Alt+key or Ctrl+key.
         ITRACE_ALL( IString( "key - Mnemonic hwnd=" ) +
                     IString( keyevt.handle().asUnsigned() ).d2x() +
                     IString( " msg=" ) +
                     IString( keyevt.eventId() ).d2x() +
                     IString( " wParam=" ) +
                     IString( keyevt.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( keyevt.parameter2().asUnsignedLong() ).d2x() );
         /**********************************************************/
         /* Emulate PM's handling of accelerators (mnemonics for   */
         /* buttons, for example).                                 */
         /**********************************************************/
         IWindowHandle hwndMnemonic(0);
         IWindow* pcvCanvas = keyevt.window();
         unsigned short usMnemonic = (unsigned short)
#ifdef IC_PM
          (keyevt.parameter2().number1());
#endif //IC_PM
#ifdef IC_WIN
          (keyevt.isCharacter() ?
             keyevt.character() :
             MapVirtualKey( keyevt.parameter1(), 2 )) ;

        // if this window is disabled no further processing needed
         if (!pcvCanvas->isEnabled())
            {
            bProcessed = true;
            usMnemonic = 0;
            }
#endif // IC_WIN

         if (usMnemonic)
         {                             // Have a possible mnemonic.
            IWindow* pwndOwner;
            hwndMnemonic = pcvCanvas->matchForMnemonic(usMnemonic);
            if (hwndMnemonic == 0  &&
                (pwndOwner = pcvCanvas->owner())  &&
                pwndOwner->isFrameWindow()
#ifdef IC_WIN
                && (ISTYLEOF( pcvCanvas->handle() ) & WS_CHILD) // not top-level
#endif // IC_WIN
                )
            {                          // Try frame extensions also.
               IWindow::ChildCursor cursor(*pwndOwner);
               IWindowHandle hwndCanvas = keyevt.handle();
               IWindow* pwndSibling;
               for (cursor.setToFirst();
                    bProcessed == false  &&  cursor.isValid();
                    cursor.setToNext())
               {                       // Get children of frame.
                  IWindowHandle hwndSibling = pwndOwner->childAt(cursor);
                  if (hwndSibling != hwndCanvas  &&
                      (pwndSibling =
                         IWindow::windowWithHandle(hwndSibling))  &&
                      (hwndMnemonic =
                         pwndSibling->matchForMnemonic(usMnemonic)))
                  {                    // Button mnemonic matched.
                     bProcessed = true;  // Flag as done.
                  }
               }                       // End for all children of frame.
            }                          // End frame extension check.
         }                             // End character key.

         if ( hwndMnemonic  &&
              ( hwndMnemonic.sendEvent( WM_QUERYDLGCODE, 0, 0 )
                            .asUnsignedLong() & DLGC_STATIC ) )
         {  // The control is a static text.
            // Find the next sibling window with a tab stop.  If this
            // canvas/frame does not have a window with a tab stop following
            // it, then treat the mnemonic as invalid (rather than trying to
            // deal with finding a next window--either by wrapping back to
            // the first sibling or crossing canvas boundaries).
            IWindowHandle
              mnemonicParent( IPARENTOF( hwndMnemonic ) );
            IWindow
              *parentWindow = IWindow::windowWithHandle( mnemonicParent );

            if ( parentWindow )
            {
               bool found = false;
               IWindow::ChildCursor
                 cursor( *parentWindow );
               for ( cursor.setToFirst();
                     !found  &&  cursor.isValid();
                     cursor.setToNext() )
               {  // First find the static text control.
                  if ( parentWindow->childAt( cursor ) == hwndMnemonic )
                  {
                     // Now start the search for a tabbable sibling.
                     // Put the result into hwndMnemonic.
                     hwndMnemonic = 0;
                     for ( cursor.setToNext();
                           !found  &&  cursor.isValid();
                           cursor.setToNext() )
                     {
                        IWindowHandle
                          hwndChild( parentWindow->childAt( cursor ) );
                        if ( ( ISTYLEOF( hwndChild )
                                & ( WS_TABSTOP | WS_DISABLED | WS_VISIBLE ) )
                                     == ( WS_TABSTOP | WS_VISIBLE ) )
                        {  // Found a good sibling window.
                           hwndMnemonic = hwndChild;
                           found = true;
                        }
                     } // End inner loop.
                  }
               } // End outer loop.
            }
            else
            {  // Don't bother with this case.
               hwndMnemonic = 0;
            }
         }

         if (hwndMnemonic)
         {                             // Found a window for it.
            /********************************************************/
            /* Note: Don't worry about returning default emphasis   */
            /* to the original default push button for now for the  */
            /* case where you're going from a push button to a      */
            /* radio button or check box.                           */
            /********************************************************/
            //---------------------------------------------------
            // Call setfocus() AFTER buttonSetup()
            //---------------------------------------------------
            buttonSetup(pcvCanvas, hwndMnemonic, enteringGroup);
#ifdef IC_WIN
            if (IEntryField::isTextSelectionEnabled()) {
               IWindow* efWindow = IWindow::windowWithHandle( hwndMnemonic, true );
               IEntryField* eField = dynamic_cast<IEntryField*>(efWindow);
               if ( eField )
               {
			         		eField->selectRange();
               }
	         	}
#endif
            ISETFOCUS(IWindow::desktopWindow()->handle(),
                                              hwndMnemonic);

            // For non radio buttons, click the button.   buttonSetup
            // handles radio buttons.
            unsigned long ulCode = hwndMnemonic.sendEvent(WM_QUERYDLGCODE, 0, 0);
            if ( !(ulCode & DLGC_RADIOBUTTON) )
            {
#ifdef IC_PM
               hwndMnemonic.sendEvent(BM_CLICK, true, 0);
#endif //IC_PM
#ifdef IC_WIN
               clickButton(hwndMnemonic);
#endif //IC_WIN
            }

            bProcessed = true;         // Try selecting the window.
         }
      }
#endif // IC_PMWIN

      if (!bProcessed  &&  keyevt.isVirtual())
      {                                // Virtual key press/repeat-press.
         ITRACE_ALL( IString( "key - virtual hwnd=" ) +
                     IString( keyevt.handle().asUnsigned() ).d2x() +
                     IString( " msg=" ) +
                     IString( keyevt.eventId() ).d2x() +
                     IString( " wParam=" ) +
                     IString( keyevt.parameter1().asUnsignedLong() ).d2x() +
                     IString( " lParam=" ) +
                     IString( keyevt.parameter2().asUnsignedLong() ).d2x() );
         IKeyboardEvent::VirtualKey vkKey = keyevt.virtualKey();
         switch (vkKey)
         {
#ifdef IC_PMWIN
            case (IKeyboardEvent::tab):
            case (IKeyboardEvent::backTab):
               bProcessed = handleCursorMovement( keyevt );
               break;

            case (IKeyboardEvent::left):
            case (IKeyboardEvent::right):
            case (IKeyboardEvent::up):
            case (IKeyboardEvent::down):
            {
               // (IKeyboardEvent::virtualKey screens out NumLock case.)
#ifdef IC_WIN
               unsigned long ulCode = 0;
               IWindowHandle hwndFocus =
                     IQUERYFOCUS(IWindow::desktopWindow()->handle());
               if (hwndFocus)
                  ulCode = hwndFocus.sendEvent( WM_QUERYDLGCODE, 0, 0 );
               if ((~ulCode & DLGC_WANTARROWS) &&
                   (~ulCode & DLGC_WANTALLKEYS))
#endif // IC_WIN
               bProcessed = handleCursorMovement( keyevt );
               break;
            }

            case (IKeyboardEvent::enter):
            case (IKeyboardEvent::newLine):
               bProcessed = handleEnter(keyevt);
               break;
#endif // IC_PMWIN

#ifdef IC_MOTIFWIN
            // Standard Windows dialogs process Esc as if the cancel
            // button was pressed.
            case (IKeyboardEvent::esc):
               if (keyevt.window()->isFrameWindow())
                  {
                  // lParam is the ID of the Cancel button if there
                  // is one, 0 if not.
                  IWindowHandle hwndCancel = IWINDOWFROMID(
                                                keyevt.handle(), IC_ID_CANCEL);
                  keyevt.handle().postEvent( WM_COMMAND,
                                            IC_ID_CANCEL,
                                            hwndCancel.asUnsigned());
                  bProcessed = true;
                  }
               break;
#endif // IC_MOTIFWIN

            default:
               break;
         }  /* end switch */
      }
   }                                   // End key press.

   return bProcessed;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleCursorMovement                                     |
|                                                                              |
| Handles tabbing between groups and canvases and cursor movement within a     |
| group.                                                                       |
| Notes: This processing is necessary since PM's frame window procedure only   |
|          gives tabbing and cursoring between immediate child windows of the  |
|          frame, and we need to support tabbing and cursoring to child        |
|          windows of a canvas (which could be a child, or grandchild, or      |
|          great grandchildren, etc., of the frame).  However, where we are    |
|          tabbing from the last child or back tabbing from the first child of |
|          a canvas, we will let the owner window determine if there is a      |
|          siblign window that should get the input focus.                     |
|        All controls in a group are assumed to be contained in (children of)  |
|          the canvas.                                                         |
------------------------------------------------------------------------------*/
bool ICVKeyboardHandler :: handleCursorMovement ( IKeyboardEvent& keyevt )
{
   bool bProcessed = false;

#ifdef IC_PMWIN
   IWindow* pcvCanvas = keyevt.window();
   IWindowHandle hwndCanvas = keyevt.handle();
   IWindowHandle hwndFocus =
                   IQUERYFOCUS(IWindow::desktopWindow()->handle());
   if (hwndFocus  &&  IISCHILD(hwndFocus, hwndCanvas))
   {                              // Child with input focus.
      unsigned long ulNextControlType;
      IKeyboardEvent::VirtualKey vkKey = keyevt.virtualKey();
      switch (vkKey)
      {
         /**********************************************************/
         /* The use of WinEnumDlgItem here assumes that either all */
         /* child windows have been created with HWND_BOTTOM       */
         /* positioning, or the z-order of the child windows has   */
         /* been reversed by the canvas, so that the order that    */
         /* child windows are returned by WinEnumDlgItem matches   */
         /* the expected order that the windows are layed out and  */
         /* tabbed through.                                        */
         /**********************************************************/
         case (IKeyboardEvent::tab):
            ulNextControlType = EDI_NEXTTABITEM;
            break;

         case (IKeyboardEvent::backTab):
            ulNextControlType = EDI_PREVTABITEM;
            break;

         case (IKeyboardEvent::right):
         case (IKeyboardEvent::down):
            ulNextControlType = EDI_NEXTGROUPITEM;
            break;

         case (IKeyboardEvent::left):
         case (IKeyboardEvent::up):
            ulNextControlType = EDI_PREVGROUPITEM;
            break;

         default:                 // Code error.
            ITHROWGUIERROR2("IKeyboardEvent::VirtualKey",
                            IBaseErrorInfo::invalidParameter,
                            IException::recoverable);
            break;
      }  /* end switch */

                                  // Find next control in this canvas.
      IWindowHandle hwndNextControl =
            IENUMDLGITEM(hwndCanvas, hwndFocus, ulNextControlType);
      if (hwndNextControl)
      {                    // Need to change control with input focus.
         bProcessed = true;       // Assume staying in canvas.
         /**********************************************************/
         /*  Let the owner (another canvas or PM's frame window    */
         /*  procedure) handle the case of tabbing and back-       */
         /*  tabbing from this canvas (if passed to                */
         /*  WinDefWindowProc, the WM_CHAR will be passed up the   */
         /*  chain to the next owner).  This class must process    */
         /*  the WM_SETFOCUS message to set the input focus from   */
         /*  the canvas window to the appropriate child control    */
         /*  when the canvas window is tabbed to.                  */
         /**********************************************************/
         if (ulNextControlType == EDI_NEXTTABITEM  &&
             hwndNextControl ==
                     IENUMDLGITEM(hwndCanvas, 0, EDI_FIRSTTABITEM))
         {            // Really tabbing off the canvas (wrapped?).
            IWindowHandle hwndChild;
            IWindowHandle hwndParent = hwndFocus;

            do
            {                     // Find immediate child of canvas.
               hwndChild = hwndParent;
               hwndParent = IPARENTOF(hwndChild);
            }
            while (hwndParent != hwndCanvas);

            bool bDone = false;
            IWindow::ChildCursor cursor(*pcvCanvas);
            for (cursor.setToFirst();
                 bDone == false  &&  cursor.isValid();
                 cursor.setToNext())
            {
               IWindowHandle hwndTest = pcvCanvas->childAt(cursor);
               if (hwndTest == hwndNextControl)
               {        // Next tab item precedes control tabbed from.
                  bProcessed = false;  // Let owner handle this.

#ifdef IC_WIN
                  // Below is a Windows only exception.
                  // In Windows, this handler is used for the frame
                  // window as well.  If we are handling the frame then we
                  // don't necessarily want to punt.  Set focus to the child
                  // if the frame is not a Windows tab control page.
                  if (pcvCanvas->isFrameWindow())
                  {
                     IWindowHandle parent = IPARENTOF( hwndCanvas );
                     IWindowHandle grandParent = (parent) ? IPARENTOF( parent ) : 0;
                     bool isPageWindow = false;
                     if (grandParent)
                     {
                        IWindowClassName className( grandParent );
                        if (className == WC_TABCONTROL ||
                            className == WC_NOTEBOOK)
                           isPageWindow = true;
                     }
                     if (isPageWindow)
                     {
                        // The Window default frame window procedure will
                        // eat the WM_CHAR.  Go ahead and send the event
                        // to the tab control.
                        grandParent.postEvent( keyevt.eventId(),
                                               keyevt.parameter1(),
                                               keyevt.parameter2() );
                     }
                     else
                        bProcessed = true;
                  }
#endif // IC_WIN
                  bDone = true;        // Allow tab off the canvas.
               }
               else if (hwndTest == hwndChild)
               {          // Control tabbed from precedes next tab item.
                  bDone = true;        // Tabbing to hwndNextControl.
               }
            }
         }
         else if (ulNextControlType == EDI_PREVTABITEM  &&
                  hwndNextControl ==
                     IENUMDLGITEM(hwndCanvas, 0, EDI_LASTTABITEM))
         {                        // Back-tabbing off the canvas?
            IWindowHandle hwndChild;
            IWindowHandle hwndParent = hwndFocus;
            do
            {                     // Find immediate child of canvas.
               hwndChild = hwndParent;
               hwndParent = IPARENTOF(hwndChild);
            }
            while (hwndParent != hwndCanvas);

            IWindow::ChildCursor cursor(*pcvCanvas);
            bool bDone = false;

            for (cursor.setToFirst();
                 bDone == false  &&  cursor.isValid();
                 cursor.setToNext())
            {
               IWindowHandle hwndTest = pcvCanvas->childAt(cursor);
               if (hwndTest == hwndChild)
               {       // Control tabbed from precedes first tab item.
                  bProcessed = false;  // Let owner handle.

#ifdef IC_WIN
                  // Below is a Windows only exception.
                  // In Windows, this handler is used for the frame
                  // window as well.  If we are handling the frame then we
                  // don't necessarily want to punt.  Set focus to the child
                  // if the frame is not a Windows tab control page.
                  if (pcvCanvas->isFrameWindow())
                  {
                     IWindowHandle parent = IPARENTOF( hwndCanvas );
                     IWindowHandle grandParent = (parent) ? IPARENTOF( parent ) : 0;
                     bool isPageWindow = false;
                     if (grandParent)
                     {
                        IWindowClassName className( grandParent );
                        if (className == WC_TABCONTROL ||
                            className == WC_NOTEBOOK)
                           isPageWindow = true;
                     }
                     if (isPageWindow)
                     {
                        // The Window default frame window procedure will
                        // eat the WM_CHAR.  Go ahead and send the event
                        // to the tab control.
                        grandParent.postEvent( keyevt.eventId(),
                                               keyevt.parameter1(),
                                               keyevt.parameter2() );
                     }
                     else
                        bProcessed = true;
                  }
#endif // IC_WIN
                  bDone = true;        // Allow back-tab off canvas.
               }
               else if (hwndTest == hwndNextControl)
               {          // Next tab item precedes control tabbed from.
                  bDone = true;        // Back-tabbing to hwndNextControl.
               }
            }
         }

         /**********************************************************/
         /* Check for any fix-ups for buttons.                     */
         /**********************************************************/
         if (bProcessed == false)
         {                  // Leaving group (owner window to handle).
            buttonSetup(pcvCanvas, hwndFocus, leavingGroup);
         }
         else if (hwndNextControl != hwndFocus)
         {                  // New child is group getting input focus.
            //-------------------------------------------------------------------
            // Only force a click of radiobutton if moving within
            // an existing radiobutton group, not tabbing into the group.
            //-------------------------------------------------------------------
            bool bClick(false);
            if ( ulNextControlType == EDI_NEXTGROUPITEM ||
                 ulNextControlType == EDI_PREVGROUPITEM )
               bClick = true;
            buttonSetup(pcvCanvas, hwndNextControl, withinGroup, bClick);
#ifdef IC_WIN
            if (IEntryField::isTextSelectionEnabled()) {
               IWindow* efWindow = IWindow::windowWithHandle( hwndNextControl, true );
               IEntryField* eField = dynamic_cast<IEntryField*>(efWindow);
               if ( eField )
               {
			         		eField->selectRange();
               }
	         	}
#endif
            ISETFOCUS(IWindow::desktopWindow()->handle(),
                      hwndNextControl);
         }
      } /* endif have a tabbable child */
#ifdef IC_WIN
      else if (pcvCanvas->isFrameWindow())
      {
         // There is no tabbable child, and this is a frame.  Set focus
         // to the frame.  This will cause gotFocus to eventually
         // run for this window and set focus to the client or first
         // child.
         buttonSetup(pcvCanvas, hwndFocus, leavingGroup);
         ISETFOCUS(IWindow::desktopWindow()->handle(),
                   pcvCanvas->handle() );
         bProcessed = true;
      }
#endif //IC_WIN

   } /* endif */
#endif // IC_PMWIN

   return bProcessed;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEnter                                              |
|                                                                              |
| Process the Enter key by causing the default push button to be selected.     |
| Notes: The push button selected is the button current hilited as the default |
|          rather than the original default push button.                       |
------------------------------------------------------------------------------*/
bool ICVKeyboardHandler :: handleEnter ( IKeyboardEvent& keyevt )
{
   IMODTRACE_ALL("ICVKeyboardHandler::handleEnter");
   bool bProcessed = false;
#ifdef IC_PMWIN
   IWindow *pwndOwner;
   IWindow *pwndCanvas = keyevt.window();
   IWindowHandle hwndDefault = pwndCanvas->defaultEmphasisButton();

   if (hwndDefault == 0  &&
       (pwndOwner = pwndCanvas->owner())  &&
       (pwndOwner->isFrameWindow()))
   {                                   // Try frame extensions also.
      IWindow::ChildCursor cursor(*pwndOwner);
      IWindowHandle hwndCanvas = keyevt.handle();
      IWindow* pwndSibling;
      for (cursor.setToFirst();
           bProcessed == false  &&  cursor.isValid();
           cursor.setToNext())
      {
         IWindowHandle hwndSibling = pwndOwner->childAt(cursor);
         if (hwndSibling != hwndCanvas  &&
             (pwndSibling = IWindow::windowWithHandle(hwndSibling))  &&
             (hwndDefault = pwndSibling->defaultEmphasisButton()))
         {                             // A default push button found.
            bProcessed = true;         // Flag as done.
         }
      }                                // End for all children of frame.
   }                                   // End frame extension check.

#ifdef IC_WIN
   if ((hwndDefault) &&
       ((ISTYLEOF(hwndDefault) & (WS_DISABLED | WS_VISIBLE)) == WS_VISIBLE) )
#else
   if (hwndDefault)
#endif
   {                                   // A default push button found.
      clickButton(hwndDefault);
      bProcessed = true;               // Flag as done.
   }
#endif // IC_PMWIN
   return bProcessed;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICVKeyboardHandler::buttonSetup                                              |
|                                                                              |
| Handles cursor selection of radio buttons and default push button hiliting   |
| with cursor movement.                                                        |
------------------------------------------------------------------------------*/
void ICVKeyboardHandler :: buttonSetup (
                                        const IWindow* pcvCanvas,
                                        const IWindowHandle& hwndChild,
                                              State state,
                                              bool  clickRadio )
{
   IASSERTPARM(pcvCanvas != 0);

   /****************************************************************/
   /* Select the specified control if it is a radio button without */
   /* the BS_NOCURSORSELECT style.                                 */
   /****************************************************************/
   unsigned long ulCode = hwndChild.sendEvent(WM_QUERYDLGCODE, 0, 0);

   //-------------------------------------------------------------------
   // Only force a click of the radiobutton if requested.
   //-------------------------------------------------------------------
   if (clickRadio && (ulCode & DLGC_RADIOBUTTON)  &&
       state != ICVKeyboardHandler::leavingGroup)
   {                              // Child window is a radio button.
#ifdef   IC_PM
      if (!(IQUERYWINDOWULONG(hwndChild, QWL_STYLE) &
                                       BS_NOCURSORSELECT))
      {               // Processing for cursor selected radio buttons.
         hwndChild.sendEvent(BM_CLICK, true, 0);
      }
#endif   //IC_PM
#ifdef   IC_WIN
      IWindow* rbWindow = IWindow::windowWithHandle( hwndChild, false );
      const IRadioButton* rButton;
      rButton = (const IRadioButton*) 0;
      rButton = dynamic_cast<const IRadioButton*>(rbWindow);
      if ( rButton && rButton->allowsMouseClickFocus() )
      {
        clickButton(hwndChild);
      }
      else if( !rButton )
      {
        // Radio button is not wrappered, so check if it
        // has the autoSelect style.  If so, then we need
        // to select it.
        //
        if ( ISTYLEOF(hwndChild) & BS_AUTORADIOBUTTON
                             & (unsigned long)~(BS_RADIOBUTTON) )
        {
          IEventResult evt = hwndChild.sendEvent(BM_QUERYCHECK,
                             IEventParameter1(0),
                             IEventParameter2(0));
          if ( !evt.asUnsignedLong() )  // if not selected
          {
            clickButton(hwndChild);
          }
        }
      }
#endif   //IC_WIN
   }

   return;
}
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                          |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: handleEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                    |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( ICanvas* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}

#ifdef IC_MOTIFWIN
/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                          |
|                                                                              |
| Attach the handler to the frame window.                                      |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: handleEventsFor ( IFrameWindow* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::handleEventsFor(canvas);
   return *this;
}

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                    |
|                                                                              |
| Detach the handler from the frame window.                                    |
------------------------------------------------------------------------------*/
ICVKeyboardHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( IFrameWindow* canvas )
{
   IASSERTPARM(canvas != 0);
   Inherited::stopHandlingEventsFor(canvas);
   return *this;
}
#endif // IC_MOTIFWIN

/*------------------------------------------------------------------------------
| ICVKeyboardHandler::handleEventsFor                                          |
------------------------------------------------------------------------------*/
IHandler& ICVKeyboardHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}
/*------------------------------------------------------------------------------
| ICVKeyboardHandler::stopHandlingEventsFor                                    |
------------------------------------------------------------------------------*/
IHandler& ICVKeyboardHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}


#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICanvasStatics::matchForMnemonic                                             |
|                                                                              |
| Returns the first child window that uses the specified character as a        |
| mnemonic.                                                                    |
------------------------------------------------------------------------------*/
IWindowHandle ICanvasStatics::matchForMnemonic (
                                 const IWindow* window,
                                 unsigned short character )
{
  IWindowHandle hwndMnemonic( 0 );     // Assume no mnemonic match.

  bool bDone = false;                  // Flag for exiting loop.
  IWindow::ChildCursor cursor( *(IWindow*) window );

  for ( cursor.setToFirst();
        bDone == false  &&  cursor.isValid();
        cursor.setToNext() )
  {                                    // Check all child windows.
     IWindowHandle hwndChild = window->childAt( cursor );

     // Windows::we do WM_MATCHMNEMONIC ourselves (see iwindow2.cpp)
     unsigned long dlgCode = hwndChild.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong();

     // only do mnemonics on buttons and statics

#ifdef IC_PM
        if(
#endif // IC_PM
#ifdef IC_WIN
     if ( (dlgCode & DLGC_BUTTON || dlgCode & DLGC_RADIOBUTTON ||
                ((dlgCode & DLGC_STATIC) && ( (ISTYLEOF(hwndChild) & SS_NOPREFIX) == 0))) &&
#endif // IC_WIN
                ((ISTYLEOF(hwndChild) & WS_DISABLED) == 0) &&
        ( hwndChild.sendEvent( WM_MATCHMNEMONIC,
                               IEventParameter1(character,0),
                               IEventParameter2(0) ).asUnsignedLong() == true ))
     {                                 // A match for the mnemonic found.
        hwndMnemonic = hwndChild;      // Return this child window.
        bDone = true;                  // Can exit loop now.
     }
  }                                    // End for all child windows.

  if ( hwndMnemonic == 0 )
  {                          // No match among children of the canvas.
     for ( cursor.setToFirst();
           bDone == false  &&  cursor.isValid();
           cursor.setToNext() )
     {                                 // Check child canvases next.
        IWindow *pwndChild =
           IWindow::windowWithHandle( window->childAt( cursor ));

        if ( pwndChild  &&
             (hwndMnemonic = pwndChild->matchForMnemonic( character )))
        {                    // Found a canvas with a mnemonic match.
           bDone = true;               // Can exit loop now.
        } /* endif */
     }                                 // End for all child windows.
  }

  return hwndMnemonic;
}
#endif // IC_PMWIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvas::matchForMnemonic                                                    |
|                                                                              |
| Returns the first child window that uses the specified character as a        |
| mnemonic.                                                                    |
------------------------------------------------------------------------------*/
IWindowHandle ICanvasStatics::matchForMnemonic (
                                 const IWindow* window,
                                 unsigned short character )
{
  IWindowHandle hwndMnemonic(0);       // Assume no mnemonic match

  bool bDone = false;                  // Flag for exiting loop
  IWindow::ChildCursor cursor( *(IWindow *) window );

  for (cursor.setToFirst();
       bDone == false  &&  cursor.isValid();
       cursor.setToNext())
  {                                    // Check all child windows
     IWindowHandle hwndChild = window->childAt(cursor);

     if ( XtIsSubclass(hwndChild, xmCascadeButtonWidgetClass) ||
          XtIsSubclass(hwndChild, xmPushButtonWidgetClass)    ||
          XtIsSubclass(hwndChild, xmToggleButtonWidgetClass)   )
     {                                 // only these buttons have & use XmNmnemonic
                                       // No support yet for static controls.  IC_NOTYET
        KeySym ksChar = 0;
        XtVaGetValues(hwndChild, XmNmnemonic, &ksChar, NULL);
        if (ksChar==character)
        {                      // A match for the mnemonic found
           hwndMnemonic = hwndChild;   // Return this child window
           bDone = true;               // Can exit loop now
        }
     }          // endif XtIsSubclass
  }           // End for all child windows

  if (hwndMnemonic == 0)
  {                          // No match among children of this canvas
     for (cursor.setToFirst();
          bDone == false  &&  cursor.isValid();
          cursor.setToNext())
     {                                 // Check child canvases next
        IWindow *pwndChild = window->windowWithHandle(window->childAt(cursor));
        if (pwndChild) {
          hwndMnemonic = pwndChild->matchForMnemonic(character);
          if (hwndMnemonic != 0)     // Found a canvas with a default push button
            bDone = true;             // Can exit loop now
        }
     }        // End for all child windows
  }         // endif

  return hwndMnemonic;
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| ICanvasStatics::nextDialogItem                                               |
| Calls GetNextDlgGroupItem/GetNextDlgTabItem based on flag value.  Flag       |
| values are those used in PM WinEnumDlgItem function.                         |
| This is used instead of DM_GETDEFID so that it can be used in canvases as    |
| well as dialogs.                                                             |
------------------------------------------------------------------------------*/
IWindowHandle ICanvasStatics::nextDialogItem (
                                  const IWindowHandle& dialog,
                                  const IWindowHandle& child,
                                  unsigned long        flags )
{
   IMODTRACE_ALL( "ICanvasStatics::nextDialogItem " ) ;
   ITRACE_ALL( IString( " dialog=" ) +
                  IString( dialog.asUnsigned() ).d2x() +
                  IString( " child=" ) +
                  IString( child.asUnsigned() ).d2x() +
                  IString( " flags=" ) +
                  IString( flags ).d2x() );
   IWindowHandle::Value hwnd = 0;

#ifdef IC_WIN
   // Set up the initial child.
   HWND hwndChild = child;
   // Find the immediate child of the dialog window.
   if (hwndChild)
   {
      HWND parent = IPARENTOF(hwndChild);
      while ( (parent != dialog) && (parent) )
      {
         hwndChild = parent;
         parent = IPARENTOF(hwndChild);
      }
      if (!parent)
         hwndChild = 0;   // bogus child ... default to first
   }

   switch (flags)
   {
      case EDI_FIRSTTABITEM    :
      case EDI_LASTTABITEM     :
         {
            // Set up the direction.
            unsigned direction =
               (flags == EDI_FIRSTTABITEM)  ? GW_HWNDNEXT : GW_HWNDPREV;
            // Set up the first window.
            hwndChild = GetWindow( dialog, GW_CHILD );
            if ((hwndChild) && (direction == GW_HWNDPREV))
               hwndChild = GetWindow( hwndChild, GW_HWNDLAST);

            // Find the next or previous child with WS_TABSTOP set.
            while ( (hwndChild ) && (hwnd == 0) )
            {
               ITRACE_ALL( IString("hwndChild=") +
                           IString( (unsigned long)hwndChild).d2x() );
               if ((ISTYLEOF(hwndChild) & (WS_TABSTOP | WS_DISABLED | WS_VISIBLE))
                    ==
                   (WS_TABSTOP | WS_VISIBLE) )
               {
               //-----------------------------------------------------------------
               // DEFECT 29047 : only go to canvas if it has tabable fields
               //-----------------------------------------------------------------
               IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
               IWindowClassName className( hwndChild );
               if ( (className == IC_WC_IOC_CANVAS ||
                     className == IC_WC_IOC_CANVAS_AP)
                    && pwndChild->isTabStop() )
                  {
                  // static with a tabstop "should be" a canvas wnd
                  // So... only use this wnd IF it has a tabstop child
                  HWND hwndChildTab;
                  hwndChildTab = IENUMDLGITEM(hwndChild, 0, EDI_FIRSTTABITEM);
                  if (hwndChildTab)
                     hwnd = hwndChild;
                  }
               else
               //-----------------------------------------------------------------
                  hwnd = hwndChild;
               }

               hwndChild = GetWindow( hwndChild, direction);
            }
         }   // case
         break;
      case EDI_NEXTTABITEM     :
      case EDI_PREVTABITEM     :
         {
            // Set up the direction.
            unsigned direction =
               (flags == EDI_NEXTTABITEM) ? GW_HWNDNEXT : GW_HWNDPREV;
            if (hwndChild == 0)
            {
               // Get a child to verify at least 1.
               hwndChild = GetWindow( dialog, GW_CHILD );
            }
            // Set the starting point.
            HWND hwndStart = hwndChild;

            ITRACE_ALL( IString("hwndStart=") +
                        IString( (unsigned long)hwndStart).d2x() );
            // Find the next or previous child with WS_TABSTOP set.
            if (hwndStart)
            {
               do
               {
                  hwndChild = GetWindow(hwndChild, direction);
                  // Handle wrap around ....
                  if (hwndChild == 0)
                  {
                     // Wrapped
                     if (direction == GW_HWNDNEXT)
                        hwndChild = GetWindow( hwndStart, GW_HWNDFIRST);
                     else
                        hwndChild = GetWindow( hwndStart, GW_HWNDLAST);
                  }
                  ITRACE_ALL( IString("hwndChild=") +
                              IString( (unsigned long)hwndChild).d2x() );

                  if ((ISTYLEOF(hwndChild) & (WS_TABSTOP | WS_DISABLED | WS_VISIBLE))
                       ==
                      (WS_TABSTOP | WS_VISIBLE) )
                  {
                      //-----------------------------------------------------------
                      // DEFECT 29047 : only go to canvas if it has tabable fields
                      //-----------------------------------------------------------
                      IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
                      IWindowClassName className( hwndChild );
                      if ( (className == IC_WC_IOC_CANVAS ||
                            className == IC_WC_IOC_CANVAS_AP)
                           && pwndChild->isTabStop() )
                         {
                         // static with a tabstop "should be" a canvas wnd
                         // So... only use this wnd IF it has a tabstop child
                         HWND hwndChildTab;
                         hwndChildTab = IENUMDLGITEM(hwndChild, 0, EDI_FIRSTTABITEM);
                         if (hwndChildTab)
                            hwnd = hwndChild;
                         }
                      else
                      //-----------------------------------------------------------
                         hwnd = hwndChild;
                  }
               }
               while ( (hwndChild != hwndStart) && (hwnd == 0) );

            }  // if

         }   // case
         break;
      case EDI_FIRSTGROUPITEM  :
         hwnd = GetNextDlgGroupItem( dialog, 0, false );
         break;
      case EDI_LASTGROUPITEM   :
         hwnd = GetNextDlgGroupItem( dialog, 0, true );
         break;
      case EDI_NEXTGROUPITEM   :
         hwnd = GetNextDlgGroupItem( dialog, hwndChild, false );
         break;
      case EDI_PREVGROUPITEM   :
         hwnd = GetNextDlgGroupItem( dialog, hwndChild, true );
         break;
   }
#endif // IC_WIN
#ifdef IC_PM

   if (! IISWINDOWENABLED(dialog) )
   {
       return hwnd;
   }

   //---------------------------------------------------
   // Do not return WinEnumDlgItem result for TAB items
   // if the hwnd is disabled
   //---------------------------------------------------
   switch (flags)
   {
      case EDI_FIRSTTABITEM    :
      case EDI_LASTTABITEM     :
         {
            HWND hwndChild, hwndWrap;
            unsigned long newFlags;
            newFlags = (flags == EDI_FIRSTTABITEM) ? EDI_NEXTTABITEM : EDI_PREVTABITEM;

            hwndWrap = hwndChild = WinEnumDlgItem(dialog, child, flags);
            while ( (hwndChild ) && (hwnd == 0) )
            {
               if ( IISWINDOWENABLED(hwndChild) )
               {
                  hwnd = hwndChild;
               }

               if (hwnd==0)   // not yet found; continue searching list for a tabstop child
               {
                  hwndChild = WinEnumDlgItem(dialog, child, newFlags);
                  if (hwndChild == hwndWrap) // wrapped around, none found.
                     hwndChild = 0;
               }
            }
         }
         break;

      case EDI_NEXTTABITEM    :
      case EDI_PREVTABITEM     :
         {
            HWND hwndChild, hwndWrap;

            hwndWrap = hwndChild = WinEnumDlgItem(dialog, child, flags);
            if ( hwndChild == child ) {   // wrapped already... NO next/prev tab...
               hwndChild = 0;
            }
            while ( (hwndChild ) && (hwnd == 0) )
            {
               if ( IISWINDOWENABLED(hwndChild) )
               {
                   hwnd = hwndChild;
               }

               if (hwnd==0)   // not yet found; continue searching list for a tabstop child
               {
                  hwndChild = WinEnumDlgItem(dialog, child, flags);
                  if (hwndChild == hwndWrap) // wrapped around, none found.
                     hwndChild = 0;
               }
            }
         }
         break;

      case EDI_FIRSTGROUPITEM  :
      case EDI_LASTGROUPITEM   :
      case EDI_NEXTGROUPITEM   :
      case EDI_PREVGROUPITEM   :
         hwnd = WinEnumDlgItem(dialog, child, flags);
         break;
   }
#endif //IC_PM
   ITRACE_ALL(("hwnd=") + IString((unsigned long)hwnd).d2x() );
   return hwnd;
}
