#ifndef ARM_H_INCLUDED
#define ARM_H_INCLUDED

/****************************************************************************/
/* arm.h - ARM API Definitions                                              */
/****************************************************************************/

#include <sys/types.h>     /* C types  definitions */

/*  Type definitions for various field sizes */

/*  64-bit integer compiler support                                         */
/*                                                                          */
/*  If a type declaration supporting 64 bit integer arithmatic is defined   */
/*  for the target platform and compiler, the "ARM_INT64" #define should be */
/*  set to that type declaration. E.g.,                                     */
/*                                                                          */
/*         #define ARM_INT64 long long   (for unix)                         */
/*                  or                                                      */
/*         #define ARM_INT64 _int64      (for NT)                           */
/*                                                                          */
/*  If 64 bit arithmetic is not supported on the target platform or         */
/*  compiler, remove (or comment out) the "ARM_INT64" #define so that the   */
/*  structures of two 32 bit values will be defined for the 64 bit fields.  */
/*                                                                          */
/*  Little endian support                                                   */
/*                                                                          */
/*  On hardware platforms which are little endian (such as x86 platforms),  */
/*  if 64 bit arithmetic is not supported, the "LITTLE_ENDIAN" #define      */
/*  should be set to 1.  This  will ensure that the upper and lower words   */
/*  of the 64 bit value are stored in the correct order.                    */

/*
#define ARM_INT64 long long
*/

/* 
#define LITTLE_ENDIAN  1
*/


typedef unsigned char  arm_bit8_t;
typedef short arm_int16_t;
typedef long  arm_int32_t;
typedef unsigned char  arm_unsigned8_t;
typedef unsigned short arm_unsigned16_t;
typedef unsigned long  arm_unsigned32_t;

#ifdef ARM_INT64
typedef ARM_INT64 arm_int64_t;
typedef unsigned ARM_INT64 arm_unsigned64_t;
#else
#ifdef LITTLE_ENDIAN
typedef struct arm_int64_t {
    arm_int32_t   lower;
    arm_int32_t   upper;
} arm_int64_t;

typedef struct arm_unsigned64 {
    arm_unsigned32_t   lower;
    arm_unsigned32_t   upper;
} arm_unsigned64_t;
#else
typedef struct arm_int64_t {
    arm_int32_t   upper;
    arm_int32_t   lower;
} arm_int64_t;

typedef struct arm_unsigned64 {
    arm_unsigned32_t   upper;
    arm_unsigned32_t   lower;
} arm_unsigned64_t;
#endif
#endif



/***    Symbol definitions    ***/


/*  Enumeration of transaction status completion codes */

enum arm_tran_status_e { ARM_GOOD = 0, ARM_ABORT, ARM_FAILED };


/*  Enumeration of user data formats */

enum arm_userdata_e { ARM_Format1 = 1, ARM_Format2, ARM_Format101 = 101 };


/*  Enumeration of metric types */
 
typedef enum arm_metric_type_e {
        ARM_Counter32 = 1, ARM_Counter64, ARM_CntrDivr32,
        ARM_Gauge32, ARM_Gauge64, ARM_GaugeDivr32, ARM_NumericID32,
        ARM_NumericID64, ARM_String8, ARM_String32,
        ARM_MetricTypeLast 
} arm_metric_type_e;
 

/***	Data definitions    ***/


/*  User metric structures */

typedef struct arm_cntrdivr32_t {		/*  Counter32 + Divisor32  */
    arm_unsigned32_t  count;
    arm_unsigned32_t  divisor;
} arm_cntrdivr32_t;

typedef struct arm_gaugedivr32_t {		/*  Gauge32 + Divisor32  */
    arm_int32_t       gauge;
    arm_unsigned32_t  divisor;
} arm_gaugedivr32_t;


/*  Union of user ARM_Format1 metric types */

typedef union arm_user_metric1_u {
    arm_unsigned32_t  counter32;       /*  Counter32  */
    arm_unsigned64_t  counter64;       /*  Counter64  */
    arm_cntrdivr32_t  cntrdivr32;      /*  Counter32 + Divisor32  */
    arm_int32_t	      gauge32;         /*  Gauge32  */
    arm_int64_t	      gauge64;         /*  Gauge64  */
    arm_gaugedivr32_t gaugedivr32;     /*  Gauge32 + Divisor32  */
    arm_unsigned32_t  numericid32;     /*  NumericID32  */
    arm_unsigned64_t  numericid64;     /*  NumericID64  */
    char              string8[8];      /*  String8  */
} arm_user_metric1_u;


/*  Application view of correlator */

typedef struct arm_app_correlator_t {
    arm_int16_t length;            /*  Length of the correlator  */
    char        agent_data[166];   /*  Agent-specific data fields  */
} arm_app_correlator_t;


/*  User metrics ARM_Format1 structure definition */

typedef struct arm_user_data1_t {
    arm_int32_t          format;        /*  Version/format id (userdata_e)  */
    arm_bit8_t           flags[4];      /*  Flags for metrics' presence  */
    arm_user_metric1_u   metric[6];     /*  User metrics  */
    char                 string32[32];  /*  32 byte non-terminated string  */
    arm_app_correlator_t correlator;    /*  Correlator  */
} arm_user_data1_t;


/*  User metrics ARM_Format2 structure definition */

typedef struct arm_user_data2_t {
    arm_int32_t   format;               /*  Version/format id (userdata_e)  */
    char          string1020[1020];     /*  1020 byte opaque blob  */
} arm_user_data2_t;


/*  User metric meta-data for ARM_Format101 structure */

typedef struct arm_user_meta101_t {
    arm_int32_t type;       /*  Type of metric (arm_user_metric_e)  */
    char        name[44];   /*  NULL-terminated string <= 44 char  */
} arm_user_meta101_t;


/*  User meta-data ARM_Format101 structure definition */

typedef struct arm_user_data101_t {
    arm_int32_t        format;     /* Version/format id (userdata_e)  */
    arm_bit8_t         flags[4];   /*  Flags for which fields are present  */
    arm_user_meta101_t meta[7];	   /*  User metrics meta-data  */
} arm_user_data101_t;


/*  Flag bit definitions (within bit8 fields) */

/*  flags[0] in arm_user_data1_t passed in arm_start */
#define ARM_CorrPar_f   0x80    /*  Correlator from parent  */
#define ARM_CorrReq_f   0x40    /*  Request correlator generation  */
#define ARM_CorrGen_f   0x20    /*  New correlator generated in data  */
#define ARM_TraceReq_f  0x10    /*  User trace request  */

/*  flags[1] in arm_user_data101_t passed in arm_get_id and */
/*  flags[1] in arm_user_data1_t passed in arm_start, arm_update and arm_end */
#define ARM_Metric1_f    0x80   /*  Metric 1 present  */
#define ARM_Metric2_f    0x40   /*  Metric 2 present  */
#define ARM_Metric3_f    0x20   /*  Metric 3 present  */
#define ARM_Metric4_f    0x10   /*  Metric 4 present  */
#define ARM_Metric5_f    0x08   /*  Metric 5 present  */
#define ARM_Metric6_f    0x04   /*  Metric 6 present  */
#define ARM_AllMetrics_f 0xfc   /*  Metrics 1 - 6 present  */
#define ARM_String1_f    0x02   /*  String 1 present  */


#if defined  _WIN32
 #include <windows.h>
 #define ARM_API WINAPI
#elif defined __OS2__
 #define ARM_API _Pascal
#elif defined _OS216
 #define  arm_data_t char _far
 #define  arm_ptr_t  char _far
 #define ARM_API _far _pascal
#elif defined _WIN16 || _WINDOWS
 #include <windows.h>
 typedef BOOL (FAR PASCAL _export * FPSTRCB) (LPSTR, LPVOID);
 #define  arm_data_t char FAR
 #define  arm_ptr_t  char FAR
 #define ARM_API WINAPI
#else  /* unix */
#define ARM_API
#endif


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef _PROTOTYPES

/***    Function prototypes    ***/

extern arm_int32_t ARM_API arm_init(
    char*        appl_name,     /* application name              */
    char*        appl_user_id,  /* Name of the application user  */
    arm_int32_t  flags,         /* Reserved = 0                  */
    char*        data,          /* Reserved = NULL               */
    arm_int32_t  data_size);    /* Reserved = 0                  */


extern arm_int32_t ARM_API arm_getid(
    arm_int32_t  appl_id,       /* application handle            */
    char*        tran_name,     /* transaction name              */
    char*        tran_detail,   /* transaction additional info   */
    arm_int32_t  flags,         /* Reserved = 0                  */
    char*        data,          /* format definition of user metrics */
    arm_int32_t  data_size);    /* length of data buffer         */


extern arm_int32_t ARM_API arm_start(
    arm_int32_t  tran_id,       /* transaction name identifier   */
    arm_int32_t  flags,         /* Reserved = 0                  */
    char*        data,          /* user metrics data             */
    arm_int32_t  data_size);    /* length of data buffer         */


extern arm_int32_t ARM_API arm_update(
    arm_int32_t  start_handle,  /* unique transaction handle     */
    arm_int32_t  flags,         /* Reserved = 0                  */
    char*        data,          /* user metrics data             */
    arm_int32_t  data_size);    /* length of data buffer         */


extern arm_int32_t ARM_API arm_stop(
    arm_int32_t  start_handle,  /* unique transaction handle     */
    arm_int32_t  tran_status,   /* Good=0, Abort=1, Failed=2     */
    arm_int32_t  flags,         /* Reserved = 0                  */
    char*        data,          /* user metrics data             */
    arm_int32_t  data_size);    /* length of data buffer         */


extern arm_int32_t ARM_API arm_end(
    arm_int32_t  appl_id,       /* application id                */
    arm_int32_t  flags,	        /* Reserved = 0                  */
    char*        data,          /* Reserved = NULL               */
    arm_int32_t  data_size);    /* Reserved = 0                  */


#else  /* _PROTOTYPES */

extern arm_int32_t   ARM_API arm_init();
extern arm_int32_t   ARM_API arm_getid();
extern arm_int32_t   ARM_API arm_start();
extern arm_int32_t   ARM_API arm_update();
extern arm_int32_t   ARM_API arm_stop();
extern arm_int32_t   ARM_API arm_end();

#endif /* _PROTOTYPES */

#ifdef __cplusplus
}
#endif /* __cplusplus */


/*  Type definitions for compatibility with version 1.0 of the ARM API */

typedef arm_int32_t        arm_appl_id_t;

typedef arm_int32_t        arm_tran_id_t;

typedef arm_int32_t        arm_start_handle_t;

typedef arm_unsigned32_t   arm_flag_t;

typedef char               arm_data_t;

typedef arm_int32_t        arm_data_sz_t;

typedef char               arm_ptr_t;

typedef arm_int32_t        arm_ret_stat_t;

typedef arm_int32_t        arm_status_t;


#endif /* ARM_H_INCLUDED */


