/****************************************************************************/
/*                                                                          */
/*                         GNAT COMPILER COMPONENTS                         */
/*                                                                          */
/*                              A - R A I S E                               */
/*                                                                          */
/*                          C Implementation File                           */
/*                                                                          */
/*                            $Revision: 1.52 $                             */
/*                                                                          */
/*          Copyright (C) 1992-1999, Free Software Foundation, Inc.         */
/*                                                                          */
/* GNAT is free software;  you can  redistribute it  and/or modify it under */
/* terms of the  GNU General Public License as published  by the Free Soft- */
/* ware  Foundation;  either version 2,  or (at your option) any later ver- */
/* sion.  GNAT is distributed in the hope that it will be useful, but WITH- */
/* OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY */
/* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License */
/* for  more details.  You should have  received  a copy of the GNU General */
/* Public License  distributed with GNAT;  see file COPYING.  If not, write */
/* to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, */
/* MA 02111-1307, USA.                                                      */
/*                                                                          */
/* As a  special  exception,  if you  link  this file  with other  files to */
/* produce an executable,  this file does not by itself cause the resulting */
/* executable to be covered by the GNU General Public License. This except- */
/* ion does not  however invalidate  any other reasons  why the  executable */
/* file might be covered by the  GNU Public License.                        */
/*                                                                          */
/* GNAT was originally developed  by the GNAT team at  New York University. */
/* It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). */
/*                                                                          */
/****************************************************************************/

/* Routines to support runtime exception handling */

/* ??? We need this to define malloc on those machines that need it, but
   this is the wrong file when this is built for libgnat.a.  */

#ifdef __alpha_vxworks
#include "vxWorks.h"
#endif

#include "config.h"
#include "a-ada.h"
#include "a-types.h"
#include "a-raise.h"
#include <stdio.h>

#define Lock_Task system__soft_links__lock_task
extern void (*Lock_Task) (void);

#define Unlock_Task system__soft_links__unlock_task
extern void (*Unlock_Task) (void);

#define Print_Unhandled_Exception ada__exceptions__print_unhandled_exception
extern void Print_Unhandled_Exception (void);

#define Raise_With_C_Msg ada__exceptions__raise_with_c_msg
extern void Raise_With_C_Msg (Exception_Id, char *);

extern struct Exception_Data constraint_error;
extern struct Exception_Data program_error;
extern struct Exception_Data storage_error;

/*  We have not yet figured out how to import this directly */

void
_gnat_builtin_longjmp (ptr, flag)
     void *ptr;
     int flag;
{
   __builtin_longjmp (ptr, flag);
}

/* When an exception is raised for which no handler exists, the procedure
   Ada.Exceptions.Unhandled_Exception is called, which performs the call
   to adafinal to complete finalization, and then prints out the error
   messages for the unhandled exception. The final step is to call this
   routine, which performs any system dependent cleanup required */

void
__gnat_unhandled_terminate ()
{
/* Special termination handling for VMS */

#ifdef VMS
    {
      long prvhnd;

      /* Remove the exception vector so it won't intercept any errors
	 in the call to exit, and go into and endless loop */

      SYS$SETEXV (1, 0, 3, &prvhnd);
      os_exit (1);
    }

/* Termination handling for all other systems. */

#elif !defined (__RT__)
    os_exit (1);
#endif
}

#if defined (WINNT) || defined (__INTERIX)
/* The following defines functions to limit the amount
   of used memory for systems that do not have OS support
   for that. The amount of available memory available for
   dynamic allocation is limited by setting the environment
   variable GNAT_MEMORY_LIMIT to the number of kilobytes
   that can be used.
*/

static __SIZE_TYPE__ available_memory = 0;
/*  Amount of memory that is available for heap allocations.
    A value of 0 means that the amount is not yet initialized */

#define MSIZE_ACCURACY 4096
/* This defines the amount of memory to add to requested allocation
   sizes, because malloc may return a bigger block than requested.
   As msize is used when free-ing, it must be used on allocation as
   well. To prevent underflow of available_memory we need to use
   a reserve.
*/

/* This routine must be called while having the task lock.
   When the memory limit is not yet initialized, it will
   be set to the value of the GNAT_MEMORY_LIMIT environment
   variable or to unlimited if that does not exist.

   If the size is larger than the amount of available memory,
   the task lock will be freed and a storage_error exception will
   be raised.
*/

static void check_available_memory (size)
   __SIZE_TYPE__ size;
{
   if (available_memory == 0)
   {
      /* The amount of available memory hasn't been initialized yet */
      char *gnat_memory_limit;
      gnat_memory_limit = (char *) getenv ("GNAT_MEMORY_LIMIT");
      if (gnat_memory_limit != 0) {
         available_memory = atoi (gnat_memory_limit) * 1024 + MSIZE_ACCURACY;
      } else {
         available_memory = (__SIZE_TYPE__) -1;
      }
   }

   if (size >= available_memory)
   {
      /* There is a memory overflow */
      (*Unlock_Task) ();
      Raise_With_C_Msg (&storage_error, "heap memory limit exceeded");
   }
}

#endif

/* Linux modules don't have malloc/free but provide a similar interface with
   kmalloc/kfree. */

#ifdef __RT__

void *
malloc (size)
     __SIZE_TYPE__ size;
{
  extern void *kmalloc();

  return kmalloc (size);
}

void
free (void *ptr);
{
  kfree (ptr);
}

void *
realloc (void *ptr, __SIZE_TYPE__ size)
{
  /* Currently do not support realloc. */
  return 0;
}

#endif

/* malloc for use by GNAT, with error checking and task lockout. */

void *
__gnat_malloc (size)
     __SIZE_TYPE__ size;
{
  void *result;

  if (size == (__SIZE_TYPE__) -1)
    Raise_With_C_Msg (&storage_error, "object too large");

  /* Change size from zero to non-zero. We still want a proper pointer
     for the zero case because pointers to zero length objects have to
     be distinct, but we can't just go ahead and allocate zero bytes,
     since some malloc's return zero for a zero argument. */

  if (size == 0)
    size = 1;

  (*Lock_Task) ();

#if defined (WINNT) || defined (__INTERIX)
  if (size + MSIZE_ACCURACY >= available_memory)
     check_available_memory (size + MSIZE_ACCURACY);
#endif

  result = (char *) malloc (size);

#if defined (WINNT) || defined (__INTERIX)
  if (result != 0)
     available_memory = available_memory - _msize (result);
#endif

  (*Unlock_Task) ();

  if (result == 0)
    Raise_With_C_Msg (&storage_error, "heap exhausted");

  return result;
}

/* free for use by GNAT, with task lockout */

void
__gnat_free (void *ptr)
{
  (*Lock_Task) ();

#if defined (WINNT) || defined (__INTERIX)
  if (ptr != 0)
     available_memory = available_memory + _msize (ptr);
#endif

  free (ptr);
  (*Unlock_Task) ();
}

/* realloc for use by GNAT, with error checking and task lockout. */

void *
__gnat_realloc (void *ptr, __SIZE_TYPE__ size)
{
  void *result;
#if defined (WINNT) || defined (__INTERIX)
  __SIZE_TYPE__ old_size;
#endif

  if (size == (__SIZE_TYPE__) -1)
    Raise_With_C_Msg (&storage_error, "object too large");

  (*Lock_Task) ();

#if defined (WINNT) || defined (__INTERIX)
  old_size = _msize (ptr);

  /* conservative check - no need to try to be precise here */
  if (size + MSIZE_ACCURACY >= available_memory)
     check_available_memory (size + MSIZE_ACCURACY);
#endif

  result = (void *) realloc (ptr, size);

#if defined (WINNT) || defined (__INTERIX)
    if (result != 0)
       available_memory = available_memory + old_size - _msize (ptr);
#endif

  (*Unlock_Task) ();

  if (result == 0)
    Raise_With_C_Msg (&storage_error, "heap exhausted");

  return result;
}
