
%PROLOG%

/* UI Class Library headers */
#include <iaccel.hpp>
#include <ititle.hpp>
#include <iapp.hpp>
#include <iinfoa.hpp>
#include <ifont.hpp>
#include <ithread.hpp>
#include <ifiledlg.hpp>
#include <imsgbox.hpp>
#include <ihelp.hpp>
#include <iexcept.hpp>

/* OS/2 Headers */
#define INCL_DOSERRORS
#define INCL_WINHELP        //  For HM_TUTORIAL message
#include <os2.h>

/* Application headers */
#include "openfile.hpp"
#include "prodinfo.hpp"
#include "msgbox.hpp"
#include "%FILE_NAME%.h"
#include "%FILE_NAME%.hpp"


/**************************************************************/
/* main() program entry point                                 */
/**************************************************************/
void main()
{
   AppWindow appWindow(IDW_FRAME_WINDOW);
   IApplication::current().run();
}


/**************************************************************/
/* Application main window constructor.  Sets events handler, */
/* icon, window style and menu bar.  It also sets static text */
/* as the client area control.  You may want to set           */
/**************************************************************/
AppWindow :: AppWindow(unsigned long windowId)
   : IFrameWindow( IFrameWindow::defaultStyle() |   // Create frame window in default style
                   IFrameWindow::accelerator,    
                   windowId), 
     canvas(ID_CANVAS,this,this),            // Create canvas for client area
     menuBar(this),                          // Create menu bar
     openFileName(EMPTY_STRING),             // No file open yet
     hasHelp(true),                          // Assume help is available for now
     statusLine (IDT_STATUS_LINE,this,this)  // Create status line
{

   // Set up status line
   statusLine.setText( IDS_GENERAL_DESC);             // Set status line text
   addExtension(&statusLine,                          // Extend client window by
                IFrameWindow::belowClient,            // placing status area below 
                (IFont(&statusLine).maxCharHeight() + 3));

   // Set up menu bar
   menuBar.setMenu( IResourceId(IDW_FRAME_WINDOW));  

   // Set up client area canvas
   setClient(&canvas);       // Set canvas control for client area

   // Set self as command event handler
   ICommandHandler::handleEventsFor(this);

   // Set application icon
   setIcon(IDI_ICON);     

   try
   {
     // Set up help, help settings
     IHelpWindow::Settings helpSettings;
     helpSettings.setMenuBar(IDM_HELP);
     helpSettings.setTutorial(TUTORIAL_FILE);
     helpSettings.setTitle(IDS_HELP_TITLE);
     helpSettings.setHelpTable(IDH_MAIN_HELPTABLE);
     helpSettings.setLibraries(HELP_FILE_NAME);
     help = new IHelpWindow(helpSettings,this);

     // Create and set help event handler
     helpHdlr = new AppHelpHandler;
     helpHdlr->handleEventsFor(this);
   }
   catch( IAccessError &exc )
   {
      // Send this window an event indicating that the help could not be loaded
      IEventParameter1 evtParm1( UM_NOHELP );
      this->postEvent( IWindow::command, evtParm1);
      // Set internal help flag to indicate that no help is available
      hasHelp = false;
   }

   // Set focus to this window and show
   setFocus();               
   show();                  
}


/***********************************************************/
/* Application main window destructor.  It stops the event */
/* handling and deletes any objects on the heap.           */
/***********************************************************/
AppWindow :: ~AppWindow ()
{
   ICommandHandler::stopHandlingEventsFor(this);
   helpHdlr->stopHandlingEventsFor(this);
   if (hasHelp) 
   {
      delete help;
      delete helpHdlr;
   }
}


/***********************************************************/
/* AppWindow main window command event handler             */
/***********************************************************/
Boolean AppWindow :: command(ICommandEvent &cmdEvent)
 {
   IResourceLibrary  resLib;
   IThread*          openThread;
   IThreadFn*        openFn;


   switch (cmdEvent.commandId()) 
   {
      case IDM_FILE_OPEN:
       {
        /*-------------------------------------------------------------*/
        /* When the user selects the File->Open menu item to open a    */
        /* file, display the standard "File open" dialog using         */
        /* IFileDialog.  The dialog is set to its most previous        */
        /* values.  Then, a thread is started to open the file and     */
        /* and perform any other application-specific file processing. */
        /*-------------------------------------------------------------*/

         IFileDialog::Settings fdSettings;  // Create file dialog settings object
         fdSettings.setOpenDialog();
         fdSettings.setOKButtonText(IResourceId(IDS_OPEN));
         fdSettings.setTitle(IResourceId(IDS_FILE_OPEN));
         if ( isFileOpen() )
            fdSettings.setFileName("*.*");
         else 
            fdSettings.setFileName(openFileName);

         // Create and associate file dialog help handler. Pass the
         // help panel resource ID for "File open".
         FileDlgHelpHandler fdHelpHdlr ( help, IDH_FILEOPEN_RESNO );
         fdHelpHdlr.handleEventsFor(this);

         // Create and show File open dialog 
         IFileDialog fileDlg (desktopWindow(),         // Parent is desktop
                              this,                    // Owner is main window
                              IFileDialog::helpButton, // Help button style
                              fdSettings);             // Here are my settings

         // Stop special file dialog help handler
         fdHelpHdlr.stopHandlingEventsFor(this);

         // Check results
         if ( fileDlg.pressedOK() )
          {
            openFileName = fileDlg.fileName();         // Store selected file name

            // Create secondary thread to handle file open operation 
            openFn = new AppOpenFileFn(this);
            openThread = new IThread(openFn);

          } 

         return true;
       }


      case IDM_FILE_SAVE:
       {
         // Write to status area
         IString tmpText = resLib.loadString(IDS_SAVE_FILE);
         tmpText.change( "%s", 
                         openFileName,            
                         1, 1);
         writeStatus (tmpText);
         return true;
       }


      case IDM_FILE_SAVEAS:
       {
        /*-------------------------------------------------------------*/
        /* When the user selects the "File->Save as" menu item,        */
        /* display the standard "Save as" dialog using IFileDialog.    */
        /* Use the IFileDialog::Settings class to specify initial      */
        /* settings, such as the currently open file.                  */
        /*-------------------------------------------------------------*/

         // Check if there is a file open and report error or write status
         if (writeStatus (IResourceId(IDS_SAVEAS)))
          {
            // Specify Save as dialog settings
            IFileDialog::Settings fdSettings;  
            fdSettings.setFileName(openFileName);
            fdSettings.setSaveAsDialog();
            fdSettings.setOKButtonText(IResourceId(IDS_SAVEAS));
            fdSettings.setTitle(IResourceId(IDS_SAVEAS));
            
            // Create and associate file dialog help handler. Pass the
            // help panel resource ID for "Save as".
            FileDlgHelpHandler fdHelpHdlr ( help, IDH_SAVEAS_RESNO );
            fdHelpHdlr.handleEventsFor(this);

            // Create and show Save as dialog 
            IFileDialog fileDlg (desktopWindow(),         // Parent is desktop
                                 this,                    // Owner is main window
                                 IFileDialog::helpButton, // Help button style
                                 fdSettings);             // Here are my settings
            
            // Stop special file dialog help handler
            fdHelpHdlr.stopHandlingEventsFor(this);

            // Check results
            if (fileDlg.pressedOK()) 
             {
               /*================================================*/
               /* Place processing to save file here.            */
               /*================================================*/
            
               // Display status 
               IString tmpStatus = resLib.loadString(IDS_FILE_SAVED);
               tmpStatus.change( "%s", 
                                 fileDlg.fileName(), 
                                 1, 1);
               writeStatus (tmpStatus);
             }  
          }  // endif   
            
         return true;
       } 


     case IDM_FILE_PRINT:
      { 
         // Write to status area
         IString tmpText = resLib.loadString(IDS_PRINT_FILE);
         tmpText.change( "%s", 
                         openFileName,            
                         1, 1);
         writeStatus (tmpText);
         return true;
       } 

      case IDM_FILE_CLOSE:
       {
         // Display "Are you sure" message box
         IMessageBox closeMsg(this);
         closeMsg.setTitle( IResourceId(IDW_FRAME_WINDOW));  // Title is application title
         IMessageBox::Response msgRC;
         msgRC = closeMsg.show( IResourceId(IDS_EXIT), 
                        IMessageBox::yesNoButton      | 
                        IMessageBox::defButton1       | 
                        IMessageBox::queryIcon        |
                        IMessageBox::applicationModal |
                        IMessageBox::moveable);


         if (msgRC == IMessageBox::yes)  // If user responded yes, close window
             close();
         return true;
       }

      case IDM_EDIT_UNDO:
         // do item action
         writeStatus (IResourceId(IDS_UNDO));
         return true;

      case IDM_EDIT_COPY:
         // do item action
         writeStatus (IResourceId(IDS_COPY));
         return true;

      case IDM_EDIT_CUT:
         // do item action
         writeStatus (IResourceId(IDS_CUT));
         return true;

      case IDM_EDIT_PASTE:
         // do item action
         writeStatus (IResourceId(IDS_PASTE));
         return true;

      case IDM_EDIT_FIND:
         // do item action
         writeStatus (IResourceId(IDS_FIND));
         return true;

      case IDM_EDIT_FINDNEXT:
         // do item action
         writeStatus (IResourceId(IDS_FINDNEXT));
         return true;


      case IDM_HELP_USING:
         // Show IPF Using Help panel
         if (hasHelp) 
           help->show(IHelpWindow::using);
         writeStatus (IResourceId(IDS_HELP_USING), true);
         return true;


      case IDM_HELP_TUTORIAL:
         // Send message to help handler
         cmdEvent.window()->sendEvent( HM_TUTORIAL );
         writeStatus (IResourceId(IDS_HELP_TUTORIAL), true);
         return true;


      case IDM_HELP_PRODINFO:
         // Show product information dialog
         productInfo();
         writeStatus (IResourceId(IDS_HELP_PRODINFO), true);
         return true;

      case UM_NOHELP:
       {
         // Tell user help could not be loaded then continue
         IMessageBox errorMsg( this );
         errorMsg.setTitle( IResourceId(IDW_FRAME_WINDOW));
         errorMsg.show( IResourceId(IDS_ERROR_NO_HELP),
                        IMessageBox::catastrophic);
       }


   } /* end switch */
   return false;
}




/*********************************************************************/
/* This productInfo function displays the product information dialog */
/*********************************************************************/
void AppWindow :: productInfo()
{
   unsigned short rc;      // Return value from show dialog

   ProdInfoDialog *prodInfo = new ProdInfoDialog(this);
   prodInfo->showModally();
   rc = prodInfo->result();       //Get result (eg OK or Cancel)         .
   delete prodInfo;
}


/***********************************************************/
/* This function writes the argument resource string onto  */
/* the status area.                                        */
/***********************************************************/
Boolean AppWindow :: writeStatus(const IResourceId status,  
                                 const Boolean isHelpCmd)
{
  IResourceLibrary resLib;

  /*---------------------------------------------------------------------------*/
  /* If the selected command is a help command, write the text unconditionally.*/
  /* Otherwise, check first whether a file has been loaded before writing the  */
  /* text.  Display an error message if no file has been loaded.               */
  /*---------------------------------------------------------------------------*/

   // Call overloaded writeStatus() function with translated status string
   return writeStatus (resLib.loadString(status), isHelpCmd);
}


/***********************************************************/
/* This function writes the argument string onto the       */
/* status area.                                            */
/***********************************************************/
Boolean AppWindow :: writeStatus(const IString &status,
                                 const Boolean isHelpCmd)
{
  /*---------------------------------------------------------------------------*/
  /* If the selected command is a help command, write the text unconditionally.*/
  /* Otherwise, check first whether a file has been loaded before writing the  */
  /* text.  Display an error message if no file has been loaded.               */
  /*---------------------------------------------------------------------------*/
   if (!isHelpCmd && isFileOpen())
    {
      /* Display "Error: no file open" message box */
      AppMessageBox errorMsg(this);
      errorMsg.show( IResourceId(IDS_ERROR_NO_FILE_LOADED),
                     IMessageBox::catastrophic,
                     IDH_FILEOPENERR2_RESNO);
      return false;
    }
   else
    {
     statusLine.setText(status);              // Set status line text
     return true;
    }

   refresh();
}


/***********************************************************/
/* This function returns TRUE if there is currently no     */
/* file open.  It checks the state of the openFileName     */
/* data member of AppWindow.                               */
/***********************************************************/
Boolean AppWindow :: isFileOpen() const
{
   return (openFileName == EMPTY_STRING);
}



/***********************************************************/
/* This function returns TRUE if there is currently no     */
/* help available to the application.                      */
/***********************************************************/
Boolean AppWindow :: isHelpAvailable() const
{
   return (hasHelp);
}



/***********************************************************/
/* This function opens the requested file and returns true */
/* if the file was opened successfully.  This function     */
/* immediately closes the file after opening it, as a      */
/* safety measure.  You can add your own file open         */
/* processing as required.                                 */
/***********************************************************/
Boolean AppWindow :: openFile()
{
 IResourceLibrary resLib;

 CHAR     szFileName[CCHMAXPATH];    // File to open 
 HFILE    hFile;                     // File handle of opened file 
 ULONG    ulAction;                  // Action taken by DosOpen 
 APIRET   rc;                        // Return code for DosOpen 

 /* Open file */
 rc = DosOpen (openFileName,              
               &hFile,
               &ulAction,
               0L,
               0L,
               OPEN_ACTION_FAIL_IF_NEW |
               OPEN_ACTION_OPEN_IF_EXISTS,
               OPEN_SHARE_DENYWRITE |
               OPEN_ACCESS_READONLY,
               0L);

 if (rc != NO_ERROR)
   {
     // Display "Error: cannot open file" message box 
     IMessageBox errorMsg(this);
     IString tmpMsg = resLib.loadString(IDS_ERROR_OPENING_FILE);
     tmpMsg.change( "%s", openFileName, 1, 1);
     errorMsg.setTitle( IResourceId(IDW_FRAME_WINDOW));  
     errorMsg.show( tmpMsg,
                    IMessageBox::catastrophic,
                    IDH_FILEOPENERR1_RESNO);

     return false;
   }
 else   // DosOpen successful.
   {
     /*=======================================================*/
     /* Add further file processing here                      */
     /*=======================================================*/

     // Display status 
     IString tmpStatus = resLib.loadString(IDS_OPEN_FILE);
     tmpStatus.change( "%s", openFileName, 1, 1);
     writeStatus(tmpStatus);

     DosClose (hFile);    // Close file for now 
     return true;
   }

   return true;
}




