/***************************************************************************
 *                                                                         *
 *  Decompression routines for the class WIArchive                         *
 *                                                                         *
 *  This file Copyright (C) 1998-99 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

/****************************************************************************
 * Decompress a previously compressed file
 *
 * Arguments - head: Pointer to file header
 *
 * Error handling is as follows:
 *  --  if a file read error occurs with the Archive file,
 *      the callback is called;
 *  --  in all other error situations, you get a return value
 *      != 0.
 *
 * Returns:
 *  0       everything OK
 *  > 1     output file handling error (e.g. disk full);
 *          this is the return value from ferror(). You should
 *          then check errno or (with VAC++) _doserrno to find
 *          out what happened.
 *  < 1     decompression error from zlib, which are:
 *               Z_DATA_ERROR
 *               Z_STREAM_ERROR
 *               Z_MEM_ERROR
 *               Z_BUF_ERROR
 *          See inflate() in zlib.h for details. This is most
 *          definitely the result of a corrupt archive.
 */
short WIArchive::Expand (WIFileHeader *head)
{
    z_stream z;
    unsigned char *outbuf;
    int status, i;
    unsigned long readcount, l;

    // First of all, do some initialization
    head->crc = INIT_CRC;
    text = new unsigned char[BUFFERSIZE];
    outbuf = new unsigned char[BUFFERSIZE];
    z.zalloc = 0;
    z.zfree  = 0;
    z.opaque = 0;
    z.next_in = text;
    readcount = head->compsize;
    if (readcount < BUFFERSIZE)
        l = readcount;
    else
        l = BUFFERSIZE;
    z.avail_in = fread (text, 1, l, Archive);
    readcount -= l;

    // read error checking added
    if (z.avail_in == 0)
    {
        // check if it's really an error and not maybe EOF
        int ferr = ferror(Archive);
        if (ferr != 0)
            if (pfnCallback != NULL)
                (*pfnCallback) (CBM_ERR_READ,
                                CBREC_CANCEL,   // allow only "Cancel"
                                head);
    }

    inflateInit (&z);

    // Let the decompression begin!
    z.next_out = outbuf;
    z.avail_out = BUFFERSIZE;
    for (;;)
    {
        if (pfnCallback != NULL)
            (*pfnCallback) (CBM_PERCENTAGE,
                            CalcPercentage (ftell (File), head->origsize),
                            head);
        if (z.avail_in == 0)
             break;
        while (z.avail_in)
        {
            status = inflate (&z, Z_SYNC_FLUSH);

            if (status < 0)
                // < 0 means ZLIB error
                return (status);    // < 0

            for (i = 0; i < BUFFERSIZE - z.avail_out; i++)
                UPDATE_CRC (text[i]);

            // write error checking added
            if (BUFFERSIZE - z.avail_out)
            {
                size_t BytesWritten = fwrite (outbuf, 1, BUFFERSIZE - z.avail_out, File);

                if (BytesWritten != (BUFFERSIZE - z.avail_out))
                {
                    // check if it's really an error and not maybe EOF
                    int ferr = ferror(File);
                    if (ferr != 0)
                    {
                        // error occured:
                        // report error to caller
                        return (abs(ferr));        // > 0
                    }
                }
            }

            z.next_out = outbuf;
            z.avail_out = BUFFERSIZE;
        }
        if (status == Z_STREAM_END)  break;
        z.next_in = text;
        if (readcount < BUFFERSIZE)
            l = readcount;
        else
            l = BUFFERSIZE;
        z.avail_in = fread (text, 1, l, Archive);
        readcount -= l;
    }

    // We're done.  Yee-hah...
    inflateEnd (&z);
    delete [] outbuf;
    delete [] text;
    return 0;
}

/****************************************************************************
 * Extract a stored file
 *
 * Arguments - head: Pointer to file header
 */
short WIArchive::Extract (WIFileHeader *head)
{
    unsigned long crc, count;
    short bytes, i;

    count = head->compsize;
    crc = INIT_CRC;

    // ----- Extract the stored file
    if (pfnCallback != NULL)
        (*pfnCallback) (CBM_PERCENTAGE,
                        CalcPercentage (ftell (File), head->origsize),
                        head);
    text = (unsigned char *)malloc(BUFFERSIZE);
    while (count)
    {
        if (count >= BUFFERSIZE)
            bytes = BUFFERSIZE;
        else
            bytes = count;

        // read error checking added
        size_t BytesRead = fread (text, 1, bytes, Archive);
        if (BytesRead == 0)
        {
            // check if it's really an error and not maybe EOF
            int ferr = ferror(Archive);
            if (ferr != 0)
                if (pfnCallback != NULL)
                    (*pfnCallback) (CBM_ERR_READ,
                                    CBREC_CANCEL,   // allow only "Cancel"
                                    head);
        }

        for (i = 0; i < bytes; i++)
            UPDATE_CRC (text[i]);

        // write error checking added
        size_t BytesWritten = fwrite (text, 1, bytes, File);
        if (BytesWritten != bytes)
        {
            // check if it's really an error and not maybe EOF
            int ferr = ferror(File);
            if (ferr != 0)
                if (pfnCallback != NULL)
                    (*pfnCallback) (CBM_ERR_WRITE,
                                    CBREC_CANCEL,   // allow only "Cancel"
                                    head);
        }

        if (pfnCallback != NULL)
            (*pfnCallback) (CBM_PERCENTAGE,
                            CalcPercentage (ftell (File), head->origsize),
                            head);
        count -= bytes;
    }
    free (text);

    // ----- Check if the CRC matched
    crc ^= INIT_CRC;
    if (crc == head->crc)
        return 1;
    else
        return 0;
}











