/***************************************************************************
 *                                                                         *
 *  Compression routines for the class WIArchive                           *
 *                                                                         *
 *  This file Copyright (C) 1998-99 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

unsigned char *text;

/****************************************************************************
 * Use the Lempel-Ziv and Huffman compression methods to reduce the size of
 * this file.
 *
 * Arguments - head: Reference to file header
 */
void WIArchive::Compress (WIFileHeader *head)
{
    z_stream z;
    unsigned char *outbuf;
    int status, count;
    unsigned int i;

    // First of all, do some initialization
    head->method = 1;
    head->crc = INIT_CRC;
    text = new unsigned char[BUFFERSIZE];
    outbuf = new unsigned char[BUFFERSIZE];
    z.zalloc = 0;
    z.zfree  = 0;
    z.opaque = 0;
    deflateInit (&z, 9);

    // Now, begin the compression!
    z.avail_in = 0;
    z.next_out = outbuf;
    z.avail_out = BUFFERSIZE;
    head->compsize = 0;
    for (;;)
    {
        if (pfnCallback != NULL)
            (*pfnCallback) (CBM_PERCENTAGE,
                            CalcPercentage (ftell (File), head->origsize),
                            head);
        if (z.avail_in == 0)
        {
            z.next_in = text;
            z.avail_in = fread (text, 1, BUFFERSIZE, File);
        }
        if (z.avail_in == 0)
             break;
        for (i = 0; i < z.avail_in; i++)
            UPDATE_CRC (text[i]);
        status = deflate (&z, Z_NO_FLUSH);
        count = BUFFERSIZE - z.avail_out;
        head->compsize += count;
        if (count)
            fwrite (outbuf, 1, count, Archive);
        z.next_out = outbuf;
        z.avail_out = BUFFERSIZE;
    }

    // Flush pending data and wrap things up
    do
    {
        status = deflate (&z, Z_FINISH);
        count = BUFFERSIZE - z.avail_out;
        head->compsize += count;
        if (count)
            fwrite (outbuf, 1, count, Archive);
        z.next_out = outbuf;
        z.avail_out = BUFFERSIZE;
    } while (status == Z_OK);
    deflateEnd (&z);
    delete [] outbuf;
    delete [] text;
}

/****************************************************************************
 * Do not apply any compression at all on this file, just calculate the CRC
 * value and store it.
 *
 * Arguments - head: Reference to file header
 */
void WIArchive::Store (WIFileHeader *head)
{
    unsigned short bytes, i;
    unsigned long crc;

    head->compsize = head->origsize;
    head->method = 0;
    crc = INIT_CRC;

    // ----- Store the file without compression
    text = (unsigned char *)malloc(BUFFERSIZE);
    bytes = fread (text, 1, BUFFERSIZE, File);
    while (bytes)
    {
        if (pfnCallback != NULL)
            (*pfnCallback) (CBM_PERCENTAGE,
                            CalcPercentage (ftell (File), head->origsize),
                            head);
        for (i = 0; i < bytes; i++)
            UPDATE_CRC (text[i]);
        fwrite (text, 1, bytes, Archive);
        bytes = fread (text, 1, BUFFERSIZE, File);
    }
    free (text);

    crc ^= INIT_CRC;
    head->crc = crc;
}










