/***************************************************************************
 *                                                                         *
 *  Main class for WarpIN, WIArchive (header file)                         *
 *                                                                         *
 *  This file Copyright (C) 1998-99 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

#ifndef WIARCHIVE_INCLUDE
#define WIARCHIVE_INCLUDE

// the following is required for VAC++ (*UM#4)
#ifdef __IBMCPP__
typedef unsigned long bool;
#endif

#include <stdio.h>
#include <list>
#include <time.h>               // needed for WIFileHeader (*UM#3)
#include "..\zlib\zlib.h"

// this is the maximum path length allowed on OS/2 (*UM#3)
#define MAXPATHLEN 256

// ----- Structures used in the archive -------------------------------------
struct WIFile
{
    char *name;                 // Name of file
    char *extra;                // Possible extra information
    short package;              // Which package it belongs to
};

struct WIArcHeader
{
    char  v1, v2, v3, v4;       // Archive verification
    short wi_revision_needed;   // Minimum WarpIN version required
    char  path[MAXPATHLEN];     // Installation path (*UM#3)
    char  name_app[64];         // Application name
    char  name_dev[64];         // Delevoper name
    char  inet_address[128];    // Internet download address
    short rev_app;              // Application revision
    short os;                   // Operating system to install to
    short packs;                // Number of packages in archive
    short origscript;           // Size of installation script
    short compscript;           // Size of compressed script
    long  extended;             // Extended data
};

struct WIPackHeader
{
    short number;               // Package number
    short files;                // Number of files in package
    long  pos;                  // Position in archive
    long  origsize;             // Size of package (original)
    long  compsize;             // Size of package (compressed)
    char  name[32];             // Name of package
};

const short WIFH_MAGIC = 0xF012;  // any value for "magic" in file header (*UM#3)

#pragma pack(1)                 // save space in archive (*UM#3)
struct WIFileHeader
{
    short magic;                // must be WIFH_MAGIC for security (*UM#3)
    short checksum;             // Header checksum
    short method;               // Compression method used
    short package;              // Which package it belongs to
    long  origsize;             // Size of file (original)
    long  compsize;             // Size of file (compressed)
    unsigned long crc;          // File CRC checksum
    char  name[MAXPATHLEN];     // Filename (*UM#3)
    time_t lastwrite;           // file's last write date/time (req. time.h) (*UM#3)
    time_t creation;            // file's creation date/time (req. time.h) (*UM#3)
            // we don't need last access date/time
    char  extended;             // Size of extended information (if any)
};
#pragma pack()                  // save space in archive (*UM#3)

// ----- Callback function definition and related stuff ---------------------

// generic callback function prototype
// this has now three parameters:
//      short         mode   one of the CBM_* flags below
//      short         param  meaning depends on CBM_* value
//      WIFileHeader* pwifh  the file header of the file in question
typedef int (FNWICALLBACK)(short, short, WIFileHeader*);
typedef FNWICALLBACK *PFNWICALLBACK;

// ----- new #define's for the "mode" parameter of the callbacks
const CBM_PERCENTAGE   = 1;
            // in this case, "param" contains the percentage of the current
            // file being processed, and pwifh the current file header.
            // The return value doesn't matter.
const CBM_UPDATING     = 2;
            // this is for compression after a file has been added to the
            // archive and the archive is to be updated; looks better if we
            // have an additional message then.
            // In this case, "param" has the compression that was achieved,
            // and pwifh the current file header. (*UM#4)
            // The return value doesn't matter.
const CBM_NEXTFILE     = 3; // (*UM#3)
            // this comes just before the WIArchive class attempts to open
            // a new output file for writing. "param" is always 0, "pwifh"
            // points to the file header which is about to be opened.
            // This allows the front-end to do two things:
            // a)   update the "current file" display
            // b)   check for whether that file exists already:
            //      -- if not, we return CBRC_PROCEED;
            //      -- if it does exist, we ask the user for whether the file
            //         may be overwritten; if so, we try to delete it.
            //         Otherwise, we return CBRC_SKIP, and the back-end does
            //         not touch the file.
            // This new approach (Alpha #3) is required because sometimes
            // the target file which already exists is _locked_, e.g. if it's
            // a system DLL in use. We then need to do a DosReplaceModule,
            // which we cannot do in the back-end if we want to keep it
            // platform-independent. Sorry for the hassle.
const CBM_ERR_MEMORY   = 4;
            // Somehow, there wasn't enough memory to do something.  Not even
            // in swap space?  Well...  We're here, anyway.
            // We should not return from the callback then, but only terminate
            // WarpIn.
const CBM_ERR_WRITE    = 5;
            // error writing into output file; this is most probable when the
            // target disk is full.
            // "param" then has one of the CBREC_* values below, which identify
            // whether the error is recoverable. Depending on that value,
            // we may return one of the CBRC_* values.
const CBM_ERR_READ     = 6;
            // error reading from input file; maybe the download is corrupt.
            // "param" then has one of the CBREC_* values below, which identify
            // whether the error is recoverable. Depending on that value,
            // we may return one of the CBRC_* values.
const CBM_ERR_ZLIB = 7; // (*UM#3)
            // error somewhere in the zlib decompression part of WIArchive;
            // this usually signifies that the archive is broken. We then
            // terminate installation, since there's no way to recover.
            // "param" then contains one of the ZLIBERR_* flags below.

// ----- Error descriptions in case of CBM_ERR_ZLIB
// the following are from zlib.h
const ZLIBERR_ERRNO           = -1;
            // unknown meaning
const ZLIBERR_STREAM_ERROR    = -2;
            // inconsistent stream data, e.g. unexpected NULL
const ZLIBERR_DATA_ERROR      = -3;
            // input stream does not match zlib format
const ZLIBERR_MEM_ERROR       = -4;
            // not enough memory
const ZLIBERR_BUF_ERROR       = -5;
            // not enough room in output buffer
const ZLIBERR_VERSION_ERROR   = -6;
            // unknown meaning
// the following are added by WIArchive
const ZLIBERR_NO_WIFH_MAGIC   = -99;
            // this occurs if the first "short" in WIFileHeader
            // does not match WIFH_MAGIC; this check is added
            // to avoid crashes in the decompression routines

// ----- Error recovery flags for callback function  (*UM#3)
// If mode == CBM_ERR_xxx, the callback gets one of the following flags
// in "param", signalling whether the error is recoverable or not.
// Depending on this flag, we may return one of the CBRC_* values.
// NOTE: We must never return CBRC_ABORT, which the WIArchive class cannot
// handle, but terminate the install thread ourselves instead.
const CBREC_CANCEL              = 1;
            // error is not recoverable: show "Cancel" only
const CBREC_RETRYCANCEL         = 2;
            // show "Cancel" and "Retry"
const CBREC_ABORTRETRYIGNORE    = 3;
            // show "Abort", "Retry", "Ignore" (== skip file)

// ----- Return values for the callback function
// If mode == CBM_ERR_xxx, the callback must return one of these three.
const CBRC_ABORT = 1;
const CBRC_RETRY = 2;
const CBRC_IGNORE = 3;

// If mode == CBM_NEXTFILE, return one of these. (*UM#3)
const CBRC_PROCEED = 4;
const CBRC_SKIP = 5;

// In all other modes, the return value is not examined by the back-end.

// Note:
// If you offer a "Cancel" option in your error / "File exists" dialogs,
// it is your own responsibility to terminate the install process. The
// WIArchive class cannot handle a "Cancel" or "Abort" return value.
// So the safest way to do all this is have the WIArchive class run in
// a separate thread while unpacking the files, and terminate the thread
// simply. This is what happens in the WarpIn front-end.

// ----- Archive internal stuff ---------------------------------------------
#ifdef WIARCHIVE_INTERNAL
    const char WI_VERIFY1 = 0x77;
    const char WI_VERIFY2 = 0x04;
    const char WI_VERIFY3 = 0x02;
    const char WI_VERIFY4 = 0xBE;
    const short WI_OS_OS2 = 1;          // 1 - OS/2

    const long INIT_CRC = 0xFFFFFFFF;
    const long CRCPOLY = 0xEBD88320;
    #define UPDATE_CRC(c) \
        head->crc = crctable[(head->crc ^ (c)) & 0xFF] ^ (head->crc >> CHAR_BIT)

    const long BUFFERSIZE = 0x6000;

    extern unsigned char *text;
    extern unsigned long crctable[];
#endif

const short WIARCHIVE_REVISION = 2;
const short WIARCHIVE_REVISION_NEEDED = 2;

/*
 *@@ PFNFORALLFILESCALLBACK:
 *      callback prototype for WIArchive::for_all_files
 *      (*UM#4)
 */

typedef int (FNFORALLFILESCALLBACK)(WIFileHeader*, unsigned long);
typedef FNFORALLFILESCALLBACK *PFNFORALLFILESCALLBACK;

// ----- WIArchive class definition -------------------------------------
class WIArchive
{
    protected:
        short CalcPercentage (long, long);
        void MakeDirectories (char *);
        FILE* OpenTemp (char *, char *);        // prototype changed (*UM#4)
        int ReadArcHeader (FILE *);             // ADD: Scan entire file for a valid archive
        void ReadFilenames (FILE *);
        void ReadPackHeaders (FILE *);
        void Update ();
        void WriteArcHeader (FILE *);
        void WritePackHeaders (FILE *);
        void Compress (WIFileHeader *);
        short Expand (WIFileHeader *);
        short Extract (WIFileHeader *);
        void Store (WIFileHeader *);

        WIArcHeader ArcHeader;
        list<WIFile *> FileList;
        list<WIFile *> ToDoList;
        list<WIPackHeader *> PackList;
        FILE *Archive;
        FILE *OldArchive;
        bool fOpenForReadOnly;          // added (*UM#4);
        FILE *File;
        int UsingTemp;
        char *Script;
        char *Extended;
        char ArchiveName[128];
        char TempArchiveName[128];
        PFNWICALLBACK pfnCallback;
        long ArcStart;
        unsigned long crctable[256];

    public:
        WIArchive ();
        ~WIArchive ();
        void add (char *, char *, short);
        void close ();
        const WIArcHeader *get_arc_header ();
        list<WIFile *> *get_file_list ();
        list<WIPackHeader *> *get_pack_list ();
        const char *get_script ();
        int open (char *, int = 0);
        void remove (char *, short);
        void set_arc_header (const WIArcHeader &);
        void set_package (short, char *);
        void set_callback_func(PFNWICALLBACK);
        void set_script (char *);
        short unpack (short);

        // this one's new, I need it for WIC.EXE (*UM#4)
        short for_all_files(WIPackHeader*, PFNFORALLFILESCALLBACK, unsigned long);
};

#endif

