/***************************************************************************
 *                                                                         *
 *  Main class for WarpIN, WIArchive.  The class handles all the grunt     *
 *  work of managing a compressed archive, including pretty advanced file  *
 *  management.  Just create a new instance of this class for each and     *
 *  every archive file that needs to be processed.                         *
 *                                                                         *
 *  This file Copyright (C) 1998-99 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

// Changes made by Ulrich Mller for Alpha #4 are marked with (*UM#4).

#include <io.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <sys\utime.h>

// the following is required for VAC++
#ifdef __IBMCPP__
#include <direct.h>
#endif

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

/***************************************************************************
 * Constructor
 */
WIArchive::WIArchive ()
{
    // Create the CRC table right now
    unsigned long i, j, r;
    for (i = 0; i <= 256; i++)
    {
        r = i;
        for (j = 0; j < 8; j++)
            if (r & 1) r = (r >> 1) ^ CRCPOLY;
            else       r >>= 1;
        crctable[i] = r;
    }

    pfnCallback = NULL;     // No percentage function available right now
    Script = NULL;          // Same goes for the script...
    Extended = NULL;        // ...and any archive extensions
    UsingTemp = 0;
    fOpenForReadOnly = 0;   // added (*UM#4);
}

/***************************************************************************
 * Destructor
 */
WIArchive::~WIArchive ()
{
    list<WIPackHeader *>::iterator pstart, pend;
    list<WIFile *>::iterator start, end;

    // See if anything exists in our file lists
    start = ToDoList.begin();  end = ToDoList.end();
    for (; start != end; start++)
    {
        if ((**start).name != NULL)  delete [] (**start).name;
        if ((**start).extra != NULL)  delete [] (**start).extra;
        delete (WIFile *)*start;
    }
    start = FileList.begin();  end = FileList.end();
    for (; start != end; start++)
    {
        if ((**start).name != NULL)  delete [] (**start).name;
        if ((**start).extra != NULL)  delete [] (**start).extra;
        delete (WIFile *)*start;
    }

    // Erase the package list too
    pstart = PackList.begin();  pend = PackList.end();
    for (; pstart != pend; pstart++)
    {
        delete (WIPackHeader *)*pstart;
    }
    if (Script != NULL)  delete [] Script;
    if (Extended != NULL)  delete [] Extended;
}

/***************************************************************************
 * Calculates a percentage value using the formula (100a + (b/2)) / b
 *
 * Arguments - a and b: Two long numbers to use during calculations
 *    Return - The percentage calculated
 */
short WIArchive::CalcPercentage (long a, long b)
{
    int i;

    for (i = 0; i < 2; i++)
        if (a <= LONG_MAX / 10)
            a *= 10;
        else
            b /= 10;
    if ((long) (a + (b >> 1)) < a)
    {
        a >>= 1;
        b >>= 1;
    }
    if (b == 0)
        return 0;
    return (short) ((a + (b >> 1)) / b);
}

/***************************************************************************
 * Recieve a filename, check if the directories exist, and try to make them
 * if they don't
 *
 * Arguments - filename: The filename and path to examine
 */
void WIArchive::MakeDirectories (char *filename)
{
    char path[MAXPATHLEN];
    char *cp, c;

    // only do anything if the filename seems to
    // include a path at all
    if (    (strlen(filename) > 3)
         && (   (strchr(filename, '\\') != 0)
             || (strchr(filename, '/') != 0)
            )
       )
    {
        // Copy the filename and prepare for some serious directory creation
        strcpy (path, filename);
        cp = path;

        // advance past the drive letter only if we have one
        if (*(cp+1) == ':')
            cp += 3;

        // Now, make the directories
        while (*cp != '\0')
        {
            if (*cp == '\\' || *cp == '/')
            {
                c = *cp;
                *cp = '\0';
                #ifdef __IBMCPP__
                    mkdir (path);       // IBM VAC++ version
                #else
                    mkdir (path, 0);
                #endif
                *cp = c;
            }
            cp++;
        }
    }
}

/***************************************************************************
 * Creates a temporary archive name and opens it
 *
 * Arguments - file: Which file to open
 *             filename: The original filename
 *             mode: File open mode
 *    Return - Zero if something went wrong
 *
 *      Prototype changed (*UM#4): was
 *              int WIArchive::OpenTemp (FILE *file, char *filename, char *mode)
 */
FILE* WIArchive::OpenTemp (char *filename, char *mode)
{
    FILE*   file = 0;
    char    TempName[256];
    char    *cp;

    strcpy (TempName, filename);
    cp = strrchr (TempName, '\\');
    if (cp == NULL) {
        if (TempName[1] == ':')
            cp = &TempName[2];
        else
            cp = TempName;
    }
    strcpy (cp, "_witemp_");
    file = fopen (TempName, mode);
    UsingTemp = 1;
    strcpy (TempArchiveName, TempName);

    return (file);
}

/***************************************************************************
 * Read the archive header and determine if it's OK or not
 *
 * Arguments - file: The file to read from
 *    Return - The standard crap, zero if things went wrong.
 */
int WIArchive::ReadArcHeader (FILE *file)
{
    z_stream z;
    long l;

    // Check if this is a valid archive
    for (l = 0; l < 200000; l++)
    {
        fseek (file, l, SEEK_SET);
        fread ((char *)&ArcHeader, 1, sizeof (WIArcHeader), file);
        if ((ArcHeader.v1 != WI_VERIFY1) || (ArcHeader.v2 != WI_VERIFY2) ||
            (ArcHeader.v3 != WI_VERIFY3) || (ArcHeader.v4 != WI_VERIFY4))
            continue;
        if (ArcHeader.wi_revision_needed > WIARCHIVE_REVISION)
            return 0;
        l = 999999;  break;
    }
    if (l != 999999)
        return 0;

    // Well, if we have come this far, we have a perfectly healthy archive
    ArcStart = l;
    if (ArcHeader.compscript > 0)
    {
        // VAC++ 3.0 doesn't like non-constant array sizes, so this
        // must be replaced, sorry. (*UM#4)
            // char tempBuffer[ArcHeader.compscript];
        char* pTempBuffer = new char[ArcHeader.compscript];
        Script = new char[ArcHeader.origscript + 1];
        fread (pTempBuffer, 1, ArcHeader.compscript, file);  // (*UM#4)

        // ----- Decompress the installation script
        z.zalloc = 0;
        z.zfree  = 0;
        z.opaque = 0;
        z.next_in = (unsigned char *)pTempBuffer; // (*UM#4)
        z.avail_in = ArcHeader.compscript;
        z.next_out = (unsigned char *)Script;
        z.avail_out = ArcHeader.origscript + 1;
        inflateInit (&z);
        inflate (&z, Z_FINISH);
        inflateEnd (&z);
        Script[ArcHeader.origscript] = '\0';
    }

    if (ArcHeader.extended > 0)
    {
        Extended = new char[ArcHeader.extended];
        fread (Extended, 1, ArcHeader.extended, file);
    }

    return 1;
}

/***************************************************************************
 * Reads all the filenames stored inside this archive
 *
 * Arguments - file: Which file to use.  Does not save the current position!
 */
void WIArchive::ReadFilenames (FILE *file)
{
    list<WIPackHeader *>::iterator pstart, pend;
    WIFileHeader fhead;
    WIFile *wifile;
    unsigned long pos;
    int i;

    pstart = PackList.begin ();  pend = PackList.end ();
    for (; pstart != pend; pstart++)
    {
        i = (**pstart).files;
        pos = (**pstart).pos;
        while (i--)
        {
            fseek (file, pos, SEEK_SET);
            fread ((char *)&fhead, 1, sizeof (WIFileHeader), file);
            wifile = new WIFile;
            wifile->package = fhead.package;
            wifile->extra = NULL;
            wifile->name = new char[strlen(fhead.name) + 1];
            strcpy (wifile->name, fhead.name);
            FileList.push_back ((WIFile *)wifile);
            pos += sizeof (WIFileHeader) + fhead.compsize;
        }
    }
}

/***************************************************************************
 * Reads the contents of the package headers from the archive
 *
 * Arguments - file: Which file to use.  Reads from current position!
 */
void WIArchive::ReadPackHeaders (FILE *file)
{
    WIPackHeader *p;
    int i;

    i = ArcHeader.packs;
    while (i--)
    {
        p = new WIPackHeader;
        fread ((char *)p, 1, sizeof (WIPackHeader), file);
        PackList.push_back (p);
    }
}

/***************************************************************************
 * Add/remove files and all that sort of stuff concerning archive update
 */
void WIArchive::Update ()
{
    list<WIPackHeader *>::iterator pstart, pend;
    list<WIFile *>::iterator start, end;
    WIFileHeader filehead;
    unsigned short s;
    long pos, pos2, packpos;

    fseek (Archive, 0, SEEK_SET);
    WriteArcHeader (Archive);
    WritePackHeaders (Archive);

    unsigned short sMax = 0;
    // find the highest package index in the archive (*UM#4)
    pstart = PackList.begin();  pend = PackList.end();
    for (; pstart != pend; pstart++)
    {
        if ((**pstart).number > sMax)
            sMax = (**pstart).number;
    }
    // sMax now has the highest package index in use (*UM#4)

    // check for every package there is
    ArcHeader.packs = 0;
    for (s = 1;
         s <= sMax;      // (*UM#4)
         s++)
    {
        bool fFound = 0;
        // see if we have a file on the to-do list which
        // uses the package with index s
        start = ToDoList.begin();  end = ToDoList.end();
        for (; start != end; start++)
            if ((**start).package == s)
            {
                fFound = 1;     // (*UM#4)
                break;
            }

        if (fFound == 0)
        {
            // no file on the to-do list is using the package with index s:
            // then check if maybe a file on the list
            // of existing files uses it
            start = FileList.begin();  end = FileList.end();
            for (; start != end; start++)
                if ((**start).package == s)
                {
                    fFound = 1;     // (*UM#4)
                    break;
                }
        }

        if (fFound == 0)
            // no file is using this package (neither on
            // to-do list nor on existing list):
            // go for next package then, we don't need this
            continue;   // for s...     (*UM#4)

        // yes, we had at least one file in this package
        // (either on to-do list or on existing list)
        ArcHeader.packs++;
        pstart = PackList.begin();  pend = PackList.end();
        for (; pstart != pend; pstart++)
        {
            if ((**pstart).number == s)
                break;
        }
        // **pstart now has the current WIPackHeader to work on

        packpos = ftell (Archive);
        (**pstart).files = 0;
        (**pstart).origsize = (**pstart).compsize = 0;

        // Process the files on our to-do list
        start = ToDoList.begin();  end = ToDoList.end();
        for (; start != end; start++)
        {
            if ((**start).package != s)
                // incorrect pack number
                continue;

            pos = ftell (Archive);      // remember current position

            // copy data, store file header and open the source file
            strcpy (filehead.name, (**start).name);
            filehead.package = (**start).package;
            File = fopen ((**start).extra, "rb");
            fseek (File, 0, SEEK_END);
            filehead.origsize = ftell (File);
            filehead.extended = 0;
            filehead.magic = WIFH_MAGIC;

            // get file date/time
            struct stat StatBuf;
            fstat(fileno(File), &StatBuf);     // gets file info
            // store last write date and creation date in file header
            filehead.lastwrite = StatBuf.st_mtime;
            filehead.creation  = StatBuf.st_ctime;

            // write file header
            fwrite ((char *)&filehead, 1, sizeof (WIFileHeader), Archive);

            // Compress this thingie!
            rewind (File);      // go to beginning of source file
            Compress (&filehead);
            if (filehead.compsize > filehead.origsize)
            {
                // compressed version is larger than original one:
                // rewind then and write the file as-is
                fseek (Archive, pos + sizeof (WIFileHeader), SEEK_SET);
                rewind (File);
                Store (&filehead);
            }
            if (pfnCallback != NULL)
                (*pfnCallback)(CBM_UPDATING,
                               CalcPercentage (filehead.compsize, filehead.origsize),
                               &filehead);  // (*UM#4)

            // remember position _after_ compressed file
            // (this is for the next file header)
            pos2 = ftell (Archive);
            // go to beginning of _this_ file and rewrite the
            // updated file header (compressed size etc.)
            fseek (Archive, pos, SEEK_SET);
            fwrite ((char *)&filehead, 1, sizeof (WIFileHeader), Archive);
            // and go back to _after_ this file for the next file header
            fseek (Archive, pos2, SEEK_SET);
            (**pstart).files++;
            (**pstart).origsize += filehead.origsize;
            (**pstart).compsize += filehead.compsize;
        }

        // copy all the old files for package index s

        // set file pointer to what the package header has
        // as the beginning of files
        fseek (OldArchive, (**pstart).pos, SEEK_SET);
        for (;;)
        {
            pos = ftell(OldArchive);
            // read the file name and see if it's in the list
            size_t stRead = fread ((char *)&filehead, 1, sizeof (WIFileHeader), OldArchive);
            if (stRead < sizeof(WIFileHeader))
                break;

            // search the list now
            start = FileList.begin();  end = FileList.end();
            for (; start != end; start++)
            {
                if (    (strcmp(filehead.name, (**start).name) == 0)
                    &&  (filehead.package == (**start).package)
                   )
                {
                    // this is the file, let's copy it
                    // 1)   copy file header we got above
                    fwrite ((char *)&filehead, 1, sizeof (WIFileHeader), Archive);

                    // 2)   copy data in chunks of BLOCKSIZE
                    unsigned char *text = (unsigned char *)malloc(BUFFERSIZE);
                    long lBytesToGo = filehead.compsize;

                    // this whole block rewritten (*UM#4)
                    while (lBytesToGo > 0)
                    {
                        long lBytesThis = lBytesToGo;
                        if (lBytesThis > BUFFERSIZE)
                            lBytesThis = BUFFERSIZE;

                        fread (text,
                               1,           // size of block
                               lBytesThis,  // no. of blocks
                               OldArchive);     // source stream

                        if (fwrite (text,
                                    1,      // size of block
                                    lBytesThis,  // no. of blocks
                                    Archive)    // target stream
                                < lBytesThis)
                            // error occured (probably disk full):
                            if (pfnCallback != NULL)
                                (*pfnCallback)(CBM_ERR_WRITE,
                                               CBREC_CANCEL,
                                               &filehead);

                        lBytesToGo -= BUFFERSIZE;
                    }
                    free (text);

                    // update package header
                    (**pstart).files++;
                    (**pstart).origsize += filehead.origsize;
                    (**pstart).compsize += filehead.compsize;
                    break; // for (; start != end; start++) (*UM#4)

                    // the source file pointer is now on the next file header
                }
            }

            // if we didn't find the file, advance the pointer
            if (start == end)
            {
                fseek (OldArchive, filehead.compsize, SEEK_CUR);
                break;  // (*UM#4)
            }
        }

        (**pstart).pos = packpos;
    } // end for (s = 1; ...

    // Now, write the correct headers to this archive
    fseek (Archive, 0, SEEK_SET);
    WriteArcHeader (Archive);
    WritePackHeaders (Archive);
    fclose (Archive);

    // Remove the old archive and replace it with the new one.
    if (UsingTemp)
    {
        fclose (OldArchive);
        unlink (ArchiveName);
        rename (TempArchiveName, ArchiveName);
        UsingTemp = 0;
    }
}

/***************************************************************************
 * Puts the present archive header into storage
 *
 * Arguments - file: Which file to write to.  Writes to the current position!
 */
void WIArchive::WriteArcHeader (FILE *file)
{
    z_stream z;
    // char tempBuffer[ArcHeader.origscript + 15];
            // Fixed two bugs: (*UM#4);
            // a) VAC++ can only handle constant array sizes, the above
            //    wouldn't compile
            // b) but even worse, at this point ArcHeader.origscript is 0,
            //    so we better allocate the buffer below only
    char *pTempBuffer = 0;

    // First of all, set the archive identifiation and such thingies
    ArcHeader.v1 = WI_VERIFY1;
    ArcHeader.v2 = WI_VERIFY2;
    ArcHeader.v3 = WI_VERIFY3;
    ArcHeader.v4 = WI_VERIFY4;
    ArcHeader.wi_revision_needed = WIARCHIVE_REVISION;
    ArcHeader.os = WI_OS_OS2;
    if (Script == NULL) {
        ArcHeader.compscript = ArcHeader.origscript = 0;
    } else {
        ArcHeader.origscript = strlen (Script) + 1; // add 0-terminator (*UM#4)
        int ScriptBufSize = ArcHeader.origscript + 15; // (*UM#4)
        pTempBuffer = new char[ScriptBufSize];  // (*UM#4)

        // ----- Compress the installation script
        z.zalloc = 0;
        z.zfree  = 0;
        z.opaque = 0;
        z.next_in = (unsigned char *)Script;
        z.avail_in = ArcHeader.origscript;
        z.next_out = (unsigned char *)pTempBuffer;  // (*UM#4)
        z.avail_out = ScriptBufSize;                // (*UM#4)
        deflateInit (&z, 9);
        deflate (&z, Z_FINISH);
        // fixed the following (*UM#4): the compscript was exactly
        // 15 bytes too small, which resulted in a corrupt
        // compressed installation script
        // ArcHeader.compscript = ArcHeader.origscript - z.avail_out;
        ArcHeader.compscript = ScriptBufSize - z.avail_out;
        deflateEnd (&z);
    }

    if (fwrite ((char *)&ArcHeader, 1, sizeof (WIArcHeader), file) < sizeof(WIArcHeader))
        // error checking added (*UM#4)
        if (pfnCallback != NULL)
            (*pfnCallback)(CBM_ERR_WRITE,
                           CBREC_CANCEL,
                           NULL);  // (*UM#4)
    if (Script != NULL)
        fwrite (pTempBuffer, 1, ArcHeader.compscript, file); // (*UM#4)
    if (Extended != NULL)
        fwrite (Extended, 1, ArcHeader.extended, file);
}

/***************************************************************************
 * Writes all the packages we have to disk
 *
 * Arguments - file: Which file to write to.  Writes to the current position!
 */
void WIArchive::WritePackHeaders (FILE *file)
{
    list<WIPackHeader *>::iterator start, end;

    start = PackList.begin();  end = PackList.end();
    for (; start != end; start++)
    {
        fwrite ((char *)(*start), 1, sizeof (WIPackHeader), file);
    }
}

/***************************************************************************
 * Adds a file to our to-do list
 *
 * Arguments - filename: Name of the file to read and understand
 *             arcname: What name the file gets in our archive
 *             package: Which package the file ends up in
 */
void WIArchive::add (char *filename, char *arcname, short package)
{
    WIFile *f;

    f = new WIFile;
    f->extra = new char [strlen (filename) + 1];
    f->name = new char [strlen (arcname) + 1];
    strcpy (f->extra, filename);
    strcpy (f->name, arcname);
    f->package = package;
    ToDoList.push_back (f);
}

/***************************************************************************
 * Close all open files and perform an archive update
 */
void WIArchive::close ()
{
    if (!fOpenForReadOnly)       // added (*UM#4);
        Update();
            // this will in turn call fclose
    else                  // added (*UM#4);
        fclose (Archive); // added (*UM#4);
}

/***************************************************************************
 * Returns a pointer to the cute archive header
 *
 * Return - Constant pointer to an archive header (hopefully, my own)
 */
const WIArcHeader *WIArchive::get_arc_header ()
{
    return &ArcHeader;
}

/***************************************************************************
 * Gives the caller a pointer to our dirty little list
 *
 * Return - A pointer to the file list
 */
list<WIFile *> *WIArchive::get_file_list ()
{
    // (*UM#4) There's a bug in that the returned file
    // list is empty. WIC.EXE cannot just list all the
    // files in the archive without initializing the
    // list first.
    ReadFilenames(Archive); // added (*UM#4)

    return &FileList;
}

/***************************************************************************
 * Returns a pointer to our list of packages
 *
 * Return - A pointer to the package list
 */
list<WIPackHeader *> *WIArchive::get_pack_list ()
{
    return &PackList;
}

/***************************************************************************
 * Returns the installation script (if any)
 *
 * Return - A pointer to the installation script, or NULL if none found
 */
const char *WIArchive::get_script ()
{
    return Script;
}

/***************************************************************************
 * Opens the archive
 *
 * Arguments - filename: Name of the archive to open
 *             mode: Which mode to open in (0 = read, 1 = read/write)
 *    Return - Always returns zero if something went wrong
 */
int WIArchive::open (char *filename, int mode)
{
    fOpenForReadOnly = (mode == 0);     // added (*UM#4); we need to remember
                                        // whether the file was read-only for close()

    strcpy (ArchiveName, filename);
    Archive = fopen (filename, "rb");
    if (Archive != NULL)
    {
        // The file exists - check if it's a valid archive
        if (ReadArcHeader (Archive) == 0)
            return 0;               // Apparently it wasn't valid
        else {
            // Read all the files and packages in this archive
            ReadPackHeaders (Archive);
                // order changed (*UM#4)
            ReadFilenames (Archive);
        }
        // If we are allowed to r/w, then create a new archive
        if (mode == 1)
        {
            fclose (Archive);
            Archive = OpenTemp (filename, "w+b");
                    // changed from "r+b" (*UM#4)
            if (Archive == NULL)
                // OK, so we couldn't create a new archive. Run away!
                return 0;
            UsingTemp = 1;
            OldArchive = fopen (filename, "rb");
        }
    } else {
        // Are we permitted to create a new archive?
        if (mode == 0) {
            return 0;               // No, we're not
        }
        Archive = fopen (filename, "w+b");
    }
    return 1;
}

/***************************************************************************
 * Removes a file from our archive
 *
 * Arguments - filename: File name to remove
 *             package: Which package it belongs to
 */
void WIArchive::remove (char *filename, short package)
{
    list<WIFile *>::iterator start, end;

    // First of all, check if the file exists in the to-do list
    start = ToDoList.begin();  end = ToDoList.end();
    for (; start != end; start++)
    {
        if (strcmp ((**start).name, filename) == 0 && (**start).package == package)
        {
            if ((**start).name != NULL)  delete [] (**start).name;
            if ((**start).extra != NULL)  delete [] (**start).extra;
            delete (WIFile *)*start;
            return;
        }
    }

    // OK, remove it from the file list...
    start = FileList.begin();  end = FileList.end();
    for (; start != end; start++)
    {
        if (strcmp ((**start).name, filename) == 0 && (**start).package == package)
        {
            if ((**start).name != NULL)  delete [] (**start).name;
            if ((**start).extra != NULL)  delete [] (**start).extra;
            delete (WIFile *)*start;
            break;
        }
    }
}

/***************************************************************************
 * Sets the important archive parts
 *
 * Arguments - head: Pointer to an archive header
 */
void WIArchive::set_arc_header (const WIArcHeader &head)
{
    char *cp;

    ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_NEEDED;
    strcpy (ArcHeader.inet_address, head.inet_address);
    strcpy (ArcHeader.name_app, head.name_app);
    strcpy (ArcHeader.name_dev, head.name_dev);
    strcpy (ArcHeader.path, head.path);
    ArcHeader.rev_app = head.rev_app;
    ArcHeader.os = head.os;

    cp = ArcHeader.path;
    if (*cp != 0)
    {
        while (*cp != '\0')  cp++;
        cp--;
        if (*cp != '\\')
        {
            cp++;  *cp = '\\';
            cp++;  *cp = '\0';
        }
    }
}

/***************************************************************************
 * Sets the percentage display function, to be used in compression and
 * decompression routines for showing the progress
 *
 * Arguments - pf: A pointer to a function that (hopefully) displays the
 *                 percentage progress.
 */
void WIArchive::set_callback_func (PFNWICALLBACK pfnCallbackNew)
{
    pfnCallback = pfnCallbackNew;
}

/***************************************************************************
 * Sets some basic information about a package
 *
 * Arguments - package: Which package number we are dealing with
 *             name: Name of this package
 */
void WIArchive::set_package (short package, char *name)
{
    list<WIPackHeader *>::iterator start, end;
    WIPackHeader *p;
    int i = 0;

    // See if we already have a package with this number
    start = PackList.begin ();  end = PackList.end ();
    for (; start != end; start++)
        if ((**start).number == package)
            {  p = *start;  i++;  break;  }
    if (i == 0)  p = new WIPackHeader;

    // Store this information
    strcpy (p->name, name);
    p->number = package;
    p->files = p->origsize = p->compsize = 0;
    if (i == 0)  PackList.push_back (p);
}

/***************************************************************************
 * Copies an installation script to the archive
 *
 * Arguments - cp: Pointer to a string containing the script
 */
void WIArchive::set_script (char *cp)
{
    int i;

    i = strlen (cp);
    if (Script != NULL)  delete [] Script;
    Script = new char[i + 1];
    strcpy (Script, cp);
}

/***************************************************************************
 * Expands a package to disk
 *
 * Arguments - package: Which package number we are ordered to expand
 *    Return - Zero if something horrid happened (like out of space)
 */
short WIArchive::unpack (short package)
{
    list<WIPackHeader *>::iterator start, end;
    WIFileHeader fhead;
    char filename[256];
    short files;

    // Find the correct package information and start processing the info
    start = PackList.begin ();  end = PackList.end ();
    for (; start != end; start++)
        if ((**start).number == package)
            break;
    fseek (Archive, (**start).pos, SEEK_SET);
    files = (**start).files;

    // The uncompress-all-files-we-can-find loop
    while (files--)
    {
        fread ((char *)&fhead, 1, sizeof (WIFileHeader), Archive);

        // check for magic bytes
        if (fhead.magic != WIFH_MAGIC)  // wiarchive.h
        {
            if (pfnCallback != NULL)
                // magic byte doesn't match:
                // report decompression error
                (*pfnCallback) (CBM_ERR_ZLIB,
                            -99,
                            &fhead);
                    // and stop!
            return (0);
        }

        // else go on
        strcpy (filename, ArcHeader.path);
        strcat (filename, fhead.name);
        MakeDirectories (filename);         // Make directories, if they don't exist

        int iStatus = 0;
            // error status, meaning:
            //  0       OK
            //  >0      file open/write error
            //  <0      decompression error (zlib)
        int file_exists_rc = CBRC_PROCEED;
            // default is still to overwrite the file,
            // if no callback exists

        // call the "next file" callback.
        if (pfnCallback != NULL)
            // This allows the front-end to do two things:
            // a)   update the "current file" display
            // b)   check for whether that file exists already:
            //      -- if not, we return CBRC_PROCEED;
            //      -- if it does exist, we ask the user for whether the file
            //         may be overwritten; if so, we try to delete it.
            //         Otherwise, we return CBRC_SKIP, and the back-end does
            //         not touch the file.
            // This new approach (Alpha #3) is required because sometimes
            // the target file which already exists is _locked_, e.g. if it's
            // a system DLL in use. We then need to do a DosReplaceModule,
            // which we cannot do in the back-end if we want to keep it
            // platform-independent. Sorry for the hassle.
            file_exists_rc = ((*pfnCallback)(CBM_NEXTFILE, 0, &fhead));

        // error checking added
        long int lCurPos = ftell(Archive);
        if (file_exists_rc == CBRC_PROCEED)
        {
            // do-while loop in case the user presses
            // "Retry" on errors below
            // normally, we don't loop
            int iRepeat = 0;
            do {
                iRepeat = 0;

                // Now, extract the file!
                File = fopen (filename, "wb");

                // error checking added (*UM Alpha #3)
                if (File)
                {
                    // file successfully opened for writing:
                    // **** go!
                    switch (fhead.method) {
                        case 0: // Stored
                            iStatus = Extract(&fhead); break;
                        case 1: // Kompressed
                            iStatus = Expand(&fhead);  break;
                        default:
                            return 0;
                    }
                    fclose (File);
                }
                else
                    // file could not be opened:
                    iStatus = 99;

                // the following blocks added
                if (iStatus == 0)
                {
                    // no error occured: set file date/time
                    struct utimbuf utb;
                    // utimbuf is declared in sys\utime.h as follows (VAC++):
                    /*   struct utimbuf
                         {
                            time_t actime;          // access time
                            time_t modtime;         // modification time
                         }; */

                    utb.actime  = fhead.lastwrite;
                    utb.modtime = fhead.lastwrite;
                    utime(filename, &utb);
                }
                else
                {
                    // iStatus is
                    //      99      if the file could not be opened above
                    //   other > 1  if Extract or Expand couldn't write to the file
                    //     < 1      if the zlib had a decompression error

                    // default: skip the file (see below)
                    file_exists_rc = CBRC_SKIP;

                    if (pfnCallback != NULL)
                    {
                        if (iStatus < 0)
                        {
                            // decompression error:
                            (*pfnCallback) (CBM_ERR_ZLIB,
                                        iStatus,
                                        &fhead);
                                // and stop!
                        }

                        // else: error related to file handling
                        int irc = 0;
                        irc = (*pfnCallback) (CBM_ERR_WRITE,
                                        CBREC_ABORTRETRYIGNORE, // allow all selections
                                        &fhead);
                        switch (irc)
                        {
                            // CBRC_ABORT is handled in the front-end
                            case CBRC_RETRY:    iRepeat = 1; break;
                                        // stay in the do-while loop
                            // if the user presses "Ignore", the default
                            // CBRC_SKIP will be used (above)
                        }
                    }
                }
            } while (iRepeat);  // only if CBRC_RETRY is returned on errors
        }

        if (file_exists_rc == CBRC_SKIP)
            // this is the case if
            // a)   CBM_NEXTFILE returned CBRC_SKIP
            //      (if the target file existed)
            // b)   CBM_ERR_WRITE returned CBRC_SKIP
            //      (if an error occured writing the file)
        {
            // file is to be skipped: move the file pointer
            // past the file data, or we'll get garbage for
            // the next file header
            fseek(Archive, lCurPos + fhead.compsize, SEEK_SET);
        }
    }

    return 1;
}

/***************************************************************************
 * Call callback for all files (added *UM#4)
 *
 * This calls pfnWICallback for all files which belong to the
 * specified package. If you want to list all the files, you
 * need to call this method for each package (by going thru
 * the package list).
 *
 * This method accesses the archive file on disk, because
 * the class-internal list does not have the WIFileHeader's,
 * which we therefore must read in here. For large archives,
 * this function can take a bit of time.
 *
 * The callback gets called with the following:
 *      WIFileHeader* pwifh  the file header of the file in question
 *      unsigned long ulUser the user parameter passed to this function;
 *                           you can use this for a pointer to some
 *                           internal data
 *
 * This function returns the number of files which were found,
 * or 0 if no files were found, or -1 if an error occured.
 */
short WIArchive::for_all_files(WIPackHeader* pckHeader,   // in: package to list files for
                               PFNFORALLFILESCALLBACK pfncbForAllFiles,
                                                // in: callback to call for each file
                               unsigned long ulUser)
                                                // in: user parameter passed to callback
{
    short sFileCount = 0;

    if (pckHeader)
    {
        // found our package:
        fseek (Archive, pckHeader->pos, SEEK_SET);

        short files = pckHeader->files;

        WIFileHeader fhead;

        // go thru all the files
        while (files--)
        {
            fread ((char *)&fhead, 1, sizeof (WIFileHeader), Archive);

            // check for magic bytes
            if (fhead.magic != WIFH_MAGIC)  // wiarchive.h
            {
                // stop!
                return (-1);
            }

            // else: OK, call callback
            (*pfncbForAllFiles)(&fhead, ulUser);
            long int lCurPos = ftell(Archive);
            fseek(Archive, lCurPos + fhead.compsize, SEEK_SET);
            sFileCount++;
        }
    }

    return (sFileCount);
}

















