
/*
 *@@sourcefile winh.c:
 *      contains Presentation Manager helper functions that are
 *      independent of a single application, i.e. these can be
 *      used w/out the rest of the XWorkplace source in any PM
 *      program.
 *
 *      Function prefixes (new with V0.81):
 *      --  winh*   Win (Presentation Manager) helper functions
 *
 *      In addition to various PM and control helper functions,
 *      this file now (V1.00) also has the following:
 *      --  progress bar support (previously in progbars.c, which
 *          has been removed; see winhProgressBarFromStatic for
 *          details);
 *      --  split windows support (all new with V1.00; see
 *          winhCreateSplitWindow for details).
 *
 *@@header "winh.h"
 */

/*
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSDEVIOCTL
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_GPI

// spooler #include's
#define INCL_BASE
#define INCL_SPL
#define INCL_SPLDOSPRINT
#define INCL_SPLERRORS

#include <os2.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "dosh.h"
#include "winh.h"
#include "prfh.h"
#include "gpih.h"

#include "undoc.h"

#define _PMPRINTF_
#include "pmprintf.h"

/* ******************************************************************
 *                                                                  *
 *   Menu helpers                                                   *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhInsertMenuItem:
 *      this inserts one one menu item into a given menu.
 *
 *      Valid menu item styles (afStyle) are:
 *
 *      --  MIS_SUBMENU
 *      --  MIS_SEPARATOR
 *      --  MIS_BITMAP: the display object is a bit map.
 *      --  MIS_TEXT: the display object is a text string.
 *      --  MIS_BUTTONSEPARATOR:
 *               The item is a menu button. Any menu can have zero,
 *               one, or two items of this type.  These are the last
 *               items in a menu and are automatically displayed after
 *               a separator bar. The user cannot move the cursor to
 *               these items, but can select them with the pointing
 *               device or with the appropriate key.
 *      --  MIS_BREAK: the item begins a new row or column.
 *      --  MIS_BREAKSEPARATOR:
 *               Same as MIS_BREAK, except that it draws a separator
 *               between rows or columns of a pull-down menu.
 *               This style can only be used within a submenu.
 *      --  MIS_SYSCOMMAND:
 *               menu posts a WM_SYSCOMMAND message rather than a
 *               WM_COMMAND message.
 *      --  MIS_OWNERDRAW:
 *               WM_DRAWITEM and WM_MEASUREITEM notification messages
 *               are sent to the owner to draw the item or determine its size.
 *      --  MIS_HELP:
 *               menu posts a WM_HELP message rather than a
 *               WM_COMMAND message.
 *      --  MIS_STATIC
 *               This type of item exists for information purposes only.
 *               It cannot be selected with the pointing device or
 *               keyboard.
 *
 *      Valid menu item attributes (afAttr) are:
 *      --  MIA_HILITED: if and only if, the item is selected.
 *      --  MIA_CHECKED: a check mark appears next to the item (submenu only).
 *      --  MIA_DISABLED: item is disabled and cannot be selected.
 *              The item is drawn in a disabled state (gray).
 *      --  MIA_FRAMED: a frame is drawn around the item (top-level menu only).
 *      --  MIA_NODISMISS:
 *               if the item is selected, the submenu remains down. A menu
 *               with this attribute is not hidden until the  application
 *               or user explicitly does so, for example by selecting either
 *               another menu on the action bar or by pressing the escape key.
 *
 *      Returns the return value of the MM_INSERTITEM msg:
 *      --  MIT_MEMERROR:    space allocation for menu item failed
 *      --  MIT_ERROR:       other error
 *      --  other:           zero-based index of new item in menu.
 */

SHORT winhInsertMenuItem(HWND hwndMenu,     // in:  menu to insert item into
                         SHORT iPosition,   // in:  zero-based index of where to
                                            //      insert or MIT_END
                         SHORT sItemId,     // in:  ID of new menu item
                         PSZ pszItemTitle,  // in:  title of new menu item
                         SHORT afStyle,     // in:  MIS_* style flags
                         SHORT afAttr)      // in:  MIA_* attribute flags
{
    MENUITEM mi;
    SHORT    src = MIT_ERROR;

    mi.iPosition = iPosition;
    mi.afStyle = afStyle;
    mi.afAttribute = afAttr;
    mi.id = sItemId;
    mi.hwndSubMenu = 0;
    mi.hItem = 0;
    src = SHORT1FROMMR(WinSendMsg(hwndMenu,
                                  MM_INSERTITEM,
                                  (MPARAM)&mi,
                                  (MPARAM)pszItemTitle));
    return (src);
}

/*
 *@@ winhInsertSubmenu:
 *      this inserts a submenu into a given menu and, if
 *      sItemId != 0, inserts one item into this new submenu also.
 *
 *      See winhInsertMenuItem for valid menu item styles and
 *      attributes.
 *
 *      Returns the HWND of the new submenu.
 */

HWND winhInsertSubmenu(HWND hwndMenu,       // in: menu to add submenu to
                       ULONG iPosition,     // in: index where to add submenu or MIT_END
                       SHORT sMenuId,       // in: menu ID of new submenu
                       PSZ pszSubmenuTitle, // in: title of new submenu
                       USHORT afMenuStyle,  // in: MIS* style flags for submenu;
                                            // MIS_SUBMENU will always be added
                       SHORT sItemId,       // in: ID of first item to add to submenu;
                                            // if 0, no first item is inserted
                       PSZ pszItemTitle,    // in: title of this item
                                            // (if sItemID != 0)
                       USHORT afItemStyle,  // in: style flags for this item, e.g. MIS_TEXT
                                            // (this is ignored if sItemID == 0)
                       USHORT afAttribute)  // in: attributes for this item, e.g. MIA_DISABLED
                                            // (this is ignored if sItemID == 0)
{
    MENUITEM mi;
    SHORT    src = MIT_ERROR;
    HWND     hwndNewMenu;

    // create new, empty menu
    hwndNewMenu = WinCreateMenu(hwndMenu,
                                NULL); // no menu template
    if (hwndNewMenu)
    {
        // add "submenu item" to this empty menu;
        // for some reason, PM always needs submenus
        // to be a menu item
        mi.iPosition = iPosition;
        mi.afStyle = afMenuStyle | MIS_SUBMENU;
        mi.afAttribute = 0;
        mi.id = sMenuId;
        mi.hwndSubMenu = hwndNewMenu;
        mi.hItem = 0;
        src = SHORT1FROMMR(WinSendMsg(hwndMenu, MM_INSERTITEM, (MPARAM)&mi, (MPARAM)pszSubmenuTitle));
        if (    (src != MIT_MEMERROR)
            &&  (src != MIT_ERROR)
           )
        {
            // set the new menu's ID to the same as the
            // submenu item
            WinSetWindowUShort(hwndNewMenu, QWS_ID, sMenuId);

            if (sItemId) {
                // item id given: insert first menu item also
                mi.iPosition = 0;
                mi.afStyle = afItemStyle;
                mi.afAttribute = afAttribute;
                mi.id = sItemId;
                mi.hwndSubMenu = 0;
                mi.hItem = 0;
                WinSendMsg(hwndNewMenu,
                           MM_INSERTITEM,
                           (MPARAM)&mi,
                           (MPARAM)pszItemTitle);
            }
        }
    }
    return (hwndNewMenu);
}

/*
 *@@ winhInsertMenuSeparator:
 *      this inserts a separator into a given menu at
 *      the given position (which may be MIT_END);
 *      returns the position at which the item was
 *      inserted.
 */

SHORT winhInsertMenuSeparator(HWND hMenu,       // in: menu to add separator to
                              SHORT iPosition,  // in: index where to add separator or MIT_END
                              SHORT sId)        // in: separator menu ID (doesn't really matter)
{
    MENUITEM mi;
    mi.iPosition = iPosition;
    mi.afStyle = MIS_SEPARATOR;             // append separator
    mi.afAttribute = 0;
    mi.id = sId;
    mi.hwndSubMenu = 0;
    mi.hItem = 0;

    return (SHORT1FROMMR(WinSendMsg(hMenu,
                    MM_INSERTITEM,
                    (MPARAM)&mi,
                    (MPARAM)"")));
}

/*
 *@@ winhMenuRemoveEllipse:
 *      removes a "..." substring from a menu item
 *      title, if found. This is useful if confirmations
 *      have been turned off for a certain menu item, which
 *      should be reflected in the menu.
 */

VOID winhMenuRemoveEllipse(HWND hwndMenu,
                           USHORT usItemId)    // in:  item to remove "..." from
{
    CHAR szBuf[255];
    CHAR *p;
    WinSendMsg(hwndMenu, MM_QUERYITEMTEXT,
            MPFROM2SHORT(usItemId, sizeof(szBuf)-1),
            (MPARAM)&szBuf);
    if (p = strstr(szBuf, "..."))
        strcpy(p, p+3);
    WinSendMsg(hwndMenu, MM_SETITEMTEXT,
            MPFROMSHORT(usItemId),
            (MPARAM)&szBuf);
}

/*
 *@@ winhQueryItemUnderMouse:
 *      this queries the menu item which corresponds
 *      to the given mouse coordinates.
 *      Returns the ID of the menu item and stores its
 *      rectangle in *prtlItem; returns (-1) upon errors.
 */

SHORT winhQueryItemUnderMouse(HWND hwndMenu,      // in: menu handle
                              POINTL *pptlMouse,  // in: mouse coordinates
                              RECTL *prtlItem)    // out: rectangle of menu item
{
    SHORT   s, sItemId, sItemCount;
    HAB     habDesktop = WinQueryAnchorBlock(HWND_DESKTOP);

    sItemCount = SHORT1FROMMR(WinSendMsg(hwndMenu, MM_QUERYITEMCOUNT, MPNULL, MPNULL));

    for (s = 0;
         s <= sItemCount;
         s++)
    {
        sItemId = SHORT1FROMMR(WinSendMsg(hwndMenu,
                                          MM_ITEMIDFROMPOSITION,
                                          (MPARAM)s, MPNULL));
        WinSendMsg(hwndMenu, MM_QUERYITEMRECT,
                MPFROM2SHORT(sItemId, FALSE),
                (MPARAM)prtlItem);
        if (WinPtInRect(habDesktop, prtlItem, pptlMouse))
            return (sItemId);
    }
    /* sItemId = (SHORT)WinSendMsg(hwndMenu, MM_ITEMIDFROMPOSITION, (MPARAM)(sItemCount-1), MPNULL);
    return (sItemId); */
    return (-1); // error: no valid menu item
}

/* ******************************************************************
 *                                                                  *
 *   "Menu button" helpers                                          *
 *                                                                  *
 ********************************************************************/

/*
 * MENUBUTTONDATA:
 *      internal data for "menu button"
 */

typedef struct _MENUBUTTONDATA
{
    PFNWP       pfnwpButtonOriginal;
    HMODULE     hmodMenu;
    ULONG       idMenu;
    BOOL        /* fHilite,
                fMouseCaptured; */
                fMouseCaptured,         // TRUE if WinSetCapture
                fMouseButton1Down,      // TRUE in between WM_BUTTON1DOWN and WM_BUTTON1UP
                fButtonDepressed,       // toggle state of the button
                fUnHiliteButton;        // flag for WM_BUTTON1UP
    HWND        hwndMenu;
} MENUBUTTONDATA, *PMENUBUTTONDATA;

/*
 *@@ fnwpSubclassedMenuButton:
 *      subclassed window proc for "menu button".
 *      See winhMenuButtonFromButton for details.
 */

MRESULT EXPENTRY fnwpSubclassedMenuButton(HWND hwndButton, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;
    PMENUBUTTONDATA pmbd = (PMENUBUTTONDATA)WinQueryWindowULong(hwndButton, QWL_USER);

    switch (msg)
    {
        /*
         * WM_BUTTON1DOWN:
         * WM_BUTTON1UP:
         *      these show/hide the menu.
         *
         *      Showing the menu follows these steps:
         *          a)  first WM_BUTTON1DOWN hilites the button;
         *          b)  first WM_BUTTON1UP shows the menu.
         *
         *      When the button is pressed again, the open
         *      menu loses focus, which results in WM_MENUEND
         *      and destroys the window automatically.
         */

        case WM_BUTTON1DOWN:
        case WM_BUTTON1DBLCLK:
            // since we're not passing the message
            // to WinDefWndProc, we need to acticate
            // the parent
            WinSetActiveWindow(HWND_DESKTOP, WinQueryWindow(hwndButton, QW_PARENT));

            if (!pmbd->fMouseCaptured)
            {
                // capture mouse events while the
                // mouse button is down
                WinSetCapture(HWND_DESKTOP, hwndButton);
                pmbd->fMouseCaptured = TRUE;
            }

            pmbd->fMouseButton1Down = FALSE;

            if (!pmbd->fButtonDepressed)
            {
                // button not hilited yet: do it now
                WinSendMsg(hwndButton,
                           BM_SETHILITE,
                           (MPARAM)TRUE,
                           (MPARAM)0);
                pmbd->fButtonDepressed = TRUE;
            }
            else
            {
                pmbd->fUnHiliteButton = TRUE;
                    // flag for WM_BUTTON1UP later
            }

            // else: the menu has just been destroyed
            // (WM_MENUEND below)

            /* else
            {
                // button already hilited:
                // this means the menu must be open, so
                // destroy it
                WinDestroyWindow(pmbd->hwndMenu);
                pmbd->hwndMenu = NULLHANDLE;
                pmbd->fHilite = FALSE;
                    // for WM_BUTTON1UP later
            } */

            mrc = (MPARAM)TRUE;     // message processed
        break;

        case WM_BUTTON1UP:
            // un-capture the mouse first
            if (pmbd->fMouseCaptured)
            {
                WinSetCapture(HWND_DESKTOP, NULLHANDLE);
                pmbd->fMouseCaptured = FALSE;
            }

            pmbd->fMouseButton1Down = FALSE;

            if (    (pmbd->fUnHiliteButton)
                 && (pmbd->fButtonDepressed)
               )
            {
                // second button up
                // (flag set by WM_BUTTON1DOWN above):
                // un-hilite button
                WinSendMsg(hwndButton,
                           BM_SETHILITE,
                           (MPARAM)FALSE,
                           (MPARAM)0);
                pmbd->fButtonDepressed = FALSE;
                pmbd->fUnHiliteButton = FALSE;
            }
            else if (    (!pmbd->fUnHiliteButton)
                      && (pmbd->fButtonDepressed)
                    )
            {
                // first button up:
                // show menu

                if (pmbd->idMenu)
                    // menu specified: load from
                    // specified resources
                    pmbd->hwndMenu = WinLoadMenu(hwndButton,
                                                 pmbd->hmodMenu,
                                                 pmbd->idMenu);
                else
                    // pmbd->idMenu == 0:
                    // send WM_COMMAND to owner
                    pmbd->hwndMenu = (HWND)WinSendMsg(WinQueryWindow(hwndButton,
                                                                     QW_OWNER),
                                                      WM_COMMAND,
                                                      (MPARAM)WinQueryWindowUShort(hwndButton,
                                                                                   QWS_ID),
                                                      (MPARAM)0);
                if (pmbd->hwndMenu)
                {
                    // menu successfully loaded:
                    // find out where to put it

                    SWP     swpButton;
                    POINTL  ptlMenu;
                    WinQueryWindowPos(hwndButton, &swpButton);
                    ptlMenu.x = swpButton.x;
                    ptlMenu.y = swpButton.y;

                    // ptlMenu now has coordinates relative
                    // to the button's parent;
                    // convert this to screen coordinates:
                    WinMapWindowPoints(WinQueryWindow(hwndButton, QW_PARENT),
                               HWND_DESKTOP,
                               &ptlMenu,
                               1);

                    // now show the menu on top of the button
                    WinPopupMenu(HWND_DESKTOP,               // menu parent
                                 hwndButton,                 // owner
                                 pmbd->hwndMenu,
                                 (SHORT)(ptlMenu.x),
                                 (SHORT)(ptlMenu.y + swpButton.cy - 1),
                                 0,                          // ID
                                  PU_NONE
                                     | PU_MOUSEBUTTON1
                                     | PU_KEYBOARD
                                     | PU_HCONSTRAIN
                                     | PU_VCONSTRAIN);
                }
            }


            /* if (pmbd->fHilite)
            {
                // button was just depressed:
            }

            if (pmbd->hwndMenu == NULLHANDLE)
            {
                // button 1 is released, but the menu is
                // not open:
                // un-hilite the button
                WinSendMsg(hwndButton,
                           BM_SETHILITE,
                           (MPARAM)FALSE,
                           (MPARAM)0);
                pmbd->fHilite = FALSE;
            } */

            mrc = (MPARAM)TRUE;     // message processed
        break;

        /*
         * WM_MENUEND:
         *      menu is destroyed
         */

        case WM_MENUEND: {
            BOOL fUnHilite = FALSE;
            // At this point, the menu has been
            // destroyed already.
            // Since WM_BUTTON1UP handles the
            // default case that the user presses
            // the menu button a second time while
            // the menu is open, we only need
            // to handle the case that the user
            // c)   presses some key on the menu (ESC, menu selection) or
            // a)   selects a menu item (which
            //      dismisses the menu) or
            // b)   clicks anywhere else.

            // Case a) if mouse button 1 is not currently down
            if ((WinGetKeyState(HWND_DESKTOP, VK_BUTTON1) & 0x8000) == 0)
                // button 1 _not_ down:
                // must be keyboard
                fUnHilite = TRUE;
            else
                // button 1 _is_ down:
                // query window under mouse pointer
                ;

            if (fUnHilite)
            {
                pmbd->fUnHiliteButton = TRUE;
                WinPostMsg(hwndButton, WM_BUTTON1UP, (MPARAM)0, (MPARAM)0);
            }

            /* else
            {
                // button 1 _is_ down:
            } */
            pmbd->hwndMenu = NULLHANDLE;
        break; }

        /*
         * WM_COMMAND:
         *      this must be from the menu, so
         *      forward this to the button's owner
         */

        case WM_COMMAND:
            WinPostMsg(WinQueryWindow(hwndButton, QW_OWNER),
                       msg,
                       mp1,
                       mp2);
        break;

        /*
         * WM_DESTROY:
         *      clean up allocated data
         */

        case WM_DESTROY:
            mrc = (*(pmbd->pfnwpButtonOriginal))(hwndButton, msg, mp1, mp2);
            free(pmbd);
        break;

        default:
            mrc = (*(pmbd->pfnwpButtonOriginal))(hwndButton, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ winhMenuButtonFromButton:
 *      this turns the specified button into a "menu button",
 *      which shows a popup menu when the button is depressed.
 *      This is done by subclassing the menu button with
 *      fnwpSubclassedMenuButton.
 *
 *      Simply call this function upon any button, and it'll
 *      turn in to a menu button.
 *
 *      When the user presses the button, the specified menu
 *      is loaded from the resources. The button will then
 *      be set as the owner, but it will forward all WM_COMMAND
 *      messages from the menu to its own owner (probably your
 *      dialog), so you can handle WM_COMMAND messages just as
 *      if the menu was owned by your dialog.
 *
 *      Alternatively, if you don't want to load a menu from
 *      the resources, you can specify idMenu == 0. In that case,
 *      when the menu button is pressed, it sends (!) a WM_COMMAND
 *      message to its owner with its ID in mp1 (as usual). The
 *      difference is that the return value from that message will
 *      be interpreted as a menu handle (HWND) to use for the button
 *      menu.
 *
 *      The subclassed button also handles menu destruction etc.
 *      by itself.
 */

BOOL winhMenuButtonFromButton(HWND hwndButton,      // in: button to subclass
                              HMODULE hmodMenu,     // in: resource module (can be NULLHANDLE for
                                                    // current EXE)
                              ULONG idMenu)         // in: resource menu ID (or 0)
{
    BOOL brc = FALSE;
    PMENUBUTTONDATA pmbd = (PMENUBUTTONDATA)malloc(sizeof(MENUBUTTONDATA));
    if (pmbd)
    {
        memset(pmbd, 0, sizeof(MENUBUTTONDATA));
        pmbd->pfnwpButtonOriginal = WinSubclassWindow(hwndButton,
                                                      fnwpSubclassedMenuButton);
        if (pmbd->pfnwpButtonOriginal)
        {
            pmbd->hmodMenu = hmodMenu;
            pmbd->idMenu = idMenu;
            WinSetWindowULong(hwndButton, QWL_USER, (ULONG)pmbd);
            brc = TRUE;
        }
        else
            free(pmbd);
    }
    return (brc);
}

/* ******************************************************************
 *                                                                  *
 *   Slider helpers                                                 *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhReplaceWithLinearSlider:
 *      this destroys the control with the ID ulID in hwndDlg
 *      and creates a linear slider at the same position with the
 *      same ID (effectively replacing it).
 *
 *      This is needed because the IBM dialog editor (DLGEDIT.EXE)
 *      keeps crashing when creating sliders. So the way to do
 *      this easily is to create some other control with DLGEDIT
 *      where the slider should be later and call this function
 *      on that control when the dialog is initialized.
 *
 *      You need to specify _one_ of the following with ulSliderStyle:
 *      -- SLS_HORIZONTAL: horizontal slider (default)
 *      -- SLS_VERTICAL: vertical slider
 *
 *      plus _one_ additional common slider style for positioning:
 *      -- for horizontal sliders: SLS_BOTTOM, SLS_CENTER, or SLS_TOP
 *      -- for vertical sliders: SLS_LEFT, SLS_CENTER, or SLS_RIGHT
 *
 *      Additional common slider styles are:
 *      -- SLS_PRIMARYSCALE1: determines the location of the scale
 *                  on the slider shaft by using increment
 *                  and spacing specified for scale 1 as
 *                  the incremental value for positioning
 *                  the slider arm. Scale 1 is displayed
 *                  above the slider shaft of a horizontal
 *                  slider and to the right of the slider
 *                  shaft of a vertical slider. This is
 *                  the default for a slider.
 *      -- SLS_PRIMARYSCALE2: not supported by this function
 *      -- SLS_READONLY: creates a read-only slider, which
 *                  presents information to the user but
 *                  allows no interaction with the user.
 *      -- SLS_RIBBONSTRIP: fills, as the slider arm moves, the
 *                  slider shaft between the home position
 *                  and the slider arm with a color value
 *                  different from slider shaft color,
 *                  similar to mercury in a thermometer.
 *      -- SLS_OWNERDRAW: notifies the application whenever the
 *                  slider shaft, the ribbon strip, the
 *                  slider arm, and the slider background
 *                  are to be drawn.
 *      -- SLS_SNAPTOINCREMENT: causes the slider arm, when positioned
 *                  between two values, to be positioned
 *                  to the nearest value and redrawn at
 *                  that position.
 *
 *      Additionally, for horizontal sliders:
 *      -- SLS_BUTTONSLEFT: specifies that the optional slider
 *                  buttons are to be used and places them
 *                  to the left of the slider shaft. The
 *                  buttons move the slider arm by one
 *                  position, left or right, in the
 *                  direction selected.
 *      -- SLS_BUTTONSRIGHT: specifies that the optional slider
 *                  buttons are to be used and places them
 *                  to the right of the slider shaft. The
 *                  buttons move the slider arm by one
 *                  position, left or right, in the
 *                  direction selected.
 *      -- SLS_HOMELEFT: specifies the slider arm's home
 *                  position. The left edge is used as the
 *                  base value for incrementing (default).
 *      -- SLS_HOMERIGHT: specifies the slider arm's home
 *                  position. The right edge is used as
 *                  the base value for incrementing.
 *
 *      Instead, for vertical sliders:
 *      -- SLS_BUTTONSBOTTOM: specifies that the optional slider
 *                  buttons are to be used and places them
 *                  at the bottom of the slider shaft. The
 *                  buttons move the slider arm by one
 *                  position, up or down, in the direction
 *                  selected.
 *      -- SLS_BUTTONSTOP: specifies that the optional slider
 *                  buttons are to be used and places them
 *                  at the top of the slider shaft. The
 *                  buttons move the slider arm by one
 *                  position, up or down, in the direction
 *                  selected.
 *      -- SLS_HOMEBOTTOM: specifies the slider arm's home
 *                  position. The bottom of the slider is
 *                  used as the base value for
 *                  incrementing.
 *      -- SLS_HOMETOP: specifies the slider arm's home
 *                  position. The top of the slider is
 *                  used as the base value for
 *                  incrementing.
 *
 *      Notes: This function automatically adds WS_PARENTCLIP,
 *      WS_TABSTOP, and WS_SYNCPAINT to the specified styles.
 *      For the WS_TABSTOP style, hwndInsertAfter is important.
 *      If you specify HWND_TOP, your window will be the first
 *      in the tab stop list.
 *
 *      It also shows the slider after having done all the
 *      processing in here by calling WinShowWindow.
 *
 *      Also, we only provide support for scale 1 here, so
 *      do not specify SLS_PRIMARYSCALE2 with ulSliderStyle,
 *      and we have the slider calculate all the spacings.
 *
 *      This returns the HWND of the slider or NULLHANDLE upon
 *      errors.
 *
 *@@added V1.00
 */

HWND winhReplaceWithLinearSlider(HWND hwndParent,   // in: parent of old control and slider
                           HWND hwndOwner,          // in: owner of old control and slider
                           HWND hwndInsertAfter,    // in: the control after which the slider should
                                                    // come up, or HWND_TOP, or HWND_BOTTOM
                           ULONG ulID,              // in: ID of old control and slider
                           ULONG ulSliderStyle,     // in: SLS_* styles
                           ULONG ulTickCount)       // in: number of ticks (scale 1)
{
    HWND    hwndSlider = NULLHANDLE;
    HWND    hwndKill = WinWindowFromID(hwndParent, ulID);
    if (hwndKill)
    {
        SWP swpControl;
        if (WinQueryWindowPos(hwndKill, &swpControl))
        {
            SLDCDATA slcd;

            // destroy the old control
            WinDestroyWindow(hwndKill);

            // initialize slider control data
            slcd.cbSize = sizeof(SLDCDATA);
            slcd.usScale1Increments = ulTickCount;
            slcd.usScale1Spacing = 0;           // have slider calculate it
            slcd.usScale2Increments = 0;
            slcd.usScale2Spacing = 0;

            // create a slider with the same ID at the same
            // position
            hwndSlider = WinCreateWindow(hwndParent,
                                         WC_SLIDER,
                                         NULL,           // no window text
                                         ulSliderStyle
                                            | WS_PARENTCLIP
                                            | WS_SYNCPAINT
                                            | WS_TABSTOP,
                                         swpControl.x,
                                         swpControl.y,
                                         swpControl.cx,
                                         swpControl.cy,
                                         hwndOwner,
                                         hwndInsertAfter,
                                         ulID,           // same ID as destroyed control
                                         &slcd,          // slider control data
                                         NULL);          // presparams

            WinSendMsg(hwndSlider,
                       SLM_SETTICKSIZE,
                       MPFROM2SHORT(SMA_SETALLTICKS,
                                    6),     // 15 pixels high
                       NULL);

            WinShowWindow(hwndSlider, TRUE);
        }
    }

    return (hwndSlider);
}

/*
 *@@ winhReplaceWithCircularSlider:
 *      this destroys the control with the ID ulID in hwndDlg
 *      and creates a linear slider at the same position with the
 *      same ID (effectively replacing it).
 *
 *      This is needed because the IBM dialog editor (DLGEDIT.EXE)
 *      cannot create circular sliders. So the way to do this
 *      easily is to create some other control with DLGEDIT
 *      where the slider should be later and call this function
 *      on that control when the dialog is initialized.
 *
 *      You need to specify the following with ulSliderStyle:
 *      --  CSS_CIRCULARVALUE: draws a circular thumb, rather than a line,
 *                  for the value indicator.
 *      --  CSS_MIDPOINT: makes the mid-point tick mark larger.
 *      --  CSS_NOBUTTON: does not display value buttons. Per default, the
 *                  slider displays "-" and "+" buttons to the bottom left
 *                  and bottom right of the knob. (BTW, these bitmaps can be
 *                  changed using CSM_SETBITMAPDATA.)
 *      --  CSS_NONUMBER: does not display the value on the dial.
 *      --  CSS_NOTEXT: does not display title text under the dial.
 *                  Otherwise, the text in the pszTitle parameter
 *                  will be used.
 *      --  CSS_NOTICKS (only listed in pmstddlg.h, not in PMREF):
 *                  obviously, this prevents tick marks from being drawn.
 *      --  CSS_POINTSELECT: permits the values on the circular slider
 *                  to change immediately when dragged.
 *                  Direct manipulation is performed by using a mouse to
 *                  click on and drag the circular slider. There are two
 *                  modes of direct manipulation for the circular slider:
 *                  <BR><B>1)</B> The default direct manipulation mode is to scroll to
 *                  the value indicated by the position of the mouse.
 *                  This could be important if you used a circular slider
 *                  for a volume control, for example. Increasing the volume
 *                  from 0% to 100% too quickly could result in damage to
 *                  both the user's ears and the equipment.
 *                  <BR><B>2)</B>The other mode of direct manipulation permits
 *                  the value on the circular slider to change immediately when dragged.
 *                  This mode is enabled using the CSS_POINTSELECT style bit. When this
 *                  style is used, the value of the dial can be changed by tracking
 *                  the value with the mouse, which changes values quickly.
 *      --  CSS_PROPORTIONALTICKS: allow the length of the tick marks to be calculated
 *                  as a percentage of the radius (for small sliders).
 *      --  CSS_360: permits the scroll range to extend 360 degrees.
 *                  CSS_360 forces the CSS_NONUMBER style on. This is necessary
 *                  to keep the value indicator from corrupting the number value.
 *
 *      FYI: The most commonly known circular slider in OS/2, the one in the
 *      default "Sound" object, has a style of 0x9002018a, meaning
 *      CSS_NOTEXT | CSS_POINTSELECT | CSS_NOTICKS.
 *
 *      Notes: This function automatically adds WS_PARENTCLIP,
 *      WS_TABSTOP, and WS_SYNCPAINT to the specified styles.
 *      For the WS_TABSTOP style, hwndInsertAfter is important.
 *      If you specify HWND_TOP, your window will be the first
 *      in the tab stop list.
 *
 *      It also shows the slider after having done all the
 *      processing in here by calling WinShowWindow.
 *
 *      This returns the HWND of the slider or NULLHANDLE upon
 *      errors.
 *
 *@@added V1.00
 */

HWND winhReplaceWithCircularSlider(HWND hwndParent,   // in: parent of old control and slider
                           HWND hwndOwner,          // in: owner of old control and slider
                           HWND hwndInsertAfter,    // in: the control after which the slider should
                                                    // come up, or HWND_TOP, or HWND_BOTTOM
                           ULONG ulID,              // in: ID of old control and slider
                           ULONG ulSliderStyle,     // in: SLS_* styles
                           SHORT sMin,              // in: minimum value (e.g. 0)
                           SHORT sMax,              // in: maximum value (e.g. 100)
                           USHORT usIncrement,      // in: minimum increment (e.g. 1)
                           USHORT usTicksEvery)     // in: ticks ever x values (e.g. 20)
{
    HWND    hwndSlider = NULLHANDLE;
    HWND    hwndKill = WinWindowFromID(hwndParent, ulID);
    if (hwndKill)
    {
        SWP swpControl;
        if (WinQueryWindowPos(hwndKill, &swpControl))
        {
            // destroy the old control
            WinDestroyWindow(hwndKill);

            // WinRegisterCircularSlider();

            // create a slider with the same ID at the same
            // position
            hwndSlider = WinCreateWindow(hwndParent,
                                         WC_CIRCULARSLIDER,
                                         "dummy",        // no window text
                                         ulSliderStyle
                                            // | WS_PARENTCLIP
                                            // | WS_SYNCPAINT
                                            | WS_TABSTOP,
                                         swpControl.x,
                                         swpControl.y,
                                         swpControl.cx,
                                         swpControl.cy,
                                         hwndOwner,
                                         hwndInsertAfter,
                                         ulID,           // same ID as destroyed control
                                         NULL,           // control data
                                         NULL);          // presparams

            if (hwndSlider)
            {
                // set slider range
                WinSendMsg(hwndSlider,
                           CSM_SETRANGE,
                           (MPARAM)sMin,
                           (MPARAM)sMax);

                // set slider increments
                WinSendMsg(hwndSlider,
                            CSM_SETINCREMENT,
                            (MPARAM)usIncrement,
                            (MPARAM)usTicksEvery);

                // set slider increments
                WinSendMsg(hwndSlider,
                            CSM_SETVALUE,
                            (MPARAM)0,
                            (MPARAM)0);

                // for some reason, the slider always has
                // WS_CLIPSIBLINGS set, even though we don't
                // set this; we must unset this now, or
                // the slider won't draw itself (%&$&%"$&%!!!)
                WinSetWindowBits(hwndSlider,
                                 QWL_STYLE,
                                 0,         // unset bit
                                 WS_CLIPSIBLINGS);

                WinShowWindow(hwndSlider, TRUE);
            }
        }
    }

    return (hwndSlider);
}

/* ******************************************************************
 *                                                                  *
 *   Spin button helpers                                            *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhSetDlgItemSpinData:
 *      sets a spin button's limits and data within a dialog window.
 *      This only works for decimal spin buttons.
 */

VOID winhSetDlgItemSpinData(HWND hwndDlg,       // in: dlg window
                           ULONG idSpinButton,  // in: item ID of spin button
                           ULONG min,           // in: minimum allowed value
                           ULONG max,           // in: maximum allowed value
                           ULONG current)       // in: new current value
{
    HWND hwndSpinButton = WinWindowFromID(hwndDlg, idSpinButton);
    if (hwndSpinButton) {
        WinSendMsg(hwndSpinButton,
                      SPBM_SETLIMITS,          // Set limits message
                      (MPARAM)max,             // Spin Button maximum setting
                      (MPARAM)min);             // Spin Button minimum setting

        WinSendMsg(hwndSpinButton,
                      SPBM_SETCURRENTVALUE,    // Set current value message
                      (MPARAM)current,
                      (MPARAM)NULL);
    }
}

/*
 *@@ winhAdjustDlgItemSpinData:
 *      this can be called on a spin button control to
 *      have its current data snap to a grid. This only
 *      works for LONG integer values.
 *
 *      For example, if you specify 100 for the grid and call
 *      this func after you have received SPBN_UP/DOWNARROW,
 *      the spin button's value will always in/decrease
 *      in steps of 100.
 *
 *      This returns the "snapped" value to which the spin
 *      button was set.
 */

LONG winhAdjustDlgItemSpinData(HWND hwndDlg,     // in: dlg window
                               USHORT usItemID,  // in: item ID of spin button
                               LONG lGrid,       // in: grid
                               USHORT usNotifyCode) // in: SPBN_UP* or *DOWNARROW of WM_CONTROL message
{
    HWND hwndSpin = WinWindowFromID(hwndDlg, usItemID);
    LONG lBottom, lTop, lValue;
    // get value, which has already increased /
    // decreased by 1
    WinSendMsg(hwndSpin,
          SPBM_QUERYVALUE,
          (MPARAM)&lValue,
          MPFROM2SHORT(0, SPBQ_ALWAYSUPDATE));

    if (    (usNotifyCode == SPBN_UPARROW)
         || (usNotifyCode == SPBN_DOWNARROW)
       )
    {
        // only if the up/down buttons were pressed,
        // snap to the nearest grid; if the user
        // manually enters something (SPBN_CHANGE),
        // we'll accept that value
        lValue = (lValue / lGrid) * lGrid;
        // add /subtract grid
        if (usNotifyCode == SPBN_UPARROW)
            lValue += lGrid;
        // else we'll have -= lGrid already

        // balance with spin button limits
        WinSendMsg(hwndSpin,
              SPBM_QUERYLIMITS,
              (MPARAM)&lTop,
              (MPARAM)&lBottom);
        if (lValue < lBottom)
            lValue = lTop;
        else if (lValue > lTop)
            lValue = lBottom;

        WinSendMsg(hwndSpin,
              SPBM_SETCURRENTVALUE,
              (MPARAM)(lValue),
              MPNULL);
    }
    return (lValue);
}

/* ******************************************************************
 *                                                                  *
 *   List box helpers                                               *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhLboxSelectAll:
 *      this selects or deselects all items in the
 *      given list box, depending on fSelect.
 *
 *      Returns the number of items in the list box.
 */

ULONG winhLboxSelectAll(HWND hwndListBox,   // in: list box
                        BOOL fSelect)       // in: TRUE = select, FALSE = deselect
{
    LONG lItemCount = WinQueryLboxCount(hwndListBox);
    ULONG ul;

    for (ul = 0; ul < lItemCount; ul++)
    {
        WinSendMsg(hwndListBox,
                   LM_SELECTITEM,
                   (MPARAM)ul,      // index
                   (MPARAM)fSelect);
    }
}

/* ******************************************************************
 *                                                                  *
 *   Container helpers                                              *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhCnrScrollToRecord:
 *      scrolls a given container control to make a given
 *      record visible.
 *
 *      Returns:
 *      --  0:       OK, scrolled
 *      --  1:       record rectangle query failed (error)
 *      --  2:       cnr viewport query failed (error)
 *      --  3:       record is already visible (scrolling not necessary)
 *      --  4:       cnrinfo query failed (error)
 *      --  5:       parent record rectangle query failed (error)
 *
 *      Note: All messages are _sent_ to the container, not posted.
 *      Scrolling therefore occurs synchroneously before this
 *      function returns.
 *
 *      This function an improved version of the one (W)(C) Dan Libby, found at
 *      http://zebra.asta.fh-weingarten.de/os2/Snippets/Howt6364.HTML
 *      Improvements (C) 1998 Ulrich Mller.
 */

ULONG winhCnrScrollToRecord(HWND hwndCnr,       // in: container window
                            PRECORDCORE pRec,   // in: record to scroll to
                            ULONG fsExtent,
                                    // in: this determines what parts of pRec
                                    // should be made visible. OR the following
                                    // flags:
                                    // -- CMA_ICON the icon rectangle
                                    // -- CMA_TEXT the record text
                                    // -- CMA_TREEICON the "+" sign in tree view
                            BOOL KeepParent)
                                    // for tree views only: whether to keep
                                    // the parent record of pRec visible when scrolling.
                                    // If scrolling to pRec would make the parent
                                    // record invisible, we instead scroll so that
                                    // the parent record appears at the top of the
                                    // container workspace (Win95 style).

{
    QUERYRECORDRECT qRect, qRect2;
    RECTL           rclRecord, rclParentRecord, rclCnr, rclCnr2;
    POINTL          ptlRecord, ptlParentRecord;
    CNRINFO         CnrInfo;
    HAB             hab = WinQueryAnchorBlock(hwndCnr);
    BOOL            KeepParent2;
    LONG            lYOfs;

    qRect.cb = sizeof(qRect);
    qRect.pRecord = (PRECORDCORE)pRec;
    qRect.fsExtent = fsExtent;

    // query record location and size of container
    if (!WinSendMsg(hwndCnr, CM_QUERYRECORDRECT, &rclRecord, &qRect))
        return 1;
    if (!WinSendMsg(hwndCnr, CM_QUERYVIEWPORTRECT, &rclCnr, MPFROM2SHORT(CMA_WINDOW, FALSE)) )
        return 2;

    // check if left bottom point of pRec is currently visible in container
    ptlRecord.x = (rclRecord.xLeft);
    ptlRecord.y = (rclRecord.yBottom);
    // ptlRecord.x = (rclRecord.xLeft + rclRecord.xRight) / 2;
    // ptlRecord.y = (rclRecord.yBottom + rclRecord.yTop) / 2;
    if (WinPtInRect(hab, &rclCnr, &ptlRecord))
         return 3;

    if (KeepParent)
        if (!WinSendMsg(hwndCnr, CM_QUERYCNRINFO, (MPARAM)&CnrInfo, (MPARAM)sizeof(CnrInfo)))
            return 4;
        else KeepParent2 = (CnrInfo.flWindowAttr & CV_TREE);
    else KeepParent2 = FALSE;

    // calculate offset to scroll to make pRec visible
    lYOfs = (rclCnr.yBottom - rclRecord.yBottom)    // this would suffice
          + (rclRecord.yTop - rclRecord.yBottom);  // but we make the next rcl visible too

    if (KeepParent2) {
        qRect2.cb = sizeof(qRect2);
        qRect2.pRecord = (PRECORDCORE)WinSendMsg(hwndCnr, CM_QUERYRECORD,
                (MPARAM)pRec, MPFROM2SHORT(CMA_PARENT, CMA_ITEMORDER));
        qRect2.fsExtent = fsExtent;

        // now query PARENT record location and size of container
        if (!WinSendMsg(hwndCnr, CM_QUERYRECORDRECT, &rclParentRecord, &qRect2))
            return 5;

        ptlParentRecord.x = (rclParentRecord.xLeft);
        ptlParentRecord.y = (rclParentRecord.yTop);
        // ptlParentRecord.x = (rclParentRecord.xLeft + rclParentRecord.xRight) / 2;
        // ptlParentRecord.y = (rclParentRecord.yBottom + rclParentRecord.yTop) / 2;
        rclCnr2 = rclCnr;
        WinOffsetRect(hab, &rclCnr2, 0, -lYOfs);
        // if ( (rclParentRecord.yBottom - rclRecord.yBottom) > (rclCnr.yTop - rclCnr.yBottom) )
        if (!(WinPtInRect(hab, &rclCnr2, &ptlParentRecord)))
        {
            lYOfs = (rclCnr.yTop - rclParentRecord.yTop) // this would suffice
                  - (rclRecord.yTop - rclRecord.yBottom);  // but we make the previous rcl visible too
        }
    }

    if (!KeepParent2)
        WinSendMsg(hwndCnr, CM_SCROLLWINDOW, (MPARAM)CMA_HORIZONTAL,
            (MPARAM)(rclRecord.xLeft - rclCnr.xLeft));
    WinSendMsg(hwndCnr, CM_SCROLLWINDOW,
        (MPARAM)CMA_VERTICAL, (MPARAM)lYOfs);

    return 0;
}

/*
 *@@ winhCnrAllocRecords:
 *      this is a shortcut to allocating memory for
 *      container record cores.
 *
 *      If (ulCount == 1), this returns the new record core.
 *      If (ulCount > 1), this returns the _first_ record
 *      core; the following record cores may be reached
 *      by following the RECORDCORE.preccNextRecord pointers
 *      (i.e. we have a linked list here).
 *
 *      The record cores returned by the container are
 *      automatically zeroed out, and their "cb" field
 *      is automatically set to the size of the record core.
 *
 *      Note that this function presently does _not_ work
 *      with MINIRECORDCOREs.
 *
 *@@changed V1.00: function prototype changed to allocate more than one record
 */

PRECORDCORE winhCnrAllocRecords(HWND hwndCnr,    // in: cnr to allocate from
                                ULONG cbrecc,
                                    // in: total size of your record core.
                                    // If you're using the default recc's, this
                                    // must be sizeof(RECORDCORE).
                                ULONG ulCount)  // in: number of records to allocate (> 0)
{
    PRECORDCORE      precc;
    precc = (PRECORDCORE)WinSendMsg(hwndCnr, CM_ALLOCRECORD,
            (MPARAM)(cbrecc-sizeof(RECORDCORE)),
            (MPARAM)ulCount);

    return (precc);
}

/*
 *@@ winhCnrInsertRecords:
 *      shortcut to inserting record cores into a container
 *      which have been allocated using winhCnrAllocRecords.
 *
 *      If (<B>ulCount</B> == 1), this inserts precc.
 *      If (ulCount > 1), this assumes precc to be the first
 *      record to be inserted, while precc->preccNextRecord
 *      must point to the next record in a linked list (as
 *      returned with winhCnrAllocRecords). (changed V1.00)
 *
 *      Note that ulCount here must be exactly the same as
 *      specified with winhCnrAllocRecords.
 *
 *      If (<B>pszText</B> != NULL), this will automatically set precc->pszIcon,
 *      precc->pszText, precc->pszName, precc->pszTree to pszText to have
 *      the same record titles in all views. If (pszText == NULL), those
 *      fields will be left alone. (changed V1.00)
 *
 *      <B>flRecordAttr</B> should have the record attributes as
 *      specified in the RECORDCORE structure, which are:
 *      --  CRA_SELECTED       record is selected
 *      --  CRA_CURSORED       cursor (keyboard focus) is on the record
 *      --  CRA_SOURCE         record has source emphasis (drag'n'drop)
 *      --  CRA_TARGET         record has target emphasis (drag'n'drop)
 *      --  CRA_INUSE          record has in-use emphasis
 *      --  CRA_FILTERED       record has been filtered
 *      --  CRA_DROPONABLE     record can be dropped something upon
 *      --  CRA_RECORDREADONLY record is read-only (no text edit)
 *      --  CRA_EXPANDED       record is expanded (tree view)
 *      --  CRA_COLLAPSED      record is collapsed (tree view)
 *      --  CRA_PICKED         record picked (Lazy Drag)
 *
 *      plus the following undocumented (haven't tested these):
 *      --  CRA_IGNORE
 *      --  CRA_DISABLED
 *      --  CRA_OWNERFREE
 *      --  CRA_OWNERDRAW
 *
 *      This func returns the number of records in the container
 *      or NULL upon errors (changed V1.00).
 *
 *@@changed V1.00: function prototype changed to insert more than one record
 */

ULONG winhCnrInsertRecords(HWND hwndCnr,   // in: container to insert into
                           PRECORDCORE preccParent,
                               // in: record core below which precc should
                               // be inserted (tree view only); if NULL, precc
                               // is inserted at "root" level
                           PRECORDCORE precc,
                               // in: record core to insert (allocated using
                               // winhCnrAllocRecords)
                           PSZ pszText,
                               // in: text for recc. in all views (or NULL)
                           ULONG flRecordAttr,
                               // in: CRA_* flags
                           ULONG ulCount)  // in: number of records to insert (> 0)
{
    ULONG           ulrc = 0;
    RECORDINSERT    ri;

    if (precc)
    {
        // RECORDCORE stuff
        precc->flRecordAttr = flRecordAttr;
        precc->preccNextRecord = NULL;

        if (pszText) // V1.00
        {
            precc->pszIcon = pszText;
            precc->pszText = pszText;
            precc->pszName = pszText;
            precc->pszTree = pszText;
        }

        // setup RECORDINSERT struct
        ri.cb = sizeof(RECORDINSERT);
        ri.pRecordOrder = (PRECORDCORE)CMA_END;
        ri.pRecordParent = (PRECORDCORE)preccParent;
        ri.zOrder = CMA_TOP;
        ri.fInvalidateRecord = TRUE;
        ri.cRecordsInsert = ulCount;        // V1.00

        ulrc = (ULONG)WinSendMsg(hwndCnr, CM_INSERTRECORD, (MPARAM)precc, (MPARAM)&ri);
    }
    return (ulrc);
}

/*
 *@@ winhCnrSetFieldInfo:
 *      this sets a FIELDINFO structure to the given
 *      data. Note that ppFieldInfo is a double pointer
 *      to the actual FIELDINFO data.
 *
 *      This is best used with the winhCnrAllocFieldInfos
 *      macro defined in winh.h.
 *
 *      After setting the data, <B>*pFieldInfo</B> is advanced
 *      to the next FIELDINFO structure so that you can
 *      call this function several times for all the
 *      FIELDINFOS that you have allocated. After the last
 *      column, this pointer will be NULL.
 *
 *      Since the pointer is modified, do not invoke this
 *      function on the original pointer returned from
 *      winhCnrAllocFieldInfos, because you'll need that
 *      pointer for winhCnrAllocFieldInfos later.
 *
 *      <B>ulDataType</B> specifies the data type of the record
 *      core field. This can be one of the following:
 *      --  CFA_BITMAPORICON: bit-map or icon data
 *      --  CFA_DATE: date format (CDATE structure)
 *      --  CFA_STRING: null-terminated string
 *      --  CFA_TIME: time format (CTIME structure)
 *      --  CFA_ULONG: unsigned number data
 *
 *      You can add the following optional flags to
 *      ulDataType:
 *      --  CFA_FIREADONLY (CFA_STRING only): disable editing
 *      --  CFA_INVISIBLE: make column invisible
 *      --  CFA_OWNER: enable owner draw for this column
 *
 *      If (fDrawLines == TRUE), we'll automatically add
 *      CFA_HORZSEPARATOR | CFA_SEPARATOR.
 *
 *      <B>ulOrientation</B> specifies the vertical and
 *      horizontal orientation of both the column title
 *      and the column data. This should be OR'ed from
 *      the following
 *      --  CFA_CENTER, CFA_LEFT, CFA_RIGHT (horizontal);
 *          the default is CFA_LEFT
 *      --  CFA_BOTTOM, CFA_TOP, CFA_VCENTER (vertical);
 *          the default is CFA_VCENTER.
 *
 *      Note that the container automatically displays
 *      data according to the settings in the "Country" object.
 *
 *      The column title will always be set to string format
 *      and CFA_FITITLEREADONLY.
 *
 *      <B>ulFieldOffset</B> should be set to the return value
 *      of the FIELDOFFSET macro, which is also redefined in
 *      winh.h (to work with C++).
 *
 *      <B>Example usage</B> of the field info functions:
 *
 +          PFIELDINFO pFieldInfoFirst, pFieldInfo2;
 +          if (pFieldInfoFirst = winhCnrAllocFieldInfos(hwndFilesCnr, NO_OF_COLUMNS))
 +                          // macro defined in winh.h
 +          {
 +              pFieldInfo2 = pFieldInfoFirst;
 +
 +              // "File name" column
 +              winhCnrSetFieldInfo(&pFieldInfo2,
 +                                  FIELDOFFSET(RECORDCORE, pszIcon),
 +                                      // icon text offset in original RECORDCORE
 +                                  "File name",
 +                                  CFA_STRING,
 +                                  CFA_LEFT,
 +                                  FALSE);     // no draw lines
 +              // "Size" column
 +              winhCnrSetFieldInfo(&pFieldInfo2,
 +                                  FIELDOFFSET(FILERECORDCORE, ulSize),
 +                                      // size data field in sample extended RECORDCORE
 +                                  "Size",
 +                                  CFA_ULONG,
 +                                  CFA_RIGHT,
 +                                  FALSE);     // no draw lines
 +              ... // set other field infos
 +          }
 +
 +          // insert field infos
 +          winhCnrInsertFieldInfos(hwndFilesCnr,
 +                                  pFieldInfoFirst,
 +                                  NO_OF_COLUMNS);
 *
 *@@added V1.00
 */

VOID winhCnrSetFieldInfo(PFIELDINFO *ppFieldInfo2,  // in/out: double ptr to FIELDINFO
                         ULONG ulFieldOffset,       // in: FIELDOFFSET(YOURRECORDCORE, yourField);
                         PSZ pszColumnTitle,        // in: column title
                         ULONG ulDataType,          // in: column data type (CFA_* flags)
                         ULONG ulOrientation,       // in: vertical and horizontal orientation (CFA_* flags)
                         BOOL fDrawLines)           // in: if TRUE, we'll draw lines around the columns
{
    if (ppFieldInfo2)
        if (*ppFieldInfo2)
        {
            ULONG flData = ulDataType | ulOrientation;
            if (fDrawLines)
                flData |= CFA_HORZSEPARATOR | CFA_SEPARATOR;

            (*ppFieldInfo2)->cb = sizeof(FIELDINFO);
            (*ppFieldInfo2)->flData = flData;
            (*ppFieldInfo2)->flTitle = CFA_FITITLEREADONLY | ulOrientation;
            (*ppFieldInfo2)->offStruct = ulFieldOffset;
            (*ppFieldInfo2)->pTitleData = strdup(pszColumnTitle);
            (*ppFieldInfo2)->pUserData   = NULL;
            *ppFieldInfo2 = (*ppFieldInfo2)->pNextFieldInfo;
        }
}

/*
 *@@ winhCnrInsertFieldInfos:
 *      this inserts field infos for Details view
 *      into the specified container.
 *
 *      pFieldInfoFirst should be the PFIELDINFO
 *      returned by winhCnrAllocFieldInfos.
 *
 *      This inserts the FIELDINFOs at the end,
 *      should any columns already exist in the container.
 *      Also, the container is invalidated.
 *
 *      Returns the return value of CM_INSERTDETAILFIELDINFO,
 *      which is the total no. of field infos in the container
 *      or null upon errors.
 *
 *@@added V1.00
 */

ULONG winhCnrInsertFieldInfos(HWND hwndCnr,                // in: cnr for Details view
                              PFIELDINFO pFieldInfoFirst,  // in: first field info as returned
                                                           // by winhCnrAllocFieldInfos
                              ULONG ulFieldCount)          // in: no. of field infos
{
    FIELDINFOINSERT fii;
    fii.cb = sizeof(FIELDINFOINSERT);
    fii.pFieldInfoOrder = (PFIELDINFO)CMA_END;
    fii.fInvalidateFieldInfo = TRUE;
    fii.cFieldInfoInsert = ulFieldCount;

    return ((ULONG)WinSendMsg(hwndCnr,
                              CM_INSERTDETAILFIELDINFO,
                              (MPARAM)pFieldInfoFirst,
                              (MPARAM)&fii));
}

/*
 *@@ winhCnrSetFieldInfos:
 *      this combines winhCnrAllocFieldInfos,
 *      winhCnrSetFieldInfo, and winhCnrInsertFieldInfos
 *      into a one-shot func. To pass all the arguments
 *      normally passed to winhCnrSetFieldInfo, we
 *      use an array of XFIELDINFO structures, which
 *      takes the same parameters.
 *
 *      See winhCnrSetFieldInfo for a description of
 *      the fields in that structure.
 *
 *      The return value is the PFIELDINFO which corresponds
 *      to the column index specified in ulFieldReturn, or
 *      NULL upon errors. This is useful for setting the
 *      position of the split bar afterwards (using the
 *      winhCnrSetSplitBarAfter macro).
 *
 *      Example usage:
 +          XFIELDINFO      xfi[3];
 +
 +          xfi[0].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszApplication);
 +          xfi[0].pszColumnTitle = "Application";
 +          xfi[0].ulDataType = CFA_STRING;
 +          xfi[0].ulOrientation = CFA_LEFT;
 +
 +          xfi[1].ulFieldOffset = FIELDOFFSET(RECORDCORE, pszIcon);
 +          xfi[1].pszColumnTitle = "Package name";
 +          xfi[1].ulDataType = CFA_STRING;
 +          xfi[1].ulOrientation = CFA_LEFT;
 +
 +          xfi[2].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszAuthor);
 +          xfi[2].pszColumnTitle = "Author";
 +          xfi[2].ulDataType = CFA_STRING;
 +
 +          winhCnrSetFieldInfos(hwndCnr,
 +                               &xfi[0],
 +                               (sizeof(xfi) / sizeof(XFIELDINFO)),   // array item count
 +                               FALSE,         // no draw lines
 +                               0);            // return first column
 *
 *@@added V1.00
 */

PFIELDINFO winhCnrSetFieldInfos(HWND hwndCnr,            // in: container hwnd
                                PXFIELDINFO paxfi,       // in: pointer to an array of ulFieldCount XFIELDINFO structures
                                ULONG ulFieldCount,      // in: no. of items in paxfi array (> 0)
                                BOOL fDrawLines,         // in: if TRUE, we'll draw lines around the columns
                                ULONG ulFieldReturn)     // in: the column index to return as PFIELDINFO
{
    PFIELDINFO  pFieldInfoFirst,
                pFieldInfo2,
                pFieldInfoReturn = NULL;

    if (pFieldInfoFirst = winhCnrAllocFieldInfos(hwndCnr, ulFieldCount))
    {
        ULONG ul = 0;
        PXFIELDINFO pxfi = NULL;

        pFieldInfo2 = pFieldInfoFirst;
        pxfi = paxfi;
        for (ul = 0; ul < ulFieldCount; ul++)
        {
            if (ul == ulFieldReturn)
                // set return value
                pFieldInfoReturn = pFieldInfo2;

            // set current field info;
            // this will modify pFieldInfo to point to the next
            winhCnrSetFieldInfo(&pFieldInfo2,
                                pxfi->ulFieldOffset,
                                pxfi->pszColumnTitle,
                                pxfi->ulDataType,
                                pxfi->ulOrientation,
                                fDrawLines);
            pxfi++;
        }

        // insert field infos
        if (winhCnrInsertFieldInfos(hwndCnr,
                                    pFieldInfoFirst,
                                    ulFieldCount) == 0)
            pFieldInfoReturn = NULL;
    }

    return (pFieldInfoReturn);
}

/*
 *@@ winhCnrForAllRecords:
 *      this monster function calls pfnwpCallback
 *      for really all the records in the container,
 *      including child records in tree view.
 *
 *      This is extremely useful for cleaning up
 *      all record cores before a container window
 *      gets destroyed.
 *
 *      This function recurses for child records.
 *      On the first call, preccParent should be
 *      NULL; you may however specify a certain
 *      record, and this function will call the
 *      callback only for that record and children.
 *
 *      pfnwpCallback gets called with the following
 *      parameters:
 *
 *      -- HWND hwnd: hwndCnr, as passed to this func
 *      -- ULONG msg: always 0
 *      -- MPARAM mp1: the current record core, as
 *                     determined by this func.
 *      -- MPARAM mp2: what you specify in mpUser.
 *
 *      The return value of the callback is ignored.
 *
 *      If you use this function for deleting record
 *      cores, you can be sure that you can delete
 *      every record, because your callback gets called
 *      for the child records before the parent record.
 *
 *      This function returns the total number of
 *      calls made to the callback.
 *
 *@@added V1.00
 */

ULONG winhCnrForAllRecords(HWND hwndCnr,
                           PRECORDCORE preccParent,
                           PFNWP pfnwpCallback,
                           MPARAM mpUser)
{
    PRECORDCORE precc2 = preccParent;
    ULONG       ulrc = 0;
    USHORT      usQuery;
    BOOL        fFirstCall = TRUE;

    while (TRUE)
    {
        if (fFirstCall)
            // first call:
            if (preccParent)
                // non-root:
                usQuery = CMA_FIRSTCHILD;
            else
                // NULL == root:
                usQuery = CMA_FIRST;
        else
            // subsequent calls:
            usQuery = CMA_NEXT;     // works as CMA_NEXTCHILD also

        precc2 =
            (PRECORDCORE)WinSendMsg(hwndCnr,
                    CM_QUERYRECORD,
                    (MPARAM)((fFirstCall)
                                // first call (CMA_FIRSTCHILD or CMA_FIRST):
                                ? preccParent   // ignored for CMA_FIRST
                                // subsequent calls (CMA_NEXTCHILD or CMA_NEXT):
                                : precc2),  // what we queried last
                    MPFROM2SHORT(
                            usQuery,    // set above
                            CMA_ITEMORDER)
                    );

        if ((precc2) && ((ULONG)precc2 != -1))
        {
            // record found:
            // recurse for that record
            ulrc += winhCnrForAllRecords(hwndCnr,
                                         precc2,        // new parent to search
                                         pfnwpCallback,
                                         mpUser);

            // _Pmpf(("Calling callback for %s", precc2->pszIcon));

            // call callback
            if (pfnwpCallback)
                (*pfnwpCallback)(hwndCnr, 0, precc2, mpUser);
            ulrc++;
        }
        else
            // no more records or error: get outta here
            break;

        fFirstCall = FALSE;
    }

    return (ulrc);
}

/*
 * winhCnrForAllChildRecords:
 *      calls the specified fncbRecc callback for
 *      the specified recc and all its child records.
 *
 *@@added V1.00
 */

VOID winhCnrForAllChildRecords(HWND hwndCnr,
                               PRECORDCORE precc,
                               PFNCBRECC pfncbRecc,
                               ULONG ulp1,
                               ULONG ulp2)
{
    PRECORDCORE precc2 = precc;
    (*pfncbRecc)(precc, ulp1, ulp2);
    do {
        precc2 =
            (PRECORDCORE)WinSendMsg(hwndCnr,
                    CM_QUERYRECORD,
                    (MPARAM)precc2,
                    MPFROM2SHORT(
                        (precc2 == precc)
                            ? CMA_FIRSTCHILD : CMA_NEXT,
                        CMA_ITEMORDER)
                    );
        if ((LONG)precc2 == -1)
            precc2 = NULL;
        if (precc2)
            // recurse again
            winhCnrForAllChildRecords(hwndCnr, precc2, pfncbRecc, ulp1, ulp2);
    } while (precc2);
}

/*
 * winhCnrForAllRecords2:
 *      this is a useful function which calls
 *      the specified callback function for
 *      really all records in the container of
 *      the main window, including child records.
 *
 *      xxx
 *
 *@@added V1.00
 */

VOID winhCnrForAllRecords2(HWND hwndCnr,
                           PFNCBRECC pfncbRecc,
                           ULONG ulp1,
                           ULONG ulp2)
{
    PRECORDCORE precc2 = NULL;
    do {
        precc2 =
            (PRECORDCORE)WinSendMsg(hwndCnr,
                    CM_QUERYRECORD,
                    (MPARAM)precc2,
                    MPFROM2SHORT(
                        ((precc2 == NULL) ? CMA_FIRST : CMA_NEXT),
                        CMA_ITEMORDER)
                    );
        if ((LONG)precc2 == -1)
            precc2 = NULL;
        if (precc2)
            // recurse again
            winhCnrForAllChildRecords(hwndCnr, precc2, pfncbRecc, ulp1, ulp2);
    } while (precc2);
}

/*
 * winhCnrForAllParentRecords:
 *      just as above, but climbs up instead.
 *      Starts with the parent record.
 *
 *@@added V1.00
 */

VOID winhCnrForAllParentRecords(HWND hwndCnr,
                                PRECORDCORE precc,
                                PFNCBRECC pfncbRecc,
                                ULONG ulp1,
                                ULONG ulp2)
{
    PRECORDCORE precc2 = precc;
    do {
        precc2 =
            (PRECORDCORE)WinSendMsg(hwndCnr,
                    CM_QUERYRECORD,
                    (MPARAM)precc2,
                    MPFROM2SHORT(CMA_PARENT,
                        CMA_ITEMORDER)
                    );
        if ((LONG)precc2 == -1)
            precc2 = NULL;
        if (precc2)
            (*pfncbRecc)(precc2, ulp1, ulp2);
    } while (precc2);
}

/*
 *@@ winhCnrMoveTree:
 *      this function moves a container record core from one
 *      tree to another. See the CM_MOVETREE message for more
 *      explanations.
 *
 *      The problem with that message is that automatic container
 *      sorting does not work in tree view. This function however
 *      is smart enough to maintain container sorting and insert
 *      the moved record core at the correct position below the
 *      target record core, if you specify a container sort function
 *      (pfnCnrSort, which works just in CNRINFO).
 *      Otherwise, we'll just insert the item as the first child
 *      of preccNewParent.
 *
 *@@added V1.00
 */

BOOL winhCnrMoveTree(HWND hwndCnr,          // in: container control
                     PRECORDCORE preccMove, // in: record core to move
                     PRECORDCORE preccNewParent, // in: new parent for preccMove
                                                 // or NULL if move to root
                     PFNCNRSORT pfnCnrSort) // in: sort function to use or NULL
{
    TREEMOVE    tm;
    PRECORDCORE preccInsertAfter = (PRECORDCORE)CMA_FIRST;
    BOOL        fBugWorkAround = FALSE,
                brc = FALSE;

    if (pfnCnrSort)
    {
        // permanent sort activated:

        // since the automatic container sort does
        // not work when moving a record core tree in
        // Tree view, we must find the recc after
        // which we'll insert the tree ourselves
        PRECORDCORE preccSearch = preccNewParent;
        BOOL        fFirstRun = TRUE;
        ULONG       ulFirstCode;

        // set the code for first-loop query:
        if (preccNewParent)
            // if new parent is non-root
            ulFirstCode = CMA_FIRSTCHILD;
        else
            // if new parent is root
            ulFirstCode = CMA_FIRST;

        while (TRUE)
        {
            preccSearch =
                (PRECORDCORE)WinSendMsg(hwndCnr,
                        CM_QUERYRECORD,
                        // the following gets either the
                        // first child recc of the target
                        // record core or the next child
                        // for consecutive loops
                        (MPARAM)preccSearch,
                        MPFROM2SHORT(
                                ((fFirstRun)
                                ? ulFirstCode  // first loop
                                : CMA_NEXT  // works for next child too
                            ),
                            CMA_ITEMORDER)
                        );
            fFirstRun = FALSE;

            if (    (preccSearch == NULL)
                 || ((ULONG)preccSearch == -1)
               )
            {
                // no more items found:
                // keep preccInsertAfter, which might be CMA_FIRST
                // or the preccSearch we have found previously.

                if (preccInsertAfter != (PRECORDCORE)CMA_FIRST)
                {
                    // Unfortunately, there is a bug in the container
                    // control which prohibits CM_MOVETREE from working
                    // if preccInsertAfter turns out to be the last
                    // record core in preccNewParent. This applies to
                    // preccInsertAfter == CMA_LAST also, and CMA_LASTCHILD
                    // doesn't work either.
                    // Duh.
                    // We'll fix this later.

                    fBugWorkAround = TRUE;
                }
                break;
            }

            if (((*pfnCnrSort)(preccSearch, preccMove, 0)) < 0)
            {
                // found record core is < our tree:
                // mark current as "insert after" for later and go on
                preccInsertAfter = preccSearch;
            }
            else
                break;
        }

        /* _Pmpf(("preccInsertAfter %s",
                (preccInsertAfter == (PRECORDCORE)CMA_FIRST) ? "CMA_FIRST"
                : (preccInsertAfter == (PRECORDCORE)CMA_LAST) ? "CMA_LAST"
                : (preccInsertAfter == NULL) ? "NULL"
                : preccInsertAfter->pszIcon
             )); */
    } // end if (CnrInfo.pSortRecord)

    if (fBugWorkAround)
        // this is TRUE only if preccInsertAfter has turned
        // out to be the last child of preccNewParent. This
        // will make the container crash, so we insert as
        // first and sort the whole damn container later.
        preccInsertAfter = (PRECORDCORE)CMA_FIRST;

    // set record to be moved
    tm.preccMove = preccMove;
    // set target record core
    tm.preccNewParent = preccNewParent;
    tm.pRecordOrder = preccInsertAfter;
    tm.flMoveSiblings = FALSE;
                // move only preccMove
    brc = (BOOL)WinSendMsg(hwndCnr,
                           CM_MOVETREE,
                           (MPARAM)&tm,
                           MPNULL);

    if (brc)
        if (fBugWorkAround)
            WinSendMsg(hwndCnr, CM_SORTRECORD, (MPARAM)pfnCnrSort, MPNULL);

    return (brc);
}

/*
 *@@ winhCnrShowContextMenu:
 *      this function shows the given menu as a context
 *      menu for the given record core (using WinPopupMenu).
 *      This function may be used when receiving WM_CONTROL
 *      with CN_CONTEXTMENU from the container, which has
 *      the preccSource in mp2.
 *
 *      In detail, this function does the following:
 *      1)  query the coordinates where to show the menu;
 *          if (preccSource), this will be next to the
 *          record core, otherwise (i.e. menu requested
 *          for whitespace) the mouse coordinates over
 *          the container;
 *      2)  give preccSource (or, if NULL, the whole
 *          container) source emphasis;
 *      3)  call WinPopupMenu.
 *
 *      Note: It is your responsibility to catch WM_MENUEND
 *      in the window procedure of hwndMenuOwner to remove
 *      the source emphasis for preccSource again.
 *
 *      This function returns FALSE if an error occured.
 *
 *@@added V1.00
 */

BOOL winhCnrShowContextMenu(HWND hwndCnr,
                            PRECORDCORE preccSource, // in: mp2 of CN_CONTEXTMENU
                            HWND hMenu,              // in: menu to show
                            HWND hwndMenuOwner)      // in: menu owner (where the
                                                     // WM_COMMAND will go to)
{
    BOOL    brc = FALSE;
    if (hMenu)
    {
        BOOL        fQueried = FALSE;

        POINTL ptl;
        if (preccSource)
        {
            CNRINFO     CnrInfo;
            winhQueryCnrInfo(hwndCnr, &CnrInfo);

            if ((CnrInfo.flWindowAttr & CV_DETAIL) == 0)
            {
                // if we're not in Details view:
                // calculate the point where to show the context
                // menu; we use the lower-right corner of the
                // source record core
                QUERYRECORDRECT qRect;
                RECTL           rclRecc;
                qRect.cb = sizeof(qRect);
                qRect.pRecord = preccSource;
                qRect.fsExtent = CMA_TEXT;
                WinSendMsg(hwndCnr,
                           CM_QUERYRECORDRECT,
                           &rclRecc,
                           &qRect);
                ptl.x = rclRecc.xRight;
                ptl.y = rclRecc.yBottom;
                    // now we have the lower-right corner in cnr coords

                // adjust if this is outside the container window
                WinQueryWindowRect(hwndCnr, &rclRecc);
                if (ptl.x > rclRecc.xRight)
                    ptl.x = rclRecc.xRight;
                if (ptl.y > rclRecc.yTop)
                    ptl.y = rclRecc.yTop;

                // convert this to screen coordinates
                WinMapWindowPoints(hwndCnr,
                           HWND_DESKTOP,
                           &ptl,
                           1);
                fQueried = TRUE;
            }
        }

        if (!fQueried)
            // else: use mouse coordinates for context menu
            WinQueryPointerPos(HWND_DESKTOP, &ptl);

        // give preccSource source emphasis;
        // if preccSource == NULL, the whole container will be
        // given source emphasis
        WinSendMsg(hwndCnr,
                   CM_SETRECORDEMPHASIS,
                   (MPARAM)preccSource,     // might be NULL for whole container
                   MPFROM2SHORT(TRUE,  // set emphasis
                            CRA_SOURCE));

        // finally, show context menu
        brc = WinPopupMenu(HWND_DESKTOP,               // menu parent
                           hwndMenuOwner,              // menu owner
                           hMenu,
                           (SHORT)ptl.x,
                           (SHORT)ptl.y,
                           0,                          // ID
                           PU_NONE
                              | PU_MOUSEBUTTON1
                              | PU_KEYBOARD
                              | PU_HCONSTRAIN
                              | PU_VCONSTRAIN);
    }

    return (brc);
}

/* ******************************************************************
 *                                                                  *
 *   Generic window helpers                                         *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhQueryWindowText:
 *      this returns the window text of the specified
 *      HWND in a newly allocated buffer, which has
 *      the exact size of the window text.
 *
 *      This buffer must be free()'d later.
 */

PSZ winhQueryWindowText(HWND hwnd)
{
    PSZ     pszText = NULL;
    ULONG   cbText = WinQueryWindowTextLength(hwnd) + 1;
                                    // additional null character
    if (cbText)
    {
        pszText = (PSZ)malloc(cbText);
        if (pszText)
        {
            WinQueryWindowText(hwnd,
                               cbText,
                               pszText);
        }
    }
    return (pszText);
}

/*
 *@@ winhSaveWindowPos:
 *      saves the position of a certain window. As opposed
 *      to the barely documented WinStoreWindowPos API, this
 *      one only saves one regular SWP structure for the given
 *      window. It does _not_ save minimized/maximized positions,
 *      presparams or anything about child windows.
 *      The window should still be visible on the screen
 *      when calling this function. Do not call it in WM_DESTROY,
 *      because then the SWP data is no longer valid.
 *      Returns TRUE if saving was successful.
 */

BOOL winhSaveWindowPos(HWND hwnd,   // in: window to save
                       HINI hIni,   // in: INI file (or HINI_USER/SYSTEM)
                       PSZ pszApp,  // in: INI application name
                       PSZ pszKey)  // in: INI key name
{
    BOOL brc = FALSE;
    SWP swp;
    if (WinQueryWindowPos(hwnd, &swp))
    {
        brc = PrfWriteProfileData(hIni, pszApp, pszKey, &swp, sizeof(swp));
    }
    return (brc);
}

/*
 *@@ winhRestoreWindowPos:
 *      this will retrieve a window position which was
 *      previously stored using winhSaveWindowPos.
 *      The window should not be visible to avoid flickering.
 *      "fl" must contain the SWP_flags as in WinSetWindowPos.
 *
 *      Note that only the following may be used:
 *      --  SWP_MOVE        reposition the window
 *      --  SWP_SIZE        also resize the window to
 *                          the stored position; this might
 *                          lead to problems with different
 *                          video resolutions, so be careful.
 *      --  SWP_SHOW        make window visible too
 *      --  SWP_NOREDRAW    changes are not redrawn
 *      --  SWP_NOADJUST    do not send a WM_ADJUSTWINDOWPOS message
 *                          before moving or sizing
 *      --  SWP_ACTIVATE    activate window (make topmost)
 *      --  SWP_DEACTIVATE  deactivate window (make bottommost)
 *
 *      Do not specify any other SWP_* flags.
 *
 *      If SWP_SIZE is not set, the window will be moved only
 *      (recommended).
 *
 *      This returns TRUE if INI data was found.
 *
 *      This function automatically checks for whether the
 *      window would be positioned outside the visible screen
 *      area and will adjust coordinates accordingly. This can
 *      happen when changing video resolutions.
 */

BOOL winhRestoreWindowPos(HWND hwnd,   // in: window to save
                       HINI hIni,   // in: INI file (or HINI_USER/SYSTEM)
                       PSZ pszApp,  // in: INI application name
                       PSZ pszKey,  // in: INI key name
                       ULONG fl)    // in: "fl" parameter for WinSetWindowPos
{
    BOOL    brc = FALSE;
    SWP     swp, swpNow;
    ULONG   cbswp = sizeof(swp);
    ULONG   fl2 = fl;

    if (PrfQueryProfileData(hIni, pszApp, pszKey, &swp, &cbswp))
    {
        ULONG ulScreenCX = WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
        ULONG ulScreenCY = WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);

        brc = TRUE;

        if ((fl & SWP_SIZE) == 0) {
            // if no resize, we need to get the current position
            brc = WinQueryWindowPos(hwnd, &swpNow);
            swp.cx = swpNow.cx;
            swp.cy = swpNow.cy;
        }

        if (brc) {
            // check for full visibility
            if ( (swp.x + swp.cx) > ulScreenCX)
                swp.x = ulScreenCX - swp.cx;
            if ( (swp.y + swp.cy) > ulScreenCY)
                swp.y = ulScreenCY - swp.cy;
        }

        brc = TRUE;

    } else {
        // window pos not found in INI: unset SWP_MOVE etc.
        fl2 &= ~(SWP_MOVE | SWP_SIZE);
    }

    WinSetWindowPos(hwnd,
                    NULLHANDLE,       // insert-behind window
                    swp.x,
                    swp.y,
                    swp.cx,
                    swp.cy,
                    fl2);        // SWP_* flags

    return (brc);
}

/*
 *@@ winhCenterWindow:
 *      centers a window within its parent window. If that's
 *      the PM desktop, it will be centered according to the
 *      whole screen.
 *      For dialog boxes, use WinCenteredDlgBox as a one-shot
 *      function.
 *
 *      Note: When calling this function, the window should
 *      not be visible to avoid flickering.
 *      This func does not show the window either, so call
 *      WinShowWindow afterwards.
 */

void winhCenterWindow(HWND hwnd)
{
   RECTL rclParent;
   RECTL rclWindow;

   WinQueryWindowRect(hwnd, &rclWindow);
   WinQueryWindowRect(WinQueryWindow(hwnd, QW_PARENT), &rclParent);

   rclWindow.xLeft   = (rclParent.xRight - rclWindow.xRight) / 2;
   rclWindow.yBottom = (rclParent.yTop   - rclWindow.yTop  ) / 2;

   WinSetWindowPos(hwnd, NULLHANDLE, rclWindow.xLeft, rclWindow.yBottom,
                    0, 0, SWP_MOVE);
}

/*
 *@@ winhCenteredDlgBox:
 *      just like WinDlgBox, but the dlg box is centered on the screen;
 *      you should mark the dlg template as not visible in the dlg
 *      editor, or display will flicker.
 *      As opposed to winhCenterWindow, this _does_ show the window.
 */

ULONG winhCenteredDlgBox(HWND hwndParent, HWND hwndOwner,
              PFNWP pfnDlgProc, HMODULE hmod, ULONG idDlg, PVOID pCreateParams)
{
    ULONG   ulReply;
    HWND    hwndDlg = WinLoadDlg(hwndParent, hwndOwner, pfnDlgProc,
                                hmod, idDlg, pCreateParams);
    winhCenterWindow(hwndDlg);
    ulReply = WinProcessDlg(hwndDlg);
    WinDestroyWindow(hwndDlg);
    return (ulReply);
}

/*
 *@@ winhQueryPresColor:
 *      returns the specified color. This is queried in the
 *      following order:
 *      1)  hwnd's pres params are searched for ulPP
 *          (which should be a PP_* index);
 *      2)  if this fails, WinQuerySysColor is called
 *          with lSysColor (which should be a SYSCLR_*
 *          index).
 *
 *      The return value is always an RGB LONG.
 *      If you do any painting with this value, you
 *      should switch the HPS you're using to RGB
 *      mode (use gpihSwitchToRGB for that).
 *
 *      Some useful ulPP / lSysColor pairs
 *      (default values as in PMREF):
 *
 +          --  PP_FOREGROUNDCOLOR          SYSCLR_WINDOWTEXT
 +                 Foreground color (default: black)
 +          --  PP_BACKGROUNDCOLOR          SYSCLR_BACKGROUND
 +                                          SYSCLR_DIALOGBACKGROUND
 +                                          SYSCLR_FIELDBACKGROUND
 +                 Background color (default: light gray)
 +          --  PP_ACTIVETEXTFGNDCOLOR
 +          --  PP_HILITEFOREGROUNDCOLOR    SYSCLR_HILITEFOREGROUND
 +                 Highlighted foreground color, for example for selected menu
 +                 (def.: white)
 +          --  PP_ACTIVETEXTBGNDCOLOR
 +          --  PP_HILITEBACKGROUNDCOLOR    SYSCLR_HILITEBACKGROUND
 +                 Highlighted background color (def.: dark gray)
 +          --  PP_INACTIVETEXTFGNDCOLOR
 +          --  PP_DISABLEDFOREGROUNDCOLOR  SYSCLR_MENUDISABLEDTEXT
 +                 Disabled foreground color (dark gray)
 +          --  PP_INACTIVETEXTBGNDCOLOR
 +          --  PP_DISABLEDBACKGROUNDCOLOR
 +                 Disabled background color
 +          --  PP_BORDERCOLOR              SYSCLR_WINDOWFRAME
 +                                          SYSCLR_INACTIVEBORDER
 +                 Border color
 +          --  PP_ACTIVECOLOR              SYSCLR_ACTIVETITLE
 +                 Active color
 +          --  PP_INACTIVECOLOR            SYSCLR_INACTIVETITLE
 +                 Inactive color
 *
 *      For menus:
 +          --  PP_MENUBACKGROUNDCOLOR      SYSCLR_MENU
 +          --  PP_MENUFOREGROUNDCOLOR      SYSCLR_MENUTEXT
 +          --  PP_MENUHILITEBGNDCOLOR      SYSCLR_MENUHILITEBGND
 +          --  PP_MENUHILITEFGNDCOLOR      SYSCLR_MENUHILITE
 *
 *  PMREF has more of these.
 *
 *@@changed V1.00: removed INI key query, using SYSCLR_* instead; function prototype changed
 */

LONG winhQueryPresColor(HWND    hwnd,       // in: window to query
                        ULONG   ulPP,       // in: PP_* index
                        LONG    lSysColor)  // in: SYSCLR_* index
{
    ULONG ul, attrFound, abValue[32];
    if (ulPP != (ULONG)-1)
        if (ul = WinQueryPresParam(hwnd,
                    ulPP,
                    0,
                    &attrFound,
                    (ULONG)sizeof(abValue),
                    (PVOID)&abValue,
                    0))
            return (abValue[0]);

    return (WinQuerySysColor(HWND_DESKTOP, lSysColor, 0));
}

/*
 *@@ winhCreateStdWindow:
 *      much like WinCreateStdWindow, but this one
 *      allows you to have the standard window
 *      positioned automatically, using a given
 *      SWP structure (*pswpFrame).
 *      If you want the window to be shown, specify
 *      SWP_SHOW (and maybe SWP_ACTIVATE) in *pswpFrame.
 */

HWND winhCreateStdWindow(HWND hwndFrameParent,
                         PSWP pswpFrame,            // in: frame wnd pos
                         ULONG flFrameCreateFlags,
                         PSZ pszClassClient,
                         PSZ pszFrameTitle,
                         ULONG flStyleClient,
                         ULONG ulID,
                         PHWND phwndClient)         // out: created client wnd
{
    FRAMECDATA  fcdata;
    HWND        hwndFrame;
    RECTL       rclClient;

    fcdata.cb            = sizeof(FRAMECDATA);
    fcdata.flCreateFlags = flFrameCreateFlags;
    fcdata.hmodResources = (HMODULE)NULL;
    fcdata.idResources   = 0;

    /* Create the frame and client windows.  */
    hwndFrame = WinCreateWindow(
                hwndFrameParent,
                WC_FRAME,        /* Frame-window class         */
                pszFrameTitle,
                0,               /* Initially invisible        */
                0,0,0,0,         /* Size and position = 0      */
                NULLHANDLE,      /* No owner                   */
                HWND_TOP,        /* Top z-order position       */
                ulID,            /* Frame-window ID            */
                &fcdata,         /* Pointer to class data      */
                NULL);           /* No presentation parameters */

    if (hwndFrame)
    {
        *phwndClient = WinCreateWindow(
                     hwndFrame,      /* Client-window parent        */
                     pszClassClient, /* Client-window class         */
                     NULL,           /* No title for client window  */
                     0,              /* Initially invisible         */
                     0,0,0,0,        /* Size and position = 0       */
                     hwndFrame,            /* Owner is frame window */
                     HWND_BOTTOM,    /* Bottom z-order position     */
                     FID_CLIENT,     /* Standard client-window ID   */
                     NULL,           /* No class data               */
                     NULL);          /* No presentation parameters  */

        if (*phwndClient)
        {
            WinSetWindowPos(hwndFrame,
                            pswpFrame->hwndInsertBehind,
                            pswpFrame->x,
                            pswpFrame->y,
                            pswpFrame->cx,
                            pswpFrame->cy,
                            pswpFrame->fl);

            /* Set the size and position of the client window. */
            WinQueryWindowRect(hwndFrame, &rclClient);
            WinCalcFrameRect(hwndFrame, &rclClient,
                             TRUE);     // calc client from frame
            WinSetWindowPos(*phwndClient,
                            HWND_TOP,
                            rclClient.xLeft,
                            rclClient.yBottom,
                            rclClient.xRight - rclClient.xLeft,
                            rclClient.yTop - rclClient.yBottom,
                            SWP_MOVE | SWP_SIZE | SWP_SHOW);
        }
    }
    return (hwndFrame);
}

/*
 *@@ winhRepaintWindows:
 *      this repaints all children of hwndParent.
 *      If this is passed as HWND_DESKTOP, the
 *      whole screen is repainted.
 */

VOID winhRepaintWindows(HWND hwndParent)
{
    HWND    hwndTop;
    HENUM   henum = WinBeginEnumWindows(HWND_DESKTOP);
    while (hwndTop = WinGetNextWindow(henum))
        if (WinIsWindowShowing(hwndTop))
            WinInvalidateRect(hwndTop, NULL, TRUE);
    WinEndEnumWindows(henum);
}

/*
 *@@ winhCreateFakeDesktop:
 *      this routine creates and displays a frameless window over
 *      the whole screen to fool the user that all windows
 *      have been closed (which in fact might not be the
 *      case).
 *      This window's background color is set to the Desktop's
 *      (PM's one, not the WPS's one).
 *      Returns the hwnd of this window.
 */

HWND winhCreateFakeDesktop(HWND hwndSibling)
{
    typedef struct _BACKGROUND {
        ULONG   cb;         //  Length of the aparam parameter, in bytes.
        ULONG   id;     //  Attribute type identity.
        ULONG   cb2;     //  Byte count of the ab parameter.
        RGB     rgb;    //  Attribute value.
     } BACKGROUND;
    RECTL       R;
    BACKGROUND  background;
    ULONG       flStyle;          // window style
    LONG        lDesktopColor;

    /* create fake desktop window = empty window with
       the size of full screen */
    lDesktopColor = WinQuerySysColor(HWND_DESKTOP,
        SYSCLR_BACKGROUND, 0);
    flStyle = WS_VISIBLE;
    R.xLeft      = 0;
    R.yBottom    = 0;
    R.xRight     = WinQuerySysValue(HWND_DESKTOP,SV_CXSCREEN);
    R.yTop       = WinQuerySysValue(HWND_DESKTOP,SV_CYSCREEN);
    background.cb = sizeof(background.id)
                  + sizeof(background.cb)
                  + sizeof(background.rgb);
    background.id = PP_BACKGROUNDCOLOR;
    background.cb2 = sizeof(RGB);
    background.rgb.bBlue = (CHAR1FROMMP(lDesktopColor));
    background.rgb.bGreen= (CHAR2FROMMP(lDesktopColor));
    background.rgb.bRed  = (CHAR3FROMMP(lDesktopColor));

    return (WinCreateWindow(HWND_DESKTOP,  // parent window
                           WC_FRAME,      // class name
                           "",            // window text
                           flStyle,       // window style
                           0, 0,          // position (x,y)
                           R.xRight,
                           R.yTop,
                           NULLHANDLE,    // owner window
                           hwndSibling,    // sibling window
                           1,             // window id
                           NULL,          // control data
                           &background)); // presentation parms
}

/*
 *@@ winhQueryCnrFromFrame:
 *      find the handle of a frame window's container; we do this
 *      by enumerating all the frame's child windows and comparing
 *      their window classes to the WC_CONTAINER code.
 *      This is not terribly fast, so use this func economically.
 *      Returns NULLHANDLE if not found.
 */

HWND winhQueryCnrFromFrame(HWND hwndFrame)
{
    HENUM               henum;
    CHAR                szClassName[256];
    HWND                hwndCnr = NULLHANDLE,
                        hwndTemp = NULLHANDLE;

    if (hwndFrame) {
        henum = WinBeginEnumWindows(hwndFrame);
        if (henum) {
            do {
                hwndTemp = WinGetNextWindow(henum);
                if (hwndTemp) {
                    if (WinQueryClassName(hwndTemp, 250, szClassName))
                        if (strcmp(szClassName, "#37") == 0)
                            // unbelievable, this is PM's code for WC_CONTAINER...
                            hwndCnr = hwndTemp;
                }
            } while (hwndTemp);
        }
        WinEndEnumWindows(henum);
    }

    return hwndCnr;
}

/*
 *@@ winhAssertWarp4Notebook:
 *      this takes hwndDlg as a notebook dialog page and
 *      goes thru all its controls. If a control with an
 *      ID <= udIdThreshold is found, this is assumed to
 *      be a button which is to be given the BS_NOTEBOOKBUTTON
 *      style. You should therefore give all your button
 *      controls which should be moved such an ID.
 *      You can also specify how many dialog units
 *      all the other controls will be moved downward in
 *      ulDownUnits; this is useful to fill up the space
 *      which was used by the buttons before moving them.
 *      Returns TRUE if anything was changed.
 *
 *      This function is useful if you wish to create
 *      notebook pages using dlgedit.exe which are compatible
 *      with both Warp 3 and Warp 4. This should be executed
 *      in WM_INITDLG of the notebook dlg function if the app
 *      has determined that it is running on Warp 4.
 */

BOOL winhAssertWarp4Notebook(HWND hwndDlg,
                    USHORT usIdThreshold,    // in: ID threshold
                    ULONG ulDownUnits)       // in: dialog units or 0
{
    BOOL brc = FALSE;
    POINTL ptl;
    HAB hab = WinQueryAnchorBlock(hwndDlg);
    BOOL    fIsVisible = WinIsWindowVisible(hwndDlg);
    if (ulDownUnits) {
        ptl.x = 0;
        ptl.y = ulDownUnits;
        WinMapDlgPoints(hwndDlg, &ptl, 1, TRUE);
    }

    WinEnableWindowUpdate(hwndDlg, FALSE);

    if (doshIsWarp4()) {
        HENUM henum = WinBeginEnumWindows(hwndDlg);
        HWND hwndItem;
        while (hwndItem = WinGetNextWindow(henum)) {
            USHORT usId = WinQueryWindowUShort(hwndItem, QWS_ID);
            // _Pmpf(("hwndItem: 0x%lX, ID: 0x%lX", hwndItem, usId));
            if (usId <= usIdThreshold) {
                // pushbutton to change:
                // _Pmpf(("  Setting bit"));
                WinSetWindowBits(hwndItem, QWL_STYLE,
                        BS_NOTEBOOKBUTTON, BS_NOTEBOOKBUTTON);
                brc = TRUE;
            } else
                // no pushbutton to change: move downwards
                // if desired
                if (ulDownUnits)
                {
                    SWP swp;
                    // _Pmpf(("Moving downwards %d pixels", ptl.y));
                    WinQueryWindowPos(hwndItem, &swp);
                    WinSetWindowPos(hwndItem, 0,
                        swp.x,
                        swp.y - ptl.y,
                        0, 0,
                        SWP_MOVE);
                }
        }
        WinEndEnumWindows(henum);
    }
    if (fIsVisible)
        WinShowWindow(hwndDlg, TRUE);
    return (brc);
}

/*
 *@@ winhDrawFormattedText:
 *      this func takes a rectangle and draws pszText into
 *      it, breaking the words as neccessary. The line spacing
 *      is determined from the font currently selected in hps.
 *      As opposed to WinDrawText, this can draw several lines
 *      at once.
 *      This returns the number of lines drawn.
 */

ULONG winhDrawFormattedText(HPS hps,     // in: presentation space; its settings
                                         // are used, but not altered
                            PRECTL prcl, // in/out: rectangle to use for drawing;
                                         // both y coords are modified to return
                                         // the space which was used for drawing
                            PSZ pszText, // in: text to draw
                            ULONG flCmd) // in: flags like in WinDrawText; I have
                                         // only tested DT_TOP and DT_LEFT though.
                                         // DT_WORDBREAK | DT_TEXTATTRS are always
                                         // set.
                                         // You can specify DT_QUERYEXTENT to only
                                         // have prcl calculated without drawing.
{
    PSZ     p = pszText;
    LONG    lDrawn = 1,
            lTotalDrawn = 0,
            lLineCount = 0,
            lOrigYTop = prcl->yTop;
    ULONG   ulTextLen = strlen(pszText),
            ulCharHeight,
            flCmd2;
    // POINTL  aptlText[TXTBOX_COUNT];
    RECTL   rcl2;

    // printf("Entering winhDrawFormattedText\n");

    flCmd2 = flCmd | DT_WORDBREAK | DT_TEXTATTRS;

    ulCharHeight = gpihQueryLineSpacing(hps, pszText);
    // _Pmpf(("  ulCharHeight: %d\n", ulCharHeight));

    while (   (lDrawn)
           && (lTotalDrawn < ulTextLen)
          )
    {
        memcpy(&rcl2, prcl, sizeof(rcl2));
        lDrawn = WinDrawText(hps,
                ulTextLen-lTotalDrawn,
                p,
                &rcl2,
                0, 0,                       // colors
                flCmd2
            );
        p += lDrawn;
        lTotalDrawn += lDrawn;
        prcl->yTop -= ulCharHeight;
        lLineCount++;
        // printf("  lDrawn: %d, lTotalDrawn: %d\n", lDrawn, lTotalDrawn);
    }
    // printf("  LineCount: %d\n", lLineCount);
    prcl->yBottom = prcl->yTop;
    prcl->yTop = lOrigYTop;

    return (lLineCount);
}

/*
 *@@ winhKillTasklist:
 *      this will destroy the Tasklist (window list) window.
 *      Note: you will only be able to get it back after a
 *      reboot, not a WPS restart. Only for use at shutdown and such.
 *      This trick by Uri J. Stern at
 *      http://zebra.asta.fh-weingarten.de/os2/Snippets/Howt8881.HTML
 */

VOID winhKillTasklist(VOID)
{
    SWBLOCK  swblock;
    HWND     hwndTasklist;
    // the tasklist has entry #0 in the SWBLOCK
    WinQuerySwitchList(NULLHANDLE, &swblock, sizeof(SWBLOCK));
    hwndTasklist = swblock.aswentry[0].swctl.hwnd;
    WinPostMsg(hwndTasklist,
                0x0454,     // undocumented msg for killing tasklist
                NULL, NULL);
}

/*
 *@@ winhQueryPendingSpoolJobs:
 *      returns the number of pending print jobs in the spooler
 *      or 0 if none. Useful for testing before shutdown.
 */

ULONG winhQueryPendingSpoolJobs(VOID)
{
    // BOOL    rcPending = FALSE;
    ULONG   ulTotalJobCount = 0;

    SPLERR splerr;
    USHORT jobCount ;
    ULONG  cbBuf ;
    ULONG  cTotal;
    ULONG  cReturned ;
    ULONG  cbNeeded ;
    ULONG  ulLevel ;
    ULONG  i,j ;
    PSZ    pszComputerName ;
    PBYTE  pBuf = NULL;
    PPRQINFO3 prq ;
    PPRJINFO2 prj2 ;

    ulLevel = 4L;
    pszComputerName = (PSZ)NULL ;
    splerr = SplEnumQueue(pszComputerName, ulLevel, pBuf, 0L, // cbBuf
                          &cReturned, &cTotal,
                          &cbNeeded, NULL);
    if ( splerr == ERROR_MORE_DATA || splerr == NERR_BufTooSmall )
    {
        if (!DosAllocMem( (PPVOID)&pBuf, cbNeeded,
                          PAG_READ | PAG_WRITE | PAG_COMMIT) )
        {
            cbBuf = cbNeeded ;
            splerr = SplEnumQueue(pszComputerName, ulLevel, pBuf, cbBuf,
                                    &cReturned, &cTotal,
                                    &cbNeeded, NULL);
            if (splerr == NO_ERROR)
            {
                // set pointer to point to the beginning of the buffer
                prq = (PPRQINFO3)pBuf ;

                // cReturned has the count of the number of PRQINFO3 structures
                for (i=0;i < cReturned ; i++)
                {
                    // save the count of jobs; there are this many PRJINFO2
                    // structures following the PRQINFO3 structure
                    jobCount = prq->cJobs;
                    // _Pmpf(( "Job count in this queue is %d",jobCount ));

                    // increment the pointer past the PRQINFO3 structure
                    prq++;

                    // set a pointer to point to the first PRJINFO2 structure
                    prj2=(PPRJINFO2)prq;
                    for (j=0;j < jobCount ; j++)
                    {
                        // increment the pointer to point to the next structure
                        prj2++;
                        // increase the job count, which we'll return
                        ulTotalJobCount++;

                    } // endfor jobCount

                    // after doing all the job structures, prj2 points to the next
                    // queue structure; set the pointer for a PRQINFO3 structure
                    prq = (PPRQINFO3)prj2;
                } //endfor cReturned
            } // endif NO_ERROR
            DosFreeMem(pBuf) ;
        }
    } // end if Q level given

    return (ulTotalJobCount);
}

/* ******************************************************************
 *                                                                  *
 *   WPS Class List helpers                                         *
 *                                                                  *
 ********************************************************************/

/*
 *@@ winhQueryWPSClassList:
 *      this returns the WPS class list in a newly
 *      allocated buffer.
 *
 *      The return value is actually of the POBJCLASS type,
 *      so you better cast this manually. We declare this
 *      this as PBYTE though because POBJCLASS requires
 *      INCL_WINWORKPLACE.
 *      See WinEnumObjectClasses() for details.
 *
 *      The buffer is allocated using malloc(), so
 *      you should free() it when it is no longer
 *      needed.
 *
 *      This returns NULL if an error occured.
 *
 *@@added V1.00
 */

PBYTE winhQueryWPSClassList(VOID)
{
    ULONG       ulSize;
    POBJCLASS   pObjClass = 0;

    // get WPS class list size
    if (WinEnumObjectClasses(NULL, &ulSize))
    {
        // allocate buffer
        pObjClass = (POBJCLASS)malloc(ulSize+1);
        // and load the classes into it
        WinEnumObjectClasses(pObjClass, &ulSize);
    }

    return ((PBYTE)pObjClass);
}

/*
 *@@ winhQueryWPSClass:
 *      this returns the POBJCLASS item if pszClass is registered
 *      with the WPS or NULL if the class could not be found.
 *
 *      The return value is actually of the POBJCLASS type,
 *      so you better cast this manually. We declare this
 *      this as PBYTE though because POBJCLASS requires
 *      INCL_WINWORKPLACE.
 *
 *      This takes as input the return value of winhQueryWPSClassList,
 *      which you must call first.
 *
 *      <B>Usage:</B>
 +          PBYTE   pClassList = winhQueryWPSClassList(),
 +                  pWPFolder;
 +          if (pClassList)
 +          {
 +              if (pWPFolder = winhQueryWPSClass(pClassList, "WPFolder"))
 +                  ...
 +              free(pClassList);
 +          }
 *
 *@@added V1.00
 */

PBYTE winhQueryWPSClass(PBYTE pObjClass,  // in: buffer returned by
                                          // winhQueryWPSClassList
                        PSZ pszClass)     // in: class name to query
{
    PBYTE   pbReturn = 0;

    POBJCLASS pocThis = (POBJCLASS)pObjClass;
    // now go thru the WPS class list
    while (pocThis)
    {
        if (strcmp(pocThis->pszClassName, pszClass) == 0)
        {
            pbReturn = (PBYTE)pocThis;
            break;
        }
        // next class
        pocThis = pocThis->pNext;
    } // end while (pocThis)

    return (pbReturn);
}

/*
 *@@ winhRegisterClass:
 *      this works just like WinRegisterObjectClass,
 *      except that it returns a more meaningful
 *      error code than just FALSE in case registering
 *      fails.
 *
 *      This returns NO_ERROR if the class was successfully
 *      registered (WinRegisterObjectClass returned TRUE).
 *
 *      Otherwise, we do a DosLoadModule if maybe the DLL
 *      couldn't be loaded in the first place. If DosLoadModule
 *      did not return NO_ERROR, this function returns that
 *      return code, which can be:
 *
 *      --  2   ERROR_FILE_NOT_FOUND: pcszModule does not exist
 *      --  2   ERROR_FILE_NOT_FOUND
 *      --  3   ERROR_PATH_NOT_FOUND
 *      --  4   ERROR_TOO_MANY_OPEN_FILES
 *      --  5   ERROR_ACCESS_DENIED
 *      --  8   ERROR_NOT_ENOUGH_MEMORY
 *      --  11  ERROR_BAD_FORMAT
 *      --  26  ERROR_NOT_DOS_DISK (unknown media type)
 *      --  32  ERROR_SHARING_VIOLATION
 *      --  33  ERROR_LOCK_VIOLATION
 *      --  36  ERROR_SHARING_BUFFER_EXCEEDED
 *      --  95  ERROR_INTERRUPT (interrupted system call)
 *      --  108 ERROR_DRIVE_LOCKED (by another process)
 *      --  123 ERROR_INVALID_NAME (illegal character or FS name not valid)
 *      --  127 ERROR_PROC_NOT_FOUND (DosQueryProcAddr error)
 *      --  180 ERROR_INVALID_SEGMENT_NUMBER
 *      --  182 ERROR_INVALID_ORDINAL
 *      --  190 ERROR_INVALID_MODULETYPE (probably an application)
 *      --  191 ERROR_INVALID_EXE_SIGNATURE (probably not LX DLL)
 *      --  192 ERROR_EXE_MARKED_INVALID (by linker)
 *      --  194 ERROR_ITERATED_DATA_EXCEEDS_64K (in a DLL segment)
 *      --  195 ERROR_INVALID_MINALLOCSIZE
 *      --  196 ERROR_DYNLINK_FROM_INVALID_RING
 *      --  198 ERROR_INVALID_SEGDPL
 *      --  199 ERROR_AUTODATASEG_EXCEEDS_64K
 *      --  201 ERROR_RELOCSRC_CHAIN_EXCEEDS_SEGLIMIT
 *      --  206 ERROR_FILENAME_EXCED_RANGE (not matching 8+3 spec)
 *      --  295 ERROR_INIT_ROUTINE_FAILED (DLL init routine failed)
 *
 *      In all these cases, pszBuf may contain a meaningful
 *      error message from DosLoadModule, especially if an import
 *      could not be resolved.
 *
 *      Still worse, if DosLoadModule returned NO_ERROR, we
 *      probably have some SOM internal error. A probable
 *      reason is that the parent class of pcszClassName
 *      is not installed, but that's WPS/SOM internal
 *      and cannot be queried from outside the WPS context.
 *
 *      In that case, ERROR_OPEN_FAILED (110) is returned.
 *      That one sounded good to me. ;-)
 */

APIRET winhRegisterClass(const char* pcszClassName, // in: e.g. "XFolder"
                         const char* pcszModule,    // in: e.g. "C:\XFOLDER\XFLDR.DLL"
                         PSZ pszBuf,                // out: error message from DosLoadModule
                         ULONG cbBuf)               // in: sizeof(*pszBuf), passed to DosLoadModule
{
    APIRET arc = NO_ERROR;

    if (!WinRegisterObjectClass(pcszClassName, pcszModule))
    {
        // failed: do more error checking then, try DosLoadModule
        HMODULE hmod = NULLHANDLE;
        arc = DosLoadModule(pszBuf, cbBuf,
                            pcszModule,
                            &hmod);
        if (arc == NO_ERROR)
        {
            // DosLoadModule succeeded:
            // some SOM error then
            DosFreeModule(hmod);
            arc = ERROR_OPEN_FAILED;
        }
    }
    // else: ulrc still 0 (== no error)

    return (arc);
}

/* ******************************************************************
 *                                                                  *
 *   Progress bars                                                  *
 *                                                                  *
 ********************************************************************/

/*
 * PaintProgress:
 *      this does the actual painting of the progress bar.
 *      It is called both by WM_PAINT and by WM_UPDATEPROGRESSBAR
 *      with different HPS's then.
 */

VOID PaintProgress(PPROGRESSBARDATA pData, HWND hwndBar, HPS hps)
{
    POINTL  ptl1, ptlText, aptlText[TXTBOX_COUNT];
    RECTL   rcl, rcl2;

    CHAR    szPercent[10] = "";

    // switch to RGB mode
    GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL);

    if (pData->ulPaintX <= pData->ulOldPaintX)
    {
        // draw frame and background only if this is either
        //    a "real" WM_PAINT (i.e., the window was overlapped
        //   and needs repainting; then ulPaintX == ulOldPaintX)
        //   or if ulNow has _de_creased
        GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0));
        ptl1.x = 0;
        ptl1.y = 0;
        GpiMove(hps, &ptl1);
        ptl1.y = (pData->rclBar.yTop);
        GpiLine(hps, &ptl1);
        ptl1.x = (pData->rclBar.xRight);
        GpiLine(hps, &ptl1);
        GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0));
        ptl1.y = 0;
        GpiLine(hps, &ptl1);
        ptl1.x = 0;
        GpiLine(hps, &ptl1);

        pData->rclBar.xLeft = 1;
        pData->rclBar.yBottom = 1;
        WinFillRect(hps, &(pData->rclBar),
            WinQuerySysColor(HWND_DESKTOP, SYSCLR_SCROLLBAR, 0));
    }

    // draw percentage?
    if (pData->ulAttr & PBA_PERCENTFLAGS)
    {
        // make string
        sprintf(szPercent, "%d %%", ((100 * pData->ulNow) / pData->ulMax) );

        // calculate string space
        GpiQueryTextBox(hps, strlen(szPercent), szPercent,
                TXTBOX_COUNT, (PPOINTL)&aptlText);

        // calculate coordinates
        ptlText.x = pData->rclBar.xLeft +
                        (   (   (pData->rclBar.xRight-pData->rclBar.xLeft)
                              - (aptlText[TXTBOX_BOTTOMRIGHT].x-aptlText[TXTBOX_BOTTOMLEFT].x)
                            )
                        / 2);
        ptlText.y = 3 + pData->rclBar.yBottom +
                        (   (   (pData->rclBar.yTop-pData->rclBar.yBottom)
                              - (aptlText[TXTBOX_TOPLEFT].y-aptlText[TXTBOX_BOTTOMLEFT].y)
                            )
                        / 2);

        // do we need to repaint the background under the percentage?
        if (    (   (ptlText.x
                        + (  aptlText[TXTBOX_BOTTOMRIGHT].x
                           - aptlText[TXTBOX_BOTTOMLEFT].x)
                    )
                  > pData->ulPaintX)
            &&  (pData->ulPaintX > pData->ulOldPaintX)
           )
        {
            // if we haven't drawn the background already,
            // we'll need to do it now for the percentage area
            rcl.xLeft      = ptlText.x;
            rcl.xRight     = ptlText.x + (aptlText[TXTBOX_BOTTOMRIGHT].x-aptlText[TXTBOX_BOTTOMLEFT].x);
            rcl.yBottom    = ptlText.y;
            rcl.yTop       = ptlText.y + (aptlText[TXTBOX_TOPLEFT].y-aptlText[TXTBOX_BOTTOMLEFT].y);
            WinFillRect(hps, &rcl,
                WinQuerySysColor(HWND_DESKTOP, SYSCLR_SCROLLBAR, 0));
        }
    }

    // now draw the actual progress
    rcl2.xLeft = pData->rclBar.xLeft;
    rcl2.xRight = (pData->ulPaintX > (rcl2.xLeft + 3))
            ? pData->ulPaintX
            : rcl2.xLeft + ((pData->ulAttr & PBA_BUTTONSTYLE)
                           ? 3 : 1);
    rcl2.yBottom = pData->rclBar.yBottom;
    rcl2.yTop = pData->rclBar.yTop-1;

    if (pData->ulAttr & PBA_BUTTONSTYLE)
    {
        RECTL rcl3 = rcl2;
        // draw "raised" inner rect
        rcl3.xLeft = rcl2.xLeft;
        rcl3.yBottom = rcl2.yBottom;
        rcl3.yTop = rcl2.yTop+1;
        rcl3.xRight = rcl2.xRight+1;
        gpihDraw3DFrame(hps, &rcl3, 2, TRUE);
        // WinInflateRect(WinQueryAnchorBlock(hwndBar), &rcl2, -2, -2);
        rcl2.xLeft += 2;
        rcl2.yBottom += 2;
        rcl2.yTop -= 2;
        rcl2.xRight -= 2;
    }

    if (rcl2.xRight > rcl2.xLeft) {
        GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP,
                    // SYSCLR_HILITEBACKGROUND,
                    SYSCLR_BUTTONMIDDLE,
                0));
        ptl1.x = rcl2.xLeft;
        ptl1.y = rcl2.yBottom;
        GpiMove(hps, &ptl1);
        ptl1.x = rcl2.xRight;
        ptl1.y = rcl2.yTop;
        GpiBox(hps, DRO_FILL | DRO_OUTLINE,
            &ptl1,
            0,
            0);
    }

    // now print the percentage
    if (pData->ulAttr & PBA_PERCENTFLAGS)
    {
        GpiMove(hps, &ptlText);
        GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP,
                // SYSCLR_HILITEFOREGROUND,
                SYSCLR_BUTTONDEFAULT,
            0));
        GpiCharString(hps, strlen(szPercent), szPercent);
    }

    // store current X position for next time
    pData->ulOldPaintX = pData->ulPaintX;
}

/*
 *@@ fnwpProgressBar:
 *      this is the window procedure for the progress bar control,
 *      which is a static rectangle control subclassed by
 *      winhProgressBarFromStatic.
 *
 *      We need to capture WM_PAINT to draw the progress bar according
 *      to the current progress, and we also update the static text field
 *      (percentage) next to it.
 *
 *      This also evaluates the WM_UPDATEPROGRESSBAR message.
 *
 *@@changed V1.00: moved this code to winh.c
 */

MRESULT EXPENTRY fnwpProgressBar(HWND hwndBar, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    HPS     hps;
    PPROGRESSBARDATA    pData = (PPROGRESSBARDATA)WinQueryWindowULong(hwndBar, QWL_USER);

    PFNWP   OldStaticProc = NULL;

    MRESULT mrc = NULL;

    if (pData) {
        OldStaticProc = pData->OldStaticProc;

        switch(msg)
        {

            /*
             * WM_UPDATEPROGRESSBAR:
             *      post or send this message to the progress
             *      bar to have a new progress displayed.
             *      Parameters:
             *          ULONG mp1   current value
             *          ULONG mp2   max value
             *      Example: mp1 = 100, mp2 = 300 will result
             *      in a progress of 33%.
             */

            case WM_UPDATEPROGRESSBAR: {
                if (    (pData->ulNow != (ULONG)mp1)
                     || (pData->ulMax != (ULONG)mp2)
                   )
                {
                    pData->ulNow = (ULONG)mp1;
                    pData->ulMax = (ULONG)mp2;
                }
                else
                    // value not changed: do nothing
                    break;

                // check validity
                if (pData->ulNow > pData->ulMax)
                    pData->ulNow = pData->ulMax;
                // avoid division by zero
                if (pData->ulMax == 0) {
                    pData->ulMax = 1;
                    pData->ulNow = 0;
                    // paint 0% then
                }

                // calculate new X position of the progress
                pData->ulPaintX =
                    (ULONG)(
                              (    (ULONG)(pData->rclBar.xRight-pData->rclBar.xLeft)
                                 * (ULONG)(pData->ulNow)
                              )
                              /  (ULONG)pData->ulMax
                           );
                if (pData->ulPaintX != pData->ulOldPaintX) {
                    // X position changed: redraw
                    // WinInvalidateRect(hwndBar, NULL, FALSE);
                    HPS hps = WinGetPS(hwndBar);
                    PaintProgress(pData, hwndBar, hps);
                    WinReleasePS(hps);
                }
            break; }

            case WM_PAINT: {
                RECTL rcl;
                hps = WinBeginPaint(hwndBar, NULLHANDLE, &rcl);
                PaintProgress(pData, hwndBar, hps);
                WinEndPaint(hps);
            break; }

            default:
                mrc = OldStaticProc(hwndBar, msg, mp1, mp2);
       }
    }
    return (mrc);
}

/*
 *@@ winhProgressBarFromStatic:
 *      this function turns an existing static rectangle control
 *      into a progress bar by subclassing its window procedure
 *      with fnwpProgressBar.
 *      This way you can easily create a progress bar as a static
 *      control in any Dialog Editor;
 *      after loading the dlg template, simply call this function
 *      with the hwnd of the static control to make it a status bar.
 *
 *      In order to _update_ the progress bar, simply post or send
 *      WM_UPDATEPROGRESSBAR to the static (= progress bar) window;
 *      this message is equal to WM_USER and needs the following
 *      parameters:
 *      --  mp1     ULONG ulNow: the current progress
 *      --  mp2     ULONG ulMax: the maximally possible progress
 *                               (= 100%)
 *
 *      The progress bar automatically calculates the current progress
 *      display. For example, if ulNow = 4096 and ulMax = 8192,
 *      a progress of 50% will be shown. It is possible to change
 *      ulMax after the progress bar has started display. If ulMax
 *      is 0, a progress of 0% will be shown (to avoid division
 *      by zero traps).
 *
 *      ulAttr accepts of the following:
 *      --  PBA_NOPERCENTAGE:    do not display percentage
 *      --  PBA_ALIGNLEFT:       left-align percentage
 *      --  PBA_ALIGNRIGHT:      right-align percentage
 *      --  PBA_ALIGNCENTER:     center percentage
 *      --  PBA_BUTTONLOOK:      no "flat", but button-like look
 *
 *@@changed V1.00: moved this code to winh.c
 */

BOOL winhProgressBarFromStatic(HWND hwndStatic, ULONG ulAttr)
{
    PFNWP OldStaticProc = WinSubclassWindow(hwndStatic, fnwpProgressBar);
    if (OldStaticProc) {
        PPROGRESSBARDATA pData = (PPROGRESSBARDATA)malloc(sizeof(PROGRESSBARDATA));
        pData->ulMax = 1;
        pData->ulNow = 0;
        pData->ulPaintX = 0;
        pData->ulAttr = ulAttr;
        pData->OldStaticProc = OldStaticProc;
        WinQueryWindowRect(hwndStatic, &(pData->rclBar));
        (pData->rclBar.xRight)--;
        (pData->rclBar.yTop)--;

        WinSetWindowULong(hwndStatic, QWL_USER, (ULONG)pData);
        return (TRUE);
    }
    else return (FALSE);
}

/* ******************************************************************
 *                                                                  *
 *   Split bars                                                     *
 *                                                                  *
 ********************************************************************/

/*
 *@@ fnwpSplitWindow:
 *      this is the window procedure of the "invisible"
 *      split window. One of these windows is created
 *      for each split view and has exactly three children:
 *
 *      1)  the split bar (fnwpSplitBar);
 *
 *      2)  the left or bottom window linked to the split bar;
 *
 *      3)  the right or top window linked to the split bar.
 *
 *      See winhCreateSplitWindow for more info and a detailed
 *      window hierarchy.
 *
 *@@added V1.00
 */

MRESULT EXPENTRY fnwpSplitWindow(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = (MRESULT)0;

    switch(msg)
    {
        case WM_SIZE: {
            winhUpdateSplitWindow(hwnd);
            // return default NULL
        break; }

        case WM_PAINT:
        {
            HPS     hps = WinBeginPaint(hwnd, (HPS)0, NULL);
            HWND    hwndSplitBar = WinWindowFromID(hwnd, ID_SPLITBAR);
            PSPLITBARDATA pData = (PSPLITBARDATA)WinQueryWindowULong(hwndSplitBar,
                                                                     QWL_USER);
            if (pData->sbcd.ulCreateFlags & SBCF_3DSUNK)
            {
                // "3D-sunk" style?
                POINTL  ptl1;
                SWP     swp;

                // switch to RGB mode
                GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL);

                WinQueryWindowPos(pData->sbcd.hwndLink1, &swp);
                GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0));
                ptl1.x = swp.x - 1;
                ptl1.y = swp.y - 1;
                GpiMove(hps, &ptl1);
                ptl1.y = swp.y + swp.cy;
                GpiLine(hps, &ptl1);
                ptl1.x = swp.x + swp.cx;
                GpiLine(hps, &ptl1);
                GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0));
                ptl1.y = swp.y - 1;
                GpiLine(hps, &ptl1);
                ptl1.x = swp.x - 1;
                GpiLine(hps, &ptl1);

                WinQueryWindowPos(pData->sbcd.hwndLink2, &swp);
                GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0));
                ptl1.x = swp.x - 1;
                ptl1.y = swp.y - 1;
                GpiMove(hps, &ptl1);
                ptl1.y = swp.y + swp.cy;
                GpiLine(hps, &ptl1);
                ptl1.x = swp.x + swp.cx;
                GpiLine(hps, &ptl1);
                GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0));
                ptl1.y = swp.y - 1;
                GpiLine(hps, &ptl1);
                ptl1.x = swp.x - 1;
                GpiLine(hps, &ptl1);
            }
            WinEndPaint(hps);

            // return default NULL
        break; }

        default:
           mrc = WinDefWindowProc(hwnd, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ fnwpSplitBar:
 *      window procedure for all split bars, horizontal and vertical.
 *      This paints the split bar and handles dragging the split bar
 *      and repositioning the "linked" windows afterwards.
 *
 *      See winhCreateSplitWindow for more info and a detailed
 *      window hierarchy.
 *
 *@@added V1.00
 */

MRESULT EXPENTRY fnwpSplitBar(HWND hwndBar, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    HPS     hps;
    PSPLITBARDATA pData = (PSPLITBARDATA)WinQueryWindowULong(hwndBar, QWL_USER);

    PFNWP   OldStaticProc = NULL;

    MRESULT mrc = NULL;

    if (pData)
    {
        OldStaticProc = pData->OldStaticProc;

        switch(msg)
        {
            case WM_MOUSEMOVE: {
                pData->hptrOld = WinSetPointer(HWND_DESKTOP, pData->hptrMove);
            break; }

            /*
             * WM_BUTTON1DOWN:
             *      this will start moving the split bar.
             *      We use WinTrackRect for this.
             */

            case WM_BUTTON1DOWN: {
                TRACKINFO track;

                track.cxBorder = 2;
                track.cyBorder = 2;
                track.cxGrid   = 1;
                track.cyGrid   = 1;
                track.cxKeyboard = 8;
                track.cyKeyboard = 8;

                // limit tracking space to client window
                WinQueryWindowRect(pData->sbcd.hwndClient, &track.rclBoundary);
                WinMapWindowPoints(pData->sbcd.hwndClient,
                                   HWND_DESKTOP,
                                   (PPOINTL)&track.rclBoundary,
                                   2);      // 2 points, since we have a RECTL

                // decrease tracking limits by what was
                // specified by the caller
                if (pData->sbcd.ulCreateFlags & SBCF_HORIZONTAL)
                {
                    // horizontal split bar
                    track.rclBoundary.yBottom += pData->sbcd.ulLeftOrBottomLimit;
                    track.rclBoundary.yTop -= pData->sbcd.ulRightOrTopLimit;
                }
                else
                {
                    // vertical split bar
                    track.rclBoundary.xLeft += pData->sbcd.ulLeftOrBottomLimit;
                    track.rclBoundary.xRight -= pData->sbcd.ulRightOrTopLimit;
                }

                // initial tracking rectangle = split bar
                WinQueryWindowRect(hwndBar, &track.rclTrack);
                WinMapWindowPoints(hwndBar,
                                   HWND_DESKTOP,
                                   (PPOINTL)&track.rclTrack,
                                   2);

                track.ptlMinTrackSize.x = track.rclTrack.xRight
                                        - track.rclTrack.xLeft;
                track.ptlMinTrackSize.y = track.rclTrack.yTop
                                        - track.rclTrack.yBottom;
                track.ptlMaxTrackSize.x = track.rclTrack.xRight
                                        - track.rclTrack.xLeft;
                track.ptlMaxTrackSize.y = track.rclTrack.yTop
                                        - track.rclTrack.yBottom;

                track.fs = TF_MOVE | TF_ALLINBOUNDARY;

                if (WinTrackRect(HWND_DESKTOP, 0, &track))
                {
                    // OK, successfully moved: reposition the
                    // windows which are linked to the split bar

                    if (pData->sbcd.ulCreateFlags & SBCF_HORIZONTAL)
                    {
                        // horizontal split bar
                        ULONG ulNewYPos = track.rclTrack.yBottom
                                        - track.rclBoundary.yBottom;
                        // add the limit specified by the caller
                        ulNewYPos += pData->sbcd.ulLeftOrBottomLimit;
                    }
                    else
                    {
                        // vertical split bar:
                        ULONG ulNewXPos = track.rclTrack.xLeft
                                        - track.rclBoundary.xLeft;
                        // add the limit specified by the caller
                        ulNewXPos += pData->sbcd.ulLeftOrBottomLimit;

                        if (pData->sbcd.ulCreateFlags & SBCF_PERCENTAGE)
                        {
                            // status bar pos is determined by
                            // a percentage:
                            // well, we'll need a new percentage then
                            RECTL rclClient;
                            WinQueryWindowRect(pData->sbcd.hwndClient, &rclClient);
                            pData->sbcd.lPos = ulNewXPos
                                               * 100 / (rclClient.xRight - rclClient.xLeft);
                        } else
                            // absolute split bar positioning:
                            if (pData->sbcd.lPos > 0)
                                // from left: easy
                                pData->sbcd.lPos = ulNewXPos;
                            else
                            {
                                // negative -> from right:
                                RECTL rclClient;
                                WinQueryWindowRect(pData->sbcd.hwndClient, &rclClient);
                                // calc new negative
                                pData->sbcd.lPos = ulNewXPos - rclClient.xRight;
                            }
                    }

                    // update parent (== "split window")
                    winhUpdateSplitWindow(WinQueryWindow(hwndBar, QW_PARENT));
                }
            break; }

            /*
             * WM_PAINT:
             *      paint the split bar
             */

            case WM_PAINT: {
                RECTL rcl,
                      rclBar;
                POINTL ptl1;
                hps = WinBeginPaint(hwndBar, NULLHANDLE, &rcl);
                WinQueryWindowRect(hwndBar, &rclBar);
                // switch to RGB mode
                GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL);

                WinFillRect(hps, &rclBar,
                    WinQuerySysColor(HWND_DESKTOP, SYSCLR_INACTIVEBORDER, 0));

                if ((pData->sbcd.ulCreateFlags & SBCF_3DSUNK) == 0)
                {
                    GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0));
                    ptl1.x = 0;
                    ptl1.y = 0;
                    GpiMove(hps, &ptl1);
                    ptl1.y = (rclBar.yTop - rclBar.yBottom) - 1;
                    GpiLine(hps, &ptl1);
                    ptl1.x = (rclBar.xRight - rclBar.xLeft) - 1;
                    GpiLine(hps, &ptl1);
                    GpiSetColor(hps, WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0));
                    ptl1.y = 0;
                    GpiLine(hps, &ptl1);
                    ptl1.x = 0;
                    GpiLine(hps, &ptl1);
                }

                WinEndPaint(hps);
            break; }

            default:
                mrc = OldStaticProc(hwndBar, msg, mp1, mp2);
        }
    }
    return (mrc);
}

/*
 *@@ winhCreateSplitWindow:
 *      this creates a new split window view according to
 *      the given PSPLITBARCDATA structure.
 *
 *      <B>Creating a split view</B>
 *
 *      For this, you need to perform the following steps:
 *
 *      1)  Create two windows that shall be separated by
 *          a split bar. These can be _any_ PM windows,
 *          even dialogs loaded using WinLoadDlg.
 *
 *          Note: Make sure all your frame windows have the
 *          FCF_NOBYTEALIGN flag set, or they'll look funny.
 *
 *      2)  Fill a SPLITBARCDATA structure with the necessary
 *          data for the split view (see winh.h for details).
 *
 *      3)  Call this function with that structure, which creates
 *          and returns the "split window", the invisible parent
 *          of the windows to be split.
 *          See the window hierarchy below.
 *
 *      4)  Position this split window within your windows,
 *          using WinSetWindowPos, which will automatically
 *          reposition the subwindows of the split window.
 *
 *      In detail, this function does the following:
 *
 *      1)  create an invisible "split window" (fnwpSplitWindow,
 *          which serves as an "abstract" parent only);
 *
 *      2)  make that window the parent of the two windows
 *          to be split
 *          (which are specified in the SPLITBARCDATA struct);
 *
 *      3)  create a split bar control (which is a static control
 *          subclassed with fnwpSplitBar) as a child of the
 *          "split window" also.
 *
 *      This function returns the HWND of the "split window".
 *      The handle of the split bar, if you absolutely need it,
 *      can be found using
 +          WinQueryWindow(hwndWhatThisReturns, ID_SPLITBAR).
 *
 *      This creates the following window hierarchy (parentship):
 *
 +      SPLITBARCDATA.hwndClient (whatever you have specified;
 +         |                      e.g. FID_CLIENT of your frame window.
 +         |                      You must intercept WM_SIZE and
 +         |                      call WinSetWindowPos on the "split window".)
 +         |
 +         +--  hwndReturn ("split window" created and returned by this
 +                 |       function; uses fnwpSplitWindow)
 +                 |
 +                 +-- SPLITBARCDATA.hwndLink1 (parent changed)
 +                 |      |
 +                 |      +-- ... (your child windows remain untouched)
 +                 |
 +                 +-- ID_SPLITBAR (uses fnwpSplitBar)
 +                 |
 +                 +-- SPLITBARCDATA.hwndLink2 (parent changed)
 +                        |
 +                        +-- ... (your child windows remain untouched)
 *
 *      Note that only parentship of hwndSplit1 and hwndSplit2
 *      is changed. Ownership remains untouched. So you can specify
 *      any window as the owner of hwndLink1/2 so that messages
 *      can be forwarded properly.
 *
 *      After the above hierarchy has been established, there are
 *      two situations where the "linked" windows will be repositioned
 *      and/or resized:
 *
 *      1)  fnwpSplitBar will automatically resize and reposition
 *          the left and right "linked" windows if the user moves the
 *          split bar.
 *
 *      2)  If the "split window" itself receives WM_SIZE, the "linked"
 *          windows will automatically be repositioned and resized. This
 *          is done in fnwpSplitWindow.
 *
 *      As a result, you must implement the following in your window
 *      procedures for the windows passed to this function:
 *
 *      1)  The window procedure of SPLITBARCDATA.hwndClient (the
 *          parent of the split window) must intercept WM_SIZE and
 *          do a WinSetWindowPos on the split window. This will
 *          readjust the split bar and SPLITBARCDATA.hwndLink1/2.
 *
 *      2)  SPLITBARCDATA.hwndLink1/2 must in turn handle WM_SIZE
 *          correctly because whenever the split window is resized,
 *          it invokes a WinSetWindowPos on SPLITBARCDATA.hwndLink1/2
 *          in turn.
 *          If you're using standard controls for your subwindows
 *          (e.g. containers), this is no problem. However, if you
 *          specify frame windows, you might want to reposition
 *          the controls in those windows in turn.
 *          Note that dialog windows do not receive WM_SIZE;
 *          you'll need to handle WM_WINDOWPOSCHANGED instead.
 *
 *      If you wish to "nest" split windows, you can do so by
 *      specifying the "split window" (the HWND which is returned
 *      by this function) as the "hwndLink1/2" to another call
 *      of this function. This way you can create a complex
 *      split window hierarchy (similar to what Netscape does
 *      with the FRAMESET tag). fnwpSplitWindow is capable of
 *      handling WM_SIZE messages correctly.
 *
 *@@added V1.00
 */

HWND winhCreateSplitWindow(HAB hab,
                           PSPLITBARCDATA psbcd)
{
    HWND hwndSplit = NULLHANDLE,
         hwndBar = NULLHANDLE;

    if (psbcd)
    {
        // register "split window" class
        WinRegisterClass(hab,
                         WC_SPLITWINDOW,
                         fnwpSplitWindow,
                         CS_SIZEREDRAW | CS_SYNCPAINT,
                         0);        // additional bytes to reserve

        hwndSplit = WinCreateWindow(psbcd->hwndClient,     // parent
                                    WC_SPLITWINDOW,
                                    "",
                                    WS_VISIBLE,
                                    0, 0, 10, 10,
                                    psbcd->hwndClient,     // owner
                                    HWND_TOP,
                                    psbcd->ulSplitWindowID,
                                    NULL,
                                    NULL);
        if (hwndSplit)
        {
            hwndBar    = WinCreateWindow(hwndSplit,    // parent
                                         WC_STATIC,
                                         "",
                                         WS_VISIBLE            // wnd style
                                           | SS_TEXT,
                                         0, 0, 10, 10,
                                         psbcd->hwndClient,    // owner
                                         HWND_TOP,
                                         ID_SPLITBAR,          // win ID
                                         NULL,                 // cdata
                                         NULL                  // presparams
                                     );
            if (hwndBar)
            {
                // subclass the static control to make it a split bar
                PFNWP OldStaticProc = WinSubclassWindow(hwndBar, fnwpSplitBar);
                if (OldStaticProc)
                {
                    PSPLITBARDATA pData = (PSPLITBARDATA)malloc(sizeof(SPLITBARDATA));
                    if (pData)
                    {
                        memcpy(&(pData->sbcd), psbcd, sizeof(SPLITBARCDATA));
                        WinQueryWindowRect(hwndBar, &(pData->rclBar));
                        (pData->rclBar.xRight)--;
                        (pData->rclBar.yTop)--;
                        pData->OldStaticProc = OldStaticProc;
                        pData->hptrOld = NULLHANDLE;
                        pData->hptrMove = WinQuerySysPointer(HWND_DESKTOP,
                                    (psbcd->ulCreateFlags & SBCF_HORIZONTAL)
                                        ? SPTR_SIZENS
                                        : SPTR_SIZEWE,
                                    FALSE);     // don't make copy
                        pData->fCaptured = FALSE;

                        WinSetWindowULong(hwndBar, QWL_USER, (ULONG)pData);
                    }

                    // change parents of the windows to link
                    WinSetParent(psbcd->hwndLink1, hwndSplit, FALSE);
                    WinSetParent(psbcd->hwndLink2, hwndSplit, FALSE);
                }
            }
        }
    }
    return (hwndSplit);
}

/*
 *@@ winhUpdateSplitWindow:
 *      this function takes a split window as input and
 *      repositions all its children (the split bars
 *      and the other subwindows which are separated by the
 *      split bars) according to the parent.
 *
 *      This function gets called from fnwpSplitWindow
 *      when WM_SIZE is received. Normally, you won't have
 *      to call this function independently; you should do
 *      a WinSetWindowPos on the split window instead.
 *
 *@@added V1.00
 */

BOOL winhUpdateSplitWindow(HWND hwndSplit)
{
    BOOL    brc = FALSE;
    ULONG   ul = 0;
    RECTL   rclSplit;
    HWND    hwndSplitBar = WinWindowFromID(hwndSplit, ID_SPLITBAR);

    if (hwndSplitBar)
    {
        PSPLITBARDATA psbd;

        WinQueryWindowRect(WinQueryWindow(hwndSplit, QW_PARENT), &rclSplit);

        WinSetWindowPos(hwndSplit, HWND_TOP,
                        rclSplit.xLeft,
                        rclSplit.yBottom,
                        rclSplit.xRight - rclSplit.xLeft,
                        rclSplit.yTop - rclSplit.yBottom,
                        SWP_MOVE | SWP_SIZE);

        // update split bar
        psbd = (PSPLITBARDATA)WinQueryWindowULong(hwndSplitBar, QWL_USER);
        if (psbd)
        {
            PSPLITBARCDATA psbcd = &(psbd->sbcd);
            RECTL rclBar;
            ULONG ul3DOfs = 0;

            if (psbcd->ulCreateFlags & SBCF_HORIZONTAL)
            {
                // _Pmpf(("Calc horizontal"));
                // vertical split bar:
                rclBar.xLeft = 0;
                rclBar.xRight = rclSplit.xRight;
                rclBar.yBottom = rclSplit.yBottom / 2;
                rclBar.yTop = rclBar.yBottom + WinQuerySysValue(HWND_DESKTOP,
                                                                SV_CYSIZEBORDER);
            }
            else
            {
                // _Pmpf(("Calc vertical"));
                // vertical split bar:
                if (psbcd->ulCreateFlags & SBCF_PERCENTAGE)
                    // take width of client and apply percentage
                    rclBar.xLeft = (rclSplit.xRight - rclSplit.xLeft)
                                    * psbcd->lPos
                                    / 100;
                else
                    if (psbcd->lPos > 0)
                        // offset from left:
                        rclBar.xLeft = psbcd->lPos;
                    else
                        // offset from right:
                        rclBar.xLeft = (rclSplit.xRight - rclSplit.xLeft)
                                       + psbcd->lPos;  // which is negative

                rclBar.xRight = rclBar.xLeft + WinQuerySysValue(HWND_DESKTOP,
                                                                SV_CXSIZEBORDER);
                rclBar.yBottom = 0;
                // take height of client
                rclBar.yTop = (rclSplit.yTop - rclSplit.yBottom);
            }

            // reposition split bar
            brc = WinSetWindowPos(hwndSplitBar,
                            HWND_TOP,
                            rclBar.xLeft,
                            rclBar.yBottom,
                            rclBar.xRight - rclBar.xLeft,
                            rclBar.yTop - rclBar.yBottom,
                            SWP_MOVE | SWP_SIZE);

            /* _Pmpf(("Set splitbar hwnd %lX to %d, %d, %d, %d; rc: %d",
                            hwndSplitBar,
                            rclBar.xLeft,
                            rclBar.yBottom,
                            rclBar.xRight - rclBar.xLeft,
                            rclBar.yTop - rclBar.yBottom,
                            brc)); */

            // reposition left/bottom window of split bar
            if (psbcd->ulCreateFlags & SBCF_3DSUNK)
                ul3DOfs = 1;
            // else 0

            WinSetWindowPos(psbcd->hwndLink1,
                            HWND_TOP,
                            ul3DOfs,
                            ul3DOfs,
                            rclBar.xLeft - ul3DOfs*2,       // the window rect is non-inclusive
                            rclBar.yTop - rclBar.yBottom - ul3DOfs*2,
                            SWP_MOVE | SWP_SIZE);

            // reposition right/top window of split bar
            WinSetWindowPos(psbcd->hwndLink2,
                            HWND_TOP,
                            rclBar.xRight + ul3DOfs,    // the window rect is non-inclusive
                            ul3DOfs,
                            rclSplit.xRight - rclBar.xRight - ul3DOfs*2,
                            rclBar.yTop - rclBar.yBottom - ul3DOfs*2,
                            SWP_MOVE | SWP_SIZE);

            // repaint split window (3D frame)
            WinInvalidateRect(hwndSplit,
                              NULL,         // all
                              FALSE);       // don't repaint children
        }
    }

    return (brc);
}

/*
 *@@ winhQuerySplitPos:
 *      this returns the position of the split bar
 *      (the child control of hwndSplit; see winhCreateSplitWindow
 *      for the window hierarchy).
 *
 *      The meaning of the return value depends on what you
 *      specified with ulCreateFlags in the SPLITBARCDATA
 *      structure passed to winhCreateSplitWindow, that is,
 *      it can be a percentage or an offset from the left
 *      or from the right of the split window.
 *
 *      This returns NULL upon errors.
 */

LONG winhQuerySplitPos(HWND hwndSplit)
{
    ULONG lrc = 0;

    // the split bar data is stored in QWL_USER of the
    // split bar (not the split window)
    HWND    hwndSplitBar = WinWindowFromID(hwndSplit, ID_SPLITBAR);
    if (hwndSplitBar)
    {
        PSPLITBARDATA psbd = (PSPLITBARDATA)WinQueryWindowULong(hwndSplitBar, QWL_USER);
        if (psbd)
            lrc = psbd->sbcd.lPos;
    }

    return (lrc);
}
