
/*
 * threads.h:
 *      header file for treads.c, which
 *      contains helper functions for creating, destroying, and
 *      synchronizing threads.
 *
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#if __cplusplus
extern "C" {
#endif

#ifndef THREADS_HEADER_INCLUDED
    #define THREADS_HEADER_INCLUDED

    /* thread info structure passed to secondary threads */
    typedef struct _THREADINFO {
            // data maintained by thr* functions
            ULONG   cb;
            ULONG   ulData;
            BOOL    fExit;
            // data to be maintained by secondary thread
            HAB     hab;
            TID     tid;
            BOOL    fExitComplete;
            ULONG   ulResult;
            ULONG   ulFuncInfo;
    } THREADINFO, *PTHREADINFO;

    typedef void _Optlink THREADFUNC (void*);
    typedef THREADFUNC *PTHREADFUNC;

    /*
     * thrCreate:
     *      this function creates a THREADINFO structure in *ppti;
     *      you must pass the thread function in pfn, which will
     *      then be executed. The thread will be passed a pointer
     *      to the new THREADINFO structure as its thread parameter.
     *      The ulData field in that structure is set to ulData
     *      here. Use whatever you like.
     *      Note: As opposed to previous versions, V0.84 now expects
     *      *pfn to have _Optlink calling convention, because now
     *      _beginthread() is used instead of DosCreateThread().
     */

    BOOL thrCreate(PTHREADINFO *ppti, PTHREADFUNC pfn, ULONG ulData);

    /*
     * thrClose:
     *      this functions sets a thread's "Close" flag to
     *      TRUE; the thread should monitor this flag
     *      periodically and then terminate itself.
     */

    BOOL thrClose(PTHREADINFO pti);


    /*
     * thrGoodbye:
     *      every thread should call this function just before
     *      it terminates itself so that the other thread funcs
     *      can react properly. This updates the THREADINFO
     *      structure, but does _not_ call _endthread(). So you
     *      should either call this function before calling
     *      _endthread() or as the last function call in the
     *      thread function itself before it exits.
     */

    VOID thrGoodbye(PTHREADINFO pti);

    /*
     * thrWait:
     *      this function waits for a thread to end
     */

    BOOL thrWait(PTHREADINFO pti);

    /*
     * thrFree:
     *      this is a combination of thrClose and
     *      thrWait; the THREADINFO block is then freed
     *      and *ppti set to NULL
     */

    BOOL thrFree(PTHREADINFO *ppti);

    /*
     * thrKill:
     *      just like thrFree, but the thread is
     *      brutally killed
     */

    BOOL thrKill(PTHREADINFO *ppti);

    /*
     * thrQueryID:
     *      returns thread ID
     */

    TID thrQueryID(PTHREADINFO pti);

    /*
     * thrQueryPriority:
     *      returns the priority of the calling thread.
     *      The low byte of the low word is a hexadecimal value
     *      representing a rank (value 0 to 31) within a priority class.
     *      Class values, found in the high byte of the low word, are
     *      as follows:
     *          0x01  idle
     *          0x02  regular
     *          0x03  time-critical
     *          0x04  server
     */

    ULONG thrQueryPriority(VOID);

#endif

#if __cplusplus
}
#endif

