
/*
 *@@sourcefile stringh.c:
 *      contains string/text helper functions that are independent
 *      of a single application, i.e. you can use them in any
 *      program. These are good for parsing/splitting strings and
 *      other stuff used throughout XWorkplace.
 *
 *      Function prefixes (new with V0.81):
 *      --  strh*       string helper functions.
 *
 *@@include #define INCL_DOSDATETIME
 *@@include #include <os2.h>
 *@@include #include "stringh.h"
 */

/*
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_WINSHELLDATA
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "stringh.h"

// uncomment to have debugging
// #define _PMPRINTF_
#include "pmprintf.h"

/*
 *@@ strhReplace:
 *      replace oldStr with newStr in str.
 *
 *      str should have enough allocated space for the replacement, no check
 *      is made for this. str and OldStr/NewStr should not overlap.
 *      The empty string ("") is found at the beginning of every string.
 *
 *      Returns: pointer to first location behind where NewStr was inserted
 *      or NULL if OldStr was not found.
 *      This is useful for multiple replacements also.
 *      (be careful not to replace the empty string this way !)
 *
 *      Author:     Gilles Kohl
 *      Started:    09.06.1992   12:16:47
 *      Modified:   09.06.1992   12:41:41
 *      Subject:    Replace one string by another in a given buffer.
 *                  This code is public domain. Use freely.
 */

char* strhReplace(char* str, char* oldStr, char* newStr)
{
      int OldLen, NewLen;
      char *p, *q;

      if (NULL == (p = strstr(str, oldStr)))
            return p;
      OldLen = strlen(oldStr);
      NewLen = strlen(newStr);
      memmove(q = p+NewLen, p+OldLen, strlen(p+OldLen)+1);
      memcpy(p, newStr, NewLen);
      return (q);
}

/*
 *@@ strhReplace2:
 *      another string replacement func. This also
 *      replaces pszSearch with pszReplace in *ppszBuf.
 *
 *      However, if pszSearch was found, *ppszBuf is
 *      re-allocated so the buffer cannot overflow.
 *
 *      Returns the length of the new string or 0
 *      if pszSearch was not found.
 */

ULONG strhReplace2(PSZ* ppszBuf,
                   PSZ pszSearch,
                   PSZ pszReplace)
{
    ULONG ulrc = 0;

    if ((ppszBuf) && (pszSearch) && (pszReplace))
    {
        ULONG   cbSearch = strlen(pszSearch),
                cbReplace = strlen(pszReplace),
                cbNew = strlen(*ppszBuf)
                        + cbReplace
                        - cbSearch
                        + 1;                  // null terminator

        PSZ     pFound = strstr(*ppszBuf, pszSearch);
        if (pFound)
        {
            // allocate new buffer
            PSZ     pszNew = (PSZ)malloc(cbNew);

            // find offset where pszSearch was found
            ULONG cbFirst = pFound - *ppszBuf;
            // copy until offset
            memcpy(pszNew,
                   *ppszBuf,
                   cbFirst);
            // copy replacement
            memcpy(pszNew + cbFirst,
                   pszReplace,
                   cbReplace);
            // copy rest
            strcpy(pszNew + cbFirst + cbReplace,
                   pFound + cbSearch);

            // replace PSZ pointer
            free(*ppszBuf);
            *ppszBuf = pszNew;
            // return new length
            ulrc = cbNew;
        }
    }
    return (ulrc);
}

/*
 *@@ strhInsert:
 *      this inserts one string into another.
 *
 *      pszInsert is inserted into pszBuffer at position
 *      pInsertPos (which must point to somewhere in pszBuffer,
 *      or this crashes).
 *
 *      A newly allocated string is returned. pszBuffer is
 *      not changed. The new string should be free()'d after
 *      use.
 *
 *      Upon errors, NULL is returned.
 *
 *@@changed V1.00: completely rewritten.
 */

PSZ strhInsert(PSZ pszBuffer,
               PSZ pszInsert,
               PSZ pInsertPos)
{
    PSZ     pszNew = NULL;

    if ((pszBuffer) && (pszInsert))
    {
        do {
            ULONG   cbBuffer = strlen(pszBuffer);
            ULONG   cbInsert = strlen(pszInsert);
            ULONG   ulInsertOfs = (pInsertPos - pszBuffer);

            if (    (pInsertPos < pszBuffer)
                 || (pInsertPos > (pszBuffer + cbBuffer))
               )
                break;  // do

            // OK, let's go.
            pszNew = (PSZ)malloc(cbBuffer + cbInsert + 1);  // additional null terminator

            // copy stuff before pInsertPos
            memcpy(pszNew,
                   pszBuffer,
                   ulInsertOfs);
            // copy string to be inserted
            memcpy(pszNew + ulInsertOfs,
                   pszInsert,
                   cbInsert);
            // copy stuff after pInsertPos
            strcpy(pszNew + ulInsertOfs + cbInsert,
                   pszBuffer + ulInsertOfs);
        } while (FALSE);
    }

    return (pszNew);
}

/*
 *@@ strhistr:
 *      like strstr, but case-insensitive.
 */

PSZ strhistr(PSZ string1, PSZ string2)
{
    PSZ prc = NULL;

    if ((string1) && (string2)) {
        PSZ pszSrchIn = strdup(string1);
        PSZ pszSrchFor = strdup(string2);
        strupr(pszSrchIn);
        strupr(pszSrchFor);

        if ((pszSrchIn) && (pszSrchFor)) {
            prc = strstr(pszSrchIn, pszSrchFor);
            if (prc) {
                // prc now has the first occurence of the string,
                // but in pszSrchIn; we need to map this
                // return value to the original string
                prc = (prc-pszSrchIn) // offset in pszSrchIn
                      + string1;
            }
        }
        if (pszSrchFor)
            free(pszSrchFor);
        if (pszSrchIn)
            free(pszSrchIn);
    }
    return (prc);
}

/*
 *@@ strhncpy0:
 *      like strncpy, but always appends a 0 character.
 */

ULONG strhncpy0(PSZ pszTarget, PSZ pszSource, ULONG cbSource)
{
    ULONG ul = 0;
    PSZ pTarget = pszTarget,
        pSource = pszSource;

    for (ul = 0; ul < cbSource; ul++)
        if (*pSource)
            *pTarget++ = *pSource++;
        else
            break;
    *pTarget = 0;

    return (ul);
}

/*
 * strhCount:
 *      this counts the occurences of c in pszSearch.
 */

ULONG strhCount(PSZ pszSearch, CHAR c)
{
    PSZ         p = pszSearch;
    ULONG       ulCount = 0;
    while (TRUE)
    {
        p = strchr(p, c);
        if (p)
        {
            ulCount++;
            p++;
        }
        else
            break;
    }
    return (ulCount);
}

/*
 *@@ strhIsDecimal:
 *      returns TRUE if psz consists of decimal digits only.
 */

BOOL strhIsDecimal(PSZ psz)
{
    PSZ p = psz;
    while (*p != 0)
    {
        if (isdigit(*p) == 0)
            return (FALSE);
        p++;
    }

    return (TRUE);
}

/*
 *@@ strhSubstr:
 *      this creates a new PSZ containing the string
 *      from pBegin to pEnd, excluding the pEnd character.
 *      The new string is null-terminated.
 *
 *      Example:
 +              "1234567890"
 +                ^      ^
 +                p1     p2
 +          strhSubstr(p1, p2)
 *      would return a new string containing "2345678".
 */

PSZ strhSubstr(PSZ pBegin, PSZ pEnd)
{
    ULONG cbSubstr = (pEnd - pBegin);
    PSZ pszSubstr = (PSZ)malloc(cbSubstr + 1);
    strhncpy0(pszSubstr, pBegin, cbSubstr);
    return (pszSubstr);
}

/*
 *@@ strhThousandsULong:
 *      converts a ULONG into a decimal string, while
 *      inserting thousands separators into it. Specify
 *      the separator char in cThousands.
 *      Returns pszTarget so you can use it directly
 *      with sprintf and the "%s" flag.
 *      For cThousands, you should use the data in
 *      OS2.INI ("PM_National" application), which is
 *      always set according to the "Country" object.
 *      Use strhThousandsDouble for "double" values.
 */

PSZ strhThousandsULong(PSZ pszTarget,       // out: decimal as string
                       ULONG ul,            // in: decimal to convert
                       CHAR cThousands)     // in: separator char (e.g. '.')
{
    USHORT ust, uss, usc;
    CHAR   szTemp[40];
    sprintf(szTemp, "%d", ul);

    ust = 0;
    usc = strlen(szTemp);
    for (uss = 0; uss < usc; uss++)
    {
        if (uss)
            if (((usc - uss) % 3) == 0)
            {
                pszTarget[ust] = cThousands;
                ust++;
            }
        pszTarget[ust] = szTemp[uss];
        ust++;
    }
    pszTarget[ust] = '\0';

    return (pszTarget);
}

/*
 *@@ strhThousandsDouble:
 *      like strhThousandsULong, but for a "double"
 *      value. Note that after-comma values are truncated.
 */

PSZ strhThousandsDouble(PSZ pszTarget, double dbl, CHAR cThousands)
{
    USHORT ust, uss, usc;
    CHAR   szTemp[40];
    sprintf(szTemp, "%.0f", floor(dbl));

    ust = 0;
    usc = strlen(szTemp);
    for (uss = 0; uss < usc; uss++)
    {
        if (uss)
            if (((usc - uss) % 3) == 0)
            {
                pszTarget[ust] = cThousands;
                ust++;
            }
        pszTarget[ust] = szTemp[uss];
        ust++;
    }
    pszTarget[ust] = '\0';

    return (pszTarget);
}

/*
 *@@ strhFileDate:
 *      converts file date data to a string (to pszBuf).
 *      You can pass any FDATE structure to this function,
 *      which are returned in those FILEFINDBUF* or
 *      FILESTATUS* structs by the Dos* functions.
 *      ulDateFormat can be queried using
 +              PrfQueryProfileInt(HINI_USER, "PM_National", "iDate", 0);
 *
 *      meaning:
 *      --  0   mm.dd.yyyy  (English)
 *      --  1   dd.mm.yyyy  (e.g. German)
 *      --  2   yyyy.mm.dd  (Japanese)
 *      --  3   yyyy.dd.mm
 *
 *      cDateSep is used as a date separator (e.g. '.').
 *      This can be queried using:
 +              PrfQueryProfileString(HINI_USER, "PM_National", "sDate", "/",
 +                  szDateSep, sizeof(szDateSep)-1);
 */

VOID strhFileDate(PSZ pszBuf,           // out: string returned
                  FDATE* pfDate,        // in: date information
                  ULONG ulDateFormat,   // in: date format (0-3)
                  CHAR cDateSep)        // in: date separator (e.g. '.')
{
    switch (ulDateFormat) {
        case 0:  // mm.dd.yyyy  (English)
            sprintf(pszBuf, "%02d%c%02d%c%04d",
                pfDate->month,
                    cDateSep,
                pfDate->day,
                    cDateSep,
                ((pfDate->year)+1980));
        break;

        case 1:  // dd.mm.yyyy  (e.g. German)
            sprintf(pszBuf, "%02d%c%02d%c%04d",
                pfDate->day,
                    cDateSep,
                pfDate->month,
                    cDateSep,
                ((pfDate->year)+1980));
        break;

        case 2: // yyyy.mm.dd  (Japanese)
            sprintf(pszBuf, "%04d%c%02d%c%02d",
                ((pfDate->year)+1980),
                    cDateSep,
                pfDate->month,
                    cDateSep,
                pfDate->day);
        break;

        default: // yyyy.dd.mm
            sprintf(pszBuf, "%04d%c%02d%c%02d",
                ((pfDate->year)+1980), cDateSep,
                pfDate->day, cDateSep,
                pfDate->month);
        break;
    }
}

/*
 *@@ strhFileTime:
 *      converts file time data to a string (to pszBuf).
 *      You can pass any FTIME structure to this function,
 *      which are returned in those FILEFINDBUF* or
 *      FILESTATUS* structs by the Dos* functions.
 *      ulTimeFormat can be queried using
 +              PrfQueryProfileInt(HINI_USER, "PM_National", "iTime", 0);
 *      meaning:
 *      --  0   12-hour clock
 *      --  >0  24-hour clock
 *
 *      cDateSep is used as a time separator (e.g. ':').
 *      This can be queried using:
 +              PrfQueryProfileString(HINI_USER, "PM_National", "sTime", ":",
 +                  szTimeSep, sizeof(szTimeSep)-1);
 *
 *@@changed 99-03-15 fixed 12-hour crash
 */

VOID strhFileTime(PSZ pszBuf,           // out: string returned
                  FTIME* pfTime,        // in: time information
                  ULONG ulTimeFormat,   // in: 24-hour time format (0 or 1)
                  CHAR cTimeSep)        // in: time separator (e.g. ':')
{
    if (ulTimeFormat == 0)
    {
        // for 12-hour clock, we need additional INI data
        CHAR szAMPM[10] = "err";

        if (pfTime->hours > 12)
        {
            // > 12h: PM.

            // Note: 12:xx noon is 12 AM, not PM (even though
            // AM stands for "ante meridiam", but English is just
            // not logical), so that's handled below.

            PrfQueryProfileString(HINI_USER,
                "PM_National",
                "s2359",        // key
                "PM",           // default
                szAMPM, sizeof(szAMPM)-1);
            sprintf(pszBuf, "%02d%c%02d%c%02d %s",
                // leave 12 == 12 (not 0)
                pfTime->hours % 12,
                    cTimeSep,
                pfTime->minutes,
                    cTimeSep,
                pfTime->twosecs*2,
                szAMPM);
        } else
        {
            // <= 12h: AM
            PrfQueryProfileString(HINI_USER,
                "PM_National",
                "s1159",        // key
                "AM",           // default
                szAMPM, sizeof(szAMPM)-1);
            sprintf(pszBuf, "%02d%c%02d%c%02d %s",
                pfTime->hours,
                    cTimeSep,
                pfTime->minutes,
                    cTimeSep,
                pfTime->twosecs*2,
                szAMPM);
        }
    }
    else
        // 24-hour clock
        sprintf(pszBuf, "%02d%c%02d%c%02d",
            pfTime->hours, cTimeSep,
            pfTime->minutes, cTimeSep,
            pfTime->twosecs*2);
}

/*
 *@@ strhFindEOL:
 *      returns a pointer to the next \r, \n or NULL character
 *      following pszSearchIn. Stores the offset in *pulOffset.
 */

PSZ strhFindEOL(PSZ pszSearchIn, PULONG pulOffset)
{
    PSZ     p = pszSearchIn,
            prc = NULL;
    while (TRUE)
    {
        if ( (*p == '\r') || (*p == '\n') || (*p == 0) )
        {
            prc = p;
            break;
        }
        p++;
    }

    if (pulOffset)
        *pulOffset = prc - pszSearchIn;
    return (prc);
}

/*
 *@@ strhFindNextLine:
 *      like strhFindEOL, but this returns the character
 *      _after_ \r or \n. Note that this might return
 *      a pointer to terminating NULL character also.
 */

PSZ strhFindNextLine(PSZ pszSearchIn, PULONG pulOffset)
{
    PSZ pEOL = strhFindEOL(pszSearchIn, NULL);
    // pEOL now points to the \r char or the terminating 0 byte;
    // if not null byte, advance pointer
    PSZ pNextLine = pEOL;
    if (*pNextLine == '\r')
        pNextLine++;
    if (*pNextLine == '\n')
        pNextLine++;
    if (pulOffset)
        *pulOffset = pNextLine - pszSearchIn;
    return (pNextLine);
}

/*
 *@@ strhFindKey:
 *      finds pszKey in pszSearchIn; similar to strhistr,
 *      but this one makes sure the key is at the beginning
 *      of a line. Spaces before the key are tolerated.
 *      Returns NULL if the key was not found.
 *      Used by strhGetParameter/strhSetParameter; useful
 *      for analyzing CONFIG.SYS settings.
 *
 *@@changed V1.00: fixed bug in that this would also return something if only the first chars matched
 */

PSZ strhFindKey(PSZ pszSearchIn,     // in: text buffer to search
                PSZ pszKey,          // in: key to search for
                PBOOL pfIsAllUpperCase) // out: TRUE if key is completely in upper case;
                                     // can be NULL if not needed
{
    PSZ     p = NULL;
    BOOL    fFound = FALSE;

    p = pszSearchIn;
    do {
        p = strhistr(p, pszKey);

        if ((p) && (p >= pszSearchIn))
        {
            // make sure the key is at the beginning of a line
            // by going backwards until we find a char != " "
            PSZ p2 = p;
            while ( (*p2 == ' ') && (p2 > pszSearchIn) )
                p2--;

            // if previous char is an EOL sign, go on
            if (    (*(p2-1) == '\r')
                 || (*(p2-1) == '\n')
                 || (p2 == pszSearchIn)
               )
            {
                // now check whether the char after the search
                // is a "=" char

                ULONG cbKey = strlen(pszKey);

                // tolerate spaces before "="
                PSZ p3 = p;
                while (*(p3+cbKey) == ' ')
                    p3++;

                if (*(p3+cbKey) == '=')
                {
                    // found:
                    fFound = TRUE; // go on, p contains found key

                    if (pfIsAllUpperCase)
                    {
                        // test for all upper case
                        ULONG cbKey = strlen(pszKey),
                              ul = 0;
                        *pfIsAllUpperCase = TRUE;
                        for (ul = 0; ul < cbKey; ul++)
                            if (islower(*(p+ul)))
                            {
                                *pfIsAllUpperCase = FALSE;
                                break; // for
                            }
                    }

                    break; // do
                }
            } // else search next key

            p++; // search on after this key
        }
    } while ((!fFound) && (p != NULL) && (p != pszSearchIn));

    return (p);
}


/*
 *@@ strhGetParameter:
 *      searches pszSearchIn for the key pszKey; if found, it
 *      returns a pointer to the following characters in pszSearchIn
 *      and, if pszCopyTo != NULL, copies the rest of the line to
 *      that buffer, of which cbCopyTo specified the size.
 *      If the key is not found, NULL is returned.
 *      String search is done by calling strhFindKey.
 *      This is useful for querying CONFIG.SYS settings.
 *
 *      <B>Example:</B> this would return "YES" if you searched
 *      for "PAUSEONERROR=", and "PAUSEONERROR=YES" existed in pszSearchIn.
 */

PSZ strhGetParameter(PSZ pszSearchIn,   // in: text buffer to search
                     PSZ pszKey,        // in: key to search for
                     PSZ pszCopyTo,     // out: key value
                     ULONG cbCopyTo)    // out: sizeof(*pszCopyTo)
{
    PSZ p = strhFindKey(pszSearchIn, pszKey, NULL),
        prc = NULL;
    // copy to pszCopyTo
    if (p) {
        prc = p + strlen(pszKey);
        if (pszCopyTo) {
            ULONG cb;
            PSZ pEOL = strhFindEOL(prc, &cb);
            if (pEOL) {
                _Pmpf(("%s: cb: %d, p2-p1: %d", pszKey, cb, pEOL-prc));
                if (cb > cbCopyTo)
                    cb = cbCopyTo-1;
                strncpy(pszCopyTo, prc, cb);
                pszCopyTo[cbCopyTo-1] = '\0';
            }
        }
    }

    return (prc);
}

/*
 *@@ strhSetParameter:
 *      searches pszSearchIn for the key pszKey; if found, it
 *      replaces the characters following this key up to the
 *      end of the line by pszParam. If pszKey is not found in
 *      pszSearchIn, it is appended to the file in a new line. You
 *      must ensure that *pszSearchin is large enough for the
 *      manipulation, i.e. it must be at least of the following
 *      size:
 +          strlen(pszSearchIn) + strlen(pszNewParam) + 1
 *
 *      in case the key was not found and will be appended.
 *      This function searches w/out case sensitivity.
 *      Returns a pointer to the new parameter.
 */

PSZ strhSetParameter(PSZ pszSearchIn,    // in: text buffer to search
                    PSZ pszKey,         // in: key to search for
                    PSZ pszNewParam,    // in: new parameter to set for key
                    BOOL fRespectCase)  // in: if TRUE, pszNewParam will
                            // be converted to upper case if the found key is
                            // in upper case also. pszNewParam should be in
                            // lower case if you use this.
{
    BOOL fIsAllUpperCase = FALSE;
    PSZ pKey = strhFindKey(pszSearchIn, pszKey, &fIsAllUpperCase),
        prc = NULL;

    if (pKey)
    {
        // key found in file:
        // replace existing parameter
        PSZ pOldParam = pKey + strlen(pszKey);
        prc = pOldParam;
        // pOldParam now has the old parameter, which we
        // will overwrite now

        if (pOldParam)
        {
            ULONG cbOldParam;
            PSZ pEOL = strhFindEOL(pOldParam, &cbOldParam);
            // pEOL now has first end-of-line after the parameter

            if (pEOL)
            {
                PSZ pszOldCopy = (PSZ)malloc(cbOldParam+1);
                strncpy(pszOldCopy, pOldParam, cbOldParam);
                pszOldCopy[cbOldParam] = '\0';

                // check for upper case desired?
                if (fRespectCase)
                    if (fIsAllUpperCase)
                        strupr(pszNewParam);

                strhReplace(pOldParam, pszOldCopy, pszNewParam);

                free(pszOldCopy);
            }
        }
    } else
    {
        // key not found: append to end of file
        strcat(pszSearchIn, "\r\n");
        strcat(pszSearchIn, pszKey);
        strcat(pszSearchIn, strupr(pszNewParam));
        strcat(pszSearchIn, "\r\n");
    }

    return (prc);
}

/*
 *@@ strhDeleteLine:
 *      this deletes the line in pszSearchIn which starts with
 *      the key pszKey. Returns TRUE if the line was found and
 *      deleted.
 */

BOOL strhDeleteLine(PSZ pszSearchIn, PSZ pszKey)
{
    BOOL fIsAllUpperCase = FALSE;
    PSZ pKey = strhFindKey(pszSearchIn, pszKey, &fIsAllUpperCase);
    BOOL brc = FALSE;

    if (pKey) {
        PSZ pEOL = strhFindEOL(pKey, NULL);
        // pEOL now has first end-of-line after the key
        if (pEOL) {
            // delete line by overwriting it with
            // the next line
            strcpy(pKey, pEOL+2);
        } else {
            // EOL not found: we must be at the end of the file
            *pKey = '\0';
        }
        brc = TRUE;
    }

    return (brc);
}

/*
 *@@ strhBeautifyTitle:
 *      replaces all line breaks (0xd, 0xa) with spaces.
 */

BOOL strhBeautifyTitle(PSZ psz)
{
    BOOL rc = FALSE;
    CHAR *p;
    while (p = strchr(psz, 0xa))
    {
        *p = ' ';
        rc = TRUE;
    }
    while (p = strchr(psz, 0xd))
    {
        *p = ' ';
        rc = TRUE;
    }
    return (rc);
}

/*
 * strhFindAttribValue:
 *      searches for pszAttrib in pszSearchIn; if found,
 *      returns the first character after the "=" char.
 *      If "=" is not found, a space, \r, and \n are
 *      also accepted.
 *
 *      <B>Example:</B>
 +          strhFindAttribValue("<PAGE BLAH="data">, "BLAH")
 +
 +          returns ....................... ^ this address.
 *
 *@@added V1.00
 */

PSZ strhFindAttribValue(PSZ pszSearchIn, PSZ pszAttrib)
{
    PSZ     prc = 0;
    PSZ     pszSearchIn2 = pszSearchIn,
            p,
            p2;
    ULONG   cbAttrib = strlen(pszAttrib);

    // 1) find space char
    while (p = strchr(pszSearchIn2, ' '))
    {
        p++;
        // now check whether the p+strlen(pszAttrib)
        // is a valid end-of-tag character
        if (    (memicmp(p, pszAttrib, cbAttrib) == 0)
             && (   (*(p+cbAttrib) == ' ')
                 || (*(p+cbAttrib) == '>')
                 || (*(p+cbAttrib) == '=')
                 || (*(p+cbAttrib) == '\r')
                 || (*(p+cbAttrib) == '\n')
                 || (*(p+cbAttrib) == 0)
                )
           )
        {
            p2 = p + cbAttrib;
            while (     (   (*p2 == ' ')
                         || (*p2 == '=')
                         || (*p2 == '\n')
                         || (*p2 == '\r')
                        )
                    &&  (*p2 != 0)
                  )
                p2++;
            prc = p2;
            break; // first while
        }
        pszSearchIn2++;
    }
    return (prc);
}

/*
 * strhGetNumAttribValue:
 *      stores the numerical parameter value of an HTML-style
 *      tag in *pl.
 *
 *      Returns the address of the tag parameter in the
 *      search buffer, if found, or NULL.
 *
 *      <B>Example:</B>
 +          strhGetNumAttribValue("<PAGE BLAH=123>, "BLAH", &l)
 *
 *      stores 123 in the "l" variable.
 *
 *@@added V1.00
 */

PSZ strhGetNumAttribValue(PSZ pszSearchIn,       // in: where to search
                         PSZ pszTag,            // e.g. "INDEX"
                         PLONG pl)              // out: numerical value
{
    PSZ pParam;
    if (pParam = strhFindAttribValue(pszSearchIn, pszTag)) {
        sscanf(pParam, "%d", pl);
    }
    return (pParam);
}

/*
 * strhGetTextAttr:
 *      retrieves the attribute value of a textual HTML-style tag
 *      in a newly allocated buffer, which is returned,
 *      or NULL if attribute not found.
 *      If an attribute value is to contain spaces, it
 *      must be enclosed in quotes.
 *
 *      The offset of the attribute data in pszSearchIn is
 *      returned in *pulOffset so that you can do multiple
 *      searches.
 *
 *      This returns a new buffer, which should be free()'d after use.
 *
 *      <B>Example:</B>
 +          ULONG   ulOfs = 0;
 +          strhGetTextAttr("<PAGE BLAH="blublub">, "BLAH", &ulOfs)
 +                           ............^ ulOfs
 *
 *      returns a new string with the value "blublub" (without
 *      quotes) and sets ulOfs to 12.
 *
 *@@added V1.00
 */

PSZ strhGetTextAttr(PSZ pszSearchIn, PSZ pszTag, PULONG pulOffset)
{
    PSZ     pParam,
            pParam2,
            prc = NULL;
    ULONG   ulCount = 0;
    LONG    lNestingLevel = 0;

    if (pParam = strhFindAttribValue(pszSearchIn, pszTag))
    {
        // determine end character to search for: a space
        CHAR cEnd = ' ';
        if (*pParam == '\"')
        {
            // or, if the data is enclosed in quotes, a quote
            cEnd = '\"';
            pParam++;
        }

        if (pulOffset)
            // store the offset
            *pulOffset = pParam - pszSearchIn;

        // now find end of attribute
        pParam2 = pParam;
        while (*pParam)
        {
            if (*pParam == cEnd)
                // end character found
                break;
            else if (*pParam == '<')
                // yet another opening tag found:
                // this is probably some "<" in the attributes
                lNestingLevel++;
            else if (*pParam == '>')
            {
                lNestingLevel--;
                if (lNestingLevel < 0)
                    // end of tag found:
                    break;
            }
            ulCount++;
            pParam++;
        }

        // copy attribute to new buffer
        if (ulCount)
        {
            prc = (PSZ)malloc(ulCount+1);
            memcpy(prc, pParam2, ulCount);
            *(prc+ulCount) = 0;
        }
    }
    return (prc);
}

/*
 * strhFindEndOfTag:
 *      returns a pointer to the ">" char
 *      which seems to terminate the tag beginning
 *      after pszBeginOfTag.
 *
 *      If additional "<" chars are found, we look
 *      for additional ">" characters too.
 *
 *      Note: You must pass the address of the opening
 *      '<' character to this function.
 *
 *      Example:
 +          PSZ pszTest = "<BODY ATTR=\"<BODY>\">";
 +          strhFindEndOfTag(pszTest)
 +      returns.................................^ this.
 *
 *@@added V1.00
 */

PSZ strhFindEndOfTag(PSZ pszBeginOfTag)
{
    PSZ     p = pszBeginOfTag,
            prc = NULL;
    LONG    lNestingLevel = 0;

    while (*p)
    {
        if (*p == '<')
            // another opening tag found:
            lNestingLevel++;
        else if (*p == '>')
        {
            // closing tag found:
            lNestingLevel--;
            if (lNestingLevel < 1)
            {
                // corresponding: return this
                prc = p;
                break;
            }
        }
        p++;
    }

    return (prc);
}

/*
 * strhGetBlock:
 *      this complex function searches the given string
 *      for a pair of opening/closing HTML-style tags.
 *
 *      If found, this routine returns TRUE and does
 *      the following:
 *
 *          1)  allocate a new buffer, copy the text
 *              enclosed by the opening/closing tags
 *              into it and set *ppszBlock to that
 *              buffer;
 *
 *          2)  if the opening tag has any attributes,
 *              allocate another buffer, copy the
 *              attributes into it and set *ppszAttrs
 *              to that buffer; if no attributes are
 *              found, *ppszAttrs will be NULL;
 *
 *          3)  set *pulOffset to the offset from the
 *              beginning of *ppszSearchIn where the
 *              opening tag was found;
 *
 *          4)  advance *ppszSearchIn to after the
 *              closing tag, so that you can do
 *              multiple searches without finding the
 *              same tags twice.
 *
 *      All buffers should be freed using free().
 *
 *      This returns the following:
 *      --  0: no error
 *      --  1: tag not found at all (doesn't have to be an error)
 *      --  2: begin tag found, but no corresponding end tag found. This
 *             is a real error.
 *      --  3: begin tag is not terminated by ">" (e.g. "<BEGINTAG whatever")
 *
 *      <B>Example:</B>
 +          PSZ pSearch = "<PAGE INDEX=1>This is page 1.</PAGE>More text."
 +          PSZ pszBlock, pszAttrs;
 +          ULONG ul;
 +          strhGetBlock(&pSearch, "PAGE", &pszBlock, &pszAttrs, &ulOfs)
 *
 *      would do the following:
 *
 *      1)  set pszBlock to a new string containing "This is page 1."
 *          without quotes;
 *
 *      2)  set pszAttrs to a new string containing "<PAGE INDEX=1>";
 *
 *      3)  set ul to 0, because "<PAGE" was found at the beginning;
 *
 *      4)  pSearch would be advanced to point to the "More text"
 *          string in the original buffer.
 *
 *      Hey-hey. A one-shot function, fairly complicated, but indispensable
 *      for HTML parsing.
 *
 *@@added V1.00
 */

ULONG strhGetBlock(PSZ *ppszSearchIn,   // in/out: where to search. If pszTag was found,
                                        // this will be advanced to after the closing tag.
                   PSZ pszTag,          // in: tags to search for without <> (e.g. "TEXT")
                   PSZ *ppszBlock,      // out: block enclosed by the tags
                   PSZ *ppszAttribs,    // out: attributes of the opening tag
                   PULONG pulOfsBeginTag, // in/out: offset from *ppszSearchIn where opening tag was found
                   PULONG pulOfsBeginBlock) // in/out: offset from *ppszSearchIn where beginning of block was found
{
    ULONG   ulrc = 1;
    PSZ     pszBeginTag = *ppszSearchIn,
            pszSearch2 = pszBeginTag,
            pszClosingTag;
    ULONG   cbTag = strlen(pszTag);

    // go thru the block and check all tags if it's the
    // begin tag we're looking for
    while (pszBeginTag = strchr(pszBeginTag, '<'))
    {
        if (memicmp(pszBeginTag+1, pszTag, strlen(pszTag)) == 0)
            // yes: stop
            break;
        else
            pszBeginTag++;
    }

    if (pszBeginTag)
    {
        ULONG   ulNestingLevel = 0;

        PSZ     pszEndOfBeginTag = strhFindEndOfTag(pszBeginTag);
                                    // strchr(pszBeginTag, '>');
        if (pszEndOfBeginTag)
        {
            // does the caller want the attributes?
            if (ppszAttribs)
            {
                // yes: then copy them
                ULONG   ulAttrLen = pszEndOfBeginTag - pszBeginTag + 1;
                PSZ     pszAttrs = (PSZ)malloc(ulAttrLen);
                strncpy(pszAttrs, pszBeginTag, ulAttrLen);
                // add terminating 0
                *(pszAttrs + ulAttrLen) = 0;

                *ppszAttribs = pszAttrs;
            }

            // increase offset of where we found the begin tag
            if (pulOfsBeginTag)
                *pulOfsBeginTag += pszBeginTag - *ppszSearchIn;

            // now find corresponding closing tag (e.g. "</BODY>"
            pszBeginTag = pszEndOfBeginTag+1;
            // now we're behind the '>' char of the opening tag
            // increase offset of that too
            if (pulOfsBeginBlock)
                *pulOfsBeginBlock += pszBeginTag - *ppszSearchIn;

            // find next closing tag;
            // for the first run, pszSearch2 points to right
            // after the '>' char of the opening tag
            pszSearch2 = pszBeginTag;
            while (pszClosingTag = strstr(pszSearch2, "<"))
            {
                // if we have another opening tag before our closing
                // tag, we need to have several closing tags before
                // we're done
                if (memicmp(pszClosingTag+1, pszTag, cbTag) == 0)
                    ulNestingLevel++;
                else
                {
                    // is this ours?
                    if (    (*(pszClosingTag+1) == '/')
                         && (memicmp(pszClosingTag+2, pszTag, cbTag) == 0)
                       )
                    {
                        // we've found a matching closing tag; is
                        // it ours?
                        if (ulNestingLevel == 0)
                        {
                            // our closing tag found:
                            // allocate mem for a new buffer
                            // and extract all the text between
                            // open and closing tags to it
                            ULONG ulLen = pszClosingTag-pszBeginTag;
                            if (ppszBlock)
                            {
                                PSZ pNew = (PSZ)malloc(ulLen+1);
                                strncpy(pNew, pszBeginTag, ulLen);
                                *(pNew+ulLen) = 0;
                                *ppszBlock = pNew;
                            }

                            // increase the input pointer
                            *ppszSearchIn = pszClosingTag + cbTag + 1;

                            ulrc = 0;

                            break;
                        } else
                            // not our closing tag:
                            ulNestingLevel--;
                    }
                }
                // no matching closing tag: search on after that
                pszSearch2 = strhFindEndOfTag(pszClosingTag);
            } // end while (pszClosingTag = strstr(pszSearch2, "<"))

            if (!pszClosingTag)
                // no matching closing tag found:
                // return 2 (closing tag not found)
                ulrc = 2;
        } // end if (pszBeginTag)
        else
            // no matching ">" for opening tag found:
            ulrc = 3;
    }

    return (ulrc);
}


