
/*
 *@@sourcefile prfh.c:
 *      contains those Presentation Manager helper functions
 *      which deal with Profile (Prf*) functions. These can be
 *      used w/out the rest of the XWorkplace source in any PM
 *      program.
 *      This file is new with V0.82.
 *
 *      Function prefixes:
 *      --  prfh*   Prf (profile, INI) helper functions
 *
 *@@include #define INCL_WINWINDOWMGR
 *@@include #define INCL_WINSHELLDATA
 *@@include #include <os2.h>
 *@@include #include "prfh.h"
 */

/*
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "dosh.h"

#include "prfh.h"

/*
 *@@ prfhQueryKeysForApp:
 *      returns the keys list for an INI application. This
 *      list is copied into a newly allocated buffer, of which
 *      the address is returned.
 *
 *      Returns NULL upon errors.
 *
 *      If the return value is != NULL, the PSZ points to a new
 *      buffer which contains all the keys within the pszApp
 *      application. Each key name in the list is terminated with
 *      a null character. The last string in the list is terminated
 *      with two null characters.
 *
 *      The returned buffer should be freed later using free().
 *
 *      <B>Example</B> for iterating over a keys list:
 +          PSZ pszKeysList = prfhQueryKeysForApp(...);
 +          if (pszKeysList)
 +          {
 +              PSZ pKey2 = pszKeysList;
 +
 +              while (*pKey2 != 0)
 +              {
 +                  ...     // pKey2 has the current key now
 +                  pKey2 += strlen(pKey2)+1; // next key
 +              }
 +              free(pszKeysList);
 +          }
 *
 *      You can also use this function to query the applications
 *      list for hIni, if you specifiy pszApp as NULL.
 */

PSZ prfhQueryKeysForApp(HINI hIni,      // in: INI handle (can be HINI_USER or HINI_SYSTEM)
                        PSZ pszApp)     // in: application to query list for (or NULL for applications list)
{
    PSZ     pKeys = NULL;
    ULONG   ulSizeOfKeysList = 0;

    // get size of keys list for pszApp
    if (PrfQueryProfileSize(hIni, pszApp, NULL, &ulSizeOfKeysList))
    {
        pKeys = (PSZ)malloc(ulSizeOfKeysList);
        if (!PrfQueryProfileData(hIni, pszApp, NULL, pKeys, &ulSizeOfKeysList))
        {
            free(pKeys);
            pKeys = NULL;
        }
    }
    return (pKeys);
}

/*
 *@@ prfhQueryProfileData:
 *      similar to PrfQueryProfileData, but this one copies
 *      the data into a newly allocated buffer, of which the
 *      address is returned.
 *
 *      Returns NULL upon errors.
 *
 *      If pcbBuf != NULL, this func will write the size of
 *      the allocated buffer into *pcbBuf.
 *
 *      The returned buffer should be freed later using free().
 */

PSZ prfhQueryProfileData(HINI hIni,      // in: INI handle (can be HINI_USER or HINI_SYSTEM)
                        PSZ pszApp,      // in: application to query
                        PSZ pszKey,      // in: key to query
                        PULONG pcbBuf)   // out: size of the returned buffer
{
    PSZ     pData = NULL;
    ULONG   ulSizeOfData = 0;

    // get size of data for pszApp/pszKey
    if (PrfQueryProfileSize(hIni, pszApp, pszKey, &ulSizeOfData))
    {
        pData = (PSZ)malloc(ulSizeOfData);
        if (!PrfQueryProfileData(hIni, pszApp, pszKey, pData, &ulSizeOfData))
        {
            free(pData);
            pData = NULL;
        }
    }

    if (pcbBuf)
        *pcbBuf = ulSizeOfData;

    return (pData);
}

/*
 *@@ prfhCopyKey:
 *      this copies one key from the given profile and application
 *      to another one.
 *
 *      pszTargetApp may be in the same profile (and must be
 *      different from pszSourceApp then) or in a different
 *      profile (and can be the same then).
 *
 *      You must specify all parameters. You cannot specify pszKey
 *      as NULL to have a whole application copied. Use prfCopyApp
 *      for that.
 *      No check is made for this.
 *
 *      Returns:
 *      --  0: no error
 *      --  PRFERR_DATASIZE: couldn't query data size for key
 *      --  PRFERR_MEMORY: couldn't allocate memory
 *      --  PRFERR_READ: couldn't read data from source (PrfQueryProfileData error)
 *      --  PRFERR_WRITE: couldn't write data to target (PrfWriteProfileData error)
 */

ULONG prfhCopyKey(HINI hiniSource,       // in: source profile (can be HINI_USER or HINI_SYSTEM)
                 PSZ pszSourceApp,      // in: source application
                 PSZ pszKey,            // in: source/target key
                 HINI hiniTarget,       // in: target profile (can be HINI_USER or HINI_SYSTEM)
                 PSZ pszTargetApp)      // in: target app
{
    ULONG   ulSizeOfData = 0,
            ulrc = 0;       // return: no error

    if (PrfQueryProfileSize(hiniSource, pszSourceApp, pszKey, &ulSizeOfData))
    {
        PSZ pData = 0;

        // copy data
        if (ulSizeOfData == 0)
        {
            // data size == 0: this shouldn't really happen,
            // but if it does, we'll just create a NULL string.
            // Users have reported that some INI files seem to
            // contain those "empty" keys. I don't see how these
            // can exist, but they seem to...
            pData = (PSZ)malloc(1);
            *pData = 0;
        }
        else
            pData = (PSZ)malloc(ulSizeOfData);

        if (pData)
        {
            if (PrfQueryProfileData(hiniSource, pszSourceApp, pszKey,
                            pData, &ulSizeOfData))
            {
                if (!PrfWriteProfileData(hiniTarget,
                                         pszTargetApp,
                                         pszKey,
                                         pData,
                                         ulSizeOfData))
                    ulrc = PRFERR_WRITE;
            }
            else
                ulrc = PRFERR_READ;

            free(pData);
        }
        else
            ulrc = PRFERR_MEMORY;
    }
    else
        ulrc = PRFERR_DATASIZE;

    return (ulrc);
}

/*
 *@@ prfhCopyApp:
 *      this copies one key from the given profile and application
 *      to another one.
 *
 *      You can use this function in several contexts:
 *      -- copy one application within the same profile
 *         (i.e. hiniSource == hiniTarget);
 *         in this case, pszSourceApp must be != pszTargetApp;
 *      -- copy an application from one profile to another
 *         (i.e. hiniSource != hiniTarget);
 *         in this case, pszSourceApp can be == pszTargetApp
 *         (but can be different also).
 *
 *      WARNING: This does _not_ check for whether the target
 *      application exists already. This has two consequences:
 *      --  existing data will be overwritten without warning;
 *      --  if the existing target application has keys that are
 *          not in the source application, they are not deleted.
 *          As a result, you might end up with more keys than
 *          in the source application.
 *
 *      So you should delete the target application before
 *      calling this function, like this:
 +          PrfWriteProfileString(hiniTarget, pszTargetApp, NULL, NULL);
 *
 *      You must specify all parameters. You cannot specify pszApp
 *      as NULL to have a whole profile copied. Use prfCopyProfile
 *      for that.
 *      No check is made for this.
 *
 *      Returns:
 *      --  0: no error
 *      --  PRFERR_KEYSLIST: couldn't query keys for pszSourceApp
 *      --  PRFERR_DATASIZE: couldn't query data size for key
 *      --  PRFERR_MEMORY: couldn't allocate memory
 *      --  PRFERR_READ: couldn't read data from source (PrfQueryProfileData error)
 *      --  PRFERR_WRITE: couldn't write data to target (PrfWriteProfileData error)
 */

ULONG prfhCopyApp(HINI hiniSource,   // in: source profile (can be HINI_USER or HINI_SYSTEM)
                  PSZ pszSourceApp,  // in: source application
                  HINI hiniTarget,   // in: target profile (can be HINI_USER or HINI_SYSTEM)
                  PSZ pszTargetApp,  // in: name of pszSourceApp in hiniTarget
                  PSZ pszErrorKey)   // out: failing key in case of error
{
    ULONG   ulrc;
    PSZ pszKeysList;

    if (pszErrorKey)
        *pszErrorKey = 0;

    pszKeysList = prfhQueryKeysForApp(hiniSource, pszSourceApp);
    if (pszKeysList)
    {
        PSZ pKey2 = pszKeysList;

        while (*pKey2 != 0)
        {
            // copy this key
            ulrc = prfhCopyKey(hiniSource,
                               pszSourceApp,
                               pKey2,
                               hiniTarget,
                               pszTargetApp);
            if (ulrc)
            {
                // error: copy failing key to buffer
                if (pszErrorKey)
                    strcpy(pszErrorKey, pKey2);
                break;
            }
            pKey2 += strlen(pKey2)+1;
        } // end while (*pKey2 != 0)

        free (pszKeysList);
    }
    else
        ulrc = PRFERR_KEYSLIST;

    return (ulrc);
}

/*
 * prfhINIError:
 *      this is called whenever an error occurs saving a
 *      profile. This will compose an error string and
 *      all the callback func fncbError.
 */

ULONG prfhINIError(ULONG ulOptions,
                   HFILE hfLog,
                   PFNWP fncbError,
                   PSZ pszErrorString)
{
    ULONG ulrc;
    CHAR szError2[2000];

    if (fncbError) {
        if (ulOptions == MB_ABORTRETRYIGNORE)
            sprintf(szError2, "%s"
                              "\nPress 'Abort' to abort saving the INI files. This will restart the WPS to avoid loss of data."
                              "\nPress 'Retry' to attempt saving this INI file again. "
                              "\nPress 'Ignore' to ignore this error. This might help, but "
                              "also lead to more errors or loss of the currently processed data.",
                              pszErrorString);
        else
            sprintf(szError2, "%s\nPress 'Cancel' to abort shutdown. ", pszErrorString);
        ulrc = ( (ULONG)(*fncbError)(0, 0, szError2, (MPARAM)ulOptions) );

    } else
        ulrc = (MBID_ABORT);

    if (hfLog)
    {
        CHAR szLog[2000];
        sprintf(szLog, "    Error occured: %s\n    Return code: %s (%d)\n",
                pszErrorString,
                    (ulrc == MBID_ABORT) ? "MBID_ABORT"
                    : (ulrc == MBID_IGNORE) ? "MBID_IGNORE"
                    : (ulrc == MBID_RETRY) ? "MBID_RETRY"
                    : "unknown",
                ulrc);
        doshWriteToLogFile(hfLog, szLog);
    }
    return (ulrc);
}

/*
 * prfhINIError2:
 *      like the previous func, but with additional info
 *      for the error string.
 */

ULONG prfhINIError2(ULONG ulOptions,
                    PSZ pszINI,
                    HFILE hfLog,
                    PFNWP fncbError,
                    PSZ pszErrorString)
{
    CHAR szError2[2000];
    sprintf(szError2, "An error occured copying the profile %s: \n%s",
            pszINI, pszErrorString);
    return (prfhINIError(ulOptions, hfLog, fncbError, szError2));
}

/*
 *@@ prfhCopyProfile:
 *      this function copies a given profile entirely into a
 *      new file. hOld is the handle, pszOld the filename
 *      of the profile to be copied (e.g. HINI_USERPROFILE
 *      and "?:\OS2\OS2.INI"; pszNew must point to a
 *      buffer which will contain the new filename ending
 *      in ".XFL" after prfhCopyProfile returns
 *      (e.g. "?:\OS2\OS2.XFL").
 *
 *      You may specify a callback procedure which gets
 *      called upon every copied application in the INI
 *      file; this way, you can provide a progress bar.
 *      The second callback, fncbError, gets called upon
 *      errors.
 */

BOOL prfhCopyProfile(HAB hab,               // in:  anchor block
                     HFILE hfLog,           // in:  logfile handle
                                            // or NULLHANDLE for no log
                     HINI hOld,             // in:  HINI to copy
                     PSZ pszOld,            // in:  fully qualif. filename of hOld
                     PSZ pszNew,            // out: new filename
                     PFNWP fncbUpdate,      // in:  progress callback
                     HWND hwnd, ULONG msg, ULONG ulCount, ULONG ulMax,
                                            // in:  passed to fncbUpdate
                     PFNWP fncbError,       // in:  error callback
                     PULONG pulFunc2)       // in:  passed to fncbError
{
    // BOOL    rc = TRUE;
    HINI    hNew;
    PSZ     pApps, pKeys,
            pApp2, pKey2;
    ULONG   ulSizeOfAppsList, ulSizeOfKeysList, ulSizeOfData;
    CHAR    szLog[1000];
    #define MBID_NOERROR 999
    ULONG   ulErrorStatus = MBID_NOERROR;
    PSZ     p_;
    CHAR    szError2[2000];

    *pulFunc2 = 20000;
    if (hfLog) {
        sprintf(szLog, "    Entering prfhCopyProfile, params: "
                    "hiniOld 0x%lX, pszOld %s, pfnwpCallback 0x%lX, "
                    "hwnd 0x%lX, msg 0x%lX, ulCount 0x%lX, ulMax 0x%lX\n",
                    hOld, pszOld, fncbUpdate, hwnd, msg, ulCount, ulMax);
        doshWriteToLogFile(hfLog, szLog);
    }

    *pulFunc2 = 20010;
    if ( (!pszOld) || (!pszNew) )
        ulErrorStatus = prfhINIError2(MB_CANCEL, pszOld, hfLog, fncbError,
                "Invalid name buffers given.");

    *pulFunc2 = 20020;
    if (ulErrorStatus == MBID_NOERROR) {
        strupr(pszOld);
        strcpy(pszNew, pszOld);
        p_ = strstr(pszNew, ".INI");
        if (!p_)
            ulErrorStatus = prfhINIError2(MB_CANCEL, pszNew, hfLog, fncbError,
                    "Error composing new filename.");
    }

    *pulFunc2 = 20030;
    if (ulErrorStatus == MBID_NOERROR)
    {
        strcpy(p_, ".XFL");

        // delete new profile, in case something's left
        if (hfLog) {
            sprintf(szLog, "    DEL %s\n", pszNew);
            doshWriteToLogFile(hfLog, szLog);
        }
        DosDelete(pszNew);

        // open new profile
        if (hfLog) {
            sprintf(szLog, "    PrfOpenProfile %s\n", pszNew);
            doshWriteToLogFile(hfLog, szLog);
        }

        *pulFunc2 = 20040;
        hNew = PrfOpenProfile(hab, pszNew);
        if (!hNew)
            ulErrorStatus = prfhINIError2(MB_CANCEL, pszNew, hfLog, fncbError,
                    "Error creating temporary profile.");
    }

    // get size of applications list
    *pulFunc2 = 20100;
    if (ulErrorStatus == MBID_NOERROR)
    {
        if (!PrfQueryProfileSize(hOld, NULL, NULL, &ulSizeOfAppsList))
            ulErrorStatus = prfhINIError2(MB_CANCEL, pszNew, hfLog, fncbError,
                    "Error querying applications list size.");
        else
        if (ulSizeOfAppsList == 0)
            ulErrorStatus = prfhINIError2(MB_CANCEL, pszNew, hfLog, fncbError,
                "Size of applications list cannot be 0.");
    }

    if (ulErrorStatus == MBID_NOERROR)
    {
        // do-while loop in case of MBID_RETRY
        do {
            ulErrorStatus = MBID_NOERROR; // if we have a retry in the do-while loop

            // get applications list
            *pulFunc2 = 20110;
            pApps = (PSZ)malloc(ulSizeOfAppsList);
            pApp2 = pApps;
            if (!PrfQueryProfileData(hOld, NULL, NULL, pApps, &ulSizeOfAppsList))
                ulErrorStatus = prfhINIError2(MB_CANCEL, pszNew, hfLog, fncbError,
                            "Could not query application list.");

            // applications loop

            *pulFunc2 = 20200;
            while (   (*pApp2 != 0)
                   && (ulErrorStatus == MBID_NOERROR)
                  )
            {
                CHAR szErrorKey[1000];
                // copy application (this will call prfhCopyKey in turn)
                ULONG ulrc = prfhCopyApp(hOld,
                                         pApp2,
                                         hNew,
                                         pApp2,
                                         szErrorKey);

                switch (ulrc)
                {
                    case 0: // no error
                        ulErrorStatus = MBID_NOERROR;
                        break;

                    case PRFERR_KEYSLIST:
                        // couldn't query keys list
                        sprintf(szError2,
                                "Error querying keys list for \"%s\".",
                                pApp2);
                        break;

                    case PRFERR_DATASIZE:
                        // couldn't query data size for key
                        sprintf(szError2,
                                "Error querying data size for \"%s\"/\"%s\".",
                                pApp2, szErrorKey);
                        break;

                    case PRFERR_MEMORY:
                        // couldn't allocate memory
                        sprintf(szError2,
                                "Error allocating memory for copying \"%s\"/\"%s\".",
                                pApp2, szErrorKey);
                        break;

                    case PRFERR_READ:
                        // couldn't read data from source (PrfQueryProfileData error)
                        sprintf(szError2,
                                "Error reading data from app \"%s\", key \"%s\".",
                                pApp2, szErrorKey);
                        break;

                    case PRFERR_WRITE:
                        // couldn't write data to target (PrfWriteProfileData error)
                        sprintf(szError2,
                                "Error writing data to app \"%s\", key \"%s\".",
                                pApp2, szErrorKey);
                        break;
                }

                *pulFunc2 = 20520;
                if (fncbUpdate) {
                    ULONG ulNow2, ulMax2;
                    ulNow2 = ((1000*(pApp2-pApps))/ulSizeOfAppsList) + (ulCount*1000);
                    ulMax2 = (ulMax+1)*1000; // ulAppsSize;
                    (*fncbUpdate)(hwnd, msg, (MPARAM)ulNow2, (MPARAM)ulMax2);
                }

                *pulFunc2 = 20530;
                // go for next app
                pApp2 += strlen(pApp2)+1;
                *pulFunc2 = 20531;

                if (ulErrorStatus == MBID_IGNORE)
                    // skip this app
                    ulErrorStatus = MBID_NOERROR;

            } // end while (*pApp2 != 0)

            *pulFunc2 = 20610;
            if (pApps)
                free(pApps);

            if (hfLog) {
                sprintf(szLog, "    Done copying apps\n");
                doshWriteToLogFile(hfLog, szLog);
            }
        } while (ulErrorStatus == MBID_RETRY);
    } // end if (ulErrorOccured == MBID_NOERROR)

    *pulFunc2 = 20620;
    if (hfLog) {
        sprintf(szLog, "    PrfCloseProfile %s\n", pszNew);
        doshWriteToLogFile(hfLog, szLog);
    }

    *pulFunc2 = 20630;
    PrfCloseProfile(hNew);
    *pulFunc2 = 20640;
    if (fncbUpdate)
        (*fncbUpdate)(hwnd, msg, (MPARAM)(ulCount+1), (MPARAM)(ulMax+1));

    *pulFunc2 = 20650;
    if (hfLog)
        doshWriteToLogFile(hfLog, "    Exiting prfhCopyProfile.\n");

    *pulFunc2 = 20660;

    return (ulErrorStatus == MBID_NOERROR); // FALSE if error occured
}

/*
 *@@ prfhSaveINIs:
 *      this will enforce saving of OS2.INI and OS2SYS.INI
 *      by calling prfhCopyProfile (above) on them; the therefrom
 *      resulting ".XFL" files will replace the original
 *      ".INI" files.
 *      Specify fncbUpdate and fncbError like in prfhCopyProfile.
 */

APIRET prfhSaveINIs(HAB hab,                // in:  anchor block
                     HFILE hfLog,           // in:  logfile handle or NULLHANDLE for no log
                     PFNWP fncbUpdate,      // in:  progress callback
                     HWND hwnd, ULONG msg,  // in:  params passed to fncbUpdate
                     PFNWP fncbError,       // in:  error callback
                     PULONG pulFunc2)       // in:  passed to fncbError
{
    PRFPROFILE Profiles;
    APIRET  arc = NO_ERROR, arc2;
    BOOL    brc = TRUE;

    // FILESTATUS3 fs3;
    CHAR    szSysNew[CCHMAXPATH],
            szUserNew[CCHMAXPATH],
            szSysBackup[CCHMAXPATH],
            szUserBackup[CCHMAXPATH],
            szLog[1000];

    // PSZ     p;

    // the following flag may be one of the following:
    //        MBID_NOERROR--- everything's fine, continue
    //        MBID_IGNORE --- error occured, but ignore
    //        MBID_RETRY  --- error occured, but retry
    //        MBID_ABORT  --- error occured, abort saving
    ULONG   ulErrorOccured = MBID_IGNORE;

    // DosEnterCritSec();

    *pulFunc2 = 10000;

    if (hfLog)
        doshWriteToLogFile(hfLog, "  Entering prfhSaveINIs...\n");

    Profiles.cchUserName = Profiles.cchSysName = 0;
    brc = PrfQueryProfile(hab, &Profiles);

    *pulFunc2 = 10010;
    if (!brc) {
        ulErrorOccured = prfhINIError(MB_CANCEL, hfLog, fncbError,
                "Error querying system profiles size.");
    }

    *pulFunc2 = 10020;
    if (ulErrorOccured == MBID_IGNORE)
    {
        Profiles.pszUserName  = (PSZ)malloc(Profiles.cchUserName);
        Profiles.pszSysName  = (PSZ)malloc(Profiles.cchSysName);
        if (    (Profiles.pszSysName == NULL)
             || (Profiles.pszUserName == NULL)
           )
            ulErrorOccured = prfhINIError(MB_CANCEL, hfLog, fncbError,
                    "Error allocating memory (1).");
    }

    *pulFunc2 = 10030;
    if (ulErrorOccured == MBID_IGNORE)
    {
        if (!PrfQueryProfile(hab, &Profiles))
            ulErrorOccured = prfhINIError(MB_CANCEL, hfLog, fncbError,
                    "Error querying profiles (2).");
    }

    *pulFunc2 = 10040;
    if (ulErrorOccured == MBID_IGNORE)
    {
        if (hfLog) {
            sprintf(szLog, "  System profiles are %s, %s\n",
                    Profiles.pszUserName, Profiles.pszSysName);
            doshWriteToLogFile(hfLog, szLog);
        }

        // create filenames
        // xxx this crashes with lower-case .INI
        strcpy(szSysBackup, Profiles.pszSysName);
        strcpy(strstr(szSysBackup, ".INI"), ".BAK");
        strcpy(szUserBackup, Profiles.pszUserName);
        strcpy(strstr(szUserBackup, ".INI"), ".BAK");

        *pulFunc2 = 10050;

        // create OS2SYS.XFL

        if (hfLog) {
            sprintf(szLog, "  Storing %s in *.XFL\n",
                    Profiles.pszSysName);
            doshWriteToLogFile(hfLog, szLog);
        }

        if (!prfhCopyProfile(hab, hfLog, HINI_SYSTEMPROFILE,
                Profiles.pszSysName, szSysNew,
                fncbUpdate, hwnd, msg, 0, 1,
                fncbError, pulFunc2))
        {
            // abort, since prfhCopyProfile already has error handling
            ulErrorOccured = MBID_ABORT;
        }
    }

    *pulFunc2 = 10060;
    if (ulErrorOccured == MBID_IGNORE)
    {
        // create OS2.XFL

        if (hfLog) {
            sprintf(szLog, "  Storing %s in *.XFL\n",
                    Profiles.pszUserName);
            doshWriteToLogFile(hfLog, szLog);
        }

        if (!prfhCopyProfile(hab, hfLog,
                HINI_USERPROFILE, Profiles.pszUserName, szUserNew,
                fncbUpdate, hwnd, msg, 1, 1,
                fncbError, pulFunc2))
        {
            // abort, since prfhCopyProfile already has error handling
            ulErrorOccured = MBID_ABORT;
        }
    }

    /*
     * renaming stuff for OS2SYS.INI
     *
     */

    if (ulErrorOccured == MBID_IGNORE)
    {
        do {
            ulErrorOccured = MBID_IGNORE; // if we have a retry in the do-while loop

            *pulFunc2 = 10070;
            if (ulErrorOccured == MBID_IGNORE) {
                // attrib -r -s -h -a OS2SYS.BAK
                if (hfLog) {
                    sprintf(szLog, "  ATTRIB -R -S -H -A %s\n",
                            szSysBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = doshSetPathAttr(szSysBackup, FILE_NORMAL);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // delete OS2SYS.BAK
                if (hfLog) {
                    sprintf(szLog, "  DEL %s\n",
                            szSysBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = DosDelete(szSysBackup);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // attrib -r -s -h -a OS2SYS.INI
                if (hfLog) {
                    sprintf(szLog, "  ATTRIB -R -S -H -A %s\n",
                            Profiles.pszSysName);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = doshSetPathAttr(Profiles.pszSysName, FILE_NORMAL);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // move OS2SYS.INI OS2SYS.BAK
                if (hfLog) {
                    sprintf(szLog, "  MOVE %s %s\n",
                            Profiles.pszSysName, szSysBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc = DosMove(Profiles.pszSysName, szSysBackup);
                if (hfLog) {
                    sprintf(szLog, "    rc: %d\n", arc);
                    doshWriteToLogFile(hfLog, szLog);
                }

                if (arc)
                    ulErrorOccured = prfhINIError(MB_ABORTRETRYIGNORE, hfLog, fncbError, "Error moving original system profile to backup.");
            }

            *pulFunc2 = 10200;
            if (ulErrorOccured == MBID_IGNORE) {
                if (hfLog) {
                    sprintf(szLog, "  MOVE %s %s\n",
                            szSysNew, Profiles.pszSysName);
                    doshWriteToLogFile(hfLog, szLog);
                }
                *pulFunc2 = 10210;
                arc = DosMove(szSysNew, Profiles.pszSysName);
                if (hfLog) {
                    sprintf(szLog, "    rc: %d\n", arc);
                    doshWriteToLogFile(hfLog, szLog);
                }
                if (arc)
                    ulErrorOccured = prfhINIError(MB_ABORTRETRYIGNORE, hfLog, fncbError, "Error moving newly created profile to system profile.");
            }
        } while (ulErrorOccured == MBID_RETRY);
    }

    /*
     * renaming stuff for OS2.INI
     *
     */

    if (ulErrorOccured == MBID_IGNORE) {
        do {
            ulErrorOccured = MBID_IGNORE; // if we have a retry in the do-while loop

            *pulFunc2 = 10300;
            if (ulErrorOccured == MBID_IGNORE) {
                // attrib -r -s -h -a OS2.BAK
                if (hfLog) {
                    sprintf(szLog, "  ATTRIB -R -S -H -A %s\n",
                            szUserBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = doshSetPathAttr(szUserBackup, FILE_NORMAL);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // delete OS2.BAK
                *pulFunc2 = 10310;
                if (hfLog) {
                    sprintf(szLog, "  DEL %s\n",
                            szUserBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = DosDelete(szUserBackup);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // attrib -r -s -h -a OS2.INI
                if (hfLog) {
                    sprintf(szLog, "  ATTRIB -R -S -H -A %s\n",
                            Profiles.pszUserName);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc2 = doshSetPathAttr(Profiles.pszUserName, FILE_NORMAL);
                if (hfLog) {
                    sprintf(szLog, "    rc2: %d\n", arc2);
                    doshWriteToLogFile(hfLog, szLog);
                }

                // move OS2.INI OS2.BAK
                if (hfLog) {
                    sprintf(szLog, "  MOVE %s %s\n",
                            Profiles.pszUserName, szUserBackup);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc = DosMove(Profiles.pszUserName, szUserBackup);
                if (hfLog) {
                    sprintf(szLog, "    rc: %d\n", arc);
                    doshWriteToLogFile(hfLog, szLog);
                }

                if (arc)
                    ulErrorOccured = prfhINIError(MB_ABORTRETRYIGNORE, hfLog, fncbError, "Error moving original user profile to backup.");
            }

            *pulFunc2 = 10350;
            if (ulErrorOccured == MBID_IGNORE) {
                // move OS2.XFL OS2.INI
                if (hfLog) {
                    sprintf(szLog, "  MOVE %s %s\n",
                            szUserNew, Profiles.pszUserName);
                    doshWriteToLogFile(hfLog, szLog);
                }
                arc = DosMove(szUserNew, Profiles.pszUserName);
                if (hfLog) {
                    sprintf(szLog, "    rc: %d\n", arc);
                    doshWriteToLogFile(hfLog, szLog);
                }

                if (arc)
                    ulErrorOccured = prfhINIError(MB_ABORTRETRYIGNORE, hfLog, fncbError, "Error moving newly created profile to user profile.");
            }
        } while (ulErrorOccured == MBID_RETRY);
    }

    // DosExitCritSec();

    *pulFunc2 = 10400;
    if (ulErrorOccured != MBID_IGNORE) {
        DosMove(szSysBackup, Profiles.pszSysName);
        DosMove(szUserBackup, Profiles.pszUserName);
    }

    *pulFunc2 = 10410;
    if (    (Profiles.pszSysName)
         && (Profiles.pszUserName)
       )
    {
        *pulFunc2 = 10420;
        free(Profiles.pszSysName);
        *pulFunc2 = 10430;
        free(Profiles.pszUserName);
    }

    *pulFunc2 = 10430;
    if (hfLog) {
        doshWriteToLogFile(hfLog, "  Done with prfhSaveINIs\n");
    }

    if (ulErrorOccured != MBID_IGNORE)
        return (999);
    else
        return (NO_ERROR);
}

/*
 *@@ prfhQueryColor:
 *      returns a system color in OS2.INI's PM_Colors as a LONG.
 */

LONG prfhQueryColor(PSZ pszKeyName, PSZ pszDefault)
{
    CHAR szColor[30];
    LONG r, g, b;
    PrfQueryProfileString(
                HINI_USER,
                "PM_Colors",
                pszKeyName,
                pszDefault,
                szColor,
                sizeof(szColor)-1);
    sscanf(szColor, "%d %d %d ", &r, &g, &b);
    return (r*0x10000 + g*0x100 + b);
}


