
/*
 *@@sourcefile gpih.c:
 *      contains GPI helper functions that are
 *      independent of a single application, i.e. these can be
 *      used w/out the rest of the XWorkplace source in any PM
 *      program.
 *
 *      Function prefixes (new with V0.81):
 *      --  gpih*   GPI helper functions
 *
 *@@include #include <os2.h>
 *@@include #include "gpih.h"
 */

/*
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSDEVIOCTL
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_GPI
#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "winh.h"
#include "gpih.h"

// array for querying device capabilities (gpihQueryDisplayCaps)
LONG            DisplayCaps[CAPS_DEVICE_POLYSET_POINTS] = {0};
BOOL            fCapsQueried = FALSE;

/*
 *@@ gpihQueryDisplayCaps:
 *      this returns certain device capabilities of
 *      the Display device. ulIndex must be one of
 *      the indices as described in DevQueryCaps.
 *
 *      This function will load all the device capabilities
 *      only once into a global array and re-use them afterwards.
 */

ULONG gpihQueryDisplayCaps(ULONG ulIndex)
{
    if (!fCapsQueried)
    {
        HPS hps = WinGetScreenPS(HWND_DESKTOP);
        HDC hdc = GpiQueryDevice(hps);
        DevQueryCaps(hdc, 0, CAPS_DEVICE_POLYSET_POINTS, &DisplayCaps[0]);
    }

    return (DisplayCaps[ulIndex]);
}

/*
 *@@ gpihQueryLineSpacing:
 *      this returns the optimal line spacing for text
 *      output with the current HPS; this is computed
 *      by evaluating those incredible FONTMETRICS.
 *
 *      This might be helpful if you write text to the
 *      screen yourself and need the height of a text
 *      line to advance to the next.
 */

LONG gpihQueryLineSpacing(HPS hps,
                          PSZ pszText)      // in:  text to output
{
    FONTMETRICS fm;
    POINTL aptlText[TXTBOX_COUNT];
    GpiQueryTextBox(hps, strlen(pszText), pszText,
             TXTBOX_COUNT, (PPOINTL)&aptlText);

    if (GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), &fm))
        return ( (  fm.lMaxBaselineExt     // max vertical font space
                   +fm.lExternalLeading)    // space advised by font designer
                 // * 12 / 10
               );
    else
        return (15);
}

/*
 *@@ gpihDrawRect:
 *      this draws a simple rectangle with the current
 *      color (use GpiSetColor before calling this function).
 *
 *      The specified rectangle is non-inclusive, that is, the top
 *      right corner specifies the top right pixel to be drawn.
 *
 *@added V1.00
 */

VOID gpihDrawRect(HPS hps, PRECTL prcl)
{
    POINTL ptl1;

    ptl1.x = prcl->xLeft;
    ptl1.y = prcl->yBottom;
    GpiMove(hps, &ptl1);
    ptl1.y = prcl->yTop-1;
    GpiLine(hps, &ptl1);
    ptl1.x = prcl->xRight-1;
    GpiLine(hps, &ptl1);
    ptl1.y = prcl->yBottom;
    GpiLine(hps, &ptl1);
    ptl1.x = prcl->xLeft;
    GpiLine(hps, &ptl1);
}

/*
 *@@ gpihDraw3DFrame:
 *      this draws a rectangle in 3D style with a given line width.
 *
 *      If (fRaised), the 3D line is given a "raised" look,
 *      "sunken" otherwise.
 *
 *      The line will be drawn using the SYSCLR_BUTTONLIGHT and
 *      SYSCLR_BUTTONDARK system colors.
 *
 *      The specified rectangle is non-inclusive, that is, the top
 *      right corner specifies the top right pixel to be drawn.
 *
 *      If usWidth > 1, the additional pixels will be drawn towards
 *      the _center_ of the rectangle. prcl thus always specifies
 *      the bottom left and top right pixels to be drawn.
 */

VOID gpihDraw3DFrame(HPS hps,
                     PRECTL prcl,       // in:  rectangle
                     USHORT usWidth,    // in:  line width (>= 1)
                     BOOL fRaised)      // in:  TRUE for "raised", FALSE "sunken"
{
    RECTL rcl2 = *prcl;
    LONG lColorLeft, lColorRight;
    USHORT us;
    POINTL ptl1;

    if (fRaised) {
        lColorLeft = WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0);
        lColorRight = WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0);
    } else {
        lColorLeft = WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONDARK, 0);
        lColorRight = WinQuerySysColor(HWND_DESKTOP, SYSCLR_BUTTONLIGHT, 0);
    }

    for (us = 0; us < usWidth; us++)
    {
        GpiSetColor(hps, lColorLeft);
        ptl1.x = rcl2.xLeft;
        ptl1.y = rcl2.yBottom;
        GpiMove(hps, &ptl1);
        ptl1.y = rcl2.yTop-1;
        GpiLine(hps, &ptl1);
        ptl1.x = rcl2.xRight-1;
        GpiLine(hps, &ptl1);
        GpiSetColor(hps, lColorRight);
        ptl1.y = rcl2.yBottom;
        GpiLine(hps, &ptl1);
        ptl1.x = rcl2.xLeft;
        GpiLine(hps, &ptl1);

        rcl2.xLeft++;
        rcl2.yBottom++;
        rcl2.xRight--;
        rcl2.yTop--;
    }
}

/*
 *@@ gpihCreateMemPS:
 *      creates a memory device context and presentation space so
 *      that they are compatible with the screen device context and
 *      presentation space. These are stored in *hdcMem and *hpsMem.
 *
 *      This is a one-shot function for the standard code that is
 *      always needed when working with bitmaps in a memory device
 *      context.
 *
 *      Returns FALSE upon errors. In that case, both hdcMem and
 *      hpsMem are set to NULLHANDLE.
 *
 *      To cleanup after this function has returned TRUE, use the
 *      following:
 +          GpiDestroyPS(hpsMem);
 +          DevCloseDC(hdcMem);
 */

BOOL gpihCreateMemPS(HAB hab,       // in: anchor block
                     HDC *hdcMem,   // out: memory DC
                     HPS *hpsMem)   // out: memory PS
{
    BOOL brc = FALSE;
    PSZ pszData[4] = { "Display", NULL, NULL, NULL };
    SIZEL sizlPage = {0, 0};

    // create new memory DC
    if (*hdcMem = DevOpenDC(hab,
                            OD_MEMORY,      // create memory DC
                            "*",            // token: do not take INI info
                            4,              // item count in pszData
                            (PDEVOPENDATA)pszData,
                            NULLHANDLE))    // compatible with screen
        // memory DC created successfully:
        // create compatible PS
        if (*hpsMem = GpiCreatePS(hab,
                                  *hdcMem,      // HDC to associate HPS with (GPIA_ASSOC);
                                                // mandatory for GPIT_MICRO
                                  &sizlPage,    // is (0, 0) == screen size
                                  PU_PELS       // presentation page units: pixels
                                    | GPIA_ASSOC    // associate with hdcMem (req. for GPIT_MICRO)
                                    | GPIT_MICRO))  // micro presentation space
            brc = TRUE;
        else
        {
            // error (hpsMem == NULLHANDLE):
            // close memory DC again
            DevCloseDC(*hdcMem);
            *hdcMem = NULLHANDLE;
        }

    return (brc);
}

/*
 *@@ gpihCreateBitmap:
 *      creates a new bitmap for a given memory PS.
 *      This bitmap will have the cPlanes and bitcount
 *      which are found in the memory PS.
 *      For all the mysterious other values, we use
 *      fixed default values, this doesn't seem to hurt.
 *
 *      Note that the bitmap is _not_ yet selected into
 *      the specified memory PS.
 *
 *      Returns the bitmap handle or NULLHANDLE upon errors.
 *
 *@@changed V1.00: function prototype changed to cx and cy
 */

HBITMAP gpihCreateBitmap(HPS hpsMem,        // in: memory DC
                         ULONG  cx,         // in: width of new bitmap
                         ULONG  cy)         // in: height of new bitmap
{
    HBITMAP hbm = NULLHANDLE;
    LONG alData[2];
    BITMAPINFOHEADER2 bih2;
    PBITMAPINFO2 pbmi = NULL;

    // determine the device's plane/bit-count format;
    // alData[0] then has cPlanes,
    // alData[1] has cBitCount
    if (GpiQueryDeviceBitmapFormats(hpsMem, 2, alData))
    {
        // set up the BITMAPINFOHEADER2 and BITMAPINFO2 structures
        bih2.cbFix = (ULONG)sizeof(BITMAPINFOHEADER2);
        bih2.cx = cx; // (prcl->xRight - prcl->xLeft);       changed V1.00
        bih2.cy = cy; // (prcl->yTop - prcl->yBottom);       changed V1.00
        bih2.cPlanes = alData[0];
        bih2.cBitCount = alData[1];
        bih2.ulCompression = BCA_UNCOMP;
        bih2.cbImage = (    (   (bih2.cx
                                    * (1 << bih2.cPlanes)
                                    * (1 << bih2.cBitCount)
                                ) + 31
                            ) / 32
                       ) * bih2.cy;
        bih2.cxResolution = 70;
        bih2.cyResolution = 70;
        bih2.cclrUsed = 2;
        bih2.cclrImportant = 0;
        bih2.usUnits = BRU_METRIC;  // measure units for cxResolution/cyResolution: pels per meter
        bih2.usReserved = 0;
        bih2.usRecording = BRA_BOTTOMUP;        // scan lines are bottom to top (default)
        bih2.usRendering = BRH_NOTHALFTONED;    // other algorithms aren't documented anyways
        bih2.cSize1 = 0;            // parameter for halftoning (undocumented anyways)
        bih2.cSize2 = 0;            // parameter for halftoning (undocumented anyways)
        bih2.ulColorEncoding = BCE_RGB;     // only possible value
        bih2.ulIdentifier = 0;              // application-specific data

        // allocate memory for info header
        if (DosAllocMem((PPVOID)&pbmi,
                        sizeof(BITMAPINFO2) +
                            (sizeof(RGB2)
                                * (1 << bih2.cPlanes)
                                * (1 << bih2.cBitCount)
                            ),
                        PAG_COMMIT | PAG_READ | PAG_WRITE)
                 == NO_ERROR)
        {
            pbmi->cbFix = bih2.cbFix;
            pbmi->cx = bih2.cx;
            pbmi->cy = bih2.cy;
            pbmi->cPlanes = bih2.cPlanes;
            pbmi->cBitCount = bih2.cBitCount;
            pbmi->ulCompression = BCA_UNCOMP;
            pbmi->cbImage = ((bih2.cx+31)/32) * bih2.cy;
            pbmi->cxResolution = 70;
            pbmi->cyResolution = 70;
            pbmi->cclrUsed = 2;
            pbmi->cclrImportant = 0;
            pbmi->usUnits = BRU_METRIC;
            pbmi->usReserved = 0;
            pbmi->usRecording = BRA_BOTTOMUP;
            pbmi->usRendering = BRH_NOTHALFTONED;
            pbmi->cSize1 = 0;
            pbmi->cSize2 = 0;
            pbmi->ulColorEncoding = BCE_RGB;
            pbmi->ulIdentifier = 0;

            // create a bit map that is compatible with the display
            hbm = GpiCreateBitmap(hpsMem,
                                  &bih2,
                                  FALSE,
                                  NULL,
                                  pbmi);

            // free the memory we allocated previously; GPI has
            // allocated all the resources it needs itself, so
            // we can release this
            DosFreeMem(pbmi);
        }
    }

    return (hbm);
}

/*
 *@@ gpihCreateBmpFromPS:
 *      this creates a new bitmap and copies a screen rectangle
 *      into it. Consider this a "screen capture" function.
 *
 *      The new bitmap (which is returned) is compatible with the
 *      device associated with hpsScreen. This function calls
 *      gpihCreateMemPS and gpihCreateBitmap to have it created.
 *      The memory PS is only temporary and freed again.
 *
 *      This returns the handle of the new bitmap,
 *      which can then be used for WinDrawBitmap and such, or
 *      NULLHANDLE upon errors.
 */

HBITMAP gpihCreateBmpFromPS(HAB hab,        // in: anchor block
                            HPS hpsScreen,  // in: screen PS to copy from
                            PRECTL prcl)    // in: rectangle to copy
{

    /* To copy an image from a display screen to a bit map:
      1. Associate the memory device context with a presentation space.
      2. Create a bit map.
      3. Select the bit map into the memory device context by calling GpiSetBitmap.
      4. Determine the location (in device coordinates) of the image.
      5. Call GpiBitBlt and copy the image to the bit map. */

    HDC hdcMem;
    HPS hpsMem;
    HBITMAP hbm = NULLHANDLE;
    POINTL aptl[3];

    if (gpihCreateMemPS(hab, &hdcMem, &hpsMem))
    {
        if (hbm = gpihCreateBitmap(hpsMem,
                                   prcl->xRight - prcl->xLeft,
                                   prcl->yTop - prcl->yBottom))
        {
            // Associate the bit map and the memory presentation space.
            if (GpiSetBitmap(hpsMem, hbm)
                    != HBM_ERROR)
            {
                // Copy the screen to the bit map.
                aptl[0].x = 0;              // lower-left corner of destination rectangle
                aptl[0].y = 0;
                aptl[1].x = prcl->xRight;   // upper-right corner for both
                aptl[1].y = prcl->yTop;
                aptl[2].x = prcl->xLeft;    // lower-left corner of source rectangle
                aptl[2].y = prcl->yBottom;

                if (GpiBitBlt(hpsMem, hpsScreen,
                              sizeof(aptl) / sizeof(POINTL), // Number of points in aptl
                              aptl,
                              ROP_SRCCOPY,
                              BBO_IGNORE)
                        == GPI_ERROR)
                {
                    // error during bitblt:
                    GpiDeleteBitmap(hbm);
                    hbm = NULLHANDLE; // for return code
                }
            } else {
                // error selecting bitmap for hpsMem:
                GpiDeleteBitmap(hbm);
                hbm = NULLHANDLE; // for return code
            }
        }

        GpiDestroyPS(hpsMem);
        DevCloseDC(hdcMem);
    } // end if (hdcMem = DevOpenDC())

    return (hbm);
}

/*
 *@@ gpihCreateHalftonedBitmap:
 *      this creates a half-toned copy of the
 *      input bitmap by doing the following:
 *
 *      1)  create a new bitmap with the size of hbmSource;
 *      2)  copy hbmSource to the new bitmap (using GpiWCBitBlt);
 *      3)  overpaint every other pixel with lColorGray.
 *
 *      Note that the memory DC is switched to RGB mode, so
 *      lColorGray better be an RGB color.
 *
 *      Note: hbmSource must _not_ be selected into any device
 *      context, or otherwise GpiWCBitBlt will fail.
 *
 *      This returns the new bitmap or NULLHANDLE upon errors.
 *
 *@added V1.00
 */

HBITMAP gpihCreateHalftonedBitmap(HAB hab,              // in: anchor block
                                  HBITMAP hbmSource,    // in: source bitmap
                                  LONG lColorGray)      // in: color used for gray
{
    HBITMAP hbmReturn = NULLHANDLE;

    HDC     hdcMem;
    HPS     hpsMem;
    BITMAPINFOHEADER2 bmi;
    RECTL   rclSource;

    if (hbmSource)
    {
        // query bitmap info
        bmi.cbFix = sizeof(bmi);
        GpiQueryBitmapInfoHeader(hbmSource, &bmi);

        if (gpihCreateMemPS(hab, &hdcMem, &hpsMem))
        {
            if (hbmReturn = gpihCreateBitmap(hpsMem,
                                             bmi.cx,
                                             bmi.cy))
            {
                if (GpiSetBitmap(hpsMem, hbmReturn) != HBM_ERROR)
                {
                    LONG lHits;
                    POINTL  aptl[4];

                    // step 1: copy bitmap
                    memset(aptl, 0, sizeof(POINTL) * 4);
                    // aptl[0]: target bottom-left, is all 0
                    // aptl[1]: target top-right (inclusive!)
                    aptl[1].x = bmi.cx - 1;
                    aptl[1].y = bmi.cy - 1;
                    // aptl[2]: source bottom-left, is all 0

                    // aptl[3]: source top-right (non-inclusive!)
                    aptl[3].x = bmi.cx;
                    aptl[3].y = bmi.cy;
                    lHits = GpiWCBitBlt(hpsMem,     // target HPS (bmp selected)
                            hbmSource,
                            4L,             // must always be 4
                            &aptl[0],       // points array
                            ROP_SRCCOPY,
                            BBO_IGNORE);
                                    // doesn't matter here, because we're not stretching

                    // step 2: overpaint bitmap
                    // with half-toned pattern

                    gpihSwitchToRGB(hpsMem);

                    GpiMove(hpsMem, &aptl[0]);  // still 0, 0
                    aptl[0].x = bmi.cx - 1;
                    aptl[0].y = bmi.cy - 1;
                    GpiSetColor(hpsMem, lColorGray);
                    GpiSetPattern(hpsMem, PATSYM_HALFTONE);
                    GpiBox(hpsMem,
                           DRO_FILL, // interior only
                           &aptl[0],
                           0, 0);    // no corner rounding

                    // unselect bitmap
                    GpiSetBitmap(hpsMem, NULLHANDLE);
                } // end if (GpiSetBitmap(hpsMem, hbmReturn) != HBM_ERROR)
                else
                {
                    // error selecting bitmap:
                    GpiDeleteBitmap(hbmReturn);
                    hbmReturn = NULLHANDLE;
                }
            } // end if (hbmReturn = gpihCreateBitmap...

            GpiDestroyPS(hpsMem);
            DevCloseDC(hdcMem);
        } // end if (gpihCreateMemPS(hab, &hdcMem, &hpsMem))
    } // end if (hbmSource)

    return (hbmReturn);
}

/*
 *@@ gpihLoadBitmapFile:
 *      this loads the specified bitmap file into
 *      the given HPS. Note that the bitmap is _not_
 *      yet selected into the HPS.
 *
 *      This function can currently only handle OS/2 1.3
 *      bitmaps.
 *
 *      Returns the new bitmap handle or NULL upon errors
 *      (e.g. if an OS/2 2.0 bitmap was accessed).
 *
 *      In the latter case, *pulError is set to one of
 *      the following:
 *      --  -1:      file not found
 *      --  -2:      malloc failed
 *      --  -3:      the bitmap data could not be read (fopen failed)
 *      --  -4:      file format not recognized (maybe OS/2 2.0 bitmap)
 *      --  -5:      GpiCreateBitmap error (maybe file corrupt)
 */

HBITMAP gpihLoadBitmapFile(HPS hps,             // in: HPS for bmp
                           PSZ pszBmpFile,      // in: bitmap filename
                           PULONG pulError)     // out: error code if FALSE is returned
{
    HBITMAP hbm;
    PBITMAPFILEHEADER2  pbfh;

    struct stat st;
    PBYTE       pBmpData;
    FILE        *BmpFile;

    if (stat(pszBmpFile, &st) == 0)
    {

        if ((pBmpData = (PBYTE)malloc(st.st_size)))
        {
            // open bmp file
            if ((BmpFile = fopen(pszBmpFile, "rb")))
            {
                // read bmp data
                fread(pBmpData, 1, st.st_size, BmpFile);
                fclose(BmpFile);

                // check bitmap magic codes
                if (pBmpData[0] == 'B' && pBmpData[1] == 'M')
                {
                    pbfh = (PBITMAPFILEHEADER2)pBmpData;
                    hbm = GpiCreateBitmap(hps,
                                &pbfh->bmp2,
                                CBM_INIT,
                                (pBmpData + pbfh->offBits),
                                (PBITMAPINFO2)&pbfh->bmp2);

                    if (hbm == NULLHANDLE)
                    {
                        if (pulError)
                        *pulError = -5;
                    }
                }
                else if (pulError)
                        *pulError = -4;

            }
            else if (pulError)
                    *pulError = -3;

            free(pBmpData);
        }
        else if (pulError)
                *pulError = -2;
    }
    else if (pulError)
            *pulError = -1;

    return (hbm);
}

/*
 *@@ gpihStretchBitmap:
 *      this copies hbmSource to the bitmap selected
 *      into hpsTarget, which must be a memory PS.
 *
 *      The source size is the whole size of hbmSource,
 *      the target size is specified in prclTarget
 *      (which is inclusive, meaning that the top right
 *      corner of that rectangle lies _outside_ the target).
 *
 *      This uses GpiWCBitBlt to stretch the bitmap.
 *      hbmSource therefore must _not_ be selected
 *      into any presentation space, or GpiWCBitBlt will
 *      fail.
 *
 *      If (fPropotional == TRUE), the target size is
 *      modified so that the proportions of the bitmap
 *      are preserved. The bitmap data will then be
 *      copied to a subrectangle of the target bitmap:
 *      there will be extra space either to the left
 *      and right of the bitmap data or to the bottom
 *      and top.
 *      The outside areas of the target bitmap are
 *      not changed then, so you might want to fill
 *      the bitmap with some color first.
 *
 *      This returns the return value of GpiWCBitBlt,
 *      which can be:
 *      --  GPI_OK
 *      --  GPI_HITS: correlate hits
 *      --  GPI_ERROR: error occured (probably either hbmSource not free
 *                     or no bitmap selected into hpsTarget)
 *
 *@added V1.00
 */

LONG gpihStretchBitmap(HPS hpsTarget,
                       HBITMAP hbmSource,
                       PRECTL prclSource,   // in: source rectangle; if NULL, use size of bitmap
                       PRECTL prclTarget,
                       BOOL fProportional)
{
    LONG                lHits = 0;
    BITMAPINFOHEADER2   bih2;
    POINTL              aptl[4];
    BOOL                fCalculated = FALSE;

    memset(aptl, 0, sizeof(POINTL) * 4);

    bih2.cbFix = sizeof(bih2);
    GpiQueryBitmapInfoHeader(hbmSource,
                             &bih2);

    // aptl[2]: source bottom-left, is all 0
    // aptl[3]: source top-right (non-inclusive!)
    aptl[3].x = bih2.cx;
    aptl[3].y = bih2.cy;

    if (fProportional)
    {
        // proportional mode:

        // 1) find out whether cx or cy is too
        // large

        ULONG ulPropSource = (bih2.cx * 1000)
                                    / bih2.cy;
                // e.g. if the bmp is 200 x 100, we now have 2000
        ULONG ulPropTarget = ((prclTarget->xRight - prclTarget->xLeft) * 1000)
                                    / (prclTarget->yTop - prclTarget->yBottom);
                // case 1: if prclTarget is 300 x 100, we now have 3000 (> ulPropSource)
                // case 2: if prclTarget is 150 x 100, we now have 1500 (< ulPropSource)

        // case 1:
        if (ulPropTarget > ulPropSource)
        {
            // prclTarget is too wide (horizontally):
            // decrease width, keep height

            ULONG cx = (prclTarget->xRight - prclTarget->xLeft);
            ULONG cxNew = (cx * ulPropSource) / ulPropTarget;

            // aptl[0]: target bottom-left
            // move left right (towards center)
            aptl[0].x = prclTarget->xLeft + ((cx - cxNew) / 2);
            aptl[0].y = prclTarget->yBottom;

            // aptl[1]: target top-right (inclusive!)
            aptl[1].x = aptl[0].x + cxNew;
            aptl[1].y = prclTarget->yTop;

            fCalculated = TRUE;
        }
        else
        {
            // prclTarget is too high (vertically):
            // keep width, decrease height

            ULONG cy = (prclTarget->yTop - prclTarget->yBottom);
            ULONG cyNew = (cy * ulPropTarget) / ulPropSource;

            // aptl[0]: target bottom-left
            aptl[0].x = prclTarget->xLeft;
            // move bottom up (towards center)
            aptl[0].y = prclTarget->yBottom + ((cy - cyNew) / 2);

            // aptl[1]: target top-right (inclusive!)
            aptl[1].x = prclTarget->xRight;
            aptl[1].y = aptl[1].y + cyNew;
                    // (prclTarget->yTop * ulPropSource) / ulPropTarget;

            fCalculated = TRUE;
        }
    } // end if (pa->ulFlags & ANF_PROPORTIONAL)

    if (!fCalculated)
    {
        // non-proportional mode or equal proportions:
        // stretch to whole size of prclTarget

        // aptl[0]: target bottom-left
        aptl[0].x = prclTarget->xLeft;
        aptl[0].y = prclTarget->yBottom;
        // aptl[1]: target top-right (inclusive!)
        aptl[1].x = prclTarget->xRight;
        aptl[1].y = prclTarget->yTop;
    }

    lHits = GpiWCBitBlt(hpsTarget,     // target HPS (bmp selected)
                        hbmSource,
                        4L,             // must always be 4
                        &aptl[0],       // points array
                        ROP_SRCCOPY,
                        BBO_IGNORE);
                                // ignore eliminated rows or
                                // columns; useful for color

    return (lHits);
}

