
/*
 *@@sourcefile eas.c:
 *      contains helper functions for handling Extended Attributes.
 *      See explanations below.
 *
 *      Function prefixes (new with V0.81):
 *      --  ea*         EA helper functions
 *
 *      This file is new with V0.81 and contains all the EA functions
 *      that were in helpers.c previously.
 *
 *      Most of the functions in here come in pairs: one for working
 *      on EAs based on a path (eaPath*), one for open HFILEs (eaHFile*).
 *      Most functions return structures which are allocated using
 *      malloc() and should be freed again later.
 *
 *      <B>Example:</B>
 +          PEABINDING peab = eaPathReadOneByName("C:\Desktop", ".LONGNAME");
 +          if (peab)
 +              PSZ pszLongName = peab->pszValue;
 +          eaFreeBinding(peab);
 *
 *@@include #include <os2.h>
 *@@include #include "eas.h"
 */

/*
 *      Most of the code in this file dealing with Extended Attributes is based
 *      on code (w) by Chris Hanson (cph@zurich.ai.mit.edu).
 *      Copyright (c) 1995 Massachusetts Institute of Technology.
 *
 *      The original code is available as EALIB.ZIP at Hobbes.
 *
 *      This material was developed by the Scheme project at the Massachusetts
 *      Institute of Technology, Department of Electrical Engineering and
 *      Computer Science.  Permission to copy this software, to redistribute
 *      it, and to use it for any purpose is granted, subject to the following
 *      restrictions and understandings.
 *
 *      1. Any copy made of this software must include this copyright notice
 *      in full.
 *
 *      2. Users of this software agree to make their best efforts (a) to
 *      return to the MIT Scheme project any improvements or extensions that
 *      they make, so that these may be included in future releases; and (b)
 *      to inform MIT of noteworthy uses of this software.
 *
 *      3. All materials developed as a consequence of the use of this
 *      software shall duly acknowledge such use, in accordance with the usual
 *      standards of acknowledging credit in academic research.
 *
 *      4. MIT has made no warrantee or representation that the operation of
 *      this software will be error-free, and MIT is under no obligation to
 *      provide any services, by way of maintenance, update, or otherwise.
 *
 *      5. In conjunction with products arising from the use of this material,
 *      there shall be no use of the name of the Massachusetts Institute of
 *      Technology nor of any adaptation thereof in any advertising,
 *      promotional, or sales literature without prior written consent from
 *      MIT in each case.
 *
 *      This file Copyright (C) 1997-99 Ulrich Mller,
 *                                      Massachusetts Institute of Technology.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "eas.h"

// #define _PMPRINTF_
#include "pmprintf.h"

/********************************************************************
 *                                                                  *
 *   Extended Attribute handling                                    *
 *                                                                  *
 ********************************************************************/

#define EA_BINDING_FLAGS(binding) ((binding)->bFlags)
#define EA_BINDING_NAME_LENGTH(binding) ((binding)->bNameLength)
#define EA_BINDING_VALUE_LENGTH(binding) ((binding)->usValueLength)
#define EA_BINDING_NAME(binding) ((binding)->pszName)
#define EA_BINDING_VALUE(binding) ((binding)->pszValue)

#define EA_LIST_BINDING(list) ((list)->peab)
#define EA_LIST_NEXT(list) ((list)->next)

// forward declarations to helper funcs at bottom
static PEALIST      ReadEAList(ULONG, PVOID);
static EABINDING *  ReadEAByIndex(ULONG, PVOID, ULONG);
static EABINDING *  ReadEAByName(ULONG, PVOID, PSZ);
static PDENA2       ReadDenaByIndex(ULONG, PVOID, ULONG);
static PEABINDING   GetEAValue(ULONG, PVOID, PDENA2);
static void         SetupQueryEAInfo(PDENA2, PEAOP2);
static PEABINDING   ConvertFeal2Binding(PFEA2LIST);
static void         WriteEAList(ULONG, PVOID, PEALIST);
static void         WriteEA(ULONG, PVOID, PEABINDING);
static PFEA2LIST    ConvertBinding2Feal(PEABINDING);

/*
 *@@ eaFreeBinding:
 *      deallocate EA binding memory that was generated
 *      by the ea...Read... procedures below. These procs
 *      assume that "malloc" was used for allocation and
 *      that the "pszName" and "pszValue" fields of each binding
 *      were also allocated using "malloc". "pszValue" may also
 *      be NULL.
 *
 *@@changed V1.00: added check for (binding != NULL)
 */

void eaFreeBinding(PEABINDING peab)
{
    if (peab)
    {
        free(EA_BINDING_NAME(peab));
        if ((EA_BINDING_VALUE(peab)) != 0)
            free(EA_BINDING_VALUE(peab));
        free(peab);
    }
}

/*
 *@@ eaFreeList:
 *      like eaFreeBinding, but for an EA binding list.
 *      This calls eaFreeBinding for each list item.
 *
 *@@changed V1.00: added check for (list != NULL)
 */

void eaFreeList(PEALIST list)
{
    if (list)
    {
        while (list != 0)
        {
            PEALIST next = (EA_LIST_NEXT (list));
            eaFreeBinding(EA_LIST_BINDING (list));
            free(list);
            list = next;
        }
    }
}

/* ******************************************************************
 *                                                                  *
 *   Read-EA functions                                              *
 *                                                                  *
 ********************************************************************/

/*
 * All of the following functions come in two flavors:
 *
 *      eaHFile*    operate on an open file handle.
 *
 *      eaPath*     operate on any file specified by its
 *                  filename, which may be fully qualified.
 */

/*
 *@@ eaPathQueryTotalSize:
 *      returns the total size of all EAs for a given file.
 *      This does not use the other EA functions, but
 *      accesses the EAs directly, so this is a lot quicker
 *      if you only want the total EA size without accessing
 *      the EA data.
 */

ULONG eaPathQueryTotalSize(PSZ pszPath)
{
    APIRET  arc;
    ULONG   ulTotalEASize = 0;
    CHAR    pchPath;
    // HDIR    hdir;
    // ULONG ulCount = 1;
    FILEFINDBUF4   ffb4;

    _Pmpf(("eaPathQueryTotalSize %s", pszPath));

    arc = DosQueryPathInfo(pszPath,
                       // &hdir,
                       // FILE_ARCHIVED | FILE_HIDDEN | FILE_SYSTEM | FILE_READONLY | FILE_DIRECTORY,
                       FIL_QUERYEASIZE,
                       &ffb4,
                       sizeof(FILEFINDBUF4));

    if (arc == NO_ERROR)
    {
        // CHAR szFile[CCHMAXPATH];
        // PBYTE pbBuffer = malloc(ffb4.cbList);
        BYTE abBuf[2000];
        LONG lCount = 0;
        PDENA2 pdena2;

        lCount = -1;

        arc = DosEnumAttribute(ENUMEA_REFTYPE_PATH,
                                      pszPath,
                                      1,
                                      abBuf,
                                      sizeof(abBuf),
                                      (PULONG)&lCount,
                                      ENUMEA_LEVEL_NO_VALUE);
            // ulCount now contains the EA count

        pdena2 = (PDENA2)abBuf;

        _Pmpf(("  %s: arc = %d, count = %d", pszPath, arc, lCount));

        if (lCount > 0) {
            ulTotalEASize = pdena2->cbName + 8;

            while (lCount > 0) {
                ulTotalEASize += (pdena2->cbValue + sizeof(DENA2));
                lCount--;
                pdena2 = (PDENA2) (((PBYTE) pdena2) +
                              pdena2->oNextEntryOffset);

            }
        }
    }

    _Pmpf(("    %s: total %d", pszPath, ulTotalEASize));

    return (ulTotalEASize);
}

/*
 *@@ eaPathReadAll:
 *      reads all of the extended attributes into an EALIST.
 *      Returns NULL if no EAs were found.
 *      The returned list should be freed using eaFreeList.
 */

PEALIST eaPathReadAll(PSZ path)
{
    return (ReadEAList(ENUMEA_REFTYPE_PATH, path));
}

/*
 *@@ eaHFileReadAll:
 *      like eaPathReadAll, but for an open file handle.
 */

PEALIST eaHFileReadAll(HFILE hfile)
{
    return (ReadEAList(ENUMEA_REFTYPE_FHANDLE, (&hfile)));
}

/*
 *@@ eaPathReadOneByIndex:
 *      returns one EA specified by a given index, counting
 *      from 1. Returns NULL if the specified index was not
 *      found, either because the file has no EAs at all or
 *      the index is too large.
 *      The returned binding should be freed using eaFreeBinding.
 */

PEABINDING eaPathReadOneByIndex(PSZ path, ULONG index)
{
    return (ReadEAByIndex(ENUMEA_REFTYPE_PATH, path, index));
}

/*
 *@@ eaHFileReadOneByIndex:
 *      like eaPathReadOneByIndex, but for an open file handle.
 */

PEABINDING eaHFileReadOneByIndex(HFILE hfile, ULONG index)
{
    return (ReadEAByIndex(ENUMEA_REFTYPE_FHANDLE, (&hfile), index));
}

/*
 *@@ eaPathReadOneByName:
 *      returns one EA specified by the given EA name (e.g.
 *      ".LONGNAME"). Returns NULL if not found.
 *      The returned binding should be freed using eaFreeBinding.
 */

PEABINDING eaPathReadOneByName(PSZ path, PSZ name)
{
    return (ReadEAByName(ENUMEA_REFTYPE_PATH, path, name));
}

/*
 *@@ eaHFileReadOneByName:
 *      like eaPathReadOneByName, but for an open file handle.
 */

PEABINDING eaHFileReadOneByName(HFILE hfile, PSZ name)
{
    return (ReadEAByName(ENUMEA_REFTYPE_FHANDLE, (&hfile), name));
}

/* ******************************************************************
 *                                                                  *
 *   Write-EA functions                                             *
 *                                                                  *
 ********************************************************************/

/*
 *@@ eaPathWriteAll:
 *      writes a list of EAs to a given file. These EAs
 *      are added to possibly existing EAs on the file;
 *      existing EAs will be overwritten.
 *
 *      A given EA is deleted if its EABINDING.usValueLength
 *      field is 0; only in that case, the EABINDING.value
 *      field may also be NULL.
 */

void eaPathWriteAll(PSZ path, PEALIST list)
{
    WriteEAList(ENUMEA_REFTYPE_PATH, path, list);
}

/*
 *@@ eaHFileWriteAll:
 *      like eaPathWriteAll, but for an open file handle.
 */

void eaHFileWriteAll(HFILE hfile, PEALIST list)
{
    WriteEAList(ENUMEA_REFTYPE_FHANDLE, (&hfile), list);
}

/*
 *@@ eaPathWriteOne:
 *      adds one EA to a given file. If the EA exists
 *      alredy, it is overwritten.
 *
 *      A given EA is deleted if its EABINDING.usValueLength
 *      field is 0; only in that case, the EABINDING.pszValue
 *      field may also be NULL.
 *
 *      To delete an EA, you may also use eaPathDeleteOne.
 */

void eaPathWriteOne(PSZ path, PEABINDING peab)
{
    WriteEA(ENUMEA_REFTYPE_PATH, path, peab);
}

/*
 *@@ eaHFileWriteOne:
 *      like eaPathWriteOne, but for an open file handle.
 */

void eaHFileWriteOne(HFILE hfile, PEABINDING peab)
{
    WriteEA(ENUMEA_REFTYPE_FHANDLE, (&hfile), peab);
}

/*
 *@@ eaPathDeleteOne:
 *      this deletes one EA by constructing a temporary
 *      EABINDING for pszEAName and calling eaPathWriteOne.
 *
 *@@added V1.00
 */

void eaPathDeleteOne(PSZ path, PSZ pszEAName)
{
    EABINDING eab;
    eab.bFlags = 0;
    eab.bNameLength = strlen(pszEAName);
    eab.pszName = pszEAName;
    eab.usValueLength = 0;
    eab.pszValue = 0;
    eaPathWriteOne(path, &eab);
}

/********************************************************************
 *                                                                  *
 *   Translation funcs                                              *
 *                                                                  *
 ********************************************************************/

/*
 *@@ eaQueryEAType:
 *      this returns the type of the EA stored
 *      in the given EA binding.
 *      See CPREF for the EAT_* codes which are
 *      returned here.
 *      Returns 0 upon errors.
 *
 *@@added V1.00
 */

USHORT eaQueryEAType(PEABINDING peab)
{
    USHORT usReturn = 0;
    if (peab)
        if (peab->pszValue)
            // first USHORT always has EAT_* flag
            usReturn = *((PUSHORT)(peab->pszValue));
    return (usReturn);
}

/*
 *@@ eaCreatePSZFromBinding:
 *      this returns a new PSZ for the given EABINDING,
 *      if this is a string EA (EAT_ASCII).
 *      This PSZ should be free()'d after use.
 *      Otherwise, NULL is returned.
 *
 *@@added V1.00
 */

PSZ eaCreatePSZFromBinding(PEABINDING peab)
{
    PSZ pszReturn = NULL;

    if (eaQueryEAType(peab) == EAT_ASCII)
    {
        // OK: next USHORT has length of string
        PUSHORT pusLength = (PUSHORT)(peab->pszValue + 2);
        pszReturn = malloc(*pusLength + 1);
        memcpy(pszReturn, peab->pszValue + 4, *pusLength);
        // add null terminator
        *(pszReturn + (*pusLength)) = 0;
    }

    return (pszReturn);
}

/*
 *@@ eaCreateBindingFromPSZ:
 *      reverse to eaCreatePSZFromBinding, this creates
 *      a new EABINDING from the given PSZ, which
 *      can be used with the write-EA functions.
 *      This EA is of type EAT_ASCII and will be
 *      made non-critical (peab->bFlags = 0).
 *      Returns NULL if the given string is NULL.
 *
 *@@added V1.00
 */

PEABINDING eaCreateBindingFromPSZ(PSZ pszEAName,     // in: EA name (e.g. ".LONGNAME")
                                  PSZ pszString)     // in: string for EAT_ASCII EA
{
    PEABINDING peab;
    if (pszString)
    {
        peab = (PEABINDING)malloc(sizeof(EABINDING));
        if (peab)
        {
            SHORT cbString = strlen(pszString);
            peab->bFlags = 0;
            peab->bNameLength = strlen(pszEAName);
            peab->pszName = strdup(pszEAName);
            peab->usValueLength = cbString + 4;
            peab->pszValue = (PSZ)malloc(peab->usValueLength);
            if (peab->pszValue)
            {
                // set first USHORT to EAT_ASCII
                *((PUSHORT)(peab->pszValue)) = EAT_ASCII;
                // set second USHORT to length of string
                *((PUSHORT)(peab->pszValue + 2)) = cbString;
                // copy string to byte 4 (no null-terminator)
                memcpy(peab->pszValue + 4, pszString, cbString);
            }
            else
            {
                // malloc error:
                if (peab->pszName)
                    free(peab->pszName);
                free(peab);
                peab = NULL;
            }
        }
    }

    return (peab);
}

/*
 *@@ eaQueryMVCount:
 *      this returns the number of subitems in a
 *      multi-value EA. This works for both EAT_MVMT
 *      and EAT_MVST (multi-type and single-type) EAs.
 *      Returns 0 upon errors.
 *
 *@@added V1.00
 */

USHORT eaQueryMVCount(PEABINDING peab,      // in: EA binding to examine (must be EAT_MVMT or EAT_MVST)
                      PUSHORT pusCodepage,  // out: codepage found in binding (ptr can be NULL)
                      PUSHORT pusEAType)    // out: either EAT_MVST or EAT_MVMT (ptr can be NULL)
{
    USHORT usReturn = 0;

    USHORT usEAType = eaQueryEAType(peab);

    if (pusEAType)
        *pusEAType = usEAType;

    if (usEAType == EAT_MVST)
    {
        // multi-value single-type:
        // pszValue is as follows (all USHORTs)
        // EAT_MVST usCodepage usCount usDataType data ....

        // store codepage
        if (pusCodepage)
            *pusCodepage = *((PUSHORT)(peab->pszValue + 2));
        // find count of entries
        usReturn = *((PUSHORT)(peab->pszValue + 4));
    } // end if (*((PUSHORT)(peab->pszValue)) == EAT_MVST)
    else if (usEAType == EAT_MVMT)
    {
        // multi-value multi-type:
        // pszValue is as follows (all USHORTs)
        // EAT_MVMT usCodepage usCount (usDataType data... )...

        // store codepage
        if (pusCodepage)
            *pusCodepage = *((PUSHORT)(peab->pszValue + 2));
        // find count of entries
        usReturn = *((PUSHORT)(peab->pszValue + 4));
    }

    return (usReturn);
}

/*
 *@@ eaQueryMVItem:
 *      this returns a pointer to the beginning
 *      of data of a subitem in a multi-value
 *      (EAT_MVST or EAT_MVMT) EA.
 *
 *      Note that this _only_ works if the data
 *      stored in the multi-value fields has
 *      length-data right after the EAT_* fields.
 *      This is true for EAT_ASCII, for example.
 *
 *      <B>Example:</B> If the EA value in the
 *      binding is like this:
 +          EAT_MVMT  codp  count [DataType   Data] ...
 +          EAT_MVMT  0000  0002  EAT_ASCII   000A Hello John
 +                                EAT_BINARY  0003 0x12 0x21 0x34
 *      calling this function with usindex==1 would return
 *      a pointer to the 0x12 byte in the EA data, set *pusCodepage
 *      to 0, and set *pusDataLength to 3.
 *
 *@@added V1.00
 */

PSZ eaQueryMVItem(PEABINDING peab,            // in: binding to examing
                  USHORT usIndex,             // in: item to search (starting at 0)
                  PUSHORT pusCodepage,        // out: codepage found in binding (ptr can be NULL)
                  PUSHORT pusEAType,          // out: EAT_* data type of data that the return value points to (ptr can be NULL)
                  PUSHORT pusDataLength)      // out: length of data that the return value points to (ptr can be NULL)
{
    PSZ pszReturn = NULL;

    USHORT usEAType = eaQueryEAType(peab);

    if (usEAType == EAT_MVST)
    {
        // multi-value single-type:
        // pszValue is as follows (all USHORTs)
        // EAT_MVST usCodepage usCount usDataType data ....
        USHORT usCount,
               usDataType;
        PUSHORT pusLengthThis;
        PSZ    pData;
        // store codepage
        if (pusCodepage)
            *pusCodepage = *((PUSHORT)(peab->pszValue + 2));
        // find count of entries
        usCount = *((PUSHORT)(peab->pszValue + 4));

        if (usIndex < usCount)
        {
            // find data type
            usDataType = *((PUSHORT)(peab->pszValue + 6));
            if (    (usDataType == EAT_ASCII)
                 || (usDataType == EAT_BINARY)
               )
            {
                USHORT us = 0;
                // find beginning of data (after length word)
                pData = peab->pszValue + 10;

                while (us < usIndex)
                {
                    pusLengthThis = (PUSHORT)(pData - 2);
                    pData += *pusLengthThis + 2;
                    us++;
                }

                // pData now points to the actual data;
                // pData - 2 has the length of the subvalue

                // return values
                if (pusEAType)
                    *pusEAType = usDataType; // same for all MVST subvalues
                if (pusDataLength)
                    *pusDataLength = *((PUSHORT)(pData - 2));
                pszReturn = pData;
            }
        }
    } // end if (usEAType == EAT_MVST)
    else if (usEAType == EAT_MVMT)
    {
        // multi-value multi-type:
        // pszValue is as follows (all USHORTs)
        // EAT_MVMT usCodepage usCount (usDataType data... )...
        USHORT usCount,
               usDataType;
        PUSHORT pusLengthThis;
        PSZ    pData;
        // store codepage
        if (pusCodepage)
            *pusCodepage = *((PUSHORT)(peab->pszValue + 2));
        // find count of entries
        usCount = *((PUSHORT)(peab->pszValue + 4));

        if (usIndex < usCount)
        {
            USHORT us = 0;
            pData = peab->pszValue + 6;

            while (us < usIndex)
            {
                PUSHORT pusDataType = (PUSHORT)(pData);

                if (    (*pusDataType == EAT_ASCII)
                     || (*pusDataType == EAT_BINARY)
                   )
                {
                    pusLengthThis = (PUSHORT)(pData + 2);
                    pData += *pusLengthThis + 4;
                    us++;
                }
                else
                {
                    pData = 0;
                    break;
                }
            }

            // pData now points to the usDataType field
            // of the subvalue;
            // pData + 2 is the length of the subvalue;
            // pData + 4 is the actual data

            // return values
            if (pData)
            {
                if (pusEAType)
                    *pusEAType = *((PUSHORT)(pData));  // different for each MVMT item
                if (pusDataLength)
                    *pusDataLength = *((PUSHORT)(pData + 2));
                pszReturn = pData + 4;
            }
        }
    } // end if (usEAType == EAT_MVMT)

    return (pszReturn);
}

/*
 *@@ eaCreatePSZFromMVBinding:
 *      this returns a new PSZ for the given EABINDING,
 *      if this is a multi-value, multi-type EA (EAT_MVMT),
 *      all of whose subitems are of EAT_ASCII though.
 *      Note that this does _not_ work for EAT_MVST EAs.
 *
 *      This format is used by the .COMMENTS and .KEYPHRASES
 *      system EAs, for example.
 *
 *      The different subitems will be separated by what
 *      you specify in pszSeparator. You can, for example,
 *      specify \n\r\0 to have CR-LF separators.
 *
 *      This PSZ should be free()'d after use.
 *      On errors, NULL is returned.
 *
 *@@added V1.00
 */

PSZ eaCreatePSZFromMVBinding(PEABINDING peab,       // in: EAT_MVMT binding
                             PSZ     pszSeparator,  // in: null-terminated string used as separator
                             PUSHORT pusCodepage)   // out: codepage found in binding (ptr can be NULL)
{
    PSZ     pszTotal = NULL;     // this will hold the whole string
    USHORT  usEAType = 0;

    USHORT usMVCount = eaQueryMVCount(peab,
                                      pusCodepage,
                                      &usEAType);
    if (    (usMVCount)
                // items found?
         && (usEAType == EAT_MVMT)
                // rule out EAT_MVST, which we cannot handle here
       )
    {
        // EAT_MVMT items found:
        // go thru all of them
        USHORT  us = 0;
        USHORT  cbComment = 0;
        USHORT  cbSeparator = strlen(pszSeparator);
        while (us < usMVCount)
        {
            USHORT  usEATypeThis = 0;
            USHORT  usEALenThis = 0;
            PSZ pszSubItem = eaQueryMVItem(peab,
                                                     us,    // index
                                                     NULL,  // codepage
                                                     &usEATypeThis,
                                                     &usEALenThis);
            if (usEATypeThis == EAT_ASCII)
                if (usEALenThis)
                {
                    PSZ pszTemp = pszTotal;
                    PSZ pTarget;
                    cbComment += usEALenThis + cbSeparator;
                    pszTotal = malloc(cbComment);
                    if (pszTemp)
                    {
                        // not first loop: append EA value
                        USHORT cbCommentOld = strlen(pszTemp);
                        // copy previous compiled string
                        strcpy(pszTotal, pszTemp);
                        // append separator
                        memcpy(pszTotal + cbCommentOld,
                               pszSeparator,
                               cbSeparator);
                        // copy the rest after the separator (below)
                        pTarget = pszTotal + cbCommentOld + cbSeparator;
                        free(pszTemp);
                    }
                    else
                        // first loop: copy to beginning
                        pTarget = pszTotal;

                    // copy EA value
                    memcpy(pTarget, pszSubItem, usEALenThis);
                    // append null terminator
                    *(pTarget + usEALenThis) = 0;
                }
            us++;
        } // end while (us < usMVCount)
    } // end if (usMVCount)

    return (pszTotal);
}

/*
 *@@ eaCreateMVBindingFromPSZ:
 *      reverse to eaCreatePSZFromMVBinding, this
 *      parses pszInput and creates an EAT_MVMT
 *      EABINDING, all of whose subvalues are of
 *      EAT_ASCII. The EABINDING is non-critical
 *      (peab->bFlags = 0).
 *
 *      pszSeparator is used in the analysis of pszInput
 *      as the separator string (e.g. "\r\n").
 *
 *      Returns NULL if the given string is NULL.
 *
 *      <B>Example:</B>
 +          PSZ pszInput = "Line 1\r\nLine2\r\nLine3\r\n";
 +          PEABINDING peab = eaCreateMVBindingFromPSZ(".KEYPHRASES",
 +                                                     pszInput,
 +                                                     "\r\n",
 +                                                     0);      // codepage
 *      will create an EABINDING with the following pszValue:
 +          EAT_MVMT codp usCount (usDataType usLen data...)
 +          EAT_MVMT   0     3    EAT_ASCII     6   Line 1
 +                                EAT_ASCII     5   Line2
 +                                EAT_ASCII     5   Line3
 *
 *@@added V1.00
 */

PEABINDING eaCreateMVBindingFromPSZ(PSZ pszEAName,      // in: EA name (e.g. ".KEYPHRASES")
                                    PSZ pszInput,       // in: string to parse
                                    PSZ pszSeparator,   // in: separator used in pszInput
                                    USHORT usCodepage)  // in: codepage to set in EAT_MVMT
{
    PEABINDING peab;
    if (pszInput)
    {
        peab = (PEABINDING)malloc(sizeof(EABINDING));
        if (peab)
        {
            PSZ     p = pszInput,
                    pSource,
                    pTarget;
            USHORT  cbInput = strlen(pszInput),
                    cbSep = strlen(pszSeparator),
                    usSepCount = 0,
                    cbToAlloc = 0,
                    cbThis,
                    us;

            peab->bFlags = 0;
            peab->bNameLength = strlen(pszEAName);
            peab->pszName = strdup(pszEAName);

            // now count the number of pszSeparators in pszInput
            while (p = strstr(p, pszSeparator))
            {
                usSepCount++;
                p += cbSep;
            }
            // usSepCount now has the separator count; that means
            // we have (usSepCount + 1) data fields

            // allocate the memory we need for the total EA:
            // EAT_MVMT usCodepage usCount (usDataType data... )...

            cbToAlloc =     6       // six fixed bytes for (EAT_MVMT usCodepage usCount)
                         +  cbInput - (usSepCount * cbSep)
                                    // length of input string minus length of all separators
                         +  ((usSepCount + 1) * 4);
                                    // for each data field (of which we'll have (usSepCount + 1)),
                                    // we need an extra four bytes for EAT_ASCII and the length
            peab->usValueLength = cbToAlloc;
            peab->pszValue = (PSZ)malloc(cbToAlloc + 1);

            // now initialize the first three fields:
            *((PUSHORT)(peab->pszValue)) = EAT_MVMT;
            *((PUSHORT)(peab->pszValue + 2)) = usCodepage;
            *((PUSHORT)(peab->pszValue + 4)) = (usSepCount + 1);

            // set pointer to first field
            pSource = pszInput;
            pTarget = peab->pszValue + 6;

            // now go thru all fields except the last
            for (us = 0;
                 us < usSepCount;       // exclude the last field
                 us++)
            {
                // find the next separator
                PSZ     pNextSep = strstr(pSource, pszSeparator);
                // calculate the length of the substring
                cbThis = pNextSep - pSource;
                // set data type in field
                *((PUSHORT)(pTarget)) = EAT_ASCII;
                // set length of this field
                *((PUSHORT)(pTarget + 2)) = cbThis;
                // copy data
                memcpy(pTarget + 4, pSource, cbThis);
                // advance source to the next (after substring and separator)
                pSource += cbThis + cbSep;
                // advance target to the next (after substring and (EAT_ASCII usLen)
                pTarget += cbThis + 4;
            }

            // now handle the last field
            cbThis = strlen(pSource);
            *((PUSHORT)(pTarget)) = EAT_ASCII;
            *((PUSHORT)(pTarget + 2)) = cbThis;
            memcpy(pTarget + 4, pSource, cbThis);
        } // end if (peab)
    } // end if (pszInput)

    return (peab);
}

/********************************************************************
 *                                                                  *
 *   EA helper funcs                                                *
 *                                                                  *
 ********************************************************************/

/*
 *  These get called by the functions above.
 *
 */

static PEALIST ReadEAList(ULONG type, PVOID pfile)
{
    ULONG index = 1;
    PEALIST head = 0;
    PEALIST tail = 0;

    while (1)
    {
        PEABINDING binding = (ReadEAByIndex(type, pfile, index));
        if (binding == 0)
            break;
        {
            PEALIST list = (malloc(sizeof (EALIST)));
            (EA_LIST_BINDING (list)) = binding;
            (EA_LIST_NEXT (list)) = 0;
            if (head == 0)
                head = list;
            else
                (EA_LIST_NEXT (tail)) = list;
            tail = list;
        }
        index += 1;
    }
    return (head);
}

static PEABINDING ReadEAByIndex(ULONG type, PVOID pfile, ULONG index)
{
    PDENA2 dena = (ReadDenaByIndex(type, pfile, index));
    return ((dena == 0)
                    ? 0
                    : (GetEAValue(type, pfile, dena)));
}

static PEABINDING ReadEAByName(ULONG type, PVOID pfile, PSZ name)
{
    ULONG index = 1;
    while (1)
    {
        PDENA2 dena = ReadDenaByIndex(type, pfile, index);
        if (dena == 0)
            return (NULL);
        if ((strcmp(name, (dena->szName))) == 0)
            return (GetEAValue(type, pfile, dena));
        free(dena);
        index += 1;
    }
}

static PDENA2 ReadDenaByIndex(ULONG type, PVOID pfile, ULONG index)
{
    ULONG count = 1;
    PDENA2 dena = (malloc(500)); // 500 is magic -- IBM doesn't explain.
    APIRET arc = DosEnumAttribute(type, pfile, index, dena, 500, (&count),
                     ENUMEA_LEVEL_NO_VALUE);
    if (count == 0)
    {
        free(dena);
        return (0);
    }
    else
        return (dena);
}

static PEABINDING GetEAValue(ULONG type, PVOID pfile, PDENA2 dena)
{
    ULONG level = FIL_QUERYEASFROMLIST;
    EAOP2 eaop;
    ULONG size = (sizeof(eaop));
    APIRET arc = NO_ERROR;
    SetupQueryEAInfo(dena, (&eaop));
    if (type == ENUMEA_REFTYPE_FHANDLE)
        arc = DosQueryFileInfo((* ((PHFILE) pfile)), level, (&eaop), size);
    else
        arc = DosQueryPathInfo(pfile, level, (&eaop), size);
    free(eaop.fpGEA2List);
    return (ConvertFeal2Binding(eaop.fpFEA2List));
}

static void SetupQueryEAInfo(PDENA2 dena, PEAOP2 eaop)
{
    unsigned int geal_size = ((sizeof (GEA2LIST)) + (dena->cbName));
    unsigned int feal_size
                 = ((sizeof (FEA2LIST)) + (dena->cbName) + (dena->cbValue));
    (eaop->fpGEA2List) = (malloc(geal_size));
    ((eaop->fpGEA2List)->cbList) = geal_size;
    (eaop->fpFEA2List) = (malloc(feal_size));
    ((eaop->fpFEA2List)->cbList) = feal_size;
    (eaop->oError) = 0;
    {
        PGEA2 gea = (&(((eaop->fpGEA2List)->list) [0]));
        (gea->oNextEntryOffset) = 0;
        (gea->cbName) = (dena->cbName);
        strcpy ((gea->szName), (dena->szName));
    }
    free(dena);
}

static PEABINDING ConvertFeal2Binding(PFEA2LIST feal)
{
    PFEA2 fea = (&((feal->list) [0]));
    PEABINDING binding = (malloc(sizeof (EABINDING)));
    (EA_BINDING_FLAGS (binding)) = (fea->fEA);
    (EA_BINDING_NAME_LENGTH (binding)) = (fea->cbName);
    (EA_BINDING_VALUE_LENGTH (binding)) = (fea->cbValue);
    (EA_BINDING_NAME (binding)) = (malloc((fea->cbName) + 1));
    strcpy ((EA_BINDING_NAME (binding)), (fea->szName));
    (EA_BINDING_VALUE (binding)) = (malloc(fea->cbValue));
    memcpy ((EA_BINDING_VALUE (binding)),
        (&((fea->szName) [(fea->cbName) + 1])),
        (fea->cbValue));
    free(feal);
    return (binding);
}

static void WriteEAList(ULONG type, PVOID pfile, PEALIST list)
{
    while (list != 0)
    {
        WriteEA(type, pfile, (EA_LIST_BINDING (list)));
        list = (EA_LIST_NEXT (list));
    }
}

static void WriteEA(ULONG type, PVOID pfile, PEABINDING binding)
{
    ULONG level = FIL_QUERYEASIZE;
    EAOP2 eaop;
    ULONG size = (sizeof (eaop));
    APIRET arc = NO_ERROR;

    (eaop.fpGEA2List) = 0;
    (eaop.fpFEA2List) = (ConvertBinding2Feal(binding));
    (eaop.oError) = 0;
    if (type == ENUMEA_REFTYPE_FHANDLE)
        arc = DosSetFileInfo((* ((PHFILE) pfile)), level, (&eaop), size);
    else
        arc = DosSetPathInfo(pfile, level, (&eaop), size, DSPI_WRTTHRU);
    free(eaop.fpFEA2List);
}

static PFEA2LIST ConvertBinding2Feal(PEABINDING binding)
{
    unsigned int feal_size
              = ((sizeof (FEA2LIST))
                 + (EA_BINDING_NAME_LENGTH (binding))
                 + (EA_BINDING_VALUE_LENGTH (binding)));
    PFEA2LIST feal = (malloc(feal_size));
    PFEA2 fea = (&((feal->list) [0]));
    (feal->cbList) = feal_size;
    (fea->oNextEntryOffset) = 0;
    (fea->fEA) = (EA_BINDING_FLAGS (binding));
    (fea->cbName) = (EA_BINDING_NAME_LENGTH (binding));
    (fea->cbValue) = (EA_BINDING_VALUE_LENGTH (binding));
    strcpy ((fea->szName), (EA_BINDING_NAME (binding)));
    if ((EA_BINDING_VALUE (binding)) != 0)
        memcpy ((&((fea->szName) [(fea->cbName) + 1])),
            (EA_BINDING_VALUE (binding)),
            (fea->cbValue));
    return (feal);
}


