
/*
 *@@sourcefile dosh.c:
 *      dosh.c contains Control Program helper functions that
 *      are independent of a single application, i.e. you can
 *      use them in any program. All of the following work with
 *      OS/2 only.
 *
 *      Function prefixes (new with V0.81):
 *      --  dosh*   Dos (Control Program) helper functions
 *
 *      These funcs are forward-declared in dosh.h, which
 *      must be #include'd first.
 *
 *      Save dosh.h to enforce a complete recompile of XWorkplace.
 *      That header is checked for in the main makefile.
 *
 *      The doshFindXXXPartition functions were kindly
 *      supplied by Christian Langanke, February 1999.
 *
 *@@header "dosh.h"
 */

/*
 *      This file Copyright (C) 1997-99 Ulrich Mller,
 *                                      Christian Langanke.
 *      This file is part of the XWorkplace source package.
 *      XWorkplace is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XWorkplace main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSDEVIOCTL
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_GPI
#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "dosh.h"

// #define _PMPRINTF_
#include "pmprintf.h"

const CHAR acDriveLetters[28] = " ABCDEFGHIJKLMNOPQRSTUVWXYZ";

/*
 *@@ doshGetULongTime:
 *      this returns the current time as a ULONG value (in milliseconds).
 *      Useful for stopping how much time the machine has spent in
 *      a certain function. To do this, simply call this function twice,
 *      and subtract the two values, which will give you the execution
 *      time in milliseconds.
 */

ULONG doshGetULongTime(VOID)
{
    DATETIME    dt;
    DosGetDateTime(&dt);
    return (10*(dt.hundredths + 100*(dt.seconds + 60*(dt.minutes + 60*(dt.hours)))));
}

/*
 *@@ doshQueryShiftState:
 *      returns TRUE if any of the SHIFT keys are
 *      currently pressed. Useful for checks during
 *      WM_COMMAND messages from menus.
 */

BOOL doshQueryShiftState(VOID)
{
    // BYTE abBuf[512];
    HFILE hfKbd;
    ULONG ulAction; //, cbRead, cbWritten;
    SHIFTSTATE      ShiftState;
    ULONG           cbDataLen = sizeof(ShiftState);

    DosOpen("KBD$", &hfKbd, &ulAction, 0,
             FILE_NORMAL, FILE_OPEN,
             OPEN_ACCESS_READONLY | OPEN_SHARE_DENYWRITE,
             (PEAOP2)NULL);

    DosDevIOCtl(hfKbd, IOCTL_KEYBOARD, KBD_GETSHIFTSTATE,
                      NULL, 0, NULL,      // no parameters
                      &ShiftState, cbDataLen, &cbDataLen);

    DosClose(hfKbd);

    return ((ShiftState.fsState & 3) != 0);
}


/*
 *@@ doshIsWarp4:
 *      returns TRUE only if at least OS/2 Warp 4 is running.
 */

BOOL doshIsWarp4(VOID)
{
    ULONG       aulBuf[3];

    DosQuerySysInfo(QSV_VERSION_MAJOR, QSV_VERSION_MINOR,
                    &aulBuf, sizeof(aulBuf));
    return ((aulBuf[0] != 20) || (aulBuf[1] > 30));
}

/*
 *@@ doshEnumDrives:
 *      this copies the drive letters of the drives on the
 *      system to pszBuffer which match the given pszFileSystem
 *      string (for example, "HPFS"). If pszFileSystem
 *      is NULL, all drives are enumerated.
 *      pszBuffer should be 27 characters in size to hold
 *      information for all drives.
 */

VOID doshEnumDrives(PSZ pszBuffer,      // out: drive letters
                    PSZ pszFileSystem)  // in: FS's to match or NULL
{
    CHAR szName[5] = "";
    ULONG ulLogicalDrive = 3, // start with drive C:
          ulFound = 0;        // found drives count
    APIRET arc             = NO_ERROR; // return code

    // go thru the drives, start with C: (== 3)
    do {
        UCHAR nonRemovable=0;
        ULONG parmSize=2;
        ULONG dataLen=1;

        #pragma pack(1)
        struct
        {
            UCHAR dummy,drive;
        } parms;
        #pragma pack()

        parms.drive=(UCHAR)(ulLogicalDrive-1);
        arc = DosDevIOCtl((HFILE)-1,
                        IOCTL_DISK,
                        DSK_BLOCKREMOVABLE,
                        &parms,
                        parmSize,
                        &parmSize,
                        &nonRemovable,
                        1,
                        &dataLen);

        if (arc == NO_ERROR)
        {
            if (nonRemovable)
            {
                ULONG  ulOrdinal       = 0;     // ordinal of entry in name list
                BYTE   fsqBuffer[sizeof(FSQBUFFER2) + (3 * CCHMAXPATH)] = {0};
                ULONG  cbBuffer   = sizeof(fsqBuffer);        // Buffer length)
                PFSQBUFFER2 pfsqBuffer = (PFSQBUFFER2)fsqBuffer;

                szName[0] = acDriveLetters[ulLogicalDrive];
                szName[1] = ':';
                szName[2] = '\0';

                arc = DosQueryFSAttach(
                                szName,          // logical drive of attached FS
                                ulOrdinal,       // ignored for FSAIL_QUERYNAME
                                FSAIL_QUERYNAME, // return data for a Drive or Device
                                pfsqBuffer,      // returned data
                                &cbBuffer);      // returned data length

                if (arc == NO_ERROR)
                {
                    // The data for the last three fields in the FSQBUFFER2
                    // structure are stored at the offset of fsqBuffer.szName.
                    // Each data field following fsqBuffer.szName begins
                    // immediately after the previous item.
                    CHAR* pszFSDName = (PSZ)&(pfsqBuffer->szName) + (pfsqBuffer->cbName) + 1;
                    if (pszFileSystem == NULL)
                    {
                        // enum-all mode: always copy
                        pszBuffer[ulFound] = szName[0]; // drive letter
                        ulFound++;
                    } else if (strcmp(pszFSDName, pszFileSystem) == 0) {
                        pszBuffer[ulFound] = szName[0]; // drive letter
                        ulFound++;
                    }
                }
            }
            ulLogicalDrive++;
        }
    } while ( arc == NO_ERROR );

    pszBuffer[ulFound] = '\0';
}

/*
 *@@ doshQueryBootDrive:
 *      returns the letter of the boot drive
 */

CHAR doshQueryBootDrive(VOID)
{
    ULONG ulBootDrive;
    DosQuerySysInfo(5, 5, &ulBootDrive, sizeof(ulBootDrive));
    return (acDriveLetters[ulBootDrive]);
}

/*
 *@@ doshIsFixedDisk:
 *      checks whether a disk is fixed or removeable.
 *      ulLogicalDrive must be 1 for drive A:, 2 for B:, ...
 *      The result is stored in *pfFixed.
 *      Returns DOS error code.
 *      Warning: This uses DosDevIOCtl, which has proved
 *      to cause problems with some device drivers for
 *      removeable disks.
 */

APIRET doshIsFixedDisk(ULONG  ulLogicalDrive,        // in: 1 for A:, 2 for B:, 3 for C:, ...
                       PBOOL  pfFixed)             // out: TRUE for fixed disks
{
    APIRET arc = ERROR_INVALID_DRIVE;

    if (ulLogicalDrive) {
        // parameter packet
        struct {
            UCHAR command, drive;
        } parms;

        // data packet
        UCHAR ucNonRemoveable;

        ULONG ulParmSize = sizeof(parms);
        ULONG ulDataSize = sizeof(ucNonRemoveable);

        parms.drive = (UCHAR)(ulLogicalDrive-1);
        arc = DosDevIOCtl((HFILE)-1,
                        IOCTL_DISK,
                        DSK_BLOCKREMOVABLE,
                        &parms,
                        ulParmSize,
                        &ulParmSize,
                        &ucNonRemoveable,
                        ulDataSize,
                        &ulDataSize);

        if (arc == NO_ERROR)
            *pfFixed = (BOOL)ucNonRemoveable;
    }

    return (arc);
}

/*
 *@@ doshAssertDrive:
 *      this checks for whether the given drive
 *      is currently available. If fSuppress == TRUE,
 *      DosError is used in a critical section to
 *      avoid "Drive not ready" popups.
 */

APIRET doshAssertDrive(ULONG ulLogicalDrive, // in: 1 for A:, 2 for B:, 3 for C:, ...
                       BOOL fSuppress)       // in: suppress-popups flag
{
    FSALLOCATE  fsa;
    APIRET arc = NO_ERROR;

    if (fSuppress) {
        DosError(FERR_DISABLEHARDERR | FERR_DISABLEEXCEPTION);
        DosEnterCritSec();
    }

    arc = DosQueryFSInfo(ulLogicalDrive, FSIL_ALLOC, &fsa, sizeof(fsa));

    if (fSuppress) {
        DosError(FERR_ENABLEHARDERR | FERR_ENABLEEXCEPTION);
        DosExitCritSec();
    }

    return (arc);
}

/*
 *@@ doshQueryDiskFree:
 *       returns the number of bytes remaining on the disk
 *       specified by the given logical drive, or -1 upon errors.
 *       Note: This returns a "double" value, because a ULONG
 *       can only hold values of some 4 billion, which would
 *       lead to funny results for drives > 4 GB.
 */

double doshQueryDiskFree(ULONG ulLogicalDrive) // in: 1 for A:, 2 for B:, 3 for C:, ...
{
    FSALLOCATE  fsa;
    double      dbl = -1;

    if (ulLogicalDrive)
        if (DosQueryFSInfo(ulLogicalDrive, FSIL_ALLOC, &fsa, sizeof(fsa))
                == NO_ERROR)
            dbl = (fsa.cSectorUnit * fsa.cbSector * fsa.cUnitAvail);

    return (dbl);
}

/*
 *@@ doshQueryDiskFSType:
 *       copies the file-system type of the given disk object
 *       (HPFS, FAT, CDFS etc.) to pszBuf.
 *       Returns the DOS error code.
 */

APIRET doshQueryDiskFSType(ULONG ulLogicalDrive, // in:  1 for A:, 2 for B:, 3 for C:, ...
                           PSZ   pszBuf)         // out: buffer for FS type
{
    APIRET arc = NO_ERROR;
    CHAR szName[5];

    szName[0] = acDriveLetters[ulLogicalDrive];
    szName[1] = ':';
    szName[2] = '\0';

    {
        ULONG  ulOrdinal       = 0;     // ordinal of entry in name list
        // PBYTE  pszFSDName      = NULL;  // pointer to FS name
        // PBYTE  prgFSAData      = NULL;  // pointer to FS data
        BYTE   fsqBuffer[sizeof(FSQBUFFER2) + (3 * CCHMAXPATH)] = {0};
        ULONG  cbBuffer   = sizeof(fsqBuffer);        // Buffer length)
        PFSQBUFFER2 pfsqBuffer = (PFSQBUFFER2)fsqBuffer;

        arc = DosQueryFSAttach(
                        szName,          // logical drive of attached FS
                        ulOrdinal,       // ignored for FSAIL_QUERYNAME
                        FSAIL_QUERYNAME, // return data for a Drive or Device
                        pfsqBuffer,      // returned data
                        &cbBuffer);      // returned data length

        if (arc == NO_ERROR)
        {
            // The data for the last three fields in the FSQBUFFER2
            // structure are stored at the offset of fsqBuffer.szName.
            // Each data field following fsqBuffer.szName begins
            // immediately after the previous item.
            PSZ pszFSDName = (CHAR*)(&pfsqBuffer->szName) + pfsqBuffer->cbName + 1;
            strcpy(pszBuf, pszFSDName);
        }
    }

    return (arc);
}

/*
 *@@ doshQueryDiskType:
 *      this retrieves more information about a given drive,
 *      which is stored in *pulDeviceType. According to the
 *      IBM Control Program Reference, this value can be:
 *      --  0  48 TPI low-density diskette drive
 *      --  1  96 TPI high-density diskette drive
 *      --  2  3.5-inch 720KB diskette drive
 *      --  3  8-Inch single-density diskette drive
 *      --  4  8-Inch double-density diskette drive
 *      --  5  Fixed disk
 *      --  6  Tape drive
 *      --  7  Other (includes 1.44MB 3.5-inch diskette drive)
 *      --  8  R/W optical disk
 *      --  9  3.5-inch 4.0MB diskette drive (2.88MB formatted)
 *
 *      This returns a DOS error code.
 *
 *      Warning: This uses DosDevIOCtl, which has proved
 *      to cause problems with some device drivers for
 *      removeable disks.
 */

APIRET doshQueryDiskType(ULONG  ulLogicalDrive,      // in: 1 = A:, 2 = B:, ...
                         PULONG pulDeviceType)       // out: device type
{
    APIRET arc = ERROR_INVALID_DRIVE;

    if (ulLogicalDrive) {
        #pragma pack(1)
        // parameter packet
        struct {
            UCHAR command, drive;
        } parms;
        // data packet
        struct {
            UCHAR   aucBPB[31];         // BIOS parameter block
            USHORT  usCylinders;
            UCHAR   ucDeviceType;
            USHORT  usDeviceAttrs;
        } data;
        #pragma pack()

        ULONG ulParmSize = sizeof(parms);
        ULONG ulDataSize = sizeof(data);

        parms.command = 1; // read currently inserted media
        parms.drive=(UCHAR)(ulLogicalDrive-1);

        arc = DosDevIOCtl((HFILE)-1,
                        IOCTL_DISK,
                        DSK_GETDEVICEPARAMS,
                        &parms, ulParmSize, &ulParmSize,
                        &data,  ulDataSize, &ulDataSize);

        if (arc == NO_ERROR)
            *pulDeviceType = (ULONG)(data.ucDeviceType);
    }

    return (arc);
}

/*
 *@@ doshIsFileOnFAT:
 *      returns TRUE if pszFileName resides on
 *      a FAT drive.
 */

BOOL doshIsFileOnFAT(const char* pcszFileName)
{
    BOOL brc = FALSE;
    CHAR szName[5];

    szName[0] = pcszFileName[0];
    szName[1] = ':';
    szName[2] = '\0';

    #ifdef DEBUG_TITLECLASH
        _Pmpf(("cmnIsFAT for %s -> %s", pcszFileName, szName));
    #endif
    {
        ULONG  ulOrdinal       = 0;     // ordinal of entry in name list
        // PBYTE  pszFSDName      = NULL;  // pointer to FS name
        // PBYTE  prgFSAData      = NULL;  // pointer to FS data
        APIRET rc              = NO_ERROR; // return code
        BYTE   fsqBuffer[sizeof(FSQBUFFER2) + (3 * CCHMAXPATH)] = {0};
        ULONG  cbBuffer   = sizeof(fsqBuffer);        // Buffer length)
        PFSQBUFFER2 pfsqBuffer = (PFSQBUFFER2)fsqBuffer;

        rc = DosQueryFSAttach(
                        szName,          // logical drive of attached FS
                        ulOrdinal,       // ignored for FSAIL_QUERYNAME
                        FSAIL_QUERYNAME, // return data for a Drive or Device
                        pfsqBuffer,      // returned data
                        &cbBuffer);      // returned data length

        if (rc == NO_ERROR)
        {
            // The data for the last three fields in the FSQBUFFER2
            // structure are stored at the offset of fsqBuffer.szName.
            // Each data field following fsqBuffer.szName begins
            // immediately after the previous item.
            PSZ pszFSDName = (PSZ)&(pfsqBuffer->szName) + pfsqBuffer->cbName + 1;
            #ifdef DEBUG_TITLECLASH
                _Pmpf(("  cmnIsFAT -> File system: %s", pszFSDName));
            #endif
            if (strncmp(pszFSDName, "FAT", 3) == 0)
                brc = TRUE;
        }
    }
    return (brc);
}

/*
 *@@ doshIsValidFileName:
 *      this returns NO_ERROR only if pszFile is a valid file name.
 *      This may include a full path.
 *
 *      If a drive letter is specified, this checks for whether
 *      that drive is a FAT drive and adjust the checks accordingly,
 *      i.e. 8+3 syntax (per path component).
 *      If no drive letter is specified, this check is performed
 *      for the current drive.
 *
 *      This also checks if pszFileNames contains characters which
 *      are invalid for the current drive.
 *
 *      Note: this performs syntactic checks only. This does not
 *      check for whether the specified path components exist.
 *      However, it _is_ checked for whether the given drive
 *      exists.
 *
 *      If an error is found, the corresponding DOS error code
 *      is returned:
 *      --  ERROR_INVALID_DRIVE
 *      --  ERROR_FILENAME_EXCED_RANGE  (on FAT: no 8+3 filename)
 *      --  ERROR_INVALID_NAME          (invalid character)
 */

APIRET doshIsValidFileName(const char* pcszFile)
{
    CHAR    szPath[CCHMAXPATH+4] = " :";
    CHAR    szComponent[CCHMAXPATH];
    PSZ     p1, p2;
    BOOL    fIsFAT = FALSE;
    PSZ     pszInvalid;

    // check drive first
    if (*(pcszFile + 1) == ':')
    {
        CHAR cDrive = toupper(*pcszFile);
        // drive specified:
        strcpy(szPath, pcszFile);
        szPath[0] = toupper(*pcszFile);
        if (doshQueryDiskFree(cDrive - 'A' + 1) == -1)
            return (ERROR_INVALID_DRIVE);
    }
    else
    {
        // no drive specified: take current
        ULONG   ulDriveNum = 0,
                ulDriveMap = 0;
        DosQueryCurrentDisk(&ulDriveNum, &ulDriveMap);
        szPath[0] = ((UCHAR)ulDriveNum) + 'A' - 1;
        szPath[1] = ':';
        strcpy(&szPath[2], pcszFile);
    }

    fIsFAT = doshIsFileOnFAT(szPath);

    pszInvalid = (fIsFAT)
                              ? "<>|+=:;,\"/[] "  // invalid characters in FAT
                              : "<>|:\"/";        // invalid characters in IFS's

    // now separate path components
    p1 = &szPath[2];       // advance past ':'

    do {

        if (*p1 == '\\')
            p1++;

        p2 = strchr(p1, '\\');
        if (p2 == NULL)
            p2 = p1 + strlen(p1);

        if (p1 != p2)
        {
            LONG    lDotOfs = -1,
                    lAfterDot = -1;
            ULONG   cbFile,
                    ul;
            PSZ     pSource = szComponent;

            strncpy(szComponent, p1, p2-p1);
            szComponent[p2-p1] = 0;
            cbFile = strlen(szComponent);

            // now check each path component
            for (ul = 0; ul < cbFile; ul++)
            {
                if (fIsFAT)
                {
                    // on FAT: only 8 characters allowed before dot
                    if (*pSource == '.')
                    {
                        lDotOfs = ul;
                        lAfterDot = 0;
                        if (ul > 7)
                            return (ERROR_FILENAME_EXCED_RANGE);
                    }
                }
                // and check for invalid characters
                if (strchr(pszInvalid, *pSource) != NULL)
                    return (ERROR_INVALID_NAME);

                pSource++;

                // on FAT, allow only three chars after dot
                if (fIsFAT)
                    if (lAfterDot != -1)
                    {
                        lAfterDot++;
                        if (lAfterDot > 3)
                            return(ERROR_FILENAME_EXCED_RANGE);
                    }
            }

            // we are still missing the case of a FAT file
            // name without extension; if so, check whether
            // the file stem is <= 8 chars
            if (fIsFAT)
                if (lDotOfs == -1)  // dot not found:
                    if (cbFile > 8)
                        return (ERROR_FILENAME_EXCED_RANGE);
        }

        // go for next component
        p1 = p2+1;
    } while (*p2);

    return (NO_ERROR);
}

/*
 *@@ doshMakeRealName:
 *      this copies pszSource to pszTarget, replacing
 *      all characters which are not supported by file
 *      systems with cReplace.
 *      pszTarget must be at least the same size as pszSource.
 *      If (fIsFAT), the file name will be made FAT-compliant (8+3).
 *      Returns TRUE if characters were replaced.
 */

BOOL doshMakeRealName(PSZ pszTarget,    // out: new real name
                      PSZ pszSource,    // in: filename to translate
                      CHAR cReplace,    // in: replacement char for invalid
                                        //     characters (e.g. '!')
                      BOOL fIsFAT)      // in: make-FAT-compatible flag
{
    ULONG ul,
          cbSource = strlen(pszSource);
    LONG  lDotOfs = -1,
          lAfterDot = -1;
    BOOL  brc = FALSE;
    PSZ   pSource = pszSource,
          pTarget = pszTarget,
          pszInvalid = (fIsFAT)
                            ? "<>|+=:;,\"/\\[] "  // invalid characters in FAT
                            : "<>|:\"/\\"; // invalid characters in IFS's

    for (ul = 0; ul < cbSource; ul++)
    {
        if (fIsFAT) {
            // on FAT: truncate filename if neccessary
            if (*pSource == '.')
            {
                lDotOfs = ul;
                lAfterDot = 0;
                if (ul > 7) {
                    // only 8 characters allowed before dot,
                    // so set target ptr to dot pos
                    pTarget = pszTarget+8;
                }
            }
        }
        // and replace invalid characters
        if (strchr(pszInvalid, *pSource) == NULL)
            *pTarget = *pSource;
        else {
            *pTarget = cReplace;
            brc = TRUE;
        }
        pTarget++;
        pSource++;

        // on FAT, allow only three chars after dot
        if (fIsFAT)
            if (lAfterDot != -1) {
                lAfterDot++;
                if (lAfterDot > 3)
                    break;
            }
    }
    *pTarget = '\0';
    #ifdef DEBUG_TITLECLASH
        _Pmpf(("      strMakeRealName: returning %s", pszTarget));
    #endif

    if (fIsFAT)
    {
        // we are still missing the case of a FAT file
        // name without extension; if so, check whether
        // the file stem is <= 8 chars
        if (lDotOfs == -1)  // dot not found:
            if (cbSource > 8)
                *(pszTarget+8) = 0; // truncate

        // convert to upper case
        strupr(pszTarget);
    }

    return (brc);
}

/*
 *@@ doshQueryFileSize:
 *      returns the size of an already opened file
 *      or 0 upon errors.
 *      Use doshQueryPathSize to query the size of
 *      any file.
 */

ULONG doshQueryFileSize(HFILE hFile)
{
    FILESTATUS3 fs3;
    if (DosQueryFileInfo(hFile, FIL_STANDARD, &fs3, sizeof(fs3)))
        return (0);
    else
        return (fs3.cbFile);
}

/*
 *@@ doshQueryPathSize:
 *      returns the size of any file,
 *      or 0 if the file could not be
 *      found.
 *      Use doshQueryFileSize instead to query the
 *      size if you have a HFILE.
 */

ULONG doshQueryPathSize(PSZ pszFile)
{
    FILESTATUS3 fs3;
    if (DosQueryPathInfo(pszFile, FIL_STANDARD, &fs3, sizeof(fs3)))
        return (0);
    else
        return (fs3.cbFile);
}

/*
 *@@ doshQueryDirExist:
 *      returns TRUE if the given directory
 *      exists.
 */

BOOL doshQueryDirExist(PSZ pszDir)
{
    FILESTATUS3 fs3;
    APIRET arc = DosQueryPathInfo(pszDir, FIL_STANDARD, &fs3, sizeof(fs3));
    if (arc == NO_ERROR)
        // file found:
        return ((fs3.attrFile & FILE_DIRECTORY) != 0);
    else
        return FALSE;
}

/*
 *@@ doshCreatePath:
 *      this creates the specified directory.
 *      As opposed to DosCreateDir, this
 *      function can create several directories
 *      at the same time, if the parent
 *      directories do not exist yet.
 */

APIRET doshCreatePath(PSZ pszPath)
{
    APIRET  arc0 = NO_ERROR;
    CHAR    path[CCHMAXPATH];
    CHAR    *cp, c;
    ULONG   cbPath;

    strcpy(path, pszPath);
    cbPath = strlen(path);

    if (path[cbPath] != '\\')
    {
        path[cbPath] = '\\';
        path[cbPath+1] = 0;
    }

    cp = path;
    // advance past the drive letter only if we have one
    if (*(cp+1) == ':')
        cp += 3;

    // Now, make the directories
    while (*cp != 0)
    {
        if (*cp == '\\')
        {
            c = *cp;
            *cp = 0;
            if (!doshQueryDirExist(path))
            {
                APIRET arc = DosCreateDir(path, 0);
                if (arc != NO_ERROR)
                {
                    arc0 = arc;
                    break;
                }
            }
            *cp = c;
        }
        cp++;
    }
    return (arc0);
}

/*
 *@@ doshSetCurrentDir:
 *      sets the current working directory
 *      to the given path.
 *      As opposed to DosSetCurrentDir, this
 *      one will change the current drive
 *      also, if one is specified.
 */

APIRET doshSetCurrentDir(PSZ pszDir)
{
    if (pszDir)
    {
        if (*pszDir != 0)
            if (*(pszDir+1) == ':')
            {
                // drive given:
                CHAR    cDrive = toupper(*(pszDir));
                APIRET  arc;
                // change drive
                arc = DosSetDefaultDisk( (ULONG)(cDrive - 'A' + 1) );
                        // 1 = A:, 2 = B:, ...
                if (arc != NO_ERROR)
                    return (arc);
            }

        return (DosSetCurrentDir(pszDir));
    }
    return (ERROR_INVALID_PARAMETER);
}

/*
 *@@ doshQuerySysErrorMsg:
 *      this retrieves the error message for a system
 *      error from the system error message file (OSO001.MSG).
 *      The error message is returned in a newly allocated
 *      buffer, which should be free()'d afterwards.
 *
 *      Returns NULL upon errors.
 */

PSZ doshQuerySysErrorMsg(APIRET arc)
{
    PSZ     pszReturn = 0;
    CHAR    szDosError[1000];
    ULONG   cbDosError = 0;
    DosGetMessage(NULL, 0,       // no string replacements
            szDosError, sizeof(szDosError),
            arc,
            "OSO001.MSG",        // default OS/2 message file
            &cbDosError);
    if (cbDosError > 2)
    {
        szDosError[cbDosError - 2] = 0;
        pszReturn = strdup(szDosError);
    }
    return (pszReturn);
}

/*
 *@@ doshReadTextFile:
 *      reads a text file from disk, allocates memory
 *      via malloc() and sets a pointer to this
 *      buffer (or NULL upon errors).
 *
 *      This returns the APIRET of DosOpen and DosRead.
 *      If any error occured, no buffer was allocated.
 *      Otherwise, you should free() the buffer when
 *      no longer needed.
 */

APIRET doshReadTextFile(PSZ pszFile,        // in: file name to read
                        PSZ* ppszContent)   // out: newly allocated buffer with file's content
{
    ULONG   ulSize,
            ulBytesRead = 0,
            ulAction, ulLocal;
    HFILE   hFile;
    PSZ     pszContent = NULL;
    APIRET arc = DosOpen(pszFile,
                         &hFile,
                         &ulAction,                      // action taken
                         5000L,                          // primary allocation size
                         FILE_ARCHIVED | FILE_NORMAL,    // file attribute
                         OPEN_ACTION_OPEN_IF_EXISTS,     // open flags
                         OPEN_FLAGS_NOINHERIT
                            | OPEN_SHARE_DENYNONE
                            | OPEN_ACCESS_READONLY,      // read-only mode
                         NULL);                          // no EAs

    if (arc == NO_ERROR)
    {
        ulSize = doshQueryFileSize(hFile);
        pszContent = (PSZ)malloc(ulSize+1);
        arc = DosSetFilePtr(hFile,
                            0L,
                            FILE_BEGIN,
                            &ulLocal);
        arc = DosRead(hFile,
                      pszContent,
                      ulSize,
                      &ulBytesRead);
        DosClose(hFile);
        *(pszContent+ulBytesRead) = '\0';

        // set output buffer pointer
        *ppszContent = pszContent;
    }

    return (arc);
}

/*
 *@@ doshCreateBackupFileName:
 *      creates a valid backup filename of pszExisting
 *      with a numerical file name extension which does
 *      not exist in the directory where pszExisting
 *      resides.
 *      Returns a PSZ to a new buffer which was allocated
 *      using malloc().
 *
 *      <B>Example:</B> returns "C:\CONFIG.002" for input
 *      "C:\CONFIG.SYS" if "C:\CONFIG.001" already exists.
 */

PSZ doshCreateBackupFileName(const char* pszExisting)
{
    // CHAR    szTemp[CCHMAXPATH];
    PSZ     pszNew = strdup(pszExisting),
            pszLastDot = strrchr(pszNew, '.');
    ULONG   ulCount = 1;
    CHAR    szCount[5];
    do {
        sprintf(szCount, ".%03d", ulCount);
        strcpy(pszLastDot, szCount);
        ulCount++;
    } while (doshQueryPathSize(pszNew) != 0);
    return (pszNew);
}

/*
 *@@ doshWriteTextFile:
 *      writes a text file to disk; pszFile must contain the
 *      whole path and filename.
 *      An existing file will be backed up if (fBackup == TRUE),
 *      using doshCreateBackupFileName; otherwise the file
 *      will be overwritten.
 *      Returns the number of bytes written or 0 upon errors.
 */

ULONG doshWriteTextFile(const char* pszFile,        // in: file name
                        const char* pszContent,     // in: text to write
                        BOOL fBackup)               // in: create-backup flag
{
    ULONG   ulWritten = 0,
            ulAction, ulLocal;
    HFILE   hFile;
    APIRET rc;

    if (fBackup)
    {
        PSZ     pszBackup = doshCreateBackupFileName(pszFile);
        DosCopy(pszFile, pszBackup, DCPY_EXISTING);
        free(pszBackup);
    }

    rc = DosOpen(pszFile,
                 &hFile,
                 &ulAction,                      // action taken
                 5000L,                          // primary allocation size
                 FILE_ARCHIVED | FILE_NORMAL,    // file attribute
                 OPEN_ACTION_CREATE_IF_NEW
                 | OPEN_ACTION_REPLACE_IF_EXISTS,  // open flags
                 OPEN_FLAGS_NOINHERIT
                 | OPEN_FLAGS_WRITE_THROUGH      // write immediately w/out cache
                 | OPEN_FLAGS_NO_CACHE           // do not store in cache
                 | OPEN_FLAGS_SEQUENTIAL         // sequential, not random access
                 | OPEN_SHARE_DENYNONE
                 | OPEN_ACCESS_WRITEONLY,        // read-write mode
                 NULL);                          // no EAs

    if (rc == NO_ERROR)
    {
        ULONG ulSize = strlen(pszContent);      // exclude 0 byte

        rc = DosSetFilePtr(hFile,
                            0L,
                            FILE_BEGIN,
                            &ulLocal);
        if (rc == NO_ERROR)
        {
            rc = DosWrite(hFile,
                          (PVOID)pszContent,
                          ulSize,
                          &ulWritten);
            if (rc == NO_ERROR)
                rc = DosSetFileSize(hFile, ulSize);
        }

        DosClose(hFile);

        if (rc != NO_ERROR)
            ulWritten = 0;
    }

    return (ulWritten);
}

/*
 *@@ doshOpenLogFile:
 *      this opens a log file in the root directory of
 *      the boot drive; it is titled pszFilename, and
 *      the file handle is returned.
 */

HFILE doshOpenLogFile(const char* pcszFilename)
{
    APIRET  rc;
    CHAR    szFileName[CCHMAXPATH];
    HFILE   hfLog;
    ULONG   ulAction;
    ULONG   ibActual;

    sprintf(szFileName, "%c:\\%s", doshQueryBootDrive(), pcszFilename);
    rc = DosOpen(szFileName, &hfLog, &ulAction, 0,
                 FILE_NORMAL,
                 OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYWRITE,
                 (PEAOP2)NULL);
    if (rc == NO_ERROR)
    {
        DosSetFilePtr(hfLog, 0, FILE_END, &ibActual);
        return (hfLog);
    }
    else
        return (0);
}

/*
 * doshWriteToLogFile
 *      writes a string to a log file, adding a
 *      leading timestamp.
 */

APIRET doshWriteToLogFile(HFILE hfLog, const char* pcsz)
{
    if (hfLog)
    {
        DATETIME dt;
        CHAR szTemp[2000];
        ULONG   cbWritten;
        DosGetDateTime(&dt);
        sprintf(szTemp, "Time: %02d:%02d:%02d %s",
            dt.hours, dt.minutes, dt.seconds,
            pcsz);
        return (DosWrite(hfLog, (PVOID)szTemp, strlen(szTemp), &cbWritten));
    }
    else return (ERROR_INVALID_HANDLE);
}

/*
 *@@ doshQuickStartSession:
 *      this is a shortcut to DosStartSession w/out having to
 *      deal with all the messy parameters.
 *
 *      This one starts pszPath as a child session and passes
 *      pszParams to it.
 *
 *      If (Wait), this function will create a termination queue
 *      and not return until the child session has ended. Otherwise
 *      the function will return immediately, and the SID/PID of
 *      the child session can be found in *pulSID and *ppid.
 *
 *      Returns the error code of DosStartSession.
 */

APIRET doshQuickStartSession(PSZ pszPath,     // in: program to start
                             PSZ pszParams,   // in: parameters for program
                             BOOL Visible,    // in: show program?
                             BOOL Wait,       // in: wait for termination?
                             PULONG pulSID,   // out: session ID
                             PPID ppid)       // out: process ID
{
    APIRET rc;
    REQUESTDATA rqdata;
    ULONG DataLength; PULONG DataAddress; BYTE elpri;
    int er;
    // Queue Stuff
    HQUEUE hq;
    PID qpid=0;
    STARTDATA   SData;
    CHAR        szObjBuf[CCHMAXPATH];

    if (Wait) {
        if (    (er=DosCreateQueue(&hq, QUE_FIFO|QUE_CONVERT_ADDRESS, "\\queues\\kfgstart.que"))
             != NO_ERROR)
        {
            char str[80];
            sprintf(str, "Create %ld\n", er);
        }

        if ((er=DosOpenQueue(&qpid, &hq, "\\queues\\kfgstart.que")) != NO_ERROR)
        {
            char str[80];
            sprintf(str, "Open %ld\n", er);
        }
    }

    SData.Length  = sizeof(STARTDATA);
    SData.Related = SSF_RELATED_CHILD; //INDEPENDENT;
    SData.FgBg    = SSF_FGBG_FORE;
    SData.TraceOpt = SSF_TRACEOPT_NONE;

    SData.PgmTitle = pszPath;       // title for window
    SData.PgmName = pszPath;
    SData.PgmInputs = pszParams;

    SData.TermQ = (Wait) ? "\\queues\\kfgstart.que" : NULL;
    SData.Environment = 0;
    SData.InheritOpt = SSF_INHERTOPT_SHELL;
    SData.SessionType = SSF_TYPE_DEFAULT;
    SData.IconFile = 0;
    SData.PgmHandle = 0;

    SData.PgmControl = ((Visible) ? SSF_CONTROL_VISIBLE : SSF_CONTROL_INVISIBLE);
    SData.InitXPos  = 30;
    SData.InitYPos  = 40;
    SData.InitXSize = 200;
    SData.InitYSize = 140;
    SData.Reserved = 0;
    SData.ObjectBuffer  = (CHAR*)&szObjBuf;
    SData.ObjectBuffLen = (ULONG)sizeof(szObjBuf);

    rc = DosStartSession(&SData, pulSID, ppid);

    if (rc == NO_ERROR)
    {
        if (Wait)
        {
            rqdata.pid=qpid;
            DosReadQueue(hq, &rqdata, &DataLength, (PVOID*)&DataAddress, 0, 0, &elpri, 0);
            DosCloseQueue(hq);
        }
    }
    return (rc);
}

/*
 *@@ doshSetPathAttr:
 *      sets the file attributes of pszFile,
 *      which can be fully qualified.
 *
 *      fAttr can be:
 *      --  FILE_ARCHIVED
 *      --  FILE_READONLY
 *      --  FILE_SYSTEM
 *      --  FILE_HIDDEN
 *
 *      Note that this simply sets all the given
 *      attributes; the existing attributes
 *      are lost.
 */

APIRET doshSetPathAttr(const char* pcszFile, ULONG fAttr)
{
    FILESTATUS3 fs3;
    APIRET rc = DosQueryPathInfo(pcszFile,
            FIL_STANDARD,
            &fs3,
            sizeof(fs3));

    if (rc == NO_ERROR) {
        fs3.attrFile = fAttr;
        rc = DosSetPathInfo(pcszFile,
                FIL_STANDARD,
                &fs3,
                sizeof(fs3),
                DSPI_WRTTHRU);
    }
    return (rc);
}

/********************************************************************
 *                                                                  *
 *   Partition functions                                            *
 *                                                                  *
 ********************************************************************/

/*
 *  The following code has been kindly supplied by Christian Langanke.
 *
 */

// query command

#define COMMAND_QUERY_PARTITION "FDISK /QUERY"

// fdisk output format:
// - start (SOCn) and end (EOCn)of column 1 to 8
// - values are zero based !
#define SOC1  0
#define SOC2  6
#define SOC3 18
#define SOC4 21
#define SOC5 29
#define SOC6 32
#define SOC7 39
#define SOC8 50

#define EOC1  5
#define EOC2 14
#define EOC3 20
#define EOC4 25
#define EOC5 31
#define EOC6 38
#define EOC7 49
#define EOC8 58

// some more macros
#define CR             '\r'
#define LN             '\n'
#define NEXTSTR(s)      (s+strlen(s) + 1)
#define MEMORY_SIG     'PL'
#define BUFFER_SIZE    1024
#define COLUMN_STATUS  38

static   USHORT         usMemorySig       = 'PL';
static   PSZ            pszBootableStatus = "13";


/*
 * PARTITIONLIST:
 *
 */

typedef struct _PARTITIONLIST
{
      ULONG          usMemorySig;
      ULONG          ulStatus;
      PSZ            pszCurrentLine;
      CHAR           szBuffer[2];      // two terminating bytes
} PARTITIONLIST, *PPARTITIONLIST;

/*
 *@@ doshFindFirstPartition:
 *      this starts a partition enumeration.
 *      Use the handle returned in *php for
 *      subsequent calls to doshFindNextPartition
 *      and doshFindClosePartition.
 *
 *      First, set *pulEntries to the number of
 *      partitions which should be returned.
 *      *ppartitionlist must be (*pulEntries * sizeof(PARTITION))
 *      in size for this to work.
 *
 *      The partitions which are returned depend
 *      on the ulStatus flag:
 *      --  PART_STATUS_STARTABLE: return the _one_ startable partition
 *                                 (i.e. bootmanager)
 *      --  PART_STATUS_BOOTABLE: return all bootable partitions
 *      --  PART_STATUS_ALL: return _all_ partitions
 *
 *      Returns:
 *      --  NO_ERROR
 *      --  ERROR_NO_MORE_FILES        ( == no more partitions)
 *      --  ERROR_INVALID_PARAMETER
 *      --  ERROR_NOT_ENOUGH_MEMORY
 *
 *      and the resultcodes of:
 *      --  doshFindNextPartition
 *      --  DosCreatePipe
 *      --  DosSearchPath
 *      --  DosExecPgm
 *      --  DosRead
 */

APIRET doshFindFirstPartition(PHPARTITION    php,            // out: enum handle
                              ULONG          ulStatus,       // in: what to query
                              PVOID          ppartitionlist, // out: PARTITION buffer
                              ULONG          ulBuflen,       // in: sizeof(*ppartitionlist)
                              PULONG         pulEntries)     // in/out: partitions to query
{
    APIRET         rc = NO_ERROR;

    HFILE          hfRead    = NULLHANDLE;
    HFILE          hfWrite   = NULLHANDLE;

    CHAR           szCommand[ 1024];

    ULONG          ulBytesRead;
    PPARTITIONLIST pplBuffer = NULL;
    PSZ            pszBuffer;
    ULONG          ulBufferLen = 0;

    PSZ            pszLine;

    do
    {
        // check parms
        if ((php            == NULL) ||
            (ppartitionlist == NULL) ||
            (pulEntries     == NULL) ||
            (*pulEntries    == 0))
        {
            rc = ERROR_INVALID_PARAMETER;
            break;
        }

        // create a pipe
        rc = DosCreatePipe( &hfRead, &hfWrite, BUFFER_SIZE);
        if (rc != NO_ERROR)
           break;

        // call fdisk
        sprintf( szCommand, COMMAND_QUERY_PARTITION " 1>&%u 2>&%u", hfWrite, hfWrite);
        rc = system( szCommand);
        if (rc == -1)
        {
            rc = 1041;
            break;
        }

        // get memory
        // - first four bytes is a ptr to current line
        // - after that a ASCII string list follows
        ulBufferLen = BUFFER_SIZE;
        pplBuffer = (PPARTITIONLIST)malloc(ulBufferLen + sizeof(PARTITIONLIST));
                        // typecast added for C++ (*UM)
        if (pplBuffer == NULL)
        {
            rc = ERROR_NOT_ENOUGH_MEMORY;
            break;
        }

        // save base adress as handle
        // and adress the base pointer
        pplBuffer->usMemorySig = usMemorySig;
        pplBuffer->ulStatus    = ulStatus;
        pszBuffer              = pplBuffer->szBuffer;
        *php                   = (HPARTITION)pplBuffer;

        // now read output
        *pszBuffer = 0;
        while (rc == NO_ERROR)
        {
            // read the pipe
            rc = DosRead( hfRead,
                          pszBuffer,
                          BUFFER_SIZE,
                          &ulBytesRead);
            if (rc != NO_ERROR)
                break;

            // make sure buffer is double zero terminated
            *(pszBuffer + ulBytesRead)     = 0;
            *(pszBuffer + ulBytesRead + 1) = 0;

            // all read ?
            if (ulBytesRead < BUFFER_SIZE)
                break;

            // prepare to read next block
            pszBuffer   += strlen( pszBuffer);
            ulBufferLen += BUFFER_SIZE;
            pszBuffer = (PSZ)realloc(pszBuffer, ulBufferLen);
                        // typecast added for C++ (*UM)
            if (pszBuffer == NULL)
            {
                rc = ERROR_NOT_ENOUGH_MEMORY;
                break;
            }
        } // end while

        // handle error
        if (rc != NO_ERROR)
           break;

        // split buffer up to lines at CRs, handle LFs later
        pszLine = pplBuffer->szBuffer;
        pszLine = strchr( pszLine, CR);
        while (pszLine)
        {
            // cut of rest of buffer
            if (pszLine)
                *pszLine++ = 0;

            // adress next line
            pszLine = strchr( pszLine, CR);
        }

        // search first line
        pszLine = pplBuffer->szBuffer;
        while (*pszLine)
        {
            // search for colon
            if (strchr( pszLine, ':'))
               break;

            // adress next line
            pszLine = NEXTSTR( pszLine);
        }

        // save ptr to this line
        pplBuffer->pszCurrentLine = pszLine;

        // now get entries
        rc = doshFindNextPartition( *php, ppartitionlist, ulBuflen, pulEntries);

    } while (FALSE);


    // cleanup
    if (hfRead)  DosClose( hfRead);
    if (hfWrite) DosClose( hfWrite);
    return rc;
}

/*
 *@@ doshFindNextPartition:
 *
 *      Returns:
 *      --  0              NO_ERROR
 *      --  18             ERROR_NO_MORE_FILES        ( == no more partitions)
 *      --  87             ERROR_INVALID_PARAMETER
 */

APIRET doshFindNextPartition(HPARTITION     hp,
                             PVOID          ppartitionlist,
                             ULONG          ulBuflen,
                             PULONG         pulEntries)
{
    APIRET         rc = NO_ERROR;

    PPARTITIONLIST pplBuffer = (PPARTITIONLIST)hp;
    PARTITION      partition;

    PSZ            pszLine;
    CHAR           szLine[ 81];

    CHAR           szFStype[ 20];
    PSZ            pszEos;

    ULONG          ulEntryCount;
    ULONG          ulMaxEntries;

    do {
        // check parms
        if (    (pplBuffer      == NULL)
             || (ppartitionlist == NULL)
             || (pplBuffer->usMemorySig != usMemorySig)
             || (pulEntries     == NULL)
             || (*pulEntries    == 0)
           )
        {
            rc = ERROR_INVALID_PARAMETER;
            break;
        }

        // adress current line
        pszLine  = pplBuffer->pszCurrentLine;
        if (*pszLine == 0)
        {
            rc = ERROR_NO_MORE_FILES;
            break;
        }

        // check if requested entries would fit into memory
        ulMaxEntries = ulBuflen / sizeof( PARTITION);
        *pulEntries  = min( ulMaxEntries, *pulEntries);

        // get next lines
        ulEntryCount = 0;
        while ((*pszLine) && (ulEntryCount < *pulEntries))
           {
           // skip linefeeds here
           while ((*pszLine) && (*pszLine < 0x20))
              {
              pszLine++;
              }

           if (*pszLine != 0)
              {
              do
                 {
                 // check if line is valid
                 strcpy( szLine, pszLine);
                 if (szLine[ EOC3 -1] != ':')
                    break;

                 memset( &partition, 0, sizeof( partition));

                 // store disk num
                 szLine[ EOC1] = 0;
                 partition.ulDisk = atol( &szLine[ SOC1]);

                 // store name
                 szLine[ EOC2] = 0;
                 strcpy( partition.szName, &szLine[ SOC2]);

                 // store drive
                 szLine[ EOC3] = 0;
                 strcpy( partition.szDrive, &szLine[ SOC3]);
                 if (partition.szDrive[ 0] == 0x20)
                    partition.szDrive[ 0] = 0;

                 // store volume type
                 szLine[ EOC4] = 0;
                 partition.ulVType = atol( &szLine[ SOC4]);

                 // store file system type
                 szLine[ EOC5] = 0;
                 strcpy( szFStype, &szLine[ SOC5]);
                 partition.ulFStype = strtol( szFStype, &pszEos, 16);

                 // store partition status
                 szLine[ EOC6] = 0;
                 partition.ulStatus = atol( &szLine[ SOC6]);

                 // check status
                 if ((pplBuffer->ulStatus != PART_STATUS_ALL) &&
                     (pplBuffer->ulStatus != partition.ulStatus))
                    break;

                 // store partition start
                 szLine[ EOC7] = 0;
                 partition.ulStart = atol( &szLine[ SOC7]);

                 // store partition size
                 szLine[ EOC8] = 0;
                 partition.ulSize = atol( &szLine[ SOC8]);

                 // store entry
                 memcpy( (PPARTITION) ppartitionlist + ulEntryCount, &partition, sizeof( partition));
                 ulEntryCount++;

                 } while (FALSE);

              } // if (*pszLine != 0)

           // adress next line
           pszLine = NEXTSTR( pszLine);

           }

        // save pointer
        pplBuffer->pszCurrentLine = pszLine;
        *pulEntries = ulEntryCount;

    } while (FALSE);

    return rc;

}

/*
 *@@ doshFindClosePartition:
 *
 *      Returns:
 *      --  0              NO_ERROR
 *      --  87             ERROR_INVALID_PARAMETER
 */

APIRET doshFindClosePartition(HPARTITION hp)
{
    APIRET         rc = NO_ERROR;

    PPARTITIONLIST pplBuffer = (PPARTITIONLIST)hp;

    do {
        // check parms
        if ((pplBuffer == NULL) ||
            (pplBuffer->usMemorySig != usMemorySig))
        {
            rc = ERROR_INVALID_PARAMETER;
            break;
        }

        // free the memory
        free( pplBuffer);

    } while (FALSE);

    return rc;
}

