
/*
 * wic.cpp:
 *      this is the command-line WarpIn Archive Creator, "wic.exe".
 *
 *      This file Copyright (C) 1998-99 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */


#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include <io.h>

#include "ansiscrn.h"
#include "dosh.h"
#include "stringh.h"

#include "wiarchive.h"

char szLastFile[300] = "";

class CancelExcpt { int i; };

/*
 * WICallback:
 *      this is the general callback from the WIArchive class.
 */

int WICallback(short mode, short s, WIFileHeader* file)
{
    if (mode == CBM_PERCENTAGE)
    {
        // s has the percentage
        if (strcmp(szLastFile, file->name) != 0) {
            strcpy(szLastFile, file->name);
        } else
            printf ("\r");
        printf ("Compressing %s (%03d%%)", file->name, s);
        fflush(stdout);
    }
    else if (mode ==  CBM_UPDATING)
    {
        // ANSI: go right 40 chars
        printf("\r%c[%dC",
                    27,     // ESC
                    40);
        printf("%02d%% compression \n",
                100 - ((file->compsize * 100) / file->origsize));
        fflush(stdout);
    }
    else {
        printf("\nError %d occured. Terminating.\n", mode);
    }
    return (CBRC_ABORT);
}

/*
 * ExplainParams:
 *      say hello if the params aren't right
 */

void ExplainParams(void)
{
    ANSI_fg_color(B_WHITE);
    printf("wic - WarpIn archive creation and maintenance\n");
    ANSI_fg_color(WHITE);
    printf("(C) 1998-99 Jens Beckmn, Ulrich Mller\n");
    ANSI_fg_color(B_WHITE);
    printf("Syntax:\n");
    ANSI_fg_color(WHITE);
    printf("    wic <arc> -<mode> <params>\n"
           "with: \n"
           "    <arc> being the archive to work on (def. ext.: .WPI)\n"
           "The different <mode>'s are:\n");
    ANSI_fg_color(B_WHITE);
    printf("Add files:");
    ANSI_fg_color(WHITE);
               printf(" wic <arc> -a {<pck> <filemask>...}...\n"
           "    adds the files in <filemask> to <pck> in <arc>.\n"
           "    You can specify several packages and/or several filemasks.\n"
           "    Examples:\n"
           "        wic test -a 1 *.cpp *.h 2 *.exe\n");
    ANSI_fg_color(B_WHITE);
    printf("Delete files:");
    ANSI_fg_color(WHITE);
                  printf(" not implemented yet\n");
    ANSI_fg_color(B_WHITE);
    printf("List packages:");
    ANSI_fg_color(WHITE);
                   printf(" wic <arc> -l [<pck>]\n"
           "    list the files in <pck> in <arc>.\n"
           "    If <pck> is not specified, an archive summary is displayed.\n");
    ANSI_fg_color(B_WHITE);
    printf("Set installation script:");
    ANSI_fg_color(WHITE);
                             printf(" wic <arc> -s <file>\n"
           "    sets <file> as the install script for <arc>.wpi. This can be\n"
           "    combined with 'Add' mode, e.g.:\n"
           "        wic test -s test.wis -a 1 *.cpp *.h 2 *.exe\n");
    ANSI_fg_color(WHITE);
    exit(1);
}

/*
 * PrintFilesCallback:
 *      callback for WIArchive::for_all_files
 *      when enumerating files in "list mode".
 */

int PrintFilesCallback(WIFileHeader* file,
                       unsigned long ulNotUsedHere)
{
    CHAR    szOriginal[20],
            szCompressed[20];

    printf(" %-40s %10s %10s   %3d%%\n",
                file->name,
                strhThousandsULong(szOriginal,
                                   file->origsize,
                                   '.'),
                strhThousandsULong(szCompressed,
                                   file->compsize,
                                   '.'),
                100 - ( (file->compsize * 100 )/ file->origsize));
}

/*
 *@@ AddToPackage:
 *
 */

struct AddToPackage
{
    LONG        lPackage;           // package to add to
    list<PSZ>   listFileMasks;      // file masks to add to package

    // constructor
    AddToPackage()
    {
        lPackage = 0;
    }
};

WIArchive Arc;
WIArcHeader ArcHeader;

char    szFilestem[256] = "";  // filestem given on the cmdline
char    szArcFile[256],
        szScriptFile[256];
PSZ     pszScriptBuf = NULL;   // for reading in the script

// list of files to work on in 'Add' mode
list<AddToPackage*> listAddToPackages;
AddToPackage*   pAddToPackageThis = 0;

LONG    lListPackage = 0;       // default: list all packages


BOOL fContinue = TRUE;

ULONG ulNextMode = 0;

/*
 *@@ AddMode:
 *
 */

void AddMode(void)
{
    if (access(szArcFile, 0) == 0)
        printf("Archive \"%s\" exists and will be updated.\n",
                szArcFile);

    if (Arc.open(szArcFile, 1) == 0)
    {
        printf("Error reading/creating \"%s\".\n", szArcFile);
        throw(CancelExcpt());
    }

    strcpy(ArcHeader.name_app, "Test application");
    strcpy(ArcHeader.name_dev, "Ulrich Mller");
    strcpy (ArcHeader.inet_address, "N/A");
    ArcHeader.rev_app = 1;
    Arc.set_arc_header(ArcHeader);

    printf("Collecting files\n");
    fflush(stdout);

    APIRET arc = 0;

    // go thru the packages which were specified on the command line;
    // each of these AddToPackage structures has in turn a list
    // of file masks to be added.
    // This list might be empty if only a script is to be added.
    list<AddToPackage*>::iterator PckStart = listAddToPackages.begin(),
                                  PckEnd = listAddToPackages.end();
    for (; PckStart != PckEnd; PckStart++)
    {
        AddToPackage* pAddToPackageThis = *PckStart;

        char szPackageName[20];
        sprintf(szPackageName, "Pck%03d", pAddToPackageThis->lPackage);
        Arc.set_package(pAddToPackageThis->lPackage, szPackageName);

        // go thru list of file masks for this package
        list<PSZ>::iterator FileMaskStart = pAddToPackageThis->listFileMasks.begin(),
                            FileMaskEnd = pAddToPackageThis->listFileMasks.end();

        for (; FileMaskStart != FileMaskEnd; FileMaskStart++)
        {
            PSZ pszFileMaskThis = *FileMaskStart;
            printf("Adding \"%s\" to package %d",
                   pszFileMaskThis,
                   pAddToPackageThis->lPackage);
            fflush(stdout);

            // DosFindFirst returns the file names only, so if the
            // user specified relative paths on the command line,
            // we'll need to manually add those again when adding
            // files to the archive.
            CHAR    szDrive[_MAX_DRIVE],        // "C:"
                    szDir[_MAX_DIR],            // "\test\"
                    szFilestem[_MAX_FNAME],     // "*"
                    szExt[_MAX_EXT];            // ".obj"
            _splitpath(pszFileMaskThis, szDrive, szDir, szFilestem, szExt);

            // now check if this is valid
            if (szDrive[0] != 0)
            {
                // drive specified: no!
                printf("\nError in file mask \"%s\":\n"
                       "You cannot store absolute drive specifications in an archive because this would\n"
                       "require the user to have the same drive letters.\n",
                       pszFileMaskThis);
                throw(CancelExcpt());
            }
            // check if first part of the path is a backslash; that's absolute too
            if (szDir[0] == '\\')
            {
                printf("\nError in file mask \"%s\":\n"
                       "You cannot store absolute path specifications in an archive because this would\n"
                       "prevent the user from installing to any directory.\n"
                       "If you specify paths in file masks, use relative specifications only.\n",
                       pszFileMaskThis);
                throw(CancelExcpt());
            }

            HDIR          hdirFindHandle = HDIR_CREATE;
            FILEFINDBUF3  ffb3           = {0};      // Returned from FindFirst/Next
            ULONG         ulResultBufLen = sizeof(FILEFINDBUF3);
            ULONG         ulFindCount    = 1;        // Look for 1 file at a time

            arc = DosFindFirst(pszFileMaskThis,       // PSZ file pattern
                                &hdirFindHandle,      // directory search handle
                                FILE_NORMAL,          // search attribute
                                &ffb3,                // result buffer
                                ulResultBufLen,       // result buffer length
                                &ulFindCount,         // number of entries to find
                                FIL_STANDARD);        // return level 1 file info

            if (arc != NO_ERROR)
            {
                printf("\nNo files for search mask \"%s\" found\n",
                        pszFileMaskThis);
                printf("SYS%04d: %s\n", arc, doshQuerySysErrorMsg(arc));
                throw(CancelExcpt());
            }

            // char szFilenameInArc[256];
            // keep finding the next file until there are no more files
            while (arc != ERROR_NO_MORE_FILES)
            {
                PSZ pszFileNameInArc;
                if (    // skip directory entries
                        (strcmp(ffb3.achName, ".") != 0)
                     && (strcmp(ffb3.achName, "..") != 0)
                        // skip the archive file being created
                     && (strcmp(ffb3.achName, szArcFile) != 0)
                   )
                {
                    CHAR    szFullName[CCHMAXPATH];
                    sprintf(szFullName, "%s%s",
                            szDir,
                            ffb3.achName);
                    // printf("Adding %s, %s\n", szFullName, szFullName);
                    printf(".");
                    fflush(stdout);
                    Arc.add(szFullName, szFullName,
                            pAddToPackageThis->lPackage);
                }

                ulFindCount = 1;                      // Reset find count.
                arc = DosFindNext(hdirFindHandle,      // Directory handle
                                 &ffb3,         // Result buffer
                                 ulResultBufLen,      // Result buffer length
                                 &ulFindCount);       // Number of entries to find

            } // endwhile

            arc = DosFindClose(hdirFindHandle);    // close our find handle
            printf("\n");

        } // end for FileMaskStart
    } // end for PckStart...

    if (szScriptFile[0] != 0)
    {
        arc = doshReadTextFile(szScriptFile, &pszScriptBuf);
        if (arc != NO_ERROR)
        {
            printf("Script file %s.wis not found. Terminating.\n", szFilestem);
            throw(CancelExcpt());
        }
        printf("Script \"%s\".wis successfully read, %d bytes.\n",
                szFilestem,
                strlen(pszScriptBuf));
        printf("Writing script to archive... ");
        fflush(stdout);
        Arc.set_script(pszScriptBuf);
        printf("OK\n");
    }

    Arc.set_callback_func (WICallback);

    Arc.close();

    printf("Done.\n");
}

/*
 *@@ ListMode:
 *
 */

void ListMode(void)
{
    CHAR szOriginal[30],
         szCompressed[30];

    if (Arc.open(szArcFile, 0) == 0)
    {
        printf("Error opening \"%s\" for reading.\n", szArcFile);
        throw(CancelExcpt());
    }

    if (lListPackage == 0)
    {
        // "list all packages" mode:
        ULONG ulPckCount = 0,
              ulTotalFiles = 0,
              ulTotalCompressed = 0,
              ulTotalOriginal = 0;
        list<WIPackHeader*> *pPackageList = Arc.get_pack_list();
        list<WIPackHeader*>::iterator pckStart, pckEnd;
        pckStart = pPackageList->begin();
        pckEnd = pPackageList->end();
        for (; pckStart != pckEnd; pckStart++)
        {
            WIPackHeader* pHeaderThis = *pckStart;

            printf("Package %d\n", pHeaderThis->number);
            printf("  %4d files, size: %10s bytes original\n"
                   "                    %10s bytes compressed (%d%% compression)\n",
                    pHeaderThis->files,
                    strhThousandsULong(szOriginal,
                                       pHeaderThis->origsize,
                                       '.'),
                    strhThousandsULong(szCompressed,
                                       pHeaderThis->compsize,
                                       '.'),
                    100 - ( (pHeaderThis->compsize * 100 ) / pHeaderThis->origsize )
                );

            ulTotalFiles += pHeaderThis->files;
            ulTotalCompressed += pHeaderThis->compsize;
            ulTotalOriginal += pHeaderThis->origsize;
            ulPckCount++;
        }
        if (ulTotalFiles == 0)
            // otherwise we'd get a division by zero below
            printf("No packages found.\n");
        else
            printf("Summary: %d package(s), %d files total\n"
                   "    %10s bytes original\n"
                   "    %10s bytes compressed (%d%% compression)\n",
                    ulPckCount,
                    ulTotalFiles,
                    strhThousandsULong(szOriginal,
                                       ulTotalOriginal,
                                       '.'),
                    strhThousandsULong(szCompressed,
                                       ulTotalCompressed,
                                       '.'),
                    100 - ( (ulTotalCompressed * 100 ) / ulTotalOriginal )
                );
    }
    else
    {
        // list one package only:
        // enumerate through package headers
        BOOL fFound = FALSE;
        list<WIPackHeader*> *pPackageList = Arc.get_pack_list();
        list<WIPackHeader*>::iterator pckStart, pckEnd;
        pckStart = pPackageList->begin();
        pckEnd = pPackageList->end();
        for (; pckStart != pckEnd; pckStart++)
            if ((**pckStart).number == lListPackage)
            {
                fFound = TRUE;
                break;
            }

        if (fFound)
        {
            printf("Listing files in package %d\n", lListPackage);
            printf(" %-40s %10s %10s   compression\n",
                        "filename",
                        "origsize",
                        "compsize");
            printf(" ----------------------------------------------------------------------------\n");
            Arc.for_all_files(*pckStart, PrintFilesCallback, 0);

            printf("Package %d summary:\n", (**pckStart).number);
            printf("  %4d files, size: %10s bytes original\n"
                   "                    %10s bytes compressed (%d%% compression)\n",
                    (**pckStart).files,
                    strhThousandsULong(szOriginal,
                                       (**pckStart).origsize,
                                       '.'),
                    strhThousandsULong(szCompressed,
                                       (**pckStart).compsize,
                                       '.'),
                    100 - ( ((**pckStart).compsize * 100 )/ (**pckStart).origsize )
                );
        }
        else
        {
            printf("Error: Specified package not found.\n");
            throw(CancelExcpt());
        }
    }
    Arc.close();
}

// mode options
#define MODE_ADD        1
#define MODE_DELETE     2
#define MODE_LIST       3
#define MODE_SCRIPT     4

/*
 *@@ main:
 *
 */

int main(int argc, char *argv[])
{
    // parse parameters on cmd line
    if (argc == 1)
        // no parameters:
        ExplainParams();

    if (argc > 1)
    {
        int i = 0;
        while ((i++ < argc-1) && (fContinue))
        {
            if (argv[i][0] == '-')
            {
                SHORT i2;
                for (i2 = 1; i2 < strlen(argv[i]); i2++)
                {
                    switch (argv[i][i2])
                    {
                        case 'a':
                            if (ulNextMode != 0)
                                if (ulNextMode != MODE_SCRIPT)
                                {
                                    printf("'Add' mode can only be combined with 'Script' mode.\n");
                                    fContinue = FALSE;
                                }
                                else
                                    // script mode before
                                    if (szScriptFile[0] == 0)
                                    {
                                        printf("No script specified for 'Script' mode.");
                                        fContinue = FALSE;
                                    }
                            ulNextMode = MODE_ADD;
                        break;

                        case 'l':
                            if (ulNextMode != 0)
                            {
                                printf("'List' mode cannot be combined with other modes.\n");
                                fContinue = FALSE;
                            }
                            ulNextMode = MODE_LIST;
                        break;

                        case 'd':
                            if (ulNextMode != 0)
                            {
                                printf("'Delete' mode cannot be combined with other modes.\n");
                                fContinue = FALSE;
                            }
                            ulNextMode = MODE_DELETE;
                        break;

                        case 's':
                            if (ulNextMode != 0)
                                if (ulNextMode != MODE_ADD)
                                {
                                    printf("'Script' mode can only be combined with 'Add' mode.\n");
                                    fContinue = FALSE;
                                }
                            ulNextMode = MODE_SCRIPT;
                        break;

                        default:  // unknown parameter
                            ExplainParams();
                        break;
                    }
                }
            }
            else
            {
                // no option ("-"): seems to be file
                if (szFilestem[0] == 0)
                    // first non-option: seems to be archive name
                    strcpy(szFilestem, argv[i]);
                else
                    // archive name already copied:
                    if (ulNextMode == MODE_SCRIPT)
                    {
                        if (szScriptFile[0] == 0)
                            strcpy(szScriptFile, argv[i]);
                        else
                        {
                            printf("Too many arguments after '-s'.");
                            fContinue = FALSE;
                        }
                    }
                    else if (ulNextMode == MODE_ADD)
                    {
                        if (strhIsDecimal(argv[i]))
                        {
                            // argument is decimal: seems to be package number;
                            // save the previous stuff then
                            if (pAddToPackageThis)
                            {
                                listAddToPackages.push_back(pAddToPackageThis);
                            }
                            pAddToPackageThis = new(AddToPackage);

                            LONG lPackage = 0;
                            sscanf(argv[i], "%d", &lPackage);
                            if (lPackage == 0)
                            {
                                printf("Error: You have specified an invalid package.\n");
                                fContinue = FALSE;
                            }
                            pAddToPackageThis->lPackage = lPackage;
                        }
                        else
                        {
                            // non-decimal: seems to be file mask
                            if (!pAddToPackageThis)
                            {
                                printf("Error: You need to specify a package number first.");
                                fContinue = FALSE;
                            }
                            else
                            {
                                pAddToPackageThis->listFileMasks.push_back(strdup(argv[i]));
                            }
                        }
                    } // end if (ulNextMode == MODE_ADD)
                    else if (ulNextMode == MODE_LIST)
                        if (strhIsDecimal(argv[i]))
                            sscanf(argv[i], "%d", &lListPackage);
                        else
                        {
                            printf("Invalid parameter for 'list' mode\n");
                            fContinue = FALSE;
                        }
            }
        }
    }

    if (ulNextMode == 0)
    {
        printf("Error: You have not specified a valid operation mode.\n");
        fContinue = FALSE;
    }

    if (    ( (ulNextMode == MODE_ADD) && (pAddToPackageThis == 0) )
         || ( (ulNextMode == MODE_SCRIPT) && (szScriptFile[0] == 0) )
         || ( (ulNextMode == MODE_LIST) && (szFilestem[0] == 0) )
       )
    {
        printf("Error: Not enough parameters specified.\n");
        fContinue = FALSE;
    }

    if (!fContinue)
        ExplainParams();        // and stop

    if (pAddToPackageThis)
        listAddToPackages.push_back(pAddToPackageThis);

    try
    {
        if (strchr(szFilestem, '.') == NULL)
            sprintf(szArcFile, "%s.wpi", szFilestem);
        else
            strcpy(szArcFile, szFilestem);

        /*
         * MODE_SCRIPT only:
         *
         */
        if (ulNextMode == MODE_SCRIPT)
            ulNextMode = MODE_ADD;

        /*
         * MODE_ADD:
         *
         */

        if (ulNextMode == MODE_ADD)
        {
            AddMode();
        }

        /*
         * MODE_DELETE:
         *
         */

        else if (ulNextMode == MODE_DELETE)
        {
            printf("Delete mode not implemented yet.\n");
            throw(CancelExcpt());
        }

        /*
         * MODE_LIST:
         *
         */

        else if (ulNextMode == MODE_LIST)
        {
            ListMode();
        } // end else if (ulNextMode == MODE_LIST)
    }
    catch (CancelExcpt& X)
    {
        printf("Terminating.\n");
    }
    return 0;
}

