
/*
 * warpin.cpp:
 *      this is the main source file for warpin.exe.
 *
 *      The WarpIn installer source code consists of three
 *      code modules, basically:
 *
 *      1)  The WIArchive class ("back-end"). This one is standard
 *          C++ code which should be completely independent of OS/2.
 *          This is in the ../wiarchive and ../zlib directories.
 *
 *      2)  This file, warpin.cpp, which contains OS/2, but no
 *          PM code. This contains the main() procedure, i.e.
 *          WarpIn's program entry point, which then opens the
 *          archive, parses the HTML-like text in there and
 *          builds lots of data in the WPIGLOBALS structure
 *          below accordingly. There's _no_ PM code in this
 *          file. However, this file calls lots of functions
 *          in gui.cpp for the user interface.
 *
 *          New with Alpha #4 is the "Database" functionality.
 *          If WarpIN is started without an archive as a
 *          parameter, it switches itself to "Database" mode.
 *          This is done in main() (warpin.cpp) also, but there
 *          are lots of helper functions for working on the
 *          global database in the new database.cpp file.
 *          warpin.cpp calls more callbacks in gui.cpp for
 *          "Database" mode now.
 *
 *      3)  Having done that, the gui* functions in gui.cpp
 *          are called for the actual user interface. That is,
 *          if you wish to rewrite the WarpIn interface, say
 *          for a VIO version of WarpIn, you "only" have to
 *          rewrite the gui.cpp code, which gets the "digested"
 *          information from the installation profiles from
 *          this file, warpin.cpp. You thus don't have to
 *          deal with the HTML string parsing.
 *
 *      Note that WarpIn is multithreaded. When installation
 *      actually starts, i.e. on the "Install" page, a second
 *      thread is started, which calls the WIArchive methods
 *      upon the archive to unpack the files. The GUI part must
 *      have a number of callbacks to update the display on the
 *      install page accordingly, while installation is in
 *      progress. If you wish to rewrite the gui.cpp part, you
 *      must be aware of which function gets called by which
 *      thread, or everything will crash pretty soon.
 *
 *      This file Copyright (C) 1998-99 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdlib.h>
#include <direct.h>
#include <string.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

#include <ctype.h>

// #include's from XFolder /HELPERS directory
#include "dosh.h"
#include "stringh.h"
#include "threads.h"

#include "warpin.h"

#include "except.h"

/*******************************************************************
 *                                                                 *
 *  Global variables                                               *
 *                                                                 *
 ******************************************************************/

// this is the large structure of variables which
// are shared with gui.cpp, declared in warpin.h.
WPIGLOBALS      WpiGlobals;

CHAR            szMessageFile[CCHMAXPATH];

void EnterArchiveMode(PSZ pszArchiveFile);

/*******************************************************************
 *                                                                 *
 *  Helper functions                                               *
 *                                                                 *
 ******************************************************************/

/*
 *@@ wpiGetMessage:
 *      like DosGetMessage, but automatically finds the
 *      (NLS) WarpIN message file.
 */

APIRET wpiGetMessage(PCHAR *pTable,     // in: array[ulTable] of PSZs with
                                        // substitutes for "%1" etc. in
                                        // the msg string
                     ULONG ulTable,     // in: size of that array
                     PSZ pszBuf,        // out: buffer to hold message string
                     ULONG cbBuf,       // in: sizeof(*pszBuf)
                     ULONG ulMsgNumber) // in: msg number to retrieve
{
    ULONG ul;
    APIRET arc;
    arc = DosGetMessage(pTable, ulTable, pszBuf, cbBuf, ulMsgNumber,
                szMessageFile, &ul);
    *(pszBuf + ul - 2) = 0;
    return (arc);
}

/*
 *@@ wpiParseError:
 *      displays an error and stops.
 */

VOID wpiParseError(ULONG ulOfs, PSZ pszInfo, PSZ pszExtra)
{
    CHAR szText[1000];
    PSZ pszText = szText;

    if (ulOfs) {
        // count lines up to offset
        ULONG ulLine = 1;
        PCSZ p = WpiGlobals.pszProfile;
        while (p < (WpiGlobals.pszProfile + ulOfs))
        {
            if (*p == '\n')
                ulLine++;
            p++;
        }
        pszText += sprintf(szText,
                    "WarpIn encountered an error in the installation"
                    " profile (line %d):\n",
                    ulLine);
    }

    sprintf(pszText, pszInfo,
                pszExtra);
    guiMessageBox("WarpIn: Error in Profile",
                  szText,
                  MB_ICONEXCLAMATION | MB_OK);

    // clean up on the way out
    throw(CancelExcpt());
}

/*
 *@@ wpiGetBlock:
 *      wrapper function for strhGetBlock, which
 *      can also display an error message.
 *
 *      strhGetBlock returns 1 if pszTag was not
 *      found and 2 if the opening, but no closing
 *      tag was found.
 */

BOOL wpiGetBlock(PSZ *ppszSearchIn,
              ULONG ulOffsetOfSearch,       // in: offset to add for error message (can be zero)
              PSZ pszTag,
              PSZ *ppszBlock,
              PSZ *ppszAttribs,
              PULONG pulOfsBeginTag,
              PULONG pulOfsBeginBlock)
{
    PSZ pszBeginTag = *ppszSearchIn;

    ULONG ulrc  = strhGetBlock(ppszSearchIn,
                               pszTag,
                               ppszBlock,
                               ppszAttribs,
                               pulOfsBeginTag,
                               pulOfsBeginBlock);

    if (ulrc == 2)
    {
        wpiParseError(ulOffsetOfSearch + (pszBeginTag - *ppszSearchIn),
                      "Missing closing tag for \"<%s>\" tag.",
                      pszTag);
            // this terminates WarpIN
    }

    return (ulrc == 0);     // TRUE if tag was found
}

/*
 *@@ wpiResolveMacros:
 *      this resolves the macros that many attributes support.
 *
 *      Specify the address of the PSZ to be resolved in ppszString.
 *      If any macros are resolved, *ppszString is free()'d and
 *      reset to a newly allocated string with the resolved macros.
 *
 *      This returns the number of macros resolved or 0 if no macros
 *      were found. In that case, ppszString is not changed.
 *
 *      The following macros can be resolved:
 *
 *      --  "?:\":      "?" is replaced with the OS/2 boot drive letter.
 *      --  "$(xxx)" (with "xxx" being a decimal number):
 *                      "xxx" is replaced with the target path of the package
 *                      with the index "xxx".
 *      --  "$(env)" (with "env" being an environment variable):
 *                      "env" is replaced with the value of an environment variable.
 */

ULONG wpiResolveMacros(PSZ *ppszString)
{
    ULONG ulrc = 0;

    PSZ     pMacroBegin = NULL;

    while (pMacroBegin = strstr(*ppszString, "$("))
    {
        // macro found:
        PSZ     pEndOfMacro = strchr(pMacroBegin, ')');
        if (pEndOfMacro)
        {
            PSZ     pszToReplace = strhSubstr(pMacroBegin,
                                              pEndOfMacro+1); // include ')'
            PSZ     pszInBrackets = strhSubstr(pMacroBegin + 2,
                                               pEndOfMacro);  // exclude ')'
            // now check the macro name if it's decimal
            if (strhIsDecimal(pszInBrackets))
            {
                // OK, we appear to have a package index:
                ULONG   ulIndex = 0;
                sscanf(pszInBrackets, "%d", &ulIndex);
                // find that package in WpiGlobals
                list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                             PckEnd = WpiGlobals.PackageInfoList.end();
                for (; PckStart != PckEnd; PckStart++)
                {
                    PackageInfo* ppiThis = *PckStart;
                    if (ppiThis->PackHeader.number == ulIndex)
                    {
                        // replace macro with target path
                        strhReplace2(ppszString,
                                     pszToReplace,              // search string
                                     ppiThis->szTargetPath);    // replacement string

                        ulrc++;     // replacement count

                        break;  // for PckStart
                    }
                }
            }
            else
            {
                // not all decimal:
                // attempt to resolve the environment variable
                PSZ pszValue = getenv(pszInBrackets);
                if (!pszValue)
                    // prompt
                    pszValue = guiEnterVariable(pszInBrackets, NULL);

                // replace macro with target path
                strhReplace2(ppszString,
                             pszToReplace,              // search string
                             pszValue);                 // replacement string
            }

            free(pszToReplace);
            free(pszInBrackets);
        } // end if (pEndOfMacro)
        else
            break;

        // OK, search next macro;
        // since the macro search string has been replaced
        // with real text, we can safely search the whole
        // buffer again
        // (and we must, because pszCurrent points to an
        // all new buffer)

    } // end while (pMacro = strstr(*ppszString, "$(");)

    return (ulrc);
}

/*
 *@@ wpiCTime2FTime:
 *      useful helper function for converting
 *      the C runtime "time_t" format to the
 *      OS/2 FDATE and FTIME format.
 */

void wpiCTime2FTime(time_t *pCTime,     // in: C runtime time_t format
                    FDATE *pfDate,      // out: OS/2 FDATE format
                    FTIME *pfTime)      // out: OS/2 FTIME format
{
    struct tm* pTimeBuf = localtime(pCTime);
    pfDate->year      = (pTimeBuf->tm_year + 1900) - 1980;
    pfDate->month     = (pTimeBuf->tm_mon + 1);
    pfDate->day       = (pTimeBuf->tm_mday);
    pfTime->hours     = (pTimeBuf->tm_hour);
    pfTime->minutes   = (pTimeBuf->tm_min);
    pfTime->twosecs   = (pTimeBuf->tm_sec) / 2;
}

/*
 *@@ wpiCompareFileDates:
 *      this function compares the last write date
 *      as found in PFILESTATUS3 (which should have
 *      been filled using DosQueryPathInfo on an
 *      existing file) to the last write date in
 *      the specified WIFileHeader.
 *
 *      The header file's last write date will be
 *      written in the two specified buffers in
 *      OS/2 date/time format. These buffers _must_
 *      be specified (or this will crash).
 *
 *      Returns:
 *          // < 0:  existing file is newer
 *          // == 0: existing file is same write date
 *          // > 0:  existing file is older
 */

int wpiCompareFileDates(WIFileHeader* pHeader,      // in: file header to compare
                        FILESTATUS3 *pfs3,          // in: actual file to compare
                        FDATE *pfdateLastWrite,     // out: last-write date of pHeader in OS/2 format
                        FTIME *pftimeLastWrite)     // out: last-write time of pHeader in OS/2 format
{
    // convert file header's time information
    wpiCTime2FTime(&pHeader->lastwrite,
                   pfdateLastWrite,
                   pftimeLastWrite);

    // and compare to the actual file's information
    static const PSZ pszFormatTimestamp = "%4u%02u%02u%02u%02u%02u%";

    CHAR szHeaderStamp[30],
         szExistingStamp[30];

    // create timestamps
    sprintf(szHeaderStamp,
            pszFormatTimestamp,
            pfdateLastWrite->year + 1980,
            pfdateLastWrite->month,
            pfdateLastWrite->day,
            pftimeLastWrite->hours,
            pftimeLastWrite->minutes,
            pftimeLastWrite->twosecs * 2);
    sprintf(szExistingStamp,
            pszFormatTimestamp,
            pfs3->fdateLastWrite.year + 1980,
            pfs3->fdateLastWrite.month,
            pfs3->fdateLastWrite.day,
            pfs3->ftimeLastWrite.hours,
            pfs3->ftimeLastWrite.minutes,
            pfs3->ftimeLastWrite.twosecs * 2);

    return (strcmp(szHeaderStamp, szExistingStamp));
}

/*******************************************************************
 *                                                                 *
 *  PART 1                                                         *
 *  ------                                                         *
 *  The following code is all running on thread 1 (the main PM     *
 *  thread). This includes main(), which will open the archive,    *
 *  analyze the install script, build a list of dialog pages etc.  *
 *                                                                 *
 ******************************************************************/

/*******************************************************************
 *                                                                 *
 *  Program entry point                                            *
 *                                                                 *
 ******************************************************************/

/*
 *@@ wpiLoadDatabase:
 *      this gets called from main() to load
 *      the database into the list at WpiGlobals.DBPckList.
 *      This returns the no. of packages found.
 */

ULONG wpiLoadDatabase(VOID)
{
    ULONG               ulPckCount = 0;
    PSZ                 pszPckID = datEnumPackages(NULL);
    PackageInfo*        pPckInfo;

    while (pszPckID)
    {
        // any packages are installed:
        // create a new package info from database
        pPckInfo = new PackageInfo(WpiGlobals.hiniDatabase, pszPckID);

        if (pPckInfo)
            WpiGlobals.DBPckList.push_back(pPckInfo);

        PSZ pszPckIDNext = datEnumPackages(pszPckID);

        free(pszPckID);
        pszPckID = pszPckIDNext;
        ulPckCount++;
    }

    return (ulPckCount);
}

/*
 *@@ WarpInExitList:
 *      registered with DosExitList.
 */

VOID APIENTRY WarpInExitList(VOID)
{
    // kill Install thread if it's running
    if (thrQueryID(WpiGlobals.ptiInstallThread))
        thrKill(&WpiGlobals.ptiInstallThread);
    // go to next exit list routine
    DosExitList(EXLST_EXIT, (PFNEXITLIST)NULL);
    return;
}

/*
 *@@ main:
 *      Program entry point.
 *      This initializes the GUI front-end and then starts
 *      parsing the install script to build lots of internal
 *      data in the WPIGLOBALS structure, which can then be
 *      used without hassle in the GUI front-end.
 *
 *      If no archive is given on the command line, we start
 *      WarpIN in "Database" mode instead.
 */

int main(int argc, char *argv[])
{
    // open generic try block for exceptions
    // which are not caught in the subroutines;
    // this especially applies to CancelExcpt
    try
    {
        // compose the name of the .msg file; we get the
        // full name of the WarpIn executable in argv[0]
        strcpy(szMessageFile, argv[0]);
        PSZ p = strrchr(szMessageFile, '\\');
        strcpy(p+1, "warpin.msg");

        // compose the name of the global database
        strcpy(WpiGlobals.szWarpINFile, argv[0]);

        // initialize the PM environment (this is
        // needed for the profile functions)
        if (!(WpiGlobals.habThread1 = WinInitialize(0)))
            return FALSE;

        // open the WARPIN.INI file
        CHAR szINIFile[2*CCHMAXPATH];
        strcpy(szINIFile, WpiGlobals.szWarpINFile);
        p = strrchr(szINIFile, '\\');
        strcpy(p+1, "warpin.ini");
        WpiGlobals.hiniWarpIN = PrfOpenProfile(WpiGlobals.habThread1, szINIFile);

        // initialize the interface
        if (!guiInitialize())
            return (1);

        // initialize the database
        if (!datInitialize(&WpiGlobals))
            throw(CancelExcpt());

        // register our exit list
        DosExitList(EXLST_ADD
                        | 0x00002A00,         /* Invocation order is 42 (0x2A) */
                        (PFNEXITLIST)WarpInExitList);

        // load the database (for both Archive and Database mode)
        wpiLoadDatabase();

        // now check arguments
        if (argc > 1)
        {
            /*
             *  "Archive" mode:
             *      an archive was given on the command line.
             */

            EnterArchiveMode(argv[1]);

        } // end if (argc > 1)
        else
        {
            /*
             *  "Database" mode:
             *      if no archive given on the command line.
             */

            if (WpiGlobals.DBPckList.empty())
                // no packages found (first run or
                // database not found):
                guiProductInfo(HWND_DESKTOP);
            else
                guiBeginDatabaseMode();
        }
    }
    catch (EnvironmentExcept& X)
    {
        // thrown by GetEnvironment
        PSZ     apsz[2] = { X.pszValue, X.pszCommand };
        guiMessageBoxMsg(HWND_DESKTOP,
                         102,   // error
                         (PCHAR*)&apsz, 2,
                         127,
                         MB_ICONEXCLAMATION | MB_OK | MB_MOVEABLE);

    }
    catch (ErrorMessageExcpt& X)
    {
        guiMessageBox("WarpIN: Error",
                      X.pszMessage,
                      MB_ICONEXCLAMATION | MB_OK);
    }
    catch(CancelExcpt&)
    {       // do nothing, this is for exiting only;
    }
    catch (...)
    {
        guiMessageBox("WarpIN: Error",
                      "Unexpected C++ exception caught.",
                      MB_ICONEXCLAMATION | MB_OK);
    }

    // clean up
    PrfCloseProfile(WpiGlobals.hiniWarpIN);
    guiCleanupBeforeExit();
    WinTerminate(WpiGlobals.habThread1);

    return (0);         // no error
}

/*******************************************************************
 *                                                                 *
 *  Archive mode: script parsing                                   *
 *                                                                 *
 ******************************************************************/

/*
 *@@ ParseConfigSys:
 *      this gets called by ParsePackages to parse
 *      the CONFIGSYS attributes of a given PCK tag
 *      and create a list of ConfigSys instances (config.h)
 *      in pConfigSysList of the specified PackageInfo.
 *
 *      Returns the number of ConfigSys instances created (0 if none found).
 *      This terminates WarpIN upon errors, giving the user
 *      a meaningful error message.
 */

LONG ParseConfigSys(PSZ pszPckAttribs,      // in: attributes block of PCK tag
                    ULONG ulAttrOfs,        // in: offset of pszPckAttribs in script (for errors)
                    PackageInfo* ppi)       // in/out: where to create the ConfigSys instances
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    PSZ         pszSearchIn = pszPckAttribs;

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = strhGetTextAttr(pszSearchIn,
                                              "CONFIGSYS", &ulOfs);
        if (!pszAttrThis)
            // no more attribs found:
                break;  // while

        // else: pszAttr now has the whole attribute

        // we now create an instance of CfgSysManip (cfgsys.h),
        // whose constructor is capable of parsing this attribute

        try {
            CfgSysManip* pCfgSysManip = new CfgSysManip(pszAttrThis);
                            // this may throw a ConfigExcpt, caught below
            ppi->listCfgSysAttrsAttrs.push_back(pCfgSysManip);
            lrc++;

            // append this attribute string to the log string
            ppi->logCfgSysAttrs.Append(pszAttrThis);

            /* if (ppi->pszConfigSys == NULL)
            {
                // first run:
                ppi->cbConfigSys = strlen(pszAttrThis) + 1;
                                                // including null byte
                ppi->pszConfigSys = (PSZ)malloc(ppi->cbConfigSys);
                strcpy(ppi->pszConfigSys, pszAttrThis);
            }
            else
            {
                // subsequent runs: append after null byte
                ULONG   cbNew = strlen(pszAttrThis) + 1;
                PSZ     pszTemp = (PSZ)malloc(ppi->cbConfigSys + cbNew);
                // copy old buffer
                memcpy(pszTemp, ppi->pszConfigSys, ppi->cbConfigSys);
                // copy new attribs behind last null byte
                strcpy(pszTemp + ppi->cbConfigSys,
                       pszAttrThis);
                // set new buffer
                free(ppi->pszConfigSys);
                ppi->pszConfigSys = pszTemp;
                ppi->cbConfigSys += cbNew;
            } */
        }
        catch (ConfigExcpt& X)
        {
            // thrown by CfgSysManip constructor:
            PSZ pszError = NULL;
            CHAR        szError[1000] = "unknown";

            switch (X.iErrorCode)
            {
                case CFGEXCPT_DOUBLEREPLACEMODE:
                    sprintf(szError,
                            "Replace mode specified twice (\"%s\")",
                            X.pszSubstring);
                    break;
                case CFGEXCPT_INVALIDVERTICAL:
                    sprintf(szError,
                            "Invalid modifier combination; vertical modifier not allowed (\"%s\")",
                            X.pszSubstring);
                    break;
                case CFGEXCPT_INVALIDSEARCH:
                    sprintf(szError,
                            "Invalid search string specified (\"%s\")",
                            X.pszSubstring);
                    break;
                case CFGEXCPT_NOTIMPLEMENTED:
                    sprintf(szError,
                            "Keyword not implemented yet (\"%s\")",
                            X.pszSubstring);
            }

            wpiParseError(ulAttrOfs,
                          "Error parsing CONFIGSYS attribute of PCK tag:\n%s",
                          szError);
                // this exits
        }
        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return (lrc);
}

/*
 *@@ ParseWPSClasses:
 *      this gets called by ParsePackages to parse
 *      the REGISTERCLASS and REPLACECLASS attributes of a given PCK tag
 *      and create two lists of RegisterClass and ReplaceClass instances (config.h)
 *      in listRegisterClassAttrs/listReplaceClass of the specified PackageInfo.
 *
 *      Returns the total number of instances created (0 if none found).
 *      This terminates WarpIN upon errors, giving the user
 *      a meaningful error message.
 */

LONG ParseWPSClasses(PSZ pszPckAttribs,  // in: attributes block of PCK tag
                          ULONG ulAttrOfs,    // in: offset of pszPckAttribs in script (for errors)
                          PackageInfo* ppi)   // in/out: where to create the RegisterClass instances
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;

    // step 1: get the REGISTERCLASS attributes
    PSZ         pszSearchIn = pszPckAttribs;

    while (lrc != -1)   // no error?
    {
        try {
            PSZ     pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                  "REGISTERCLASS", &ulOfs);
            if (!pszAttrThis)
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "classname|classmodule"
            // (without quotes)

            RegisterClass* pRegisterClass = new RegisterClass(pszAttrThis);
                            // this may throw a ConfigExcpt, caught below
            ppi->listRegisterClassAttrs.push_back(pRegisterClass);
            lrc++;

            // append this attribute string to the log string
            ppi->logRegisterClassAttrsAttrs.Append(pszAttrThis);

            free(pszAttrThis);

            // advance search pointer for next search
            pszSearchIn += ulOfs;
        }
        catch (ConfigExcpt& X)
        {
            // thrown by CreateWPSObject constructor
            PSZ pszError = NULL;
            CHAR        szError[1000] = "unknown";

            switch (X.iErrorCode)
            {
                case REGEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid class specification; must be \"classname|classdll\" (is: \"%s\")",
                            X.pszSubstring);
                break;
            }
            wpiParseError(ulAttrOfs,
                          "Error parsing REGISTERCLASS attribute of PCK tag:\n%s",
                          szError);
                // this exits
        }
    }

    // step 2: get the REPLACECLASS attributes

    // reset the search pointer
    pszSearchIn = pszPckAttribs;

    while (lrc != -1)   // no error?
    {
        try {
            PSZ     pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                  "REPLACECLASS", &ulOfs);
            if (!pszAttrThis)
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "oldclassname|newclassname"
            // (without quotes)

            ReplaceClass* pReplaceClass = new ReplaceClass(pszAttrThis);
                            // this may throw a ConfigExcpt, caught below
            ppi->listReplaceClass.push_back(pReplaceClass);
            lrc++;

            // append this attribute string to the log string
            ppi->logReplaceClassAttrs.Append(pszAttrThis);

            free(pszAttrThis);

            // advance search pointer for next search
            pszSearchIn += ulOfs;
        }
        catch (ConfigExcpt& X)
        {
            // thrown by CreateWPSObject constructor
            PSZ pszError = NULL;
            CHAR        szError[1000] = "unknown";

            switch (X.iErrorCode)
            {
                case REGEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid class specification; must be \"classname|classdll\" (is: \"%s\")",
                            X.pszSubstring);
                break;
            }
            wpiParseError(ulAttrOfs,
                          "Error parsing REPLACECLASS attribute of PCK tag:\n%s",
                          szError);
                // this exits
        }
    }

    return (lrc);
}

/*
 *@@ ParseWPSObjects:
 *      this gets called by ParsePackages to parse
 *      the CREATEOBJECT attributes of a given PCK tag
 *      and create a list of CreateWPSObject instances (config.h)
 *      in listWPSObjectAttrs of the specified PackageInfo.
 *
 *      Returns the number of CreateWPSObject instances created (0 if none found).
 *      This terminates WarpIN upon errors, giving the user
 *      a meaningful error message.
 */

LONG ParseWPSObjects(PSZ pszPckAttribs,  // in: attributes block of PCK tag
                     ULONG ulAttrOfs,    // in: offset of pszPckAttribs in script (for errors)
                     PackageInfo* ppi)   // in/out: where to create the CreateWPSObject instances
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    PSZ         pszSearchIn = pszPckAttribs;

    while (lrc != -1)   // no error?
    {
        try {
            PSZ     pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                  "CREATEOBJECT", &ulOfs);
            if (!pszAttrThis)
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "[REPLACE] classname|title|location[|config]"
            // (without quotes)

            CreateWPSObject* pCreateWPSObject = new CreateWPSObject(pszAttrThis);
                            // this may throw a ConfigExcpt, caught below
            ppi->listWPSObjectAttrs.push_back(pCreateWPSObject);
            lrc++;

            // append this attribute string to the log string
            ppi->logWPSObjectAttrsAttrs.Append(pszAttrThis);

            free(pszAttrThis);

            // advance search pointer for next search
            pszSearchIn += ulOfs;
        }
        catch (ConfigExcpt& X)
        {
            // thrown by CreateWPSObject constructor (ParseCreateObject)
            PSZ pszError = NULL;
            CHAR        szError[1000] = "unknown";

            switch (X.iErrorCode)
            {
                case WPOEXCPT_NOLOCATION:
                    sprintf(szError,
                            "No location specified (\"%s\")",
                            X.pszSubstring);
                    break;
                case WPOEXCPT_NOTITLE:
                    sprintf(szError,
                            "No title specified (\"%s\")",
                            X.pszSubstring);
                    break;
                case WPOEXCPT_NOCLASS:
                    sprintf(szError,
                            "No class specified (\"%s\")",
                            X.pszSubstring);
                    break;
            }
            wpiParseError(ulAttrOfs,
                          "Error parsing CREATEOBJECT attribute of PCK tag:\n%s",
                          szError);
                // this exits
        }
    }

    return (lrc);
}

/*
 *@@ ParsePackages:
 *      this gets called by EnterArchiveMode() to set up
 *      the list of package infos. To do this, we analyze the
 *      PCK tags in the HEAD block
 *      of the installation profile.
 *
 *      This routine possibly calls itself recursively
 *      if GROUP tags are found.
 *
 *      It is initially called with pszBlock containing
 *      the HEAD block and will then call
 *      itself with a new block if GROUP are found
 *      therein.
 */

ULONG ParsePackages(PSZ pszBlock,              // in: HEAD block to process (first call)
                                               // or GROUP block (recursive calls)
                    PackageInfo* ppiGroup,     // in: group package to append
                                               // new packages to; initially NULL
                    ULONG ulHeadBlockOfs)      // in: offset of HEAD block (for error messages)
{
    ULONG   ulrc = 0;
    PSZ     pszSearch = pszBlock,
            pszPackage,
            pszGroup;

    // while we have packages to go
    while (pszPackage = strhistr(pszSearch, "<PCK"))
    {
        // check if we have a <GROUP> in this block
        pszGroup = strhistr(pszSearch, "<GROUP");
        if (    // <GROUP> found?
                (pszGroup)
                // <GROUP> before next <PCK>?
             && (pszGroup < pszPackage)
           )
        {
            // if so: extract the text between <GROUP> tags
            // and recurse!
            PSZ     pszGroupBlock,
                    pszGroupAttribs;
            ULONG   ulGroupTagOfs = 0,
                    ulGroupBlockOfs = 0;

            // extract new block
            wpiGetBlock(&pszSearch,
                        ulHeadBlockOfs,
                        "GROUP",
                        &pszGroupBlock,
                        &pszGroupAttribs,
                        &ulGroupTagOfs, &ulGroupBlockOfs);

            // get group attributes
            PackageInfo* ppi = new PackageInfo;

            ppi->pszTitle = strhGetTextAttr(pszGroupAttribs, "TITLE", NULL);
            if (!ppi->pszTitle)
                wpiParseError(ulHeadBlockOfs + ulGroupTagOfs,
                        "No \"TITLE\" attribute given for \"GROUP\" tag.",
                        NULL);

            // mark as group
            ppi->ulType = PCK_GROUP;
            ppi->ppiGroup = ppiGroup; // ulGroup = ulGroupCount;
            ppi->ulSelection = 0;

            // add this to the package info list
            // (it's a group, but it's in the list anyways)
            WpiGlobals.PackageInfoList.push_back(ppi);

            // recurse with the group buffer
            ParsePackages(pszGroupBlock,
                            ppi,
                            ulHeadBlockOfs + ulGroupBlockOfs);

            /* free(pszGroupBlock);
            free(pszGroupAttribs); */
        }
        else
        {
            // no group, but package next:
            LONG    lPckIndex = 0;
            // extract text between <PCK> and </PCK> tags
            PSZ     pszInfo = 0,
                    pszPckAttribs = 0;
            ULONG   ulPckTagOfs = 0,
                    ulPckBlockOfs = 0;
            if (!wpiGetBlock(&pszSearch,
                             ulHeadBlockOfs,
                             "PCK",
                             &pszInfo,
                             &pszPckAttribs,
                             &ulPckTagOfs, &ulPckBlockOfs))
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                        "You have specified no information within the PCK block. ", NULL);

            // so now we have:
            //      -- pszInfo: the "..." stuff between the <PCK>...</PCK> tags
            //      -- pszPckAttribs: the "..." stuff after the <PCK ...> tag

            // "INDEX="
            if (!strhGetNumAttribValue(pszPckAttribs, "INDEX", &lPckIndex))
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                              "A <PCK> tag is lacking the required INDEX attribute.",
                              0);

            // look up the package in the archive
            WIPackHeader* pPackHeader = NULL;
            list<WIPackHeader*>::iterator start = WpiGlobals.pPackageList->begin(),
                                          end = WpiGlobals.pPackageList->end();
            for (; start != end; start++)
                if ((**start).number == lPckIndex)
                {
                    pPackHeader = *start;
                    break;
                }

            if (!pPackHeader)
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                        "The index specified with the <PCK> tag is invalid. No "
                        "corresponding package found in the archive.",
                        0);

            PackageInfo* ppi = new PackageInfo;
                    // this now has a constructor

            // mark as package
            ppi->ulType = PCK_PCK;

            ppi->ppiGroup = ppiGroup;
            memcpy(&ppi->PackHeader, pPackHeader, sizeof(*pPackHeader));

            ppi->pszTitle = strhGetTextAttr(pszPckAttribs, "TITLE", NULL);
            if (!ppi->pszTitle)
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                              "TITLE attribute missing for package "
                              "\"%d\".", (PSZ)lPckIndex);

            ppi->pszInfo = pszInfo;

            // "TARGET="
            PSZ pszTargetPath;
            if (    (pszTargetPath = strhGetTextAttr(pszPckAttribs, "TARGET", NULL))
                    == NULL)
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                              "TARGET attribute missing for package "
                              "\"%s\".", ppi->pszTitle);
            strcpy(ppi->szTargetPath, pszTargetPath);
            // free(pszTargetPath);

            // "PACKAGEID="
            if (    (ppi->pszID = strhGetTextAttr(pszPckAttribs, "PACKAGEID", NULL))
                    == NULL)
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                              "Package ID missing for package "
                              "\"%s\".", ppi->pszTitle);

            // check ID; it must contain exactly four backslashes
            if (strhCount(ppi->pszID, '\\') != 4)
                wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                              "Package ID for package "
                              "\"%s\" has invalid syntax. ",
                              ppi->pszTitle);

            // "REQUIRES="; this is optional
            ULONG   ulOfs = 0;
            ppi->pszRequiresID = strhGetTextAttr(pszPckAttribs, "REQUIRES", &ulOfs);
            if (ppi->pszRequiresID)
                if (strhCount(ppi->pszRequiresID, '\\') != 4)
                    wpiParseError(ulHeadBlockOfs + ulPckTagOfs,
                                  "\"REQUIRES\" ID for package "
                                  "\"%s\" has invalid syntax. ",
                                  ppi->pszTitle);

            // CONFIGSYS attributes (errors are handled in ParseConfigSys)
            if ((ParseConfigSys(pszPckAttribs,
                                ulHeadBlockOfs + ulPckTagOfs,
                                ppi))
                    > 0)
                // something found: set global flag
                WpiGlobals.ulConfigureFlags |= CONFIG_CONFIGSYS;

            // "REGISTERCLASS=" attributes (errors are handled in ParseWPSClasses)
            if ((ParseWPSClasses(pszPckAttribs,
                                      ulHeadBlockOfs + ulPckTagOfs,
                                      ppi))
                    > 0)
                // something found: set global flag
                WpiGlobals.ulConfigureFlags |= CONFIG_WPSCLASSES;

            // "CREATEOBJECT=" attributes (errors are handled in ParseWPSObjects)
            if (ParseWPSObjects(pszPckAttribs,
                                ulHeadBlockOfs + ulPckTagOfs,
                                ppi)
                    > 0)
                // something found: add global flag
                WpiGlobals.ulConfigureFlags |= CONFIG_WPSOBJECTS;

            // "EXECUTE="; this is optional
            ppi->pszExecute = strhGetTextAttr(pszPckAttribs,
                                              "EXECUTE", &ulOfs);

            // convert package size to KByte
            ppi->ulSize = (pPackHeader->origsize + 512) / 1024;

            ppi->ulSelection = (strhFindAttribValue(pszPckAttribs, "SELECT") != 0);
            ppi->ulPathType = 0;
            if (strhFindAttribValue(pszPckAttribs, "BASE") != 0)
                ppi->ulPathType |= PATH_BASE;
            if (strhFindAttribValue(pszPckAttribs, "FIXED") == 0)
                ppi->ulPathType |= PATH_VARIABLE;
            ppi->fNoDeselect = (strhFindAttribValue(pszPckAttribs, "NODESELECT") != 0);

            // add this to the package info list
            WpiGlobals.PackageInfoList.push_back(ppi);

            // free(pszPckAttribs);

            ulrc++;
        }
    }

    return (ulrc);
}

/*
 *@@ ParsePages:
 *      this gets called by main() at the beginning
 *      to set up a list in WPIGLOBALS containing all
 *      the WPIPAGEINFOs. To do this, we analyze the
 *      <PAGE> blocks in the <BODY>...</BODY> block
 *      of the installation profile.
 *      This routine only gets called with pszBlock
 *      containing the <BODY>...</BODY> block.
 *      It does not call itself recursively.
 */

ULONG ParsePages(PSZ pszBlock,               // in: block to process
                 ULONG ulBodyBlockOfs)       // in: for error messages
{
    // start with page one
    LONG    lCurrentPage = 1;

    do {
        PSZ     pszSearch = WpiGlobals.pszProfile,
                        // start search at the beginning
                pszPageBlock,
                        // this will contain the <PAGE>...</PAGE> block
                pszPageAttrs;

        ULONG   ulPageTagOfs = 0,
                ulPageBlockOfs = 0;

        // find the page we're looking for (lCurrentPage)
        do {
            // get the next <PAGE> block; this will
            // advance pszSearch to after the block
            if (wpiGetBlock(&pszSearch,
                            ulBodyBlockOfs,
                            "PAGE",
                            &pszPageBlock,
                            &pszPageAttrs,
                            &ulPageTagOfs, &ulPageBlockOfs))
            {
                LONG lPageFound = 0;
                // another <PAGE> block found:
                // query page index
                if (!(strhGetNumAttribValue(pszPageAttrs, "INDEX", &lPageFound)))
                {
                    wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                            "The <PAGE> tag contains no INDEX attribute. ", NULL);
                }

                if (lPageFound == lCurrentPage)
                {
                    // it's the page we're looking for:
                    // get out of here
                    break;
                } else {
                    /* free(pszPageBlock);
                    free(pszPageAttrs); */
                }
            } else
                pszPageBlock = NULL;
        } while (pszPageBlock);

        if (pszPageBlock == NULL)
            // page not found: probably the
            // last page; exit the outer do loop
            break;

        // else page found: create a new WPIPAGEINFO structure
        PWPIPAGEINFO pPageInfoNew = new WPIPAGEINFO;
                    // this now has a constructor

        pPageInfoNew->lPageIndex = lCurrentPage;

        // determine page type
        PSZ pszTypeAttr;
        if (!(pszTypeAttr = strhGetTextAttr(pszPageAttrs, "TYPE", NULL)))
        {
            wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                    "The PAGE tag contains no TYPE attribute. ", NULL);
        }
        else if (stricmp(pszTypeAttr, "TEXT") == 0)
        {
            pPageInfoNew->lPageMode = MODE_TEXT;
        }
        else if (stricmp(pszTypeAttr, "README") == 0)
        {
            pPageInfoNew->lPageMode = MODE_README;
        }
        else if (stricmp(pszTypeAttr, "CONTAINER") == 0)
        {
            pPageInfoNew->lPageMode = MODE_CONTAINER;
        }
        else if (stricmp(pszTypeAttr, "CONFIGURE") == 0)
        {
            pPageInfoNew->lPageMode = MODE_CONFIGURE;
        }
        /* else if (stricmp(pszTypeAttr, "INSTALL") == 0)
        {
            pPageInfoNew->lPageMode = MODE_INSTALL;
        } */
        else {
            wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                    "The PAGE tag contains an invalid TYPE attribute (\"%s\"). ",
                            pszTypeAttr);
        }
        free(pszTypeAttr);

        // page intro text
        PSZ pszPageBlockTemp = pszPageBlock;
        PSZ pszTextBody;
        if (!(wpiGetBlock(&pszPageBlockTemp, ulBodyBlockOfs + ulPageBlockOfs, "TEXT",
                &pszTextBody, 0, 0, 0)))
        {
            wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                        "Every page must contain exactly one \"<TEXT ...>\" tag. "
                        "Page %d does not.",
                        (PSZ)lCurrentPage);
        }
        pPageInfoNew->pszInfoText = pszTextBody;

        // get the buttons
        PSZ     pszButtonTitle = NULL,
                pszButtonAttrs = NULL;
        ULONG   ulButtonTagOfs = 0;

        LONG lTarget = -1;

        pszButtonTitle = NULL;

        if (wpiGetBlock(&pszPageBlock,
                        ulBodyBlockOfs + ulPageBlockOfs,
                        "NEXTBUTTON",
                        &pszButtonTitle,
                        &pszButtonAttrs,
                        &ulButtonTagOfs, 0))
        {
            // query button target page
            if (!(strhGetNumAttribValue(pszButtonAttrs, "TARGET", &lTarget)))
                lTarget = -1;

            pPageInfoNew->lNextButton = lTarget;
            if (pszButtonTitle)
                pPageInfoNew->pszNextButtonTitle = pszButtonTitle;
            else
                wpiParseError(ulBodyBlockOfs + ulPageBlockOfs + ulButtonTagOfs,
                        "The NEXTBUTTON tag has no title. ", NULL);
        }
        else
            wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                        "The <NEXTBUTTON>...</NEXTBUTTON> block for page %d "
                        "is missing.",
                        (PSZ)lCurrentPage);

        if (pPageInfoNew->lPageMode == MODE_README)
        {
            pszPageBlockTemp = pszPageBlock;
            PSZ     pszReadmeBody = 0;
            if (!(wpiGetBlock(&pszPageBlockTemp, ulBodyBlockOfs + ulPageBlockOfs, "README",
                    &pszReadmeBody, 0, 0, 0)))
            {
                wpiParseError(ulBodyBlockOfs + ulPageTagOfs,
                            "You have specified that page %d should be a README page, but "
                            "the <README>...</README> block is missing.",
                            (PSZ)lCurrentPage);
            }

            pPageInfoNew->pszReadmeText = pszReadmeBody;
        }

        // alright, done with this page info;
        // store the page info in the list
        WpiGlobals.PageInfoList.push_back(pPageInfoNew);
        // go for next
        lCurrentPage++;

        /* free(pszPageBlock);
        free(pszPageAttrs); */

    } while (TRUE);
        // loop forever; there's a break in the loop

    return (lCurrentPage);
}

/*
 *@@ wpiVerifyInstallStatus:
 *      this gets called from main() to create
 *      a list of packages in the archive (checking
 *      WpiGlobals.PackageInfoList) which are
 *      already installed.
 *
 *      The list of packages which are already
 *      installed is created in WpiGlobals.AlreadyInstalledList.
 */

VOID wpiVerifyInstallStatus(VOID)
{
    // if we've been here before, clean up
    WpiGlobals.AlreadyInstalledList.clear();

    list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                 PckEnd = WpiGlobals.PackageInfoList.end();
    for (; PckStart != PckEnd; PckStart++)
    {

       (**PckStart).ulInstallStatus = INSTALLED_NO;

        PPACKAGEID pPckIDThis = datSplitPackageID((**PckStart).pszID);
        if (pPckIDThis)
        {
            PSZ pszInstalledID = datFindPackage(pPckIDThis->pszAuthor,
                                                pPckIDThis->pszApplication,
                                                pPckIDThis->pszPackage);
            if (pszInstalledID)
            {
                // package found:
                PPACKAGEID pPckIDInstalled = datSplitPackageID(pszInstalledID);
                CHAR    szVersionThis[30],
                        szVersionInstalled[30];
                sprintf(szVersionThis, "%010d.%010d",
                        pPckIDThis->ulVersionMajor,
                        pPckIDThis->ulVersionMinor);
                sprintf(szVersionInstalled, "%010d.%010d",
                        (**PckStart).ulInstalledVersionMajor,
                        (**PckStart).ulInstalledVersionMinor);
                int icomp = strcmp(szVersionInstalled, szVersionThis);
                if (icomp > 0)
                {
                    (**PckStart).ulInstallStatus = INSTALLED_NEWER;
                    // deselect package per default, we don't want
                    // to overwrite with outdated version
                    (**PckStart).Select(FALSE, WpiGlobals.PackageInfoList);
                }
                else if (icomp < 0)
                {
                    (**PckStart).ulInstallStatus = INSTALLED_OLDER;
                    // select package per default, since the installed
                    // one is outdated
                    (**PckStart).Select(TRUE, WpiGlobals.PackageInfoList);
                }
                else    // 0:
                {
                    (**PckStart).ulInstallStatus = INSTALLED_SAME;
                    (**PckStart).Select(TRUE, WpiGlobals.PackageInfoList);
                }

                // store this in list of already installed
                // packages
                WpiGlobals.AlreadyInstalledList.push_back(*PckStart);

                free(pszInstalledID);
                delete pPckIDInstalled;
            } // end if (pszInstalledID)

            delete pPckIDThis;
        } // end if (pPckIDThis)
    } // end for (; PckStart != PckEnd; PckStart++)
}

/*
 *@@ GetEnvironment:
 *      this gets called from main() after everything
 *      has been set up and right before we'd normally
 *      go into "Archive" mode.
 *
 *      This will override various settings according
 *      to the currently defined environment variables.
 *
 *      This throws an EnvironmentExcept if invalid
 *      values are found.
 */

VOID GetEnvironment(VOID) throw(EnvironmentExcept)
{
    PSZ     pszCommand = 0,
            pszValue = 0;

    // SET WARPIN_DISPLAYPAGES={YES|NO}

    // SET WARPIN_INSTALLACTION={REPAIR|ADDREMOVE|DEINSTALL}
    // the default at this time is either PROMPT if packages
    // have been installed already or ADDREMOVE if this archive
    // is all new
    pszCommand = "WARPIN_INSTALLACTION";
    if (pszValue = getenv(pszCommand))
    {
        if (stricmp(pszValue, "REPAIR") == 0)
            WpiGlobals.ulInstallAction = ACTION_REPAIR;
        else if (stricmp(pszValue, "ADDREMOVE") == 0)
            WpiGlobals.ulInstallAction = ACTION_ADDREMOVE;
        else if (stricmp(pszValue, "DEINSTALL") == 0)
            WpiGlobals.ulInstallAction = ACTION_DEINSTALL;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));
    }

    // modify packages
    list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                 PckEnd = WpiGlobals.PackageInfoList.end();
    for (; PckStart != PckEnd; PckStart++)
    {
        PackageInfo* ppi = *PckStart;
        CHAR szVarname[100];
        // SET WARPIN_SELECTPACKAGEx={YES|NO}
        sprintf(szVarname, "WARPIN_SELECTPACKAGE%d", ppi->PackHeader.number);
        pszValue = getenv(szVarname);
        if (pszValue)
            if (stricmp(pszValue, "YES") == 0)
                ppi->Select(TRUE, WpiGlobals.PackageInfoList);
            else if (stricmp(pszValue, "NO") == 0)
                ppi->Select(FALSE, WpiGlobals.PackageInfoList);
            else
                // invalid value:
                throw(EnvironmentExcept(szVarname, pszValue));

        // SET WARPIN_TARGETPATHx=path
        sprintf(szVarname, "WARPIN_TARGETPATH%d", ppi->PackHeader.number);
        pszValue = getenv(szVarname);
        if (pszValue)
            if (doshIsValidFileName(pszValue))
                strcpy(ppi->szTargetPath, pszValue);
            else
                // invalid value:
                throw(EnvironmentExcept(szVarname, pszValue));
    }

    // SET WARPIN_UPDATECONFIGSYS={YES|NO}
    pszCommand = "WARPIN_UPDATECONFIGSYS";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "YES") == 0)
            WpiGlobals.ulDoConfiguration |= CONFIG_CONFIGSYS;
        else if (stricmp(pszValue, "NO") == 0)
            WpiGlobals.ulDoConfiguration &= ~CONFIG_CONFIGSYS;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));

    // SET WARPIN_INSTALLWPSCLASSES={YES|NO}
    pszCommand = "WARPIN_INSTALLWPSCLASSES";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "YES") == 0)
            WpiGlobals.ulDoConfiguration |= CONFIG_WPSCLASSES;
        else if (stricmp(pszValue, "NO") == 0)
            WpiGlobals.ulDoConfiguration &= ~CONFIG_WPSCLASSES;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));

    // SET WARPIN_CREATEWPSOBJECTS={YES|NO}
    pszCommand = "WARPIN_CREATEWPSOBJECTS";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "YES") == 0)
            WpiGlobals.ulDoConfiguration |= CONFIG_WPSOBJECTS;
        else if (stricmp(pszValue, "NO") == 0)
            WpiGlobals.ulDoConfiguration &= ~CONFIG_WPSOBJECTS;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));

    // SET WARPIN_IFSAMEDATE={PROMPT|SKIP|OVERWRITE}
    pszCommand = "WARPIN_IFSAMEDATE";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "PROMPT") == 0)
            WpiGlobals.ulIfSameDate = FE_PROMPT;
        else if (stricmp(pszValue, "SKIP") == 0)
            WpiGlobals.ulIfSameDate = FE_SKIP;
        else if (stricmp(pszValue, "OVERWRITE") == 0)
            WpiGlobals.ulIfSameDate = FE_OVERWRITE;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));

    // SET WARPIN_IFEXISTINGOLDER={PROMPT|SKIP|OVERWRITE}
    pszCommand = "WARPIN_IFEXISTINGOLDER";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "PROMPT") == 0)
            WpiGlobals.ulIfExistingOlder = FE_PROMPT;
        else if (stricmp(pszValue, "SKIP") == 0)
            WpiGlobals.ulIfExistingOlder = FE_SKIP;
        else if (stricmp(pszValue, "OVERWRITE") == 0)
            WpiGlobals.ulIfExistingOlder = FE_OVERWRITE;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));

    // SET WARPIN_IFEXISTINGNEWER={PROMPT|SKIP|OVERWRITE}
    pszCommand = "WARPIN_IFEXISTINGNEWER";
    if (pszValue = getenv(pszCommand))
        if (stricmp(pszValue, "PROMPT") == 0)
            WpiGlobals.ulIfExistingNewer = FE_PROMPT;
        else if (stricmp(pszValue, "SKIP") == 0)
            WpiGlobals.ulIfExistingNewer = FE_SKIP;
        else if (stricmp(pszValue, "OVERWRITE") == 0)
            WpiGlobals.ulIfExistingNewer = FE_OVERWRITE;
        else
            // invalid value:
            throw(EnvironmentExcept(pszCommand, pszValue));
}

/*
 *@@ EnterArchiveMode:
 *      this gets called from main() when WarpIN was
 *      started with an archive as a parameter.
 */

void EnterArchiveMode(PSZ pszArchiveFile)
{
    // install the LOUD exception handler (except.h);
    // all the string parsing might crash...
    TRY_LOUD(excpt1)
    {
        /*
         *  "Archive" mode:
         *      archive given on command line
         */

        // open the archive in param 1 (WIArchive back-end)
        if (WpiGlobals.Arc.open(pszArchiveFile) != 0)
        {
            strcpy(WpiGlobals.szArcFilename, pszArchiveFile);

            WpiGlobals.pArcHeader = WpiGlobals.Arc.get_arc_header ();

            WpiGlobals.pPackageList = WpiGlobals.Arc.get_pack_list();

            // get the installation profile from the archive
            PCSZ pcszProfileTemp = WpiGlobals.Arc.get_script();
            if (pcszProfileTemp)
                WpiGlobals.pszProfile = strdup(pcszProfileTemp);

            if (WpiGlobals.pszProfile == NULL)
            {
                guiDisplayErrorMessage(102,
                                       103,  // "no profile"
                                       MB_ICONEXCLAMATION | MB_OK);
                throw(CancelExcpt());
            }

            // read in the packages from the profile;
            // first, extract the <HEAD>...</HEAD> part
            PSZ     pszProfile2 = WpiGlobals.pszProfile;
            PSZ     pszHeadBlock, pszHeadAttrs;
            ULONG   ulHeadTagOfs = 0,
                    ulHeadBlockOfs = 0;
            if (!wpiGetBlock(&pszProfile2, 0, "HEAD",
                             &pszHeadBlock, &pszHeadAttrs,
                             &ulHeadTagOfs, &ulHeadBlockOfs))
            {
                guiDisplayErrorMessage(102,
                                       104,  // "no <HEAD>"
                                       MB_ICONEXCLAMATION | MB_OK);
                throw(CancelExcpt());
            }

            // check whether a <MSG>...</MSG> block
            // exists in the HEAD block; if so, display
            // message and stop
            PSZ pszHeadBlock2 = pszHeadBlock,
                pszMessage;
            if (wpiGetBlock(&pszHeadBlock2, 0, "MSG",
                            &pszMessage, 0, 0, 0))
            {
                CHAR    szMessage[200];
                wpiGetMessage(NULL, 0,
                              szMessage, sizeof(szMessage),
                              128);
                guiMessageBox(szMessage,
                              pszMessage,
                              MB_ICONEXCLAMATION | MB_OK);
                throw(CancelExcpt());
            }

            // else: set up the list of PackageInfos
            ParsePackages(pszHeadBlock,
                          NULL,             // for first call (this might recurse)
                          ulHeadBlockOfs);

            // this has set up WpiGlobals.ulConfigureFlags;
            // now copy ulConfigureFlags into ulDoConfiguration
            // so that per default all possible configuration is
            // enabled (this will be overridden by environment vars)
            WpiGlobals.ulDoConfiguration = WpiGlobals.ulConfigureFlags;

            // we now have the list of PackageInfos in
            // WpiGlobals.PackageInfoList;
            // now query the database for whether any
            // of these packages are already installed
            wpiVerifyInstallStatus();
                // this updates WPIGLOBALS
                // and selects packages which are already installed and
                //  a)  outdated
                //  b)  same version

            // then, extract the <BODY>...</BODY> part
            pszProfile2 = WpiGlobals.pszProfile;
            PSZ     pszBodyBlock, pszBodyAttrs;
            ULONG   ulBodyTagOfs = 0,
                    ulBodyBlockOfs = 0;
            if (!wpiGetBlock(&pszProfile2, 0, "BODY",
                             &pszBodyBlock, &pszBodyAttrs,
                             &ulBodyTagOfs, &ulBodyBlockOfs))
            {
                guiDisplayErrorMessage(102,
                                       105, // "no <BODY>"
                                       MB_ICONEXCLAMATION | MB_OK);
                throw(CancelExcpt());
            }

            // this will set up the list of WPIPAGEINFOs
            ParsePages(pszBodyBlock, ulBodyBlockOfs);

            // check for whether any packages are already
            // installed
            if (!WpiGlobals.AlreadyInstalledList.empty())
                WpiGlobals.ulInstallAction = ACTION_PROMPT;
                // this can be overridden with the environment

            // now parse all the environment variables to
            // override certain settings
            GetEnvironment();

            if (WpiGlobals.ulInstallAction == ACTION_PROMPT)
                WpiGlobals.ulInstallAction = guiAlreadyInstalled();
                        // this may throw a CancelExcpt

            BOOL    fComplete = FALSE;

            switch (WpiGlobals.ulInstallAction)
            {
                case ACTION_REPAIR:     // repair
                    // wpiVerifyInstallStatus has selected the
                    // packages which are already installed, so just
                    // go for installation now
                    if (guiStartInstall())
                    {
                        // TRUE means successfull installation:
                        // store all packages in the database
                        list<PackageInfo*>::iterator
                                PckStart = WpiGlobals.PackageInfoList.begin(),
                                PckEnd = WpiGlobals.PackageInfoList.end();
                        for (; PckStart != PckEnd; PckStart++)
                            if (!(**PckStart).IsGroup())
                                if ((**PckStart).Selection())
                                    (**PckStart).Store(WpiGlobals.hiniDatabase);

                        fComplete = TRUE;
                    }
                    break;

                case ACTION_ADDREMOVE:  // add/remove packages; default
                    // *** go!
                    // this does not return until installation
                    // is completely finished or aborted
                    if (guiBeginAddRemove())
                    {
                        // TRUE means start installation:
                        if (guiStartInstall())
                        {
                            // TRUE means successfull installation:
                            // store all packages in the database
                            list<PackageInfo*>::iterator
                                    PckStart = WpiGlobals.PackageInfoList.begin(),
                                    PckEnd = WpiGlobals.PackageInfoList.end();
                            for (; PckStart != PckEnd; PckStart++)
                                if (!(**PckStart).IsGroup())
                                    if ((**PckStart).Selection())
                                        (**PckStart).Store(WpiGlobals.hiniDatabase);

                            fComplete = TRUE;
                        }
                    }
                    break;

                case ACTION_DEINSTALL: { // deinstall:
                    // 1) for all packages in the archive which
                    //    are also in the database, create a new
                    //    PackageInfo (the items in
                    //    WpiGlobals.AlreadyInstalledList point to
                    //    the packages found in the archive)
                    list<PackageInfo*>::iterator
                            PckStart = WpiGlobals.AlreadyInstalledList.begin(),
                            PckEnd = WpiGlobals.AlreadyInstalledList.end();
                    list<PackageInfo*> DeinstallList;
                    for (; PckStart != PckEnd; PckStart++)
                    {
                        // create from database
                        PackageInfo* pPckInfo = new PackageInfo(WpiGlobals.hiniDatabase,
                                                                (**PckStart).pszID);
                        DeinstallList.push_back(pPckInfo);
                    }

                    LONG lrc = guiConfirmDeinstall(HWND_DESKTOP,
                                        &DeinstallList);
                    if (lrc >= 0)
                    {
                        // OK, deinstall:
                        guiDeinstallArchive(&DeinstallList, lrc);
                        fComplete = TRUE;
                    }

                    DeinstallList.clear();
                break; }
            }

            if (fComplete)
            {
                // done: check what we've done and display an according message
                CHAR    szDone[2000];
                wpiGetMessage(NULL, 0,
                              szDone, sizeof(szDone),
                              143      // installation has completed
                              );
                strcat(szDone, " ");
                ULONG   cbDone = strlen(szDone);
                if (WpiGlobals.ulConfigurationDone & CONFIG_WPSOBJECTS)
                {
                    // created objects:
                    wpiGetMessage(NULL, 0,
                                  szDone + cbDone,
                                  sizeof(szDone) - cbDone,
                                  144);  // "objects created"
                    strcat(szDone, " ");
                    cbDone = strlen(szDone);
                }
                if ((WpiGlobals.ulConfigurationDone & (CONFIG_WPSCLASSES | CONFIG_CONFIGSYS))
                            == CONFIG_CONFIGSYS)
                {
                    wpiGetMessage(NULL, 0,
                                  szDone + cbDone,
                                  sizeof(szDone) - cbDone,
                                  145); // "CONFIG.SYS changed",
                }
                else if ((WpiGlobals.ulConfigurationDone & (CONFIG_WPSCLASSES | CONFIG_CONFIGSYS))
                            == CONFIG_WPSCLASSES)
                {
                    wpiGetMessage(NULL, 0,
                                  szDone + cbDone,
                                  sizeof(szDone) - cbDone,
                                  146);  // "WPS class list changed",
                }
                else if ((WpiGlobals.ulConfigurationDone & (CONFIG_WPSCLASSES | CONFIG_CONFIGSYS))
                            == (CONFIG_WPSCLASSES | CONFIG_CONFIGSYS))
                {
                    wpiGetMessage(NULL, 0,
                                  szDone + cbDone,
                                  sizeof(szDone) - cbDone,
                                  147);  // "both CONFIG.SYS and WPS class list changed",
                }

                guiMessageBox("WarpIN",
                              szDone,
                              MB_INFORMATION | MB_OK);
            }
        } else
            guiDisplayErrorMessage(102,
                                   106,  // "error opening archive"
                                   MB_ICONEXCLAMATION | MB_OK);

    }
    CATCH(excpt1)
    {
        // exception occured:
        // kill Install thread if it's running
        if (thrQueryID(WpiGlobals.ptiInstallThread))
            thrKill(&WpiGlobals.ptiInstallThread);

        guiDisplayErrorMessage(102,
                               107,         // "exception in main thread"
                               MB_ICONEXCLAMATION | MB_OK);
    } END_CATCH;
}

/*******************************************************************
 *                                                                 *
 *  PART 2                                                         *
 *  ------                                                         *
 *  The following code is started by part 1 when installation      *
 *  is actually about to begin, i.e. the packages shall be         *
 *  extracted from the archive. All this is running on a separate  *
 *  thread, the Install thread.                                    *
 *                                                                 *
 ******************************************************************/

ULONG           ulCurrentFile = 0,
                ulFilesTotal = 0,
                ulCurrentPackage = 0,
                ulPackagesTotal = 0;
double          dTotalBytesToDo = 0,
                dBytesDone = 0,
                dBytesThisPackage = 0,
                dBytesOfLastFile = 0;
char            szLastFile[300] = "";

// current package being unpacked
PackageInfo*    pPckInfoCurrent = NULL;

/*
 *@@ WICallback:
 *      this is the callback function which is
 *      given to the WIArchive class. We analyze
 *      the parameters and call the PM callbacks
 *      accordingly.
 */

int  WICallback(short mode,         // in: CBM_* flags def'd in wiarchive.h
                short s,            // in: percentage of current file
                WIFileHeader* pwifh) // in: file header of current file
{
    int rc = CBRC_ABORT;

    switch (mode)
    {
        /*
         * CBM_NEXTFILE:
         *      this comes just before the WIArchive class attempts to open
         *      a new output file for writing. "param" is always 0, "pwifh"
         *      points to the file header which is about to be opened.
         *      This allows the front-end to do two things:
         *      a)   update the "current file" display by calling
         *           guiUpdateFile();
         *      b)   check for whether that file exists already:
         *           -- if not, we return CBRC_PROCEED;
         *           -- if it does exist, we compare the existing date
         *              with the one in the archive and call guiFileExists()
         *              if WpiGlobals says we should prompt. In any case,
         *              we need to return CBRC_PROCEED (overwrite) or
         *              CBRC_SKIP.
         */

        case CBM_NEXTFILE:
        {
            ulCurrentFile++;

            // store the new current file for later
            strcpy(szLastFile, pwifh->name);

            // increase the bytes we've done
            dBytesDone += dBytesOfLastFile;
            // remember the size of this file for later,
            // so we can calculate the total percentage
            dBytesOfLastFile = pwifh->origsize;

            // update display
            guiUpdateFile(pwifh, ulCurrentFile, ulFilesTotal);

            rc = CBRC_PROCEED;

            // check for whether the file exists
            FILE *TestFile = fopen (pwifh->name, "rb");
            if (TestFile != NULL)
            {
                // file exists:
                fseek (TestFile, 0, SEEK_END);
                if (ftell (TestFile) != 0)
                {
                    // file exists and has more than 0 bytes:
                    FILEEXISTS fi;

                    // get file name (in current working directory)
                    _getcwd(fi.szFilename, sizeof(fi.szFilename));
                    strcat(fi.szFilename, "\\");
                    strcat(fi.szFilename, pwifh->name);

                    // 1) get file info for file on disk
                    DosQueryPathInfo(fi.szFilename,
                                    FIL_STANDARD,
                                    &fi.fs3Existing, sizeof(fi.fs3Existing));

                    // 2) info for file in archive:
                    // WIFileHeader has two time_t fields which we need
                    // to convert to human-readable format first.
                    fi.iComp = wpiCompareFileDates(pwifh,
                                                   &fi.fs3Existing,
                                                   &fi.fDateHeader,
                                                   &fi.fTimeHeader);
                        // this returns:
                        // < 0:  existing file is newer
                        // == 0: existing file is same write date
                        // > 0:  existing file is older

                    // now check WPIGLOBALS for what we need to do
                    BOOL    fPrompt = FALSE;
                    PULONG  pulQuery = 0;
                    if (fi.iComp < 0)
                        // existing is newer:
                        pulQuery = &WpiGlobals.ulIfExistingNewer;
                    else if (fi.iComp == 0)
                        // same date:
                        pulQuery = &WpiGlobals.ulIfSameDate;
                    else
                        // existing is older:
                        pulQuery = &WpiGlobals.ulIfExistingOlder;

                    switch (*pulQuery)
                    {
                        // 0: prompt
                        // 1: skip
                        // 2: overwrite
                        case 0: fPrompt = TRUE; break;
                        case 1: rc = CBRC_SKIP; break;
                        case 2: rc = CBRC_PROCEED; break;
                    }

                    if (fPrompt)
                    {
                        fi.pwifh = pwifh;
                        rc = guiFileExists(&fi);
                                // this returns:
                                //      ERROR_PROCEED    replace this file
                                //      ERROR_SKIP       skip this file
                    }
                }
                fclose(TestFile);
            }

            // store the new current file in the
            // current package's PSZ file list for
            // the global database later
            if (pPckInfoCurrent)
                datAddFileHeader(pPckInfoCurrent, pwifh);

        break; }

        /*
         * CBM_PERCENTAGE:
         *      in this case, "param" contains the percentage of the current
         *      file being processed, and pwifh the current file header.
         *      The return value doesn't matter.
         */

        case CBM_PERCENTAGE:
        {
            // calculate the bytes we've really done so far
            double dBytesDone2 = dBytesDone + ((pwifh->origsize * s) / 100);
            // calculate total percentage
            // SHORT sPercent = (SHORT)((dBytesDone2 / dTotalBytesToDo) * 100);
            // call percentage callback (for progress bar)
            guiUpdateBytes(dBytesDone2, dTotalBytesToDo);
        break; }

        /*
         * CBM_ERR_WRITE:
         *      error writing into output file; this is most probable when the
         *      target disk is full.
         *      "param" then has one of the CBREC_* values, which identify
         *      whether the error is recoverable. Depending on that value,
         *      we may return one of the CBRC_* values.
         */

        case CBM_ERR_WRITE:
        {
            CHAR    szMessage[1000];
            PSZ     apsz = pwifh->name;
            wpiGetMessage(&apsz, 1,      // replacement table
                          szMessage, sizeof(szMessage),
                          129);     // "WarpIn was unable to write the file %1."

            // get the DOS error number; this appears to be
            // specific to VAC++, at least I have found no
            // corresponding thing for EMX/GCC.
            int iError = _doserrno;

            // now check if maybe the write error was due
            // to a full disk, because in that case, it's
            // not ERROR_DISK_FULL which is returned, so
            // we better do this
            ULONG   ulDisk, ulMap;
            DosQueryCurrentDisk(&ulDisk, &ulMap);
            if (doshQueryDiskFree(ulDisk) < pwifh->origsize)
                iError = ERROR_DISK_FULL;

            rc = guiInstallError(szMessage, ECLS_BACKEND, iError,
                            s); // CBREC_* param passed from WIArchive
                // if the user presses "Cancel" there,
                // that function will terminate everthing
        break; }

        /*
         * CBM_ERR_READ:
         *      error reading from input file; maybe the download is corrupt.
         *      "param" then has one of the CBREC_* values below, which identify
         *      whether the error is recoverable. Depending on that value,
         *      we may return one of the CBRC_* values.
         */

        case CBM_ERR_READ:
        {
            CHAR    szMessage[1000];
            PSZ     apsz = pwifh->name;
            wpiGetMessage(&apsz, 1,
                          szMessage, sizeof(szMessage),
                          131);     // "unable to read file %1 from archive..."
            rc = guiInstallError(szMessage, ECLS_BACKEND, _doserrno,
                            s); // CBREC_* param passed from WIArchive
                // if the user presses "Cancel" there,
                // that function will terminate everthing
        break; }

        /*
         * CBM_ERR_ZLIB:
         *      error somewhere in the zlib decompression part of WIArchive;
         *      this usually signifies that the archive is broken. We then
         *      terminate installation, since there's no way to recover.
         *      "param" then contains one of the ZLIBERR_* flags below.
         */

        case CBM_ERR_ZLIB: {
            CHAR szMessage[1000];
            wpiGetMessage(NULL, 0,      // no string replacements
                          szMessage, sizeof(szMessage),
                          132);         // "corrupt archive"
            rc = guiInstallError(szMessage, ECLS_DECOMP, s,
                            CBREC_CANCEL);
                            // and stop!
        break; }

        /*
         * default:
         *      some other error occured. This is probably
         *      just as serious, so we'll stop.
         */

        default: {
            CHAR szMessage[1000];
            wpiGetMessage(NULL, 0,     // no string replacements
                          szMessage, sizeof(szMessage),
                          133);         // "unknown error"
            rc = guiInstallError(szMessage, ECLS_UNKNOWN, s,
                            CBREC_CANCEL);
                            // and stop!
        }
    }

    return (rc);
}

/*
 *@@ wpiInstallThread:
 *      thread function for the Install thread.
 */

void _Optlink wpiInstallThread(PVOID ptiMyself)
{
    // install the LOUD exception handler (except.h)
    TRY_LOUD(excpt1)
    {
        WpiGlobals.Arc.set_callback_func(WICallback);

        // first go thru the list of packages which are to
        // be installed to find out the total bytes
        // that will be installed
        list<PackageInfo*>::iterator PackageStart = WpiGlobals.PackageInfoList.begin(),
                                     PackageEnd = WpiGlobals.PackageInfoList.end();
        for (; PackageStart != PackageEnd; PackageStart++)
            if ((**PackageStart).ulType == PCK_PCK)
                if ((**PackageStart).ulSelection)
                {
                    ulPackagesTotal++;
                    ulFilesTotal += (**PackageStart).PackHeader.files;
                    dTotalBytesToDo += (**PackageStart).PackHeader.origsize;
                }

        CHAR    szStartDir[CCHMAXPATH];
        ULONG   cbStartDir = sizeof(cbStartDir);
        DosQueryCurrentDir(0, szStartDir, &cbStartDir);

        // alright, now let's go
        PackageStart = WpiGlobals.PackageInfoList.begin();
        PackageEnd = WpiGlobals.PackageInfoList.end();
        for (; PackageStart != PackageEnd; PackageStart++)
        {
            // package, not group? We cannot install groups.
            if ((**PackageStart).ulType == PCK_PCK)
            {
                // package selected?
                if ((**PackageStart).ulSelection)
                {
                    // alright, install package.
                    ulCurrentPackage++;
                    dBytesThisPackage = (**PackageStart).PackHeader.origsize;
                    guiUpdatePackage(*PackageStart,
                                     ulCurrentPackage, ulPackagesTotal);

                    // create target directory
                    APIRET arc = NO_ERROR;
                    do {
                        // create path and change dir to it
                        arc = doshCreatePath((**PackageStart).szTargetPath);
                        if (arc == NO_ERROR)
                            arc = doshSetCurrentDir((**PackageStart).szTargetPath);

                        if (arc == NO_ERROR)
                        {
                            // set global variable to current package
                            pPckInfoCurrent = *PackageStart;

                            // unpack this package; this will keep
                            // calling the callback above
                            WpiGlobals.Arc.unpack((**PackageStart).PackHeader.number);
                        }
                        else
                        {
                            CHAR    szMsg[1000];
                            PSZ     apsz = (**PackageStart).szTargetPath;
                            wpiGetMessage(&apsz, 1,
                                          szMsg, sizeof(szMsg),
                                          134);  // The directory \"%s\" could not be created.
                            guiInstallError(szMsg, ECLS_DOS, arc, CBREC_RETRYCANCEL);
                                // if the user presses "Cancel" there,
                                // that will terminate everthing
                        }
                    } while (arc != NO_ERROR);
                } // end if ((**PackageStart).ulSelection)
            } // end if ((**PackageStart).ulType == PCK_PCK)
        } // end for (; PackageStart != PackageEnd; PackageStart++)

        // finally, set percentage to 100
        guiUpdateBytes(dTotalBytesToDo, dTotalBytesToDo);
        DosSleep(300);
        // PM function to clean up
        guiDoneWithInstall();
    }
    CATCH (excpt1)
    {
    /*     // exception occured:
        guiInstallError("An exception occured in the WarpIn install thread. "
                    "A file named \"WPITRAP.LOG\" was created in the root "
                    "directory of your boot drive, which contains more information.",
                    ECLS_EXCEPTION, 0,
                            CBREC_CANCEL); // terminate only
    */
    } END_CATCH;

    thrGoodbye((PTHREADINFO)ptiMyself);
}

/*
 *@@ wpiStartInstallThread:
 *      this gets called from gui.cpp when the
 *      user turns to the "Install" page.
 *
 *      This returns immediately. From now on,
 *      the GUI gets lots of callbacks on the
 *      Install thread.
 */

VOID wpiStartInstallThread(VOID)
{
    thrCreate(&WpiGlobals.ptiInstallThread,
              wpiInstallThread,
              0);
}

/*******************************************************************
 *                                                                 *
 *  PART 3                                                         *
 *  ------                                                         *
 *  This has the configuration stuff: CONFIG.SYS changes,          *
 *  installing WPS classes, etc.                                   *
 *  These functions get called from the GUI front-end (gui.cpp).   *
 *  This should be called from thread 1.                           *
 *                                                                 *
 ******************************************************************/

/*
 *@@ wpiCreateCIDFile:
 *      this creates a CID file according to the current
 *      global and package data. pszFilename must contain
 *      the full pathname of where the CID file should be
 *      created. This will be a plain OS/2 batch (.CMD)
 *      file and should have the .CMD extension.
 *
 *      This gets called from the GUI when the "Create CID file"
 *      button is pressed.
 */

BOOL wpiCreateCIDFile(PSZ pszFilename)
{
    BOOL brc = FALSE;

    FILE* CIDFile = fopen(pszFilename, "w");  // text file for writing, replace;
    if (CIDFile)
    {
        fprintf(CIDFile, "REM WarpIN CID file\n");
        DATETIME DT;
        DosGetDateTime(&DT);
        fprintf(CIDFile, "REM Created %02d/%02d/%04d, %02d:%02d:%02d\n",
            DT.month, DT.day, DT.year,
            DT.hours, DT.minutes, DT.seconds);
        fprintf(CIDFile, "\n");

        fprintf(CIDFile, "\nREM Set the following to NO to automate installation\n");
        fprintf(CIDFile, "SET WARPIN_DISPLAYPAGES=YES\n");

        fprintf(CIDFile, "\nSET WARPIN_INSTALLACTION=ADDREMOVE\n");
        fprintf(CIDFile, "REM                      or REPAIR or DEINSTALL\n");

        fprintf(CIDFile, "\nREM Here come the packages\n");
        list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                     PckEnd = WpiGlobals.PackageInfoList.end();
        for (; PckStart != PckEnd; PckStart++)
        {
            if ((**PckStart).ulType == PCK_PCK)
            {
                fprintf(CIDFile,
                        "SET WARPIN_SELECTPACKAGE%d=%s\n",
                        (**PckStart).PackHeader.number,
                        ((**PckStart).ulSelection == 1)
                              ? "YES"
                              : "NO");
                fprintf(CIDFile,
                        "SET WARPIN_TARGETPATH%d=%s\n",
                        (**PckStart).PackHeader.number,
                        (**PckStart).szTargetPath);
            }
        }

        fprintf(CIDFile, "\nREM Here comes configuration: either YES or NO\n");
        fprintf(CIDFile,
                "SET WARPIN_UPDATECONFIGSYS=%s\n",
                ((WpiGlobals.ulDoConfiguration & CONFIG_CONFIGSYS) != 0)
                      ? "YES"
                      : "NO");
        fprintf(CIDFile,
                "SET WARPIN_INSTALLWPSCLASSES=%s\n",
                ((WpiGlobals.ulDoConfiguration & CONFIG_WPSCLASSES) != 0)
                      ? "YES"
                      : "NO");
        fprintf(CIDFile,
                "SET WARPIN_CREATEWPSOBJECTS=%s\n",
                ((WpiGlobals.ulDoConfiguration & CONFIG_WPSOBJECTS) != 0)
                      ? "YES"
                      : "NO");

        fprintf(CIDFile, "\nREM Set the following to any of PROMPT, SKIP, or OVERWRITE\n");
        fprintf(CIDFile,
                "SET WARPIN_IFSAMEDATE=%s\n",
                (WpiGlobals.ulIfSameDate == FE_PROMPT) ? "PROMPT"
                : (WpiGlobals.ulIfSameDate == FE_SKIP) ? "SKIP"
                : "OVERWRITE");
        fprintf(CIDFile,
                "SET WARPIN_IFEXISTINGOLDER=%s\n",
                (WpiGlobals.ulIfExistingOlder == FE_PROMPT) ? "PROMPT"
                : (WpiGlobals.ulIfExistingOlder == FE_SKIP) ? "SKIP"
                : "OVERWRITE");
        fprintf(CIDFile,
                "SET WARPIN_IFEXISTINGNEWER=%s\n",
                (WpiGlobals.ulIfExistingNewer == FE_PROMPT) ? "PROMPT"
                : (WpiGlobals.ulIfExistingNewer == FE_SKIP) ? "SKIP"
                : "OVERWRITE");

        fclose(CIDFile);
        brc = TRUE;
    }
    return (brc);
}

/*
 *@@ wpiConfigSys:
 *      this changes CONFIG.SYS by going thru the list
 *      of packages in WPIGLOBALS which were installed,
 *      using the ConfigSys class (config.cpp).
 *
 *      This replaces macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      This is designed to be called from gui.cpp after
 *      the Install thread has finished.
 *      Call this first, before wpiWPSClasses and
 *      wpiCreateObjects.
 */

BOOL wpiConfigSys()
{
    BOOL    brc = TRUE;

    if (WpiGlobals.ulDoConfiguration | CONFIG_CONFIGSYS)
    {
        // create instance of ConfigSys; this will load CONFIG.SYS
        ConfigSys* pConfigSys = new ConfigSys;

        // go thru the list of all packages that were installed
        list<PackageInfo*>::iterator PackageStart = WpiGlobals.PackageInfoList.begin(),
                                     PackageEnd = WpiGlobals.PackageInfoList.end();
        for (; PackageStart != PackageEnd; PackageStart++)
        {
            PackageInfo* ppiThis = *PackageStart;
            if (ppiThis->ulType == PCK_PCK)         // package, not group?
                if (ppiThis->ulSelection)           // selected?
                {
                    // now go thru the CONFIG.SYS manipulators in the
                    // PackageInfo and apply them to the ConfigSys object;
                    // there may be none of these
                    list<CfgSysManip*>::iterator ManipFirst = ppiThis->listCfgSysAttrsAttrs.begin(),
                                                 ManipEnd = ppiThis->listCfgSysAttrsAttrs.end();

                    for (; ManipFirst != ManipEnd; ManipFirst++)
                    {
                        CfgSysManip* pManip = *ManipFirst;
                        // resolve macros
                        wpiResolveMacros(&pManip->pszNewLine);
                        // apply to CONFIG.SYS text
                        try {
                            pConfigSys->Manipulate(*pManip,
                                                        // the manipulator
                                                   &ppiThis->logCfgSysDone);
                                                        // CONFIG.SYS logger for database
                                // this may throw ConfigExcpt
                            WpiGlobals.ulConfigurationDone |= CONFIG_CONFIGSYS;
                        }
                        catch(ConfigExcpt&)
                        {
                            guiDisplayErrorMessage(102,
                                                   124, // "error"
                                                   MB_ICONEXCLAMATION | MB_OK);
                            brc = FALSE;
                        }
                    }
                }
        }

        // rewrite CONFIG.SYS
        pConfigSys->WriteBack(TRUE);
        delete pConfigSys;
    } // end if (WpiGlobals.ulDoConfiguration | CONFIG_CONFIGSYS)

    return (brc);
}

/*
 *@@ wpiWPSClasses:
 *      this registers and replaces WPS classes by going thru
 *      the list of packages in WPIGLOBALS which were installed.
 *      Each PackageInfo has a list of RegisterClass instances
 *      and another one of ReplaceClass instances,
 *      which at this point have the necessary information
 *      (set up at initialization by ParseWPSClasses).
 *
 *      This invokes RegisterClass::Register for each
 *      class to be registered, after having replaced macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      This is designed to be called from gui.cpp after
 *      the Install thread has finished.
 *      Call this second, after wpiConfigSys, but before
 *      wpiCreateObjects.
 */

BOOL wpiWPSClasses(VOID)
{
    BOOL    brc = TRUE;

    if (WpiGlobals.ulDoConfiguration | CONFIG_WPSCLASSES)
    {
        // go thru the list of all packages that were installed
        list<PackageInfo*>::iterator PackageStart = WpiGlobals.PackageInfoList.begin(),
                                     PackageEnd = WpiGlobals.PackageInfoList.end();
        for (; PackageStart != PackageEnd; PackageStart++)
        {
            PackageInfo* ppiThis = *PackageStart;
            if (ppiThis->ulType == PCK_PCK)         // package, not group?
                if (ppiThis->ulSelection)           // selected?
                {
                    // now go thru the RegisterClass objects in the list in
                    // PackageInfo and invoke Register() on them;
                    // there may be none of these
                    list<RegisterClass*>::iterator RegStart = ppiThis->listRegisterClassAttrs.begin(),
                                                   RegEnd = ppiThis->listRegisterClassAttrs.end();
                    for (; RegStart != RegEnd; RegStart++)
                    {
                        RegisterClass* pRegThis = *RegStart;

                        // resolve macros
                        wpiResolveMacros(&(pRegThis->pszModule));

                        BOOL    fRetry = FALSE;
                        BOOL    fReplace = FALSE;

                        do {
                            fRetry = FALSE;

                            try
                            {
                                WpiGlobals.ulConfigurationDone |= CONFIG_WPSCLASSES;
                                pRegThis->Register(fReplace,
                                                   &ppiThis->logRegisterClassAttrsDone);
                                                        // classes logger for database
                            }
                            catch(ConfigExcpt& X)
                            {
                                brc = FALSE;
                                CHAR    szError[1000] = "unknown";
                                PSZ     apsz[2];
                                ULONG   ulRplCount = 0;
                                ULONG   ulMsg;
                                ULONG   ulFlags = MB_ICONEXCLAMATION | MB_OK;
                                PSZ     pszSysError = 0;

                                switch(X.iErrorCode)
                                {
                                    case REGEXCPT_QUERYCLASSLIST:
                                        ulMsg = 135; // error querying WPS class list
                                    break;

                                    case REGEXCPT_ALREADYREGISTERED: {
                                        // RegisterClass::Register(FALSE) has determined that the
                                        // class is already registered;
                                        // ConfigExcpt.pszSubstring has the currently registered DLL
                                        apsz[0] = pRegThis->pszClassName;
                                        apsz[1] = X.pszSubstring;
                                        ulRplCount = 2;
                                        ulMsg = 136; // The class "%1" is already registered with the DLL "%2".
                                        ulFlags = MB_ICONQUESTION | MB_YESNO;
                                    break; }

                                    case REGEXCPT_REGISTER: {
                                        // RegisterClass::Register failed;
                                        // ConfigExcpt.iData has the APIRET of winhRegisterClass
                                        pszSysError = doshQuerySysErrorMsg(X.iData);
                                        apsz[0] = pRegThis->pszClassName;
                                        apsz[1] = pszSysError;
                                        ulRplCount = 2;
                                        ulMsg = 137; // The class "%1" could not be registered. The following error was reported:
                                    break; }
                                } // end switch

                                if (guiMessageBoxMsg(HWND_DESKTOP,
                                                     138,  // "WarpIN: Class Registering Error"
                                                     apsz, ulRplCount,
                                                     ulMsg,
                                                     ulFlags)
                                        == MBID_YES)
                                {
                                    fRetry = TRUE;
                                    fReplace = TRUE;
                                }
                                if (pszSysError)
                                    free(pszSysError);
                            }
                        } while (fRetry);
                    }

                    // now go thru the ReplaceClass objects in the list in
                    // PackageInfo and invoke Replace() on them;
                    // there may be none of these
                    list<ReplaceClass*>::iterator ReplStart = ppiThis->listReplaceClass.begin(),
                                                  ReplEnd = ppiThis->listReplaceClass.end();
                    for (; ReplStart != ReplEnd; ReplStart++)
                    {
                        ReplaceClass* pReplThis = *ReplStart;

                        try
                        {
                            WpiGlobals.ulConfigurationDone |= CONFIG_WPSCLASSES;
                            pReplThis->Replace(&ppiThis->plogReplaceClassDone);
                                                    // classes logger for database
                        }
                        catch(ConfigExcpt& X)
                        {
                            brc = FALSE;
                            PSZ     apsz[2];
                            ULONG   ulRplCount = 0,
                                    ulMsg = 142;    // unknown error

                            switch(X.iErrorCode)
                            {
                                case REGEXCPT_REPLACE:
                                                // error querying WPS class list
                                    apsz[0] = pReplThis->pszOldClassName;
                                    apsz[1] = pReplThis->pszNewClassName;
                                    ulRplCount = 2;
                                    ulMsg = 140;    // Error replacing WPS class "%1" with "%2".
                                break;
                            } // end switch

                            guiMessageBoxMsg(HWND_DESKTOP,
                                             139,   // WarpIN: Class Replacement Error
                                             apsz, ulRplCount,
                                             ulMsg,
                                             MB_ICONEXCLAMATION | MB_OK);
                        }
                    }
                } // end if (ppiThis->ulSelection)
        }
    } // end if (WpiGlobals.ulDoConfiguration | CONFIG_WPSCLASSES)

    return (brc);
}

/*
 *@@ wpiCreateObjects:
 *      this creates WPS objects by going thru the list of
 *      packages in WPIGLOBALS which were installed. Each
 *      PackageInfo has a list of CreateWPSObject instances,
 *      which at this point have the necessary information
 *      (set up at initialization by ParseWPSObjects).
 *
 *      This invokes CreateWPSObject::CreateObject for each
 *      object to be created, which in turn stores the
 *      HOBJECT in its instance data, after having replaced macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      This is designed to be called from gui.cpp after
 *      the Install thread has finished.
 *      Call this last, after wpiConfigSys and wpiWPSClasses.
 */

BOOL wpiCreateObjects(VOID)
{
    BOOL    brc = TRUE;

    if (WpiGlobals.ulDoConfiguration | CONFIG_WPSOBJECTS)
    {
        // go thru the list of all packages that were installed
        list<PackageInfo*>::iterator PackageStart = WpiGlobals.PackageInfoList.begin(),
                                     PackageEnd = WpiGlobals.PackageInfoList.end();
        for (; PackageStart != PackageEnd; PackageStart++)
        {
            PackageInfo* ppiThis = *PackageStart;
            if (ppiThis->ulType == PCK_PCK)         // package, not group?
                if (ppiThis->ulSelection)           // selected?
                {
                    // now go thru the CreateWPSObjects objects in the list in
                    // PackageInfo and invoke CreateObject() on them;
                    // there may be none of these
                    list<CreateWPSObject*>::iterator CrObjStart = ppiThis->listWPSObjectAttrs.begin(),
                                               CrObjEnd = ppiThis->listWPSObjectAttrs.end();
                    for (; CrObjStart != CrObjEnd; CrObjStart++)
                    {
                        CreateWPSObject* pcroThis = *CrObjStart;

                        // resolve macros
                        wpiResolveMacros(&(pcroThis->pszSetupString));
                        wpiResolveMacros(&(pcroThis->pszLocation));

                        try
                        {
                            pcroThis->CreateObject(&ppiThis->logWPSObjectAttrsDone);
                                                // WPS objects logger for database
                                    // this may throw ConfigExcpt
                            WpiGlobals.ulConfigurationDone |= CONFIG_WPSOBJECTS;
                        }
                        catch(ConfigExcpt&)
                        {
                            // error occured:
                            PSZ apsz[4] =  {
                                              pcroThis->pszTitle,
                                              pcroThis->pszClassName,
                                              pcroThis->pszLocation,
                                              pcroThis->pszSetupString
                                           };
                            brc = FALSE;
                            guiMessageBoxMsg(HWND_DESKTOP,
                                             102, // "WarpIN: Error"
                                             apsz, 4,
                                             141,   // "WPS object could not be created"
                                             MB_ICONEXCLAMATION | MB_OK);
                        }
                    }
                }
        }
    } // end if (WpiGlobals.ulDoConfiguration | CONFIG_WPSOBJECTS)

    return (brc);
}


