
/*
 * pckinfo.cpp:
 *      this implements the PackageInfo class (declared in pckinfo.h).
 *
 *      Instances of this class are essential to all of WarpIN's operation.
 *      They are created by ParsePackages when WarpIN is opened with an
 *      archive, and by the database functions when WarpIN is started
 *      in "Database" mode. Essentially, without this class, nothing works.
 *
 *      The PackageInfo class has two overloaded constructors:
 *      --  the first one takes no arguments and is used by main() to
 *          create an empty instance, which is later filled according
 *          to the install script;
 *      --  the second one takes a HINI and a five-part package ID to
 *          create a ready-made package info from database information.
 */

/*
 *      This file Copyright (C) 1999 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <string.h>

#include "prfh.h"

#include "wiarchive.h"
#include "errors.h"
#include "config.h"
#include "pckinfo.h"
#include "calbacks.h"

// INI keys in database
#define WPIKEY_FILESLIST        "Files"
#define WPIKEY_TARGETPATH       "TargetPath"
#define WPIKEY_REQUIRES         "Requires"
#define WPIKEY_CONFIGATTRS      "ConfigSysAttrs"
#define WPIKEY_CONFIGDONE       "ConfigSysDone"
#define WPIKEY_REGISTERATTRS    "RegisterAttrs"
#define WPIKEY_REGISTERDONE     "RegisterDone"
#define WPIKEY_REPLACEATTRS     "ReplaceAttrs"
#define WPIKEY_REPLACEDONE      "ReplaceDone"
#define WPIKEY_WPSOBJECTATTRS   "WPSObjectAttrs"
#define WPIKEY_WPSOBJECTDONE    "WPSObjectDone"
#define WPIKEY_EXECUTE          "Execute"
#define WPIKEY_WIPACKHEADER     "WIPackHeader"

/*
 *@@ Init:
 *      this gets called from the constructors to
 *      initialize all fields to zero and create
 *      all lists and other objects which are
 *      mandatory.
 */

void PackageInfo::Init()
{
    ulType = 0;
    ppiGroup = 0;
    pszTitle = 0;
    pszInfo = 0;
    pszID = 0;
    pszRequiresID = 0;
    szTargetPath[0] = 0;
    ulSize = 0;
    ulSelection = 0;
    fNoDeselect = FALSE;
    pszExecute = 0;
    ulPathType = 0;
    pvGuiData = 0;
    memset(&PackHeader, 0, sizeof(PackHeader));
    pszFilesList = 0;
    cbFilesList = 0;
    ulInstallStatus = 0;
    ulInstalledVersionMajor = 0;
    ulInstalledVersionMinor = 0;
    ulStatus = 0;
}

/*
 *@@ PackageInfo:
 *      default constructor for an empty PackageInfo.
 */

PackageInfo::PackageInfo()
{
    Init();
}

/*
 *@@ PackageInfo:
 *      this is the overloaded constructor to
 *      fill the PackageInfo with data from the
 *      global database.
 *
 *      pszPackageID must be in the "package ID" format
 *      (five-part package ID, separated by four backslashes).
 *
 *      This uses "new" to create the PackageInfo.
 *      You should therefore use "delete" to free it.
 *      A copy of pszPackage is created in PackageInfo.pszID
 *      using strdup, so you can free() pszPackage after
 *      calling this.
 *
 *      Upon errors, NULL is returned.
 *
 *      This initializes most fields in the returned
 *      PackageInfo structure, including the WIPackHeader
 *      substructure.
 */

PackageInfo::PackageInfo(HINI hiniDatabase,     // in: profile to load from
                         PSZ pszPackageID)      // in: package ID (== application) to load from
{
    Init();

    // check if file list exists in database
    pszFilesList = prfhQueryProfileData(hiniDatabase,
                                        pszPackageID,
                                        WPIKEY_FILESLIST,
                                        &cbFilesList);
    if (pszFilesList)
    {
        // create file header list

        // store package ID
        pszID = strdup(pszPackageID);

        // store files list
        cbFilesList = cbFilesList;

        // target path
        ULONG   cbTargetPath = sizeof(      szTargetPath);
        PrfQueryProfileData(hiniDatabase,
                            pszPackageID,
                            WPIKEY_TARGETPATH,
                                  szTargetPath,
                            &cbTargetPath);

        pszRequiresID = prfhQueryProfileData(hiniDatabase,
                                             pszPackageID,
                                             WPIKEY_REQUIRES,
                                             NULL);

        logCfgSysAttrs.Load(hiniDatabase,
                              pszPackageID,
                              WPIKEY_CONFIGATTRS);
        logCfgSysDone.Load(hiniDatabase,
                             pszPackageID,
                             WPIKEY_CONFIGDONE);
        // create undo list
        CfgSysManip::AddToUndoList(listUndoCfgSys,  // list to append to
                                   logCfgSysDone);  // logger restored above

        logRegisterClassAttrsAttrs.Load(hiniDatabase,
                                     pszPackageID,
                                     WPIKEY_REGISTERATTRS);
        logRegisterClassAttrsDone.Load(hiniDatabase,
                                    pszPackageID,
                                    WPIKEY_REGISTERDONE);
        // create undo list
        DeregisterClass::AddToUndoList(listUndoRegisterClass,  // list to append to
                                       logRegisterClassAttrsDone);  // logger restored above

        // class replacements
        logReplaceClassAttrs.Load(hiniDatabase,
                                     pszPackageID,
                                     WPIKEY_REPLACEATTRS);
        plogReplaceClassDone.Load(hiniDatabase,
                                    pszPackageID,
                                    WPIKEY_REPLACEDONE);
        // create undo list
        UnreplaceClass::AddToUndoList(listUndoReplaceClass,  // list to append to
                                      plogReplaceClassDone);  // logger restored above

        // WPS objects
        logWPSObjectAttrsAttrs.Load(hiniDatabase,
                                 pszPackageID,
                                 WPIKEY_WPSOBJECTATTRS);
        logWPSObjectAttrsDone.Load(hiniDatabase,
                                pszPackageID,
                                WPIKEY_WPSOBJECTDONE);
        DeleteWPSObject::AddToUndoList(listUndoWPSObject,
                                       logWPSObjectAttrsDone);

        pszExecute = prfhQueryProfileData(hiniDatabase,
                                          pszPackageID,
                                          WPIKEY_EXECUTE,
                                          NULL);

        ULONG cbPackHeader = sizeof(WIPackHeader);
        PrfQueryProfileData(hiniDatabase,
                            pszPackageID,
                            WPIKEY_WIPACKHEADER,
                            &      PackHeader,
                            &cbPackHeader);
    }
}

/*
 *@@ ~PackageInfo:
 *      destructor
 */

PackageInfo::~PackageInfo()
{
    if (pszTitle)
        free(pszTitle);
    if (pszInfo)
        free(pszInfo);
    if (pszID)
        free(pszID);
    if (pszRequiresID)
        free(pszRequiresID);
    if (pszExecute)
        free(pszExecute);
    if (pszFilesList)
        free(pszFilesList);
}

/*
 *@@ IsGroup:
 *      returns TRUE if this package is not
 *      a real package, but a group.
 */

BOOL PackageInfo::IsGroup(VOID)
{
    return (ulType == PCK_GROUP);
}

/*
 *@@ Select:
 *      this selects the package for installation
 *      or deselects it.
 *
 *      The package is _not_ deselected if the fNoDeselect
 *      flag has been set for this package.
 *
 *      This method updates other packages, if
 *      necessary. That is, if this package is a
 *      group, all subpackages are selected/deselected.
 *      If this package is not a group (but a "real"
 *      package), the parent group's status is
 *      updated.
 *
 *      For this to work, this method needs the list
 *      of all packages from WpiGlobals.
 *
 *      This calls guiPackageSelected for each package
 *      whose selection was changed so that the display
 *      can be updated.
 *
 *      This function returns TRUE unless you attempt
 *      to deselect a package for which the fNoDeselect
 *      flag has been set.
 */

BOOL PackageInfo::Select(BOOL fSelect,
                                    // in: new selection state for this package
                         list<PackageInfo*>& PckInfoList)
                                    // in: list of all packages which are to be updated
                                    // (from WpiGlobals)
{
    BOOL    brc = TRUE;
    ULONG   ulOldSelection = ulSelection;

    if (fSelect)
        ulSelection = 1;
    else
        if (!fNoDeselect)
            ulSelection = 0;
        else
            brc = FALSE;

    if (ulOldSelection != ulSelection)
        guiPackageSelected(this);

    return (brc);
}

/*
 *@@ ToggleSelection:
 *      similar to PackageInfo::Select, but this toggles
 *      the selection.
 *
 *      This returns the new selection status.
 */

BOOL PackageInfo::ToggleSelection(list<PackageInfo*>& PckInfoList)
{
    if (ulSelection == 1)
        // selected:
        Select(FALSE, PckInfoList);
    else
        // ulSelect == 0 or 2 (for groups):
        // select
        Select(TRUE, PckInfoList);

    return (ulSelection != 0);
}

/*
 *@@ Selection:
 *      returns the selection status of this package,
 *      which is 0 for "not selected" or 1 for "selected".
 *      If this package is a group, this might also return
 *      2 for "partly selected" (i.e. only some subpackages
 *      selected).
 */

ULONG PackageInfo::Selection(VOID)
{
    return (ulSelection);
}

/*
 *@@ Store:
 *      this stores the package in a profile.
 *
 *      This gets called from main() after installation
 *      was complete and successful to store the package
 *      information in the global database.
 *
 *      We store the whole package in the database then,
 *      using the package ID stored in pszID.
 *      This better be valid.
 *
 *      This returns FALSE upon errors.
 */

bool PackageInfo::Store(HINI hiniDatabase)
{
    BOOL    brc = FALSE;
    PSZ     pszApplication = pszID;

    if (pszApplication)
    {

        // write files list which was set by the
        // Install thread
        brc = PrfWriteProfileData(hiniDatabase,
                                  pszApplication,
                                  WPIKEY_FILESLIST,
                                  pszFilesList,
                                  cbFilesList);

        PrfWriteProfileString(hiniDatabase,
                              pszApplication,
                              WPIKEY_TARGETPATH,
                              szTargetPath);

        PrfWriteProfileString(hiniDatabase,
                              pszApplication,
                              WPIKEY_REQUIRES,
                              pszRequiresID);

        logCfgSysAttrs.Store(hiniDatabase,
                               pszApplication,
                               WPIKEY_CONFIGATTRS);
        logCfgSysDone.Store(hiniDatabase,
                              pszApplication,
                              WPIKEY_CONFIGDONE);

        logRegisterClassAttrsAttrs.Store(hiniDatabase,
                                      pszApplication,
                                      WPIKEY_REGISTERATTRS);
        logRegisterClassAttrsDone.Store(hiniDatabase,
                                     pszApplication,
                                     WPIKEY_REGISTERDONE);

        logReplaceClassAttrs.Store(hiniDatabase,
                                     pszApplication,
                                     WPIKEY_REPLACEATTRS);
        plogReplaceClassDone.Store(hiniDatabase,
                                    pszApplication,
                                    WPIKEY_REPLACEDONE);

        logWPSObjectAttrsAttrs.Store(hiniDatabase,
                                  pszApplication,
                                  WPIKEY_WPSOBJECTATTRS);
        logWPSObjectAttrsDone.Store(hiniDatabase,
                                 pszApplication,
                                 WPIKEY_WPSOBJECTDONE);

        PrfWriteProfileString(hiniDatabase,
                              pszApplication,
                              WPIKEY_EXECUTE,
                              pszExecute);

        PrfWriteProfileData(hiniDatabase,
                            pszApplication,
                            WPIKEY_WIPACKHEADER,
                            &PackHeader,
                            sizeof(PackHeader));
    }

    return (brc);
}


