
/*
 * gui.cpp:
 *      this contains the PM-specific stuff. The functions
 *      in here get called from main() in warpin.cpp.
 *
 *      This is the only PM-specific file. See warpin.cpp
 *      for details about this concept.
 *
 *      Note that all functions which get called from outside
 *      this file (i.e. either warpin.cpp or database.cpp)
 *      have the gui* prefix. These functions are all prototyped
 *      in warpin.h.
 *
 *      This file Copyright (C) 1998-99 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_WINWORKPLACE
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#include <os2.h>

#include <stdlib.h>
// #include <stdio.h>           // included with wiarchive.h
#include <direct.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h
#include <io.h>

// #include's from XFolder /HELPERS directory
#include "dosh.h"
#include "winh.h"
#include "gpih.h"
#include "stringh.h"
#include "threads.h"

#include "warpin.h"
#include "gui.h"

#include "except.h"

#pragma hdrstop

/*******************************************************************
 *                                                                 *
 *  Global variables                                               *
 *                                                                 *
 ******************************************************************/

extern WPIGLOBALS   WpiGlobals;

HMQ                 hmqThread1;

NLSSTRINGS          NLSStrings;

HWND                hwndMainDlg = NULLHANDLE,
                    hwndPageText = NULLHANDLE,
                    hwndPageReadme = NULLHANDLE,
                    hwndPageContainer = NULLHANDLE,
                    hwndPageConfigure = NULLHANDLE,
                    // hwndPageInstall = NULLHANDLE,
                    hwndDriveInfo = NULLHANDLE,
                    hwndCnrOnCnrPage = NULLHANDLE,
                    hwndWarpInStartup = NULLHANDLE;

HWND                hwndInstallStatus = NULLHANDLE;

HPOINTER            hptrSelect = NULLHANDLE,
                    hptrDeselect = NULLHANDLE,
                    hptrSomeselect = NULLHANDLE,
                    hptrError = NULLHANDLE,
                    hptrQMark = NULLHANDLE,
                    hptrOK    = NULLHANDLE,
                    hptrApp   = NULLHANDLE;

// system mini icon size (16 or 20)
ULONG               ulMiniIconSize;

CHAR                szMainWindowTitle[400];

// this flag is set by the "Drive info" window and
// set to FALSE if there's not enough space on a
// drive to hold the installation
BOOL                fPlentyOfSpace = TRUE;

// the following is TRUE after the package record cores
// have been created on the "Container" page
BOOL                fContainerPageReady = FALSE;

PPCKRECORDCORE      preccSelected = NULL;

PWPIPAGEINFO        pCurrentPageInfo = NULL;

int                 iErrorRC = -1;

ULONG               ulInstallTimeBegin = 0,
                    ulSecsElapsed = 0,
                    ulPercent = 0;

// NLS settings from OS2.INI
ULONG               ulDateFormat;
ULONG               ulTimeFormat;
CHAR                szDateSep[10],
                    szTimeSep[10],
                    szThousand[10];

/*
 *@@ GuiSettings:
 *      global structure for GUI settings, which
 *      are stored in WARPIN.INI.
 */

struct
{
    ULONG           ulDatabaseView;  // one of the following:
                       // -- ID_WIMI_DATABASE_TREEVIEW: applications/packages tree
                       // -- ID_WIMI_DATABASE_DETAILSVIEW: packages details
} GuiSettings;

MRESULT EXPENTRY fnwpPages(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
MRESULT EXPENTRY fnwpDriveInfo(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
VOID TurnToPage(LONG lSearchPage, BOOL fStoreThis);
BOOL Configure(VOID);

/*******************************************************************
 *                                                                 *
 *  PM implementation part 1:                                      *
 *      Lotsa functions invoked by the stuff in part 2             *
 *      (that is, the required GUI callbacks).                     *
 *                                                                 *
 ******************************************************************/

/*
 *@@ LoadSettings:
 *      this loads the user settings from WARPIN.INI.
 *      This gets called from guiInitialize.
 */

VOID LoadSettings(VOID)
{
    GuiSettings.ulDatabaseView = ID_WIMI_DATABASE_DETAILSVIEW;

    ULONG cbData = sizeof(GuiSettings);
    PrfQueryProfileData(WpiGlobals.hiniWarpIN, "GUI", "Database",
                        &GuiSettings, &cbData);
    cbData = sizeof(WpiGlobals.ulIfSameDate);
    PrfQueryProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfSameDate",
                        &WpiGlobals.ulIfSameDate, &cbData);
    cbData = sizeof(WpiGlobals.ulIfExistingOlder);
    PrfQueryProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfExistingOlder",
                        &WpiGlobals.ulIfExistingOlder, &cbData);
    cbData = sizeof(WpiGlobals.ulIfExistingNewer);
    PrfQueryProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfExistingNewer",
                        &WpiGlobals.ulIfExistingNewer, &cbData);
}

/*
 *@@ SaveSettings:
 *      this saves the user settings back into WARPIN.INI.
 */

VOID SaveSettings(VOID)
{
    PrfWriteProfileData(WpiGlobals.hiniWarpIN, "GUI", "Database",
                        &GuiSettings, sizeof(GuiSettings));
    PrfWriteProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfSameDate",
                        &WpiGlobals.ulIfSameDate, sizeof(WpiGlobals.ulIfSameDate));
    PrfWriteProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfExistingOlder",
                        &WpiGlobals.ulIfExistingOlder, sizeof(WpiGlobals.ulIfExistingOlder));
    PrfWriteProfileData(WpiGlobals.hiniWarpIN, "Settings", "ulIfExistingNewer",
                        &WpiGlobals.ulIfExistingNewer, sizeof(WpiGlobals.ulIfExistingNewer));
}

/*
 *@@ LoadString:
 *      this loads one NLS string from the resources.
 *      Used during guiInitialize.
 */

VOID LoadString(ULONG ulID, PSZ *ppsz)
{
    CHAR    szBuf[255];
    if (*ppsz)
        free(*ppsz);
    if (WinLoadString(WpiGlobals.habThread1,
                      NULLHANDLE, ulID ,
                      sizeof(szBuf), szBuf))
        *ppsz = strdup(szBuf);
    else
        *ppsz = strdup("Error: String resource not found");
}

/*
 *@@ SetControlsFont:
 *      this sets the font for all the controls of hwndDlg
 *      which have a control ID in the range of usIDMin to
 *      usIDMax. "Unused" IDs (i.e. -1) will also be set.
 *
 *      On Warp 3, we'll use "8.Helv", on Warp 4 "9.WarpSans".
 */

VOID SetControlsFont(HWND hwndDlg, SHORT usIDMin, SHORT usIDMax)
{
    HENUM henum;
    HWND hwndItem;
    CHAR szFont[20];
    ULONG cbFont;

    // set font for all the dialog controls
    if (doshIsWarp4())
        strcpy(szFont, "9.WarpSans");
    else
        strcpy(szFont, "8.Helv");
    cbFont = strlen(szFont)+1;
    henum = WinBeginEnumWindows(hwndDlg);
    while (hwndItem = WinGetNextWindow(henum))
    {
        SHORT sID = WinQueryWindowUShort(hwndItem, QWS_ID);
        if (    (sID == -1)
             || (sID > usIDMin) && (sID < usIDMax)
           )
            WinSetPresParam(hwndItem,
                    PP_FONTNAMESIZE,
                    cbFont,
                    szFont);
    }
    WinEndEnumWindows(henum);
}

/*
 *@@ CreateDBAppRecord:
 *
 */

PDATABASERECORD CreateDBAppRecord(HWND hwndCnrAlloc, PPACKAGEID pPckID)
{
    PDATABASERECORD preccApplication =
            (PDATABASERECORD)winhCnrAllocRecords(hwndCnrAlloc,
                                                 sizeof(DATABASERECORD),
                                                 1);
    if (preccApplication)
    {
        // set package info to NULL;
        // this signifies that this is an application
        preccApplication->pPckInfo = NULL;

        preccApplication->recc.hptrIcon
            = preccApplication->recc.hptrMiniIcon
            = hptrApp;

        // store other fields
        preccApplication->pszAuthor = strdup(pPckID->pszAuthor);
        preccApplication->pszApplication = strdup(pPckID->pszApplication);
    }

    return (preccApplication);
}

/*
 *@@ CreateDBPckRecord:
 *
 */

PDATABASERECORD CreateDBPckRecord(HWND hwndCnrAlloc, PackageInfo* ppiPackage)
{
    PDATABASERECORD preccPck = NULL;

    PPACKAGEID pPckID = datSplitPackageID(ppiPackage->pszID);
    if (pPckID)
    {
        preccPck = (PDATABASERECORD)winhCnrAllocRecords(hwndCnrAlloc,
                                                        sizeof(DATABASERECORD),
                                                        1);
        if (preccPck)
        {
            // store package info in recc
            preccPck->pPckInfo = ppiPackage;
            // and recc in package info
            ppiPackage->pvGuiData = (PRECORDCORE)preccPck;

            preccPck->recc.hptrIcon
                = preccPck->recc.hptrMiniIcon
                = hptrQMark;

            // store other fields
            preccPck->pszAuthor = strdup(pPckID->pszAuthor);
            preccPck->pszApplication = strdup(pPckID->pszApplication);
            preccPck->pszPackageName = strdup(pPckID->pszPackage);
            preccPck->ulVersionMajor = pPckID->ulVersionMajor;
            preccPck->ulVersionMinor = pPckID->ulVersionMinor;
            sprintf(preccPck->szVersion, "%d.%d",
                    pPckID->ulVersionMajor,
                    pPckID->ulVersionMinor);
            preccPck->pszVersion = preccPck->szVersion;
            preccPck->ulFiles = preccPck->pPckInfo->PackHeader.files;
            preccPck->ulTotalSize = preccPck->pPckInfo->PackHeader.origsize;
            preccPck->pszTargetPath = preccPck->pPckInfo->szTargetPath;
        }

        delete pPckID;
    }

    return (preccPck);
}

/*
 *@@ ShowConfirmations:
 *      this shows the "Confirmations" dialog and
 *      updates the flags in WpiGlobals, if necessary.
 */

VOID ShowConfirmations(HWND hwndOwner)
{
    HWND hwndDlg = WinLoadDlg(HWND_DESKTOP, hwndOwner,
                                   WinDefDlgProc,
                                   NULLHANDLE, ID_WID_CONFIRMATIONS,
                                   NULL);
    winhCenterWindow(hwndDlg);
    SetControlsFont(hwndDlg, 0, 2000);
    // set controls
    ULONG idSet = 0;
    switch (WpiGlobals.ulIfExistingOlder)
    {
        case 0:  idSet = ID_WID_EXISTOLDER_PROMPT; break;
        case 1:  idSet = ID_WID_EXISTOLDER_SKIP; break;
        case 2:  idSet = ID_WID_EXISTOLDER_OVERWRITE; break;
    }
    winhSetDlgItemChecked(hwndDlg, idSet, TRUE);
    switch (WpiGlobals.ulIfSameDate)
    {
        case 0:  idSet = ID_WID_EXISTSAME_PROMPT; break;
        case 1:  idSet = ID_WID_EXISTSAME_SKIP; break;
        case 2:  idSet = ID_WID_EXISTSAME_OVERWRITE; break;
    }
    winhSetDlgItemChecked(hwndDlg, idSet, TRUE);
    switch (WpiGlobals.ulIfExistingNewer)
    {
        case 0:  idSet = ID_WID_EXISTNEWER_PROMPT; break;
        case 1:  idSet = ID_WID_EXISTNEWER_SKIP; break;
        case 2:  idSet = ID_WID_EXISTNEWER_OVERWRITE; break;
    }
    winhSetDlgItemChecked(hwndDlg, idSet, TRUE);

    if (WinProcessDlg(hwndDlg) == DID_OK)
    {
        // read controls
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTOLDER_PROMPT))
            WpiGlobals.ulIfExistingOlder = 0;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTOLDER_SKIP))
            WpiGlobals.ulIfExistingOlder = 1;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTOLDER_OVERWRITE))
            WpiGlobals.ulIfExistingOlder = 2;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTSAME_PROMPT))
            WpiGlobals.ulIfSameDate = 0;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTSAME_SKIP))
            WpiGlobals.ulIfSameDate = 1;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTSAME_OVERWRITE))
            WpiGlobals.ulIfSameDate = 2;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTNEWER_PROMPT))
            WpiGlobals.ulIfExistingNewer = 0;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTNEWER_SKIP))
            WpiGlobals.ulIfExistingNewer = 1;
        if (winhIsDlgItemChecked(hwndDlg, ID_WID_EXISTNEWER_OVERWRITE))
            WpiGlobals.ulIfExistingNewer = 2;

        SaveSettings();
    }
    WinDestroyWindow(hwndDlg);
}

/*
 *@@ DisplayMode:
 *      shows/hides controls in the dlg window according
 *      to the mode flag.
 */

VOID DisplayMode(ULONG ulMode)
{
    WinShowWindow(hwndPageText, ulMode == MODE_TEXT);
    WinShowWindow(hwndPageReadme, ulMode == MODE_README);
    WinShowWindow(hwndPageContainer, ulMode == MODE_CONTAINER);
    WinShowWindow(hwndPageConfigure, ulMode == MODE_CONFIGURE);
    // WinShowWindow(hwndPageInstall, ulMode == MODE_INSTALL);
}

/*
 *@@ UpdatePackageDisplay:
 *      this updates the display if package
 *      selections have changed on the "Container"
 *      page.
 */

VOID UpdatePackageDisplay(VOID)
{
    BOOL fShow = FALSE;
    if (preccSelected)
        fShow = (preccSelected->pPckInfo->ulType == PCK_PCK);
    winhShowDlgItem(hwndPageContainer, ID_WIDI_PATHENTRY,
            fShow);
    winhShowDlgItem(hwndPageContainer, ID_WIDI_PATHTEXT,
            fShow);
    if (fShow)
    {
        BOOL fEnable = (    (preccSelected->pPckInfo->ulSelection == 1)
                         && (preccSelected->pPckInfo->ulPathType & PATH_VARIABLE)
                       );
        winhEnableDlgItem(hwndPageContainer, ID_WIDI_PATHENTRY,
                (fEnable));
        winhEnableDlgItem(hwndPageContainer, ID_WIDI_PATHTEXT,
                (fEnable));
    }
}

/*
 *@@ fncbreccSelectAll:
 *      callback for selecting all child records.
 */

VOID EXPENTRY fncbreccSelectAll(PRECORDCORE precc, ULONG ul1, ULONG ul2)
{
    ((PPCKRECORDCORE)precc)->pPckInfo->ulSelection = ul1;
    WinSendMsg(hwndCnrOnCnrPage,
               CM_INVALIDATERECORD,
               (MPARAM)&precc,
               MPFROM2SHORT(1, CMA_ERASE));
}

/*
 *@@ fncbreccUpdateParents:
 *      callback for updating parent records (groups)
 *      when the selection state of a package or group
 *      has changed.
 */

VOID EXPENTRY fncbreccUpdateParents(PRECORDCORE precc, ULONG ul1, ULONG ul2)
{
    BOOL fUnselectedFound = FALSE;
    BOOL fSelectedFound = FALSE;
    ULONG ulSelection = 0;

    if (((PPCKRECORDCORE)precc)->pPckInfo->ulType == PCK_GROUP)
    {
        PPCKRECORDCORE precc2 = (PPCKRECORDCORE)precc;
        do {
            precc2 =
                (PPCKRECORDCORE)WinSendMsg(hwndCnrOnCnrPage,
                        CM_QUERYRECORD,
                        (MPARAM)precc2,
                        MPFROM2SHORT(
                            (precc2 == (PPCKRECORDCORE)precc)
                                ? CMA_FIRSTCHILD : CMA_NEXT,
                            CMA_ITEMORDER)
                        );
            if ((LONG)precc2 == -1)
                precc2 = NULL;
            if (precc2)
                // check if this child is selected
                if (precc2->pPckInfo->ulSelection == 1)
                    fSelectedFound = TRUE;
                else if (precc2->pPckInfo->ulSelection == 0)
                    fUnselectedFound = TRUE;
                else if (precc2->pPckInfo->ulSelection == 0)
                    fUnselectedFound = TRUE;
                else { // 2: some selected
                    fSelectedFound = TRUE;
                    fUnselectedFound = TRUE;
                    break;
                }
        }
        while (precc2);

        if (fUnselectedFound)
            if (fSelectedFound)
                ulSelection = 2;
            else
                ulSelection = 0;
        else
            if (fSelectedFound)
                ulSelection = 1;
            else // no unselected, no selected
                ulSelection = 2;

        if (ulSelection != ((PPCKRECORDCORE)precc)->pPckInfo->ulSelection)
        {
            ((PPCKRECORDCORE)precc)->pPckInfo->ulSelection = ulSelection;
            WinSendMsg(hwndCnrOnCnrPage,
                       CM_INVALIDATERECORD,
                       (MPARAM)&precc,
                       MPFROM2SHORT(1, CMA_ERASE));
        }
    }
}

/*
 *@@ fncbreccUpdatePathBase:
 *      callback for updating target paths when
 *      a "base" target path has been changed.
 */

VOID EXPENTRY fncbreccUpdatePathBase(PRECORDCORE precc, ULONG ulSearchFor, ULONG ulBaseRecc)
{
    if (ulBaseRecc != (ULONG)precc)
    {
        strhReplace(((PPCKRECORDCORE)precc)->pPckInfo->szTargetPath,
                (PSZ)ulSearchFor,
                ((PPCKRECORDCORE)ulBaseRecc)->pPckInfo->szTargetPath);
    }
}

/*
 *@@ ConfirmCancelInstall:
 *      helper func which is used every time
 *      the user tries to cancel installation.
 *      This wants this decision confirmed
 *      and does all the cleanup, if neccessary.
 */

BOOL ConfirmCancelInstall(HWND hwndOwner)
{
    ULONG ulmbrc = MBID_NO;
    BOOL  brc = FALSE;

    if (!WpiGlobals.fInstallationComplete)
    {
        TID     tidInstall;

        // suspend Install thread if it's running;
        // we don't want the unpacking of files to
        // continue while the dialog is being
        // displayed
        if (tidInstall = thrQueryID(WpiGlobals.ptiInstallThread))
            DosSuspendThread(tidInstall);

        ulmbrc = guiMessageBoxMsg(hwndOwner,
                        108, // "Warning"
                        0, 0,
                        109, // "Really cancel?"
                        MB_ICONQUESTION | MB_YESNO | MB_MOVEABLE);

        if (ulmbrc == MBID_YES)
        {
            if (WpiGlobals.ptiInstallThread)
                WpiGlobals.ptiInstallThread->fExit = TRUE;
            WinPostMsg(hwndMainDlg, WM_CLOSE, 0, 0);
            brc = TRUE;
        }

        if (tidInstall)
            DosResumeThread(tidInstall);
    }
    else {
        WinPostMsg(hwndOwner, WM_CLOSE, 0, 0);
        brc = TRUE;
    }

    return (brc);
}

/*
 *@@ fnwpPages:
 *      window proc for the main window. This is
 *      responsible for almost everything that the
 *      front-end can do.
 */

MRESULT EXPENTRY fnwpPages(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      initialize some more controls
         */

        case WM_INITDLG:
        {
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            SetControlsFont(hwndDlg, 0, 500);
        break; }

        /*
         * WPIM_TURNTOPAGE:
         *      this use message gets posted when
         *      we need to switch to another page.
         *      The new page is in (ULONG)mp1;
         *      if ((BOOL)mp2 == TRUE), the new page
         *      will be stored in the "visited" list.
         */

        case WPIM_TURNTOPAGE:
        {
            // set up the dialog controls for the page
            // that we want
            TurnToPage((LONG)mp1, (BOOL)mp2);
        break; }

        /*
         * WM_CONTROL:
         *
         */

        case WM_CONTROL:
        {
            USHORT  usControl = SHORT1FROMMP(mp1);
            USHORT  usNotifyCode = SHORT2FROMMP(mp1);

            switch (usControl)
            {
                /*
                 * ID_WIDI_CNR:
                 *      selection container
                 *      on "Container" page
                 */

                case ID_WIDI_CNR:
                {
                    switch (usNotifyCode)
                    {
                        /*
                         * CN_ENTER:
                         *      double-click or enter on a
                         *      container record core; update
                         *      package selections then.
                         */

                        case CN_ENTER: {
                            PNOTIFYRECORDENTER pnre = (PNOTIFYRECORDENTER)mp2;
                            PPCKRECORDCORE  precc = (PPCKRECORDCORE)pnre->pRecord;

                            if (precc)
                            {
                                // double-click on container, not whitespace:
                                PackageInfo* ppi = precc->pPckInfo;
                                if (ppi)
                                {
                                    ppi->ToggleSelection(WpiGlobals.PackageInfoList);

                                    // update drive info
                                    WinPostMsg(hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);

                                    UpdatePackageDisplay();
                                }
                            }
    /*                             if (!precc->pPckInfo->fNoDeselect)
                                {
                                    if (precc->pPckInfo->ulSelection == 1)
                                        ulNewSelection = 0;
                                    else // 0 or 2
                                        ulNewSelection = 1;
                                    // if we have a group, we need to select all
                                    // the child records too
                                    winhCnrForAllChildRecords(hwndCnrOnCnrPage,
                                                              (PRECORDCORE)precc,
                                                              &fncbreccSelectAll,
                                                              ulNewSelection,
                                                              0);
                                    // and now update all the parent records with
                                    // full / half / no selection
                                    winhCnrForAllParentRecords(hwndCnrOnCnrPage,
                                                               (PRECORDCORE)precc2,
                                                               &fncbreccUpdateParents,
                                                               0, 0);

                                } */
                        break; }

                        /*
                         * CN_EMPHASIS:
                         *      package selection changed; update
                         *      dependent info in dialog
                         */

                        case CN_EMPHASIS: {
                            PNOTIFYRECORDEMPHASIS pnre = (PNOTIFYRECORDEMPHASIS)mp2;
                            preccSelected = (PPCKRECORDCORE)pnre->pRecord;
                            if (pnre->fEmphasisMask & CRA_SELECTED)
                                WinSetDlgItemText(hwndPageContainer, ID_WIDI_PATHENTRY,
                                    preccSelected->pPckInfo->szTargetPath);

                            UpdatePackageDisplay();
                        break; }

                    }
                break; }

                /*
                 * ID_WIDI_PATHENTRY:
                 *      "Path" entry field
                 *      on "Container" page
                 */

                case ID_WIDI_PATHENTRY:
                {
                    switch (usNotifyCode)
                    {
                        /*
                         * EN_CHANGE:
                         *      PATH entry field has some new content;
                         *      we check for whether a new drive was enterd,
                         *      and if so, we need to update the recc data and,
                         *      if this is the "base" recc, examine all
                         *      other record cores too
                         */

                        case EN_CHANGE: {
                            // backup old path
                            PSZ pszTemp = strdup(preccSelected->pPckInfo->szTargetPath);

                            WinQueryDlgItemText(hwndPageContainer, ID_WIDI_PATHENTRY,
                                    sizeof(preccSelected->pPckInfo->szTargetPath),
                                    preccSelected->pPckInfo->szTargetPath);

                            if (preccSelected->pPckInfo->ulPathType & PATH_BASE)
                            {
                                // "base" record core: go thru all
                                // reccs and replace the old path by
                                // the new path, if found
                                winhCnrForAllRecords2(hwndCnrOnCnrPage,
                                                      fncbreccUpdatePathBase,
                                                      (ULONG)pszTemp,         // search for
                                                      (ULONG)preccSelected);  // base record with new path
                            }

                            if (*pszTemp != *(preccSelected->pPckInfo->szTargetPath))
                            {
                                // drive letter changed:
                                // update drive info, because the drive might
                                // have changed
                                WinPostMsg(hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);
                            }

                            free(pszTemp);
                        break; }
                    }
                } break;

                /*
                 * ID_WIDI_UPDATECONFIGSYS etc.:
                 *      checkboxes on "Configure" page.
                 */

                case ID_WIDI_UPDATECONFIGSYS:
                case ID_WIDI_INSTALLWPSCLASSES:
                case ID_WIDI_CREATEWPSOBJECTS:
                    if (    (usNotifyCode == BN_CLICKED)
                         || (usNotifyCode == BN_DBLCLICKED)
                       )
                    {
                        WpiGlobals.ulDoConfiguration = 0;
                        if (winhIsDlgItemChecked(hwndPageConfigure, ID_WIDI_UPDATECONFIGSYS))
                            WpiGlobals.ulDoConfiguration |= CONFIG_CONFIGSYS;
                        if (winhIsDlgItemChecked(hwndPageConfigure, ID_WIDI_INSTALLWPSCLASSES))
                            WpiGlobals.ulDoConfiguration |= CONFIG_WPSCLASSES;
                        if (winhIsDlgItemChecked(hwndPageConfigure, ID_WIDI_CREATEWPSOBJECTS))
                            WpiGlobals.ulDoConfiguration |= CONFIG_WPSOBJECTS;
                    }
                break;

            } // end switch(usControl)
        break; }

        /*
         * WM_DRAWITEM:
         *      container record core owner-draw
         */

        case WM_DRAWITEM:
        {
            // get generic DRAWITEM structure
            POWNERITEM poi = (POWNERITEM)mp2;

            // check if we're to draw the text
            // or the icon
            if (poi->idItem == CMA_ICON)
            {
                // icon:

                PCNRDRAWITEMINFO pcdii = (PCNRDRAWITEMINFO)poi->hItem;
                HPOINTER hptr;

                // 1) draw current install status to the left
                PackageInfo* pPckInfo = ((PPCKRECORDCORE)(pcdii->pRecord))->pPckInfo;
                if (pPckInfo->ulInstallStatus)
                    WinDrawPointer(poi->hps,
                                   poi->rclItem.xLeft,
                                   poi->rclItem.yBottom,
                                   hptrOK,
                                   DP_MINI);

                // 2) draw selection status (for installation)
                if (pPckInfo->ulSelection == 1)
                    hptr = hptrSelect;
                else if (pPckInfo->ulSelection == 2)
                    hptr = hptrSomeselect;
                else    // 0:
                    hptr = hptrDeselect;
                WinDrawPointer(poi->hps,
                        poi->rclItem.xLeft + ulMiniIconSize + 5,
                        poi->rclItem.yBottom,
                        hptr,
                        DP_MINI);
            }
            /* else if (poi->idItem == CMA_TEXT)
            {
                // get container-specific draw-item struct
                PCNRDRAWITEMINFO pcdii = (PCNRDRAWITEMINFO)poi->hItem;
                ULONG flCmd = DT_LEFT | DT_VCENTER | DT_ERASERECT;
                RECTL rcl2;

                // set draw colors
                LONG  lBackground = CLR_WHITE;
                LONG  lForeground = CLR_BLACK;

                // switch to RGB
                GpiCreateLogColorTable(poi->hps, 0,
                        LCOLF_RGB,
                        0, 0, NULL);

                // if (((pcdii->pRecord->flRecordAttr) & CRA_DISABLED) == 0) {
                    // not disabled == valid WPS class
                    if ((pcdii->pRecord->flRecordAttr) & CRA_SELECTED) {
                        lBackground = CLR_RED;
                        lForeground = CLR_WHITE;
                    }
                 } else {
                    if ((pcdii->pRecord->flRecordAttr) & CRA_SELECTED) {
                        lBackground = CLR_BLUE;
                        lForeground = CLR_WHITE;
                    } else
                        lForeground = CLR_BLUE;
                }

                memcpy(&rcl2, &(poi->rclItem), sizeof(rcl2));
                WinDrawText(poi->hps,
                        strlen(pcdii->pRecord->pszText),
                        pcdii->pRecord->pszText,
                        &rcl2,
                        lForeground,  // foreground
                        lBackground,
                        flCmd);
                mrc = (MPARAM)TRUE; // tell cnr that we've drawn the item
            } */

            else
                mrc = (MPARAM)FALSE; // tell cnr to draw the item
        break; }

        /*
         * WM_COMMAND:
         *
         */

        case WM_COMMAND:
        {
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIDI_NEXTBUTTON:
                    if (pCurrentPageInfo->lNextButton > 0)
                        WinPostMsg(hwndMainDlg, WPIM_TURNTOPAGE,
                                (MPARAM)(pCurrentPageInfo->lNextButton),
                                (MPARAM)TRUE);  // store in "visited" list
                    else
                        // button index == 0:
                        // this means start installation
                        WinDismissDlg(hwndMainDlg, DID_OK);
                break;

                case ID_WIDI_BACKBUTTON:
                    // TurnToPage has stored the last page in WpiGlobals
                    if (!WpiGlobals.lPagesList.empty())
                    {
                        LONG lLast = WpiGlobals.lPagesList.back();
                        // delete last item in list
                        WpiGlobals.lPagesList.pop_back();
                        WinPostMsg(hwndMainDlg, WPIM_TURNTOPAGE,
                                   (MPARAM)lLast,
                                   (MPARAM)FALSE);  // store in "visited" list
                    }
                break;

                /*
                 * ID_WIMI_CONFIRMATIONS:
                 *      "Confirmations" menu entry.
                 */

                case ID_WIMI_CONFIRMATIONS:
                    ShowConfirmations(hwndDlg);
                break;

                case ID_WIMI_HELPINDEX:
                case ID_WIMI_HELPGENERAL:
                case ID_WIMI_HELPUSINGHELP:
                    DebugBox("WarpIn", "Sigh... not implemented yet.");
                break;

                case ID_WIMI_HELPPRODUCTINFO:
                    guiProductInfo(hwndMainDlg);
                break;

                // "Create CID file" button
                case ID_WIDI_CREATECIDCMD: {
                    FILEDLG fd;
                    memset(&fd, 0, sizeof(FILEDLG));
                    fd.cbSize = sizeof(FILEDLG);
                    fd.fl = FDS_SAVEAS_DIALOG
                              | FDS_ENABLEFILELB
                              | FDS_CENTER;

                    strcpy(fd.szFullFile, WpiGlobals.szArcFilename);
                    PSZ p = strrchr(fd.szFullFile, '.');
                    if (p)
                        strcpy(p, ".cmd");

                    if (    WinFileDlg(HWND_DESKTOP,    // parent
                                       hwndDlg,
                                       &fd)
                        && (fd.lReturn == DID_OK)
                       )
                    {
                        PSZ pszTable = fd.szFullFile;
                        if (access(fd.szFullFile, 0) == 0)
                            // file exists:
                            if (guiMessageBoxMsg(hwndDlg,
                                                 108,
                                                 &pszTable,
                                                 1,
                                                 125,
                                                 MB_ICONQUESTION | MB_YESNO | MB_MOVEABLE)
                                    != MBID_YES)
                                break;
                        // else:
                        wpiCreateCIDFile(fd.szFullFile);
                    }
                break; }

                default:
                    // else: other command = cancel install
                    ConfirmCancelInstall(hwndMainDlg);
            }

        break; }

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ TurnToPage:
 *      this gets called by fnwpPages when the
 *      "Next" or "Back" buttons are pressed.
 *      We update the dialogs then according
 *      to the page data which was prepared
 *      by warpin.cpp in WPIGLOBALS.
 */

VOID TurnToPage(LONG lSearchPage, BOOL fStoreThis)
{
    /*
     * 1) handle "leaving page" stuff:
     *
     */

    if (pCurrentPageInfo)
    {
        // this still has the old page
        switch (pCurrentPageInfo->lPageMode)
        {
            /*
             * MODE_CONTAINER:
             *      leaving container page
             */

            case MODE_CONTAINER: {
                // PmPrintf("  Leaving container page");

                // if lSearchPage comes _after_ the current
                // page, we must check the validity of package
                // selections (we don't want to do this if
                // the user presses the "Back" button)

                if (lSearchPage > pCurrentPageInfo->lPageIndex)
                {
                    if (!guiCheckPackages())
                        // error found: do not leave container page
                        return;
                }

                // finally, leaving container page:
                // hide "drive info" window
                WinPostMsg(hwndDriveInfo, WPIM_UPDATE, (MPARAM)0, NULL);

            break; } // case MODE_CONTAINER

            /*
             * MODE_CONFIGURE:
             *      leaving "configure" page
             */

            case MODE_CONFIGURE:
            break;

        } // end switch (pCurrentPageInfo->lPageMode)

        // store the index of the page that we're leaving
        // in the list in WpiGlobals,
        if (fStoreThis)
            WpiGlobals.lPagesList.push_back(pCurrentPageInfo->lPageIndex);

    } // end if (pCurrentPageInfo)

    // find the page we're looking for (lSearchPage)
    pCurrentPageInfo = NULL;
    list<PWPIPAGEINFO>::iterator PageStart, PageEnd;
    PageStart = WpiGlobals.PageInfoList.begin();
    PageEnd = WpiGlobals.PageInfoList.end();
    for (; PageStart != PageEnd; PageStart++)
        if ((**PageStart).lPageIndex == lSearchPage)
        {
            // page found: mark it
            pCurrentPageInfo = *PageStart;
            break;
        }

    if (pCurrentPageInfo == NULL)
        wpiParseError(NULL,
                "Unable to find page %d in the "
                "installation profile. The \"<PAGE ...>\" tags "
                "always need to have an INDEX attribute, and there "
                "needs to be at least one page.",
                (PSZ)lSearchPage);

    // set page text
    HWND hwndSubDialogTemp = NULLHANDLE;
    switch (pCurrentPageInfo->lPageMode)
    {
        case MODE_TEXT: hwndSubDialogTemp = hwndPageText; break;
        case MODE_README: hwndSubDialogTemp = hwndPageReadme; break;
        case MODE_CONTAINER: hwndSubDialogTemp = hwndPageContainer; break;
        case MODE_CONFIGURE: hwndSubDialogTemp = hwndPageConfigure; break;
        // case MODE_INSTALL: hwndSubDialogTemp = hwndPageInstall; break;
    }
    // each of the four subdialogs has a static text with ID_WIDI_INFOTEXT ID
    WinSetDlgItemText(hwndSubDialogTemp, ID_WIDI_INFOTEXT,
                      pCurrentPageInfo->pszInfoText);

    // now initialize the page we're turning to
    switch (pCurrentPageInfo->lPageMode)
    {

        /*
         * MODE_README:
         *      set the MLE with text
         */

        case MODE_README:
            WinSetDlgItemText(hwndSubDialogTemp, ID_WIDI_READMEMLE,
                              pCurrentPageInfo->pszReadmeText);
        break;

        /*
         * MODE_CONTAINER:
         *      entering container page:
         *      create record cores etc.
         */

        case MODE_CONTAINER:
        {
            // did we create record cores before?
            if (!fContainerPageReady)
            {
                ULONG ulCount;
                PPCKRECORDCORE preccThis;
                CHAR szTitleWithSize[200];

                // go thru the list of packages
                // (this has both "groups" and real packages)
                list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                             PckEnd = WpiGlobals.PackageInfoList.end();
                for (; (PckStart != PckEnd); PckStart++)
                {
                    PackageInfo* ppiThis = *PckStart;
                    // create record core for this package
                    preccThis = (PPCKRECORDCORE)winhCnrAllocRecords(hwndCnrOnCnrPage,
                                                sizeof(PCKRECORDCORE),
                                                1);
                    // store PackageInfoin recc
                    preccThis->pPckInfo = ppiThis;
                    // and recc in PACKAGEINFO
                    ppiThis->pvGuiData = (PRECORDCORE)preccThis;

                    sprintf(szTitleWithSize, "%s\n(%d KB)",
                                ppiThis->pszTitle,
                                ppiThis->ulSize);
                    // insert record core into cnr
                    winhCnrInsertRecords(hwndCnrOnCnrPage,
                            (PRECORDCORE)(ppiThis->ppiGroup)
                                            ? (PRECORDCORE)ppiThis->ppiGroup->pvGuiData
                                            : NULL,
                            (PRECORDCORE)preccThis,
                            strdup(szTitleWithSize),
                            CRA_RECORDREADONLY | CRA_COLLAPSED,
                            1);
                }

                fContainerPageReady = TRUE;
            }
            // show and update the drive info window
            WinPostMsg(hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);
        break; } // case MODE_CONTAINER:

        /*
         * MODE_CONFIGURE:
         *      entering configure page
         */

        case MODE_CONFIGURE: {
            // enable/disable the checkboxes;
            // the WpiGlobals.ulConfigureFlags flags have
            // been set by wpiParsePackages if _any_
            // package has corresponding information
            // in the install script.
            // We _enable_ the dlg items according to ulConfigureFlags,
            // we _set_ the dlg items according to ulDoConfiguration.
            winhEnableDlgItem(hwndPageConfigure,
                              ID_WIDI_UPDATECONFIGSYS,
                              ((WpiGlobals.ulConfigureFlags & CONFIG_CONFIGSYS) != 0));
            winhSetDlgItemChecked(hwndPageConfigure,
                              ID_WIDI_UPDATECONFIGSYS,
                              ((WpiGlobals.ulDoConfiguration & CONFIG_CONFIGSYS) != 0));
            winhSetDlgItemChecked(hwndPageConfigure,
                              ID_WIDI_INSTALLWPSCLASSES,
                              ((WpiGlobals.ulConfigureFlags & CONFIG_WPSCLASSES) != 0));
            winhEnableDlgItem(hwndPageConfigure,
                              ID_WIDI_INSTALLWPSCLASSES,
                              ((WpiGlobals.ulDoConfiguration & CONFIG_WPSCLASSES) != 0));
            winhSetDlgItemChecked(hwndPageConfigure,
                              ID_WIDI_CREATEWPSOBJECTS,
                              ((WpiGlobals.ulConfigureFlags & CONFIG_WPSOBJECTS) != 0));
            winhEnableDlgItem(hwndPageConfigure,
                              ID_WIDI_CREATEWPSOBJECTS,
                              ((WpiGlobals.ulDoConfiguration & CONFIG_WPSOBJECTS) != 0));
        break; }

    } // end switch (pCurrentPageInfo->lPageMode)

    HWND    hwndFocusButton = WinWindowFromID(hwndMainDlg, ID_WIDI_CANCELBUTTON);

    // set "Back" button (all pages)
    winhShowDlgItem(hwndMainDlg, ID_WIDI_BACKBUTTON,
                    !(WpiGlobals.lPagesList.empty()));

    // set "Next" button (all pages)
    BOOL fShow =   (pCurrentPageInfo->lNextButton >= 0);
    HWND hwndNextButton = WinWindowFromID(hwndMainDlg, ID_WIDI_NEXTBUTTON);
    WinShowWindow(hwndNextButton, fShow);
    if (fShow)
    {
        WinSetWindowText(hwndNextButton, pCurrentPageInfo->pszNextButtonTitle);
        if (fShow)
            hwndFocusButton = hwndNextButton;
    }

    WinSetFocus(HWND_DESKTOP, hwndFocusButton);

    // PmPrintf("Calling DisplayMode");
    // show/hide controls
    DisplayMode(pCurrentPageInfo->lPageMode);

    WinSetActiveWindow(HWND_DESKTOP, hwndMainDlg);
}

/*
 *@@ fnwpFileExists:
 *      window proc for "file exists" confirmation
 *      dialog. This is only for an additional
 *      confirmation when "Cancel" or "Confirmations" are pressed.
 */

MRESULT EXPENTRY fnwpFileExists(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        case WM_COMMAND: {
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIMI_CONFIRMATIONS:
                    ShowConfirmations(hwndDlg);
                break;

                case DID_CANCEL:
                    if (!ConfirmCancelInstall(hwndDlg))
                        break;

                default:
                    mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        break; }

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ FileExistsDlg:
 *      this gets called from fnwpPages if a file already
 *      exists on disk and we need to prompt the user.
 *      This must return one of the following:
 *          CBRC_PROCEED    replace this file
 *          CBRC_SKIP       skip this file
 *          CBRC_CANCEL     abort installation
 */

int FileExistsDlg(HWND hwndOwner, PFILEEXISTS pfi)
{
    int irc = CBRC_SKIP;

    static int      last_sel = ID_WIDI_EXISTS_SKIP;

    // lNextTime is the global (static) flag
    // in which we remember the last user selection

    HWND hwndConfirm = WinLoadDlg(HWND_DESKTOP, hwndOwner,
                                  fnwpFileExists,
                                  NULLHANDLE,
                                  ID_WID_FILEEXISTS,
                                  NULL);
    SetControlsFont(hwndConfirm, 0, 1003);

    // now set fields in dialog
    // 1)   existing file
    CHAR szTemp[300];
    strhFileDate(szTemp, &(pfi->fs3Existing.fdateLastWrite), ulDateFormat, szDateSep[0]);
    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_DATEOLD,
                    szTemp);
    strhFileTime(szTemp, &(pfi->fs3Existing.ftimeLastWrite), ulTimeFormat, szTimeSep[0]);
    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_TIMEOLD,
                    szTemp);

    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_SIZEOLD,
                    strhThousandsULong(szTemp, pfi->fs3Existing.cbFile, szThousand[0]));

    // 2)   file in archive
    strhFileDate(szTemp, &pfi->fDateHeader, ulDateFormat, szDateSep[0]);
    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_DATENEW,
                    szTemp);
    strhFileTime(szTemp, &pfi->fTimeHeader, ulTimeFormat, szTimeSep[0]);
    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_TIMENEW,
                    szTemp);

    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_SIZENEW,
                    strhThousandsULong(szTemp, pfi->pwifh->origsize, szThousand[0]));

    // 3)  issue warning text?
    HWND hwndInfoText = WinWindowFromID(hwndConfirm, ID_WIDI_EXISTS_INFOTEXT);
    if (pfi->iComp < 0)
    {
        WinSetWindowText(hwndInfoText,
                         NLSStrings.pszWarningExistingNewer);
                                // "Warning: The existing file is newer!"
        WinShowWindow(hwndInfoText, TRUE);
    }
    else
        WinShowWindow(hwndInfoText, FALSE);

    // set file name (replace %1 and %2 in static text)
    CHAR        szMessage[1000];
    WinQueryDlgItemText(hwndConfirm, ID_WIDI_EXISTS_TXT1,
            sizeof(szMessage), szMessage);
    PSZ pszLastBackslash = strrchr(pfi->szFilename, '\\');
    strhReplace(szMessage, "%1", pszLastBackslash+1);
    *pszLastBackslash = 0;
    strhReplace(szMessage, "%2", pfi->szFilename);
    WinSetDlgItemText(hwndConfirm, ID_WIDI_EXISTS_TXT1,
            szMessage);

    // check last selection item
    winhSetDlgItemChecked(hwndConfirm, last_sel, 1);

    // run confirmation dialog
    ULONG ulrc = WinProcessDlg(hwndConfirm);

    if (ulrc == DID_OK)
    {
        // and now, do what?
        if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_EXISTS_SKIP))
        {
            last_sel = ID_WIDI_EXISTS_SKIP;
        } else if (winhIsDlgItemChecked(hwndConfirm, ID_WIDI_EXISTS_RENAME))
        {
            last_sel = ID_WIDI_EXISTS_RENAME;
        }
    }
    // "Cancel" is handled by fnwpFileExists

    WinDestroyWindow(hwndConfirm);

    // last_sel always has the action to do now

    if (last_sel == ID_WIDI_EXISTS_SKIP)
        irc = CBRC_SKIP;
    else
        irc = CBRC_PROCEED;

    return (irc);
}

/*
 *@@ Configure:
 *      this gets called from fnwpPages upon receiving
 *      WPIM_DONEWITHINSTALL (i.e. after the Install
 *      thread is done).
 *
 *      This calls the functions in warpin.cpp for
 *      changing CONFIG.SYS, registering WPS classes,
 *      creating WPS objects etc.
 *
 *      This is running on thread 1. As a result, we
 *      block the PM input queue while this is going on.
 *      We probably should do this while creating WPS
 *      objects etc. in the first place. ;-)
 */

BOOL Configure(VOID)
{
    BOOL        brc = FALSE;

    if (WpiGlobals.ulConfigureFlags == 0)
        // we don't have anything to configure at all:
        return (FALSE);

    HWND hwndStatus = WinLoadDlg(HWND_DESKTOP,
                                 HWND_DESKTOP,
                                 WinDefDlgProc,
                                 NULLHANDLE,      // our own EXE
                                 ID_WID_WARPINSTARTUP,
                                 NULL);
    HWND hwndInfoInStatus = WinWindowFromID(hwndStatus, ID_WIDI_INFOTEXT);
    winhCenterWindow(hwndStatus);

    if (WpiGlobals.ulConfigureFlags & CONFIG_CONFIGSYS)
    {
        WinSetWindowText(hwndInfoInStatus,
                         NLSStrings.pszUpdatingConfigSys);
                            // "Updating CONFIG.SYS");
        WinShowWindow(hwndStatus, TRUE);

        wpiConfigSys();
    }

    if (WpiGlobals.ulConfigureFlags & CONFIG_WPSCLASSES)
    {
        WinSetWindowText(hwndInfoInStatus,
                         NLSStrings.pszRegisteringClasses);
                            // "Registering WPS classes");
        WinShowWindow(hwndStatus, TRUE);

        wpiWPSClasses();
    }

    if (WpiGlobals.ulConfigureFlags & CONFIG_WPSOBJECTS)
    {
        // create WPS objects
        WinSetWindowText(hwndInfoInStatus,
                         NLSStrings.pszCreatingObjects);
                            // "Creating WPS objects");
        WinShowWindow(hwndStatus, TRUE);

        wpiCreateObjects();
    }

    WinDestroyWindow(hwndStatus);
}

/*
 *@@ fnwpInstall:
 *      window proc for the "Install" window after we're
 *      done with all the pages and the Install thread is
 *      running.
 *
 *      The various gui* callbacks keep posting messages
 *      to us.
 */

MRESULT EXPENTRY fnwpInstall(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    static HWND         hwndProgressBar = NULLHANDLE;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      initialize some more controls
         */

        case WM_INITDLG:
        {
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            SetControlsFont(hwndDlg, 0, 500);

            // create a progress bar
            hwndProgressBar = WinWindowFromID(hwndDlg, ID_WIDI_PROGRESSBAR);
            winhProgressBarFromStatic(hwndProgressBar,
                        PBA_ALIGNCENTER | PBA_BUTTONSTYLE);  // (*UM#4)

            // start the timer
            ulInstallTimeBegin = doshGetULongTime();
            WinStartTimer(WpiGlobals.habThread1, hwndDlg, 1, 1000);

            // start Install thread
            wpiStartInstallThread();
        break; }

        /*
         * WPIM_UPDATEPACKAGE:
         *      user msg posted from the Install thread.
         *      when we advance to the next package.
         *      Parameters:
         *          PSZ mp1  current package name
         *          PSZ mp2  string with "pck 1 out of 9"
         */

        case WPIM_UPDATEPACKAGE:
        {
            if (mp1 != NULL) {
                PackageInfo* pPckInfo = (PackageInfo*)mp1;
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURPCKTITLE,
                            pPckInfo->pszTitle);
                WinSetDlgItemText(hwndDlg, ID_WIDI_TARGETPATHDIR,
                            pPckInfo->szTargetPath);
            }
            if (mp2 != NULL) {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURPCKCOUNT, (PSZ)mp2);
                free(mp2);
            }
        break; }

        /*
         * WPIM_UPDATEFILE:
         *      user msg posted from the Install thread
         *      when we advance to the next file.
         *      Parameters:
         *          PSZ mp1  current file name
         *          PSZ mp2  string with "file 1 out of 9"
         */

        case WPIM_UPDATEFILE:
        {
            if (mp1 != NULL) {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURFILENAME, (PSZ)mp1);
                free(mp1);
            }
            if (mp2 != NULL) {
                WinSetDlgItemText(hwndDlg, ID_WIDI_CURFILECOUNT, (PSZ)mp2);
                free(mp2);
            }
            ULONG ulDrive, ulMap;
            DosQueryCurrentDisk(&ulDrive, &ulMap);
            CHAR szMsg[100], sz2[20];
            sprintf(szMsg, NLSStrings.pszDriveXFree, // "Drive %c: %s KB free",
                        UCHAR(ulDrive & 0xFF) + 'A' - 1,
                        strhThousandsDouble(sz2,
                                        doshQueryDiskFree(ulDrive) / 1024,
                                        szThousand[0]));
            WinSetDlgItemText(hwndDlg, ID_WIDI_TARGETPATHFREE, szMsg);
        break; }

        /*
         * WPIM_UPDATEBYTES:
         *      user msg posted from the Install thread
         *      when the "bytes" display needs updating.
         *      Parameters:
         *          SHORT mp1  current percentage
         */

        case WPIM_UPDATEBYTES: {
            ulPercent = (ULONG)mp1;
            WinPostMsg(hwndProgressBar, WM_UPDATEPROGRESSBAR,
                    (MPARAM)ulPercent,      // current value
                    (MPARAM)100);           // maximum value
        break; }

        /*
         * WM_TIMER:
         *      the timer is only running on the
         *      "Install" page to update the
         *      elapsed time once a second.
         *      We also attempt to have a meaningful
         *      "time to go" display, but this isn't
         *      always working right.
         */

        case WM_TIMER:
        {
            ULONG ulTime2 = doshGetULongTime();

            ulSecsElapsed = (ulTime2 - ulInstallTimeBegin) / 1000;

            ULONG   ulHours = ulSecsElapsed / 60 /60;
            ULONG   ulMinutes = ((ulSecsElapsed / 60) - (ulHours * 60));
            ULONG   ulSecs = ulSecsElapsed
                                - (ulHours * 60 * 60)
                                - (ulMinutes * 60);
            CHAR    szTime[30];
            sprintf(szTime, "%02d%c%02d%c%02d %s",
                    ulHours,    szTimeSep[0],
                    ulMinutes,  szTimeSep[0],
                    ulSecs,
                    NLSStrings.pszElapsed);

            WinSetDlgItemText(hwndDlg, ID_WIDI_TIMEELAPSED, szTime);

            // calculate time to go
            if ((ulSecsElapsed) && (ulPercent > 10))
            {
                ULONG   ulSecsToGo = (ulSecsElapsed * (100-ulPercent) ) / ulPercent;

                ULONG   ulHours = ulSecsToGo / 60 /60;
                ULONG   ulMinutes = ((ulSecsToGo / 60) - (ulHours * 60));
                ULONG   ulSecs = ulSecsToGo
                                    - (ulHours * 60 * 60)
                                    - (ulMinutes * 60);
                CHAR    szTime[30];
                sprintf(szTime, "%02d%c%02d%c%02d %s",
                        ulHours,    szTimeSep[0],
                        ulMinutes,  szTimeSep[0],
                        ulSecs,
                        NLSStrings.pszToGo);

                WinSetDlgItemText(hwndDlg, ID_WIDI_TIMELEFT, szTime);
            }
        break; }

        /*
         * WPIM_FILEEXISTS:
         *      this is posted by the guiFileExists callback
         *      on the Install thread if a file exists already.
         *      mp1 has the PFILEEXISTS structure in question.
         *
         *      Note that guiFileExists waits until the global
         *      iErrorRC variable is set to anything but -1.
         *      The Install thread then resumes operation
         *      depending on that value.
         *
         *      This must set iErrorRC to one of the following:
         *          CBRC_PROCEED    replace this file
         *          CBRC_SKIP       skip this file
         *      Those are the return values of FileExistsDlg.
         */

        case WPIM_FILEEXISTS:
            iErrorRC = FileExistsDlg(hwndDlg, (PFILEEXISTS)mp1);
        break;

        /*
         * WPIM_INSTALLERROR:
         *      posted from the Install thread also.
         *      Parameters:
         *          PSZ mp1     the error message; this is free()'d here
         *          ULONG mp2   CBREC_* flags (wiarchive.h)
         *
         *      Note that guiFileExists waits until the global
         *      iErrorRC variable is set to anything but -1.
         *      The Install thread then resumes operation
         *      depending on that value.
         */

        case WPIM_INSTALLERROR:
        {
            ULONG ulErrRsp;

            switch ((ULONG)mp2)
            {
                case CBREC_RETRYCANCEL:         ulErrRsp = MB_RETRYCANCEL; break;
                case CBREC_ABORTRETRYIGNORE:    ulErrRsp = MB_ABORTRETRYIGNORE; break;
                default:                        ulErrRsp = MB_CANCEL;
            }

            ULONG ulmbrc = WinMessageBox(HWND_DESKTOP,      // parent
                                         hwndDlg,       // owner
                                         (PCSZ)mp1,
                                         "WarpIn: Install Error",   // xxx
                                         0,
                                         MB_ICONEXCLAMATION | ulErrRsp | MB_MOVEABLE);

            switch (ulmbrc)
            {
                case MBID_RETRY:  iErrorRC = CBRC_RETRY; break;
                case MBID_IGNORE: iErrorRC = CBRC_IGNORE; break;
                case MBID_CANCEL: iErrorRC = CBRC_ABORT; break;
                case MBID_ABORT:  iErrorRC = CBRC_ABORT; break;
                default:          iErrorRC = CBRC_ABORT; break;
            }

            free(mp1);
        break; }

        /*
         * WPIM_DONEWITHINSTALL:
         *      posted from the Install thread
         *      when all files have been unpacked.
         *      The Install thread terminates itself
         *      automatically. We need to advance to
         *      the next page after the "Install" page.
         */

        case WPIM_DONEWITHINSTALL:
        {
            // start configuring (CONFIG.SYS, WPS classes etc.)
            Configure();

            // stop timer on container page
            WinStopTimer(WpiGlobals.habThread1, hwndDlg, 1);
            // and go to next page
            // TurnToPage(pCurrentPageInfo->lPageIndex + 1, FALSE);

            WpiGlobals.fInstallationComplete = TRUE;

            // get outta here
            WinDismissDlg(hwndDlg, DID_OK);
        break; }

        case WM_COMMAND:
            ConfirmCancelInstall(hwndDlg);
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

    } // end switch

    return (mrc);
}

/*******************************************************************
 *                                                                 *
 *  Drive information window                                       *
 *                                                                 *
 ******************************************************************/

/*
 *  This is additional code for the "free space on drives"
 *  window which is displayed only on the container page
 *  of the main window. This basically deals with the ugly
 *  Details view container messaging and updating it.
 */

/*
 *@@ fncbreccSumUpSize:
 *      callback for ForAllRecords upon the Drive info
 *      window.
 *      This goes thru all record cores and sums up
 *      the total free and required space on all the
 *      drives.
 *      This is needed for the bottommost row in the
 *      drive info container.
 */

ULONG               aulSpaceRequired[28];
PDRIVERECORDCORE    pDrvReccsInvalid[28];

VOID EXPENTRY fncbreccSumUpSize(PRECORDCORE precc, ULONG ul1, ULONG ul2)
{
    if (((PPCKRECORDCORE)precc)->pPckInfo->ulSelection == 1) {
        // package selected:
        LONG lDriveNum = (toupper(((PPCKRECORDCORE)precc)->pPckInfo->szTargetPath[0]))-'A'+1;
        // drive number (A = 1, B = 2, ...)
        if ((lDriveNum > 0) && (lDriveNum < 27))
            aulSpaceRequired[lDriveNum] += ((PPCKRECORDCORE)precc)->pPckInfo->ulSize;
    }
}

/*
 *@@ fnwpDriveInfo:
 *      wnd proc for the "Drive Info" window.
 *      This needs to handle a lot of ugly messages
 *      for cnr Details view.
 */

MRESULT EXPENTRY fnwpDriveInfo(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg) {

        /*
         * WM_INITDLG:
         *
         */

        case WM_INITDLG: {
            HENUM henum;
            HWND hwndItem;
            CHAR szFont[20];
            ULONG cbFont;

            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            // set fonts
            if (doshIsWarp4())
                // on Warp 4, use WarpSans
                strcpy(szFont, "9.WarpSans");
            else
                // on Warp 3, use Helv
                strcpy(szFont, "8.Helv");
            cbFont = strlen(szFont)+1;

            henum = WinBeginEnumWindows(hwndDlg);
            while (hwndItem = WinGetNextWindow(henum))
            {
                SHORT sID = (SHORT)WinQueryWindowUShort(hwndItem, QWS_ID);
                if ((sID > 0) && (sID < 500))
                    WinSetPresParam(hwndItem,
                            PP_FONTNAMESIZE,
                            cbFont,
                            szFont);
            }
            WinEndEnumWindows(henum);

            // initialize container data
            WinPostMsg(hwndDriveInfo, WPIM_QUERYDRIVES, NULL, NULL);
        break; }

        /*
         * WPIM_QUERYDRIVES:
         *      this is posted to initialize the container data.
         */

        case WPIM_QUERYDRIVES: {
            HWND hwndDriveCnr = WinWindowFromID(hwndDriveInfo, ID_WIDI_DRIVECNR);
            CHAR    szAllDrives[30];
            ULONG   cbDrives,
                    ul,
                    ulTotalFree = 0;
            PFIELDINFO  pFieldInfoFirst,
                        pFieldInfo2;
            PDRIVERECORDCORE precc;
            PSZ     pszTitle;


            // setup the container
            BEGIN_CNRINFO()
            {
                winhCnrSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
            } END_CNRINFO(hwndDriveCnr);

            // set up data for Details view columns
            XFIELDINFO      xfi[3];

            xfi[0].ulFieldOffset = FIELDOFFSET(RECORDCORE, pszIcon);
            xfi[0].pszColumnTitle = NLSStrings.pszDrive;
            xfi[0].ulDataType = CFA_STRING;
            xfi[0].ulOrientation = CFA_LEFT;

            xfi[1].ulFieldOffset = FIELDOFFSET(DRIVERECORDCORE, ulFree);
            xfi[1].pszColumnTitle = NLSStrings.pszFreeKB;
            xfi[1].ulDataType = CFA_ULONG;
            xfi[1].ulOrientation = CFA_RIGHT;

            xfi[2].ulFieldOffset = FIELDOFFSET(DRIVERECORDCORE, ulRequired);
            xfi[2].pszColumnTitle = NLSStrings.pszRequiredKB;
            xfi[2].ulDataType = CFA_ULONG;
            xfi[2].ulOrientation = CFA_RIGHT;

            PFIELDINFO pFieldInfoLast =
                winhCnrSetFieldInfos(hwndDriveCnr,
                                     &xfi[0],
                                     (sizeof(xfi) / sizeof(XFIELDINFO)),   // array item count
                                     TRUE,          // draw lines
                                     0);            // column index to return

            // now insert the record cores

            doshEnumDrives(szAllDrives, NULL); // all filesystems
            cbDrives = strlen(szAllDrives);

            for (ul = 0; ul <= cbDrives; ul++)
            {
                FSALLOCATE  fsa;
                precc = (PDRIVERECORDCORE)winhCnrAllocRecords(hwndDriveCnr,
                                            sizeof(DRIVERECORDCORE),
                                            1);
                pszTitle = (PSZ)malloc(100);
                if (ul < cbDrives)
                {
                    sprintf(pszTitle, "%c:", szAllDrives[ul]);
                    precc->ulDrive = ul + 3; // start with drive C:
                    DosQueryFSInfo(precc->ulDrive, FSIL_ALLOC, &fsa, sizeof(fsa));
                    precc->ulFree = ((fsa.cSectorUnit * fsa.cbSector * fsa.cUnitAvail)
                                    + 512)
                                    / 1024;
                    ulTotalFree += precc->ulFree;
                }
                else {
                    precc->ulDrive = 0;     // total entry has drive no. 0
                    sprintf(pszTitle, "Total:");
                    precc->ulFree = ulTotalFree;
                }

                winhCnrInsertRecords(hwndDriveCnr,
                        (PRECORDCORE)NULL,
                        (PRECORDCORE)precc,
                        pszTitle,
                        CRA_RECORDREADONLY,
                        1);
            }

            // WinPostMsg(hwndDriveInfo, WPIM_UPDATE, NULL, NULL);
        break; }

        /*
         * WPIM_UPDATE:
         *      this is posted whenever the drive display needs
         *      to be updated, e.g. when the user enters a different
         *      target path for a package.
         *      Parameters:
         *          BOOL mp1    if TRUE, the window will be updated
         *                      and shown and a timer will be started;
         *                      if FALSE, the window will be hidden
         *                      and the timer will be stopped.
         */

        case WPIM_UPDATE: {
            HWND hwndDriveCnr = WinWindowFromID(hwndDriveInfo, ID_WIDI_DRIVECNR);

            static ULONG    idTimerRunning = 0;

            if (mp1)
            {
                // show window:
                PDRIVERECORDCORE precc = NULL;
                ULONG   ulDrive = 3,
                        ulTotalSize = 0,
                        ul;
                // go thru the cnr in the main window
                // and update drive/size array
                memset(aulSpaceRequired, 0, sizeof(aulSpaceRequired));
                memset(pDrvReccsInvalid, 0, sizeof(pDrvReccsInvalid));
                winhCnrForAllRecords2(hwndCnrOnCnrPage, &fncbreccSumUpSize, 0, 0);

                // now update the drives window with the
                // new required disk space per drive by
                // going over all the record cores

                // reset the global warning flag
                fPlentyOfSpace = TRUE;
                do {
                    precc =
                        (PDRIVERECORDCORE)WinSendMsg(hwndDriveCnr,
                                CM_QUERYRECORD,
                                (MPARAM)precc,
                                MPFROM2SHORT(
                                    ((precc == NULL) ? CMA_FIRST : CMA_NEXT),
                                    CMA_ITEMORDER)
                                );
                    if ((LONG)precc == -1)
                        precc = NULL;
                    if (precc) {
                        if (precc->ulDrive) {
                            precc->ulRequired = aulSpaceRequired[ulDrive];
                            ulTotalSize += aulSpaceRequired[ulDrive];
                        } else
                            // "Total" entry
                            precc->ulRequired = ulTotalSize;

                        // if the required space on a drive
                        // exceeds the available space, draw
                        // the drive with "picked" emphasis
                        if (precc->ulRequired > precc->ulFree)
                        {
                            // not enough space:
                            fPlentyOfSpace = FALSE;
                            precc->recc.flRecordAttr |= CRA_SELECTED; // CRA_PICKED;
                            pDrvReccsInvalid[ulDrive] = precc;
                        } else
                        {
                            precc->recc.flRecordAttr &= ~CRA_SELECTED; // CRA_PICKED;
                            pDrvReccsInvalid[ulDrive] = 0;
                        }
                    }

                    ulDrive++;
                } while (precc);

                WinSendMsg(hwndDriveCnr,
                        CM_INVALIDATERECORD,
                        (MPARAM)NULL,       // all records
                        MPFROM2SHORT(0, CMA_ERASE));

                WinShowWindow(hwndDriveInfo, TRUE);

                if (idTimerRunning == 0)
                    idTimerRunning = WinStartTimer(WpiGlobals.habThread1, hwndDriveInfo,
                                            2, 400);
            }
            else
            {
                // hide window:
                if (idTimerRunning) {
                    WinStopTimer(WpiGlobals.habThread1, hwndDriveInfo, 2);
                    idTimerRunning = 0;
                }

                WinShowWindow(hwndDriveInfo, FALSE);
            }
        break; }

        /*
         * WM_TIMER:
         *
         */

        case WM_TIMER: {
            HWND hwndDriveCnr = WinWindowFromID(hwndDriveInfo, ID_WIDI_DRIVECNR);
            ULONG ul;

            for (ul = 0; ul < 28; ul++) {
                if (pDrvReccsInvalid[ul])
                {
                    pDrvReccsInvalid[ul]->recc.flRecordAttr ^= CRA_SELECTED; // XOR
                    WinSendMsg(hwndDriveCnr,
                            CM_SETRECORDEMPHASIS,
                            (MPARAM)pDrvReccsInvalid[ul],
                            MPFROM2SHORT(!((pDrvReccsInvalid[ul]->recc.flRecordAttr & CRA_SELECTED) != 0),
                                        CRA_SELECTED));
                }
            }
        break; }

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*******************************************************************
 *                                                                 *
 *  PM implementation part 2:                                      *
 *      Required callbacks on thread 1.                            *
 *                                                                 *
 ******************************************************************/

/*
 *@@ guiInitialize:
 *      this is supposed to set up the GUI environment.
 *      We are also passed the WPIGLOBALS structure
 *      in warpin.cpp so we can store a pointer to
 *      it here.
 *
 *      This _always_ gets called at the very beginning.
 *
 *      Note that this gets called _before_ WarpIN sets
 *      the various fields in WpiGlobals from the environment
 *      variables, so it's safe to set fields here from
 *      the user settings, because these will be overridden
 *      if environment variables have been defined.
 *
 *      At this point, the WARPIN.INI file has been opened
 *      and its HINI has been stored in WpiGlobals.
 *
 *      Required callback.
 */

BOOL guiInitialize(void)
{
    if (!(hmqThread1 = WinCreateMsgQueue(WpiGlobals.habThread1, 0)))
        return FALSE;

    LoadSettings();

    hwndWarpInStartup = WinLoadDlg(HWND_DESKTOP,
                                   HWND_DESKTOP,
                                   WinDefDlgProc,
                                   NULLHANDLE,      // our own EXE
                                   ID_WID_WARPINSTARTUP,
                                   NULL);
    winhCenterWindow(hwndWarpInStartup);
    WinShowWindow(hwndWarpInStartup, TRUE);

    // query all kinds of NLS data from OS2.INI
    // which we will use to display data
    ulDateFormat =
        PrfQueryProfileInt(HINI_USER, "PM_National", "iDate", 0);
    ulTimeFormat =
        PrfQueryProfileInt(HINI_USER, "PM_National", "iTime", 0);
    PrfQueryProfileString(HINI_USER, "PM_National", "sDate", "/",
        szDateSep, sizeof(szDateSep)-1);
    PrfQueryProfileString(HINI_USER, "PM_National", "sTime", ":",
        szTimeSep, sizeof(szTimeSep)-1);
    PrfQueryProfileString(HINI_USER, "PM_Default_National", "sThousand",
            ",", szThousand, sizeof(szThousand)-1);

    // load icons which we'll need
    hptrSelect   = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_SELECTED);
    hptrDeselect = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DESELECTED);
    hptrSomeselect = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_SOMESELECTED);
    hptrError    = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_ERROR);
    hptrQMark    = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_QMARK);
    hptrOK       = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_OK);
    hptrApp      = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_APP);

    // system icon size
    ulMiniIconSize = WinQuerySysValue(HWND_DESKTOP, SV_CYICON) / 2;

    // load NLS strings
    LoadString(WPSI_FILENOTFOUND, &NLSStrings.pszFileNotFound);
    LoadString(WPSI_EXISTINGFILENEWER, &NLSStrings.pszExistingFileNewer);
    LoadString(WPSI_EXISTINGFILEOLDER, &NLSStrings.pszExistingFileOlder);
    LoadString(WPSI_DELETEERROR, &NLSStrings.pszDeleteError);
    LoadString(WPSI_DELETEDIRERROR, &NLSStrings.pszDeleteDirError);
    LoadString(WPSI_PACKAGEREMOVED, &NLSStrings.pszPackageRemoved);
    LoadString(WPSI_CLASSDEREGISTERED, &NLSStrings.pszClassDeregistered);
    LoadString(WPSI_ERRORDEREGISTERINGCLASS, &NLSStrings.pszErrorDeregisteringClass);
    LoadString(WPSI_CLASSUNREPLACED, &NLSStrings.pszClassUnreplaced);
    LoadString(WPSI_ERRORUNREPLACINGCLASS, &NLSStrings.pszErrorUnreplacingClass);
    LoadString(WPSI_OBJECTDELETED, &NLSStrings.pszObjectDeleted);
    LoadString(WPSI_ERRORDELETINGOBJECT, &NLSStrings.pszErrorDeletingObject);
    LoadString(WPSI_CFGSYSUNDONE, &NLSStrings.pszCfgSysUndone);
    LoadString(WPSI_ERRORUNDOINGCFGSYS, &NLSStrings.pszErrorUndoingCfgSys);

    LoadString(WPSI_WARNINGEXISTINGNEWER, &NLSStrings.pszWarningExistingNewer);

    LoadString(WPSI_UPDATINGCONFIGSYS, &NLSStrings.pszUpdatingConfigSys);
    LoadString(WPSI_REGISTERINGCLASSES, &NLSStrings.pszRegisteringClasses);
    LoadString(WPSI_CREATINGOBJECTS, &NLSStrings.pszCreatingObjects);

    LoadString(WPSI_DRIVEXFREE, &NLSStrings.pszDriveXFree);
    LoadString(WPSI_ELAPSED, &NLSStrings.pszElapsed);
    LoadString(WPSI_TOGO, &NLSStrings.pszToGo);
    LoadString(WPSI_PACKAGEX, &NLSStrings.pszPackageX);
    LoadString(WPSI_FILEX, &NLSStrings.pszFileX);

    LoadString(WPSI_DRIVE, &NLSStrings.pszDrive);
    LoadString(WPSI_FREEKB, &NLSStrings.pszFreeKB);
    LoadString(WPSI_REQUIREDKB, &NLSStrings.pszRequiredKB);

    LoadString(WPSI_DEINSTALLINGARCHIVE, &NLSStrings.pszDeinstallingArchive);
    LoadString(WPSI_DEINSTALLINGPACKAGE, &NLSStrings.pszDeinstallingPackage);
    LoadString(WPSI_DEINSTALLINGAPPLICATION, &NLSStrings.pszDeinstallingApplication);
    LoadString(WPSI_VERIFYINGPACKAGE, &NLSStrings.pszVerifyingPackage);
    LoadString(WPSI_VERIFYINGAPPLICATION, &NLSStrings.pszVerifyingApplication);

    LoadString(WPSI_FILENAME, &NLSStrings.pszFileName);
    LoadString(WPSI_SIZE, &NLSStrings.pszSize);
    LoadString(WPSI_CREATIONDATE, &NLSStrings.pszCreationDate);
    LoadString(WPSI_LASTWRITEDATE, &NLSStrings.pszLastWriteDate);

    LoadString(WPSI_PACKAGENAME, &NLSStrings.pszPackageName);
    LoadString(WPSI_VERSION, &NLSStrings.pszVersion);
    LoadString(WPSI_AUTHOR, &NLSStrings.pszAuthor);
    LoadString(WPSI_TARGETPATH, &NLSStrings.pszTargetPath);
    LoadString(WPSI_FILES, &NLSStrings.pszFiles);
    LoadString(WPSI_INSTALLDATE, &NLSStrings.pszInstallDate);

    LoadString(WPSI_VERIFYING, &NLSStrings.pszVerifying);
    LoadString(WPSI_DELETING, &NLSStrings.pszDeleting);
    LoadString(WPSI_DONE, &NLSStrings.pszDone);

    // OK
    return (TRUE);
}

/*
 *@@ guiCleanupBeforeExit:
 *      this _always_ gets called before WarpIN exists.
 *
 *      Required callback.
 */

VOID guiCleanupBeforeExit(void)
{
    WinDestroyPointer(hptrSelect);
    WinDestroyPointer(hptrDeselect);
    WinDestroyPointer(hptrSomeselect);
    WinDestroyPointer(hptrError);
    WinDestroyPointer(hptrQMark);
    WinDestroyPointer(hptrOK);

    // clean up on the way out
    WinDestroyMsgQueue(hmqThread1);
}

/*
 *@@ guiMessageBox:
 *      this should display a message box.
 *      Note that this is called by warpin.cpp,
 *      so this cannot handle PM-specific stuff
 *      such as owner windows.
 *
 *      Required callback.
 */

ULONG guiMessageBox(PSZ pszTitle,
                    PSZ pszMessage,
                    ULONG ulFlags)
{
    return (WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
                          pszMessage,
                          pszTitle,
                          0,
                          ulFlags | MB_MOVEABLE));
}

/*
 *@@ guiDisplayErrorMessage:
 *      this should display a message box also,
 *      but translate the given messages from
 *      the WarpIN message file into strings
 *      beforehand.
 *      Note that this is called by warpin.cpp,
 *      so this cannot handle PM-specific stuff
 *      such as owner windows.
 *
 *      Required callback.
 */

VOID guiDisplayErrorMessage(ULONG ulTitleMsg,
                            ULONG ulMsgMsg,
                            ULONG ulFlags)
{
    CHAR    szTitle[200], szMsg[2000];
    wpiGetMessage(NULL, 0,
            szTitle, sizeof(szTitle),
            ulTitleMsg);
    wpiGetMessage(NULL, 0,
            szMsg, sizeof(szMsg),
            ulMsgMsg);
    guiMessageBox(szTitle, szMsg, ulFlags);
}

/*
 *@@ guiMessageBoxMsg:
 *      like before, but this one is a bit
 *      more sophisticated in that it can
 *      handle owner windows and style flags
 *      like in WinMessageBox.
 *      This can also do DosGetMessage string
 *      replacements.
 *
 *      Required callback.
 */

ULONG guiMessageBoxMsg(HWND hwndOwner,      // in: owner window
                       ULONG ulTitleMsg,    // in: dlg title message
                       PCHAR *pTable,       // in: string replacements for
                       ULONG cTable,        //     DosGetMessage
                       ULONG ulMsgMsg,      // in: dlg message
                       ULONG ulFlags)       // in: WinMessageBox style flags
{
    CHAR    szTitle[200], szMsg[2000];
    wpiGetMessage(NULL, 0,
            szTitle, sizeof(szTitle),
            ulTitleMsg);
    wpiGetMessage(pTable, cTable,
            szMsg, sizeof(szMsg),
            ulMsgMsg);
    return (WinMessageBox(HWND_DESKTOP, hwndOwner,
            szMsg, szTitle,
            0, ulFlags | MB_MOVEABLE));
}

/*
 *@@ guiAlreadyInstalled:
 *      this gets called before guiBeginAddRemove
 *      if main() has determined that packages from
 *      the given archive have been installed already.
 *
 *      This function must display a dialog offering
 *      the three "Repair", "Add/Remove", and "Deinstall"
 *      functions.
 *
 *      This returns:
 *      -- ACTION_CANCEL:      cancel (exit WarpIN)
 *      -- ACTION_REPAIR:      "Repair" selected
 *      -- ACTION_ADDREMOVE    "Add/Remove" selected (guiBeginAddRemove will be called then)
 *
 *      This function may also throw a CancelExcpt to terminate WarpIN.
 *
 *      Required callback.
 */

ULONG guiAlreadyInstalled(VOID) throw(CancelExcpt)
{
    ULONG   ulrc = 0;
    HWND    hwndDlg = WinLoadDlg(HWND_DESKTOP,
                                 HWND_DESKTOP,
                                 WinDefDlgProc,
                                 NULLHANDLE, ID_WID_ARCHIVEINSTALLED,
                                 NULL);

    winhCenterWindow(hwndDlg);
    SetControlsFont(hwndDlg, 0, 10);    // skip the buttons

    switch(WinProcessDlg(hwndDlg))
    {
        case ID_WIDI_AI_REPAIR:     ulrc = ACTION_REPAIR; break;
        case ID_WIDI_AI_ADDREMOVE:  ulrc = ACTION_ADDREMOVE; break;
        case ID_WIDI_AI_DEINSTALL:  ulrc = ACTION_DEINSTALL; break;

        default:        // cancel
            throw(CancelExcpt());
    }

    WinDestroyWindow(hwndDlg);

    return (ulrc);
}

/*
 *@@ guiBeginAddRemove:
 *      this gets called from main() when all the following
 *      is true:
 *      --  an archive was passed on the command line,
 *      --  the archive was successfully loaded,
 *      --  the install script was successfully parsed,
 *      --  the archive has not been installed yet or the
 *          user has selected "Add/Remove" in the dialog
 *          displayed by guiAlreadyInstalled or an environment
 *          variable has been set accordingly.
 *
 *      Required callback.
 *
 *      This func has no arguments, because all the
 *      needed data has been stored in WPIGLOBALS.
 *
 *      It is the responsibility of this function to start
 *      displaying all the pages described in the install
 *      script. After the last page has been turned to
 *      (i.e. installation is about to begin), this function
 *      must return with TRUE. If the user chooses to cancel
 *      the install process beforehand, this must return FALSE.
 *
 *      If TRUE is returned, guiStartInstall gets called
 *      afterwards.
 */

BOOL guiBeginAddRemove(void)
{

    /*
     * Initialize GUI:
     *
     */

    SWCNTRL     swctl;
    PID         pid;
    HSWITCH     hswitch;
    HPOINTER    hIcon = WinLoadPointer(HWND_DESKTOP,
                                NULLHANDLE,
                                ID_ICON);
    CHAR        szWinTitle[300] = "WarpIn";

    strcpy(szWinTitle, WpiGlobals.pArcHeader->name_app);

    // OK: now load the WarpIN main window.
    // All further processing is done in fnwpPages.
    hwndMainDlg = WinLoadDlg(HWND_DESKTOP, HWND_DESKTOP,
            fnwpPages,
            0, ID_WID_MAIN,
            NULL);
    if (hwndMainDlg == NULLHANDLE)
        guiMessageBox("WarpIn Error",
                      "The main window could not be loaded. ",
                      MB_ICONEXCLAMATION | MB_OK);

    // the main window is still hidden!

    // load menu -- the ol' menu in dlg trick
    WinLoadMenu(hwndMainDlg, NULLHANDLE, ID_WIM_MAINMENU);
    WinSendMsg(hwndMainDlg, WM_UPDATEFRAME, (MPARAM)FCF_MENU, 0);

    winhCenterWindow(hwndMainDlg); // invisibly

    // now load the four subdialogs, which will be
    // children of the main window. They all use
    // fnwpPages as their window procedure also,
    // since we have all the message handling in
    // there.
    hwndPageText = WinLoadDlg(hwndMainDlg,     // parent
                              hwndMainDlg,
                              fnwpPages,
                              NULLHANDLE,
                              ID_WID_PAGE_TEXT,
                              0);
    hwndPageReadme = WinLoadDlg(hwndMainDlg,     // parent
                               hwndMainDlg,
                               fnwpPages,
                               NULLHANDLE,
                               ID_WID_PAGE_README,
                               0);
    hwndPageContainer = WinLoadDlg(hwndMainDlg,     // parent
                              hwndMainDlg,
                              fnwpPages,
                              NULLHANDLE,
                              ID_WID_PAGE_CONTAINER,
                              0);
    hwndPageConfigure = WinLoadDlg(hwndMainDlg,     // parent
                              hwndMainDlg,
                              fnwpPages,
                              NULLHANDLE,
                              ID_WID_PAGE_CONFIGURE,
                              0);

    // this loads the drive-info window, which just
    // has the cnr for displaying drives.
    hwndDriveInfo = WinLoadDlg(HWND_DESKTOP,        // parent
                               hwndMainDlg,         // owner is the main window
                               fnwpDriveInfo,
                               0, ID_WID_DRIVEINFO,
                               NULL);

    // give the dlg an icon
    WinSendMsg(hwndMainDlg,
               WM_SETICON,
               (MPARAM)hIcon,
                NULL);

    // add ourselves to the tasklist
    swctl.hwnd = hwndMainDlg;              // window handle
    swctl.hwndIcon = hIcon;                // icon handle
    swctl.hprog = NULLHANDLE;              // program handle
    WinQueryWindowProcess(hwndMainDlg, &(swctl.idProcess), NULL);
                                           // process identifier
    swctl.idSession = 0;                   // session identifier ?
    swctl.uchVisibility = SWL_VISIBLE;     // visibility
    swctl.fbJump = SWL_JUMPABLE;           // jump indicator
    strcpy(swctl.szSwtitle, szWinTitle);
    swctl.bProgType = PROG_DEFAULT;        // program type
    WinSetWindowText(hwndMainDlg, szWinTitle);
    hswitch = WinAddSwitchEntry(&swctl);

    // now set up the controls in the dialogs;
    // most of these are currently hidden, but
    // we can initialize them anyways

    // setup the container
    hwndCnrOnCnrPage = WinWindowFromID(hwndPageContainer, ID_WIDI_CNR);
    BEGIN_CNRINFO()
    {
        winhCnrSetView(CV_TREE | CV_ICON | CA_TREELINE | CA_OWNERDRAW);
        // CnrInfo.cxTreeIndent = 15;
        // set icon sizes to system mini-icon size
        winhCnrSetBmpOrIconSize((ulMiniIconSize * 2 + 5), ulMiniIconSize);
    } END_CNRINFO(hwndCnrOnCnrPage);

    // hide the startup window
    WinDestroyWindow(hwndWarpInStartup);

    /*
     * Start user interaction:
     *
     */

    // start with page 1
    WinPostMsg(hwndMainDlg, WPIM_TURNTOPAGE, (MPARAM)1, MPNULL);
    // go!
    ULONG ulrc = WinProcessDlg(hwndMainDlg);
            // fnwpPages takes over;
            // this does not return until we either
            // get to the last page and the button was
            // pressed there (then DID_OK is returned)
            // or everything has been cancelled

    WinDestroyWindow(hwndMainDlg);

    return (ulrc == DID_OK);
}

/*
 *@@ guiPackageSelected:
 *      this gets called from the PackageInfo::Select
 *      method for every package selection that has
 *      changed.
 *
 *      The GUI should update its display for that
 *      package then.
 *
 *      Required callback.
 */

VOID guiPackageSelected(PackageInfo* pPckInfo)
{
    if (fContainerPageReady)
    {
        // get the record core which we created for
        // this package
        PPCKRECORDCORE precc = (PPCKRECORDCORE)(pPckInfo->pvGuiData);

        // repaint the record core
        WinSendMsg(hwndCnrOnCnrPage,
                   CM_INVALIDATERECORD,
                   (MPARAM)&precc,
                   MPFROM2SHORT(1, CMA_ERASE));
    }
}

/*
 *@@ guiCheckPackages:
 *      this gets called from both TurnToPage and from
 *      main() to check if the current package selections
 *      are valid.
 *
 *      This returns TRUE if no errors were found.
 *
 *      Required callback.
 */

BOOL guiCheckPackages()
{
    list<PackageInfo*>::iterator PckStart = WpiGlobals.PackageInfoList.begin(),
                                 PckEnd = WpiGlobals.PackageInfoList.end();

    CHAR    szDirs[1000] = "";

    // check whether the paths are valid
    BOOL    fInvalidDirsFound = FALSE;
    BOOL    fPackagesToAdd = FALSE,
            fPackagesToRemove = FALSE;

    for (; (PckStart != PckEnd); PckStart++)
    {
        // skip groups
        if (!(**PckStart).IsGroup())
        {
            if ((**PckStart).Selection())
                fPackagesToAdd = TRUE;
            else
                if ((**PckStart).ulInstallStatus != INSTALLED_NO)
                    fPackagesToRemove = TRUE;

            BOOL    fInvalidThis = FALSE;
            PSZ     pszTargetPath = &(**PckStart).szTargetPath[0];

            if (strlen(pszTargetPath) < 4)
                fInvalidThis = TRUE;
            else if (*(pszTargetPath+1) != ':')
                fInvalidThis = TRUE;
            else if (*(pszTargetPath+2) != '\\')
                fInvalidThis = TRUE;
            else if (doshIsValidFileName(pszTargetPath))
                fInvalidThis = TRUE;

            if (fInvalidThis)
            {
                sprintf(szDirs+strlen(szDirs), "\n    %s", (**PckStart).szTargetPath);
                fInvalidDirsFound = TRUE;
            }
        }
    }

    if ((!fPackagesToAdd) && (!fPackagesToRemove))
    {
        // no packages selected for installation/deinstallation:
        // this doesn't make any sense
        guiMessageBoxMsg(hwndMainDlg,
                         102,       // "Error"
                         NULL, 0,
                         126,       // "no packages"
                         MB_ICONEXCLAMATION | MB_OK | MB_MOVEABLE);
        // report error
        return (FALSE);
    }

    if (fInvalidDirsFound)
    {
        PSZ pTable = szDirs;
        guiMessageBoxMsg(hwndMainDlg,
                         114,    // "invalid directories"
                         &pTable, 1,
                         115,
                         MB_ICONEXCLAMATION | MB_OK | MB_MOVEABLE);
        // report error
        return (FALSE);
    }

    // check if disk space on target drives is sufficient
    // by evaluating the global flag which is maintained
    // by the "Drive info" window (fnwpDriveInfo)
    if (!fPlentyOfSpace)
        if (guiMessageBoxMsg(hwndMainDlg,
                     108,   // "WarpIn: Warning"
                     0, 0,
                     110,   // "not enough space"
                     MB_ICONEXCLAMATION | MB_YESNO | MB_MOVEABLE)
                != MBID_YES)
            // report error
            return (FALSE);

    // ask whether to create directories which
    // don't exist yet
    szDirs[0] = 0;
    BOOL    fDirsMissing = FALSE;
    PckStart = WpiGlobals.PackageInfoList.begin();
    PckEnd = WpiGlobals.PackageInfoList.end();
    for (; (PckStart != PckEnd); PckStart++)
    {
        if (!doshQueryDirExist((**PckStart).szTargetPath))
        {
            sprintf(szDirs+strlen(szDirs), "\n    %s", (**PckStart).szTargetPath);
            fDirsMissing = TRUE;
        }
    }

    if (fDirsMissing)
    {
        PSZ pTable = szDirs;
        if (guiMessageBoxMsg(hwndMainDlg,
                            116,    // "create directories"
                            &pTable,
                            1,
                            117,
                            MB_ICONEXCLAMATION | MB_YESNO | MB_MOVEABLE)
                    != MBID_YES)
            // report error
            return (FALSE);
    }

    return (TRUE);
}

/*
 *@@ guiEnterVariable:
 *      this gets called by wpiResolveMacros if an
 *      environment variable needs to be resolved and
 *      a value could not be found.
 *
 *      This must return the new value for that variable
 *      in a new buffer allocated using malloc().
 *
 *      Required callback.
 */

PSZ guiEnterVariable(PSZ pszVarName,        // in: name of the variable (e.g. "PATH")
                     PSZ pszDescription)    // in: description of that variable
                                            // (to be displayed to the user)
                     throw(CancelExcpt)
{
    PSZ     pszValue;
    HWND hwndDlg = WinLoadDlg(HWND_DESKTOP, HWND_DESKTOP,
                                   WinDefDlgProc,
                                   NULLHANDLE, ID_WID_ENTERVARIABLE,
                                   NULL);
    winhCenterWindow(hwndDlg);
    SetControlsFont(hwndDlg, 0, 2000);

    CHAR szVarName[300];
    sprintf(szVarName, "SET %s=", pszVarName);
    WinSetDlgItemText(hwndDlg, ID_WIDI_EV_SETTEXT, szVarName);

    HWND hwndEntryField = WinWindowFromID(hwndDlg, ID_WIDI_EV_ENTRYFIELD);
    winhSetEntryFieldLimit(hwndEntryField, 255);

    if (WinProcessDlg(hwndDlg) == DID_OK)
    {
        pszValue = winhQueryWindowText(hwndEntryField);
    }
    else
        throw(CancelExcpt());

    WinDestroyWindow(hwndDlg);
    return (pszValue);
}

/*
 *@@ guiStartInstall:
 *
 *
 *      If this returns TRUE, this means the packages
 *      are to be stored in the database afterwards.
 *
 *      Required callback.
 */

BOOL guiStartInstall(void)
{
    hwndInstallStatus = WinLoadDlg(HWND_DESKTOP,     // parent
                                   HWND_DESKTOP,
                                   fnwpInstall,
                                   NULLHANDLE,
                                   ID_WID_PAGE_INSTALL,
                                   0);
    winhCenterWindow(hwndInstallStatus);
    WinShowWindow(hwndInstallStatus, TRUE);
    ULONG ulrc = WinProcessDlg(hwndInstallStatus);
    hwndInstallStatus = NULLHANDLE;
    return (ulrc = DID_OK);
}

/*
 *@@ guiConfirmDeinstall:
 *      this displays the "Confirm Deinstall" dialog.
 *      pPckInfoList has the packages to deinstall,
 *      depending on which the dialog will be filled
 *      with data and such.
 *
 *      This gets called in two situations:
 *      1)  from main() after the user has selected
 *          "Deinstall" in the "Archive installed" dialog;
 *      2)  from fnwpDatabaseMain also before deinstalling
 *          any packages.
 *
 *      This returns -1 if the user pressed Cancel.
 *      Otherwise a value >= 0 is returned, that is 0 ORed
 *      with the DBDEINST_* flags def'd in warpin.h:
 *      -- DBDEINST_CONFIGSYS:  remove CONFIG.SYS entries
 *      -- DBDEINST_WPSCLASSES: de-install WPS classes
 *      -- DBDEINST_WPSOBJECTS: destroy WPS objects
 *
 *      Required callback.
 */

LONG guiConfirmDeinstall(HWND hwndOwner,                   // in: dialog owner
                         list<PackageInfo*>* pPckInfoList) // in: package list to deinstall.
{
    LONG   lrc = -1;
    if (pPckInfoList)
    {
        HWND hwndDlg = WinLoadDlg(HWND_DESKTOP,
                                  hwndOwner,
                                  WinDefDlgProc,
                                  NULLHANDLE, ID_WID_CONFIRMDEINSTALL,
                                  NULL);
        winhCenterWindow(hwndDlg);
        SetControlsFont(hwndDlg, 0, 2000);

        BOOL        fHasFiles = FALSE,
                    fHasConfigSys = FALSE,
                    fHasClasses = FALSE,
                    fHasObjects = FALSE;
        HWND        hwndFilesLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_FILESLIST),
                    hwndPckLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_PACKAGESLIST),
                    hwndCfgSysLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_CONFIGSYSLIST),
                    hwndClassesLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_WPSCLASSESLIST),
                    hwndObjectsLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_WPSOBJECTSLIST);
        list<PackageInfo*>::iterator PckStart = pPckInfoList->begin(),
                                     PckEnd = pPckInfoList->end();
        for (; PckStart != PckEnd; PckStart++)
        {
            PackageInfo* ppi = *PckStart;
            WinInsertLboxItem(hwndPckLbox, LIT_END, ppi->pszID);

            // check files
            if (ppi->pszFilesList)
            {
                ULONG ulHeaderCount = 0;
                list<WIFileHeader*>* pFileHeadersList = datGetFileHeaders(ppi,
                                                                      &ulHeaderCount);
                if (pFileHeadersList)
                {
                    list<WIFileHeader*>::iterator HeaderFirst = pFileHeadersList->begin(),
                                                  HeaderLast = pFileHeadersList->end();
                    for (; HeaderFirst != HeaderLast; HeaderFirst++)
                    {
                        WinInsertLboxItem(hwndFilesLbox, LIT_END, (**HeaderFirst).name);
                        delete *HeaderFirst;
                    }
                    delete pFileHeadersList;
                    fHasFiles = TRUE;
                }
            }

            // check CONFIG.SYS items
            list<CfgSysManip*>::iterator CfgStart = ppi->listUndoCfgSys.begin(),
                                         CfgEnd   = ppi->listUndoCfgSys.end();
            for (; CfgStart != CfgEnd; CfgStart++)
            {
                WinInsertLboxItem(hwndCfgSysLbox, LIT_END, (**CfgStart).pszNewLine);
                fHasConfigSys = TRUE;
            }

            // check registered classes
            list<DeregisterClass*>::iterator RegStart = ppi->listUndoRegisterClass.begin(),
                                             RegEnd   = ppi->listUndoRegisterClass.end();
            for (; RegStart != RegEnd; RegStart++)
            {
                WinInsertLboxItem(hwndClassesLbox, LIT_END, (**RegStart).pszClassName);
                fHasClasses = TRUE;
            }

            // check WPS objects created
            list<DeleteWPSObject*>::iterator ObjStart = ppi->listUndoWPSObject.begin(),
                                             ObjEnd   = ppi->listUndoWPSObject.end();
            for (; ObjStart != ObjEnd; ObjStart++)
            {
                WinInsertLboxItem(hwndObjectsLbox, LIT_END, (**ObjStart).pszObjectID);
                fHasObjects = TRUE;
            }
        }

        winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_DELETEFILES, fHasFiles);
        winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_DELETEFILES, fHasFiles);

        winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_CONFIGSYS, fHasConfigSys);
        winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_CONFIGSYS, fHasConfigSys);

        winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_WPSCLASSES, fHasClasses);
        winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_WPSCLASSES, fHasClasses);

        winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_WPSOBJECTS, fHasObjects);
        winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_WPSOBJECTS, fHasObjects);

        if (WinProcessDlg(hwndDlg) == DID_OK)
        {
            lrc = 0;
            if (winhIsDlgItemChecked(hwndDlg, ID_WIDI_CFD_CONFIGSYS))
                lrc |= DBDEINST_CONFIGSYS;
            if (winhIsDlgItemChecked(hwndDlg, ID_WIDI_CFD_WPSCLASSES))
                lrc |= DBDEINST_WPSCLASSES;
            if (winhIsDlgItemChecked(hwndDlg, ID_WIDI_CFD_WPSOBJECTS))
                lrc |= DBDEINST_WPSOBJECTS;
        }
        WinDestroyWindow(hwndDlg);
    }

    return (lrc);
}

/*
 *@@ guiDeinstallArchive:
 *      this gets called from main() after
 *      the user has selected "Deinstall" in the
 *      "Archive installed" dialog and guiConfirmDeinstall
 *      has not returned -1.
 *
 *      This will also open the Database status window
 *      (fnwpDatabaseStatus), but this time as a modal
 *      dialog.
 *
 *      Required callback.
 */

BOOL guiDeinstallArchive(list<PackageInfo*>* pPckInfoList,
                         ULONG ulFlags) // DBDEINST_* flags, as returned by
                                        // guiConfirmDeinstall
{
    DATABASESTATUSINFO dbsi(DBACTION_DEINSTALL,
                            ulFlags,
                            pPckInfoList,
                            NLSStrings.pszDeinstallingArchive,
                            NULLHANDLE);

    HWND hwndStatus = WinLoadDlg(HWND_DESKTOP,   // parent
                                 HWND_DESKTOP,   // owner
                                 fnwpDatabaseStatus,
                                 NULLHANDLE,      // our own EXE
                                 ID_WID_DATABASETHREAD,
                                 // pass the command to the dlg
                                 (PVOID)&dbsi);
    WinProcessDlg(hwndStatus);
        // modal this time
    WinDestroyWindow(hwndStatus);
}

/*
 *@@ guiProductInfo:
 *      this should display the WarpIN product
 *      info window.
 *
 *      Required callback.
 */

VOID guiProductInfo(HWND hwndOwner)
{
    HWND hwndProdInfo = WinLoadDlg(HWND_DESKTOP, hwndOwner,
                                   WinDefDlgProc,
                                   NULLHANDLE, ID_WID_PRODUCTINFO,
                                   NULL);
    CHAR szGPLInfo[2000];
    wpiGetMessage(NULL, 0,
            szGPLInfo, sizeof(szGPLInfo),
            101);
    WinSetDlgItemText(hwndProdInfo, ID_WIDI_INFOTEXT, szGPLInfo);
    winhCenterWindow(hwndProdInfo);
    WinProcessDlg(hwndProdInfo);
    WinDestroyWindow(hwndProdInfo);
}

/*******************************************************************
 *                                                                 *
 *  PM implementation part 3:                                      *
 *      this gets called on the Install thread.                    *
 *                                                                 *
 ******************************************************************/

/*  General note for the following functions:
 *      Most of these are callbacks from the
 *      main Install thread on which the "Unpack"
 *      methods of the WIArchive class are
 *      running. In other words, it's the Install
 *      thread which creates all the files.
 *      Since all the following functions therefore
 *      run on the Install thread, and since the
 *      Install thread does _not_ have a message
 *      queue, we better not manipulate windows
 *      here. Instead, we post msgs to the main
 *      window which does the updating on thread 1.
 */

/*
 *@@ guiUpdatePackage:
 *      GUI callback from warpin.cpp
 *      to update the "Package" display.
 *
 *      Required callback.
 */

void guiUpdatePackage(PackageInfo* pPckInfo, ULONG ulPckThis, ULONG ulPcksTotal)
{
    CHAR szMsg[100];
    sprintf(szMsg, NLSStrings.pszPackageX,  // "Package %d out of %d"
                   ulPckThis, ulPcksTotal);
    WinPostMsg(hwndInstallStatus, WPIM_UPDATEPACKAGE,
               pPckInfo,
               strdup(szMsg));
        // the window proc will release the string memory
}

/*
 *@@ guiUpdateFile:
 *      GUI callback from warpin.cpp
 *      to update the "File" display.
 *
 *      Required callback.
 */

void guiUpdateFile(WIFileHeader* pwifh, ULONG ulFileThis, ULONG ulFilesTotal)
{


    // CRASH;


    CHAR szMsg[100];
    sprintf(szMsg, NLSStrings.pszFileX, // "File %d out of %d",
                   ulFileThis, ulFilesTotal);
    WinPostMsg(hwndInstallStatus, WPIM_UPDATEFILE,
               strdup(pwifh->name), strdup(szMsg));
        // the window proc will release the string memory
}

/*
 *@@ guiUpdateBytes:
 *      GUI callback from warpin.cpp
 *      to update the progress display.
 *
 *      Required callback.
 */

void guiUpdateBytes(double dBytesDone, double dBytesTotal)
{
    SHORT sPercent = (SHORT)((dBytesDone / dBytesTotal) * 100);
    WinPostMsg(hwndInstallStatus, WPIM_UPDATEBYTES,
               (MPARAM)sPercent, 0);
}

/*
 *@@ guiFileExists:
 *      callback for the Install thread to confirm
 *      whether a file should be replaced.
 *      This only gets called if WICallback (warpin.cpp)
 *      has determined that the user should be prompted
 *      for what to do, depending on ulIfSameDate,
 *      ulIfExistingNewer, and ulIfExistingOlder in WpiGlobals.
 *
 *      This must return one of the following:
 *          CBRC_PROCEED    replace this file
 *          CBRC_SKIP       skip this file
 *      If the user presses "Cancel", it's our own
 *      responsibility to terminate everthing.
 *      We must _not_ return from this function then.
 *
 *      This callback should also offer the user an
 *      opportunity to reset the three flags in WpiGlobals
 *      so that prompting can be disabled for subsequent
 *      files.
 *
 *      Required callback.
 */

int guiFileExists(PFILEEXISTS pfi)
{
    // we cannot do a WinSendMsg, because this would
    // require the Install thread to have a PM message
    // queue.
    // WinPostMsg works though, so we rely on a global
    // variable to get the return code. This is safe
    // since this variable is only set by thread 1
    // after it receives our message.

    iErrorRC = -1;

    WinPostMsg(hwndInstallStatus, WPIM_FILEEXISTS,
               (MPARAM)pfi,     // pass structure to main window
               (MPARAM)0);

    // wait until the user has selected something
    // in the "File exists" dialog on thread 1;
    // WPIM_FILEEXISTS in fnwpPages (above) sets
    // iErrorRC to either CBRC_CANCEL, CBRC_SKIP
    // or CBRC_PROCEED
    while (iErrorRC == -1)
        DosSleep(100);  // that's ugly, but it works

    if (iErrorRC == CBRC_ABORT)
        // "Cancel" pressed: we need to terminate
        // our thread then, or OS/2 will hang
        _endthread();

    return (iErrorRC);
}

/*
 *@@ guiInstallError:
 *      this is called whenever an error occurs on
 *      the Install thread.
 *      This must return one of the following:
 *          CBRC_RETRY     retry the failing operation
 *      If the user presses "Cancel", it's our own
 *      responsibility to terminate everthing.
 *      We must _not_ return from this function then.
 *
 *      Required callback.
 */

int guiInstallError(char* pszMessage,   // in: message generated by warpin.cpp
                    int iErrorClass,    // in: ECLS_* flag
                    int iErrorCode,     // in: error code; meaning depends on iErrorClass
                    int iErrorRsp)      // in: possible response (ERSP_* flags)
{
    // we cannot do a WinSendMsg, because this would
    // require the Install thread to have a PM message
    // queue.
    // WinPostMsg works though, so we rely on a global
    // variable to get the return code. This is safe
    // since this variable is only set by thread 1
    // after it receives our message.

    PSZ     pszMessage2 = NULL;

    iErrorRC = -1;

    switch (iErrorClass)
    {
        case ECLS_DOS:
        case ECLS_BACKEND:
        {
            PSZ pszSysError = doshQuerySysErrorMsg(iErrorCode);
            if (pszSysError)
            {
                ULONG   cbMessage2 = strlen(pszSysError) + strlen(pszMessage) + 400;
                pszMessage2 = (PSZ)malloc(cbMessage2);

                PSZ pTable[2] = {pszMessage, pszSysError};

                wpiGetMessage(pTable, 2,
                        pszMessage2, cbMessage2,
                        118);       // "OS/2 reported..."
            }
        break; }

        case ECLS_EXCEPTION: {
            pszMessage2 = strdup(pszMessage);
        break; }

        default:
        // includes case ECLS_DECOMP:
        {
            ULONG   cbMessage2 = strlen(pszMessage) + 400;
            pszMessage2 = (PSZ)malloc(cbMessage2);

            PSZ pszExpl = "Unknown error";
            switch (iErrorCode) {
                case ZLIBERR_STREAM_ERROR:
                        pszExpl = "Inconsistent input stream data"; break;
                case ZLIBERR_DATA_ERROR:
                        pszExpl = "Input stream data does not match zlib format"; break;
                case ZLIBERR_MEM_ERROR:
                        pszExpl = "Not enough memory"; break;
                case ZLIBERR_BUF_ERROR:
                        pszExpl = "Not enough room in output buffer"; break;
                // the following are added by WIArchive
                case ZLIBERR_NO_WIFH_MAGIC:
                        pszExpl = "Invalid file header format in archive; probably outdated archive format"; break;
            }

            CHAR szCode[20];
            PSZ pTable[3] = {pszMessage, pszExpl, szCode};
            sprintf(szCode, "%d", iErrorCode);
            wpiGetMessage(pTable, 3,
                    pszMessage2, cbMessage2,
                    119);       // "Decompression engine reported..."
        break; }
    }

    // add button explanations from message file
    ULONG ulMsg = 0;
    switch(iErrorRsp)
    {
        case CBREC_RETRYCANCEL:
            ulMsg = 111;
        break;

        case CBREC_ABORTRETRYIGNORE:
            ulMsg = 112;
        break;

        default:
            ulMsg = 113;
    }

    PSZ p = pszMessage2 + strlen(pszMessage2);
    wpiGetMessage(NULL, 0, p, 400, ulMsg);

    WinPostMsg(hwndInstallStatus, WPIM_INSTALLERROR,
               (MPARAM)pszMessage2,
               (MPARAM)iErrorRsp);

    // wait until the user has selected something
    while (iErrorRC == -1)
        DosSleep(100);  // that's ugly, but it works

    if (iErrorRC == CBRC_ABORT)
    {
        // "Cancel" pressed: we need to terminate
        // our thread then, or OS/2 will hang
        WinPostMsg(hwndInstallStatus, WM_CLOSE, 0, 0);
        _endthread();
    }

    return (iErrorRC);
}

/*
 *@@ guiDoneWithInstall:
 *      callback when all files have been
 *      unpacked successfully.
 *      After this, the Install thread will
 *      terminate itself (warpin.cpp).
 *
 *      Required callback.
 */

void guiDoneWithInstall(void)
{
    WinPostMsg(hwndInstallStatus, WPIM_DONEWITHINSTALL,
               NULL,
               NULL);
        // this will call Configure() below
}

/*******************************************************************
 *                                                                 *
 *  PM implementation part 3:                                      *
 *      show database                                              *
 *                                                                 *
 ******************************************************************/

PDATABASERECORD         preccSource = NULL;

// list of all record cores for applications (tree view only)
list<PDATABASERECORD>   AppRecordsList;

// list of all record cores for packages (all views)
list<PDATABASERECORD>   PckRecordsList;

MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);


/*
 *@@ fnwpFileList:
 *      window procedure for the "File list" dialogs.
 *      The only purpose of this is to link the dialog
 *      to the corresponding DATABASERECORD in the
 *      main dialog.
 */

MRESULT EXPENTRY fnwpFileList(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      we are given the package record core
         *      (PDATABASERECORD) as a pCreateParam in mp2.
         *      Store this in QWL_USER.
         */

        case WM_INITDLG:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            WinSetWindowPtr(hwndDlg, QWL_USER,
                            mp2);       // PDATABASERECORD
        break;

        case WM_COMMAND: {
            WinDestroyWindow(hwndDlg);
        break; }

        case WM_DESTROY: {
            // get the database record again
            PDATABASERECORD precc = (PDATABASERECORD)WinQueryWindowPtr(hwndDlg, QWL_USER);
            if (precc)
                precc->hwndFileList = NULLHANDLE;
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        break; }

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ OpenFileList:
 *      this gets called from fnwpDatabaseMain when
 *      the file list is requested for a package.
 *      So this is what we'll do.
 *
 *      Note that precc must be a package record core
 *      from the main container (not an application
 *      record core, because applications have no
 *      file lists).
 *
 *      This stores the HWND of the file list window
 *      in precc. If this is called and a file list
 *      window is already open for precc, that window
 *      is activated only.
 */

VOID OpenFileList(PDATABASERECORD precc)    // in: package record core
{
    // is precc a package record core?
    if (precc->pPckInfo)
    {
        if (precc->hwndFileList)
            // file list already open for this record core:
            // activate it
            WinSetActiveWindow(HWND_DESKTOP, precc->hwndFileList);
        else
        {
            precc->hwndFileList = WinLoadDlg(HWND_DESKTOP,
                                           HWND_DESKTOP,
                                           fnwpFileList,
                                           NULLHANDLE,      // our own EXE
                                           ID_WID_FILELIST,
                                           precc);          // for WM_INITDLG
                // this is set back to NULLHANDLE upon
                // WM_DESTROY in fnwpFileList

            HENUM henum;
            HWND hwndItem;
            CHAR szFont[20];
            ULONG cbFont;
            // set font for all the dialog controls
            if (doshIsWarp4())
                strcpy(szFont, "9.WarpSans");
            else
                strcpy(szFont, "8.Helv");
            cbFont = strlen(szFont)+1;
            henum = WinBeginEnumWindows(precc->hwndFileList);
            while (hwndItem = WinGetNextWindow(henum))
            {
                SHORT sID = WinQueryWindowUShort(hwndItem, QWS_ID);
                if ((sID > -2) && (sID < 2000))
                    WinSetPresParam(hwndItem,
                            PP_FONTNAMESIZE,
                            cbFont,
                            szFont);
            }
            WinEndEnumWindows(henum);

            // set up the container for Details view

            CNRINFO CnrInfo;
            HWND    hwndFilesCnr = WinWindowFromID(precc->hwndFileList, ID_WIDI_CNR);

            // set up data for Details view columns
            XFIELDINFO      xfi[4];

            xfi[0].ulFieldOffset = FIELDOFFSET(RECORDCORE, pszIcon);
            xfi[0].pszColumnTitle = NLSStrings.pszFileName; // "File name";
            xfi[0].ulDataType = CFA_STRING;
            xfi[0].ulOrientation = CFA_LEFT;

            xfi[1].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, ulSize);
            xfi[1].pszColumnTitle = NLSStrings.pszSize; // "Size";
            xfi[1].ulDataType = CFA_ULONG;
            xfi[1].ulOrientation = CFA_RIGHT;

            xfi[2].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, cdateCreation);
            xfi[2].pszColumnTitle = NLSStrings.pszCreationDate; // "Creation date";
            xfi[2].ulDataType = CFA_DATE;
            xfi[2].ulOrientation = CFA_RIGHT;

            xfi[3].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, cdateLastWrite);
            xfi[3].pszColumnTitle = NLSStrings.pszLastWriteDate; // "Last write date";
            xfi[3].ulDataType = CFA_DATE;
            xfi[3].ulOrientation = CFA_RIGHT;

            PFIELDINFO pFieldInfoLast =
                winhCnrSetFieldInfos(hwndFilesCnr,
                                     &xfi[0],
                                     (sizeof(xfi) / sizeof(XFIELDINFO)),   // array item count
                                     TRUE,          // draw lines
                                     0);            // column index to return
            BEGIN_CNRINFO()
            {
                // set split bar
                winhCnrSetSplitBarAfter(pFieldInfoLast);
                winhCnrSetSplitBarPos(100);
                // switch view
                winhCnrSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
            } END_CNRINFO(hwndFilesCnr);

            // now fill the container
            PackageInfo*    pPckInfo = precc->pPckInfo;
            ULONG           ulHeaderCount = 0;

            list<WIFileHeader*>* pHeaderList = datGetFileHeaders(pPckInfo,
                                                                 &ulHeaderCount);
            if (pHeaderList)
            {
                // allocate memory for file record cores;
                // the number of items has been returned by datGetFileHeaders
                PFILERECORDCORE preccFileFirst = (PFILERECORDCORE)winhCnrAllocRecords(
                                                                hwndFilesCnr,
                                                                sizeof(FILERECORDCORE),
                                                                ulHeaderCount);

                // preccFile will be used for filling the current recc;
                // the record cores are linked using the preccNextRecord fields
                PFILERECORDCORE preccFile = preccFileFirst;

                list<WIFileHeader*>::iterator HeaderStart, HeaderEnd;
                HeaderStart = pHeaderList->begin();
                HeaderEnd = pHeaderList->end();

                for (; (HeaderStart != HeaderEnd); HeaderStart++)
                {
                    preccFile->recc.pszIcon =
                    preccFile->recc.pszText =
                    preccFile->recc.pszName =
                    preccFile->recc.pszTree = (**HeaderStart).name;
                    preccFile->ulSize = (**HeaderStart).origsize;
                    // convert time_t to human-readable format
                    struct tm* pTimeBuf = localtime(&(**HeaderStart).lastwrite);
                    preccFile->cdateLastWrite.day    = pTimeBuf->tm_mday;
                    preccFile->cdateLastWrite.month  = pTimeBuf->tm_mon + 1;
                    preccFile->cdateLastWrite.year   = pTimeBuf->tm_year + 1900;
                    pTimeBuf = localtime(&(**HeaderStart).creation);
                    preccFile->cdateCreation.day   = pTimeBuf->tm_mday;
                    preccFile->cdateCreation.month = pTimeBuf->tm_mon + 1;
                    preccFile->cdateCreation.year  = pTimeBuf->tm_year + 1900;

                    preccFile = (PFILERECORDCORE)(preccFile->recc.preccNextRecord);
                }

                // insert the record core linked list
                RECORDINSERT ri;
                memset(&ri, 0, sizeof(ri));
                ri.cb = sizeof(ri);
                ri.pRecordOrder = (PRECORDCORE)CMA_END;
                ri.pRecordParent = 0;
                ri.fInvalidateRecord = TRUE;
                ri.zOrder = CMA_TOP;
                ri.cRecordsInsert = ulHeaderCount;

                WinSendMsg(hwndFilesCnr,
                           CM_INSERTRECORD,
                           (MPARAM)preccFileFirst,
                           (MPARAM)&ri);

                // delete pHeaderList;
            }

            // OK, show window

            WinShowWindow(precc->hwndFileList, TRUE);
        } // end elseif (precc->hwndFileList)
    } // end if (precc->pPckInfo)
}

/*
 *@@ FreeDatabaseRecord:
 *      this removes preccRemove from the given container
 *      and frees all associated data.
 */

VOID FreeDatabaseRecord(HWND hwndCnr, PDATABASERECORD preccRemove)
{
    if (preccRemove->pPckInfo)
        // is package:
        PckRecordsList.remove(preccRemove);
    else
        // is application:
        AppRecordsList.remove(preccRemove);

    if (preccRemove->pszAuthor)
        free(preccRemove->pszAuthor);
    if (preccRemove->pszApplication)
        free(preccRemove->pszApplication);
    if (preccRemove->pszPackageName)
        free(preccRemove->pszPackageName);

    WinSendMsg(hwndCnr,
               CM_REMOVERECORD,
               (MPARAM)&preccRemove,
               MPFROM2SHORT(1,
                            CMA_FREE | CMA_INVALIDATE));

}

/*
 *@@ fnwpFreeReccFields:
 *      callback for winhCnrForAllRecords, used from
 *      ClearDatabaseContainer.
 *
 *      This gets the following parameters:
 *          --  hwndCnr:    the container window
 *          --  msg:        always 0
 *          --  mp1:        the container record core
 *          --  mp2:        ulUser (0 here)
 */

MRESULT EXPENTRY fnwpRemoveRecord(HWND hwndCnr, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    if (mp1)
    {
        WinSendMsg(hwndCnr,
                   CM_REMOVERECORD,
                   (MPARAM)&mp1,
                   MPFROM2SHORT(1,      // one record
                                CMA_INVALIDATE));
                                        // no CMA_FREE, because we need
                                        // the recc later
    }
}

/*
 *@@ ClearDatabaseContainer:
 *      this removes all record cores from the given
 *      container. It is assumed that the container
 *      uses DATABASERECORDs for its record cores,
 *      whose data is free()'d here.
 */

VOID ClearDatabaseContainer(HWND hwndCnr)
{
    winhCnrForAllRecords(hwndCnr, NULL, fnwpRemoveRecord, 0);
}

/*
 *@@ FillDatabaseContainer:
 *      this fills the container with all applications/packages.
 *
 *      Depending on GuiSettings.ulDatabaseView, this is either
 *      a Tree view or a Details view.
 */

VOID FillDatabaseContainer(HWND hwndCnr)                 // in: cnr to fill
{
    ClearDatabaseContainer(hwndCnr);

    /*
     * ID_WIMI_DATABASE_TREEVIEW:
     *
     */

    if (GuiSettings.ulDatabaseView == ID_WIMI_DATABASE_TREEVIEW)
    {
        // CnrInfo.cxTreeIndent = 15;

        // switch cnr to Details view
        BEGIN_CNRINFO()
        {
            // set bitmap size
            winhCnrSetBmpOrIconSize(16, 16);
            // switch view
            winhCnrSetView(CV_TREE | CV_ICON | CA_TREELINE | CV_MINI);
                                // | CA_DRAWBITMAP);
        } END_CNRINFO(hwndCnr);

        // insert all application record cores to root level
        list<PDATABASERECORD>::iterator AppsStart, AppsEnd;
        AppsStart = AppRecordsList.begin();
        AppsEnd = AppRecordsList.end();

        for (; (AppsStart != AppsEnd); AppsStart++)
        {
            PDATABASERECORD precc = (*AppsStart);
            winhCnrInsertRecords(hwndCnr,
                               (PRECORDCORE)NULL,       // root level
                               (PRECORDCORE)precc,
                               precc->pszApplication,
                               CRA_RECORDREADONLY | CRA_COLLAPSED,
                               1);
        }

        // insert all packages as children of the applications
        AppsStart = AppRecordsList.begin();
        AppsEnd = AppRecordsList.end();

        for (; (AppsStart != AppsEnd); AppsStart++)
        {
            PDATABASERECORD preccApplication = (*AppsStart);

            // AppsStart now has the current application recc;
            // go thru the package recc list and add all package
            // records which have this application

            list<PDATABASERECORD>::iterator ReccStart, ReccEnd;
            ReccStart = PckRecordsList.begin();
            ReccEnd = PckRecordsList.end();

            for (; (ReccStart != ReccEnd); ReccStart++)
            {

                PDATABASERECORD preccPck = (*ReccStart);

                if (    (strcmp(preccPck->pszAuthor, preccApplication->pszAuthor) == 0)
                     && (strcmp(preccPck->pszApplication, preccApplication->pszApplication) == 0)
                   )
                {
                    winhCnrInsertRecords(hwndCnr,
                                       (PRECORDCORE)preccApplication,   // parent
                                       (PRECORDCORE)preccPck,
                                       preccPck->pszPackageName,
                                       CRA_RECORDREADONLY | CRA_COLLAPSED,
                                       1);
                }
            }
        }
    }

    /*
     * ID_WIMI_DATABASE_DETAILSVIEW:
     *
     */

    else if (GuiSettings.ulDatabaseView == ID_WIMI_DATABASE_DETAILSVIEW)
    {
        // switch cnr to Details view
        BEGIN_CNRINFO()
        {
            // set bitmap size
            winhCnrSetBmpOrIconSize(16, 16);
            // switch view
            winhCnrSetView(CV_DETAIL
                                | CA_DETAILSVIEWTITLES);
                                // | CA_DRAWBITMAP);
        } END_CNRINFO(hwndCnr);

        // the container already has the details fields we need,
        // so just insert the records now
        list<PDATABASERECORD>::iterator ReccStart =PckRecordsList.begin(),
                                        ReccEnd = PckRecordsList.end();

        for (; (ReccStart != ReccEnd); ReccStart++)
        {

            PDATABASERECORD preccPck = (*ReccStart);
            winhCnrInsertRecords(hwndCnr,
                               (PRECORDCORE)NULL,       // no parent for Details view
                               (PRECORDCORE)preccPck,
                               preccPck->pszPackageName,
                               CRA_RECORDREADONLY | CRA_COLLAPSED,
                               1);
        }
    }
}

/*
 *@@ fnwpDatabaseMain:
 *      window procedure for the main "database" dialog.
 */

MRESULT EXPENTRY fnwpDatabaseMain(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *
         */

        case WM_INITDLG:
        {
            HENUM henum;
            HWND hwndItem;
            CHAR szFont[20];
            ULONG cbFont;

            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            // set font for all the dialog controls
            if (doshIsWarp4())
                strcpy(szFont, "9.WarpSans");
            else
                strcpy(szFont, "8.Helv");
            cbFont = strlen(szFont)+1;
            henum = WinBeginEnumWindows(hwndDlg);
            while (hwndItem = WinGetNextWindow(henum))
            {
                SHORT sID = WinQueryWindowUShort(hwndItem, QWS_ID);
                if ((sID > -2) && (sID < 2000))
                    WinSetPresParam(hwndItem,
                            PP_FONTNAMESIZE,
                            cbFont,
                            szFont);
            }
            WinEndEnumWindows(henum);
        break; }

        /*
         * WM_CONTROL:
         *
         */

        case WM_CONTROL: {
            USHORT   usID = SHORT1FROMMP(mp1);
            USHORT   usNotifyCode = SHORT2FROMMP(mp1);
            switch (usID)
            {
                case ID_WIDI_CNR:
                    switch (usNotifyCode)
                    {

                        /*
                         * CN_EMPHASIS:
                         *      selection changed: update display
                         */

                        case CN_EMPHASIS: {
                            // get cnr notification struct
                            PNOTIFYRECORDEMPHASIS pnre = (PNOTIFYRECORDEMPHASIS)mp2;

                            if (pnre)
                                if (    (pnre->fEmphasisMask & CRA_SELECTED)
                                     && (pnre->pRecord)
                                   )
                                {
                                    PDATABASERECORD precc =
                                            (PDATABASERECORD)pnre->pRecord;
                                    CHAR    szVersion[100] = "";
                                    PSZ     pszVersion = 0,
                                            pszInstallDate = 0,
                                            pszTargetDir = 0;
                                    CHAR    szFiles[200] = "";

                                    // set the following for both apps and pcks
                                    WinSetDlgItemText(hwndDlg, ID_WIDI_DB_AUTHOR,
                                                      precc->pszAuthor);

                                    // set the following for pcks only
                                    if (precc->pPckInfo)
                                    {
                                        sprintf(szVersion, "%d.%d",
                                                precc->ulVersionMajor,
                                                precc->ulVersionMinor);
                                        pszTargetDir = precc->pPckInfo->szTargetPath;
                                        sprintf(szFiles, "%d",
                                                precc->pPckInfo->PackHeader.files);
                                    }
                                    WinSetDlgItemText(hwndDlg, ID_WIDI_DB_VERSION,
                                                      szVersion);
                                    WinSetDlgItemText(hwndDlg, ID_WIDI_DB_INSTALLDATE,
                                                      pszInstallDate);
                                    WinSetDlgItemText(hwndDlg, ID_WIDI_DB_TARGETDIR,
                                                      pszTargetDir);
                                    WinSetDlgItemText(hwndDlg, ID_WIDI_DB_FILES,
                                                      szFiles);
                                }
                        break; }

                        /*
                         * CN_CONTEXTMENU:
                         *      context menu requested for item;
                         *      mp2 has record core or NULL for
                         *      whitespace
                         */

                        case CN_CONTEXTMENU: {
                            USHORT usMenuID = 0;

                            preccSource = (PDATABASERECORD)mp2;
                            if (preccSource)
                            {
                                // on context record:
                                if (preccSource->pPckInfo)
                                    // on package:
                                    usMenuID = ID_WIM_DATABASE_PACKAGE;
                                else
                                    // on application:
                                    usMenuID = ID_WIM_DATABASE_APP;
                            }
                            else
                                // cnr whitespace:
                                usMenuID = ID_WIM_DATABASE_WHITESPACE;

                            HWND hPopupMenu = WinLoadMenu(hwndDlg,
                                                          NULL,     // our EXE
                                                          usMenuID);

                            WinCheckMenuItem(hPopupMenu,
                                             GuiSettings.ulDatabaseView,
                                             TRUE);

                            winhCnrShowContextMenu(WinWindowFromID(hwndDlg, usID),
                                                   (PRECORDCORE)preccSource,
                                                   hPopupMenu,
                                                   hwndDlg);
                        break; }

                        /*
                         * CN_ENTER:
                         *      enter key or double-click:
                         *      show file list
                         */

                        case CN_ENTER: {
                            PNOTIFYRECORDENTER pnre = PNOTIFYRECORDENTER(mp2);
                            if (pnre)
                                if (pnre->pRecord)
                                {
                                    PDATABASERECORD precc =
                                            (PDATABASERECORD)(pnre->pRecord);
                                    if (precc->pPckInfo)
                                        OpenFileList(precc);
                                }
                        break; }

                    } // end switch (usNotifyCode)

                break;      // case ID_WIDI_CNR
            }
        break; }

        /*
         * WM_MENUEND:
         *      if the context menu is dismissed, we'll need
         *      to remove the cnr source emphasis which was
         *      set above when showing the context menu.
         */

        case WM_MENUEND: {
            winhCnrSetSourceEmphasis(WinWindowFromID(hwndDlg, ID_WIDI_CNR),
                                     preccSource,
                                     FALSE);
        break; }

        /*
         * WM_COMMAND:
         *
         */

        case WM_COMMAND: {
            ULONG       ulCmd = (ULONG)mp1;

            switch (ulCmd)
            {
                /*
                 * ID_WIMI_DATABASE_SHOWPCK:
                 *      open "File list" window
                 */

                case ID_WIMI_DATABASE_SHOWPCK:
                    OpenFileList(preccSource);
                break;

                /*
                 * ID_WIMI_DATABASE_VERIFYAPP etc.:
                 *      open "Status" window, which
                 *      starts the Database thread
                 */

                case ID_WIMI_DATABASE_VERIFYAPP:
                case ID_WIMI_DATABASE_VERIFYPCK:
                case ID_WIMI_DATABASE_DEINSTALLAPP:
                case ID_WIMI_DATABASE_DEINSTALLPCK: {
                    list<PackageInfo*> PckInfoList;

                    // build list
                    switch (ulCmd)
                    {
                        // work on application:
                        case ID_WIMI_DATABASE_VERIFYAPP:
                        case ID_WIMI_DATABASE_DEINSTALLAPP: {
                            // find package infos which match author
                            // and application by building a comparison
                            // string
                            PSZ pszAuthorAndApp = (PSZ)malloc(strlen(preccSource->pszAuthor)
                                                              + strlen(preccSource->pszApplication)
                                                              + 3);
                            sprintf(pszAuthorAndApp,
                                    "%s\\%s\\",
                                    preccSource->pszAuthor,
                                    preccSource->pszApplication);
                            ULONG cbAuthorAndApp = strlen(pszAuthorAndApp);
                            list<PackageInfo*>::iterator PckStart = WpiGlobals.DBPckList.begin(),
                                                         PckEnd = WpiGlobals.DBPckList.end();
                            for (; PckStart != PckEnd; PckStart++)
                            {
                                PackageInfo* ppiThis = (*PckStart);
                                // check this package ID's author and application
                                if (strlen(ppiThis->pszID) > cbAuthorAndApp)
                                    if (memcmp(ppiThis->pszID, pszAuthorAndApp, cbAuthorAndApp) == 0)
                                        // matches: store this package in the list
                                        PckInfoList.push_back(ppiThis);
                            }
                            free(pszAuthorAndApp);
                        break; }

                        // work on one package only: that's easy
                        case ID_WIMI_DATABASE_DEINSTALLPCK:
                        case ID_WIMI_DATABASE_VERIFYPCK:
                            // just store the package info of the current recc
                            PckInfoList.push_back(preccSource->pPckInfo);
                    } // end switch (ulCmd)

                    BOOL    fStartThread = TRUE;
                    ULONG   ulThreadCommand = DBACTION_DEINSTALL;
                    PSZ     pszWindowTitle = 0;
                    LONG    lrc = 0;
                    // confirmations/setup

                    switch (ulCmd)
                    {
                        case ID_WIMI_DATABASE_DEINSTALLPCK:
                            lrc = guiConfirmDeinstall(hwndDlg, &PckInfoList);
                            if (lrc < 0)
                                // cancel:
                                fStartThread = FALSE;
                            else
                                pszWindowTitle = NLSStrings.pszDeinstallingPackage; // "Deinstall Package";
                        break;

                        case ID_WIMI_DATABASE_DEINSTALLAPP:
                            lrc = guiConfirmDeinstall(hwndDlg, &PckInfoList);
                            if (lrc < 0)
                                // cancel:
                                fStartThread = FALSE;
                            else
                                pszWindowTitle = NLSStrings.pszDeinstallingApplication; // "Deinstall Application";

                        break;

                        case ID_WIMI_DATABASE_VERIFYPCK:
                            pszWindowTitle = NLSStrings.pszVerifyingPackage; // "Verify Package";
                            ulThreadCommand = DBACTION_VERIFY;
                        break;

                        case ID_WIMI_DATABASE_VERIFYAPP:
                            pszWindowTitle = NLSStrings.pszVerifyingApplication;// "Verify Application";
                            ulThreadCommand = DBACTION_VERIFY;
                        break;
                    } // end switch (ulCmd)

                    if (fStartThread)
                    {
                        // for all verify "package" and "application",
                        // plus deinstall "package"  and "application",
                        // we load the database thread status window
                        // (fnwpDatabaseStatus)
                        DATABASESTATUSINFO dbsi(ulThreadCommand,
                                                lrc,
                                                &PckInfoList,
                                                pszWindowTitle,
                                                WinWindowFromID(hwndDlg, ID_WIDI_CNR));
                                                        // container window to update

                        preccSource->hwndStatus = WinLoadDlg(HWND_DESKTOP,   // parent
                                                             hwndDlg,        // owner
                                                             fnwpDatabaseStatus,
                                                             NULLHANDLE,      // our own EXE
                                                             ID_WID_DATABASETHREAD,
                                                             // pass the command to the dlg
                                                             (PVOID)&dbsi);
                        WinShowWindow(preccSource->hwndStatus, TRUE);
                                // non-modal
                    }

                    // since we have a block here, the
                    // list will be destroyed automatically now
                break; }

                case ID_WIMI_DATABASE_REMOVEPCK:
                    if (guiMessageBoxMsg(hwndDlg,
                                         108,       // "Warning"
                                         &(preccSource->recc.pszIcon),
                                         1,
                                         122,
                                         MB_ICONQUESTION | MB_YESNO | MB_MOVEABLE)
                                == MBID_YES)
                        datRemovePackage(preccSource->pPckInfo);
                            // this will call guiDatabaseCallback
                break;

                /*
                 * ID_WIMI_DATABASE_TREEVIEW:
                 * ID_WIMI_DATABASE_DETAILSVIEW:
                 *      switch container view
                 */

                case ID_WIMI_DATABASE_TREEVIEW:
                case ID_WIMI_DATABASE_DETAILSVIEW:
                    if (GuiSettings.ulDatabaseView != ulCmd)
                    {
                        // view changed:
                        GuiSettings.ulDatabaseView = ulCmd;
                        FillDatabaseContainer(WinWindowFromID(hwndDlg, ID_WIDI_CNR));
                        SaveSettings();
                    }
                break;

                case ID_WIDI_CANCELBUTTON:
                    WinDismissDlg(hwndDlg, ulCmd);
                break;
            }
        break; }

        /*
         * WPIM_PACKAGEREMOVED:
         *      this gets posted by guiDatabaseCallback.
         *      mp1 has the PackageInfo* which has been
         *      removed by the Database thread.
         */

        case WPIM_PACKAGEREMOVED: {
            PackageInfo* pPckInfoRemove = (PackageInfo*)mp1;
            if (pPckInfoRemove)
            {
                // get the record core from the package info
                PDATABASERECORD preccRemove = (PDATABASERECORD)pPckInfoRemove->pvGuiData;
                if (preccRemove)
                {
                    HWND hwndDatabaseCnr = WinWindowFromID(hwndDlg, ID_WIDI_CNR);
                    PDATABASERECORD preccApplication =
                                (PDATABASERECORD)winhCnrQueryParentRecord(hwndDatabaseCnr,
                                                                          preccRemove);

                    FreeDatabaseRecord(hwndDatabaseCnr, preccRemove);

                    // now check if the application record has
                    // any more children (i.e. any more packages)
                    if (preccApplication)
                        if (winhCnrQueryFirstChildRecord(hwndDatabaseCnr, preccApplication)
                                        == NULL)
                            FreeDatabaseRecord(hwndDatabaseCnr, preccApplication);
                }
            }
        break; }

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }
    return (mrc);
}

HWND hwndDatabaseMain = NULLHANDLE;

/*
 *@@ guiBeginDatabaseMode:
 *      this gets called from main() when WarpIN is
 *      started without any archive as a parameter.
 *      Before calling this, main() has created a
 *      list of all the packages which are currently
 *      stored in the database in WPIGLOBALS.pDBPckList.
 *
 *      This routine does not get called if the
 *      database is empty.
 *
 *      This routine is responsible for displaying
 *      the installed packages and calling
 *      datVerifyApplication, datVerifyPackage,
 *      datDeinstallApplication, and datDeinstallPackage
 *      on them, if the user desires this.
 *
 *      For the PM version of this function, this
 *      is done from the Database Status dialog window
 *      (fnwpDatabaseStatus), which is created from
 *      this function.
 *
 *      Required callback.
 */

BOOL guiBeginDatabaseMode()
{
    // create main database window
    hwndDatabaseMain = WinLoadDlg(HWND_DESKTOP,
                                   HWND_DESKTOP,
                                   fnwpDatabaseMain,
                                   NULLHANDLE,      // our own EXE
                                   ID_WID_DATABASE,
                                   NULL);
    winhCenterWindow(hwndDatabaseMain);

    HWND hwndDatabaseCnr = WinWindowFromID(hwndDatabaseMain, ID_WIDI_CNR);

    // set up the container for Details view

    // 1) set up data for Details view columns

    // set up data for Details view columns
    XFIELDINFO      xfi[9];
    memset(&xfi, 0, sizeof(xfi));

    int i = 0;

    xfi[i].ulFieldOffset = FIELDOFFSET(RECORDCORE, hptrIcon);
    xfi[i].pszColumnTitle = "";
    xfi[i].ulDataType = CFA_BITMAPORICON;       // it's an icon, really
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszApplication);
    xfi[i].pszColumnTitle = "Application";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(RECORDCORE, pszIcon);
    xfi[i].pszColumnTitle = NLSStrings.pszPackageName; // "Package name";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszVersion);
    xfi[i].pszColumnTitle = NLSStrings.pszVersion; // "Version";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_RIGHT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszAuthor);
    xfi[i].pszColumnTitle = NLSStrings.pszAuthor; // "Author";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pszTargetPath);
    xfi[i].pszColumnTitle = NLSStrings.pszTargetPath; // "Target directory";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, ulFiles);
    xfi[i].pszColumnTitle = NLSStrings.pszFiles; // "Files";
    xfi[i].ulDataType = CFA_ULONG;
    xfi[i++].ulOrientation = CFA_RIGHT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, ulTotalSize);
    xfi[i].pszColumnTitle = NLSStrings.pszSize; // "Size";
    xfi[i].ulDataType = CFA_ULONG;
    xfi[i++].ulOrientation = CFA_RIGHT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, cdateInstall);
    xfi[i].pszColumnTitle = NLSStrings.pszInstallDate; // "Install date";
    xfi[i].ulDataType = CFA_DATE;
    xfi[i++].ulOrientation = CFA_RIGHT;

    PFIELDINFO pFieldInfoLast =
             winhCnrSetFieldInfos(hwndDatabaseCnr,
                                  &xfi[0],
                                  i,        // array item count
                                  TRUE,     // draw lines
                                  3);       // PFIELDINFO index to return

    // set split bar
    BEGIN_CNRINFO()
    {
        winhCnrSetSplitBarAfter(pFieldInfoLast);
        winhCnrSetSplitBarPos(200);
    } END_CNRINFO(hwndDatabaseCnr);

    // 2) create record cores for all applications
    list<PackageInfo*>::iterator PckStart = WpiGlobals.DBPckList.begin(),
                                 PckEnd = WpiGlobals.DBPckList.end();

    for (; (PckStart != PckEnd); PckStart++)
    {
        PPACKAGEID pPckID = datSplitPackageID((**PckStart).pszID);

        // check if this application exists in the list already
        BOOL fFound = FALSE;
        list<PDATABASERECORD>::iterator AppsStart = AppRecordsList.begin(),
                                        AppsEnd = AppRecordsList.end();
        for (; (AppsStart != AppsEnd); AppsStart++)
            if (strcmp((**AppsStart).pszApplication,
                       pPckID->pszApplication)
                    == 0)
                fFound = TRUE;

        if (!fFound)
        {
            // not found:
            // create application record core
            PDATABASERECORD preccApplication = CreateDBAppRecord(hwndDatabaseCnr, pPckID);

            // store in global list
            AppRecordsList.push_back(preccApplication);

        } // end if (!fFound)

        delete pPckID;
    } // for (; (PckStart != PckEnd); PckStart++)

    // 3) create record cores for all packages
    PckStart = WpiGlobals.DBPckList.begin();
    PckEnd = WpiGlobals.DBPckList.end();
    for (; (PckStart != PckEnd); PckStart++)
    {
        PDATABASERECORD preccPck = CreateDBPckRecord(hwndDatabaseCnr, *PckStart);

        // store in global list
        PckRecordsList.push_back(preccPck);
    } // for (; (PckStart != PckEnd); PckStart++)

    FillDatabaseContainer(hwndDatabaseCnr);

    // hide the startup window
    WinDestroyWindow(hwndWarpInStartup);

    WinProcessDlg(hwndDatabaseMain);
    WinDestroyWindow(hwndDatabaseMain);
}

/*******************************************************************
 *                                                                 *
 *  PM implementation part 3:                                      *
 *      database callbacks for verification/deinstallation         *
 *                                                                 *
 ******************************************************************/

/*
 *@@ fnwpDatabaseStatus:
 *      this is the window procedure for the Database
 *      thread status window. This runs on thread 1 and
 *      receives messages from guiDatabaseCallback
 *      (running on the Database thread).
 *
 *      When a window is created using this function,
 *      the Database thread is started anew. mp2 of
 *      WM_INITDIALOG (pCreate of WinLoadDlg) must
 *      have a command value of what we're supposed
 *      to do here.
 */

MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    PDATABASESTATUSINFO pdbsi = (PDATABASESTATUSINFO)WinQueryWindowPtr(hwndDlg, QWL_USER);
    static PackageInfo* pPckInfoLast = NULL;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      since this can only be created when WM_COMMAND
         *      is evaluated in fnwpDatabaseMain, and since this
         *      must come from a context menu item, the global
         *      variable preccSource still has the DATABASERECORD
         *      of the application or package which should be
         *      worked on.
         *
         *      mp2 (pCreate) has the DATABASESTATUSINFO structure
         *      for this dialog.
         */

        case WM_INITDLG:
        {
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            pPckInfoLast = NULL;

            // create progress bar
            winhProgressBarFromStatic(WinWindowFromID(hwndDlg,
                                                      ID_WIDI_DBT_PROGRESSBAR),
                                      PBA_ALIGNCENTER | PBA_BUTTONSTYLE);

            // get the create struct which was passed in mp2
            PDATABASESTATUSINFO pdbsiPassed = (PDATABASESTATUSINFO)mp2;
            // copy that structure
            pdbsi = new DATABASESTATUSINFO(*pdbsiPassed);

            WinSetWindowPtr(hwndDlg, QWL_USER, pdbsi);

            WinSetWindowText(hwndDlg, pdbsi->pszWindowTitle);

            switch (pdbsi->ulCmd)
            {
                case DBACTION_VERIFY:
                    datVerifyPackages(pdbsi->pPckInfoList,
                                      (ULONG)hwndDlg);       // passed to callback
                break;

                case DBACTION_DEINSTALL:
                    datDeinstallPackages(pdbsi->pPckInfoList,
                                         pdbsi->ulFlags,
                                         (ULONG)hwndDlg);    // passed to callback
                break;
            }

            // disable "Save Report As" button
            winhEnableDlgItem(hwndDlg, ID_WIDI_DBT_SAVEREPORTAS, FALSE);
        break; }

        /*
         * WPIM_UPDATEPACKAGE:
         *      this gets posted by guiDatabaseCallback.
         *      mp1 has a PackageInfo* (or NULL when
         *      we're done).
         *      mp2 is a PULONG to a pseudo-semaphore
         *      which guiDatabaseCallback is waiting
         *      for to be set to 1.
         */

        case WPIM_UPDATEPACKAGE: {

            if (pPckInfoLast)
            {
                // running for second package (or we're done):
                PDATABASERECORD precc = (PDATABASERECORD)(pPckInfoLast->pvGuiData);
                if ((precc) && (pdbsi->hwndMainCnr))
                {
                    // non-modal mode:
                    if (pPckInfoLast->ulStatus == 1)
                        // error occured:
                        precc->recc.hptrIcon
                            = precc->recc.hptrMiniIcon
                            = hptrError;
                    else
                        // status should be 2:
                        precc->recc.hptrIcon
                            = precc->recc.hptrMiniIcon
                            = hptrOK;
                    WinSendMsg(pdbsi->hwndMainCnr,
                               CM_INVALIDATERECORD,
                               (MPARAM)(&precc),
                               MPFROM2SHORT(1, CMA_ERASE));
                }
            }

            PackageInfo* pPckInfo = (PackageInfo*)mp1;
            pPckInfoLast = pPckInfo;

            if (pPckInfo)
            {
                PSZ     pszApplication = NULL,
                        pszPackage = NULL;

                PPACKAGEID pPckID = datSplitPackageID(pPckInfo->pszID);

                pdbsi->ulTotalFiles = pPckInfo->PackHeader.files;

                if (pPckID->pszApplication)
                    WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_APPLICATION,
                                      pPckID->pszApplication);

                if (pPckID->pszPackage)
                    WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_PACKAGE,
                                      pPckID->pszPackage);

                delete pPckID;
            }
            else
            {
                // pPckInfo == NULL: we're done
                WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_PROGRESSBAR),
                           WM_UPDATEPROGRESSBAR,
                           (MPARAM)1,
                           (MPARAM)1);      // 100%

                // if no errors were found:
                if (!pdbsi->fErrorsFound)
                    WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_REPORTMLE),
                               MLM_INSERT,
                               (MPARAM)NLSStrings.pszDone,
                               MPNULL);
                else
                   // enable "Save Report As" button
                   winhEnableDlgItem(hwndDlg, ID_WIDI_DBT_SAVEREPORTAS, TRUE);

                // rename "Cancel" button
                WinSetDlgItemText(hwndDlg, DID_CANCEL, "~Close"); // xxx
                // empty "File" static text
                WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_FILE, "");

                // for next run:
                pPckInfoLast = NULL;
            }
            // set pseudo-semaphore
            *((PULONG)mp2) = 1;
        break; }

        /*
         * WPIM_UPDATEFILE:
         *      this gets posted by guiDatabaseCallback.
         *      mp1 has the WIFileHeader*.
         *      mp2 is a PULONG to a pseudo-semaphore
         *      which guiDatabaseCallback is waiting
         *      for to be set to 1.
         */

        case WPIM_UPDATEFILE: {
            WIFileHeader* pHeader = (WIFileHeader*)mp1;

            WinSetDlgItemText(hwndDlg, ID_WIDI_DBT_FILE,
                              pHeader->name);

            WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_PROGRESSBAR),
                       WM_UPDATEPROGRESSBAR,
                       (MPARAM)(pdbsi->ulCurrentFile),      // current value
                       (MPARAM)(pdbsi->ulTotalFiles));    // maximum value

            CHAR szForMLE[1000];
            sprintf(szForMLE,
                    "%s \%s\\%s\n",
                    (pdbsi->ulCmd == DBACTION_VERIFY)
                        ? NLSStrings.pszVerifying
                        : NLSStrings.pszDeleting,
                    pPckInfoLast->szTargetPath,
                    pHeader->name);

            WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_REPORTMLE),
                       MLM_INSERT,
                       (MPARAM)szForMLE,
                       MPNULL);

            pdbsi->ulCurrentFile++;

            // set pseudo-semaphore
            *((PULONG)mp2) = 1;
        break; }

        /*
         * WPIM_ADDSTRING:
         *      generic message from guiDatabaseCallback to
         *      add a string. mp1 points to a static buffer
         *      containing the string.
         *      mp2 is a PULONG to a pseudo-semaphore
         *      which guiDatabaseCallback is waiting
         *      for to be set to 1.
         */

        case WPIM_ADDSTRING:
        {
            WinSendMsg(WinWindowFromID(hwndDlg, ID_WIDI_DBT_REPORTMLE),
                       MLM_INSERT,
                       (MPARAM)mp1,
                       MPNULL);
            // set pseudo-semaphore
            *((PULONG)mp2) = 1;
        break; }


        /*
         * WM_COMMAND:
         *      intercept the "Save as" button
         */

        case WM_COMMAND: {
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIDI_DBT_SAVEREPORTAS: {
                    FILEDLG fd;
                    memset(&fd, 0, sizeof(FILEDLG));
                    fd.cbSize = sizeof(FILEDLG);
                    fd.fl = FDS_SAVEAS_DIALOG
                              | FDS_ENABLEFILELB
                              | FDS_CENTER;
                    sprintf(fd.szFullFile, "*.log");
                    if (    WinFileDlg(HWND_DESKTOP,    // parent
                                       hwndDlg,
                                       &fd)
                        && (fd.lReturn == DID_OK)
                       )
                    {
                        PSZ pszLogText = winhQueryWindowText(WinWindowFromID(hwndDlg,
                                                                             ID_WIDI_DBT_REPORTMLE));
                        doshWriteTextFile(fd.szFullFile, pszLogText, TRUE);
                        free(pszLogText);
                    }
                break; }

                default:
                    mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        break; }

        case WM_DESTROY:
            delete pdbsi;
            WinSetWindowPtr(hwndDlg, QWL_USER, NULL);
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ guiDatabaseCallback:
 *      this callback gets called from the Database
 *      thread during de-installation/verification.
 *
 *      Required callback.
 *
 *      Note: this is _not_ running on thread 1, but
 *      on the Database thread. Be careful with global
 *      variables.
 *
 *      The callbacks come in as follows:
 *
 *      1) DBC_NEXTPACKAGE with pPckInfo pointing to the package,
 *
 *      2) DBC_NEXTFILE with pHeader pointing to the current header;
 *
 *      3) _possibly_ one of DBC_FILENOTFOUND, DBC_EXISTINGFILENEWER,
 *         or DBC_EXISTINGFILEOLDER; if the existing file is OK,
 *         no callback occurs;
 *
 *      4) next file: go back to 2).
 *
 *      5) next package (only for datVerifyApplication): go back to 1).
 *
 *      5) After all files have been processed, another DBC_NEXTPACKAGE
 *         comes in with (pPckInfo == 0).
 */

ULONG guiDatabaseCallback(ULONG ulReportCode,       // in: DBC_* code
                          PackageInfo* pPckInfo,    // in: package info (mostly constant)
                          WIFileHeader* pHeader,    // in: file header from database
                          ULONG ulExtra,            // in: extra info
                          ULONG ulUser)             // in: ulUser passed to database thread;
                                                    // this has the HWND of the status window
{
    HWND        hwndStatus = (HWND)ulUser;
    ULONG       ulPseudoSem = 0;
    CHAR        szString[1000] = "",
                szFullPath[2*CCHMAXPATH] = "",
                szExistingDate[40] = "",
                szExistingTime[40] = "",
                szDatabaseDate[40] = "",
                szDatabaseTime[40] = "";

    if ((pPckInfo) && (pHeader))
        sprintf(szFullPath, "%s\\%s",
                pPckInfo->szTargetPath,
                pHeader->name);

    switch (ulReportCode)
    {
        /*
         * DBC_PACKAGEREMOVED:
         *      this results from datRemovePackage.
         *      Note that in this case, this callback does not
         *      necessarily get called from the Database thread,
         *      but from any thread which calls datRemovePackage.
         *      pPckInfo has the package that was just removed
         *      from the database.
         */

        case DBC_PACKAGEREMOVED:
            // results from datRemovePackage;
            // in this case, we're running on the
            // thread which called that func, which
            // _can_, but does not have to be the
            // Database thread
            WinPostMsg(hwndDatabaseMain,        // not hwndStatus!
                    WPIM_PACKAGEREMOVED,
                    (MPARAM)pPckInfo,
                    (MPARAM)NULL);
            // do not wait
        break;

        /*
         * DBC_NEXTPACKAGE:
         *      Database thread is progressing to next package (pPckInfo).
         *      In this case, pHeader and ulExtra are NULL.
         *      If pPckInfo is NULL, the Database thread is done.
         */

        case DBC_NEXTPACKAGE:
            WinPostMsg(hwndStatus, WPIM_UPDATEPACKAGE,
                    (MPARAM)pPckInfo,           // warning: can be NULL when done
                    (MPARAM)&ulPseudoSem);

            // wait for thread 1 to set the pseudo-semaphore
            while (ulPseudoSem == 0)
                DosSleep(10);
        break;

        /*
         * DBC_NEXTFILE:
         *      Database thread is progressing
         *      to next file (useful for progress bars).
         *      This comes in for _every_ file in the
         *      database file list; there might be a second
         *      call with of the following error codes afterwards.
         *      pPckInfo has the same package info as in the
         *      previous call with DBC_NEXTPACKAGE, pHeader
         *      has the current file.
         *      ulExtra is NULL.
         */

        case DBC_NEXTFILE:
            WinPostMsg(hwndStatus, WPIM_UPDATEFILE,
                    (MPARAM)pHeader,
                    (MPARAM)&ulPseudoSem);
            // wait for thread 1 to set the pseudo-semaphore
            while (ulPseudoSem == 0)
                DosSleep(10);
        break;

        /*
         * DBC_OBJECTDELETED:
         *      Database thread has successfully deleted a WPS object;
         *      pPckInfo has the package for which the class was registered,
         *      ulExtra is a PSZ to the original class name.
         */

        case DBC_OBJECTDELETED:
            sprintf(szString, NLSStrings.pszObjectDeleted, // "Deleted object \"%s\"\n",
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_CLASSDEREGISTERED:
         *      Database thread has successfully deregistered a class;
         *      pPckInfo has the package for which the class was registered,
         *      ulExtra is a PSZ to the class name.
         */

        case DBC_CLASSDEREGISTERED:
            sprintf(szString, NLSStrings.pszClassDeregistered, // "Deregistered class \"%s\"\n",
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_CLASSUNREPLACED:
         *      Database thread has successfully unreplaced a class;
         *      pPckInfo has the package for which the class was registered,
         *      ulExtra is a PSZ to the original class name.
         */

        case DBC_CLASSUNREPLACED:
            sprintf(szString, NLSStrings.pszClassUnreplaced, // "Unreplaced class \"%s\"\n",
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_CFGSYSUNDONE:
         *      Database thread has successfully undone a CONFIG.SYS change.
         *      pPckInfo has the package for which the change was made,
         *      pHeader is NULL, and ulExtra is a a pointer to the
         *      CfgSysManip in question.
         */

        case DBC_CFGSYSUNDONE:
            sprintf(szString, NLSStrings.pszCfgSysUndone,
                    ((CfgSysManip*)(ulExtra))->pszNewLine);
        break;

        /*
         * DBC_FILENOTFOUND:
         *      a file was not found on disk which
         *      is in the database of installed files though
         *      (verify mode only).
         *      pPckInfo has the same package info as in the
         *      previous call with DBC_NEXTPACKAGE, pHeader
         *      has the current file (as in the previous DBC_NEXTFILE
         *      call).
         *      ulExtra is NULL.
         */

        case DBC_FILENOTFOUND:
            sprintf(szString, NLSStrings.pszFileNotFound,    // "File \"%s\" not found"\n",
                    szFullPath);
        break;

        /*
         * DBC_EXISTINGFILENEWER:
         *      the last write date of a file on
         *      disk is newer than that stored in the database
         *      (verify mode only).
         *      This is probably not serious.
         *      pPckInfo has the same package info as in the
         *      previous call with DBC_NEXTPACKAGE, pHeader
         *      has the current file (as in the previous DBC_NEXTFILE
         *      call).
         *      In this case, ulExtra is pointer to a FILESTATUS3
         *      structure with information about the existing file on disk.         */

        case DBC_EXISTINGFILENEWER:

        /*
         * DBC_EXISTINGFILEOLDER:
         *      the last write date of a file on
         *      disk is _older_ than that stored in the database
         *      (verify mode only).
         *      This could lead to problems.
         *      pPckInfo, pHeader, and ulExtra are set like with
         *      DBC_EXISTINGFILENEWER.         */

        case DBC_EXISTINGFILEOLDER: {
            FILESTATUS3* pfs3 = (FILESTATUS3*)(ulExtra);
            FDATE       fdateLastWrite;
            FTIME       ftimeLastWrite;
            strhFileDate(szExistingDate,
                         &(pfs3->fdateLastWrite),
                         ulDateFormat, szDateSep[0]);
            strhFileTime(szExistingTime,
                         &(pfs3->ftimeLastWrite),
                         ulTimeFormat,
                         szTimeSep[0]);
            // convert file header's time information
            wpiCTime2FTime(&pHeader->lastwrite,
                           &fdateLastWrite,
                           &ftimeLastWrite);
            strhFileDate(szDatabaseDate,
                         &fdateLastWrite,
                         ulDateFormat, szDateSep[0]);
            strhFileTime(szDatabaseTime,
                         &ftimeLastWrite,
                         ulTimeFormat,
                         szTimeSep[0]);
            sprintf(szString,
                    (ulReportCode == DBC_EXISTINGFILENEWER)
                        ? NLSStrings.pszExistingFileNewer
                        : NLSStrings.pszExistingFileOlder,
                    szFullPath,
                    szExistingDate,
                    szExistingTime,
                    szDatabaseDate,
                    szDatabaseTime);
        break; }

        /*
         * DBC_DELETEERROR:
         *      a file could not be deleted
         *      (de-install mode only).
         *      pPckInfo and pHeader are set like with
         *      DBC_EXISTINGFILENEWER.
         *      In this case, ulExtra is the APIRET of DosDelete.         */

        case DBC_DELETEERROR: {
            PSZ pszSysError = doshQuerySysErrorMsg(ulExtra);
            sprintf(szString, NLSStrings.pszDeleteError,
                        // "File \"%s\" could not be deleted. OS/2 reported: \"%s\"\r"
                    szFullPath,
                    pszSysError);
            if (pszSysError)
                free (pszSysError);
        break; }

        case DBC_DELETEDIRERROR: {
            PSZ pszSysError = doshQuerySysErrorMsg(ulExtra);
            sprintf(szString, NLSStrings.pszDeleteDirError,
                        // "Directory \"%s\" could not be removed. OS/2 reported: \"%s\"\r\n",
                    pPckInfo->szTargetPath,
                    pszSysError);
            if (pszSysError)
                free (pszSysError);
        break; }

        /*
         * DBC_OBJECTDELETEERROR:
         *      a WPS object could not be deleted
         *      (de-install mode only).
         *      pPckInfo is set like DBC_EXISTINGFILENEWER,
         *      pHeader is NULL, and ulExtra is a PSZ to the object ID.
         */

        case DBC_OBJECTDELETEERROR:
            sprintf(szString, NLSStrings.pszErrorDeletingObject,
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_UNREPLACECLASSERROR:
         *      error deregistering a class (de-install mode only).
         *      pPckInfo is set like DBC_EXISTINGFILENEWER,
         *      pHeader is NULL, and ulExtra is a PSZ to the
         *      old class name.
         */

        case DBC_UNREPLACECLASSERROR:
            sprintf(szString, NLSStrings.pszErrorUnreplacingClass,
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_DEREGISTERCLASSERROR:
         *      error deregistering a class (de-install mode only).
         *      pPckInfo is set like DBC_EXISTINGFILENEWER,
         *      pHeader is NULL, and ulExtra is a PSZ to the
         *      class name.
         */

        case DBC_DEREGISTERCLASSERROR:
            sprintf(szString, NLSStrings.pszErrorDeregisteringClass,
                    (PSZ)ulExtra);
        break;

        /*
         * DBC_UNDOCFGSYSERROR:
         *      error undoing a CONFIG.SYS change (de-install mode only).
         *      pPckInfo is set like DBC_EXISTINGFILENEWER,
         *      pHeader is NULL, and ulExtra is a a pointer to the
         *      CfgSysManip in question.
         */

        case DBC_UNDOCFGSYSERROR:
            sprintf(szString, NLSStrings.pszErrorUndoingCfgSys,
                    ((CfgSysManip*)(ulExtra))->pszNewLine);
        break;
    }

    if (szString[0] != 0)
    {
        // something string-like was added for posting:
        strcat(szString, "\r\n");
        WinPostMsg(hwndStatus, WPIM_ADDSTRING,
                (MPARAM)szString,
                (MPARAM)&ulPseudoSem);
        while (ulPseudoSem == 0)
            DosSleep(10);
    }

    return (0);
}


