
/*
 * database.cpp:
 *      this contains the database handling.
 *
 *      Currently, this is implemented as a large INI file.
 *      Note that the functions in here are _not_ thread-safe,
 *      so you must make sure that only one thread at a time
 *      calls these.
 *
 *      This file Copyright (C) 1999 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_WINWORKPLACE
#include <os2.h>
#include <stdlib.h>
#include <direct.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

// #include's from XFolder /HELPERS directory
#include "dosh.h"
#include "winh.h"
#include "prfh.h"
#include "stringh.h"
#include "threads.h"

#include "warpin.h"
#include "database.h"

#include "except.h"

/*******************************************************************
 *                                                                 *
 *  Global variables                                               *
 *                                                                 *
 ******************************************************************/

extern WPIGLOBALS WpiGlobals;

PTHREADINFO         ptiDatabaseThread = NULL;

/*
 *@@ datInitialize:
 *      initialize the database.
 *      At this point, WpiGlobals.szDatabaseFile has been
 *      set by main() in warpin.cpp.
 */

BOOL datInitialize(PWPIGLOBALS pWpiGlobalsPassed)
{
    // pWpiGlobals2 = pWpiGlobalsPassed;

    CHAR szDatabaseFile[2*CCHMAXPATH];
    strcpy(szDatabaseFile, WpiGlobals.szWarpINFile);
    PSZ p = strrchr(szDatabaseFile, '\\');
    strcpy(p+1, "database.ini");
    WpiGlobals.hiniDatabase = PrfOpenProfile(WpiGlobals.habThread1,
                                  szDatabaseFile);
    return (TRUE);
}

/*******************************************************************
 *
 *  List functions:
 *      the following are used to convert the strings which are
 *      used in the database (INI file) to lists of various formats
 *      and vice versa.
 *
 *      Note that these are only for conversion. No data is queried
 *      from the database; instead, the PackageInfo structures
 *      are used.
 *
 ******************************************************************/

/*
 *@@ datAddFileHeader:
 *      this appends the given WIFileHeader to
 *      pPckInfo->pszFilesList.
 *
 *      This function updates pszFilesList and
 *      cbFilesList in pPckInfo. With each call,
 *      the buffer that pszFilesList points to
 *      is reallocated.
 *
 *      The format of each entry in pszFilesList is as follows:
 *
 +          CHAR[x]   BYTE time_t  time_t  ULONG    BYTE
 +          filename   0   writedt creatdt filesize  0
 +                                         (uncompressed)
 *      All file entries are simply stored in sequential
 *      order.
 *
 *      <B>Preconditions:</B>
 *      If this is the first call, pszFilesList and
 *      cbFilesList in pPckInfo must be 0.
 *      No check is made for whether the given file
 *      is already in the list.
 */

VOID datAddFileHeader(PackageInfo* pPckInfo, WIFileHeader* pwifh)
{
    ULONG cbFilesListOld = pPckInfo->cbFilesList;
    ULONG cbFilename = strlen(pwifh->name);

    pPckInfo->cbFilesList += (cbFilename
                              + (2 * sizeof(time_t))
                              + sizeof(ULONG)
                              + 2);
                                  // two null bytes

    PSZ pszTemp = pPckInfo->pszFilesList;
    pPckInfo->pszFilesList = (PSZ)malloc(pPckInfo->cbFilesList);

    PSZ pszTarget = pPckInfo->pszFilesList + cbFilesListOld;

    if (pszTemp)
    {
        // not first call:
        // copy old file list
        memcpy(pPckInfo->pszFilesList, pszTemp, cbFilesListOld);
        // append new file name
        free(pszTemp);
    }

    strcpy(pszTarget, pwifh->name);     // this appends null terminator
    pszTarget += (cbFilename + 1);
    memcpy(pszTarget, &(pwifh->lastwrite), sizeof(time_t));
    pszTarget += sizeof(time_t);        // with VAC++, this is a long
    memcpy(pszTarget, &(pwifh->creation), sizeof(time_t));
    pszTarget += sizeof(time_t);        // with VAC++, this is a long
    memcpy(pszTarget, &(pwifh->origsize), sizeof(ULONG));
    pszTarget += sizeof(ULONG);
    *pszTarget = 0;
}

/*
 *@@ datGetFileHeaders:
 *      quite reverse to the previous, this creates
 *      a list of WIFileHeaders from the given
 *      pPckInfo. The number of list items is returned
 *      in *pulHeaderCount.
 *
 *      Returns NULL upon errors.
 *
 *      This may be used in conjunction with datLoadPackage
 *      to get the file list for a given package ID.
 *
 *      Each file header on the list is created using new,
 *      just as the returned list itself.
 */

list<WIFileHeader*>* datGetFileHeaders(PackageInfo* pPckInfo,   // in: package info
                                       PULONG pulHeaderCount)   // out: no. of list items
                                                                // in returned list
{
    list<WIFileHeader*>*    pList = 0;
    ULONG                   ulHeaderCount = 0;

    if (pPckInfo)
        if (pPckInfo->cbFilesList)
        {
            pList = new list<WIFileHeader*>;

            PSZ p = pPckInfo->pszFilesList;

            while (p < (pPckInfo->pszFilesList + pPckInfo->cbFilesList) - 1)
            {
                WIFileHeader* pHeader = new WIFileHeader;

                // format of each entry:
                //    CHAR[x]   BYTE time_t  time_t  ULONG    BYTE
                //    filename   0   writedt creatdt filesize  0
                //                                   (uncompressed)

                // p now points to the filename
                strcpy(pHeader->name, p);
                p += strlen(p) + 1;     // go beyond null terminator

                // p now points to last write date
                pHeader->lastwrite = *((time_t*)p);
                p += sizeof(time_t);

                // p now points to last write date
                pHeader->creation = *((time_t*)p);
                p += sizeof(time_t);

                // p now points to the uncompressed file size
                pHeader->origsize = *((long*)p);

                p += sizeof(long);
                if (*p != 0)
                {
                    delete pList;
                    pList = 0;
                    break;
                }

                p++;    // go beyond last null terminator

                pList->push_back(pHeader);
                ulHeaderCount++;
            }

            if (pulHeaderCount)
                *pulHeaderCount = ulHeaderCount;
        }

    return (pList);
}

/*******************************************************************
 *                                                                 *
 *  Package functions                                              *
 *                                                                 *
 ******************************************************************/

/*
 *@@ datSplitPackageID:
 *      this splits the five-part package ID into
 *      five buffers, which are returned in a newly
 *      created PACKAGEID structure.
 *
 *      The package ID must have the following format:
 +          author\application\packagename\majorversion\minorversion
 *
 *      This returns NULL upon errors.
 *
 *      The returned structure must be freed using delete.
 */

PPACKAGEID datSplitPackageID(PSZ pszPackageID)
{
    if (pszPackageID)
    {
        PSZ     p,
                p2,
                pszAuthor,
                pszApplication,
                pszPackageName;
        ULONG   ulVersionMajor,
                ulVersionMinor;

        p = strchr(pszPackageID, '\\');
        if (p)
        {
            // valid author:
            pszAuthor = strhSubstr(pszPackageID, p);

            p2 = strchr(p+1, '\\');
            if (p2)
            {
                // valid application:
                pszApplication = strhSubstr(p+1, p2);

                p = p2;
                p2 = strchr(p+1, '\\');
                if (p2)
                {
                    // valid package name:
                    pszPackageName = strhSubstr(p+1, p2);

                    p = p2;
                    p2 = strchr(p+1, '\\');
                    if (p2)
                    {
                        // valid major version:
                        PSZ pszMajorVersion = strhSubstr(p+1, p2);
                        sscanf(pszMajorVersion, "%d", &ulVersionMajor);
                        free(pszMajorVersion);

                        p = p2;
                        if (strlen(p+1))
                        {
                            sscanf(p+1, "%d", &ulVersionMinor);

                            PPACKAGEID pPckID = new PACKAGEID;

                            pPckID->pszAuthor = pszAuthor;
                            pPckID->pszApplication = pszApplication;
                            pPckID->pszPackage = pszPackageName;
                            pPckID->ulVersionMajor = ulVersionMajor;
                            pPckID->ulVersionMinor = ulVersionMinor;

                            return (pPckID);
                        }
                    }
                    free(pszPackageName);
                }
                free(pszApplication);
            }
            free(pszAuthor);
        }
    }

    return (NULL);
}

/*
 *@@ datEnumPackages:
 *      this enumerates the packages which are stored in
 *      the database.
 *
 *      This works in two modes:
 *      --  If pszPackageSearch is NULL, the first package
 *          in the database is returned.
 *      --  If pszPackageSearch is != NULL, the package
 *          which comes after pszPackageSearch in the
 *          database is returned.
 *
 *      In any case, this function returns a PSZ to a
 *      new buffer, which must be free()'d after use.
 *      If no (more) packages are found, NULL is returned.
 *
 *      The return value and pszPackageIDSearch are both
 *      in the "package ID" format (five-part package ID,
 *      separated by four backslashes).
 */

PSZ datEnumPackages(PSZ pszPackageIDSearch)
{
    PSZ pszReturn = NULL;

    PSZ pszPackageList = prfhQueryKeysForApp(WpiGlobals.hiniDatabase,
                                             NULL); // query applications list

    if (pszPackageList)
    {
        if (pszPackageIDSearch == NULL)
            // first package queried:
            pszReturn = strdup(pszPackageList); // first app is null-terminated
        else
        {
            // search list for pszPackageSearch
            PSZ pPackage = pszPackageList;

            while (*pPackage != 0)
            {
                // pPackage has the current key now
                if (strcmp(pszPackageIDSearch, pPackage) == 0)
                {
                    // pPackage found: return next
                    pPackage += strlen(pPackage)+1;
                    if (*pPackage)
                        // not last:
                        pszReturn = strdup(pPackage);

                    // in any case, stop search
                    break;
                }

                // else: next package (application)
                pPackage += strlen(pPackage)+1;
            }
        }
        free(pszPackageList);
    }

    // return found package or NULL
    return (pszReturn);
}

/*
 *@@ datFindPackage:
 *      this returns the five-part package ID
 *      for the package which matches the
 *      given attribute. Since WarpIN does
 *      not allow several packages with the
 *      same author/application/name, but
 *      different version numbers, the
 *      returned information is guaranteed
 *      to be unique.
 *
 *      This returns NULL if no matching package was found.
 *
 *      The return value must be free()'d
 *      after use.
 */

PSZ datFindPackage(PSZ pszAuthor,
                   PSZ pszApplication,
                   PSZ pszPackageName)
{
    PSZ pszReturn = NULL;

    CHAR szCompare[2000];
    sprintf(szCompare, "%s\\%s\\%s\\", pszAuthor,
                                       pszApplication,
                                       pszPackageName);

    ULONG cbCompare = strlen(szCompare);

    PSZ pszPackageList = prfhQueryKeysForApp(WpiGlobals.hiniDatabase,
                                             NULL); // query applications list

    if (pszPackageList)
    {
        PSZ pPackageID = pszPackageList;

        while (*pPackageID != 0)
        {
            // pPackageID has the current application (== package ID) now
            if (strlen(pPackageID) > cbCompare)
                // compare the first part
                if (memcmp(pPackageID, szCompare, cbCompare) == 0)
                {
                    // pPackageID found: return
                    pszReturn = strdup(pPackageID);
                    // stop search
                    break;
                }

            // else: next package (application)
            pPackageID += strlen(pPackageID)+1;
        }
        free(pszPackageList);
    }

    // return found package or NULL
    return (pszReturn);
}

/*
 *@@ datRemovePackage:
 *      this removes a package from the database.
 *      Note that this removes _only_ the database
 *      information. It does _not_ de-install the
 *      package (which is not possible after calling
 *      this function, because all database info
 *      for that package will be lost).
 *
 *      This calls guiDatabaseCallback with DBC_PACKAGEREMOVED.
 *
 *      Note that pPckInfo is removed from WPIGLOBALS.DBPckList,
 *      but is not deleted itself. It's still valid after
 *      this call.
 *
 *      Returns FALSE upon errors.
 */

BOOL datRemovePackage(PackageInfo* ppiRemove)   // in: package to remove
{
    BOOL brc = FALSE;
    if (ppiRemove)
        if (ppiRemove->pszID)
        {
            if (PrfWriteProfileString(WpiGlobals.hiniDatabase,
                                      ppiRemove->pszID,
                                      NULL,             // delete whole application
                                      NULL))
            {
                // OK, package deleted from database:

                list<PackageInfo*>::iterator PckBegin = WpiGlobals.DBPckList.begin(),
                                             PckEnd = WpiGlobals.DBPckList.end();

                // remove it from the list in memory too
                WpiGlobals.DBPckList.remove(ppiRemove);

                // and call the GUI callback
                guiDatabaseCallback(DBC_PACKAGEREMOVED,
                                    ppiRemove,
                                    NULL,
                                    NULL,
                                    0);

                brc = TRUE;
            }
        }

    return (brc);
}

/*******************************************************************
 *                                                                 *
 *  Verification/Deinstallation functions                          *
 *                                                                 *
 ******************************************************************/

/*
 *@@ VerifyFile:
 *      this gets called from datDatabaseThread to
 *      verify a single file.
 *
 *      Returns FALSE if an error was found or TRUE
 *      if the file is OK.
 */

BOOL VerifyFile(PDATABASETHREADINFO pdbti,      // in: database thread info
                PackageInfo* pPckInfo,          // in: current package
                WIFileHeader* pHeader)          // in: file to verify
{
    BOOL brc = FALSE;

    CHAR szFullFile[2*CCHMAXPATH];

    sprintf(szFullFile,
            "%s\\%s",
            pPckInfo->szTargetPath,
            pHeader->name);

    FILESTATUS3 fs3;
    if (DosQueryPathInfo(szFullFile, FIL_STANDARD, &fs3, sizeof(fs3)))
    {
        // file not found: complain at callback
        guiDatabaseCallback(DBC_FILENOTFOUND,
                            pPckInfo,
                            pHeader,
                            NULL,
                            pdbti->ulUser);
    }
    else
    {
        // file found: compare last write dates
        // of database and real file
        FDATE fdateHeaderLastWrite;
        FTIME ftimeHeaderLastWrite;
        int iComp = wpiCompareFileDates(pHeader,
                                        &fs3,
                                        &fdateHeaderLastWrite,
                                        &ftimeHeaderLastWrite);
        if (iComp < 0)
            guiDatabaseCallback(DBC_EXISTINGFILENEWER,
                                pPckInfo,
                                pHeader,
                                (ULONG)&fs3,
                                pdbti->ulUser);
        else if (iComp > 0)
            guiDatabaseCallback(DBC_EXISTINGFILEOLDER,
                                pPckInfo,
                                pHeader,
                                (ULONG)&fs3,
                                pdbti->ulUser);
        else
            // else 0: file dates are the same
            brc = TRUE;
    }
    return (brc);
}

/*
 *@@ DeinstallFile:
 *      this gets called from datDatabaseThread to
 *      deinstall (= delete) a single file.
 *
 *      Returns FALSE if an error was found or TRUE
 *      if the file is OK.
 */

BOOL DeinstallFile(PDATABASETHREADINFO pdbti,      // in: database thread info
                   PackageInfo* pPckInfo,          // in: current package
                   WIFileHeader* pHeader)          // in: file to verify
{
    BOOL brc = FALSE;
    CHAR szFullFile[2*CCHMAXPATH];

    sprintf(szFullFile,
            "%s\\%s",
            pPckInfo->szTargetPath,
            pHeader->name);

    // reset the file's attributes, because
    // DosDelete fails if this has READONLY
    APIRET arc = doshSetPathAttr(szFullFile, 0);
    if (arc != NO_ERROR)
        // file not found: complain at callback
        guiDatabaseCallback(DBC_FILENOTFOUND,
                            pPckInfo,
                            pHeader,
                            arc,
                            pdbti->ulUser);
    else
    {
        // file exists: delete it
        arc = DosDelete(szFullFile);
        if (arc != NO_ERROR)
        {
            // report error
            guiDatabaseCallback(DBC_DELETEERROR,
                                pPckInfo,
                                pHeader,
                                arc,
                                pdbti->ulUser);
        }
        else
            brc = TRUE;
    }

    return (brc);
}

/*
 *@@ datDatabaseThread:
 *      this thread is started by datVerifyPackages or,
 *      datDeinstallPackages to compare package info
 *      with the actual files on disk.
 *
 *      ((PDATABASETHREADINFO)ptiMyself)->ulData is a
 *      pointer to the DATABASEINFO structure. It is the
 *      responsibility of this thread to free this
 *      structure.
 *
 *      This calls guiDatabaseCallback all the time.
 */

void _Optlink datDatabaseThread(PVOID ptiMyself)
{
    // install the LOUD exception handler (except.h)
    TRY_LOUD(excpt1)
    {
        PDATABASETHREADINFO pdbti = (PDATABASETHREADINFO)(((PTHREADINFO)ptiMyself)->ulData);
        ConfigSys*          pConfigSys = 0;

        // to list list, we'll add the target directories of
        // the packages which we have removed
        list<PackageInfo*> TargetDirList;

        // if we're in de-install mode and we're to undo
        // changes to CONFIG.SYS, load CONFIG.SYS now
        // (by creating a ConfigSys instance)
        if (    (pdbti->ulDeinstall & DBDEINST_CONFIGSYS)
             && (pdbti->ulTask == DBT_DEINSTALL)
           )
            pConfigSys = new(ConfigSys);

        // go thru all packages given to us
        list<PackageInfo*>::iterator PckBegin = pdbti->PckInfoList.begin(),
                                     PckEnd = pdbti->PckInfoList.end();

        for (; (PckBegin != PckEnd); PckBegin++)
        {
            PackageInfo* ppi = (*PckBegin);
                            // set to FALSE if any error occured

            ppi->ulStatus = 2;
                    // package status = "no errors found"
                    // (was either 0 or 2 before)

            // report package
            guiDatabaseCallback(DBC_NEXTPACKAGE,
                                ppi,
                                NULL,
                                NULL,
                                pdbti->ulUser);

            ULONG ulHeaderCount = 0;

            if (!doshQueryDirExist(ppi->szTargetPath))
            {
                // target directory not found:
                // call GUI callback with error
                guiDatabaseCallback(DBC_TARGETDIRNOTFOUND,
                                    ppi,
                                    NULL,
                                    0,
                                    pdbti->ulUser);
            }

            // in de-install mode, undo CONFIG.SYS/WPS classes/WPS objects
            if (    (ppi->ulStatus == 2)        // no errors found
                 && (pdbti->ulTask == DBT_DEINSTALL)
               )
            {
                // delete WPS objects?
                if (pdbti->ulDeinstall & DBDEINST_WPSOBJECTS)
                {
                    // use a reverse_iterator for deleting the
                    // objects because there might be a folder
                    // first which contains the other objects
                    // and we'd get error messages then
                    list<DeleteWPSObject*>::reverse_iterator
                            DelStart = ppi->listUndoWPSObject.rbegin(),
                            DelEnd   = ppi->listUndoWPSObject.rend();
                    for (; DelStart != DelEnd; DelStart++)
                    {
                        try
                        {
                            (**DelStart).Delete();
                            guiDatabaseCallback(DBC_OBJECTDELETED,
                                                ppi,
                                                NULL,
                                                (ULONG)((**DelStart).pszObjectID),
                                                pdbti->ulUser);
                        }
                        catch(ConfigExcpt& X)
                        {
                            // error deleting WPS object:
                            // call GUI callback with error
                            guiDatabaseCallback(DBC_OBJECTDELETEERROR,
                                                ppi,
                                                NULL,
                                                (ULONG)((**DelStart).pszObjectID),
                                                pdbti->ulUser);
                        }
                    }
                }

                // unreplace/deregister WPS classes?
                if (pdbti->ulDeinstall & DBDEINST_WPSCLASSES)
                {
                    // 1)   first unreplace classes
                    list<UnreplaceClass*>::iterator
                            ReplStart = ppi->listUndoReplaceClass.begin(),
                            ReplEnd   = ppi->listUndoReplaceClass.end();
                    for (; ReplStart != ReplEnd; ReplStart++)
                    {
                        try
                        {
                            (**ReplStart).Unreplace();
                            guiDatabaseCallback(DBC_CLASSUNREPLACED,
                                                ppi,
                                                NULL,
                                                (ULONG)((**ReplStart).pszOldClassName),
                                                pdbti->ulUser);
                        }
                        catch(ConfigExcpt& X)
                        {
                            // error deregistering class:
                            // call GUI callback with error
                            guiDatabaseCallback(DBC_UNREPLACECLASSERROR,
                                                ppi,
                                                NULL,
                                                (ULONG)((**ReplStart).pszOldClassName),
                                                pdbti->ulUser);
                        }
                    }

                    // 2)   then deregister classes
                    list<DeregisterClass*>::iterator
                            RegStart = ppi->listUndoRegisterClass.begin(),
                            RegEnd   = ppi->listUndoRegisterClass.end();
                    for (; RegStart != RegEnd; RegStart++)
                    {
                        try
                        {
                            (**RegStart).Deregister();
                            guiDatabaseCallback(DBC_CLASSDEREGISTERED,
                                                ppi,
                                                NULL,
                                                (ULONG)((**RegStart).pszClassName),
                                                pdbti->ulUser);
                        }
                        catch(ConfigExcpt& X)
                        {
                            // error deregistering class:
                            // call GUI callback with error
                            guiDatabaseCallback(DBC_DEREGISTERCLASSERROR,
                                                ppi,
                                                NULL,
                                                (ULONG)(X.pszSubstring),
                                                pdbti->ulUser);
                        }
                    }
                }

                // undo CONFIG.SYS changes?
                if (    (pdbti->ulDeinstall & DBDEINST_CONFIGSYS)
                     && (pConfigSys)
                   )
                {
                    list<CfgSysManip*>::iterator
                            CfgStart = ppi->listUndoCfgSys.begin(),
                            CfgEnd   = ppi->listUndoCfgSys.end();
                    for (; CfgStart != CfgEnd; CfgStart++)
                    {
                        try
                        {
                            // the items on this list have been
                            // created just according to what needs
                            // to be undone by the PackageInfo constructor
                            pConfigSys->Manipulate(**CfgStart,
                                                   NULL);   // no logger here
                            guiDatabaseCallback(DBC_CFGSYSUNDONE,
                                                ppi,
                                                NULL,
                                                (ULONG)(*CfgStart),
                                                pdbti->ulUser);
                        }
                        catch (ConfigExcpt& X)
                        {
                            // error deregistering class:
                            // call GUI callback with error
                            guiDatabaseCallback(DBC_UNDOCFGSYSERROR,
                                                ppi,
                                                NULL,
                                                (ULONG)(*CfgStart),
                                                pdbti->ulUser);
                        }
                    }
                }
            } // end if ... (pdbti->ulTask == DBT_DEINSTALL)

            // create a list of file headers
            list<WIFileHeader*>* pFileHeadersList = datGetFileHeaders(ppi,
                                                                      &ulHeaderCount);
            // now verify or delete all files
            if (pFileHeadersList)
            {
                list<WIFileHeader*>::iterator HeaderStart = pFileHeadersList->begin(),
                                              HeaderEnd = pFileHeadersList->end();

                for (; HeaderStart != HeaderEnd; HeaderStart++)
                {
                    WIFileHeader *pHeader = (*HeaderStart);

                    // report file
                    guiDatabaseCallback(DBC_NEXTFILE,
                                        ppi,
                                        pHeader,
                                        NULL,
                                        pdbti->ulUser);

                    switch (pdbti->ulTask)
                    {
                        case DBT_VERIFY:
                            if (!VerifyFile(pdbti, ppi, pHeader))
                                ppi->ulStatus = 1;         // == errors found
                        break;

                        case DBT_DEINSTALL:
                            if (!DeinstallFile(pdbti, ppi, pHeader))
                                ppi->ulStatus = 1;         // == errors found
                        break;
                    }
                }

                delete pFileHeadersList;
            }
            // now done with the files of this package

            // in de-install mode, do additional stuff
            if (    (ppi->ulStatus == 2)        // no errors found
                 && (pdbti->ulTask == DBT_DEINSTALL)
               )
            {
                // check if the target dir of this package is
                // already in the list
                list<PackageInfo*>::iterator TargetDirFirst = TargetDirList.begin(),
                                             TargetDirLast = TargetDirList.end();
                BOOL fFound = FALSE;
                for (; TargetDirFirst != TargetDirLast; TargetDirFirst++)
                {
                    if (strcmp((**TargetDirFirst).szTargetPath, ppi->szTargetPath) == 0)
                    {
                        fFound = TRUE;
                        break;
                    }
                }
                if (!fFound)
                    // not in list: add
                    TargetDirList.push_back(ppi);

                // remove this package from the database;
                // this will call guiDatabaseCallback again
                datRemovePackage(ppi);
            }

        } // end for (; (PckBegin != PckEnd); PckBegin++)

        // remove all directories on the target dir list
        // (this list is empty for verify mode)
        list<PackageInfo*>::iterator TargetDirFirst = TargetDirList.begin(),
                                     TargetDirLast = TargetDirList.end();
        for (; TargetDirFirst != TargetDirLast; TargetDirFirst++)
        {
            APIRET arc = DosDeleteDir((**TargetDirFirst).szTargetPath);
            if (arc != NO_ERROR)
            {
                guiDatabaseCallback(DBC_DELETEDIRERROR,
                                    *TargetDirFirst,
                                    NULL,
                                    arc,
                                    pdbti->ulUser);
                (*TargetDirFirst)->ulStatus = 1;         // == errors found
            }
        }

        // write ConfigSys back to disk
        if (pConfigSys)
        {
            pConfigSys->WriteBack(TRUE);        // create backup
            delete(pConfigSys);
        }

        // report NULL package (== we're done)
        guiDatabaseCallback(DBC_NEXTPACKAGE,
                            NULL,
                            NULL,
                            NULL,
                            pdbti->ulUser);

        delete pdbti;       // this has a destructor
    }
    CATCH (excpt1)
    {
    } END_CATCH;

    thrGoodbye((PTHREADINFO)ptiMyself);
}

/*
 *@@ datVerifyPackages:
 *      this starts the Database thread to verify the
 *      given packages.
 *
 *      The pPckInfoList is copied into a new list for the Database
 *      thread, so this does not have to be permanent.
 */

BOOL datVerifyPackages(list<PackageInfo*>* pPckInfoList,
                       ULONG ulUser)            // in: user parameter passed to
                                                // guiDatabaseCallback (could be HWND)
{
    BOOL brc = FALSE;

    if (pPckInfoList)
    {
        // now we have the package info which we need to verify;
        // create a thread info for the database thread
        PDATABASETHREADINFO pdbti = new DATABASETHREADINFO(DBT_VERIFY,  // task
                                                           0, // deinstall flags, don't care
                                                           ulUser);
        // copy the list
        list<PackageInfo*>::iterator PckStart = pPckInfoList->begin(),
                                     PckEnd = pPckInfoList->end();
        for (; PckStart != PckEnd; PckStart++)
            pdbti->PckInfoList.push_back(*PckStart);

        // start database thread
        thrCreate(&ptiDatabaseThread ,       // global PTHREADINFO at top
                  datDatabaseThread,         // thread func
                  (ULONG)pdbti);             // data passed to thread
        // pdbti will be deleted by the database thread

        brc = TRUE;
    }

    return (brc);
}

/*
 *@@ datDeinstallPackages:
 *      this starts the Database thread to de-install the
 *      given packages.
 *
 *      If ulFlags is 0, only the files will be removed.
 *      You can however OR any of the following:
 *      -- DBDEINST_CONFIGSYS:  remove CONFIG.SYS entries
 *      -- DBDEINST_WPSCLASSES: de-install WPS classes
 *      -- DBDEINST_WPSOBJECTS: destroy WPS objects
 */

BOOL datDeinstallPackages(list<PackageInfo*>* pPckInfoList,
                          ULONG ulFlags,        // DBDEINST_* flags
                          ULONG ulUser)         // in: user parameter passed to
                                                // guiDatabaseCallback (could be HWND)
{
    BOOL brc = FALSE;

    if (pPckInfoList)
    {
        // now we have the package info which we need to verify;
        // create a thread info for the database thread
        PDATABASETHREADINFO pdbti = new DATABASETHREADINFO(DBT_DEINSTALL,  // task
                                                           ulFlags,
                                                           ulUser);
        // copy the list
        list<PackageInfo*>::iterator PckStart = pPckInfoList->begin(),
                                     PckEnd = pPckInfoList->end();
        for (; PckStart != PckEnd; PckStart++)
            pdbti->PckInfoList.push_back(*PckStart);

        // start database thread
        thrCreate(&ptiDatabaseThread,       // global PTHREADINFO at top
                  datDatabaseThread,        // thread func
                  (ULONG)pdbti);            // data passed to thread
        // pdbti will be deleted by the database thread

        brc = TRUE;
    }

    return (brc);
}


