
/*
 * config.h:
 *      header file for config.c, which has the various configuration
 *      classes. See cfgsys.c for details.
 *
 *      Required #include's before #include'ing this file:
 +          #define INCL_WINWORKPLACE
 +          #include <os2.h>
 +          #include "common.h"
 +          #include <list>
 *
 *      This file Copyright (C) 1999 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#ifndef WPI_CONFIGSYS_INCLUDE
#define WPI_CONFIGSYS_INCLUDE

// error codes for ConfigExcpt thrown by ConfigSys class
const int CFGEXCPT_OPEN                 = 1;
                // error opening CONFIG.SYS (ConfigSys);
                // ConfigExcpt.iData has the APIRET
const int CFGEXCPT_DOUBLEREPLACEMODE    = 2;
                // error parsing CONFIGSYS attribute (CfgSysManip):
                // replace mode specified twice,
                // ConfigExcpt.pszSubstring has the failing part
const int CFGEXCPT_INVALIDVERTICAL      = 3;
                // error parsing CONFIGSYS attribute (CfgSysManip):
                // vertical modifier not allowed here
                // ConfigExcpt.pszSubstring has the failing part
const int CFGEXCPT_INVALIDSEARCH        = 4;
                // invalid search string (CfgSysManip)
                // ConfigExcpt.pszSubstring has the failing part
const int CFGEXCPT_MANIPULATE           = 5;
                // error in ConfigSys::Manipulate()
                // (should not happen)
const int CFGEXCPT_NOTIMPLEMENTED       = 6;
                // not implemented yet
const int CFGEXCPT_NOSEPARATOR          = 7;
                // this pops up while modifying CONFIG.SYS
                // and ADDLEFT or ADDRIGHT have been specified,
                // but the line in CONFIG.SYS has no "=" character
const int CFGEXCPT_SYNTAX               = 8;
                // invalid attribute passed to CfgSysManip::CfgSysManip
const int CFGEXCPT_PARSELOG             = 9;
                // syntax error parsing log entry in CfgSysManip::AddToUndoList

// error codes for ConfigExcpt thrown by the RegisterClass, DeregisterClass,
// ReplaceClass classes
const int REGEXCPT_SYNTAX               = 60;
                // syntax error in constructor (invalid pszRegisterClass);
                // ConfigExcpt.pszSubstring has the whole failing pszRegisterClass.
                // This is also used by ReplaceClass::ReplaceClass.
const int REGEXCPT_QUERYCLASSLIST       = 61;
                // error querying WPS class list
const int REGEXCPT_ALREADYREGISTERED    = 62;
                // RegisterClass::Register(FALSE) has determined that the
                // class is already registered;
                // ConfigExcpt.pszSubstring has the currently registered DLL
const int REGEXCPT_REGISTER             = 63;
                // RegisterClass::Register failed;
                // ConfigExcpt.iData has the APIRET of winhRegisterClass
const int REGEXCPT_DEREGISTER           = 64;
                // DeregisterClass::Deregister failed
const int REGEXCPT_REPLACE              = 65;
                // ReplaceClass::Replace failed (WinReplaceObjectClass returned FALSE)
const int REGEXCPT_UNREPLACE            = 66;
                // UnreplaceClass::Unreplace failed (WinReplaceObjectClass returned FALSE)

// error codes for ConfigExcpt thrown by CreateWPSObject class
const int WPOEXCPT_NOLOCATION           = 100;
                // no location given
const int WPOEXCPT_NOTITLE              = 101;
                // no title given
const int WPOEXCPT_NOCLASS              = 102;
                // no class given
const int WPOEXCPT_CREATE               = 103;
                // error in CreateWPSObject::CreateObject
const int WPOEXCPT_DELETEOBJECT         = 104;
                // error deleting object (DeleteWPSObject::Delete)

/*
 *@@ ConfigExcpt:
 *      exception thrown by many of the configuration
 *      methods.
 */

class ConfigExcpt
{
    public:
        int     iErrorCode;
        int     iData;          // with CFGSYS_OPEN, this has the APIRET
        char*   pszSubstring;

        // constructor 1: for integer data only
        ConfigExcpt(int iErrorCode_, int iData_)
        {
            iErrorCode = iErrorCode_;
            iData = iData_;
            pszSubstring = 0;
        }

        // constructor 2: report string (should be called with strdup())
        ConfigExcpt(int iErrorCode_, char* pszSubstring_)
        {
            iErrorCode = iErrorCode_;
            iData = 0;
            pszSubstring = pszSubstring_;
        }

        // constructor 3: both integer and string
        ConfigExcpt(int iErrorCode_, int iData_, char* pszSubstring_)
        {
            iErrorCode = iErrorCode_;
            iData = iData_;
            pszSubstring = pszSubstring_;
        }

        // destructor
        ~ConfigExcpt()
        {
            if (pszSubstring)
                free(pszSubstring);
        }
};

/*
 *@@ INILogger:
 *      generic logging class which supports
 *      putting any kind of data into a single
 *      memory block. Useful for storing logs
 *      in an INI file.
 *
 *      Note that the logger has no idea what the
 *      the data it stores means. It is the exclusive
 *      responsibility of the caller to be able to
 *      compose and parse this data.
 *
 *      No instances of INILogger are ever created directly,
 *      but only of the derived logger classes (below).
 *      These derived classes are only implemented to
 *      be able to use C++ type checking though. They
 *      do not modify the base logger class in any
 *      way.
 */

class INILogger
{
    public:
        char*           pabLogString;
        unsigned long   cbLogString;

        INILogger();
        ~INILogger();
        void Append(char* pszData, unsigned long cbData);
        void Append(char* psz);
        void Append(char* psz1, char* psz2);

        bool Empty() const;

        bool Store(HINI hini, const char* pszApp, const char* pszKey) const;
        bool Load(HINI hini, const char* pszApp, const char* pszKey);
};

/*
 *@@ AttrLogger:
 *      class derived from Logger for storing string attributes.
 *      This is identical to Logger and not used either, only
 *      for deriving more logger classes.
 */

class AttrLogger : public INILogger { };

/*
 *@@ DoneLogger:
 *      class derived from Logger for storing the actual system changes made.
 *      This is identical to Logger and not used either, only
 *      for deriving more logger classes.
 */

class DoneLogger : public INILogger { };

/*
 *@@ CfgSysDoneLogger:
 *      this logs changes made to CONFIG.SYS.
 *      This is derived from DoneLogger and only
 *      declared to make sure that it's a CONFIG.SYS
 *      logger passed to the ConfigSys methods.
 */

class CfgSysDoneLogger : public DoneLogger { };

/*
 *@@ RegisterDoneLogger:
 *      this logs registered WPS classes.
 *      This is derived from DoneLogger and only
 *      declared to make sure that only a suitable
 *      logger is passed to the RegisterClass methods.
 */

class RegisterDoneLogger : public DoneLogger { };

/*
 *@@ ReplaceDoneLogger:
 *      this logs replaced WPS classes.
 *      This is derived from DoneLogger and only
 *      declared to make sure that only a suitable
 *      logger is passed to the ReplaceClass methods.
 */

class ReplaceDoneLogger : public DoneLogger { };

/*
 *@@ WPSObjectsDoneLogger:
 *      this logs created WPS objects.
 *      This is derived from DoneLogger and only
 *      declared to make sure that only a suitable
 *      logger is passed to the CreateWPSObject methods.
 */

class WPSObjectsDoneLogger : public DoneLogger { };

// flags for CfgSysManip (below)
typedef int CFGRPL;
const CFGRPL CFGRPL_ADD            = 0;
const CFGRPL CFGRPL_UNIQUE         = 1;
const CFGRPL CFGRPL_ADDLEFT        = 2;
const CFGRPL CFGRPL_ADDRIGHT       = 3;
const CFGRPL CFGRPL_REMOVELINE     = 4;
const CFGRPL CFGRPL_REMOVEPART     = 5;

typedef int CFGVRT;
const CFGVRT CFGVRT_BOTTOM         = 0;
const CFGVRT CFGVRT_TOP            = 1;
const CFGVRT CFGVRT_BEFORE         = 2;
const CFGVRT CFGVRT_AFTER          = 3;

/*
 *@@ CfgSysManip:
 *      the "CONFIG.SYS manipulator" class.
 *      An instance of this must be created
 *      with a suitable attribute string and
 *      passed to ConfigSys::Manipulate.
 *
 *      See CfgSysManip::CfgSysManip for details.
 */

class CfgSysManip
{
    // allow ConfigSys access to private members
    friend class ConfigSys;

    private:
        CFGRPL      iReplaceMode;   // this is an int, really
                    // 0: just add line (default; iVertical applies)
                    // 1: UNIQUE mode (line is replaced;
                    //    if not found, iVertical applies)
                    // 2: ADDLEFT mode (line is updated to the left;
                    //    if not found, iVertical applies)
                    // 3: ADDRIGHT mode (line is updated to the right;
                    //    if not found, iVertical applies)
                    // 4: REMOVELINE mode (all the following are ignored)
                    // 5: REMOVEPART mode (all the following are ignored)

        CFGVRT      iVertical;      // this is an int, really
                    // 0: add to bottom (default)
                    // 1: add to top
                    // 2: add before pszSearchString
                    // 3: add after pszSearchString
        char*       pszSearchString;
                    // for iVertical == 2 or 3

    public:
        char*       pszNewLine;
                    // this is a copy of stuff before the "|" char
                    // in pszConfigSys given to the constructor

        CfgSysManip(char* pszConfigSys) throw (ConfigExcpt);
        ~CfgSysManip();

        static int AddToUndoList(list<CfgSysManip*> &List,
                                 CfgSysDoneLogger &Logger);
};

/*
 *@@ ConfigSys:
 *      this is the ConfigSys class for CONFIG.SYS manipulation.
 *      If an instance of this created, CONFIG.SYS is read in
 *      by the constructor, so only create an instance of this
 *      if you're actually to modify the file.
 *
 *      This class works in conjunction with CfgSysManip instances.
 *
 *      The destructor does _not_ write the file. Invoke
 *      ConfigSys::WriteBack() for that.
 */

class ConfigSys
{
    protected:
        char*       pszContent;
        char        szFilename[30];         // "?:\CONFIG.SYS"

    public:
        ConfigSys() throw (ConfigExcpt);
        ~ConfigSys();

        int Manipulate(CfgSysManip& Manip,
                       CfgSysDoneLogger* pLogger)
            throw(ConfigExcpt);
        int Undo(CfgSysManip& Manip)
            throw (ConfigExcpt);
        int WriteBack(bool fBackup)
            const
            throw (ConfigExcpt);
};

/*
 *@@ RegisterClass:
 *      this is the RegisterClass class for registering WPS classes.
 */

class RegisterClass
{
    public:
        // all these must be public because otherwise
        // WarpIN cannot resolve macros
        char*       pszClassName;
        char*       pszModule;

        RegisterClass(char* pszRegisterClass) throw(ConfigExcpt);
        ~RegisterClass();

        bool IsRegistered(char* pszBuffer)
             const;

        int Register(bool fReplace,
                     RegisterDoneLogger* pLogger)
                     const throw(ConfigExcpt);
};

/*
 *@@ DeregisterClass:
 *      the reverse to the above. This is used to deregister
 *      classes that were previously registered using the
 *      RegisterClass class.
 */

class DeregisterClass
{
    public:
        char*       pszClassName;

        DeregisterClass(char* pszDeregisterClass);
        ~DeregisterClass();

        int Deregister()
            const
            throw(ConfigExcpt);

        static int AddToUndoList(list<DeregisterClass*> &List,
                                 RegisterDoneLogger &Logger);
};

/*
 *@@ ReplaceClass:
 *      this is the ReplaceClass class for replacing WPS classes.
 */

class ReplaceClass
{
    public:
        // all these must be public because otherwise
        // WarpIN cannot resolve macros
        char*       pszOldClassName;
        char*       pszNewClassName;

        ReplaceClass(char* pszReplaceClass) throw(ConfigExcpt);
        ~ReplaceClass();

        int Replace(ReplaceDoneLogger* pLogger)
            const
            throw(ConfigExcpt);
};

/*
 *@@ UnreplaceClass:
 *      the reverse to the above. This is used to un-replace
 *      classes that were previously replaced using the
 *      RegisterClass class.
 *
 *      Since the ReplaceDoneLogger class stores items just
 *      like the input to ReplaceClass was, we inherit from
 *      that class to use the same parsing routines.
 */

class UnreplaceClass : public ReplaceClass
{
    public:
        // inherit constructor from ReplaceClass
        UnreplaceClass(char* pszUnreplaceClass)
                    throw(ConfigExcpt)
                : ReplaceClass(pszUnreplaceClass)
        {};

        int Unreplace()
            const
            throw(ConfigExcpt);

        static int AddToUndoList(list<UnreplaceClass*> &List,
                                 ReplaceDoneLogger &Logger);
};

/*
 *@@ CreateWPSObject:
 *      this has configuration info for creating
 *      objects.
 */

class CreateWPSObject
{
    public:
        // the following should always be set
        PSZ             pszClassName,
                        pszTitle,
                        pszSetupString,     // this one might be NULL
                        pszLocation;
        BOOL            fReplace;

        // the following is set by ConfigExcpt::CreateObject
        // if an object has been created successfully
        HOBJECT         hobj;

        CreateWPSObject(char* pszCreateObject) throw(ConfigExcpt);
        ~CreateWPSObject();

        void CreateObject(WPSObjectsDoneLogger* pLogger)
             throw(ConfigExcpt);
        bool DoesObjectExist()
             throw(ConfigExcpt);
        void DestroyObject()
             throw(ConfigExcpt);
};

/*
 *@@ DeleteWPSObject:
 *      the reverse to the above. This is used to delete
 *      objects that were previously created using the
 *      CreateWPSObject class.
 */

class DeleteWPSObject
{
    public:
        PSZ             pszObjectID;

        DeleteWPSObject(char* pszID2Delete);
        ~DeleteWPSObject();

        int Delete()
            const
            throw(ConfigExcpt);

        static int AddToUndoList(list<DeleteWPSObject*> &List,
                                 WPSObjectsDoneLogger &Logger);
};

#endif

