
/*
 * config.cpp:
 *      this has the code for the classes which manipulate CONFIG.SYS,
 *      the WPS class list, and create WPS objects.
 *
 *      These classes are designed to work independently of WarpIN
 *      and could be used in any program.
 *      I have attempted to design these classes for easy, yet flexible use.
 *
 *      The general idea of these classes is that you have one class
 *      for each manipulation to be done and another class which undoes
 *      the change again. (The exception is the ConfigSys class, which
 *      has an "Undo" method instead of an undo class.)
 *
 *      For each group of classes, there is one AddToUndoList static
 *      (class) method, which creates a list of Undo objects from the
 *      logger which was used to do the manipulation before.
 *      See CfgSysManip::CfgSysManip for an example usage.
 *
 *      Some methods throw instances of the ConfigExcpt class def'd in
 *      config.h, which you should handle. If so, a "throw (ConfigExcpt)"
 *      has been added to the respective function headers.
 *
 *      The manipulation method of each of the manipulation classes
 *      take some kind of INILogger instance as input (see the description
 *      below). This instance logs all the changes made and can then
 *      be passed to the "Undo" class/method to undo the change again.
 *
 *      These are the classes implemented here:
 *      --  INILogger:
 *          base logger class, which is capable of maintaining
 *          a log of strings. Unmodified subclasses of this are
 *          declared in config.h and used by all the following classes
 *          to log changes made to the system, which can then be stored
 *          in the database and passed to the "Undo" classes/methods to undo
 *          the changes again.
 *          See INILogger::INILogger for details.
 *          The format of data stored in the logger is dependent of the
 *          class which writes data into it. The logger itself has no
 *          idea what the data it stores means. It is the exclusive
 *          responsibility of the caller to write and parse that data
 *          (which the below classes are capable of).
 *      --  CfgSysManip:
 *          a "manipulator" object of this class holds information
 *          for a CONFIG.SYS change, which is passed to ConfigSys::Manipulate.
 *          See CfgSysManip::CfgSysManip for a description.
 *      --  ConfigSys:
 *          the actual class which holds the CONFIG.SYS text file
 *          and allows changes to be made.
 *          This class does not have a corresponding "Undo" class,
 *          but a ConfigSys::Undo method instead.
 *      --  RegisterClass:
 *          this allows registering classes with the WPS.
 *      --  DeregisterClass:
 *          the "Undo" class to the previous.
 *      --  ReplaceClass:
 *          this allows WPS classes to be replaced.
 *      --  UnreplaceClass:
 *          the "Undo" class to the previous.
 *      --  CreateWPSObject:
 *          for creating a WPS object.
 *      --  DeleteWPSObject:
 *          the "Undo" class for the previous, to delete a WPS object again.
 *
 *      See config.h for the declarations of these classes.
 */

/*
 *      This file Copyright (C) 1999 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

// all these are required for winh.h
#define INCL_DOSERRORS
#define INCL_WINWINDOWMGR
#define INCL_WINMESSAGEMGR
#define INCL_WINDIALOGS
#define INCL_WINSTDCNR
#define INCL_WININPUT
#define INCL_WINSYS
#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"

#include <list>

#include "dosh.h"
#include "winh.h"
#include "prfh.h"
#include "stringh.h"

#include "errors.h"

#include "config.h"

/*******************************************************************
 *                                                                 *
 *  INILogger class                                                *
 *                                                                 *
 ******************************************************************/

/*
 *@@ INILogger:
 *      default constructor for creating an empty log string.
 *
 *      See INILogger::Append for example usage.
 *
 *      INILogger::Store and INILogger::Load can be used for
 *      storing a logger into or retrieving one from an INI file.
 */

INILogger::INILogger()
{
    pabLogString = 0;
    cbLogString = 0;
}

/*
 *@@ ~INILogger:
 *      destructor
 */

INILogger::~INILogger()
{
    if (pabLogString)
        free(pabLogString);
}

/*
 *@@ Append:
 *      this appends binary data to the log string.
 *      This can be any data, but you should be able
 *      to decode it again so if each data is variable
 *      in length, you should add some markers.
 *
 *      Note: as opposed to the other (overloaded)
 *      "Append" methods, this does _not_ make a copy
 *      of the data, so you must have allocated memory
 *      dynamically for this.
 *
 *      Example 1: you could add plain text strings.
 *                 Done forget to store the null terminator then too.
 +          Logger.Append(strdup("Hello"), strlen("Hello") + 1);
 *
 *      Example 2: you could add binary data, with
 *                 each item having a predefined length.
 +          char    *pabData = (char*)malloc(10);
 +          pabData = ....
 +          Logger.Append(pabData, 10);
 *
 *      Use
 */

void INILogger::Append(char* pszData,           // in: data block to append (must be terminated with null)
                       unsigned long cbData)    // in: sizeof(*pszData), including the null byte
{
    if (pabLogString == NULL)
    {
        // first run:
        pabLogString = pszData;
        cbLogString = cbData;
    }
    else
    {
        // subsequent runs: append after the existing data
        char*   pszTemp = (char*)malloc(cbLogString + cbData);
        // copy old buffer
        memcpy(pszTemp, pabLogString, cbLogString);
        // copy new attribs behind last null byte
        memcpy(pszTemp + cbLogString,
               pszData,
               cbData);
        // set new buffer
        free(pabLogString);
        pabLogString = pszTemp;
        cbLogString += cbData;
    }
}

/*
 *@@ Append:
 *      overloaded Append, this does a strdup on the string
 *      and finds the length automatically  (including the null terminator).
 *
 *      Example:
 +          Logger.Append("Hello")
 *      is equal to
 +          Logger.Append(strdup("Hello"), strlen("Hello") + 1);
 */

void INILogger::Append(char* psz)
{
    unsigned long cbData = strlen(psz)+1;
    char* pData = (char*)malloc(cbData);
    strcpy(pData, psz);
    Append(pData, cbData);
}

/*
 *@@ Append:
 *      yet another overloaded Append, which concatenates two strings
 *      and finds the length automatically (including the null terminator).
 *
 *      Example:
 +          Logger.Append("Hello ", "World");
 *      is equal to:
 +          Logger.Append(strdup("Hello World"), strlen("Hello World") + 1);
 */

void INILogger::Append(char* psz1, char* psz2)
{
    unsigned long cbData1 = strlen(psz1);
    unsigned long cbData2 = strlen(psz2);
    char* pData = (char*)malloc(cbData1 + cbData2 + 1);
    memcpy(pData, psz1, cbData1);
    strcpy(pData + cbData1, psz2);
    Append(pData, cbData1 + cbData2 + 1);
}

/*
 *@@ Empty:
 *      returns TRUE if the logger is currently empty.
 */

bool INILogger::Empty() const
{
    return (pabLogString != 0);
}

/*
 *@@ Store:
 *      this stores the whole INILogger in the given profile key.
 *      Returns TRUE if successfully written.
 *
 *      Note: if the logger is currently empty, this will delete
 *      the given INI key.
 */

bool INILogger::Store(HINI hini,            // in: INI handle
                      const char* pszApp,   // in: INI application
                      const char* pszKey)   // in: INI key
                const
{
    return (PrfWriteProfileData(hini, pszApp, pszKey, pabLogString, cbLogString));
}

/*
 *@@ Load:
 *      reverse to INILogger::Store, this loads the data from INI.
 *      This will overwrite the current contents of the logger.
 *
 *      Returns TRUE if data was found.
 */

bool INILogger::Load(HINI hini,             // in: INI handle
                     const char* pszApp,    // in: INI application
                     const char* pszKey)    // in: INI key
{
    if (pabLogString)
        free(pabLogString);
    pabLogString = prfhQueryProfileData(hini,
                                        (PSZ)pszApp,
                                        (PSZ)pszKey,
                                        &cbLogString);
    return (pabLogString != 0);
}

/*******************************************************************
 *                                                                 *
 *  CfgSysManip class                                              *
 *                                                                 *
 ******************************************************************/

/*
 *@@ CfgSysManip(char* pszConfigSys):
 *      this constructor translates a CONFIGSYS attribute (as used
 *      in the PCK tag and stored in the database) into a CONFIGSYSITEM
 *      structure.
 *
 *      The CfgSysManip class is designed for use with the ConfigSys class.
 *      This class describes manipulations to be done upon the CONFIG.SYS
 *      file (represented by the ConfigSys class).
 *
 *      Manipulating CONFIG.SYS works just as follows:
 *
 *      1)  Create an instance of ConfigSys:
 *
 +              ConfigSys* pConfigSys = new ConfigSys;
 *
 *          This will load your CONFIG.SYS file into the instance's memory.
 *
 *      2)  Create an instance of CfgSysManip and specify the manipulation
 *          in the constructor:
 *
 +              CfgSysManip* pCfgSysManip = new CfgSysManip("SET TEST=YES | UNIQUE");
 *
 *          The CfgSysManip class has a single constructor which takes a PSZ
 *          (char*) as input. This input string has exactly the format like
 *          with the CONFIGSYS attribute to the PCK tag in WarpIN installation
 *          scripts, like this (see the WarpIN Programmer's Reference for details):
 *
 +              statement [| modifiers]
 *
 *          "modifiers" can be one of the following:
 +              [UNIQUE] [vertical]
 +              ADDRIGHT [vertical]
 +              ADDLEFT [vertical]
 +              REMOVELINE
 +              REMOVEPART
 *
 *          "vertical" can be one of the following:
 +              ADDTOP
 +              ADDAFTER(xxx)
 +              ADDBEFORE(xxx)
 *          with "xxx" being a search string.
 *
 *          Examples:
 +              "BASEDEV=NEWDRIVR.SYS /WHATEVER | UNIQUE ADDAFTER(IBM1S506.ADD)"
 +              "SET PATH=C:\BLAH | ADDRIGHT"
 *
 *          After this constructor has successfully converted pszConfigSys,
 *          all the instance data is valid (see cfgsys.h).
 *
 *          However, this does _not_ handle macro resultion like in WarpIn
 *          scripts (which is done in warpin.cpp before calling the Manipulate
 *          method), because in this file scope we know nothing about the
 *          PackageInfo instances.
 *
 *      3)  Create a CfgSysDone logger instance:
 +              CfgSysDoneLogger Logger;
 *
 *      4)  Invoke the ConfigSys::Manipulate method with the CfgSysManip
 *          instance (this will add data to the logger):
 +              pConfigSys->Manipulate(Logger, Manip);
 *
 *      5)  Write CONFIG.SYS back to disk and clean up.
 +              pConfigSys->WriteBack(TRUE);
 +              delete pCfgSysManip;
 +              delete pConfigSys;
 *
 *      Now, if you want to undo the changes later, call the static
 *      CfgSysManip::AddToUndoList method with the logger passed to
 *      to the various constructor calls (above), which will create
 *      a list of CfgSysManip instance which will be able to undo
 *      the changes again:
 +          list<CfgSysManip*> UndoList;
 +          CfgSysManip::AddToUndoList(UndoList, Logger);
 *      and iterate over the list and call ConfigSys::Manipulate with
 *      the objects on that list to undo the changes.
 */

CfgSysManip::CfgSysManip(char* pszConfigSys)
             throw (ConfigExcpt)
{
    // initialize all fields to zero
    memset(this, 0, sizeof(*this));

    if (!pszConfigSys)
        throw(ConfigExcpt(CFGEXCPT_SYNTAX, 0));

    // now check if we have modifiers
    PSZ     pSep = strchr(pszConfigSys, '|');
    if (!pSep)
        // no modifiers: just copy
        pszNewLine = strdup(pszConfigSys);
    else
    {
        // we do have modifiers:
        BOOL    fVerticalsAllowed = TRUE;

        // get rid of spaces before '|'
        PSZ     pSep2 = pSep-1;
        while (*pSep2 == ' ')
            pSep2--;

        // get the "statement" part
        pszNewLine = strhSubstr(pszConfigSys, pSep2 + 1);

        // remember the modifiers position
        PSZ     pModifiers = pSep + 1;
        BOOL    fReplaceModeFound = FALSE;
        // now check for the replace mode;
        // the default is 0 (CFGRPL_ADD)
        iReplaceMode = CFGRPL_ADD;
        fVerticalsAllowed = TRUE;

        if (strhistr(pModifiers, "UNIQUE"))
        {
            iReplaceMode = CFGRPL_UNIQUE;
            fVerticalsAllowed = TRUE;
            fReplaceModeFound = TRUE;
        }
        if (strhistr(pModifiers, "ADDRIGHT"))
        {
            if (!fReplaceModeFound)
            {
                iReplaceMode = CFGRPL_ADDRIGHT;
                fVerticalsAllowed = TRUE;
                fReplaceModeFound = TRUE;
            }
            else
                // double replace mode found:
                throw(ConfigExcpt(CFGEXCPT_DOUBLEREPLACEMODE,
                                  strdup(pModifiers)));
        }
        if (strhistr(pModifiers, "ADDLEFT"))
        {
            if (!fReplaceModeFound)
            {
                iReplaceMode = CFGRPL_ADDLEFT;
                fVerticalsAllowed = TRUE;
                fReplaceModeFound = TRUE;
            }
            else
                // double replace mode found:
                throw(ConfigExcpt(CFGEXCPT_DOUBLEREPLACEMODE,
                                  strdup(pModifiers)));
        }
        if (strhistr(pModifiers, "REMOVELINE"))
        {
            if (!fReplaceModeFound)
            {
                iReplaceMode = CFGRPL_REMOVELINE;
                fVerticalsAllowed = FALSE;
                fReplaceModeFound = TRUE;
            }
            else
                // double replace mode found:
                throw(ConfigExcpt(CFGEXCPT_DOUBLEREPLACEMODE,
                                  strdup(pModifiers)));
        }
        if (strhistr(pModifiers, "REMOVEPART"))
        {
            if (!fReplaceModeFound)
            {
                iReplaceMode = CFGRPL_REMOVEPART;
                fVerticalsAllowed = FALSE;
                fReplaceModeFound = TRUE;
            }
            else
                // double replace mode found:
                throw(ConfigExcpt(CFGEXCPT_DOUBLEREPLACEMODE,
                                  strdup(pModifiers)));
        }

        // now parse vertical modifiers
        iVertical = CFGVRT_BOTTOM;      // default
        BOOL fVerticalFound = FALSE;
        if (strhistr(pModifiers, "ADDTOP"))
        {
            iVertical = CFGVRT_TOP;
            fVerticalFound = TRUE;
        }
        PSZ p2;
        if (p2 = strhistr(pModifiers, "ADDAFTER("))
        {
            if (!fVerticalFound)
            {
                PSZ pEndOfSearch = strchr(p2, ')');
                if (!pEndOfSearch)
                    throw(ConfigExcpt(CFGEXCPT_INVALIDSEARCH, strdup(p2)));
                else
                    pszSearchString = strhSubstr(p2 + 9,        // strlen("ADDAFTER(")
                                                 pEndOfSearch); // excluding that char

                iVertical = CFGVRT_AFTER;
                fVerticalFound = TRUE;
            }
            else
                throw(ConfigExcpt(CFGEXCPT_INVALIDVERTICAL, strdup(p2)));
        }
        if (p2 = strhistr(pModifiers, "ADDBEFORE("))
        {
            if (!fVerticalFound)
            {
                PSZ pEndOfSearch = strchr(p2, ')');
                if (!pEndOfSearch)
                    throw(ConfigExcpt(CFGEXCPT_INVALIDSEARCH, strdup(p2)));
                else
                    pszSearchString = strhSubstr(p2 + 10,       // strlen("ADDBEFORE(")
                                                 pEndOfSearch); // excluding that char

                iVertical = CFGVRT_BEFORE;
                fVerticalFound = TRUE;
            }
            else
                throw(ConfigExcpt(CFGEXCPT_INVALIDVERTICAL, strdup(p2)));
        }

        // finally check if vertical modifier is allowed at all
        if ((fVerticalFound) && (!fVerticalsAllowed))
            throw(ConfigExcpt(CFGEXCPT_INVALIDVERTICAL, strdup(pModifiers)));

    } // end elseif (!pSep)
}

/*
 *@@ ~CfgSysManip:
 *      destructor.
 */

CfgSysManip::~CfgSysManip()
{
    if (pszNewLine)
        free(pszNewLine);
    if (pszSearchString)
        free(pszSearchString);
}

/*
 *@@ AddToUndoList:
 *      static class method to create CfgSysManip
 *      instances from the logger instance that was
 *      previously used with ConfigSys::Manipulate.
 *
 *      The new CfgSysManip objects are appended to
 *      the specified list and are exact opposites
 *      to the CfgSysManip objects that were stored
 *      in the logger. That is, if the logger registered
 *      that something was added to CONFIG.SYS, we
 *      create an object which removes that text again,
 *      and vice versa.
 *
 *      See CfgSysManip::CfgSysManip for an example usage.
 *
 *      This returns the number of items created.
 */

int CfgSysManip::AddToUndoList(list<CfgSysManip*> &List,
                               CfgSysDoneLogger &Logger)
{
    // the logger in this case has a list of special PSZs,
    // where the first three characters of each entry
    // signify to us what was changed in CONFIG.SYS

    PSZ     pszLogStart = Logger.pabLogString,
            pszLogThis = pszLogStart;
    int     iCount = 0;

    while (pszLogThis < pszLogStart + Logger.cbLogString)
    {
        ULONG   cbLogThis = strlen(pszLogThis);
        PSZ     pszNewModifiers = 0;

        // now check what we have in the log; the
        // first three characters (followed by a
        // space) siginy what happened:
        //  --  "DLL":      deleted line
        //  --  "DLP":      deleted part of line
        //  --  "NWL":      added an all new line
        //  --  "NWP":      added a new part to an existing line

        if (memicmp(pszLogThis, "DLL ", 4) == 0)
        {
            // line was deleted: re-insert that line (UNIQUE mode)
            pszNewModifiers = "UNIQUE";
        }
        else if (memicmp(pszLogThis, "DLP ", 4) == 0)
        {
            // part of line was deleted: re-insert that line (ADDRIGHT mode)
            // line was deleted: re-insert that line (UNIQUE mode)
            pszNewModifiers = "ADDRIGHT";
        }
        else if (memicmp(pszLogThis, "NWL ", 4) == 0)
        {
            // line was added: remove that whole line
            pszNewModifiers = "REMOVELINE";
        }
        else if (memicmp(pszLogThis, "NWP ", 4) == 0)
            // part of line was added: remove that part
            pszNewModifiers = "REMOVEPART";

        if (pszNewModifiers == 0)
            // none of the above: error
            throw(ConfigExcpt(CFGEXCPT_PARSELOG, strdup(pszLogThis)));
        else
        {
            // something found: compose attribute string for manipulator
            PSZ pszNewAttrs = (PSZ)malloc(strlen(pszLogThis)
                                          + strlen(pszNewModifiers)
                                          + 10);
            sprintf(pszNewAttrs, "%s | %s",
                    pszLogThis + 4,    // stuff after "DLL " or whatever
                    pszNewModifiers);

            // add the undo manipulator to the _front_ of the
            // list so that items are undone in reverse order
            // (because if a line was replaced originally,
            // we first have a "delete line" and then an
            // "add line" entry in the log)
            List.push_front(new CfgSysManip(pszNewAttrs));
            free(pszNewAttrs);
        }

        pszLogThis += cbLogThis + 1;    // go beyond null byte
        iCount++;
    }

    return (iCount);
}

/*******************************************************************
 *                                                                 *
 *  ConfigSys class                                                *
 *                                                                 *
 ******************************************************************/

/*
 *@@ ConfigSys:
 *      the constructor, which loads the current CONFIG.SYS
 *      file from OS/2 boot drive into instance memory.
 */

ConfigSys::ConfigSys()
           throw (ConfigExcpt)
{
    pszContent = NULL;

    sprintf(szFilename, "%c:\\config.sys", doshQueryBootDrive());
    // now read CONFIG.SYS file to initialize the dlg items
    APIRET arc = doshReadTextFile(szFilename, &pszContent);
    if (arc != NO_ERROR)
        throw(ConfigExcpt(CFGEXCPT_OPEN, arc));
}

/*
 *@@ ~ConfigSys:
 *      the destructor.
 *
 *      This does _not_ write the file. Use ConfigSys::WriteBack()
 *      before deleting an instance of this.
 */

ConfigSys::~ConfigSys()
{
    if (pszContent)
        free(pszContent);
}

/*
 *@@ Manipulate:
 *      this monster method changes the data in our memory copy of
 *      CONFIG.SYS according to the CfgSysManip object, which you must
 *      have created before. See CfgSysManip::CfgSysManip for a
 *      description of this usage.
 *
 *      This also takes a CfgSysDoneLogger as input where all the
 *      changes made to the CONFIG.SYS memory copy are logged. This
 *      logger can then be passed to ConfigSys::Undo to have the
 *      changes undone again.
 *
 *      Call ConfigSys::WriteBack to have CONFIG.SYS written back to disk.
 *
 *      This throws ConfigExcpt(CFGEXCPT_MANIPULATE, 0)) upon errors.
 */

int ConfigSys::Manipulate(CfgSysManip& Manip,       // in: manipulator object
                          CfgSysDoneLogger* pLogger) // in: logger object to append items to (can be NULL)
               throw(ConfigExcpt)
{
    int irc = 0;

    PSZ     pszCommand = 0,
            pszArgNew = 0;
    BOOL    fIsAllUpperCase = FALSE;

    /*
     * Prepare data
     *
     */

    // First, check if the new-line string has a "=" char
    PSZ pSep = strchr(Manip.pszNewLine, '=');
    if (pSep)
    {
        // yes: separate in two strings
        pszCommand = strhSubstr(Manip.pszNewLine, pSep);
        pszArgNew = strdup(pSep+1);
    }
    else
    {
        // no:
        pszCommand = strdup(Manip.pszNewLine);
        pszArgNew = 0;
    }

    // The CONFIG.SYS manipulation is a six-step process,
    // depending on the data in the CfgSysManip instance.
    // This is an outline what needs to be done when;
    // the implementation follows below.

    // 1)  First find out if a line needs to be updated or deleted
    //     or if we only need to write a new line.
    //     This depends on CfgSysManip.iReplaceMode.
           BOOL    fFindLine = FALSE;

    // 2)  If we need to find a line, find the
    //     line and copy it into pszLineToUpdate.
    //     Set pInsert to the original line, if found.
    //     If not found, set fInsertWRTVertical to TRUE.
           PSZ     pLineToUpdate = 0;
                // pointer to that line in original buffer; NULL if not found
           PSZ     pInsert = 0;
                // address of that line in original buffer if (fFindLine == TRUE) and found
           BOOL    fInsertWRTVertical = FALSE;
                // TRUE means set a new pInsert depending on Manip.iVertical;
                // this is set to TRUE if (fFindLine == FALSE) or line not found

    // 3)  If a line needs to be deleted, well, delete it then.
           PSZ     pLineToDelete = 0;
                // pointer to start of original line (pLineToUpdate);
                // points to character right after \r\n
           PSZ     pszDeletedLine = 0;
                // copy of that line if deleted (this has the stuff between
                // the EOL chars, not including those)

    // 4)  If (fInsertWRTVertical == TRUE), set pInsert to a position in
    //     the original buffer, depending on Manip.iVertical.

    // 5)  Compose a new line to be inserted at pInsert.
           PSZ     pszLineToInsert = 0;

    // 6)  Insert that line at pInsert.

    /*
     * GO!!
     *
     */

    // **** 1): do we need to search for a line?
    switch (Manip.iReplaceMode)
    {
        case CFGRPL_ADD:
            fFindLine = FALSE;
            fInsertWRTVertical = TRUE;      // always find new position
        break;

        case CFGRPL_UNIQUE:
        case CFGRPL_ADDLEFT:
        case CFGRPL_ADDRIGHT:
        case CFGRPL_REMOVELINE:
        case CFGRPL_REMOVEPART:
            fFindLine = TRUE;               // search line first; this sets
                                            // fInsertWRTVertical if not found
        break;
    }

    // **** 2): search line, if needed
    if (fFindLine)
    {
        pLineToUpdate = strhFindKey(pszContent,             // CONFIG.SYS text
                                    pszCommand,             // stuff to search for
                                    &fIsAllUpperCase);

        if (pLineToUpdate)
        {
            // line found:
            pLineToDelete = pLineToUpdate;
            pInsert = pLineToUpdate;
        }
        else
            // line not found:
            // insert new line then,
            // respecting the iVertical flag,
            // unless we are in "remove" mode
            if (    (Manip.iReplaceMode != CFGRPL_REMOVELINE)
                 && (Manip.iReplaceMode != CFGRPL_REMOVEPART)
               )
                fInsertWRTVertical = TRUE;
    }

    // **** 3): delete original line (OK)
    if (pLineToDelete)
    {
        // pLineToDelete is only != NULL if we searched and found a line above
        // and points to the character right after \r\n;
        // find end of this line
        PSZ pEOL = strhFindEOL(pLineToDelete, NULL);
        // pEOL now points to the \r char or the terminating 0 byte;
        // if not null byte, advance pointer
        PSZ pNextLine = pEOL;
        if (*pNextLine == '\r')
            pNextLine++;
        if (*pNextLine == '\n')
            pNextLine++;

        // make a copy of the line to be deleted;
        // this might be needed later
        pszDeletedLine = strhSubstr(pLineToDelete, pEOL);

        // overwrite beginning of line with next line;
        // the buffers overlap, but this works
        strcpy(pLineToDelete, pNextLine);
    }

    // **** 4):
    if (fInsertWRTVertical)
    {
        // this is only TRUE if a) we didn't search for a line
        // or b) we did search, but we didn't find a line
        // (but never with CFGRPL_REMOVE* mode);
        // we need to find a new vertical position in CONFIG.SYS then
        // and set pInsert to the beginning of a line (after \r\n)
        switch (Manip.iVertical)
        {
            case CFGVRT_BOTTOM:
                // insert at the very bottom (default)
                pInsert = pszContent + strlen(pszContent);
            break;

            case CFGVRT_TOP:
                // insert at the very top
                pInsert = pszContent;
            break;

            case CFGVRT_BEFORE: {
                // Manip.pszSearchString has the search string
                // before whose line we must insert
                PSZ pFound = strhistr(pszContent, Manip.pszSearchString);
                if (pFound)
                {
                    // search string found:
                    // find beginning of line
                    while (pFound > pszContent)
                        if ((*pFound == '\r') || (*pFound == '\n'))
                            break; // while
                        else
                            pFound--;

                    pInsert = pFound;
                } else
                    // search string not found: insert at top then
                    pInsert = pszContent;
            break; }

            case CFGVRT_AFTER:
                // Manip.pszSearchString has the search string
                // after whose line we must insert
                PSZ pFound = strhistr(pszContent, Manip.pszSearchString);
                if (pFound)
                {
                    // search string found:
                    // find end of line
                    pInsert = strhFindNextLine(pFound, NULL);
                } else
                    // search string not found: insert at bottom then
                    pInsert = pszContent + strlen(pszContent);
            break;
        }
    }

    // **** 5): compose new line
    switch (Manip.iReplaceMode)
    {
        case CFGRPL_ADD:
        case CFGRPL_UNIQUE:
            // that's easy, the new line is simply given
            pszLineToInsert = strdup(Manip.pszNewLine);
        break;

        case CFGRPL_REMOVELINE:
            // that's easy, just leave pszLineToInsert = 0;
        break;

        case CFGRPL_REMOVEPART:
            if (pszDeletedLine)
            {
                // parse the line which we removed above,
                // find the part which was to be removed,
                // and set pszLineToInsert to the rest of that
                PSZ     pArgToDel = 0;
                // find the subpart to look for
                if (pArgToDel = strhistr(pszDeletedLine, pszArgNew))
                {
                    // pArgToDel now has the position of the
                    // part to be deleted in the old part
                    pszLineToInsert = strdup(pszDeletedLine);
                    strhReplace2(&pszLineToInsert,
                                 pszArgNew,   // search string
                                 "");         // replacement string
                }
                else
                    // subpart not found:
                    // reinsert the old line
                    pszLineToInsert = strdup(pszDeletedLine);
            }
            // else line to be deleted was not found: leave pszLineToInsert = 0;
        break;

        case CFGRPL_ADDLEFT:
        case CFGRPL_ADDRIGHT:
            // add something to the left or right of the existing
            // argument:
            // check if we found a line above
            if (pszDeletedLine)
            {
                PSZ     pszCommandOld = 0,
                        pszArgOld = 0;
                // yes: split the existing line
                pSep = strchr(pszDeletedLine, '=');
                if (pSep)
                {
                    // "=" found: separate in two strings
                    pszCommandOld = strhSubstr(pszDeletedLine, pSep);
                    pszArgOld = strdup(pSep+1);
                }
                else
                    // no "=" found: that's strange
                    throw(ConfigExcpt(CFGEXCPT_NOSEPARATOR,
                                      strdup(pszDeletedLine)));

                if ((pszCommandOld) && (pszArgOld))
                {
                    // does the new arg exist in old arg already?
                    if (strhistr(pszArgOld, pszArgNew) == 0)
                    {
                        // no: go!
                        ULONG   cbCommandOld = strlen(pszCommandOld);
                        ULONG   cbArgOld = strlen(pszArgOld);
                        ULONG   cbArgNew = strlen(pszArgNew);   // created at the very top

                        // compose new line
                        pszLineToInsert = (PSZ)malloc(cbCommandOld
                                                        + cbArgOld
                                                        + cbArgNew
                                                        + 5);

                        if (Manip.iReplaceMode == CFGRPL_ADDLEFT)
                            // add new arg to the left:
                            sprintf(pszLineToInsert,
                                    "%s=%s;%s",
                                    pszCommandOld,
                                    pszArgNew,
                                    pszArgOld);
                        else
                            // add new arg to the right:
                            if (*(pszArgOld + cbArgOld - 1) == ';')
                                // old arg has terminating ";" already:
                                sprintf(pszLineToInsert,
                                        "%s=%s%s",
                                        pszCommandOld,
                                        pszArgOld,
                                        pszArgNew);
                            else
                                // we need to append a ";":
                                sprintf(pszLineToInsert,
                                        "%s=%s;%s",
                                        pszCommandOld,
                                        pszArgOld,
                                        pszArgNew);
                    } // end if (stristr(pszArgOld, pszArgNew) == 0)
                    else
                        // exists already:
                        // reinsert the line we deleted above
                        pszLineToInsert = strdup(pszDeletedLine);

                    free(pszCommandOld);
                    free(pszArgOld);
                }
            } // end if (pszDeletedLine)
            else
                // line was not found: add a new one then
                // (the position has been determined above)
                pszLineToInsert = strdup(Manip.pszNewLine);
        break;
    }

    // **** 6): insert new line at pInsert
    if (pszLineToInsert)
    {
        PSZ     pszInsert2 = 0;
        ULONG   cbInsert = strlen(pszLineToInsert);
        BOOL    fLineBreakNeeded = FALSE;

        // check if the char before pInsert is EOL
        if (pInsert > pszContent)
            if (    (*(pInsert-1) != '\r')
                 && (*(pInsert-1) != '\n')
               )
                fLineBreakNeeded = TRUE;

        if (fLineBreakNeeded)
        {
            // not EOL (this might be the case if we're
            // appending to the very bottom of CONFIG.SYS)
            pszInsert2 = (PSZ)malloc(cbInsert + 5);
            // insert an extra line break then
            memcpy(pszInsert2, "\r\n", 2);
            memcpy(pszInsert2 + 2, pszLineToInsert, cbInsert);
            strcpy(pszInsert2 + cbInsert + 2, "\r\n");
        }
        else
        {
            // OK:
            pszInsert2 = (PSZ)malloc(cbInsert + 3);
            memcpy(pszInsert2, pszLineToInsert, cbInsert);
            strcpy(pszInsert2 + cbInsert, "\r\n");
        }

        PSZ pszNew = strhInsert(pszContent,         // source
                                pszInsert2,         // new
                                pInsert);           // where to insert
        if (pszNew)
        {
            free(pszInsert2);
            free(pszContent);
            pszContent = pszNew;
        }
        else
            throw(ConfigExcpt(CFGEXCPT_MANIPULATE, 0));
    }
    else
        if (    (Manip.iReplaceMode != CFGRPL_REMOVELINE)
             && (Manip.iReplaceMode != CFGRPL_REMOVEPART)
           )
            throw(ConfigExcpt(CFGEXCPT_MANIPULATE, 0));

    // finally, create log entries for Undo().
    // This uses the first three characters of the
    // each log entry to signify what happend:
    //  --  "DLL":      deleted line
    //  --  "DLP":      deleted part of line
    //  --  "NWL":      added an all new line
    //  --  "NWP":      added a new part to an existing line
    // There may be several log entries per manipulator
    // if a line was deleted and then re-inserted (i.e. updated).
    if (pLogger)
        switch (Manip.iReplaceMode)
        {
            case CFGRPL_UNIQUE:
                if (pszDeletedLine)
                    // did we delete a line?
                    pLogger->Append("DLL ", pszDeletedLine);

            // no break, because we also added something

            case CFGRPL_ADD:
                if (pszLineToInsert)
                    // added a line:
                    pLogger->Append("NWL ", pszLineToInsert);
            break;

            case CFGRPL_ADDLEFT:
            case CFGRPL_ADDRIGHT:
                if (pszDeletedLine)
                    // a line was found and updated:
                    // store the new part only
                    pLogger->Append("NWP ", Manip.pszNewLine);
                else
                    pLogger->Append("NWL ", pszLineToInsert);
            break;

            case CFGRPL_REMOVELINE:
                if (pszDeletedLine)
                    // did we delete a line?
                    pLogger->Append("DLL ", pszDeletedLine);
            break;

            case CFGRPL_REMOVEPART:
                if (pszDeletedLine)
                    // did we delete a line?  xxx
                    pLogger->Append("DLP ", Manip.pszNewLine);
            break;
        }

    // clean up
    if (pszCommand)
        free(pszCommand);
    if (pszArgNew)
        free(pszArgNew);
    if (pszDeletedLine)
        free(pszDeletedLine);
    if (pszLineToInsert)
        free(pszLineToInsert);

    return (irc);
}

/*
 *@@ Undo:
 *      this attempts to undo a previous manipulation of
 *      CONFIG.SYS. To this method, you should pass the
 *      same (or recreated) CfgSysManip object that you
 *      passed to ConfigSys::Manipulate earlier.
 *
 *      Call ConfigSys::WriteBack to have CONFIG.SYS written back to disk.
 *
 *      This throws ConfigExcpt upon errors:
 *      --  CFGEXCPT_NOUNDOREMOVE: REMOVELINE and REMOVEPART
 *          cannot be undone, because we cannot know what
 *          the previous line was.
 *      --
 */

int ConfigSys::Undo(CfgSysManip& Manip)
               throw (ConfigExcpt)
{
    // What we do here is analyse the CfgSysManip object
    // to find out what might have been changed and create
    // a second object which undoes the change. We then
    // call Manipulate with the second object.

    switch (Manip.iReplaceMode)
    {
    }
}

/*
 *@@ WriteBack:
 *      this rewrites the CONFIG.SYS file on disk with the
 *      data we have in memory.
 *
 *      This makes a backup copy in "CONFIG.003" style if
 *      (fBackup == TRUE). See doshCreateBackupFileName for
 *      details.
 */

int ConfigSys::WriteBack(bool fBackup)      // in: create backup?
               const
{
    int irc = 0;
    if (pszContent)
        doshWriteTextFile(szFilename, pszContent,
                          fBackup);        // create backup
    else
        throw(ConfigExcpt(CFGEXCPT_MANIPULATE, 0));
    return (irc);
}

/*******************************************************************
 *                                                                 *
 *  RegisterClass class                                            *
 *                                                                 *
 ******************************************************************/

/*
 *@@ RegisterClass:
 *      this constructor translates a REGISTERCLASS attribute (as used
 *      in the PCK tag and stored in the database) into the RegisterClass
 *      instance data.
 */

RegisterClass::RegisterClass(char* pszRegisterClass)
               throw(ConfigExcpt)
{
    pszClassName = 0;
    pszModule = 0;

    // find separator
    PSZ pBegin = pszRegisterClass;
    PSZ pEnd = strchr(pBegin, '|');
    if (pEnd)
    {
        // strip trailing spaces
        PSZ pEnd2 = pEnd;
        while ((*pEnd2 == ' ') && (pEnd2 > pszRegisterClass))
            pEnd2--;

        pszClassName = strhSubstr(pBegin, pEnd2);
        if (!pszClassName)
            throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszRegisterClass)));
        else
        {
            pBegin = pEnd + 1;
            // strip leading spaces
            while ((*pBegin) && (*pBegin == ' '))
                pBegin++;
            pszModule = strdup(pBegin);
            if (!pszModule)
                throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszRegisterClass)));
        }
    }
    else
        throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszRegisterClass)));
}

/*
 *@@ ~RegisterClass:
 *      the destructor.
 */

RegisterClass::~RegisterClass()
{
    if (pszClassName)
        free(pszClassName);
    if (pszModule)
        free(pszModule);
}

/*
 *@@ IsRegistered:
 *      this returns TRUE if a class of the same WPS class name
 *      as this instance is already registered with the WPS.
 *      In that case, the module of the registered class is
 *      copied into pszBuffer, which should be sufficient in size.
 *
 *      This throws a ConfigExcpt with REGEXCPT_QUERYCLASSLIST
 *      if the class list could not be queried.
 */

bool RegisterClass::IsRegistered(char* pszBuffer)
                    const
{
    BOOL    fInstalled = FALSE;
    PBYTE   pClassList = winhQueryWPSClassList(),
            pClassThis = 0;
    if (pClassList)
    {
        if (pClassThis = winhQueryWPSClass(pClassList, pszClassName))
        {
            fInstalled = TRUE;
            if (pszBuffer)
                strcpy(pszBuffer, ((POBJCLASS)pClassThis)->pszModName);
        }
        free(pClassList);
    }
    else
        throw(ConfigExcpt(REGEXCPT_QUERYCLASSLIST, 0));

    return (fInstalled);
}

/*
 *@@ Register:
 *      this attempts to register the class.
 *
 *      If (fReplace == TRUE), we do not call
 *      RegisterClass::IsRegistered before registering
 *      this, i.e. we will always register the class,
 *      even if it's already registered.
 *
 *      If (fReplace == FALSE) and RegisterClass::IsRegistered
 *      returned TRUE, we throw a ConfigExcpt with
 *      REGEXCPT_ALREADYREGISTERED and pszSubstr set to the
 *      file name of the registered DLL. Note that IsRegistered might
 *      in turn throw ConfigExcpt with REGEXCPT_QUERYCLASSLIST.
 *
 *      If registering the class failed, this throws a
 *      ConfigExcpt with REGEXCPT_REGISTER and iData set
 *      to the APIRET of winhRegisterClass.
 *
 *      This method can take a DoneLogger object as input, which
 *      can later be used with the DeregisterClass::AddToUndoList
 *      static class method to easily create a list of DeregisterClass
 *      objects to undo the changes made.
 *
 *      Example usage (exception handling omitted):
 +          RegisterDoneLogger Logger;
 +                      // create logger instance
 +          RegisterClass RegClass("XFolder;C:\XFOLDER\XFLDR.DLL");
 +                      // create RegisterClass instance:
 +          RegClass.Register(TRUE, Logger);
 +          ... // register more classes with the same logger
 +
 +          // now create undo list
 +          list<DeregisterClass*> List;
 +          DeregisterClass::AddToUndoList(List, Logger);
 +          list<DeregisterClass*>::iterator DeregStart = List.begin(),
 +                                           DeregEnd = List.end();
 +          for (; DeregStart != DeregEnd; DeregStart++)
 +              (**DeregStart).Deregister;
 */

int RegisterClass::Register(bool fReplace,
                            RegisterDoneLogger* pLogger)      // in: logger object to append items to (can be NULL)
                            const throw(ConfigExcpt)
{
    int irc = 0;

    if (fReplace == FALSE)
    {
        char szBuffer[300];
        if (IsRegistered(szBuffer))
            throw(ConfigExcpt(REGEXCPT_ALREADYREGISTERED, strdup(szBuffer)));
    }

    CHAR   szBuf[1000];
    APIRET arc = winhRegisterClass(pszClassName,
                                   pszModule,
                                   szBuf,
                                   sizeof(szBuf));
    // always record in logger, even if failed,
    // because the class is in the class list anyways
    if (pLogger)
        pLogger->Append(pszClassName);

    if (arc != NO_ERROR)
        throw(ConfigExcpt(REGEXCPT_REGISTER, arc));

    return (irc);
}

/*******************************************************************
 *                                                                 *
 *  DeregisterClass class                                          *
 *                                                                 *
 ******************************************************************/

/*
 *@@ DeregisterClass:
 *      the constructor, which in this case takes a simple
 *      PSZ class name as input.
 */

DeregisterClass::DeregisterClass(char* pszDeregisterClass)
{
    pszClassName = strdup(pszDeregisterClass);
}

/*
 *@@ ~DeregisterClass:
 *      the destructor.
 */

DeregisterClass::~DeregisterClass()
{
    if (pszClassName)
        free(pszClassName);
}

/*
 *@@ Deregister:
 *      this attempts to deregister the class.
 *      Throws a ConfigExcpt upon failure.
 */

int DeregisterClass::Deregister()
                     const
                     throw(ConfigExcpt)
{
    if (!WinDeregisterObjectClass(pszClassName))
        throw(ConfigExcpt(REGEXCPT_DEREGISTER, 0));
}

/*
 *@@ AddToUndoList:
 *      static class method to create DeregisterClass
 *      instances from the logger instance that was
 *      previously used with RegisterClass::Register.
 *
 *      The new DeregisterClass objects are appended to
 *      the specified list.
 *
 *      See RegisterClass::Register for an example usage.
 *
 *      This returns the number of items created.
 */

int DeregisterClass::AddToUndoList(list<DeregisterClass*> &List,
                                   RegisterDoneLogger &Logger)
{
    // the logger in this case has a simple list of PSZs with all
    // the class names that were registered, each of which is terminated
    // with a null character

    PSZ     pszLogStart = Logger.pabLogString,
            pszLogThis = pszLogStart;
    int     iCount = 0;

    while (pszLogThis < pszLogStart + Logger.cbLogString)
    {
        ULONG cbLogThis = strlen(pszLogThis);
        List.push_back(new DeregisterClass(pszLogThis));
        pszLogThis += cbLogThis + 1;    // go beyond null byte
        iCount++;
    }

    return (iCount);
}

/*******************************************************************
 *                                                                 *
 *  ReplaceClass class                                             *
 *                                                                 *
 ******************************************************************/

/*
 *@@ ReplaceClass:
 *      this constructor translates a REPLACECLASS attribute (as used
 *      in the PCK tag and stored in the database) into the ReplaceClass
 *      instance data.
 */

ReplaceClass::ReplaceClass(char* pszReplaceClass)
              throw(ConfigExcpt)
{
    pszOldClassName = 0;
    pszNewClassName = 0;

    // find separator
    PSZ pBegin = pszReplaceClass;
    PSZ pEnd = strchr(pBegin, '|');
    if (pEnd)
    {
        // strip trailing spaces
        PSZ pEnd2 = pEnd;
        while ((*pEnd2 == ' ') && (pEnd2 > pszReplaceClass))
            pEnd2--;

        pszOldClassName = strhSubstr(pBegin, pEnd2);
        if (!pszOldClassName)
            throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszReplaceClass)));
        else
        {
            pBegin = pEnd + 1;
            // strip leading spaces
            while ((*pBegin) && (*pBegin == ' '))
                pBegin++;
            pszNewClassName = strdup(pBegin);
            if (!pszNewClassName)
                throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszReplaceClass)));
        }
    }
    else
        throw(ConfigExcpt(REGEXCPT_SYNTAX, strdup(pszReplaceClass)));
}

/*
 *@@ ~ReplaceClass:
 *      the destructor.
 */

ReplaceClass::~ReplaceClass()
{
    if (pszOldClassName)
        free(pszOldClassName);
    if (pszNewClassName)
        free(pszNewClassName);
}

/*
 *@@ Replace:
 *      this attempts to replace the class or undo
 *      an existing replacement (if fReplace == FALSE).
 *
 *      If replacing the class failed, this throws a
 *      ConfigExcpt with REGEXCPT_REPLACE.
 *
 *      Example usage (exception handling omitted):
 +          ReplaceDoneLogger Logger;
 +                      // create logger instance
 +          ReplaceClass ReplClass("WPFolder;XFolder");
 +                      // create ReplaceClass instance:
 +          ReplClass.Replace(TRUE, Logger);
 +          ... // replace more classes with the same logger
 +
 +          // now create undo list
 +          list<UnreplaceClass*> List;
 +          UnreplaceClass::AddToUndoList(List, Logger);
 +          list<UnreplaceClass*>::iterator UnrplStart = List.begin(),
 +                                          UnrplEnd = List.end();
 +          for (; UnrplStart != UnrplEnd; UnrplStart++)
 +              (**UnrplStart).Unreplace;
 */

int ReplaceClass::Replace(ReplaceDoneLogger* pLogger)      // in: logger object to append items to (can be NULL)
                  const
                  throw(ConfigExcpt)
{
    int irc = 0;

    if (!WinReplaceObjectClass(pszOldClassName, pszNewClassName,
                               TRUE))       // replace
        throw(ConfigExcpt(REGEXCPT_REPLACE, 0));

    return (irc);
}

/*******************************************************************
 *                                                                 *
 *  UnreplaceClass class                                           *
 *                                                                 *
 ******************************************************************/

/*
 *@@ Unreplace:
 *      this attempts to unreplace the class.
 *      Throws a ConfigExcpt upon failure.
 */

int UnreplaceClass::Unreplace() const throw(ConfigExcpt)
{
    if (!WinReplaceObjectClass(pszOldClassName, pszNewClassName, FALSE))
        throw(ConfigExcpt(REGEXCPT_UNREPLACE, 0));
}

/*
 *@@ AddToUndoList:
 *      static class method to create UnreplaceClass
 *      instances from the logger instance that was
 *      previously used with RegisterClass::Register.
 *
 *      The new UnreplaceClass objects are appended to
 *      the specified list.
 *
 *      See ReplaceClass::Replace for an example usage.
 *
 *      This returns the number of items created.
 */

int UnreplaceClass::AddToUndoList(list<UnreplaceClass*> &List,
                                  ReplaceDoneLogger &Logger)
{
    // the logger in this case has a simple list of PSZs with all
    // the class names that were registered, each of which is terminated
    // with a null character

    PSZ     pszLogStart = Logger.pabLogString,
            pszLogThis = pszLogStart;
    int     iCount = 0;

    while (pszLogThis < pszLogStart + Logger.cbLogString)
    {
        ULONG cbLogThis = strlen(pszLogThis);
        List.push_back(new UnreplaceClass(pszLogThis));
                                    // this calls the inherited ReplaceClass
                                    // constructor, which will parse the thing
        pszLogThis += cbLogThis + 1;    // go beyond null byte
        iCount++;
    }

    return (iCount);
}

/*******************************************************************
 *                                                                 *
 *  CreateWPSObject class                                          *
 *                                                                 *
 ******************************************************************/

/*
 *@@ CreateWPSObject:
 *      this constructor translates a CREATEOBJECT attribute (as used
 *      in the PCK tag and stored in the database) into the CreateWPSObject
 *      instance data.
 */

CreateWPSObject::CreateWPSObject(char* pszCreateObject)
                 throw(ConfigExcpt)
{
    // extract class name
    PSZ pBegin = pszCreateObject;
    PSZ pEnd = strchr(pBegin, '|');
    if (pEnd)
    {
        pszClassName = strhSubstr(pBegin, pEnd);

        // extract title
        pBegin = pEnd+1;
        pEnd = strchr(pBegin, '|');
        if (pEnd)
        {
            pszTitle = strhSubstr(pBegin, pEnd);

            // extract location
            pBegin = pEnd+1;
            pEnd = strchr(pBegin, '|');
            if (pEnd)
            {
                // yet another separator found: we then have a config parameter
                pszLocation = strhSubstr(pBegin, pEnd);

                // extract config (rest of pszCreateObject)
                pBegin = pEnd+1;
                pszSetupString = strdup(pBegin);
            }
            else
                // no separator after "location" found:
                // duplicate whole string
                      pszLocation = strdup(pBegin);

            if (!pszLocation)
                throw(ConfigExcpt(WPOEXCPT_NOLOCATION, strdup(pszCreateObject)));
        }
        else
            throw(ConfigExcpt(WPOEXCPT_NOTITLE, strdup(pszCreateObject)));
    }
    else
        throw(ConfigExcpt(WPOEXCPT_NOCLASS, strdup(pszCreateObject)));
}

/*
 *@@ ~CreateWPSObject:
 *      the destructor.
 */

CreateWPSObject::~CreateWPSObject()
{
    if (pszClassName)
        free(pszClassName);
    if (pszTitle)
        free(pszTitle);
    if (pszSetupString)
        free(pszSetupString);
    if (pszLocation)
        free(pszLocation);
}

/*
 *@@ CreateObject:
 *      this attempts to create the WPS object
 *      with the instance data.
 *
 *      Throws a ConfigExcpt with WPOEXCPT_CREATE
 *      if this fails.
 */

void CreateWPSObject::CreateObject(WPSObjectsDoneLogger* pLogger) // in: logger object to append items to (can be NULL)
                      throw(ConfigExcpt)
{
    hobj = WinCreateObject(pszClassName,
                           pszTitle,
                           pszSetupString,
                           pszLocation,
                           (fReplace)
                               ? CO_REPLACEIFEXISTS
                               : CO_UPDATEIFEXISTS);
    if (hobj == NULLHANDLE)
        throw(ConfigExcpt(WPOEXCPT_CREATE, strdup(pszTitle)));

    // store the object ID in the logger
    PSZ pObjectID = strhistr(pszSetupString, "OBJECTID=<");
    if (pObjectID)
    {
        PSZ pBegin = pObjectID + 9;
        PSZ pEnd = strchr(pBegin, '>');
        if (pEnd)
        {
            PSZ pszObjectID = strhSubstr(pBegin, pEnd+1);
            if (pLogger)
                pLogger->Append(pszObjectID);
            free(pszObjectID);
        }
    }
}

/*******************************************************************
 *                                                                 *
 *  DeleteWPSObject class                                          *
 *                                                                 *
 ******************************************************************/

/*
 *@@ DeleteWPSObject:
 *      the constructor, which in this case takes a simple
 *      PSZ object ID as input.
 */

DeleteWPSObject::DeleteWPSObject(char* pszID2Delete)
{
    pszObjectID = strdup(pszID2Delete);
}

/*
 *@@ ~DeleteWPSObject:
 *      the destructor.
 */

DeleteWPSObject::~DeleteWPSObject()
{
    if (pszObjectID)
        free(pszObjectID);
}

/*
 *@@ Delete:
 *      this attempts to delete the object.
 *      Throws a ConfigExcpt upon failure.
 */

int DeleteWPSObject::Delete() const throw(ConfigExcpt)
{
    HOBJECT     hobj = WinQueryObject(pszObjectID);
    if (!hobj)
        throw(ConfigExcpt(WPOEXCPT_DELETEOBJECT, 0));

    if (!WinDestroyObject(hobj))
        throw(ConfigExcpt(WPOEXCPT_DELETEOBJECT, 0));
}

/*
 *@@ AddToUndoList:
 *      static class method to create DeleteWPSObject
 *      instances from the logger instance that was
 *      previously used with RegisterClass::Register.
 *
 *      The new DeleteWPSObject objects are appended to
 *      the specified list.
 *
 *      This returns the number of items created.
 */

int DeleteWPSObject::AddToUndoList(list<DeleteWPSObject*> &List,
                                   WPSObjectsDoneLogger &Logger)
{
    // the logger in this case has a simple list of PSZs with all
    // the object IDs that were created, each of which is terminated
    // with a null character

    PSZ     pszLogStart = Logger.pabLogString,
            pszLogThis = pszLogStart;
    int     iCount = 0;

    while (pszLogThis < pszLogStart + Logger.cbLogString)
    {
        ULONG cbLogThis = strlen(pszLogThis);
        List.push_back(new DeleteWPSObject(pszLogThis));
        pszLogThis += cbLogThis + 1;    // go beyond null byte
        iCount++;
    }

    return (iCount);
}


