(* access of pci hardware *)

{&Use32+}
{$G+}
{$R+}
{$S-}
{$I+}
{$N-}

unit pci_hw;

interface

type
  {$IfDef VirtualPascal}
  PWord         =^SmallWord;
  {$Else}
  SmallWord     =Word;
  {$EndIf}
  PLongint      =^Longint;

var
  errcode       :byte;
  failed        :boolean;
  usebios       :boolean;

  PCIverhi,
  PCIverlo,
  PCIchar,
  PCI_hibus     :byte;

  conmap,
  len           :word;

  irqbuff       : array[0..1023+6] of byte;
  irqbuffR      :
    packed record
      length_of_IRQ_routing_table_buffer        :SmallWord;
      IRQ_routing_table_array_buffer            :pointer;
      IRQ_routing_table_entry_Array             :array[0..63] of
        packed record
          case integer of
            1:(binary                             :array[0..15] of byte);
            2:(PCI_bus_number                     :byte;
               PCI_device_number                  :byte;
               INTABCD                            :array[0..3] of
                 packed record
                   link_value                     :byte;
                   IRQ_connectivity_bit_map       :SmallWord;
                 end;
               device_slot_number                 :byte;
               reserved                           :byte);
        end;
    end absolute irqbuff;

procedure pci_present_test;
function lookup(deviceid,func,bus:byte;index_:word) : byte;
procedure write_dword(deviceid,func,bus:byte;index_,data:longint);
procedure write_word(deviceid,func,bus:byte;index_,data:word);
procedure write_byte(deviceid,func,bus:byte;index_,data:byte);

function Mem_F000(const i:word):byte;
function MemW_F000(const i:word):smallword;
function MemL_F000(const i:word):longint;
procedure load_irqbuff;

{$IfDef OS2}
procedure Open_oemhlp;
procedure Close_oemhlp;
procedure os2_read_bios;
{$EndIf}

implementation

{$IfDef DPMI32}
uses
  Dpmi32Df,
  Dpmi32;
{$EndIf}

{$IfDef OS2}

uses
  Os2Base,
  Os2Def,
  Strings;

var
  biosf000:array[0..$ffff] of byte;

procedure os2_read_bios;

  var
    hand,
    action,
    rc                  :longint;

    ParmRec1:
      record            // Input parameter record
        phys32          :longint;
        laenge          :smallword;
      end;

    ParmRec2:
      record
        sel             :smallword;
      end;

    ParmLen             : ULong;  // Parameter length in bytes
    DataLen             : ULong;  // Data length in bytes
    Data1:
      record
        sel             :smallword;
      end;

  begin
    FillChar(biosf000,SizeOf(biosf000),0);

    if DosOpen('SCREEN$',hand,action,0,0,1,$40,nil)<>0 then
      exit;

    ParmLen:=SizeOf(ParmRec1);

    with ParmRec1 do
      begin
        phys32:=$000f0000;
        laenge:=0;
      end;

    datalen:=SizeOf(data1);
    rc:=DosDevIOCtl(
            hand,                       // Handle to device
            IOCTL_SCR_AND_PTRDRAW,      // Category of request
            SCR_ALLOCLDT,               // Function being requested
            @ParmRec1,                  // Input/Output parameter list
            ParmLen,                    // Maximum output parameter size
            @ParmLen,                   // Input:  size of parameter list
                                        // Output: size of parameters returned
            @Data1,                     // Input/Output data area
            Datalen,                    // Maximum output data size
            @DataLen);                  // Input:  size of input data area
    if rc=0 then
      begin

        asm {&Saves None}
          push gs

            sub esi,esi
            mov gs,data1.sel

            mov edi,offset biosf000
            mov ecx,$10000
            cld
          @l1:
            mov al,gs:[esi]
            inc esi
            stosb
            loop @l1

          pop gs
        end;

        ParmLen:=SizeOf(ParmRec2);

        with ParmRec2 do
          begin
            sel:=data1.sel;
          end;

        DataLen:=0;
        rc:=DosDevIOCtl(
                hand,                           // Handle to device
                IOCTL_SCR_AND_PTRDRAW,          // Category of request
                SCR_DEALLOCLDT,                 // Function being requested
                @ParmRec2,                      // Input/Output parameter list
                ParmLen,                        // Maximum output parameter size
                @ParmLen,                       // Input:  size of parameter list
                                                // Output: size of parameters returned
                nil,                            // Input/Output data area
                Datalen,                        // Maximum output data size
                @DataLen);                      // Input:  size of input data area

      end;

    DosClose(hand);
  end;

{$EndIf}

function Mem_F000(const i:word):byte;
  begin
    {$IfDef VirtualPascal}

      {$IfDef DPMI32}
      Mem_F000:=Mem[$f0000+i];
      {$EndIf}

      {$IfDef OS2}
      Mem_F000:=biosf000[i];
      {$EndIf}


    {$Else}
    Mem_F000:=Mem[$f000:i];
    {$EndIf}
  end;

function MemW_F000(const i:word):SmallWord;
  begin
    {$IfDef VirtualPascal}

      {$IfDef DPMI32}
      MemW_F000:=MemW[$f0000+i];
      {$EndIf}

      {$IfDef OS2}
      MemW_F000:=PWord(@biosf000[i])^;
      {$EndIf}


    {$Else}
    MemW_F000:=MemW[$f000:i];
    {$EndIf}
  end;

function MemL_F000(const i:word):longint;
  begin
    {$IfDef VirtualPascal}

      {$IfDef DPMI32}
      MemL_F000:=MemL[$f0000+i];
      {$EndIf}

      {$IfDef OS2}
      MemL_F000:=PLongint(@biosf000[i])^;
      {$EndIf}


    {$Else}
    MemL_F000:=MemL[$f000:i];
    {$EndIf}
  end;

{$IfDef OS2}
var
  oemhlp_handle :longint;

procedure open_oemhlp;
  begin
    if SysFileOpen('OEMHLP$',open_access_readonly+open_share_denynone,oemhlp_handle)<>0 then
       oemhlp_handle:=-1;
  end;

procedure close_oemhlp;
  begin
    SysFileClose(oemhlp_handle);
  end;
{$EndIf}

{$IfDef VirtualPascal}

{$IfDef OS2}
function lookup_bios(deviceid,func,bus:byte;index_:word) : byte;

  var
    para              :
      packed record
        subfuction    :byte;
        busnumber     :byte;
        devfuncnumber :byte;
        configregister:byte;
        size          :byte;
      end;

    data              :
      packed record
        returncode    :byte;
        data          :longint;
      end;

    para_len,data_len :longint;


  begin
    with para do
      begin
        subfuction:=3; (* read configuartion byte ($1a/$b108) *)
        busnumber:=bus;
        devfuncnumber:=deviceid shl 3+func;
        configregister:=index_;
        size:=SizeOf(byte);
      end;
    para_len:=SizeOf(para);

    with data do
      begin
        returncode:=0;
        data:=0;
      end;
    data_len:=SizeOf(data);

    errcode:=
      DosDevIoCtl(
        oemhlp_handle,
        $80,              (* oemhlp/testcfg/.. *)
        $0b,              (* PCI *)
        @para,SizeOf(para),@para_len,
        @data,SizeOf(data),@data_len);

    if errcode=$00 then
      begin
        failed:=false;
        lookup_bios:=Lo(data.data);
      end;
  end;


procedure write_data_bios(deviceid,func,bus:byte;index_,value:longint;datasize:byte);
  var
    para              :
      packed record
        subfuction    :byte;
        busnumber     :byte;
        devfuncnumber :byte;
        configregister:byte;
        size          :byte;
        data          :longint;
      end;

    data              :
      packed record
        returncode    :byte;
      end;

    para_len,data_len :longint;


  begin
    with para do
      begin
        subfuction:=4; (* write configuartion space *)
        busnumber:=bus;
        devfuncnumber:=deviceid shl 3+func;
        configregister:=index_;
        size:=datasize;
        data:=value;
      end;
    para_len:=SizeOf(para);

    with data do
      begin
        returncode:=0;
      end;
    data_len:=SizeOf(data);

    errcode:=
      DosDevIoCtl(
        oemhlp_handle,
        $80,              (* oemhlp/testcfg/.. *)
        $0b,              (* PCI *)
        @para,SizeOf(para),@para_len,
        @data,SizeOf(data),@data_len);

    if errcode=$00 then
      failed:=false;
  end;

procedure pci_present_test;
  var
    para              :
      packed record
        subfuction    :byte;
      end;

    data              :
      packed record
        returncode    :byte;
        hardwaremech  :byte;
        majorver      :byte;
        minorver      :byte;
        lastbus       :byte;
      end;

    para_len,data_len :longint;


  begin
    with para do
      begin
        subfuction:=0; (* Query PCI *)
      end;
    para_len:=SizeOf(para);

    FillChar(data,SizeOf(data),0);
    data_len:=SizeOf(data);

    errcode:=
      DosDevIoCtl(
        oemhlp_handle,
        $80,              (* oemhlp/testcfg/.. *)
        $0b,              (* PCI *)
        @para,SizeOf(para),@para_len,
        @data,SizeOf(data),@data_len);

    if errcode=$00 then
      with data do
        begin
          PCIchar:=hardwaremech;
          PCI_hibus:=lastbus;
          PCIverlo:=minorver;
          PCIverhi:=majorver;
          failed:=false;
        end;
  end;


procedure load_irqbuff;

  function compare(const bios;const code:string):boolean;
    var
      z               :longint;
      bios_a          :array[1..100] of char absolute bios;
    begin
      compare:=false;

      for z:=1 to Length(code) do
        if  (bios_a[z]<>code[z]) and (code[z]<>'?') then
          Exit
        else
          if z=Length(code) then
            begin
              compare:=true;
              Exit;
            end;
    end;

  const
    award_call        :string
      =#$3c#$0e                       // cmp al,$0e
      +#$75#$05                       // jne @al_0f
      +#$e8'??'                       // call pci_0e
      +#$eb'?'                        // jmp @exit
      +#$3c#$0f
      +#$75#$05
      +#$e8'??';

    award_pci_0e      :string
      =#$be'??'                       // mov si,0B90
      +#$8b#$0c                       // mov cx,[si]
      +#$26#$3b#$0d                   // cmp cx,es:[di]
      +#$26#$89#$0d                   // mov es:[di],cx
      +#$76#$06                       // jbe @0F9D6B
      +#$C6#$46#$01#$89               // mov byte ptr [bp+1],89
      +#$F9                           // stc
      +#$C3                           // ret
                                      // @0F9D6B:
      +#$06                           // push es
      +#$26#$C4#$7D#$02               // les di,es:[di+2]
      +#$BE'??'                       // mov si,0B20
      +#$C1#$E9#$02                   // shr cx,2
      +#$FC                           // cld
      +#$F3#$66#$A5                   // repz movsd
      +#$07                           // pop es
      +#$c6#$46#$01#$00               // mov byte ptr [bp+1],0
      +#$f8                           // clc
      +#$c3;                          // ret

    systemsoft_pci_0e :string
      =#$66#$57                       // push edi
      +#$F6#$C4#$40                   // test ah, 40h
      +#$75#$07                       // jnz @792E
                                      //
      +#$66#$81#$E7#$FF#$FF#$00#$00   // and edi,$FFFF
                                      //
      +#$67#$26#$81#$3F'??'           // cmp word ptr es:[edi],$80h
      +#$72'?'                        // jb zu_klein
                                      //
      +#$06                           // push es
      +#$66#$57                       // push edi
      +#$67#$26#$C7#$07#$00#$00       // mov word ptr es:[edi],0
      +#$F6#$C4#$40                   // test ah, 40h
      +#$74#$15                       // jz @7959
                                      //
      +#$67#$66#$26#$C4#$7F#$02       // les  edi, es:[edi+2]
      +#$66#$BE#$00#$80#$0E#$00       // mov esi,$E8000
      +#$66#$81#$C6'??'#$00#$00       // add esi,offset PCI_routing_tabelle
      +#$EB#$0B                       // jmp short @7964
                                      //
                                      // @7959:
      +#$67#$26#$C4#$7F#$02           // les di,es:[edi+2]
      +#$66#$BE'??'#$00#$00           // mov esi,offset PCI_routing_tabelle
                                      //
                                      // @7964:
      +#$66#$B9'??'#$00#$00;          // mov ecx,$80

  var
    ca,pci_0e,z       :longint;
  begin
    (* failed:=true; *)

    if StrLComp(@biosf000[$e000],'Award',Length('Award'))=0 then
      begin
        ca:=0;
        repeat

          if biosf000[ca]=Ord(award_call[1]) then
            if compare(biosf000[ca],award_call) then
              begin
                pci_0e:=ca+4+3+biosf000[ca+5]+biosf000[ca+6] shl 8;
                if compare(biosf000[pci_0e],award_pci_0e) then
                  begin
                    conmap:=0;
                    len:=PWord(@biosf000[PWord(@biosf000[pci_0e+1])^])^;
                    irqbuffR.length_of_IRQ_routing_table_buffer:=len;
                    Move(biosf000[PWord(@biosf000[pci_0e+25])^],irqbuffR.IRQ_routing_table_entry_Array,len);
                    failed:=false;
                    Exit;
                  end;
              end;

          Inc(ca);
        until ca>$fff0;

      end; (* Award BIOS *)

    if StrLComp(@biosf000[$e010],'SystemSoft',Length('SystemSoft'))=0 then
      begin
        pci_0e:=0;
        repeat
          if biosf000[pci_0e]=Ord(systemsoft_pci_0e[1]) then
            if compare(biosf000[pci_0e],systemsoft_pci_0e) then
              begin
                conmap:=0;
                len:=PWord(@biosf000[pci_0e+14+4])^;
                irqbuffR.length_of_IRQ_routing_table_buffer:=len;
                Move(biosf000[PWord(@biosf000[pci_0e+48+3])^],irqbuffR.IRQ_routing_table_entry_Array,len);
                failed:=false;
                Exit;
              end;

          Inc(pci_0e);

        until ca>$ff00;
      end; (* SystemSoft *)
  end;

{$EndIf OS2}

{$IfDef DPMI32}
function lookup_bios(deviceid,func,bus:byte;index_:word) : byte;assembler;
  {&Uses EBX,ECX,EDX,EDI}{&Frame-}
  asm
    mov ax,$b108
    mov bl,deviceid
    shl bl,3
    add bl,func
    mov bh,bus
    mov edi,index_
    int $1a
    jc @exit

    mov failed,false
  @exit:
    mov errcode,ah
    mov al,cl
  end;


procedure write_data_bios(deviceid,func,bus:byte;index_,data:longint;datasize:byte);assembler;
  {&Uses All}{&Frame-}
  asm
    mov eax,$b10b               (* byte *)
    cmp datasize,1
    je @size_correct
    mov al,$0c                  (* word *)
    cmp datasize,2
    je @size_correct
    mov al,$0d                  (* dword *)
  @size_correct:
    mov bl,deviceid
    shl bl,3
    add bl,func
    mov bh,bus
    mov edi,index_
    mov ecx,data
    int $1a
    mov errcode,ah
  end;


procedure pci_present_test;assembler;
  {&Uses All}{&Frame-}
  asm
    mov ax,$b101
    int $1a
    jc @exit

    cmp dx,$4350
    jne @exit

    mov PCIchar,al
    mov PCI_hibus,cl
    mov PCIverlo,bl
    mov PCIverhi,bh
    mov failed,false

  @exit:
  end;

procedure load_irqbuff;
  var
    irq16     :smallword;
    r         :real_mode_call_structure_typ;
  begin
    if GetDosMem(irq16,SizeOf(irqbuff))<>0 then Exit;
    FillChar(Mem[irq16 shl 4],SizeOf(irqbuff),0);

    MemW[irq16 shl 4+0]:=SizeOf(irqbuff)-6;
    MemW[irq16 shl 4+2]:=6;
    MemW[irq16 shl 4+4]:=irq16;

    with r do
      begin
        init_register(r);
        ax_:=$b10e;
        bx_:=$0000;
        ds_:=$f000;
        es_:=irq16;
        edi_:=0;

        intr_realmode(r,$1a);
        Move(Ptr(irq16 shl 4)^,irqbuff,SizeOf(irqbuff));
        len:=MemW[es_ shl 4+edi_];
        freedosmem(irq16);

        if ah_<>0 then Exit;

        conmap:=bx_;
        failed:=false;

      end;
  end;
{$EndIf DPMI32}

function lookup_hw(deviceid,func,bus:byte;index_:word) : byte;assembler;
  {&Uses ECX}{&Frame+}
  asm
    mov ah,$80
    mov al,bus
    shl eax,16
    mov al,byte ptr[index_]
    and al,$fc
    mov ah,deviceid
    shl ah,3
    add ah,func

    push eax
    push $0cf8
    call _Out32

    mov ecx,index_
    and ecx,3
    shl ecx,3 (* *8  *)

    push $0cfc
    call _In32
    shr eax,cl
    mov cl,al
    mov failed,false

    push 0
    push $0cf8
    call _Out32

    mov al,cl
  end;


procedure write_data_hw(deviceid,func,bus:byte;index_,data:longint;datasize:byte);assembler;
  {&Uses EAX,ECX}{&Frame+}
  asm
    mov ah,$80
    mov al,bus
    shl eax,16
    mov al,byte ptr[index_]
    and al,$fc
    mov ah,deviceid
    shl ah,3
    add ah,func

    push eax
    push $0cf8
    call _Out32

    mov ecx,index_
    and ecx,3
    shl ecx,3 (* *8  *)

    mov eax,data

    push eax
    push $0cfc
    cmp datasize,1
    jne @not_size_8
    call _Out8
    jmp @continue
  @not_size_8:
    cmp datasize,2
    jne @not_size_16
    call _Out16
    jmp @continue
  @not_size_16:
    call _Out32
  @continue:
    mov failed,false

    push 0
    push $0cf8
    call _Out32
  end;

{$Else} (* BP 7.0 *)

function lookup_bios(deviceid,func,bus:byte;index:word) : byte; assembler;
  asm
    mov ax,$b108
    mov bl,deviceid
    shl bl,3
    add bl,func
    mov bh,bus
    mov di,index
    int $1a
    jc @exit

    mov failed,false
    mov al,cl           (* result *)
  @exit:
    mov errcode,ah
  end;


function lookup_hw(deviceid,func,bus:byte;index:word) : byte; assembler;
  var
    inf:byte;

  asm
    mov ax,$8000
    mov al,bus
    db $66;shl ax,16

    mov ax,index
    and ax,00fch
    mov ah,deviceid
    shl ah,3
    add ah,func

    mov dx,0cf8h
    db $66;out dx,ax

    mov ax,index
    and ax,3
    shl ax,3
    mov cx,ax

    mov dx,0cfch
    db $66;in ax,dx
    db $66;shr ax,cl
    mov inf,al
    mov failed,false


    db $66;xor ax,ax
    mov dx,0cf8h
    db $66;out dx,ax

    mov al,inf
  end;


procedure write_data_bios(deviceid,func,bus:byte;index:word;data:longint;datasize:byte); assembler;
  asm
    mov ax,$b10d                (* byte *)
    cmp datasize,1
    je @size_correct
    mov al,$0c                  (* word *)
    cmp datasize,2
    je @size_correct
    mov al,$0d                  (* dword *)
  @size_correct:
    mov bl,deviceid
    shl bl,3
    add bl,func
    mov bh,bus
    mov di,index
    db $66; mov cx,word ptr data
    int $1a
    mov errcode,ah
  end;

procedure write_data_hw(deviceid,func,bus:byte;index:word;data:longint;datasize:byte); assembler;
  asm
    mov ax,$8000
    mov al,bus
    db $66;shl ax,16

    mov ax,index
    and ax,00fch
    mov ah,deviceid
    shl ah,3
    add ah,func

    mov dx,0cf8h
    db $66;out dx,ax

    mov dx,index
    and dx,3
    add dx,0cfch

    db $66;mov ax,word ptr data

    cmp datasize,1
    jne @not_size_8
    out dx,al
    jmp @continue
  @not_size_8:
    cmp datasize,2
    jne @not_size_16
    out dx,ax
    jmp @continue
  @not_size_16:
    db $66;out dx,ax
  @continue:
    mov failed,false


    db $66;xor ax,ax
    mov dx,0cf8h
    db $66;out dx,ax


  end;

procedure pci_present_test; assembler;
  asm
    mov ax,$b101
    int $1a
    jc @exit

    cmp dx,$4350
    jne @exit

    mov PCIchar,al
    mov PCI_hibus,cl
    mov PCIverlo,bl
    mov PCIverhi,bh
    mov failed,false

  @exit:
  end;

procedure load_irqbuff;assembler;
  const
    irq_buf_size=SizeOf(irqbuffR.IRQ_routing_table_entry_Array);
  asm
    push ds

    mov bx,0
    mov ax,seg irqbuff
    mov es,ax
    mov di,offset irqbuff
    mov word ptr es:[di+0],irq_buf_size
    lea ax,[di+6]
    mov es:[di+2],ax
    mov es:[di+4],es

    mov ax,0f000h
    mov ds,ax
    mov ax,0b10eh

    int $1a
    pop ds

    mov cx,word ptr es:[di]

    cmp ah,0
    jne @exit


    mov conmap,bx
    mov len,cx
    mov failed,false

  @exit:
  end;
{$EndIf}


function lookup(deviceid,func,bus:byte;index_:word) : byte;
  begin
    if usebios then
      lookup:=lookup_bios(deviceid,func,bus,index_)
    else
      lookup:=lookup_hw  (deviceid,func,bus,index_);
  end;

procedure write_dword(deviceid,func,bus:byte;index_,data:longint);
  begin
    if usebios then
      write_data_bios(deviceid,func,bus,index_,data,SizeOf(longint))
    else
      write_data_hw  (deviceid,func,bus,index_,data,SizeOf(longint));
  end;

procedure write_byte(deviceid,func,bus:byte;index_,data:byte);
  begin
    if usebios then
      write_data_bios(deviceid,func,bus,index_,data,SizeOf(byte))
    else
      write_data_hw  (deviceid,func,bus,index_,data,SizeOf(byte));
  end;

procedure write_word(deviceid,func,bus:byte;index_,data:word);
  begin
    if usebios then
      write_data_bios(deviceid,func,bus,index_,data,SizeOf(word))
    else
      write_data_hw  (deviceid,func,bus,index_,data,SizeOf(word));
  end;

end.

