/* $Id: kSFNContainer.cpp,v 1.1 2000/04/29 19:06:35 stknut Exp $
 *
 * kSFNContainer (kTaskMgr) - generic SystemFileNumber (SFN) container.
 *
 * Copyright (c) 2000 knut st. osmundsen (knut.stange.osmundsen@pmsc.no)
 *
 */

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#define INCL_WIN
#define INCL_GPI
#define INCL_BASE


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <os2.h>
#ifdef USE_KLIB
    #include <kAssert.h>
    #include <kLog.h>
    #include <kHeap.h>
#else
    #include <malloc.h>
#endif
#include <memory.h>
#include <string.h>
#include <stddef.h>
#include <stdio.h>

#include "kBase.h"
#include "kError.h"
#include "kDlgBase.h"
#include "kMenuBase.h"
#include "kClickDlg.h"
#include "kContainer.h"

#include "kQuerySysState.h"
#include "kNotebookBase.h"
#include "kNotebookPageBase.h"
#include "kSFNRecord.h"
#include "kSFNContainer.h"
#include "kTaskMgr.h"
#include "kTaskMgr_defs.h"


/**
 * Updates the content of the container - inserts SFN a single process.
 * @returns   success indicator.
 * @author    knut st. osmundsen (knut.stange.osmundsen@pmsc.no)
 */
BOOL   kSFNContainer::insertSFNsPid()
{
    int             c,                  /* Count of SFNs in this FS record. */
                    i;                  /* Loop variable */
    kSFNRecord    * pCurCnrRec,         /* Current container record. */
                  * pCnrRec;            /* Pointer to first container record. */
    PPROCESSDATA    pProcData;          /* Pointer to process data. */

    ASSERT(usPid < 0xFFFF);

    /*
     * Remove all existing container records!
     */
    removeAllRecords();

    /*
     * Get process data
     */
    pProcData = QSGetProcessData(usPid);
    if (pProcData == NULL)
        return FALSE;


    /*
     * Allocate container records for all the objects.
     *
     * FEATURE/BUGBUG: cFH isn't correct - it don't match with the number of
     *      entries in the pFSRec array. (qsProc_t)
     *
     *      This is verified! There is a "bug" in the _qsProcFS worker (this
     *      is the function which generates the SFT table pointed to by pFSRec).
     *      cFH is set to JFN_Length by _qsReadPtda. While _qsProcFS skips the
     *      entries containing 0xFFFF (unused handle placeholder).
     *      This way the cFH is really the maximum number of handles and the
     *      size of the pFSRec table has to be determined by checking pointers
     *      to the next structure. Thread records are by specifications
     *      following the FS records, pointer to these is used to do this.
     *
     *      Note. JFN is not equal to the index into the SFN table due to this bug.
     */
    if ((char*)&pProcData->pProcRec->pFSRec[pProcData->pProcRec->cFH] > (char*)pProcData->pProcRec->pThrdRec)
        c = (int)((char*)pProcData->pProcRec->pThrdRec - (char*)pProcData->pProcRec->pFSRec) / sizeof(pProcData->pProcRec->pFSRec[0]);
    else
        c = (int)pProcData->pProcRec->cFH;
    pCurCnrRec = pCnrRec = (kSFNRecord*)allocMiniRec(sizeof(kSFNRecord), c);
    if (pCurCnrRec == NULL)
        return FALSE;               /* FIXME: complain about this! */

    /*
     * Loop thru all SFNs for this MFT (Main(?)FileTable).
     */
    for (i = 0; i < c; i++, pCurCnrRec = (kSFNRecord*)pCurCnrRec->getNext())
    {
        qsSft_t         SFTDummy;       /* Dummy SFT record. (used when FS data isn't found) */
        struct _FSDummy
        {
            qsFrec_t    rec;
            char        name[64];
        }           FSDummy;            /* Dummy FS record with filename. (used when FS data isn't found) */
        qsFrec_t *      pFSData;        /* Pointer to FS data record. */
        qsSft_t *       pSFTData;       /* Pointer to SFT data record. */


        /*
         * Get FS and SFT data.
         */
        pFSData = QSGetSfnData(pProcData->pProcRec->pFSRec[i]);
        if (pFSData != NULL)
        {
            int j;
            /* find the correct SFT record in this FS record */
            for (j = 0, pSFTData = pFSData->pSft; j < pFSData->ctSft; j++)
                if (pSFTData->sfn == pProcData->pProcRec->pFSRec[i])
                    break;
            ASSERT(pSFTData != NULL);
        }
        else
        {   /* create dummy FS and SFT records */
            memset(&SFTDummy, 0, sizeof(SFTDummy));
            SFTDummy.sfn = pProcData->pProcRec->pFSRec[i];
            FSDummy.rec.RecType = QS_FILESYS;
            FSDummy.rec.pNextRec = NULL;
            FSDummy.rec.ctSft = 1;
            FSDummy.rec.pSft = &SFTDummy;
            if (SFTDummy.sfn == 0xffff)
                strcpy(FSDummy.name, "<unused>");
            else
                strcpy(FSDummy.name, "<Not found due to BUGBUG in DosQuerySysState / _qsReadMFT ???>");
            pFSData = &FSDummy.rec;
            pSFTData = &SFTDummy;
        }


        /*
         * Init and set the record.
         */
        pCurCnrRec->init();
        pCurCnrRec->set(pSFTData, (char*)pFSData + sizeof(*pFSData), (USHORT)i);
    }


    /*
     * Insert the record at bottom of the container.
     */
    if (!insertAtTop(pCnrRec, c))
        return FALSE;

    /* resort container */
    return enableSorting();
}


/**
 * Updates the content of the container - inserts SFN for the entire system
 * @returns   success indicator.
 * @author    knut st. osmundsen (knut.stange.osmundsen@pmsc.no)
 */
BOOL   kSFNContainer::insertSFNsAll()
{
    qsFrec_t *      pFSData;            /* Pointer to current FS data record. */
    int             cRecords;           /* Count of records to allocate / allocated. */
    kSFNRecord    * pCurCnrRec,
                  * pCnrRec;

    /*
     * Remove all existing container records!
     */
    removeAllRecords();

    /*
     * Get data for this MTE and get the count of files.
     */
    cRecords = 0;
    pFSData = QSGetFSFirstData();
    while (pFSData != NULL && pFSData->RecType == QS_FILESYS)
    {
        /*
         * BUGBUG! - ctSft isn't correct, or rather Sft records are missing!!! (qsFrec_t)
         *      The number of qsSft_t records is always 1, no matter what the
         *      ctSft member says!
         *      This is verified! The defect is located in _qsReadMFT. When
         *      writing the qsSft_t records the record pointer is always
         *      initialized to _pCurCache in each iteration. This is probably
         *      correct, but _pCurCache is _NOT_ changed during the iteration.
         *      With the excelent result that the next Sft record overwrites
         *      the previous one.
         *
         *      Solution: To move the call to _qsUpdatePtrs into the loopbody,
         *                after the Sft record is written.
         *                This way _pCurCache will be updated at end of each
         *                iteration.
         *      This is a problem on Aurora, Warp 4 and Warp 3.
         */
        if (pFSData->ctSft > 1 && &pFSData->pSft[pFSData->ctSft] > pFSData->pNextRec && pFSData->pNextRec != NULL)
        {
            fprintf(stderr, "- start - pFSData->ctSft correction - end -\nold pFSData->ctSft = %d\n", pFSData->ctSft);
            pFSData->ctSft = (((char*)pFSData->pNextRec - (char*)pFSData->pSft)) / SIZEOFQSSFT_T;
            fprintf(stderr, "new pFSData->ctSft = %d\n- end   - pFSData->ctSft correction - end -\n", pFSData->ctSft);
        }

        /* Add; Next record */
        cRecords += pFSData->ctSft;
        pFSData = (qsFrec_t*)pFSData->pNextRec;
    }


    /*
     * Allocate container records for all the objects.
     */
    pCurCnrRec = pCnrRec = (kSFNRecord*)allocMiniRec(sizeof(kSFNRecord), cRecords);
    if (pCurCnrRec == NULL)
        return FALSE;               /* FIXME: complain about this! */


    /*
     * Loop thru all filesystem records insering all system filenumbers.
     */
    pFSData = QSGetFSFirstData();
    while (pFSData != NULL && pFSData->RecType == QS_FILESYS)
    {
        int         i;                  /* Inner SFN loop variable */
        qsSft_t *   pSftRec;            /* Pointer to current Sft data */


        /*
         * Loop thru all SFNs for this MFT (Main(?)FileTable).
         */
        for (i = 0, pSftRec = pFSData->pSft;
             i < pFSData->ctSft;
             i++, pCurCnrRec = (kSFNRecord*)pCurCnrRec->getNext(), pSftRec = (qsSft_t*)((char*)pSftRec + SIZEOFQSSFT_T)
             )
        {
            /*
             * Init and set the record.
             */
            pCurCnrRec->init();
            pCurCnrRec->set(pSftRec, (char*)pFSData + sizeof(*pFSData));
        }

        /*
         * Next filesystem record.
         */
        pFSData = (qsFrec_t*)pFSData->pNextRec;
    }

    /*
     * Insert the records.
     */
    if (!insertAtTop(pCnrRec, cRecords))
        return FALSE;

    /* resort container */
    return enableSorting();
}


/**
 * Menu is closing. Remove emphasis.
 * @param     usMenuId  Menu id.
 * @param     hwndMnu   Handle to menu window.
 */
VOID kSFNContainer::menuEnd(USHORT usMenuId, HWND hwndMnu)
{
    setRecordEmphasis(pCurRecord, FALSE, CRA_SOURCE);
    hwndMnu = hwndMnu;
    usMenuId = usMenuId;
}


/**
 * Command events - none yet.
 * @param     usCmd     Control id which send/posted the message.
 * @param     usSource  Source id.
 * @param     fPointer  Mouse pointer flag.
 */
VOID  kSFNContainer::command(USHORT usCmd, USHORT usSource, BOOL fPointer)
{
    switch (usCmd)
    {
        /*
         * Refresh menu item were selected - update the container!
         */
        case IDM_CNR_SFN_ALL_REFRESH:
            update();
            break;


        /*
         * One of the sort menu items were selected.
         *      If it's an uncheck request Then
         *          Set back to default sorting. (if allready default sorting - do nothing - break)
         *      Endif.
         *      Save new sort id.
         *      Update menuitems.
         *      (re)Enable sorting.
         */
        case IDM_CNR_SFN_ALL_SORT_JFN:
        case IDM_CNR_SFN_ALL_SORT_SFN:
        case IDM_CNR_SFN_ALL_SORT_REFCNT:
        case IDM_CNR_SFN_ALL_SORT_FLAGS:
        case IDM_CNR_SFN_ALL_SORT_FLAGS2:
        case IDM_CNR_SFN_ALL_SORT_MODE:
        case IDM_CNR_SFN_ALL_SORT_MODE2:
        case IDM_CNR_SFN_ALL_SORT_SIZE:
        case IDM_CNR_SFN_ALL_SORT_HVPB:
        case IDM_CNR_SFN_ALL_SORT_ATTR:
        case IDM_CNR_SFN_ALL_SORT_NAME:
            if (usCmd == usSortId)
            {
                #if 0 //JFN is disabled due to Bug/Feature in _qsProcFS.
                if (usCmd == IDM_CNR_SFN_ALL_SORT_SFN && usPid >= 0xFFFF
                    || usCmd == IDM_CNR_SFN_ALL_SORT_JFN && usPid < 0xFFFF
                    )
                    break;
                usCmd = (USHORT)(usPid < 0xFFFF ? IDM_CNR_SFN_ALL_SORT_JFN : IDM_CNR_SFN_ALL_SORT_SFN);
                #else
                if (usCmd == IDM_CNR_SFN_ALL_SORT_SFN)
                    break;
                usCmd = IDM_CNR_SFN_ALL_SORT_SFN;
                #endif
            }
            usSortId = usCmd;

            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_JFN   , usCmd == IDM_CNR_SFN_ALL_SORT_JFN   );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_SFN   , usCmd == IDM_CNR_SFN_ALL_SORT_SFN   );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_REFCNT, usCmd == IDM_CNR_SFN_ALL_SORT_REFCNT);
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_FLAGS , usCmd == IDM_CNR_SFN_ALL_SORT_FLAGS );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_FLAGS2, usCmd == IDM_CNR_SFN_ALL_SORT_FLAGS2);
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_MODE  , usCmd == IDM_CNR_SFN_ALL_SORT_MODE  );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_MODE2 , usCmd == IDM_CNR_SFN_ALL_SORT_MODE2 );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_SIZE  , usCmd == IDM_CNR_SFN_ALL_SORT_SIZE  );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_HVPB  , usCmd == IDM_CNR_SFN_ALL_SORT_HVPB  );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_ATTR  , usCmd == IDM_CNR_SFN_ALL_SORT_ATTR  );
            pMenuCnrAll->checkMenuItem(IDM_CNR_SFN_ALL_SORT_NAME  , usCmd == IDM_CNR_SFN_ALL_SORT_NAME  );

            /* resort container */
            enableSorting();
            break;
    }
    usSource = usSource;
    fPointer = fPointer;
}


/**
 * Record sort callback function - compares two records.
 * @returns   >  0  when pRecord1  >  pRecord2
 *            <  0  when pRecord1  <  pRecord2
 *            == 0  when pRecord1 ==  pRecord2
 * @param     pRecord1  Pointer to first record.
 * @param     pRecord2  Pointer to second record.
 */
SHORT  kSFNContainer::sortCallBack(kCnrMiniRecord *pRecord1, kCnrMiniRecord *pRecord2)
{
    kSFNRecord *    pRec1 = (kSFNRecord *)pRecord1;
    kSFNRecord *    pRec2 = (kSFNRecord *)pRecord2;

    if (pRec1 != NULL && pRec2 != NULL)
    {
        int iTmp = 1;
        switch (usSortId)
        {
            case IDM_CNR_SFN_ALL_SORT_JFN:
                iTmp = pRec1->getJFN() - pRec2->getJFN();
                break;

            case IDM_CNR_SFN_ALL_SORT_SFN:
                iTmp = pRec1->getSFN() - pRec2->getSFN();
                break;

            case IDM_CNR_SFN_ALL_SORT_REFCNT:
                iTmp = strcmp(pRec1->getRefCnt(), pRec2->getRefCnt());
                break;

            case IDM_CNR_SFN_ALL_SORT_FLAGS:
                iTmp = strcmp(pRec1->getFlags(), pRec2->getFlags());
                break;

            case IDM_CNR_SFN_ALL_SORT_FLAGS2:
                iTmp = strcmp(pRec1->getFlags2(), pRec2->getFlags2());
                break;

            case IDM_CNR_SFN_ALL_SORT_MODE:
                iTmp = strcmp(pRec1->getMode(), pRec2->getMode());
                break;

            case IDM_CNR_SFN_ALL_SORT_MODE2:
                iTmp = strcmp(pRec1->getMode2(), pRec2->getMode2());
                break;

            case IDM_CNR_SFN_ALL_SORT_SIZE:
                iTmp = strcmp(pRec1->getSize(), pRec2->getSize());
                break;

            case IDM_CNR_SFN_ALL_SORT_HVPB:
                iTmp = strcmp(pRec1->gethVPB(), pRec2->gethVPB());
                break;

            case IDM_CNR_SFN_ALL_SORT_ATTR:
                iTmp = strcmp(pRec1->getAttr(), pRec2->getAttr());
                break;

            case IDM_CNR_SFN_ALL_SORT_NAME:
                iTmp = strcmp(pRec1->getName(), pRec2->getName());
                break;
        }
        return (SHORT)((iTmp > 0) ? 1 : (iTmp < 0) ? -1 : 0);
    }
    return 1;
}





/**
 * Constructor.
 * @param     hwndDlg   Handle to dialog window.
 * @param     ulCnrId   ID of the container dialog item in hwndDlg.
 * @param     usPid      = 0xFFFF: View all SFNs.
 *                      != 0:      View only SFNs for the given process.
 * @author    knut st. osmundsen (knut.stange.osmundsen@pmsc.no)
 */
kSFNContainer::kSFNContainer(HWND hwndDlg, ULONG ulCnrId, USHORT usPid /* = 0xFFFF*/) throw(kError)
    : kDetailCnr(WinWindowFromID(hwndDlg, ulCnrId),
                 0,
                 "System File Numbers (SFN) Overview",
#if 0 //JFN is disabled due to Bug/Feature in _qsProcFS.
                 kSFNRecord::cFieldInfo - (usPid < 0xFFFF ? 0 : 1),
                 (PFIELDINFO)&kSFNRecord::aFieldInfo[usPid < 0xFFFF ? 0 : 1]),
#else
                 kSFNRecord::cFieldInfo - 1,
                 (PFIELDINFO)&kSFNRecord::aFieldInfo[1]),
#endif
    usPid(usPid), pCurRecord(NULL), pMenuCnrAll(NULL), usSortId(0xFFFF)
{
    /*
     * Create menu, set default sorting and enable sorting.
     */
    pMenuCnrAll = new kMenuBase(IDM_CNR_SFN_ALL, NULLHANDLE, hwndCnr);
    #if 0 //JFN is disabled due to Bug/Feature in _qsProcFS.
    if (usPid < 0xFFFF)
        usSortId = IDM_CNR_SFN_ALL_SORT_JFN;
    else
    #endif
    {
        pMenuCnrAll->enableMenuItem(IDM_CNR_SFN_ALL_SORT_JFN, FALSE);
        usSortId = IDM_CNR_SFN_ALL_SORT_SFN;
    }
    enableSorting();
    pMenuCnrAll->checkMenuItem(usSortId, TRUE);
}


/**
 * Destructor.
 */
kSFNContainer::~kSFNContainer()
{
}


/**
 * Displays the popup menu for the container.
 * @param     usId     Container id.
 * @param     pRecord  Pointer to the record which is selected by either the key
 */
VOID kSFNContainer::cnrContextMenu(USHORT usId, PRECORDCORE pRecord)
{
    if (pMenuCnrAll)
    {
        pCurRecord = (kSFNRecord*)pRecord;
        setRecordEmphasis(pCurRecord, TRUE, CRA_SOURCE);
        if (pRecord == NULL)
            pMenuCnrAll->popup();
        /*
        else
            pMenuModule->popup();
        */
    }
    usId = usId;
}


/**
 * Enter or double click on record in the container.
 * This action will bring up the detail dialog for the record.
 */
VOID kSFNContainer::cnrEnter(USHORT usId, HWND hwndCnr, PRECORDCORE pRecord, ULONG fKey)
{
    if (pRecord != NULL)
    {
        pCurRecord = (kSFNRecord*)pRecord;
        /* command(IDM_CNR_MOD_DETAILS, 0, 0); */
    }
    usId = usId;
    fKey = fKey;
    hwndCnr = hwndCnr;
}



/**
 * Updates the contents of the container.
 * @author    knut st. osmundsen (knut.stange.osmundsen@pmsc.no)
 */
VOID  kSFNContainer::update()
{
    if (usPid < 0xFFFF)
        insertSFNsPid();
    else
        insertSFNsAll();
}


