
/*
	Hugh Fisher
	Dec 91
	
	Operations for GMT grids. The I/O code
	has a remarkable similarity to that
	in gmt_grdio.c, but uses different
	parameters and error handling.
					*/

#include <stdio.h>
#include <errno.h>
#include <X11/Intrinsic.h>

#include "xgrid_utility.h"
#include "xgrid_GMTgrid.h"

static void clear (grid)
	GMTGrid * grid;
{
  grid->methods.width     = 0;
  grid->methods.height    = 0;
  grid->methods.coordType = GridCoordNodes;
  
  grid->value = NULL;
  
  grid->header.x_min = 0.0;
  grid->header.x_max = 0.0;
  grid->header.y_min = 0.0;
  grid->header.y_max = 0.0;
  grid->header.z_min = 0.0;
  grid->header.z_max = 0.0;
  grid->header.x_inc = 0.0;
  grid->header.y_inc = 0.0;
  grid->header.z_scale_factor	= 1.0;
  grid->header.z_add_offset	= 0.0;
  grid->header.nx = 0;
  grid->header.ny = 0;
  grid->header.node_offset = GridCoordNodes;

  strcpy(grid->header.x_units, "X units");
  strcpy(grid->header.y_units, "Y units");
  strcpy(grid->header.z_units, "Z units");
  strcpy(grid->header.title, "Untitled");
  memset(grid->header.command, 0, sizeof(grid->header.command));
  memset(grid->header.remark, 0, sizeof(grid->header.remark));
}

static void readFromFile (grid, fileName, status)
	GMTGrid * grid;
	String    fileName;
	int *     status;
{
  int	 input;
  int	 xRange, yRange, zRange,
  	 spacing, dimension, values;
  int	 start[1], edge[1], size[2];
  double bounds[2];
  char	 text[512];
  
  *status = 0;
  
  /* File OK? */
  input = ncopen(fileName, NC_NOWRITE);
  if (input == -1) {
    *status = errno;
    return;
  }
  /* Read the header */
  xRange    = ncvarid(input, "x_range");
  yRange    = ncvarid(input, "y_range");
  zRange    = ncvarid(input, "z_range");
  spacing   = ncvarid(input, "spacing");
  dimension = ncvarid(input, "dimension");
  values    = ncvarid(input, "z");

  /* Grid units */
  ncattget(input, xRange, "units", grid->header.x_units);
  ncattget(input, yRange, "units", grid->header.y_units);
  ncattget(input, zRange, "units", grid->header.z_units);
  ncattget(input, values, "scale_factor", &grid->header.z_scale_factor);
  ncattget(input, values, "add_offset", &grid->header.z_add_offset);
  ncattget(input, values, "node_offset", &grid->header.node_offset);
  
  /* Full title and generating command */
  ncattget(input, NC_GLOBAL, "title", grid->header.title);
  ncattget(input, NC_GLOBAL, "source", text);
  strncpy(grid->header.command, text, sizeof(grid->header.command));
  strncpy(grid->header.remark, &(text[sizeof(grid->header.command)]),
  				sizeof(grid->header.remark));
  
  /* Grid coordinate system */
  start[0] = 0;
  edge[0]  = 2;
  ncvarget(input, xRange, start, edge, bounds);
  grid->header.x_min = bounds[0];
  grid->header.x_max = bounds[1];
  ncvarget(input, yRange, start, edge, bounds);
  grid->header.y_min = bounds[0];
  grid->header.y_max = bounds[1];
  ncvarget(input, spacing, start, edge, bounds);
  grid->header.x_inc = bounds[0];
  grid->header.y_inc = bounds[1];
  ncvarget(input, zRange, start, edge, bounds);
  grid->header.z_min = bounds[0];
  grid->header.z_max = bounds[1];
  
  /* Actual data */
  ncvarget(input, dimension, start, edge, size);
  grid->header.nx = size[0];
  grid->header.ny = size[1];

  grid->value = (GridValue *) XtMalloc (sizeof(GridValue)
  				* grid->header.nx * grid->header.ny);
  start[0] = 0;
  edge[0]  = grid->header.nx * grid->header.ny;
  ncvarget(input, values, start, edge, grid->value);

  /* Update generic grid values */    
  grid->methods.width  = grid->header.nx;
  grid->methods.height = grid->header.ny;
  grid->methods.coordType = grid->header.node_offset;
    
  /* Done */
  ncclose(input);
}

static void writeToFile (grid, fileName, status)
	GMTGrid * grid;
	String    fileName;
	int *	  status;
{
  int	 output;
  int	 xRange, yRange, zRange,
  	 spacing, dimension, values,
	 sides, xysize;
  int    start[1], edge[1], dimensions[1], size[2];
  double bounds[2];
  char   text[480];
  int	 loop;

  *status = 0;
  
  /* Create file, which enters define mode as well */  	
  output = nccreate(fileName, NC_CLOBBER);
  if (output == -1) {
    *status = errno;
    return;
  }

  /* Define a two dimensional array */  
  sides  = ncdimdef(output, "side", 2);
  xysize = ncdimdef(output, "xysize", grid->header.nx * grid->header.ny);
  dimensions[0] = sides;

  /* Define variables in grid file */  
  xRange    = ncvardef(output, "x_range", NC_DOUBLE, 1, dimensions);
  yRange    = ncvardef(output, "y_range", NC_DOUBLE, 1, dimensions);
  zRange    = ncvardef(output, "z_range", NC_DOUBLE, 1, dimensions);
  spacing   = ncvardef(output, "spacing", NC_DOUBLE, 1, dimensions);
  dimension = ncvardef(output, "dimension", NC_LONG, 1, dimensions);

  dimensions[0] = xysize;
  values = ncvardef(output, "z", NC_FLOAT, 1, dimensions);
  
  /* Define and set grid attributes */
  strcpy(text, grid->header.command);
  ncattput(output, NC_GLOBAL, "title", NC_CHAR,
  	sizeof(grid->header.title), grid->header.title);
  ncattput(output, values, "long_name", NC_CHAR,
  	sizeof(grid->header.title), grid->header.title);
	
  strcpy(&(text[sizeof(grid->header.command)]), grid->header.remark);
  ncattput(output, NC_GLOBAL, "source", NC_CHAR, sizeof(text), text);
  
  ncattput(output, xRange, "units", NC_CHAR,
  	sizeof(grid->header.x_units), grid->header.x_units);
  ncattput(output, yRange, "units", NC_CHAR,
  	sizeof(grid->header.y_units), grid->header.y_units);
  ncattput(output, zRange, "units", NC_CHAR,
  	sizeof(grid->header.z_units), grid->header.z_units);
  ncattput(output, values, "scale_factor", NC_DOUBLE,
  	1, &grid->header.z_scale_factor);
  ncattput(output, values, "add_offset", NC_DOUBLE,
  	1, &grid->header.z_add_offset);
  ncattput(output, values, "node_offset", NC_LONG,
  	1, &grid->header.node_offset);
  
  /* Finished defining things, now do the header */
  ncendef(output);
  
  start[0] = 0;
  edge[0]  = 2;
  
  bounds[0] = grid->header.x_min;
  bounds[1] = grid->header.x_max;
  ncvarput(output, xRange, start, edge, bounds);
  bounds[0] = grid->header.y_min;
  bounds[1] = grid->header.y_max;
  ncvarput(output, yRange, start, edge, bounds);
  bounds[0] = grid->header.x_inc;
  bounds[1] = grid->header.y_inc;
  ncvarput(output, spacing, start, edge, bounds);
  size[0] = grid->header.nx;
  size[1] = grid->header.ny;
  ncvarput(output, dimension, start, edge, size);
  
  /* Recalc min and max data values so that client
     program doesn't have to  */
  for (loop = 0; loop < grid->header.nx * grid->header.ny; loop ++) {

/* Changed to use gmt.h's "GMT_is_fnan" macro (now in GMTgrid.h) vs. isnan */
/* to be more generic, specifically for SGIs.  J. Lillibridge  8/4/93    */

    if (GMT_is_fnan(grid->value[loop]))
      continue;
    grid->header.z_min = Min(grid->header.z_min, grid->value[loop]);
    grid->header.z_max = Max(grid->header.z_max, grid->value[loop]);
  }
  bounds[0] = grid->header.z_min;
  bounds[1] = grid->header.z_max;
  ncvarput(output, zRange, start, edge, bounds);

  /* OK, now do the actual values */
  edge[0] = grid->header.nx * grid->header.ny;
  ncvarput(output, values, start, edge, grid->value);
  
  /* Done */
  ncclose (output);
}

static void dispose (grid)
	GMTGrid * grid;
{
  XtFree(grid->value);
  grid->value = NULL;
}

static void dumpGrid (grid)
	GMTGrid * grid;
{
  int outer, inner;
  GridPoint  gPt;
  XPoint     xPt;
  
  printf("Grid: %s\n", grid->header.title);
  printf("Width, height = %d x %d\n", grid->header.nx, grid->header.ny);
  printf("X Coordinates: %f to %f by %f\n",
  	grid->header.x_min, grid->header.x_max, grid->header.x_inc);
  printf("Y Coordinates: %f to %f by %f\n",
  	grid->header.y_min, grid->header.y_max, grid->header.y_inc);
  if (grid->header.node_offset == GridCoordNodes)
    printf("Data points centred on coordinates\n");
  else if (grid->header.node_offset == GridCoordPixels)
    printf("Data points between coordinates\n");  
  printf("Z range = %f to %f, scale by %f and add %f\n",
  	grid->header.z_min, grid->header.z_max,
	grid->header.z_scale_factor, grid->header.z_add_offset);
  /* Write out the values in same format as grd2xyz uses */	
  for (outer = 0; outer < grid->header.ny; outer ++) {
    for (inner = 0; inner < grid->header.nx; inner ++) {
      xPt.x = inner; xPt.y = outer;
      GetGridCoords(grid, &xPt, &gPt);
      printf("%lg\t%lg\t%lg\n", gPt.x, gPt.y, GetGrid(grid, inner, outer));
    }
    printf("\n");
  }
}

static void getIndexes (grid, coord, index)
	GMTGrid *   grid;
	GridPoint * coord;
	XPoint *    index;
{
  int column, row;
  
  column = (int)((coord->x - grid->header.x_min) / grid->header.x_inc);
  row    = (int)((coord->y - grid->header.y_min) / grid->header.y_inc);
  index->x = column;
  /* Allow for origin being at the bottom rather than the top */
  index->y = grid->header.ny - row;
}

static void getCoords (grid, index, coord)
	GMTGrid *   grid;
	XPoint *    index;
	GridPoint * coord;
{
  coord->x = grid->header.x_min + index->x * grid->header.x_inc;
  coord->y = grid->header.y_max - index->y * grid->header.y_inc;
}

static void set (grid, xIndex, yIndex, value)
	GMTGrid * grid;
	int	  xIndex;
	int	  yIndex;
	GridValue value;
{
  grid->value[yIndex * grid->header.nx + xIndex] = value;
}

static GridValue get (grid, xIndex, yIndex)
	GMTGrid * grid;
	int	  xIndex;
	int	  yIndex;
{

/* Original Version had "grid->header.ny": should be "grid->header.nx */
/* Fixed 8/4/93 by J. Lillibrige @ NOAA/NOS to eliminate core dumps!  */

  return grid->value[yIndex * grid->header.nx + xIndex];
}

GMTGrid * CreateGMTGrid ()
{
  GMTGrid * result;
  
  result = (GMTGrid *)XtMalloc(sizeof(GMTGrid));
  clear(result);
  /* Assign methods */
  result->methods.dispose	= dispose;
  result->methods.readFromFile	= readFromFile;
  result->methods.writeToFile	= writeToFile;
  result->methods.getIndexes	= getIndexes;
  result->methods.getCoords	= getCoords;
  result->methods.set		= set;
  result->methods.get		= get;
  
  return result;
}
