/*--------------------------------------------------------------------
 *    @(#)libspotter.c	1.4  06/24/99
 *
 *   Copyright (c) 1999 by P. Wessel
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; version 2 of the License.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   Contact info: www.soest.hawaii.edu/wessel
 *--------------------------------------------------------------------*/
/*
 * SPOTTER: functions for moving points along small circles on a sphere.
 *
 * Paul Wessel, University of Hawaii
 * June 09, 1999
 * Version 1.0
 *
 * The user-callable functions in this library are:
 *
 * spotter_init			: Load stage poles from file
 * spotter_backtrack		: Trace track from seamount to hotspot
 * spotter_forthtrack		: Trace track from hotspot to seamount
 *
 * programs must first call spotter_init() which reads a file of
 * backward stage poles.  Given the right flag it can convert these
 * to forward stage poles.
 *
 * Then to draw a hotspot track the program can:
 *	1. Draw FROM the hotspot TO a seamount: Use spotter_forthtrack
 *	2. Draw FROM a seamount BACK TO a hotspot: Use spotter_backtrack
 *
 * To draw crustal flowlines (seamounts motion over mantle) do select
 * flowline = TRUE when calling spotter_init and then:
 *	1. Draw FROM a hotspot TO a seamount: Use spotter_backtrack
 *	2. Draw FROM a seamount TO a hotspot (and beyond): Use spotter_forthtrack
 */

#include "spotter.h"

/* Internal functions */

void spotter_forward_poles (struct EULER p[], int ns);
void spotter_rotate_fwd (double lon, double lat, double *tlon, double *tlat, struct EULER *p);
void spotter_rotate_inv (double *lon, double *lat, double tlon, double tlat, struct EULER *p);

int spotter_init (char *file, struct EULER **p, int flowline, double t_max)
{
	/* file;	Name of file with backward stage poles */
	/* p;		Pointer to stage pole array */
	/* flowline;	TRUE if flowlines rather than hotspot-tracks are needed */
	/* t_max;	Extend earliest stage pole back to this age */
	FILE *fp;
	struct EULER *e;
	char  buffer[BUFSIZ];
	int n, i = 0, n_alloc = GMT_SMALL_CHUNK;
	double x, y, last_t = DBL_MAX;

	e = (struct EULER *) GMT_memory (VNULL, n_alloc, sizeof (struct EULER), "libspotter");

	if ((fp = fopen (file, "r")) == NULL) {
		fprintf (stderr, "EULER: Cannot open stage pole file: %s\n", file);
		exit (EXIT_FAILURE);
	}

	while (fgets (buffer, 512, fp) != NULL) { /* Expects lon lat t0 t1 ccw-angle */
		if (buffer[0] == '#' || buffer[0] == '\n') continue;

		sscanf (buffer, "%lf %lf %lf %lf %lf", &e[i].lon, &e[i].lat, &e[i].t_start, &e[i].t_stop, &e[i].omega);

		if (e[i].t_stop >= e[i].t_start) {
			fprintf (stderr, "libspotter: ERROR: Stage rotation %d has start time younger than stop time\n", i);
			exit (EXIT_FAILURE);
		}
		if (e[i].t_stop > last_t) {
			fprintf (stderr, "libspotter: ERROR: Stage rotations must go from oldest to youngest\n");
			exit (EXIT_FAILURE);
		}
		last_t = e[i].t_stop;

		e[i].duration = e[i].t_start - e[i].t_stop;
		e[i].omega /= e[i].duration;
		e[i].omega_r = e[i].omega * D2R;
		e[i].sin_lat = sin (e[i].lat * D2R);
		e[i].cos_lat = cos (e[i].lat * D2R);
		x = e[i].lon * D2R;	y = e[i].lat * D2R;
		e[i].lon_r = x;		e[i].lat_r = y;
		i++;
		if (i == n_alloc) {
			n_alloc += GMT_SMALL_CHUNK;
			e = (struct EULER *) GMT_memory ((void *)e, n_alloc, sizeof (struct EULER), "libspotter");
		}
	}
	n = i;

	/* Extend oldest stage pole back to t_max Ma */

	if (t_max > 0.0 && e[0].t_start < t_max) {
		fprintf (stderr, "Extending oldest stage pole back to %lg Ma\n", t_max);

		e[0].t_start = t_max;
		e[0].duration = e[0].t_start - e[0].t_stop;
	}

	e = (struct EULER *) GMT_memory ((void *)e, n, sizeof (struct EULER), "libspotter");
	if (flowline) spotter_forward_poles (e, n);
	*p = e;

	return (n);
}

void spotter_forward_poles (struct EULER p[], int ns)
{
	int i, j;
	double d_lon, tlon, tlat;

	for (i = ns-1; i > 0; i--) {	/* From current to previous stages */
		for (j = 0; j < i; j++) {	/* Rotate the older stage poles */
			d_lon = p[i].omega_r * p[i].duration;
			spotter_rotate_fwd (p[j].lon_r, p[j].lat_r, &tlon, &tlat, &p[i]);
			tlon += d_lon;
			spotter_rotate_inv (&p[j].lon_r, &p[j].lat_r, tlon, tlat, &p[i]);
		}
	}
	for (i = 0; i < ns; i++) {
		p[i].lon = p[i].lon_r * R2D;
		p[i].lat = p[i].lat_r * R2D;
		p[i].sin_lat = sin (p[i].lat_r);
		p[i].cos_lat = cos (p[i].lat_r);
		p[i].omega = -p[i].omega;
		p[i].omega_r = -p[i].omega_r;
	}
}

/* spotter_backtrack: Given a seamount location and age, trace the
 *	hotspot-track between this seamount and a seamount of 
 *	age t_zero.  For t_zero = 0 this means the hotspot
 */

int spotter_backtrack (double xp[], double yp[], double tp[], int np, struct EULER p[], int ns, double d_km, double t_zero, BOOLEAN do_time, double **c)
/* xp, yp;	Points, in RADIANS */
/* tp;		Age of feature in m.y. */
/* np;		# of points */
/* p;		Stage poles */
/* ns;		# of stage poles */
/* d_km;	Create track point every d_km km */
/* t_zero;	Backtrack up to this age */
/* do_time;	TRUE if we want to interpolate and return time along track */
/* **c;		Pointer to return trac vector */
{
	int i, j, k, kk = 0, start_k, nd, nn, n_alloc = 2 * GMT_CHUNK;
	BOOLEAN path;
	double t, tt, dt, d_lon, tlon, dd, i_km, xnew, xx, yy, *track;
	double s_lat, c_lat, s_lon, c_lon, cc, ss, cs;

	path = (d_km > SMALL);

	if (path) {
		track = (double *) GMT_memory (VNULL, n_alloc, sizeof (double), "libspotter");
		i_km = EQ_RAD / d_km;
	}

	for (i = 0; i < np; i++) {

		if (path) {
			start_k = kk++;
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
		}
		nn = 0;
			
		t = tp[i];
		while (t > t_zero) {	/* As long as we're not back at zero age */

			j = 0;
			while (j < ns && t <= p[j].t_stop) j++;	/* Find first applicable stage pole */
			dt = MIN (p[j].duration, t - MAX(p[j].t_stop, t_zero));
			d_lon = p[j].omega_r * dt;

			/* spotter_rotate_fwd (xp[i], yp[i], &tlon, &tlat, &p[j]); */

			xnew = xp[i] - p[j].lon_r;
#ifdef SINCOS
			sincos (yp[i], &s_lat, &c_lat);
			sincos (xnew, &s_lon, &c_lon);
#else
			s_lat = sin (yp[i]);
			c_lat = cos (yp[i]);
			s_lon = sin (xnew);
			c_lon = cos (xnew);
#endif
			cc = c_lat * c_lon;
			tlon = d_atan2 (c_lat * s_lon, p[j].sin_lat * cc - p[j].cos_lat * s_lat);
			s_lat = p[j].sin_lat * s_lat + p[j].cos_lat * cc;
			c_lat = sqrt (1.0 - s_lat * s_lat);
			ss = p[j].sin_lat * s_lat;
			cs = p[j].cos_lat * s_lat;

			if (path) {
				nd = (int) ceil ((fabs (d_lon) * c_lat) * i_km);
				dd = d_lon / nd;
				tt = dt / nd;
				track[kk++] = xp[i];
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
				track[kk++] = yp[i];
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
				if (do_time) {
					track[kk++] = t;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
				}
				for (k = 1; k < nd; k++) {
					/* spotter_rotate_inv (&xx, &yy, tlon + k * dd, tlat, &p[j]); */

					xnew = tlon + k * dd;
#ifdef SINCOS
					sincos (xnew, &s_lon, &c_lon);
#else
					s_lon = sin (xnew);
					c_lon = cos (xnew);
#endif
					cc = c_lat * c_lon;
					yy = d_asin (ss - p[j].cos_lat * cc);
					xx = p[j].lon_r + d_atan2 (c_lat * s_lon, p[j].sin_lat * cc + cs);

					if (xx < 0.0) xx += TWO_PI;
					if (xx >= TWO_PI) xx -= TWO_PI;
					track[kk++] = xx;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
					track[kk++] = yy;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
					if (do_time) {
						track[kk++] = t - k * tt;
						if (kk == n_alloc) {
							n_alloc += BIG_CHUNK;
							track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
						}
					}
				}
				nn += nd;
			}
			/* spotter_rotate_inv (&xp[i], &yp[i], tlon + d_lon, tlat, &p[j]); */
			xnew = tlon + d_lon;
#ifdef SINCOS
			sincos (xnew, &s_lon, &c_lon);
#else
			s_lon = sin (xnew);
			c_lon = cos (xnew);
#endif
			cc = c_lat * c_lon;
			yp[i] = d_asin (ss - p[j].cos_lat * cc);
			xp[i] = p[j].lon_r + d_atan2 (c_lat * s_lon, p[j].sin_lat * cc + cs);

			if (xp[i] < 0.0) xp[i] += TWO_PI;
			if (xp[i] >= TWO_PI) xp[i] -= TWO_PI;
			t -= dt;
		}
		if (path) {
			track[kk++] = xp[i];
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
			track[kk++] = yp[i];
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
			if (do_time) {
				track[kk++] = t;
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
			}
			track[start_k] = nn+1;
		}
	}
	if (path) {
		track = (double *) GMT_memory ((void *)track, (size_t)kk, sizeof (double), "libspotter");
		*c = track;
		return (kk);
	}

	return (np);
}

/* spotter_forthtrack: Given a hotspot location and final age, trace the
 *	hotspot-track between the seamount created at t_zero and a
 *	seamount of age tp.  For t_zero = 0 this means from the hotspot.
 */

int spotter_forthtrack (double xp[], double yp[], double tp[], int np, struct EULER p[], int ns, double d_km, double t_zero, BOOLEAN do_time, double **c)
/* xp, yp;	Points, in RADIANS */
/* tp;		Age of feature in m.y. */
/* np;		# of points */
/* p;		Stage poles */
/* ns;		# of stage poles */
/* d_km;	Create track point every d_km km */
/* t_zero;	Foretrack from this age forward */
/* do_time;	TRUE if we want to interpolate and return time along track */
/* c;		Pointer to return trac vector */
{
	int i, j, k, kk = 0, start_k, nd, nn, n_alloc = 2 * GMT_CHUNK;
	BOOLEAN path;
	double t, tt, dt, d_lon, tlon, dd, i_km, xnew, xx, yy, *track;
	double s_lat, c_lat, s_lon, c_lon, cc, ss, cs, i_nd;

	path = (d_km > SMALL);

	if (path) {
		track = (double *) GMT_memory (VNULL, n_alloc, sizeof (double), "libspotter");
		i_km = EQ_RAD / d_km;
	}

	for (i = 0; i < np; i++) {

		if (path) {
			start_k = kk++;
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
		}
		nn = 0;

		t = t_zero;
		while (t < tp[i]) {	/* As long as we're not back at zero age */

			j = 0;
			while (j < ns && t < p[j].t_stop) j++;	/* Find first applicable stage pole */
			dt = MIN (tp[i], p[j].t_start) - t;	/* Time interval to rotate */
			d_lon = p[j].omega_r * dt;		/* Rotation angle (radians) */
			/* spotter_rotate_fwd (xp[i], yp[i], &tlon, &tlat, &p[j]); */

			xnew = xp[i] - p[j].lon_r;
#ifdef SINCOS
			sincos (yp[i], &s_lat, &c_lat);
			sincos (xnew, &s_lon, &c_lon);
#else
			s_lat = sin (yp[i]);
			c_lat = cos (yp[i]);
			s_lon = sin (xnew);
			c_lon = cos (xnew);
#endif
			cc = c_lat * c_lon;
			tlon = d_atan2 (c_lat * s_lon, p[j].sin_lat * cc - p[j].cos_lat * s_lat);
			s_lat = p[j].sin_lat * s_lat + p[j].cos_lat * cc;
			c_lat = sqrt (1.0 - s_lat * s_lat);
			ss = p[j].sin_lat * s_lat;
			cs = p[j].cos_lat * s_lat;

			if (path) {
				nd = (int) ceil ((fabs (d_lon) * c_lat) * i_km);
				i_nd = 1.0 / nd;
				dd = d_lon * i_nd;
				tt = dt * i_nd;
				track[kk++] = xp[i];
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
				track[kk++] = yp[i];
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
				if (do_time) {
					track[kk++] = t;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
				}
				for (k = 1; k < nd; k++) {
					/* spotter_rotate_inv (&xx, &yy, tlon - k * dd, tlat, &p[j]); */
					xnew = tlon - k * dd;
#ifdef SINCOS
					sincos (xnew, &s_lon, &c_lon);
#else
					s_lon = sin (xnew);
					c_lon = cos (xnew);
#endif
					cc = c_lat * c_lon;
					yy = d_asin (ss - p[j].cos_lat * cc);
					xx = p[j].lon_r + d_atan2 (c_lat * s_lon, p[j].sin_lat * cc + cs);

					if (xx < 0.0) xx += TWO_PI;
					if (xx >= TWO_PI) xx -= TWO_PI;
					track[kk++] = xx;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
					track[kk++] = yy;
					if (kk == n_alloc) {
						n_alloc += BIG_CHUNK;
						track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
					}
					if (do_time) {
						track[kk++] = t + k * tt;
						if (kk == n_alloc) {
							n_alloc += BIG_CHUNK;
							track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
						}
					}
				}
				nn += nd;
			}
			/* spotter_rotate_inv (&xp[i], &yp[i], tlon - d_lon, tlat, &p[j]); */
			xnew = tlon - d_lon;
#ifdef SINCOS
			sincos (xnew, &s_lon, &c_lon);
#else
			s_lon = sin (xnew);
			c_lon = cos (xnew);
#endif
			cc = c_lat * c_lon;
			yp[i] = d_asin (ss - p[j].cos_lat * cc);
			xp[i] = p[j].lon_r + d_atan2 (c_lat * s_lon, p[j].sin_lat * cc + cs);

			if (xp[i] < 0.0) xp[i] += TWO_PI;
			if (xp[i] >= TWO_PI) xp[i] -= TWO_PI;
			t += dt;
		}
		if (path) {
			track[kk++] = xp[i];
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
			track[kk++] = yp[i];
			if (kk == n_alloc) {
				n_alloc += BIG_CHUNK;
				track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
			}
			if (do_time) {
				track[kk++] = t;
				if (kk == n_alloc) {
					n_alloc += BIG_CHUNK;
					track = (double *) GMT_memory ((void *)track, (size_t)n_alloc, sizeof (double), "libspotter");
				}
			}
			track[start_k] = nn+1;
		}
	}
	if (path) {
		track = (double *) GMT_memory ((void *)track, (size_t)kk, sizeof (double), "libspotter");
		*c = track;
		return (kk);
	}

	return (np);
}

void spotter_rotate_fwd (double lon, double lat, double *tlon, double *tlat, struct EULER *p)
{
	/* Given the pole position in p, oblique coordinates
	 * are computed from geographical coordinates assuming a spherical earth.
	 * All values in RADIANS
	 */
	 
	double dlon, cc, test, s_lat, c_lat, c_lon, s_lon;
	 
	dlon = lon - p->lon_r;
#ifdef SINCOS
	sincos (lat, &s_lat, &c_lat);
	sincos (dlon, &s_lon, &c_lon);
#else
	s_lat = sin (lat);
	c_lat = cos (lat);
	s_lon = sin (dlon);
	c_lon = cos (dlon);
#endif
	cc = c_lat * c_lon;
	test = p->sin_lat * s_lat + p->cos_lat * cc;
	*tlat = d_asin (test);
	*tlon = d_atan2 (c_lat * s_lon, p->sin_lat * cc - p->cos_lat * s_lat);
}
	 
void spotter_rotate_inv (double *lon, double *lat, double tlon, double tlat, struct EULER *p)
{
	/* Given the pole position in project_info, geographical coordinates 
	 * are computed from oblique coordinates assuming a spherical earth.
	 * All values in RADIANS
	 */
	 
	double dlon, test, s_lat, c_lat, c_lon, s_lon, cc;
	 
	dlon = tlon;
#ifdef SINCOS
	sincos (tlat, &s_lat, &c_lat);
	sincos (dlon, &s_lon, &c_lon);
#else
	s_lat = sin (tlat);
	c_lat = cos (tlat);
	s_lon = sin (dlon);
	c_lon = cos (dlon);
#endif
	cc = c_lat * c_lon;
	test = p->sin_lat * s_lat - p->cos_lat * cc;
	*lat = d_asin (test);
	*lon = p->lon_r + d_atan2 (c_lat * s_lon, p->sin_lat * cc + p->cos_lat * s_lat);
}
