/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdtrack.c	2.65  06/20/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * grdtrack reads a xyfile, opens the 2d binary gridded grdfile,
 * and samples the dataset at the xy positions with a bilinear or bicubic
 * interpolant.  This new data is added to the input as an extra column
 * and printed to standard output.  In order to evaluate derivatives along
 * the edges of the grdfile region, we assume natural bicubic spline
 * boundary conditions (d2z/dn2 = 0, n being the normal to the edge;
 * d2z/dxdy = 0 in the corners).  Rectangles of size x_inc by y_inc are
 * mapped to [0,1] x [0,1] by affine transformation, and the interpolation
 * done on the normalized rectangle.
 *
 * Author:	Walter H F Smith
 * Date:	23-SEP-1993
 *
 * Based on the original grdtrack, which had this authorship/date/history:
 *
 * Author:	Paul Wessel
 * Date:	29-JUN-1988
 * Revised:	5-JAN-1990	PW: Updated to v.2.0
 *		4-AUG-1993	PW: Added -Q
 *		14-AUG-1998	PW: GMT 3.1
 *		10 Aug 1999 by AHC to add EMX options for binary stdin
 */

#include "gmt.h"
#include "gmt_boundcond.h"
#include "gmt_bcr.h"

float *f;

main (int argc, char **argv)
{
	int i, ix, iy, mx, my, nx, ny, n_read = 0, n_fields, n_points = 0, pad[4], one_or_zero;
	int n_output = 0, n_expected_fields = 0;
	
	BOOLEAN error = FALSE, bilinear = FALSE, suppress = FALSE, pure_ascii = FALSE;
	
	double value, west, east, south, north, *in, *out;
	
	char *grdfile, stuff[BUFSIZ], line[BUFSIZ], format1[BUFSIZ];
	
	FILE *fp = NULL;
	
	struct GRD_HEADER grd;
	
	struct GMT_EDGEINFO edgeinfo;

	grdfile = CNULL;
	west = east = south = north = 0.0;
	stuff[0] = 0;
	out = (double *)NULL;
	
	argc = GMT_begin (argc, argv);
	GMT_boundcond_init (&edgeinfo);

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			
				/* Common parameters */
			
				case 'H':
				case 'R':
				case 'V':
				case ':':
				case '\0':
					error += GMT_get_common_args (argv[i], &west, &east, &south, &north);
					break;

				/* Supplemental parameters */
				
				case 'b':
					error += GMT_io_selection (&argv[i][2]);
					break;
				case 'G':
					grdfile = &argv[i][2];
					break;
				case 'L':
					error += GMT_boundcond_parse (&edgeinfo, &argv[i][2]);
					break;
				case 'M':
					GMT_multisegment (&argv[i][2]);
					break;
				case 'Q':
					bilinear = TRUE;
					break;
				case 'S':
					suppress = TRUE;
					break;

				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else if ((fp = fopen (argv[i], GMT_io.r_mode)) == NULL) {
			fprintf (stderr, "%s: Cannot open file %s\n", GMT_program, argv[i]);
			exit (EXIT_FAILURE);
		}
	}
	
	if (argc == 1 || GMT_quick) {
		fprintf (stderr,"grdtrack %s - Sampling of a 2-D gridded netCDF grdfile along 1-D trackline\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdtrack <xyfile> -G<grdfile> [-H[<nrec>]] [-L<flag>] [-M[<flag>]] [-Q] [-Rw/s/e/n[r]] [-S] [-V] [-:] [-bi[s][<n>]] [-bo[s]]\n");
		
		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf (stderr, "	<xyfile> is an multicolumn ASCII file with (lon,lat) in the first two columns\n");
		fprintf (stderr, "	-G <grdfile> is the name of the 2-D binary data set\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		GMT_explain_option ('H');
		fprintf (stderr, "	-L sets boundary conditions.  <flag> can be either\n");
		fprintf (stderr, "	   g for geographic boundary conditions\n");
		fprintf (stderr, "	   or one or both of\n");
		fprintf (stderr, "	   x for periodic boundary conditions on x\n");
		fprintf (stderr, "	   y for periodic boundary conditions on y\n");
		GMT_explain_option ('M');
		fprintf (stderr, "	-Q Quick mode, use bilinear rather than bicubic interpolation\n");
		GMT_explain_option ('R');
		fprintf (stderr, "	-S Suppress output when result equals NaN\n");
		GMT_explain_option ('V');
		GMT_explain_option (':');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		fprintf (stderr, "	  Default is 2 input columns\n");
		GMT_explain_option ('o');
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}
	
	if (!grdfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G:  Must specify output file\n", GMT_program);
		error++;
	}
	if (GMT_io.binary[0] && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
		error++;
	}
        if (GMT_io.binary[0] && GMT_io.ncol[0] == 0) GMT_io.ncol[0] = 2;
        if (GMT_io.binary[0] && GMT_io.ncol[0] < 2) {
                fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data (-bi) must have at least 2 columns\n", GMT_program);
		error++;
	}
	
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands */

	if (GMT_io.binary[0] && gmtdefs.verbose) {
		char *type[2] = {"double", "single"};
		fprintf (stderr, "%s: Expects %d-column %s-precision binary data\n", GMT_program, GMT_io.ncol[0], type[GMT_io.single_precision[0]]);
	}

	pure_ascii = !(GMT_io.binary[0] || GMT_io.binary[1]);

	if (fp == NULL) {
		fp = GMT_stdin;
		if (gmtdefs.verbose) fprintf (stderr, "%s: Reads from standard input\n", GMT_program);
	}

#ifdef __EMX__	  /* If EMX is set, set mode of stdin to 'binary' */
		if (GMT_io.binary[0]) {
			fflush(GMT_stdin);
			_fsetmode(GMT_stdin,"b");
		}
#endif

	if (GMT_read_grd_info (grdfile, &grd)) {
		fprintf (stderr, "%s: Error opening file %s\n", GMT_program, grdfile);
		exit (EXIT_FAILURE);
	}
	
	if (west == east) {	/* No subset asked for */
		west = grd.x_min;
		east = grd.x_max;
		south = grd.y_min;
		north = grd.y_max;
	}
	one_or_zero = (grd.node_offset) ? 0 : 1;
	nx = irint ( (east - west) / grd.x_inc) + one_or_zero;
	ny = irint ( (north - south) / grd.y_inc) + one_or_zero;
	mx = nx + 4;
	my = ny + 4;
	
	f = (float *) GMT_memory (VNULL, (size_t)(mx * my), sizeof (float), GMT_program);

	pad[0] = pad[1] = pad[2] = pad[3] = 2;
	if (GMT_read_grd (grdfile, &grd, f, west, east, south, north, pad, FALSE)) {
		fprintf (stderr, "%s: Error reading file %s\n", GMT_program, grdfile);
		exit (EXIT_FAILURE);
	}

	GMT_boundcond_param_prep (&grd, &edgeinfo);

	/* Initialize bcr structure:  */

	GMT_bcr_init (&grd, pad, bilinear);

	/* Set boundary conditions  */
	
	GMT_boundcond_set (&grd, &edgeinfo, pad, f);
	
	sprintf (format1, "%s\t%s\t%%s\t%s\n\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
	
	if (gmtdefs.io_header) {	/* First echo headers, if any */
		for (i = 0; i < gmtdefs.n_header_recs - 1; i++) {
			fgets (line, BUFSIZ, fp);
			if (!GMT_io.binary[1]) printf ("%s", line);
		}
		fgets (line, BUFSIZ, fp);
		line[strlen(line)-1] = 0;
		if (!GMT_io.binary[1]) printf ("%s\tsample\n", line);
	}

	ix = (gmtdefs.xy_toggle) ? 1 : 0;	iy = 1 - ix;		/* Set up which columns have x and y */

	n_expected_fields = (GMT_io.ncol[0]) ? GMT_io.ncol[0] : BUFSIZ;

	while ((n_fields = GMT_input (fp, &n_expected_fields, &in)) >= 0 && !(GMT_io.status & GMT_IO_EOF)) {

		while (GMT_io.status & GMT_IO_SEGMENT_HEADER) {
			GMT_write_segmentheader (GMT_stdout, n_expected_fields);
			n_fields = GMT_input (fp,  &n_expected_fields, &in);
		}
		if ((GMT_io.status & GMT_IO_EOF)) continue;	/* At EOF */

		if (GMT_io.status & GMT_IO_MISMATCH) {
			fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields, n_expected_fields, n_read);
			exit (EXIT_FAILURE);
		}
		if (n_output == 0) n_output = n_expected_fields + 1;

		n_read++;

		/* If point is outside grd area, shift it using periodicity or skip if not periodic. */

		while ( (in[iy] < grd.y_min) && (edgeinfo.nyp > 0) ) in[iy] += (grd.y_inc * edgeinfo.nyp);
		if (in[iy] < grd.y_min) continue;

		while ( (in[iy] > grd.y_max) && (edgeinfo.nyp > 0) ) in[iy] -= (grd.y_inc * edgeinfo.nyp);
		if (in[iy] > grd.y_max) continue;

		while ( (in[ix] < grd.x_min) && (edgeinfo.nxp > 0) ) in[ix] += (grd.x_inc * edgeinfo.nxp);
		if (in[ix] < grd.x_min) continue;

		while ( (in[ix] > grd.x_max) && (edgeinfo.nxp > 0) ) in[ix] -= (grd.x_inc * edgeinfo.nxp);
		if (in[ix] > grd.x_max) continue;
		
		value = GMT_get_bcr_z(&grd, in[ix], in[iy], f, &edgeinfo);

		if (suppress && GMT_is_dnan (value)) continue;

		if (!out) out = (double *) GMT_memory (VNULL, (size_t)n_output, sizeof (double), GMT_program);

		if (pure_ascii && n_expected_fields > 2) {
			/* Special case: Ascii i/o and at least 3 columns:
			   Columns beyond first two could be text strings */

			/* First get rid of any commas that may cause grief */
			for (i = 0; GMT_io.current_record[i]; i++) if (GMT_io.current_record[i] == ',') GMT_io.current_record[i] = ' ';
			sscanf (GMT_io.current_record, "%*lf %*lf %[^\n]", stuff);
			printf (format1, in[ix], in[iy], stuff, value);
		}
		else {	/* Simply copy other columns, append value, and output */
			out[0] = in[ix];
			out[1] = in[iy];
			for (i = 2; i < n_expected_fields; i++) out[i] = in[i];
			out[i] = value;
			GMT_output (GMT_stdout, n_output, out);
		}

		n_points++;
	}
	fclose(fp);
	
	if (gmtdefs.verbose) fprintf (stderr, "%s: Sampled %d points from grid %s (%d x %d)\n", GMT_program,
		n_points, grdfile, grd.nx, grd.ny);
	
	free ((void *)f);
	free ((void *)out);
	
	GMT_end (argc, argv);
}
