/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdreformat.c	3.31  04/04/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * grdreformat.c reads a grd file in one format and outputs it in another
 *
 * Author:	Paul Wessel
 * Date:	3-JAN-1991-1999
 * Version:	3.1
 */

#include "gmt.h"

float *z;

main (int argc, char **argv)
{
	BOOLEAN error = FALSE, global = FALSE;

	int i, nfiles = 0, nm, nx, ny, one_or_zero, dummy[4];

	double w, e, s, n, scale = 1.0, offset = 0.0;

	char *file[2], fname[2][BUFSIZ], line[BUFSIZ];

	struct GRD_HEADER grd;

	FILE *fp;
	
	file[0] = file[1] = CNULL;

	argc = GMT_begin (argc, argv);

	w = e = s = n = 0.0;
	dummy[0] = dummy[1] = dummy[2] = dummy[3] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'R':
				case 'V':
				case '\0':
					error += GMT_get_common_args (argv[i], &w, &e, &s, &n);
					break;
					
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else if (nfiles < 2) {
			file[nfiles] = argv[i];
			nfiles++;
		}
		else
			nfiles++;
	}
	
	if (argc == 1 || GMT_quick) {	/* Display usage */
		fprintf (stderr, "grdreformat %s - Converting between different grdfile formats\n\n", GMT_VERSION);
		fprintf( stderr, "usage: grdreformat ingrdfile[=id[/scale/offset]] outgrdfile[=id[/scale/offset]] [-Rw/e/s/n] [-V]\n");

		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf (stderr, "	ingrdfile is the grd file to convert\n");
		fprintf (stderr, "	outgrdfile is the new converted grd file\n");
		fprintf( stderr, "	   scale and offset, if given, will multiply data by scale and add offset.\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		GMT_explain_option ('r');
		GMT_explain_option ('V');
		
		fprintf (stderr, "	The following formats are supported\n\n");
		sprintf (line, "%s%clib%cgmtformats.d\0", GMTHOME, DIR_DELIM, DIR_DELIM);
		if ((fp = fopen (line, "r")) == NULL) {
			fprintf (stderr, "%s: GMT ERROR: Cannot read file %s\n", GMT_program, line);
			exit (EXIT_FAILURE);
		}
		while (fgets (line, BUFSIZ, fp)) if (line[0] != '#') fprintf (stderr, "\t%s", line);
		fclose (fp);
		exit (EXIT_FAILURE);
	}
	
	if (nfiles != 2) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Must specify both input and output file names\n", GMT_program);
		error++;
	}
	
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands */
	
	GMT_grd_i_format = GMT_grd_get_i_format (file[0], fname[0], &scale, &offset);
	GMT_grd_o_format = GMT_grd_get_o_format (file[1], fname[1], &scale, &offset);
	
	if (GMT_grd_i_format == GMT_grd_o_format) {
		fprintf (stderr, "%s: Formats are identical - aborts\n", GMT_program);
		exit (EXIT_FAILURE);
	}
	
	if (gmtdefs.verbose) {
		if (file[0][0] == '=') strcpy (fname[0], "<stdin>");
		if (file[1][0] == '=') strcpy (fname[1], "<stdout>");
		fprintf (stderr, "%s: Translating file %s (format = %d) to file %s (format = %d)\n", GMT_program, fname[0], GMT_grd_i_format, fname[1], GMT_grd_o_format);
	}
	
	if (GMT_read_grd_info (file[0], &grd)) {
		fprintf (stderr, "%s: Error opening file %s\n", GMT_program, fname[0]);
		exit (EXIT_FAILURE);
	}
	
	nm = grd.nx * grd.ny;
	
	if (e > w && n > s) {
		global = (fabs (grd.x_max - grd.x_min) == 360.0);
		if (!global && (w < grd.x_min || e > grd.x_max)) error = TRUE;
		if (s < grd.y_min || n > grd.y_max) error = TRUE;
		if (error) {
			fprintf (stderr, "%s: Subset exceeds data domain!\n", GMT_program);
			exit (EXIT_FAILURE);
		}
		one_or_zero = (grd.node_offset) ? 0 : 1;
		nx = irint ((e - w) / grd.x_inc) + one_or_zero;
		ny = irint ((n - s) / grd.y_inc) + one_or_zero;
		
		z = (float *) GMT_memory (VNULL, (size_t)(nx * ny), sizeof (float), GMT_program);
		
		if (GMT_read_grd (file[0], &grd, z, w, e, s, n, dummy, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, fname[0]);
			exit (EXIT_FAILURE);
		}
	}
	else {
		z = (float *) GMT_memory (VNULL, (size_t)nm, sizeof (float), GMT_program);

		if (GMT_read_grd (file[0], &grd, z, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, fname[0]);
			exit (EXIT_FAILURE);
		}
	}

	if (GMT_write_grd (file[1], &grd, z, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
		fprintf (stderr, "%s: Error writing file %s\n", GMT_program, fname[1]);
		exit (EXIT_FAILURE);
	}
	
	free ((void *)z);
		
	GMT_end (argc, argv);
}
