/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdproject.c	2.39  06/20/99
 *
 *	Copyright (c) 1991-1999 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: www.soest.hawaii.edu/gmt
 *--------------------------------------------------------------------*/
/*
 * grdproject reads a geographical grdfile and evaluates the grid at new grid positions
 * specified by the map projection and new dx/dy values using a weighted average of all
 * points within the search radius. Optionally, grdproject may perform the inverse
 * transformation, going from rectangular coordinates to geographical.
 *
 * Author:	Paul Wessel
 * Date:	31-May-1999
 * Ver:		3.3
 *
 */

#include "gmt.h"

float *geo, *rect;

void special_fix_for_grdproject (int argc, char **argv);

main (int argc, char **argv)
{
	int i, dpi, nx, ny, dummy[4];
	
	BOOLEAN error = FALSE, inverse = FALSE, n_set = FALSE, set_n = FALSE;
	BOOLEAN d_set = FALSE, e_set = FALSE, map_center = FALSE, offset, toggle_offset = FALSE;
	
	double w, e, s, n, x_inc = 0.0, y_inc = 0.0, max_radius = 0.0;
	double xmin, xmax, ymin, ymax;
	
	char *infile, *outfile, format[BUFSIZ];
	
	struct GRD_HEADER g_head, r_head;
	
	argc = GMT_begin (argc, argv);
	
	special_fix_for_grdproject (argc, argv);

	infile = outfile = CNULL;
	w = e = s = n = 0.0;
	dummy[3] = dummy[2] = dummy[1] = dummy[0] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'J':
				case 'R':
				case 'V':
				case '\0':
					error += GMT_get_common_args (argv[i], &w, &e, &s, &n);
					break;
				
				/* Supplemental parameters */
				
				case 'D':
					GMT_getinc (&argv[i][2], &x_inc, &y_inc);
					d_set = TRUE;
					break;
				case 'E':
					dpi = atoi (&argv[i][2]);
					e_set = TRUE;
					break;
				case 'F':
					toggle_offset = TRUE;
					break;
				case 'G':
					outfile = &argv[i][2];
					break;
				case 'I':
					inverse = TRUE;
					break;
				case 'M':
					map_center = TRUE;
					if (argv[i][2] == 'm') gmtdefs.measure_unit = 2;	/* Select meters */
					break;
				case 'N':
					sscanf (&argv[i][2], "%d/%d", &nx, &ny);
					if (ny == 0) ny = nx;
					n_set = TRUE;
					break;
				case 'S':
					max_radius = atof (&argv[i][2]);
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else 
			infile = argv[i];
	}
	
	if ((d_set + e_set + n_set) == 0) n_set = set_n = TRUE;

	if (argc == 1 || GMT_quick) {
		fprintf (stderr, "grdproject %s - Project geographical grid to/from rectangular grid\n\n", GMT_VERSION);
		fprintf(stderr, "usage: grdproject <in_grdfile> -J<parameters> -R<west/east/south/north>\n");
		fprintf(stderr, "	[-D<dx[m|c]>[/dy[m|c]]] [-E<dpi>] [-F] [-G<out_grdfile>] [-I] [-M]\n");
		fprintf(stderr, "	[-N<nx/ny>] [-S<radius>] [-V]\n\n");
		
		if (GMT_quick) exit (EXIT_FAILURE);
		
		fprintf(stderr, "	<in_grdfile> is data set to be transformed\n");
		GMT_explain_option ('J');
		GMT_explain_option ('R');
		fprintf(stderr, "\n\tOPTIONS:\n");
		fprintf(stderr, "	-D sets the grid spacing for the new grid\n");
		fprintf(stderr, "	-E sets dpi for output grid\n");
		fprintf(stderr, "	-F toggle between pixel and grid registration  [Default is same as input]\n");
		fprintf(stderr, "	-G name of output grid\n");
		fprintf(stderr, "	-I Inverse transformation from rectangular to geographical\n");
		fprintf (stderr, "	-M coordinates relative to projection center [Default is relative to lower left corner]\n");
		fprintf(stderr, "	   Append m to set meter as the measure unit\n");
		fprintf(stderr, "	-N sets the number of nodes for the new grid\n");
		fprintf(stderr, "	   Only one of -D, -E, and -N can be specified!\n");
		fprintf(stderr, "	   If none are specified, nx,ny of the input file is used\n");
		fprintf(stderr, "	-S sets the search radius in projected units [Default avoids aliasing]\n");
		GMT_explain_option ('V');
		
		exit (EXIT_FAILURE);
	}
	
	if (!infile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", GMT_program);
		error++;
	}
	if (!outfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option:  Must specify output file\n", GMT_program);
		error++;
	}
	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", GMT_program);
		error++;
	}
	if (max_radius < 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S:  Max search radius, if specified, must be positive\n", GMT_program);
		error++;
	}
	if ((d_set + e_set + n_set) != 1) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify only one of -D, -E, or -N\n", GMT_program);
		error++;
	}
	if (d_set && (x_inc <= 0.0 || y_inc <= 0.0)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -D option.  Must specify positive increment(s)\n", GMT_program);
		error++;
	}
	if (n_set && !set_n && (nx <= 0 || ny <= 0)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -N option.  Must specify positive integers\n", GMT_program);
		error++;
	}
	if (e_set && dpi <= 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option.  Must specify positive dpi\n", GMT_program);
		error++;
	}
	
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands */

	GMT_map_setup (w, e, s, n);
	
	xmin = (map_center) ? project_info.xmin - project_info.x0 : project_info.xmin;
	xmax = (map_center) ? project_info.xmax - project_info.x0 : project_info.xmax;
	ymin = (map_center) ? project_info.ymin - project_info.y0 : project_info.ymin;
	ymax = (map_center) ? project_info.ymax - project_info.y0 : project_info.ymax;
	
	GMT_grd_init (&r_head, argc, argv, FALSE);
	GMT_grd_init (&g_head, argc, argv, FALSE);

	sprintf (format, "(%s/%s/%s/%s)\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);

	if (inverse) {	/* Transforming from rectangular projection to geographical */
	
		if (!project_info.region) d_swap (s, e);  /* Got w/s/e/n, make into w/e/s/n */

		g_head.x_min = w;	g_head.x_max = e;	g_head.y_min = s;	g_head.y_max = n;
	
		if (GMT_read_grd_info (infile, &r_head)) {
			fprintf (stderr, "%s: Error opening file %s\n", GMT_program, infile);
			exit (EXIT_FAILURE);
		}
		rect = (float *) GMT_memory (VNULL, (size_t)(r_head.nx * r_head.ny), sizeof (float), GMT_program);
		if (GMT_read_grd (infile, &r_head, rect, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", infile, GMT_program);
			exit (EXIT_FAILURE);
		}
		offset = r_head.node_offset;		/* Same as input */
		if (toggle_offset) offset = !offset;	/* Toggle */

		if (set_n) {
			nx = r_head.nx;
			ny = r_head.ny;
		}
		GMT_grdproject_init (&g_head, x_inc, y_inc, nx, ny, dpi, offset);
		geo = (float *) GMT_memory (VNULL, (size_t)(g_head.nx * g_head.ny), sizeof (float), GMT_program);
		if (gmtdefs.verbose) {
			fprintf (stderr, "%s:  Transform ", GMT_program);
			fprintf (stderr, format, g_head.x_min, g_head.x_max, g_head.y_min, g_head.y_max);
			fprintf (stderr, " <-- ");
			fprintf (stderr, format, xmin, xmax, ymin, ymax);
			fprintf (stderr, "\n");
		}

		GMT_grd_inverse (geo, &g_head, rect, &r_head, max_radius, map_center);
			
		if (GMT_write_grd (outfile, &g_head, geo, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "%s: Error writing file %s\n", GMT_program, outfile);
			exit (EXIT_FAILURE);
		}
		
	}
	else {	/* Forward projection from geographical to rectangular grid */
	
		if (GMT_read_grd_info (infile, &g_head)) {
			fprintf (stderr, "%s: Error opening file %s\n", GMT_program, infile);
			exit (EXIT_FAILURE);
		}

		if (project_info.projection == MOLLWEIDE && (fabs (g_head.x_min - g_head.x_max) < 360.0 || (g_head.y_max - g_head.y_min) < 180.0)) {
			fprintf (stderr, "%s: Selected projection works on global datasets only\n", GMT_program);
			exit (EXIT_FAILURE);
		}
	
		geo = (float *) GMT_memory (VNULL, (size_t)(g_head.nx * g_head.ny), sizeof (float), GMT_program);
		if (GMT_read_grd (infile, &g_head, geo, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, infile);
			exit (EXIT_FAILURE);
		}
	
		r_head.x_min = xmin;	r_head.x_max = xmax;
		r_head.y_min = ymin;	r_head.y_max = ymax;
		if (set_n) {
			nx = g_head.nx;
			ny = g_head.ny;
		}
		
		if (gmtdefs.verbose) {
			fprintf (stderr, "%s:  Transform ", GMT_program);
			fprintf (stderr, format, g_head.x_min, g_head.x_max, g_head.y_min, g_head.y_max);
			fprintf (stderr, " --> ");
			fprintf (stderr, format, xmin, xmax, ymin, ymax);
			fprintf (stderr, "\n");
		}

		offset = g_head.node_offset;		/* Same as input */
		if (toggle_offset) offset = !offset;	/* Toggle */

		GMT_grdproject_init (&r_head, x_inc, y_inc, nx, ny, dpi, offset);
		rect = (float *) GMT_memory (VNULL, (size_t)(r_head.nx * r_head.ny), sizeof (float), GMT_program);
		GMT_grd_forward (geo, &g_head, rect, &r_head, max_radius, map_center);
		
		if (GMT_write_grd (outfile, &r_head, rect, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "%s: Error writing file %s\n", GMT_program, outfile);
			exit (EXIT_FAILURE);
		}
		
	}
	
	free ((void *)geo);
	free ((void *)rect);
	
	GMT_end (argc, argv);
}

/* Because -M here sets parameters that are used when -J is
 * processed we must make sure that -M occurs before -J
 * in the argument order.  This is a so-called Cheap Fix
 * of the 2nd Kind.
 */

void special_fix_for_grdproject (int argc, char **argv) {

	int i, i_j = 0, i_m = 0;
	char *hold;

	/* Find positions in argv for -M and -J */

	for (i = 1; i < argc; i++) {
		if (!strncmp (argv[i], "-J", 2)) i_j = i;
		if (!strncmp (argv[i], "-M", 2)) i_m = i;
	}

	if (i_m == 0 || i_j == 0) return;	/* No need: either no -M [ok] or missin -J [error caught later] */
	if (i_m < i_j) return;			/* No need: -M already ahead of -J */

	/* Now swap pointers to the two arguments */

	hold = argv[i_j];	/* Pointer to the -J argument */
	argv[i_j] = argv[i_m];	/* Now points to -M */
	argv[i_m] = hold;	/* The swap is complete - life is again beautiful */
}

	
